{$S-,R-,V-,I-,B-,O+,F+,A-}

{*********************************************************}
{*                   APANSI.PAS 1.03                     *}
{*     Copyright (c) TurboPower Software 1991.           *}
{*                 All rights reserved.                  *}
{*********************************************************}

Unit ANSiDrv;
  {-Does ANSI screen writing and produces ANSI sequences}

Interface

Uses
  Crt, DOS;

Type
  ClearScreenProc = Procedure;
  SoundBellProc  = Procedure;
  
Const
  {Prevent remote from changing our video mode}
  InhibitModeChange : Boolean = False;
  UseVT100Mode : Boolean = False;                                     {!!.02}
  MaskBlinkOnInverse : Boolean = True;                                {!!.02}
  {For sizing the screen}
  AnsiWidth  : Word = 80;
  AnsiHeight : Word = 24;
  
Procedure WriteCharAnsi (C : Char);
Procedure WriteStringAnsi (S : String);
Procedure WriteStringANSINOCR (S : String);
Procedure SetClearScreenProc (CSP : ClearScreenProc);
Procedure SetSoundBellProc (SBP : SoundBellProc);

Implementation

Type
  {Token types}
  ParserType = (GotNone, GotEscape, GotBracket, GotSemiColon,
  GotParm, GotCommand);
  
Var
  CR : String;
  
Const
  {Special parser characters}
  Escape = #27;
  LeftBracket = #91;
  Semicolon = #59;
  FormFeed = #12;
  BellChar = #07;
  EqualSign = #61;
  QuestionMark = #63;                                                 {!!.02}
  
  {For sizing parser}
  MaxQueueChars = 10;
  MaxParms = 5;
  
  {For saving TextAttr states}
  Inverse : Boolean = False;
  Intense : Boolean = False;
  
  {For saving and restoring the cursor state}
  SaveX : Byte = 1;
  SaveY : Byte = 1;
  
Var
  {For saving invalid escape sequences}
  SaveCharQueue : Array [1..MaxQueueChars] Of Char;
  QTail : Byte;
  
  {For collecting and converting parameters}
  Parms : Array [1..MaxParms] Of String [5];
  ParmInt : Array [1..MaxParms] Of Integer;
  ParmDefault : Array [1..MaxParms] Of Boolean;
  ParmIndex : Byte;
  
  {Current token}
  ParserState : ParserType;
  
  {User hooks}
  ClearScreen    : ClearScreenProc;
  SoundBell      : SoundBellProc;
  
Procedure WriteStringAnsi (S : String);
    {-Writes S (and handles ANSI escape sequences)}
  Var
    I : Byte;
  Begin
    S := S + #10#13;
    For I := 1 To Length (S) Do
      WriteCharAnsi (S [I] );
  End;

    Procedure WriteStringAnsiNOCR (S : String);
    {-Writes S (and handles ANSI escape sequences)}
  Var
    I : Byte;
  Begin
    For I := 1 To Length (S) Do
      WriteCharAnsi (S [I] );
  End;


  Procedure InitParser;
    {-Initialize parser for next ansi sequence}
  Var
    I : Byte;
  Begin
    QTail := 0;
    ParmIndex := 1;
    For I := 1 To MaxParms Do Begin
      Parms [I] := '';
      ParmDefault [I] := False;
    End;
    ParserState := GotNone;
  End;

  Procedure PushChar (C : Char);
    {-Push C into the saved char queue}
  Begin
    If QTail < MaxQueueChars Then Begin
      Inc (QTail);
      SaveCharQueue [QTail] := C;
    End;
  End;

  Function HeadChar (Var C : Char) : Boolean;
    {-Returns the first character on the saved stack and moves the rest down}
  Begin
    If QTail > 0 Then Begin
      C := SaveCharQueue [1];
      HeadChar := True;
      Dec (QTail);
      Move (SaveCharQueue [2], SaveCharQueue [1], QTail);
    End Else
      HeadChar := False;
  End;

  Procedure BuildParm (C : Char);
    {-Gets the next character of the current parameter}
  Begin
    Parms [ParmIndex] := Parms [ParmIndex] + C;
  End;

  Procedure ConvertParms (C : Char);                                   {!!.02}
    {-Convert the parms into integers}
  Var
    I, Code : Integer;
  Begin
    For I := 1 To MaxParms Do Begin
      Val (Parms [I], ParmInt [I], Code);
      If Code <> 0 Then Begin
        ParmInt [I] := 1;
        ParmDefault [I] := True;
      End;
    End;
    If ParmDefault [1] And (C In ['J', 'K'] ) Then                      {!!.02}
      If UseVT100Mode Then                                            {!!.02}
        ParmInt [1] := 0                                               {!!.02}
      Else                                                            {!!.02}
        ParmInt [1] := 2;                                              {!!.02}
    If (ParmInt [1] = 0) And (C In ['A', 'B', 'C', 'D'] ) Then             {!!.03}
      ParmInt [1] := 1;                                                {!!.03}
  End;

  Procedure ClearPart (X1, Y1, X2, Y2 : Integer);
    {-Clear from X1, Y1 to X2, Y2}
  Var
    Row, Col : Integer;
    SaveX, SaveY : Word;
    
  Procedure ClearRow (X1, X2 : Integer);
    Var
      I : Integer;
    Begin
      GotoXY (X1, WhereY);
      If X2 = AnsiWidth Then
        ClrEol
      Else
        For I := X1 To X2 Do
          Write (' ');
    End;

  Begin
    {Save cursor position}
    SaveX := WhereX;
    SaveY := WhereY;
    GotoXY (X1, Y1);
    
    If Y1 = Y2 Then ClearRow (X1, X2)
    Else Begin
      ClearRow (X1, AnsiWidth);
      If Y1 + 1 <= Y2 - 1 Then
        For Row := Y1 + 1 To Y2 - 1 Do Begin
          GotoXY (1, Row);
          ClearRow (1, AnsiWidth);
        End;
      GotoXY (1, Y2);
      ClearRow (1, X2);
    End;
    GotoXY (SaveX, SaveY);
  End;

  Procedure GotoXYCheck (X, Y : Integer);
    {-GotoXY that checks against negative numbers}
  Begin
    If X < 1 Then X := 1;
    If Y < 1 Then Y := 1;
    GotoXY (X, Y);
  End;

  Procedure ReportCursorPosition;                                      {!!.02}
    {-Output CPR sequence with cursor position (no error checking)}
  Const
    AnsiStart = #27'[';
  Var
    S1, S2 : String [8];
    I : Byte;
  Begin
    
    {Make an output string like so: <esc>[<wherex>;<wherey>R}
    Str (WhereX, S1);
    Str (WhereY, S2);
    S1 := AnsiStart + S1 + ';' + S2 + 'R';
  End;

  Procedure ProcessCommand (C : Char);
    {-Process the current command}
  Var
    X, I, TextFg, TextBk : Byte;
    
    {  Procedure SenditAll(S:String);
    Var X:Byte;
    Begin
    For X:=1 To Length(S) Do Write(S[X]);
    End;
    }
  Begin
    {Convert parameter strings to integers (and assign defaults)}
    ConvertParms (C);
    
    {Act on the accumulated parameters}
    Case C Of
      {'~' : {SendItAll('PrvT%');}
      'f' : 
            Begin
              gotoxyCheck (parmint [2], parmint [1] );  {HVP - horizontal and vertical position}
            End;
      
      'H' : {CUP - cursor position}
            Begin
              GotoXYcheck (ParmInt [2], ParmInt [1] );
            End;
      
      'A' : {CUU - cursor up}
              GotoXYCheck (WhereX, WhereY - ParmInt [1] );
      
      'B' : {CUD - cursor down}
              GotoXYCheck (WhereX, WhereY + ParmInt [1] );
      
      'C' : {CUF - cursor forward}
              GotoXYCheck (WhereX + ParmInt [1], WhereY);
      
      'D' : {CUB - cursor back}
              GotoXYCheck (WhereX - ParmInt [1], WhereY);
      
      'J' : {ED - erase display}
              Case ParmInt [1] Of
                0 : ClearPart (WhereX, WhereY, AnsiWidth, AnsiHeight);
                1 : ClearPart (1, 1, WhereX, WhereY);
                2 : For X := 1 To 25 Do Begin GotoXY (1, X); ClrEol; GotoXY (1, 1) End;
              End;
      
      'K' : {EL - erase in line}
            Begin
              If ParmDefault [1] Then
                ParmInt [1] := 0;
              Case ParmInt [1] Of
                0 : ClrEol;
                1 : ClearPart (1, WhereY, WhereX, WhereY);
                2 : ClearPart (1, WhereY, AnsiWidth, WhereY);
              End;
            End;
      
      'l',
      'h' : {SM - set mode (supports text modes only)}
              If Not InhibitModeChange Then Begin
                Case ParmInt [1] Of
                  0 : TextMode (BW40);
                  1 : TextMode (CO40);
                  2 : TextMode (BW80);
                  3 : TextMode (CO80);
                  4 : TextMode (c80 + Font8X8);
                End;
                Case ParmInt [1] Of
                  0, 1 : AnsiWidth := 40;
                  2, 3 : AnsiWidth := 80;
                End;
              End;
      
      'm' : {SGR - set graphics rendition (set background color)}
            Begin
              For I := 1 To ParmIndex Do Begin
                If Inverse Then
                  {Restore inverted TextAttr before continuing}
                  TextAttr := (TextAttr ShL 4) Or (TextAttr ShR 4);
                
                {Separate out the forground and background bits}
                TextFg := TextAttr And $0F;
                TextBk := TextAttr And $F0;
                
                {Process the color command}
                Case ParmInt [I] Of
                  0  : Begin
                    TextAttr := $07;                {White on black}
                    Inverse := False;
                    Intense := False;
                  End;
                  1  : Intense  := True;               {Set intense bit later}
                  4  : Intense  := True;               {Subst intense for underline}
                  5  : TextAttr := TextAttr Or $80;    {Set blinking on}
                  7  : Inverse  := True;               {Invert TextAttr later}
                  8  : TextAttr := $00;                {Invisible}
                  27 : Inverse  := False;              {Stop inverting TextAttr}
                  30 : TextAttr := TextBk Or $00;      {Black foreground}
                  31 : TextAttr := TextBk Or $04;      {Red foreground}
                  32 : TextAttr := TextBk Or $02;      {Green foreground}
                  33 : TextAttr := TextBk Or $06;      {Yellow forground}
                  34 : TextAttr := TextBk Or $01;      {Blue foreground}
                  35 : TextAttr := TextBk Or $05;      {Magenta foreground}
                  36 : TextAttr := TextBk Or $03;      {Cyan foreground}
                  37 : TextAttr := TextBk Or $07;      {White foreground}
                  40 : TextAttr := TextFg;             {Black background}
                  41 : TextAttr := TextFg Or $40;      {Red background}
                  42 : TextAttr := TextFg Or $20;      {Green background}
                  43 : TextAttr := TextFg Or $60;      {Yellow background}
                  44 : TextAttr := TextFg Or $10;      {Blue background}
                  45 : TextAttr := TextFg Or $50;      {Magenta background}
                  46 : TextAttr := TextFg Or $30;      {Cyan background}
                  47 : TextAttr := TextFg Or $70;      {White background}
                End;
                
                {Fix up TextAttr for inverse and intense}
                If Inverse Then Begin                                     {!!.02}
                  TextAttr := (TextAttr ShL 4) Or (TextAttr ShR 4);
                  If MaskBlinkOnInverse Then                              {!!.02}
                    TextAttr := TextAttr And $7F;                         {!!.02}
                End;                                                      {!!.02}
                If Intense Then
                  TextAttr := TextAttr Or $08;
              End;
            End;
      
      'n' : {DSR - device status report}                             {!!.02}
              If ParmInt [1] = 6 Then                                       {!!.02}
                ReportCursorPosition;                                      {!!.02}
      
      's' : {SCP - save cursor position}
            Begin
              SaveX := WhereX;
              SaveY := WhereY;
            End;
      
      'u' : {RCP - restore cursor position}
              GotoXY (SaveX, SaveY);
      
      Else
        {Invalid esc sequence - display all the characters accumulated so far}
        While HeadChar (C) Do
          Case C Of
            FormFeed : ClearScreen;
            BellChar : SoundBell;
            Else Write (C);
          End;
    End;
  End;

  Procedure WriteCharAnsi (C : Char);
    {-Writes C (and handles ANSI sequences)}
  Label
    ErrorExit;
  Begin
    PushChar (C);
    
    Case ParserState Of
      GotNone : {Not in an ANSI sequence}
                Begin
                  If C = Escape Then
                    ParserState := GotEscape
                  Else
                    Case C Of
                      FormFeed : {Special case - clear screen on formfeed}
                                   ClearScreen;
                      BellChar : {Special case - ring bell on bell character}
                                   SoundBell;
                      Else       {Normal character, just write it}
                        Write (C);
                    End;
                  {Fast reinit of parser}
                  QTail := 0;
                End;
      
      GotEscape : {Last character was escape -- need [}
                    If C = LeftBracket Then
                      ParserState := GotBracket
                    Else
                      Goto ErrorExit;
      
      GotParm,
      GotBracket,
      GotSemicolon : {Need parameter char, semicolon, equalsign or command}
                       If (C >= #48) And (C <= #57) Then Begin
                         {It's a number, go add it to the current parameter}
                         BuildParm (C);
                         ParserState := GotParm;
                       End Else If (C = EqualSign) Or (C = QuestionMark) Then        {!!.02}
        {just ignore it}
      Else If C = Semicolon Then
        {It's a semicolon, prepare for next parameter}
        If ParserState = GotSemicolon Then
          Goto ErrorExit
        Else Begin
          ParserState := GotSemicolon;
          Inc (ParmIndex);
          If ParmIndex > MaxParms Then
            Goto ErrorExit;
        End
      Else Begin
        {Must be a command, go process it}
        ProcessCommand (C);
        InitParser;
      End;
    End;
    Exit;
    
    ErrorExit:
    {Invalid escape sequence -- display all the characters accumulated so far}
    While HeadChar (C) Do
      Write (C);
    InitParser;
  End;

  Procedure DefClearScreen;
  Begin
    Window (1, 1, 80, 23);
    ClrScr;
  End;

  Procedure DefSoundBell;
  Begin
    Sound (220);
    Delay (200);
    NoSound;
  End;
               Procedure SetClearScreenProc (CSP : ClearScreenProc);
    {-Sets a ClearScreen procedure to be called on FormFeed characters}
  Begin
    ClearScreen := CSP;
  End;

  Procedure SetSoundBellProc (SBP : SoundBellProc);
    {-Sets a SoundBell procedure to be called on Bell characters}
  Begin
    SoundBell := SBP;
  End;

Begin
  InitParser;
  SoundBell := DefSoundBell;
  ClearScreen := DefClearScreen;
End.
