/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.util;

import org.ckkloverdos.collection.IL;
import org.ckkloverdos.java.JavaPlatform;
import org.ckkloverdos.log.StdLog;

import java.io.File;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Array;
import java.net.InetAddress;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.text.SimpleDateFormat;
import java.util.*;

/**
 * General utility methods.
 * 
 * @author Christos KK Loverdos
 */
public final class Util
{
    /**
     * Empty object array
     */
    public static final Object[] EMPTY_ARRAY = new Object[0];

    /**
     * A utility random instance.
     */
    public static final Random RANDOM = getRandom();

    /**
     * The default date format
     */
    public static final SimpleDateFormat STD_DATE_FMT = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

    private Util() {}

    /**
     * Formats the current date by using the given <code>SimpleDateFormat</code>.
     * @param sdf
     * @return the formatted date.
     */
    public static String getFormattedDate(SimpleDateFormat sdf)
    {
        return sdf.format(new Date());
    }

    /**
     * Formats the current date by using the {@link #STD_DATE_FMT default} date format.
     * 
     * @return the formatted date.
     */
    public static String getFormattedDate()
    {
        return STD_DATE_FMT.format(new Date());
    }

    /**
     * Creates a new {@link java.util.Random} instance, preferably of the <code>SHA1PRNG</code>
     * algorithm. If this is not possible, then just a <code>new Random()</code> is returned.
     *
     * @return the created <code>Random</code>.
     */
    public static Random getRandom()
    {
        return getRandom("SHA1PRNG");
    }

    /**
     * Creates a new {@link java.util.Random} instance, of the <code>algorithm</code>
     * provided. If this is not possible, then just a <code>new Random()</code> is returned.
     *
     * @param  algorithm the desired algorithm.
     * @return the created <code>Random</code>.
     */
    public static Random getRandom(String algorithm)
    {
        try
        {
            SecureRandom r = SecureRandom.getInstance(algorithm);
            return r;
        }
        catch(NoSuchAlgorithmException e)
        {
            StdLog.error(e);
            Random r = new Random();
            r.setSeed(System.currentTimeMillis() ^ new Object().hashCode());
            return r;
        }
    }

    /**
     * Checks if the provided string is empty, after it is trimmed.
     *
     * In case the string may be <code>null</code>, then you should consider using
     * {@link #trimmedEmptySafe(String)}.
     * 
     * @param s the string to be checked.
     * @return true iff the following condition is true: <code>0 == s.trim().length()</code>
     */
    public static boolean trimmedEmpty(String s)
    {
        return 0 == s.trim().length();
    }

    /**
     * Returns the <code>localhost</code> name. If this cannot be determined, then
     * <code>localhost</code> is returned.
     *
     * @return the name of <code>localhost</code>.
     */
    public static String getLocalHost()
    {
        try
        {
            return InetAddress.getLocalHost().toString();
        }
        catch(Throwable e)
        {
            return "localhost";
        }
    }

    /**
     * Returns the string representation of the stacktrace provided by <code>throwable</code>.
     *
     * In effect, the following algorithm is applied:
     * <blockquote><pre>
     *   StringWriter w = new StringWriter();
     *   PrintWriter pw = new PrintWriter(w);
     *   throwable.printStackTrace(pw);
     *   return w.toString();
     * </pre></blockquote>
     * @param throwable
     * @return the stacktrace in string form.
     */
    public static String getStackTrace(Throwable throwable)
    {
        StringWriter w = new StringWriter();
        PrintWriter pw = new PrintWriter(w);
        throwable.printStackTrace(pw);

        return w.toString();
    }

    /**
     * Tries to convert <code>s</code> to a {@link URL} and ignores a possible {@link MalformedURLException}.
     *
     * @param s
     * @return the string as a <code>URL</code>
     */
    public static URL toURL(String s)
    {
        try
        {
            return new URL(s);
        }
        catch(MalformedURLException e)
        {
            StdLog.error(e);
            return null;
        }
    }

    /**
     * Tries to obtain <code>name</code> either as a system property or as an environment variable.
     *
     * First, it tries to obtain <code>name</code> by using <code>System.getProperty(name)</code>.
     * If this returns <code>null</code> and we are under Java 1.5+, then it uses <code>System.getenv(name)</code>.
     * 
     * @param name
     * @return The system property or environment variable with the given <code>name</code>.
     */
    public static String getPropertyOrEnv(String name)
    {
        String p = System.getProperty(name);
        if(null == p && JavaPlatform.version().compare(1, 5) >= 0)
        {
            p = System.getenv(name);
        }
        return p;
    }

    /**
     * Tries to obtain <code>name</code> either as an environment variable or as a system property.
     *
     * First, if we are under Java version < 1.5, it just returns <code>System.getProperty(name)</code>.
     * Otherwise, if we are under Java version >= 1.5, it first tries to obtain <code>name</code> by using <code>System.getenv(name)</code>.
     * If this returns <code>null</code> it returns <code>System.getProperty(name)</code>.
     *
     * @param name
     * @return The environment variable or system property with the given <code>name</code>.
     */
    public static String getEnvOrProperty(String name)
    {
        if(JavaPlatform.version().compare(1, 5) < 0)
        {
            return System.getProperty(name);
        }
        else
        {
            String p = System.getenv(name);
            if(null == p)
            {
                p = System.getProperty(name);
            }
            return p;
        }
    }

    /**
     * Returns <code>true</code> iff <code>s</code> can be parsed correctly by
     * <code>Integer.parseInt(s)</code>.
     * 
     * @param s
     * @return <code>true</code> iff <code>s</code> contains an integer.
     */
    public static boolean isInteger(String s)
    {
        try
        {
            Integer.parseInt(s);
        }
        catch(Exception e)
        {
            return false;
        }

        return true;
    }

    /**
     * Returns <code>true</code> iff <code>s</code> can be parsed correctly by
     * {@link Long#parseLong(String)}.
     *
     * @param s
     * @return <code>true</code> iff <code>s</code> contains a long.
     */
    public static boolean isLong(String s)
    {
        try
        {
            Long.parseLong(s);
        }
        catch(Exception e)
        {
            return false;
        }

        return true;
    }
    /**
     * Returns the environment variable of the given <code>name</code>.
     *
     * Since {@link System#getenv(String)} is deprecated for java 1.4, if we are
     * in a 1.4 VM then {@link System#getProperty(String)} is called, otherwise
     * {@link System#getenv(String)} is called.
     * 
     * @param name
     * @return the environment variable.
     */
    public static String getenv(String name)
    {
        if(JavaPlatform.version().compare(1, 5) < 0)
        {
            return System.getProperty(name);
        }
        else
        {
            return System.getenv(name);
        }
    }

    /**
     * A wrapper call around {@link System#getProperty(String)}.
     * 
     * @param name
     * @return the system property <code>name</code>. 
     */
    public static String getProperty(String name)
    {
        return System.getProperty(name);
    }

    /**
     * Checks if the two objects are equal by also taking into account that one of them
     * (or even both) may be null.
     *
     * This is useful in situations where we are not certain of the contents of the two objects
     * and we need to avoid <code>NullPointerException</code>s.
     * 
     * @param a
     * @param b
     * @return <code>true</code> iff <code>a</code> and <code>b</code> are equal.
     */
    public static boolean equalSafe(Object a, Object b)
    {
        if(null == a)
        {
            return null == b;
        }

        return a.equals(b);
    }

    /**
     * Checks if the provided string is empty.
     * A <code>null</code> string is also considered empty.
     *
     * @param s the string to be checked.
     * @return true iff the following condition is true: <code>s == null || 0 == s.length()</code>
     */
    public static boolean emptySafe(String s)
    {
        return s == null || 0 == s.length();
    }

    /**
     * Checks if the provided string is empty, after it is trimmed.
     * A <code>null</code> string is also considered empty.
     *
     * @param s the string to be checked.
     * @return true iff the following condition is true: <code>s == null || 0 == s.trim().length()</code>
     */
    public static boolean trimmedEmptySafe(String s)
    {
        return s == null || 0 == s.trim().length();
    }

    /**
     * Returns a "safe" value for <code>s</code>.
     *
     * If <code>s</code> is <code>null</code>, then the empty string <code>""</code> is returned.
     * @param s
     * @return <code>s</code> if it is not <code>null</code>, otherwise <code>""</code>.
     */
    public static String safe(String s)
    {
        return s == null ? "" : s;
    }

    /**
     * Returns a "safe" value for <code>array</code> of the given <code>componentType</code>.
     *
     * If <code>array</code> is <code>null</code>, then an empty <code>Object[]</code> array is returned.
     * @param array
     * @param componentType
     * @return <code>array</code> if it is not <code>null</code>, otherwise an empty <code>Object[]</code> array.
     */
    public static Object[] safe(Object[] array, Class componentType)
    {
        return null != array ? array : arrayFromItem(null, componentType);
    }
    /**
     * Returns a "safe" value for <code>set</code>.
     *
     * If <code>set</code> is <code>null</code>, then an empty <code>Set</code> is returned.
     * @param set
     * @return <code>set</code> if it is not <code>null</code>, otherwise an empty <code>Set</code>.
     */
    public static Set safe(Set set)
    {
        return null == set ? Collections.EMPTY_SET : set;
    }

    /**
     * Returns either an empty array or a one-item array containing the given <code>object</code>.
     *
     * If <code>object</code> is <code>null</code>, then an empty array is returned.
     * Otherwise, an array of type <code>object.getJavaClass()</code> and length 1 is created
     * and filled with <code>object</code>.
     * @param object
     * @return the computed array.
     */
    public static Object[] arrayFromItem(Object object)
    {
        if(null == object)
        {
            return new Object[0];
        }
        return arrayFromItem(object, object.getClass());
    }

    /**
     * Returns either an empty array or a one-item array containing the given <code>component</code>.
     *
     * The array component type is determined by <code>componentType</code>.
     *
     * If <code>component</code> is <code>null</code>, then an empty array is returned.
     *
     * <p>After succesfull return, the client can safely cast the return value to the
     * proper array type.
     * 
     * @param component
     * @param componentType
     * @return the computed array
     * @throws NullPointerException if <code>componentType</code> is <code>null</code>.
     */
    public static Object[] arrayFromItem(Object component, Class componentType)
    {
        int size = null == component ? 0 : 1;
        Object[] array = (Object[]) Array.newInstance(componentType, size);
        if(1 == size)
        {
            array[0] = component;
        }

        return array;
    }

    /**
     * Checks if the provided array is empty.
     * A <code>null</code> value for <code>array</code> is also considered as an empty array.
     *
     * @param array the array to be checked.
     * @return true iff the following condition is true: <code>null == array ? true : 0 == array.length</code>
     */
    public static boolean emptySafe(Object[] array)
    {
        return null == array ? true : 0 == array.length;
    }

    /**
     * Returns array "safe" value for <code>array</code>.
     *
     * If <code>array</code> is <code>null</code>, then an empty <code>String[]</code> array is returned.
     * @param array
     * @return <code>array</code> if it is not <code>null</code>, otherwise an empty <code>String[]</code> array.
     */
    public static String[] safe(String[] array)
    {
        return null == array ? new String[0] : array;
    }

    /**
     * Returns array "safe" value for <code>array</code>.
     *
     * If <code>array</code> is <code>null</code>, then an empty <code>File[]</code> array is returned.
     * @param array
     * @return <code>array</code> if it is not <code>null</code>, otherwise an empty <code>File[]</code> array.
     */
    public static File[] safe(File[] array)
    {
        return null == array ? new File[0] : array;
    }

    /**
     * Returns array "safe" value for <code>array</code>.
     *
     * If <code>array</code> is <code>null</code>, then an empty <code>ClassLoader[]</code> array is returned.
     * @param array
     * @return <code>array</code> if it is not <code>null</code>, otherwise an empty <code>ClassLoader[]</code> array.
     */
    public static ClassLoader[] safe(ClassLoader[] array)
    {
        return null == array ? new ClassLoader[0] : array;
    }

    /**
     * Checks if the provided <code>collection</code> is empty.
     * A <code>null</code> value for <code>collection</code> is also considered as an empty collection.
     *
     * @param collection the collection to be checked.
     * @return true iff it is safe to consider <code>collection</code> as empty.
     */
    public static boolean emptySafe(Collection collection)
    {
        return null == collection ? true : 0 == collection.size();
    }

    /**
     * Checks if the provided <code>list</code> is empty.
     * A <code>null</code> value for <code>list</code> is also considered as an empty list.
     * @param list
     * @return <code>true</code> iff it is safe to consider <code>list</code> as empty.
     */
    public static boolean emptySafe(IL list)
    {
        return null == list ? true : 0 == list.size();
    }
    
    /**
     * Returns a "safe" value of the <code>int</code> represented by <code>String s</code>.
     *
     * If <code>s</code> cannot be parsed correctly as an <code>int</code>, then the value
     * <code>safe</code> is returned.
     *
     * @param s the string representation of the <code>int</code>.
     * @param safe the safe value for the <code>int</code>.
     * @return a safe value for the <code>int</code> provided in <code>String s</code>.
     */
    public static int safe(String s, int safe)
    {
        try
        {
            return Integer.parseInt(s);
        }
        catch(Exception e)
        {
            return safe;
        }
    }

    /**
     * Returns a "safe" value of the <code>long</code> represented by <code>String s</code>.
     *
     * If <code>s</code> cannot be parsed correctly as an <code>long</code>, then the value
     * <code>safe</code> is returned.
     *
     * @param s the string representation of the <code>long</code>.
     * @param safe the safe value for the <code>long</code>.
     * @return a safe value for the <code>long</code> provided in <code>String s</code>.
     */
    public static long safe(String s, long safe)
    {
        try
        {
            return Long.parseLong(s);
        }
        catch(Exception e)
        {
            return safe;
        }
    }

    /**
     * Returns a "safe" value of the <code>double</code> represented by <code>String s</code>.
     *
     * If <code>s</code> cannot be parsed correctly as an <code>double</code>, then the value
     * <code>safe</code> is returned.
     *
     * @param s the string representation of the <code>double</code>.
     * @param safe the safe value for the <code>double</code>.
     * @return a safe value for the <code>double</code> provided in <code>String s</code>.
     */
    public static double safe(String s, double safe)
    {
        try
        {
            return Double.parseDouble(s);
        }
        catch(Exception e)
        {
            return safe;
        }
    }
}
