/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.util;

/**
 * Provides utility methods that reveal information regarding the call stack.
 * 
 * @author Christos KK Loverdos
 */
public final class CallUtil
{
    private CallUtil()
    {
    }


    /**
     * Returns class and method names information at a specific depth.
     * <p/>
     * Depth == 0 means the context of the caller of this method, so normally,
     * when issuing <code>UtilLib.getStackInfo(0)</code> from a method
     * <code>foo</code> of class <code>package.SomeClass</code>, a string
     * of the form <code>package.SomeClass::foo</code> will be returned.
     *
     */
    public static String getStackInfo(int depth)
    {
        depth += 1;
        try
        {
            throw new Exception();
        }
        catch(Exception e)
        {
            StackTraceElement[] st = e.getStackTrace();
            StringBuffer sb = new StringBuffer();
            if(st.length > depth)
            {
                sb.append(st[depth].getClassName());
                sb.append(".");
                sb.append(st[depth].getMethodName());
//				sb.append(".");
//				sb.append(st[depth].getLineNumber());
            }
            else
            {
                sb.append("::");
            }

            return sb.toString();
        }
    }

    public static StackTraceElement[] getStackTrace()
    {
        try
        {
            throw new Exception();
        }
        catch(Exception e)
        {
            StackTraceElement[] st = e.getStackTrace();
            StackTraceElement[] newSt = new StackTraceElement[st.length - 1];
            System.arraycopy(st, 1, newSt, 0, newSt.length);
            return newSt;
        }
    }
    
    public static StackTraceElement getStackTrace(int depth)
    {
        depth +=1;
        try
        {
            throw new Exception();
        }
        catch(Exception e)
        {
            StackTraceElement[] st = e.getStackTrace();
            if(st.length > depth)
            {
                return st[depth];
            }
            else
            {
                return null;
            }
        }
    }

    public static StackTraceElement getThisStackInfo()
    {
        return getStackTrace(1 + 0);
    }

    public static StackTraceElement getCallerStackInfo()
    {
        return getStackTrace(1 + 1);
    }

    public static StackTraceElement getCallerCallerStackInfo()
    {
        return getStackTrace(1 + 2);
    }

    public static StackTraceElement getCaller3StackInfo()
    {
        return getStackTrace(1 + 3);
    }

    public static String getStackShortInfo(int depth)
    {
        depth += 1;
        try
        {
            throw new Exception();
        }
        catch(Exception e)
        {
            StackTraceElement[] st = e.getStackTrace();
            StringBuffer sb = new StringBuffer();
            if(st.length > depth)
            {
                sb.append(ClassUtil.getShortClassName(st[depth].getClassName()));
                sb.append(".");
                sb.append(st[depth].getMethodName());
//				sb.append(".");
//				sb.append(st[depth].getLineNumber());
            }
            else
            {
                sb.append("::");
            }

            return sb.toString();
        }
    }

    /**
     * Get information for the method and class calling this method.
     * <p/>
     * <p/> This is equivalent to calling <code>getStackInfo(1 + 0)</code>.
     *
     */
    public static String getThisInfo()
    {
        return getStackInfo(1 + 0);
    }

    public static String getThisShortInfo()
    {
        return getStackShortInfo(1 + 0);
    }

    /**
     * Get information for the method and class calling the caller of this method.
     * <p/>
     * <p/> This is equivalent to calling <code>getStackInfo(1 + 1)</code>.
     *
     */
    public static String getCallerInfo()
    {
        return getStackInfo(1 + 1);
    }

    public static String getCallerShortInfo()
    {
        return getStackShortInfo(1 + 1);
    }

    /**
     * Get information for the method and class calling the caller of the caller of this method.
     * <p/>
     * <p/> This is equivalent to calling <code>getStackInfo(1 + 2)</code>.
     *
     */
    public static String getCallerCallerInfo()
    {
        return getStackInfo(1 + 2);
    }

    public static String getCallerCallerShortInfo()
    {
        return getStackShortInfo(1 + 2);
    }

    public static String getCaller3Info()
    {
        return getStackInfo(1 + 3);
    }

    public static String getCaller3ShortInfo()
    {
        return getStackShortInfo(1 + 3);
    }
}
