/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.tuple;

/**
 * Implements {@link org.ckkloverdos.tuple.IPair}.
 *
 * @author Christos KK Loverdos
 */
public class Pair extends TupleSkeleton implements IPair
{
    protected Object a;
    protected Object b;

    /**
     * Contsructs the pair from the two objects.
     * @param a
     * @param b
     */
    public Pair(Object a, Object b)
    {
        this.a = a;
        this.b = b;
    }

    /**
     * Returns the first element.
     * This is equivalent to {@link #get(int) get(0)}.
     */
    public Object getA()
    {
        return a;
    }

    /**
     * Returns the second element.
     * This is equivalent to {@link #get(int) get(1)}.
     */
    public Object getB()
    {
        return b;
    }

    /**
     * Returns the <code>n</code>th element of the tuple.
     * The first element is at index zero, the second element
     * is at index one.
     * @param n
     * @throws IndexOutOfBoundsException if the index <code>n</code>
     * is other than zero or one.
     */
    public Object get(int n)
    {
        switch(n)
        {
            case 0:
                return a;
            case 1:
                return b;
            default:
                throw new IndexOutOfBoundsException("Invalid index " + n + " for a pair.");
        }
    }

    /**
     * Returns the size of the pair, which is always two.
     */
    public int size()
    {
        return 2;
    }

    /**
     * Returns <code>true</code> iff the <code>n</code>th element
     * of the tuple is <code>null</code>.
     * The first element is at index zero, the second element
     * is at index one.
     * @param n
     * @throws IndexOutOfBoundsException if the index <code>n</code>
     * is other than zero or one.
     */
    public boolean isNull(int n)
    {
        switch(n)
        {
            case 0:
                return null == a;
            case 1:
                return null == b;
            default:
                throw new IndexOutOfBoundsException("Invalid index " + n + " for a pair.");
        }
    }

    /**
     * Always returns <code>false</code>, since a pair contains exactly two
     * elements, even if they are <code>null</code>.
     */
    public boolean isEmpty()
    {
        return false;
    }
}
