/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.time;

/**
 * A simple start-stop counter. Not-synchronized.
 * @author Christos KK Loverdos
 */
public class SimpleTimeCounter
{
    private long start;
    private long stop;

    /**
     * Creates a new instance and starts the counter immediately.
     */
    public SimpleTimeCounter()
    {
        this.start = System.currentTimeMillis();
        this.stop = -1L;
    }

    /**
     * Creates a new instance, optionally starting the counter
     * if <code>start</code> is <code>true</code>.
     * @param start
     */
    public SimpleTimeCounter(boolean start)
    {
        if(start)
        {
            this.start = System.currentTimeMillis();
        }
        else
        {
            this.start = -1L;
        }
        this.stop = -1L;
    }

    /**
     * Starts the counter.
     */
    public void start()
    {
        this.start = System.currentTimeMillis();
        this.stop = -1L;
    }

    /**
     * Stops the counter and returns the time difference between now and the previous
     * <code>start()</code> action.
     */
    public long stop()
    {
        this.stop = System.currentTimeMillis();
        return this.stop - this.start;
    }

    /**
     * Computes the same value as {@link #stop()}. In effect it successively stops
     * and starts the counter in one step.
     */
    public long restart()
    {
        long time = stop();
        start();
        return time;
    }

    /**
     * If the timer has not been started, returns zero, otherwise returns the time of the last
     * stop minus the time of the last start.
     * If the timer hasnot been stopped, then it returns the current time minus the
     * time of the last start.
     */
    public long millis()
    {
        if(-1L == start)
        {
            return 0;
        }

        long stop = -1L == this.stop ? System.currentTimeMillis() : this.stop;
        return stop - start;
    }

    public double seconds()
    {
        return (double) millis() / 1000.0D;
    }

    public String secondsInfo()
    {
        return seconds() + " sec.";
    }

    public String millisInfo()
    {
        return millis() + " ms.";
    }

    /**
     * Returns a string representation of the time passed, in the format
     * <code>[x days,] HH:MM:SS.LLL</code>, where <code>LLL</code> are the
     * milliseconds and the rest is obvious :-).
     */
    public String timeInfo()
    {
        return TimeUtil.timeInfo(millis());
    }

    public String toString()
    {
        return timeInfo();
    }

    public String unitsPerSec(int units, String unitsname)
    {
        return ((double) units / seconds() ) + " " + unitsname + "/sec"; 
    }

    public String unitsPerMillis(int units, String unitsname)
    {
        return ((double) units / millis()) + " " + unitsname + "/ms";
    }
}

