/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.string;

import org.ckkloverdos.log.StdLog;
import org.ckkloverdos.util.Util;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.io.UnsupportedEncodingException;
import java.io.StringReader;
import java.io.BufferedReader;
import java.io.IOException;
import java.util.regex.Pattern;
import java.util.Random;

/**
 * Utility, string-based methods.
 * @author Christos KK Loverdos
 */
public final class StringUtil
{
    public static Pattern DOT_PATTERN = Pattern.compile("\\.");

    /**
     * UTF-8
     */
    public static final String UTF8 = "UTF-8";

    /**
     * ""
     */
    public static final String EMPTY_STRING = "";

    /**
     * {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F'}
     */
    public static final char[] HEX_DIGITS =
            {
                    '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F',
            };

    private StringUtil() {}


    /**
     * Returns the hex representation of the input.
     * @param b
     */
    public static String toHex(byte[] b)
    {
        StringBuffer sb = new StringBuffer(2 * b.length);
        for(int i = 0; i < b.length; i++)
        {
            int high = (b[i] >>> 4) & 0x0F;
            int low = b[i] & 0x0F;

            sb.append(hexToAscii(high));
            sb.append(hexToAscii(low));
        }

        return sb.toString();
    }

    /**
     * Returns the hex representation of the input.
     * @param b
     */
    public static String toHex(byte b)
    {
        int high = (b >>> 4) & 0x0F;
        int low = b & 0x0F;

        StringBuffer sb = new StringBuffer(2);
        sb.append(hexToAscii(high));
        sb.append(hexToAscii(low));

        return sb.toString();
    }

    /**
     * Returns the <code>ASCII</code> character represententation of the
     * hexadecimal value <code>h</code>.
     * @param h
     */
    public static char hexToAscii(int h)
    {
        if(h >= 10 && h <= 15)
        {
            return (char) ('A' + (h - 10));
        }
        if((h >= 0) && (h <= 9))
        {
            return (char) ('0' + h);
        }
        return '0';
    }

    /**
     * Computes a message digest for the given <code>input</code>.
     * @param input
     */
    public static String messageDigest(String input)
    {
        return messageDigest(input, null, null);
    }

    /**
     * Computes a message digest for the given <code>input</code>,
     * by using the specified <code>algorithm</code>.
     * @param input
     * @param algorithm
     */
    public static String messageDigest(String input, String algorithm)
    {
        return messageDigest(input, algorithm, null);
    }

    /**
     * Computes a message digest for the given <code>input</code>,
     * by using the specified <code>algorithm</code>.
     * The <code>encoding</code> is used to obtain byte values from
     * the <code>input</code>, by using {@link String#getBytes(String)}.
     *
     * If the <code>algorithm</code> is <code>null</code>, then <code>SHA1</code> is used.
     * If the <code>encoding</code> is <code>null</code>, then {@link #UTF8}
     * is used.
     *
     * @param msg
     * @param algorithm
     * @param encoding
     */
    public static String messageDigest(String msg, String algorithm, String encoding)
    {
        if(null == algorithm)
        {
            algorithm = "SHA1";
        }
        if(null == encoding)
        {
            encoding = UTF8;
        }
        try
        {
            MessageDigest sha1 = MessageDigest.getInstance(algorithm);
            sha1.reset();
            sha1.update(msg.getBytes(encoding));
            return toHex(sha1.digest());
        }
        catch(NoSuchAlgorithmException e)
        {
            StdLog.warn("SHA1 is not implemented! Going to use cleartext password...");
            return msg;
        }
        catch(UnsupportedEncodingException e)
        {
            StdLog.warn("Encoding " + encoding + " is not supported???");
            StdLog.warn("Going to use cleartext password...");
            return msg;
        }
    }

    /**
     * Safely removes the <code>suffix</code> from <code>str</code>. This is performed
     * until <code>str</code> no more {@link String#endsWith(String) ends with}
     * <code>suffix</code>.
     * If any of the input is <code>null</code>, then a {@link Util#safe(String) safe}
     * value for <code>str</code> is returned.
     *
     * @param str
     * @param suffix
     */
    public static String safeRemoveSuffixRepeat(String str, String suffix)
    {
        if(null != suffix && null != str)
        {
            while(str.endsWith(suffix))
            {
                str = str.substring(0, str.length() - suffix.length());
            }

            str = new String(str); // GC-friendly
        }

        return Util.safe(str);
    }

    /**
     * Splits <code>str</code> according to the given <code>regularExpression</code> and
     * returns the <code>index</code>th item.
     * It returns <code>null</code> in case of an <code>ArrayIndexOutOfBoundsException</code>, <code>null</code>.
     *  
     * @param str
     * @param regularExpression
     * @param index
     */
    public static String getStringPart(String str, String regularExpression, int index)
    {
        try
        {
            return str.split(regularExpression)[index];
        }
        catch(ArrayIndexOutOfBoundsException e)
        {
            StdLog.error(e);
            return null;
        }
    }

    /**
     * Safely removes the <code>suffix</code> from <code>str</code>. This is performed
     * once, in contrast to {@link #safeRemoveSuffixRepeat(String, String)}.
     * If any of the input is <code>null</code>, then a {@link Util#safe(String) safe}
     * value for <code>str</code> is returned.
     * @param str
     * @param suffix
     */
    public static String safeRemoveSuffix(String str, String suffix)
    {
        if(null != suffix && null != str && str.endsWith(suffix))
        {
            str = new String(str.substring(0, str.length() - suffix.length()));
        }

        return Util.safe(str);
    }

    /**
     * GC-friendly version of {@link String#substring(int)}.
     * See <a href="http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4513622">this bug</a>
     * @param s
     * @param beginIndex
     */
    public static String substring(String s, int beginIndex)
    {
        return new String(s.substring(beginIndex));
    }

    /**
     * GC-friendly version of {@link String#substring(int, int)}.
     * See <a href="http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4513622">this bug</a>
     * @param s
     * @param beginIndex
     * @param endIndex
     */
    public static String substring(String s, int beginIndex, int endIndex)
    {
        return new String(s.substring(beginIndex, endIndex));
    }

    /**
     * Splits the input by dots.
     * @param s
     */
    public static String[] splitDots(String s)
    {
        return DOT_PATTERN.split(s);
    }

    /**
     * Generates a Unique ID in string form.
     */
    public static String getStringUID()
    {
        return getStringUID(Util.RANDOM);
    }

    /**
     * Generates a Unique ID in string form by using the provided random number
     * generator.
     * @param random
     */
    public static String getStringUID(Random random)
    {
        StringBuffer sb = new StringBuffer();
        sb.append(Util.getLocalHost());
        sb.append("-");
        sb.append(random.nextLong());
        sb.append("-");
        double d1 = random.nextDouble();
        double d2 = random.nextDouble();
        sb.append((d1 - d2) * (d1 - d2) + d1 * d2);

        String md = messageDigest(sb.toString());

        //StdLog.debug("getStringUID(\"" + sb + "\") = " + md);

        return md;
    }

    /**
     * Splits the provided <code>text</code> into lines and returns the
     * <code>n</code>th line. Line numbering starts from one.
     * @param text
     * @param n
     */
    public static String getTextLine(String text, int n)
    {
        if(n <= 0)
        {
            return null;
        }

        StringReader sr = new StringReader(text);
        BufferedReader br = new BufferedReader(sr);

        try
        {
            int _n = 1;
            for(String s = br.readLine(); null != s; s = br.readLine(), _n++)
            {
                if(_n == n)
                {
                    return s;
                }
            }
        }
        catch(IOException e)
        {
            StdLog.error(e);
        }

        return null;
    }
}
