/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.source;

import org.ckkloverdos.string.StringUtil;

import java.io.*;

/**
 * A {@link org.ckkloverdos.source.ISource} whose contents come directly from a string.
 *
 * The source is valid iff the underlying string is not <code>null</code>.
 * @author Christos KK Loverdos
 */
public class StringSource extends SourceSkeleton
{
    private String theString;

    public StringSource(String name, SourceType type, String theString)
    {
        super(name, type, null);
        this.theString = theString;
        setValid(null != theString);
    }

    public StringSource(String name, SourceType type, StringBuffer theString)
    {
        this(name, type, theString.toString());
    }

    public Reader getReader() throws UnsupportedEncodingException
    {
        return new StringReader(theString);
    }

    public Reader getReader(String encoding) throws IOException
    {
        return new StringReader(getString(encoding));
    }

    public String getString() throws IOException
    {
        return theString;
    }

    public String getString(String encoding) throws IOException
    {
        return new String(theString.getBytes(), encoding);
    }

    /**
     * Returns an input stream obtained from the UTF8 representation of
     * the underlying string.
     * @throws IOException
     */
    public InputStream getInputStream() throws IOException
    {
        return new ByteArrayInputStream(getString(StringUtil.UTF8).getBytes());
    }
}