/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.source;

/**
 * Type representation for sources.
 *
 * @author Christos KK Loverdos
 *
 * @see org.ckkloverdos.source.ISource
 */
public class SourceType
{
    private String name;
    private boolean textOverride;

    public SourceType(String name)
    {
        if(null == name)
        {
            throw new NullPointerException("Null name for SourceType");
        }
        this.name = name;
    }

    /**
     * Returns <code>true</code> iff the <code>other</code> type has the same name as
     * this one.
     * @param other
     */
    public boolean is(SourceType other)
    {
        return this.equals(other);
    }

    /**
     * Returns <code>true</code> iff the <code>other</code> type has the same
     * {@link #getContentType() Content Type} as this one.
     * @param other
     */
    public boolean isContentType(SourceType other)
	{
		return this.getContentType().equals(other.getContentType());
	}

    /**
     * Returns <code>true</code> iff <code>other</code> equals the
     * {@link #getContentType() Content Type} of this instance.
     *
     * @param other
     */
    public boolean isContentType(String other)
    {
        return this.getContentType().equals(other);
    }

    /**
     * Returns <code>true</code> iff the <code>other</code> type has the same
     * {@link #getContentSubtype() Content Subtype} as this one.
     *
     * @param other
     */
    public boolean isContentSubtype(SourceType other)
    {
        return this.getContentSubtype().equals(other.getContentSubtype());
    }

    /**
     * Returns <code>true</code> iff <code>other</code> equals the
     * {@link #getContentSubtype() Content Subtype} of this instance.
     *
     * @param other
     */
    public boolean isContentSubtype(String other)
    {
        return this.getContentSubtype().equals(other);
    }

    /**
     * Returns the name of this type.
     */
    public String getName()
	{
		return this.name;
	}

    /**
     * Returns the Content Type for this type. If the name of this type is of the form:
     * <pre>
     * content-type/content-subtype
     * </pre>
     * (notice the slash) then the string <code>content-type</code> is returned, otherwise the whole name
     * of this type is returned.
     */
    public String getContentType()
    {
        int pos = name.indexOf('/');
        if(-1 != pos)
        {
            return name.substring(0, pos);
        }
        return name;
    }

    /**
     * Returns the Content Subtype for this type. If the name of this type is of the form:
     * <pre>
     * content-type/content-subtype
     * </pre>
     * (notice the slash) then the string <code>content-subtype</code> is returned, otherwise the whole name
     * of this type is returned.
     */
    public String getContentSubtype()
    {
        int pos = name.indexOf('/');
        if(-1 != pos)
        {
            return name.substring(pos + 1);
        }
        return name;
    }

    /**
     * Returns <code>true</code> iff the Content Type of this instance is <code>"text"</code>. 
     */
    public boolean isText()
    {
        return textOverride || "text".equals(getContentType());
    }

    /**
     * Returns <code>true</code> iff <code>other</code> has the same name
     * as this type.
     * @param o
     */
    public boolean equals(Object o)
    {
        if(this == o)
        {
            return true;
        }
        if(o == null || getClass() != o.getClass())
        {
            return false;
        }

        SourceType that = (SourceType) o;

        if(!name.equals(that.name))
        {
            return false;
        }

        return true;
    }

    public int hashCode()
    {
        return name.hashCode();
    }

    /**
     * Returns <code>true</code> if {@link #setTextOverride()} has been called.
     */
    public boolean isTextOverride()
    {
        return textOverride;
    }

    /**
     * Make {@link #isText()} always return true, regardless of the Content Type.
     * @return this
     */
    public SourceType setTextOverride()
    {
        this.textOverride = true;
        return this;    
    }
}