/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.resource;

import org.ckkloverdos.util.Util;
import org.ckkloverdos.util.ClassUtil;
import org.ckkloverdos.log.StdLog;
import org.ckkloverdos.string.KVToString;

/**
 * Resource-related utilities.
 * 
 * @author Christos KK Loverdos
 */

public final class RcUtil
{
    private RcUtil()
    {
    }

    /**
     * Returns the provided resource if not <code>null</code>, otherwise it returns
     * {@link org.ckkloverdos.resource.EmptyResource#INSTANCE}.
     *
     * @param rc
     */
    public static IResource safe(IResource rc)
    {
        return null != rc ? rc : EmptyResource.INSTANCE;
    }

    /**
     * Normalizes the resource name, so that a resource context will use
     * the normalized name to resolve the resource.
     * All dots except the last one (which is considered as part of
     * the resource extension) are translated to slashes.
     * 
     * @param name
     */
    public static String normalize(String name)
    {
        String suffix = null;
        int dotPos = name.lastIndexOf('.');
        if(dotPos >= 0)
        {
            suffix = name.substring(dotPos);
            name = name.substring(0, dotPos);
        }
        name = name.replace('.', '/');
        if(dotPos >= 0)
        {
            name = name + suffix;
        }

        return name;
    }

    /**
     * Returns a file resource context from the <code>user.dir</code> property.
     */
    public static FileResourceContext getUserDirResourceContext()
    {
        return new FileResourceContext(Util.getProperty("user.dir"));
    }

    /**
     * Constructs a new name under the package of the given class.
     * @param name
     * @param c
     */
    public static String nameUnderPackage(String name, Class c)
    {
        return nameUnderPackage(name, c.getPackage());
    }

    /**
     * Constructs a new name under the given package.
     * @param name
     * @param p
     */
    public static String nameUnderPackage(String name, Package p)
    {
        return p.getName() + '.' + name;
    }

    /**
     * Returns a class loader resource context based on the {@link Thread#getContextClassLoader() context} class loader.
     */
    public static ClassLoaderResourceContext getThreadClassLoaderContext()
    {
        return new ClassLoaderResourceContext(ClassUtil.getContextClassLoader());
    }

    private static void resolve(IResourceContext ctx, String name)
    {
        StdLog.log("<****> " + name);
        IResource rc = ctx.getResource(name);
        StdLog.log("rc = " + rc);
    }

    public static void main(String[] args)
    {
        String fileName = nameUnderPackage("test.txt", RcUtil.class);

        FileResourceContext fileCtx = new FileResourceContext(".");
        StdLog.log(new KVToString("fileName", fileName));
        ClassLoaderResourceContext cpCtx = getThreadClassLoaderContext();
        CompositeResourceContext ctx = new CompositeResourceContext(
            fileCtx,
            cpCtx
        );

//        StdLog.debug("-------------------------------------------");

        resolve(ctx, "java.lang.String.class");
        resolve(ctx, "classes\\production\\ckkl-Core\\com\\ckkloverdos\\resource\\" + "test.txt");

    }
}
