/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.resource;

import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;

import java.net.URL;
import java.net.MalformedURLException;
import java.io.File;
import java.io.InputStream;
import java.io.IOException;
import java.io.FileInputStream;

/**
 * File-based resource.
 * 
 * @author Christos KK Loverdos
 */
public class FileResource implements IResource, IToStringAware
{
    private File file;

    /**
     * Constructs a new instance from the given file.
     * @param file
     */
    public FileResource(File file)
    {
        this.file = file;
    }

    /**
     * Constructs a new instance from the given file name.
     * @param name
     */
    public FileResource(String name)
    {
         this.file = new File(name);
    }

    /**
     * Returns <code>true</code> iff the resource exists.
     */
    public boolean exists()
    {
        return file.exists();
    }

    /**
     * Returns <code>true</code> iff the resource is a file.
     * This always returns <code>true</code> for a {@link org.ckkloverdos.resource.FileResource}.
     * You should consult {@link #exists()} to see if the file exists.
     */
    public boolean isFile()
    {
        return true;
    }

    /**
     * Returns <code>true</code> iff the resource is a directory.
     */
    public boolean isDirectory()
    {
        return file.isDirectory();
    }

    /**
     * Returns the file object for this resource.
     */
    public File getFile()
    {
        return file;
    }

    /**
     * Returns the file name of this resource.
     */
    public String getName()
    {
        return file.getName();
    }

    /**
     * Returns the URL representation of the resource file.
     * This is equivalent to <code>getFile().toURL()</code>.
     */
    public URL getURL()
    {
        try
        {
            return file.toURI().toURL();
        }
        catch(MalformedURLException e)
        {
            return null;
        }
    }

    /**
     * Returns a new {@link java.io.FileInputStream} constructed from the
     * resource file.
     * @throws IOException
     */
    public InputStream getInputStream() throws IOException
    {
        return new FileInputStream(file);
    }

    public String toString()
    {
        ToString ts = new ToString(this);
        toStringAware(ts);
        return ts.toString();
    }

    public void toStringAware(ToString ts)
    {
        ts.add(file);
    }
}
