/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.resource;

import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;

/**
 * A composite resource context which uses other context to resolve resources.
 * @author Christos KK Loverdos
 */
public class CompositeResourceContext extends ResourseContextSkeleton implements IToStringAware
{
    private IResourceContext[] contexts;

    /**
     * Constructs a new instance based on the provided contexts and parent.
     * @param contexts
     * @param parent
     */
    public CompositeResourceContext(IResourceContext[] contexts, IResourceContext parent)
    {
        super(parent);
        this.contexts = contexts;
    }

    /**
     * Constructs a new instance based on the provided contexts.
     * @param contexts
     */
    public CompositeResourceContext(IResourceContext[] contexts)
    {
        this(contexts, null);
    }

    /**
     * Constructs a new instance based on the provided contexts.
     * @param a
     * @param b
     */
    public CompositeResourceContext(IResourceContext a, IResourceContext b)
    {
        this(new IResourceContext[]{a, b});
    }

    /**
     * Constructs a new instance based on the provided contexts.
     * @param a
     * @param b
     * @param c
     */
    public CompositeResourceContext(IResourceContext a, IResourceContext b, IResourceContext c)
    {
        this(new IResourceContext[]{a, b, c});
    }

    /**
     * Constructs a new instance based on the provided contexts.
     * @param a
     * @param b
     * @param c
     * @param d
     */
    public CompositeResourceContext(IResourceContext a, IResourceContext b, IResourceContext c, IResourceContext d)
    {
        this(new IResourceContext[]{a, b, c, d});
    }

    /**
     * The contexts that make up this composite context are consulted for the
     * requested resource in the order they were specified in one of the constructors.
     * @param name
     */
    public IResource getLocalResource(String name)
    {
        for(int i = 0; i < contexts.length; i++)
        {
            IResourceContext context = contexts[i];
            IResource rc = context.getResource(name);
            if(rc.exists())
            {
                return rc;
            }
        }

        return EmptyResource.INSTANCE;
    }

    public String toString()
    {
        ToString ts = new ToString(this);
        toStringAware(ts);
        return ts.toString();
    }

    public void toStringAware(ToString ts)
    {
        ToString ts2 = ts.save();
        ts
            .add(contexts)
            .restore(ts2);
    }
}
