/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.resource;

import org.ckkloverdos.util.Assert;
import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;

import java.net.URL;

/**
 * Classloader-based resource context.
 * 
 * @author Christos KK Loverdos
 */
public class ClassLoaderResourceContext extends ResourseContextSkeleton implements IToStringAware
{
    private ClassLoader loader;

    /**
     * Constructs a new context that will use the provided class loader
     * to resolve resources.
     * @param loader
     */
    public ClassLoaderResourceContext(ClassLoader loader)
    {
        this(loader, null);
    }

    /**
     * Constructs a new context that will first use the provided class loader
     * to resolve resources and the parent context on failure.
     * @param loader
     * @param parent
     */
    public ClassLoaderResourceContext(ClassLoader loader, IResourceContext parent)
    {
        super(parent);
        Assert.notNull("Class loader cannot be null");

        this.loader = loader;
    }

    /**
     * Returns the class loader used to resolve resources.
     */
    public ClassLoader getClassLoader()
    {
        return loader;
    }

    /**
     * Returns the resource with the given name, using only the underlying
     * class loader to resolve it (the parent context is not consulted).
     * @param name
     * @return the resource if found or {@link org.ckkloverdos.resource.EmptyResource#INSTANCE} otherwise.
     */
    public IResource getLocalResource(String name)
    {
        name = normalize(name);
        URL url = loader.getResource(name);
        if(null == url)
        {
            return EmptyResource.INSTANCE;
        }
        return new ClassLoaderResource(url);
    }

    public String toString()
    {
        ToString ts = new ToString(this);
        toStringAware(ts);
        return ts.toString();
    }

    public void toStringAware(ToString ts)
    {
        ts.add(loader);
    }
}
