/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.log;

import org.ckkloverdos.string.KVToString;
import org.ckkloverdos.string.ToString;
import org.ckkloverdos.util.CallUtil;
import org.ckkloverdos.util.Util;
import org.ckkloverdos.util.ClassUtil;
import org.ckkloverdos.java.JavaPlatform;

/**
 * A simple logging class.
 *
 * This a very limited logging API but with many utility methods, usually not
 * found in more complete APIs.
 * 
 * @author Christos KK Loverdos
 */
public class StdLogger
{
    public static final String DEBUG = "DEBUG";
    public static final String INFO  = " INFO";
    public static final String WARN  = " WARN";
    public static final String ERROR = "ERROR";

    private boolean printDate;
    private String uid;
	private String where;
    private int stackDepthOverride;
    private int indent;
    private boolean withSite;

    public StdLogger()
    {
        this(null, null, true);
    }

    /**
     * Constructs a new instance for which all output will include the
     * provided <code>uid</code>. A typical use of the <code>uid</code>
     * is to characterise a unique client request in some application
     * or library.
     * @param uid
     */
    public StdLogger(String uid)
    {
        this(uid, true);
    }

    /**
     * Constructs a new instance for which all output will include the
     * provided <code>uid</code> and the date/time of the logging call.
     * A typical use of the <code>uid</code> is to characterise a unique
     * client request in some application or library.
     */
    public StdLogger(String uid, boolean printDate)
    {
        this(uid, null, printDate);
    }

    /**
     * Constructs a new instance for which all output will include the
     * provided <code>uid</code> and a description of <code>where</code>
     * the logging call is made.
     * <p/>
     * A typical use of the <code>uid</code> is to characterise a unique
     * client request in some application or library.
     * If <code>where</code> is null, then the position of the logging call
     * will be determined on-the-fly by inspecting the stack.
     * If <code>where</code> is the empty string, the no information will
     * be provided regarding the calling site.
     */
    public StdLogger(String uid, String where)
    {
        this(uid, where, true);
    }
    
    public StdLogger(String uid, String where, boolean printDate)
    {
        this(uid, where, printDate, 0);
    }

    StdLogger(String uid, String where, boolean printDate, int stackDepthOverride)
    {
        this(uid, where, printDate, stackDepthOverride, false);
    }

    private StdLogger(String uid, String where, boolean printDate, int stackDepthOverride, boolean withSite)
    {
        this.uid = uid;
        this.where = where;
        this.printDate = printDate;
        this.stackDepthOverride = 3 + stackDepthOverride;
        this.withSite = withSite;
    }

    private String getPrintString(String level, String msg)
    {
        String line = "";
        if(printDate)
        {
            line += Util.getFormattedDate();
        }
        if(null != level)
        {
            if(line.length() > 0)
            {
                line += " ";
            }
            line += level;
        }

        if(null != where)
        {
            if(0 != where.length())
            {
                if(line.length() > 0)
                {
                    line += " ";
                }

                String _where;
                if(withSite)
                {
                    _where = CallUtil.getStackShortInfo(stackDepthOverride) + "." + where;
                }
                else
                {
                    _where = where;
                }

                line += "[" + _where + "]";
            }
        }
        else
        {
            if(line.length() > 0)
            {
                line += " ";
            }
            line += "[" + CallUtil.getStackShortInfo(stackDepthOverride) + "]";
        }

        if(null != uid && uid.length() > 0)
        {
            if(line.length() > 0)
            {
                line += " ";
            }
            line += uid;
        }

        msg = String.valueOf(msg);

        if(line.length() > 0 && msg.length() > 0)
        {
            line += " ";
        }

        return line + indentedMsg(msg);
    }

    private void print(String level, String msg)
    {
        System.out.println(getPrintString(level, msg));
    }

    private void print(String level, ToString msg)
    {
        System.out.println(getPrintString(level, msg.toString()));
    }

    private void print(String level, Throwable e)
    {
        System.out.println(getPrintString(level, Util.getStackTrace(e)));
    }

    private void print(String level, String msg, Throwable e)
    {
        System.out.println(
            getPrintString(level,
                           msg +
                           " [" + ClassUtil.getShortClassName(e) + "]" +
                           (null == e.getMessage() ? "" : " " + e.getMessage()) + 
                           JavaPlatform.LINE_SEPARATOR + Util.getStackTrace(e)));
    }

    private String indentedMsg(String msg)
    {
        if(indent <= 0)
        {
            return msg;
        }
        StringBuffer sb = new StringBuffer();
        for(int i=0; i<indent; i++)
        {
            sb.append("  ");
        }
        sb.append(msg);
        return sb.toString();
    }

    /**
     * Constructs a new instance for with the description given by <code>where</code>
     * will be appended the the information of the calling site.
     *
     * <p/>
     * This is useful when we have logger in some method A, which calls another
     * method B, and we want the logs printed from method B to include the
     * information that it has come from method A.
     *
     * <p/>
     * If in method A we have a logger:
     * <pre>
     * StdLogger logA = new StdLogger();
     * </pre>
     * and a log from A is of the form:
     * <pre>
     * 2007/10/21 13:22:55 [SomeClass.A] hello world!
     * </pre>
     * then if a new logger is created and passed to B:
     * <pre>
     * StdLogger logB = logA.forSite("B");
     * B(logB, ....)
     * </pre>
     * the output in B will look like:
     * <pre>
     * 2007/10/21 13:22:55 [SomeClass.A.B] hello world!
     * </pre>
     * Notice the call site descriptions provided in the brackets,
     * <code>[SomeClass.A]</code> and <code>[SomeClass.A.B]</code>.
     *
     * <p/>
     * In the future, this info will be provided dynamically from the call stack.
     * @param where
     */
    public StdLogger forSite(String where)
    {
        if(null != this.where && 0 != this.where.length())
        {
            where = this.where + "." + where;
        }
        return new StdLogger(uid, where, printDate, 0, true);
    }

    public StdLogger indent()
    {
        indent++;
        return this;
    }

    public StdLogger unindent()
    {
        indent--;
        return this;
    }
    
    public void log(String msg)
    {
        print(null, msg);
    }

    public void log(boolean b)
    {
        print(null, String.valueOf(b));
    }

    public void log(String key, Object val)
    {
        print(null, new KVToString(key, val));
    }

    public void log(String msg, String key, Object val)
    {
        print(null, new KVToString(msg + key, val));
    }

    public void log(String key, Object val, String k2, Object v2)
    {
        print(null, new KVToString(key, val, k2, v2));
    }

    public void log(String msg, String key, Object val, String k2, Object v2)
    {
        print(null, new KVToString(msg + key, val, k2, v2));
    }

    public void log(String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(null, new KVToString(key, val, k2, v2, k3, v3));
    }

    public void log(String msg, String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(null, new KVToString(msg + key, val, k2, v2, k3, v3));
    }

    public void log(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(null, new KVToString(key, val, k2, v2, k3, v3, k4, v4));
    }

    public void log(String msg, String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(null, new KVToString(msg + key, val, k2, v2, k3, v3, k4, v4));
    }

    public void log(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4, String k5, Object v5)
    {
        print(null, new KVToString(key, val, k2, v2, k3, v3, k4, v4, k5, v5));
    }

    public void log(String msg, String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4, String k5, Object v5)
    {
        print(null, new KVToString(msg + key, val, k2, v2, k3, v3, k4, v4, k5, v5));
    }

    public void log(ToString msg)
    {
        print(null, msg);
    }

    public void log(Throwable msg)
    {
        print(null, msg);
    }

    public void log(Object msg)
    {
        print(null, String.valueOf(msg));
    }

    public void debug(String msg)
    {
        print(DEBUG, msg);
    }

    public void debug(ToString msg)
    {
        print(DEBUG, msg);
    }

    public void debug(Throwable e)
    {
        print(DEBUG, e);
    }

    public void debug(String msg, Throwable e)
    {
        print(DEBUG, msg, e);
    }

    public void debug(ToString msg, Throwable e)
    {
        print(DEBUG, msg.toString(), e);
    }

    public void debug(Object o, Throwable e)
    {
        print(DEBUG, String.valueOf(o), e);
    }


    public void debug(Object msg)
    {
        print(DEBUG, String.valueOf(msg));
    }

    //////////////////////////////////////
    public void debug(String key, Object val)
    {
        print(DEBUG, new KVToString(key, val));
    }

    public void debug(String msg, String key, Object val)
    {
        print(DEBUG, new KVToString(msg + key, val));
    }

    public void debug(String key, Object val, String k2, Object v2)
    {
        print(DEBUG, new KVToString(key, val, k2, v2));
    }

    public void debug(String msg, String key, Object val, String k2, Object v2)
    {
        print(DEBUG, new KVToString(msg + key, val, k2, v2));
    }

    public void debug(String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(DEBUG, new KVToString(key, val, k2, v2, k3, v3));
    }

    public void debug(String msg, String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(DEBUG, new KVToString(msg + key, val, k2, v2, k3, v3));
    }

    public void debug(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(DEBUG, new KVToString(key, val, k2, v2, k3, v3, k4, v4));
    }

    public void debug(String msg, String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(DEBUG, new KVToString(msg + key, val, k2, v2, k3, v3, k4, v4));
    }

    public void debug(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4, String k5, Object v5)
    {
        print(DEBUG, new KVToString(key, val, k2, v2, k3, v3, k4, v4, k5, v5));
    }
    //////////////////////////////////////

    public void info(ToString msg)
    {
        print(INFO, msg);
    }

    public void info(Throwable e)
    {
        print(INFO, e);
    }

    public void info(String msg, Throwable e)
    {
        print(INFO, msg, e);
    }

    public void info(ToString msg, Throwable e)
    {
        print(INFO, msg.toString(), e);
    }

    public void info(Object o, Throwable e)
    {
        print(INFO, String.valueOf(o), e);
    }

    public void info(Object msg)
    {
        print(INFO, String.valueOf(msg));
    }

    //////////////////////////////////////
    public void info(String key, Object val)
    {
        print(INFO, new KVToString(key, val));
    }

    public void info(String msg, String key, Object val)
    {
        print(INFO, new KVToString(msg + key, val));
    }

    public void info(String key, Object val, String k2, Object v2)
    {
        print(INFO, new KVToString(key, val, k2, v2));
    }

    public void info(String msg, String key, Object val, String k2, Object v2)
    {
        print(INFO, new KVToString(msg + key, val, k2, v2));
    }

    public void info(String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(INFO, new KVToString(key, val, k2, v2, k3, v3));
    }

    public void info(String msg, String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(INFO, new KVToString(msg + key, val, k2, v2, k3, v3));
    }

    public void info(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(INFO, new KVToString(key, val, k2, v2, k3, v3, k4, v4));
    }

    public void info(String msg, String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(INFO, new KVToString(msg + key, val, k2, v2, k3, v3, k4, v4));
    }

    public void info(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4, String k5, Object v5)
    {
        print(INFO, new KVToString(key, val, k2, v2, k3, v3, k4, v4, k5, v5));
    }
    //////////////////////////////////////
    
    public void warn(String msg)
    {
        print(WARN, msg);
    }

    public void warn(ToString msg)
    {
        print(WARN, msg);
    }

    public void warn(Throwable e)
    {
        print(WARN, e);
    }

    public void warn(String msg, Throwable e)
    {
        print(WARN, msg, e);
    }

    public void warn(ToString msg, Throwable e)
    {
        print(WARN, msg.toString(), e);
    }

    public void warn(Object o, Throwable e)
    {
        print(WARN, String.valueOf(o), e);
    }

    public void warn(Object msg)
    {
        print(WARN, String.valueOf(msg));
    }

    //////////////////////////////////////
    public void warn(String key, Object val)
    {
        print(WARN, new KVToString(key, val));
    }

    public void warn(String msg, String key, Object val)
    {
        print(WARN, new KVToString(msg + key, val));
    }

    public void warn(String key, Object val, String k2, Object v2)
    {
        print(WARN, new KVToString(key, val, k2, v2));
    }

    public void warn(String msg, String key, Object val, String k2, Object v2)
    {
        print(WARN, new KVToString(msg + key, val, k2, v2));
    }

    public void warn(String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(WARN, new KVToString(key, val, k2, v2, k3, v3));
    }

    public void warn(String msg, String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(WARN, new KVToString(msg + key, val, k2, v2, k3, v3));
    }

    public void warn(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(WARN, new KVToString(key, val, k2, v2, k3, v3, k4, v4));
    }

    public void warn(String msg, String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(WARN, new KVToString(msg + key, val, k2, v2, k3, v3, k4, v4));
    }

    public void warn(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4, String k5, Object v5)
    {
        print(WARN, new KVToString(key, val, k2, v2, k3, v3, k4, v4, k5, v5));
    }
    //////////////////////////////////////

    public void error(String msg)
    {
        print(ERROR, msg);
    }

    public void error(ToString msg)
    {
        print(ERROR, msg);
    }

    public void error(Throwable e)
    {
        print(ERROR, e);
    }

    public void error(String msg, Throwable e)
    {
        print(ERROR, msg, e);
    }

    public void error(ToString msg, Throwable e)
    {
        print(ERROR, msg.toString(), e);
    }

    public void error(Object o, Throwable e)
    {
        print(ERROR, String.valueOf(o) , e);
    }

    public void error(Object msg)
    {
        print(ERROR, String.valueOf(msg));
    }

    //////////////////////////////////////
    public void error(String key, Object val)
    {
        print(ERROR, new KVToString(key, val));
    }

    public void error(String msg, String key, Object val)
    {
        print(ERROR, new KVToString(msg + key, val));
    }

    public void error(String key, Object val, String k2, Object v2)
    {
        print(ERROR, new KVToString(key, val, k2, v2));
    }

    public void error(String msg, String key, Object val, String k2, Object v2)
    {
        print(ERROR, new KVToString(msg + key, val, k2, v2));
    }

    public void error(String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(ERROR, new KVToString(key, val, k2, v2, k3, v3));
    }

    public void error(String msg, String key, Object val, String k2, Object v2, String k3, Object v3)
    {
        print(ERROR, new KVToString(msg + key, val, k2, v2, k3, v3));
    }

    public void error(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(ERROR, new KVToString(key, val, k2, v2, k3, v3, k4, v4));
    }

    public void error(String msg, String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4)
    {
        print(ERROR, new KVToString(msg + key, val, k2, v2, k3, v3, k4, v4));
    }

    public void error(String key, Object val, String k2, Object v2, String k3, Object v3, String k4, Object v4, String k5, Object v5)
    {
        print(ERROR, new KVToString(key, val, k2, v2, k3, v3, k4, v4, k5, v5));
    }
    //////////////////////////////////////

    public void setUID(String uid)
	{
		this.uid = uid;
	}

    public void setWhere(String where)
    {
        this.where = where;
    }

    public String getWhere()
    {
        return where;
    }

    public String getUID()
	{
		return uid;
	}

    public void enter()
    {
        print(null, ">>>ENTER");
    }

    public void exit()
    {
        print(null, "<<<EXIT");
    }
}
