/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.java;

import org.ckkloverdos.util.Util;

/**
 * Utility class that wraps the <code>java.version</code> system property and provides
 * comparison methods.
 * 
 * @author Christos KK Loverdos
 */
public final class JavaVersion
{
    private int major;
    private int minor;
    private String patch="";
    private double patchNumber;

    public static final JavaVersion JAVA_VERSION = new JavaVersion();

    public JavaVersion()
    {
        String version = System.getProperty("java.version");
        String[] parts = version.split("\\.");

        major = Util.safe(parts[0], 0);
        if(parts.length > 1)
        {
            minor = Util.safe(parts[1], 0);

            if(parts.length > 2)
            {
                patch = parts[2];
                patchNumber = getPatch(patch);
            }
        }
    }

    public int getMajor()
    {
        return major;
    }

    public int getMinor()
    {
        return minor;
    }

    private double getPatch(String patch)
    {
        return Double.parseDouble(patch.replace('_', '.'));
    }

    public int compare(int major, int minor, String patch)
    {
        int dmajor = JAVA_VERSION.major - major;
        if(0 != dmajor)
        {
            return dmajor;
        }
        int dminor = JAVA_VERSION.minor - minor;
        if(0 != dminor)
        {
            return dminor;
        }
        double dpatchN = JAVA_VERSION.patchNumber - getPatch(patch);
        return dpatchN < 0.0 ? -1 : dpatchN > 0.0 ? 1 : 0;
    }

    public int compare(int major, int minor)
    {
        int dmajor = JAVA_VERSION.major - major;
        if(0 != dmajor)
        {
            return dmajor;
        }
        return JAVA_VERSION.minor - minor;
    }

    public String toString()
    {
        return major + "." + minor + "." + patchNumber;
    }

//    public static void main(String[] args)
//    {
//        System.out.println(JavaVersion.JAVA_VERSION);
//        System.out.println(JavaVersion.JAVA_VERSION.compare(1, 4, "2.08"));
//    }
}
