/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.java;

import org.ckkloverdos.util.Util;

/**
 * Provides information about the underlying Java platform.
 * 
 * @author Christos KK Loverdos
 */
public final class JavaPlatform
{
    public static String LINE_SEPARATOR = System.getProperty("line.separator");
    public static final String EOL = getProperty("line.separator");
    public static final String FILE_SEP = getProperty("file.separator");
    public static final String PATH_SEP = getProperty("path.separator");
    public static final JavaVersion JAVA_VERSION = JavaVersion.JAVA_VERSION;

    public static final String OS_NAME = getProperty("os.name");
    public static final String OS_VERSION = getProperty("os.version");
    
    private static final String OS_NAME_LOWER = OS_NAME.toLowerCase();

    private JavaPlatform() {}

    public static String getProperty(String name)
    {
        return System.getProperty(name);
    }

    public static boolean isWindows()
    {
        return OS_NAME.indexOf("Windows") != -1;
    }

    public static boolean isSolaris()
    {
        return OS_NAME.indexOf("Solaris") != -1 ||
               OS_NAME.indexOf("SunOS") != -1;
    }

    public static boolean isLinux()
    {
        return OS_NAME.indexOf("Linux") != -1;
    }

    public static boolean isUnix()
    {
        return !isWindows() && !isMac();
    }

    public static boolean isMac()
    {
        return OS_NAME.startsWith("Mac OS");
    }

    public static boolean isMacOSX()
    {
        return isMac() && OS_NAME.endsWith("X");
    }

    public static JavaVersion version()
    {
        return JAVA_VERSION;
    }

    public static boolean isAtLeastJdk14()
    {
        return JAVA_VERSION.compare(1, 4) >= 0;
    }

    public static boolean isAtLeastJdk15()
    {
        return JAVA_VERSION.compare(1, 5) >= 0;
    }

    public static boolean isAtLeastJdk16()
    {
        return JAVA_VERSION.compare(1, 6) >= 0;
    }

    public static String path()
    {
        String path = Util.getEnvOrProperty("PATH");
        if(null == path && isWindows())
        {
            path =  getProperty("java.library.path");
        }
        else if(null == path)
        {
            path = "";
        }
        return path;
    }

    public static String[] splitpath()
    {
        return path().split(PATH_SEP);
    }
}
