/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.collection;

import org.ckkloverdos.filter.IFilter;
import org.ckkloverdos.log.StdLog;

import java.lang.reflect.Array;
import java.util.*;

/**
 * Utility methods for collections, either as input or output.
 * 
 * @author Christos KK Loverdos
 */

public final class CollectionUtil
{
    private CollectionUtil()
    {
    }

    /**
     * Returns the given <code>array</code> as a <code>List</code>.
     * If <code>array</code> is null, then an empty list is returned.
     * @param array
     */
    public static List toList(Object[] array)
    {
        return null == array ? new ArrayList() : Arrays.asList(array);
    }

    /**
     * Constructs a list containing the elements of the enumeration.
     * @param en
     */
    public static List toList(Enumeration en)
    {
        List list = new ArrayList();
        while(en.hasMoreElements())
        {
            list.add(en.nextElement());
        }
        return list;
    }
    
    /**
     * Returns the given <code>collection</code> as an array whose components have
     * the type <code>componentType</code>.
     * If the <code>collection</code> is null, then an empty array is returned.
     * It is assumed that the <code>collection</code> elements are of the given
     * <code>componentType</code>.
     *
     * @param collection
     * @param componentType
     */
    public static Object toArray(Collection collection, Class componentType)
    {
        if(null == collection)
        {
            return Array.newInstance(componentType, 0);
        }

        Object[] array = (Object[]) Array.newInstance(componentType, collection.size());
        int i = 0;
        for(Iterator cItems = collection.iterator(); cItems.hasNext(); i++)
        {
            array[i] = cItems.next();
        }

        return array;
    }

    /**
     * Returns the given <code>collection</code> as an array of strings.
     * This is essentially equivalent to <code>(String[]) asArray(collection, String.class)</code>. 
     * @param collection
     */
    public static String[] toStringArray(Collection collection)
    {
        return (String[]) toArray(collection, String.class);
    }

    public static Set union(Set a, Set b)
    {
        return new L(a).union(new L(b)).toSet();
    }

    public static Set intersect(Set a, Set b)
    {
        return new L(a).intersect(new L(b)).toSet();
    }

    public static Set minus(Set a, Set b)
    {
        return new L(a).minus(new L(b)).toSet();
    }

    public static IL filterKeys(Map map, IFilter filter, Object hints)
    {
        return new L(map.keySet()).filter(filter, hints);
    }

    /**
     * Contructs a new map keeping only the keys of the provided list.
     * @param map
     * @param keys
     */
    public static Map filterMap(Map map, IL keys)
    {
        Map r = newLike(map);
        for(Iterator iter = keys.iterator(); iter.hasNext();)
        {
            Object key = iter.next();
            if(map.containsKey(key))
            {
                r.put(key, map.get(key));
            }
        }

        return r;
    }

    public static Collection newCollectionLike(IL il)
    {
        if(il.isCollection())
        {
            Collection collection = il.toCollection();
            try
            {
                return (Collection) collection.getClass().newInstance();
            }
            catch(Exception e)
            {
            }
        }

        return new ArrayList();
    }

    public static List newListLike(IL il)
    {
        if(il.isList())
        {
            List list = il.toList();
            try
            {
                return (List) list.getClass().newInstance();
            }
            catch(Exception e)
            {
            }
        }

        return new ArrayList();
    }

    public static Set newSetLike(IL il)
    {
        if(il.isSet())
        {
            Set set = il.toSet();
            try
            {
                return (Set) set.getClass().newInstance();
            }
            catch(Exception e)
            {
            }
        }
        return new HashSet();
    }

    public static Map newLike(Map map)
    {
        try
        {
            return (Map) map.getClass().newInstance();
        }
        catch(Exception e)
        {
            StdLog.error(e);
        }

        return new HashMap();
    }

    /**
     * Contructs a new map with keys those accepted by <code>filter</code>.
     * The second parameter <code>hints</code> is passed to <code>filter</code>.
     * @param map
     * @param filter
     * @param hints
     */
    public static Map filterMap(Map map, IFilter filter, Object hints)
    {
        Map filtered = newLike(map);
        Iterator iKeys = map.keySet().iterator();
        while(iKeys.hasNext())
        {
            Object o = iKeys.next();
            if(filter.accept(o, hints))
            {
                filtered.put(o, map.get(o));
            }
        }

        return filtered;
    }

    public static void main(String[] args)
    {
        L a = new L(new L(new String[]{"one", "two", "ten"}).toSet());
        L b = new L(new L(new String[]{"one", "three"}).toSet());
        L c = new L(new String[]{"nine", "eleven", "twenty"});

        System.out.println("a = " + a);
        System.out.println("b = " + b);
        
        System.out.println("union        = " + a.union(b));
        System.out.println("sorted union = " + a.union(b).sort().toCollection());
        System.out.println("intersection = " + a.intersect(b));
        System.out.println("difference   = " + a.minus(b));

        System.out.println("a.get(0) = " + a.get(0));
        System.out.println("a.get(2) = " + a.get(2));

        System.out.println("");
        System.out.println(a.toArray());
        System.out.println(a.toArray(String[].class));
        System.out.println(c.toArray(String[].class));
    }
}
