/*
 * Copyright 1999-2007 Christos KK Loverdos.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.ckkloverdos.collection;

import org.ckkloverdos.string.IToStringAware;
import org.ckkloverdos.string.ToString;
import org.ckkloverdos.util.Util;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

/**
 * Utility class to uniformly handle collections and object arrays.
 * @author Christos KK Loverdos
 */
public final class CollectionProxy implements IToStringAware// implements Collection
{
    private ICollectionProxyImpl impl;

    public CollectionProxy(Collection col)
    {
        impl = new CProxy(col);
    }

    public CollectionProxy(Object[] array)
    {
        impl = new AProxy(array);
    }

    public CollectionProxy(L l)
    {
        if(l.isCollection())
        {
            impl = new CProxy(l.toCollection());
        }
        else
        {
            impl = new AProxy(l.toArray());
        }
    }

    public Iterator iterator()
    {
        return impl.iterator();
    }

    public CollectionProxy addAll(IL l)
    {
        impl.addAll(l);
        return this;
    }
    
    public CollectionProxy addAll(Collection c)
    {
        impl.addAll(c);
        return this;
    }

    public CollectionProxy addAll(Object[] array)
    {
        impl.addAll(array);
        return this;
    }

    public CollectionProxy add(Object o)
    {
        impl.add(o);
        return this;
    }

    public CollectionProxy remove(Object o)
    {
        impl.remove(o);
        return this;
    }

    public CollectionProxy clear(Object o)
    {
        impl.clear(o);
        return this;
    }

    public boolean isCollection()
    {
        return impl.isCollection();
    }

    public Collection getCollection()
    {
        return impl.getCollection();
    }

    public Object[] getArray()
    {
        return impl.getArray();
    }
    
    public L toL()
    {
        return impl.toL();
    }

    public static CollectionProxy newLike(IL l)
    {
        if(l.isCollection())
        {
            return new CollectionProxy(CollectionUtil.newCollectionLike(l));
        }
        else
        {
            return new CollectionProxy(new Object[]{});
        }
    }

    public void toStringAware(ToString ts)
    {
        impl.toStringAware(ts);
    }

    public String toString()
    {
        return impl.toString();
    }

    public void beginMassiveAdd()
    {
    }

    public void endMassiveAdd()
    {
    }

    private static interface ICollectionProxyImpl extends IToStringAware
    {
        Iterator iterator();
        void addAll(IL l);
        void addAll(Collection c);
        void addAll(Object[] array);
        void add(Object o);
        void remove(Object o);
        void clear(Object o);
        boolean isCollection();
        Collection getCollection();
        Object[] getArray();
        L toL();
        void beginMassiveAdd();
        void endMassiveAdd();
    }

    private static final class CProxy implements ICollectionProxyImpl
    {
        private Collection col;

        private CProxy(Collection col)
        {
            this.col = col;
        }

        public Iterator iterator()
        {
            return col.iterator();
        }

        public void addAll(IL l)
        {
            Iterator iter = l.iterator();
            while(iter.hasNext())
            {
                col.add(iter.next());
            }
        }

        public void addAll(Collection c)
        {
            col.addAll(c);
        }

        public void addAll(Object[] array)
        {
            if(null != array)
            {
                for(int i = 0; i < array.length; i++)
                {
                    col.add(array[i]);
                }
            }
        }

        public void add(Object o)
        {
            col.add(o);
        }

        public void remove(Object o)
        {
            col.remove(o);
        }

        public void clear(Object o)
        {
            while(col.remove(o))
            {
            }
        }

        public boolean isCollection()
        {
            return true;
        }

        public Collection getCollection()
        {
            return col;
        }

        public Object[] getArray()
        {
            return null;
        }

        public L toL()
        {
            return new L(col);
        }

        public void beginMassiveAdd()
        {
        }

        public void endMassiveAdd()
        {
        }

        public void toStringAware(ToString ts)
        {
            ts.add(col);
        }

        public String toString()
        {
            ToString ts = new ToString();
            toStringAware(ts);
            return ts.toString();
        }
    }

    private static final class AProxy implements ICollectionProxyImpl
    {
        private Object[] array;
        private ArrayList massiveAddHelper;

        private AProxy(Object[] array)
        {
            this.array = array;
        }

        public Iterator iterator()
        {
            return new ArrayIterator(array);
        }

        public void addAll(IL l)
        {
            Iterator iter = l.iterator();
            beginMassiveAdd();
            while(iter.hasNext())
            {
                massiveAddHelper.add(iter.next());
            }
            endMassiveAdd();
        }

        public void addAll(Collection c)
        {
            beginMassiveAdd();
            for(Iterator iterator = c.iterator(); iterator.hasNext();)
            {
                massiveAddHelper.add(iterator.next());
            }
            endMassiveAdd();
        }

        public void addAll(Object[] array)
        {
            beginMassiveAdd();
            for(int i = 0; i < array.length; i++)
            {
                massiveAddHelper.add(array[i]);
            }
            endMassiveAdd();
        }

        private boolean inMassiveAdd() {return null != massiveAddHelper;}

        private Object newArray(int length)
        {
            Class componentType = array.getClass().getComponentType();
            Object newArray = Array.newInstance(componentType, length);
            return newArray;
        }

        public void add(Object o)
        {
            if(inMassiveAdd())
            {
                massiveAddHelper.add(o);
            }
            else
            {
                Object newArray = newArray(array.length + 1);
                System.arraycopy(array, 0, newArray, 0, array.length);
                Array.set(newArray, array.length, o);
                array = (Object[]) newArray;
            }
        }

        public void remove(Object o)
        {
            if(array.length > 0)
            {
                int delIndex = -1;
                for(int i = 0; i < array.length; i++)
                {
                    if(Util.equalSafe(array[i], o))
                    {
                        delIndex = i;
                        break;
                    }
                }
                if(delIndex >= 0)
                {
                    Object newArray = newArray(array.length - 1);
                    System.arraycopy(array, 0, newArray, 0, delIndex);
                    System.arraycopy(array, delIndex + 1, newArray, delIndex, array.length - delIndex - 1);
                    array = (Object[]) newArray;
                }
            }
        }

        public void clear(Object o)
        {
            if(array.length > 0)
            {
                IntArray delPositions = new IntArray();
                for(int i = 0; i < array.length; i++)
                {
                    if(Util.equalSafe(array[i], o))
                    {
                        delPositions.add(i);
                    }
                }

                int howmanyToDelete = delPositions.length();
                if(howmanyToDelete > 0)
                {
                    int newLength = array.length - howmanyToDelete;
                    Object newArray = newArray(newLength);

                    int[] dpArray = delPositions.array();
                    int dpIndex = 0;
                    int iToRead = 0;
                    int iToWrite = 0;
                    int wrote = 0;
                    for(; iToRead < array.length && dpIndex < howmanyToDelete; iToRead++)
                    {
                        int dPos = dpArray[dpIndex];
                        if(iToRead < dPos)
                        {
                            Array.set(newArray, iToWrite++, array[iToRead]);
                            wrote++;
                        }
                        else if(iToRead == dPos)
                        {
                            dpIndex++;
                        }
                    }

                    // copy the rest
                    if(iToRead < array.length)
                    {
                        int remainingToWrite = newLength - wrote;
                        System.arraycopy(array, iToRead, newArray, iToWrite, remainingToWrite);
                    }

                    array = (Object[]) newArray;
                }
            }
        }

        public boolean isCollection()
        {
            return false;
        }

        public Collection getCollection()
        {
            return null;
        }

        public Object[] getArray()
        {
            return array;
        }

        public L toL()
        {
            return new L(array);
        }

        public void beginMassiveAdd()
        {
            massiveAddHelper = new ArrayList();
        }

        public void endMassiveAdd()
        {
            if(null != massiveAddHelper)
            {
                int total = array.length + massiveAddHelper.size();
                Object[] massive = massiveAddHelper.toArray();
                Object newArray = newArray(total);
                System.arraycopy(array, 0, newArray, 0, array.length);
                System.arraycopy(massive, 0, newArray, array.length, massive.length);
                array = (Object[]) newArray;
                massiveAddHelper = null;
            }
        }

        public void toStringAware(ToString ts)
        {
            ts.add(array);
        }

        public String toString()
        {
            ToString ts = new ToString();
            toStringAware(ts);
            return ts.toString();
        }
    }
}
