/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* userbase.c - Userbase handling */

/* See end of read_user() for user time/day limits. Default is:

    if (nodenum == 3)
        time_left = 3600;
    else
        time_left = user.TodayMinutes > 60 ? 0 : 60-user.TodayMinutes;

    (unlimited time for node #3, 60min/day for others)
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "os.h"
#include "memory.h"
#include "output.h"
#include "timeslic.h"
#include "files.h"
#include "mareas.h"
#include "logfile.h"
#include "bbs_func.h"
#include "concord.h"
#include "userbase.h"
#include "crc32.h"
#include "ask_str.h"
#include "language.h"
#include "config.h"

typedef struct
{
    void *next;
    int user_in;
    char name[36];
}
GROUP_REC;


/* Local variables */
static int Fuidx = -1, Fudat = -1;
static GROUP_REC *first_group;
static GROUP_REC *group_rec;


/* Global variables */
unsigned long user_num;
time_t checked_newfiles;
int time_left;

USER_SUBRECORDS usrsub;
USER_REC user;

char current_charset[20];
char current_emulation[20];

int open_userbase(char *fname)
{
    char tmp[256];

    if (Fuidx != -1) return 1; /* Already opened */

    first_group = NULL;

    sprintf(tmp, "%s"SSLASH"%s.dat", data_path, fname);
    Fudat = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYNO);
    if (Fudat == -1)
    {
        /* Userbase not found, create new */
        Fudat = FileCreate(tmp, CREATE_MODE);
        if (Fudat == -1) return 0;
        FileMode(Fudat,O_BINARY);

        /* Create .idx file */
        sprintf(tmp,"%s.idx",fname);
        Fuidx = FileCreate(tmp, CREATE_MODE);
        if (Fuidx == -1)
        {
            /* Could not create userbase index file */
            FileClose(Fudat);
            return 0;
        }
        FileMode(Fuidx,O_BINARY);
    }
    else
    {
        /* Open .idx file */
        sprintf(tmp, "%s"SSLASH"%s.idx", data_path, fname);
        Fuidx = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYNO);
        if (Fuidx == -1)
        {
            /* Not found! */
            printf("Userbase indexes should be created.\n");
            return 0;
        }
    }
    return 1;
}

void close_userbase(void)
{
    close_groups();

    FileClose(Fudat); Fudat = -1;
    FileClose(Fuidx); Fuidx = -1;
}

unsigned long users(void)
{
    return FileSeek(Fuidx, 0, SEEK_END) / sizeof(USER_IDX);
}

unsigned long scan_user(char *name)
{
    unsigned long crc,msgnum;
    USER_IDX uidx;

    /* Calculate CRC32 of user name */
    crc = up_crc32(name);

    /* Scan user indexes */
    FileSeek(Fuidx,0,SEEK_SET);
    msgnum = 1;
    while (FileRead(Fuidx,&uidx,sizeof(uidx)) == sizeof(uidx))
    {
        if (uidx.UserCRC == crc) return msgnum; /* Found! */
        msgnum++;
    }

    /* User not found */
    return 0;
}

static USER_LASTREAD_REC *ulast;
static unsigned ulast_readed;

int read_user(unsigned long usernum, int justread)
{
    USER_IDX uidx;
    USER_SUBFIELD usub;
    time_t _tim;
    struct tm *tim,*ltim,ntim;
    PROTO_REC *proto;
    LANG_REC *lang;
    PACKER_REC *packer;

    unsigned SubReaded;

    if (!FileLock(Fuidx))
    {
        write_log("read_user() : Can't lock userbase.");
        return 0;
    }

    /* Read user index */
    FileSeek(Fuidx,(usernum-1)*sizeof(uidx),SEEK_SET);
    if (FileRead(Fuidx,&uidx,sizeof(uidx)) != sizeof(uidx))
    {
        FileUnlock(Fuidx);
        return 0;
    }

    /* Read user data */
    FileSeek(Fudat,uidx.SeekPos,SEEK_SET);
    if (FileRead(Fudat,&user,sizeof(user)) != sizeof(user))
    {
        FileUnlock(Fuidx);
        return 0;
    }

    /* Read subfields */
    memset(&usrsub,0,sizeof(usrsub));
    SubReaded = 0; ulast = NULL;
    while (SubReaded < user.SubfieldLen)
    {
        /* Read subfield ID */
        if (FileRead(Fudat,&usub,sizeof(usub)) != sizeof(usub))
        {
            FileUnlock(Fuidx);
            return 0;
        }

        /* Read subfield data */
        switch (usub.ID)
        {
            case USERSUB_NAME:
                usrsub.name[FileRead(Fudat,usrsub.name,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_ALIAS:
                usrsub.alias[FileRead(Fudat,usrsub.alias,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_PASSWORD:
                usrsub.password[FileRead(Fudat,usrsub.password,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_ADDRESS1:
                usrsub.address1[FileRead(Fudat,usrsub.address1,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_ADDRESS2:
                usrsub.address2[FileRead(Fudat,usrsub.address2,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_ADDRESS3:
                usrsub.address3[FileRead(Fudat,usrsub.address3,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_ADDRESS4:
                usrsub.address4[FileRead(Fudat,usrsub.address4,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_ADDRESS5:
                usrsub.address5[FileRead(Fudat,usrsub.address5,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_SYSOP_CMNT:
                usrsub.sysop_cmnt[FileRead(Fudat,usrsub.sysop_cmnt,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_VOICE:
                usrsub.voice[FileRead(Fudat,usrsub.voice,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_DATA:
                usrsub.data[FileRead(Fudat,usrsub.data,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_CITY:
                usrsub.city[FileRead(Fudat,usrsub.city,(unsigned) usub.Len)] = '\0';
                break;
            case USERSUB_LASTREAD:
                if (!justread)
                {
                    ulast = (USER_LASTREAD_REC *) _malloc((int) usub.Len);
                    if (ulast != NULL)
                    {
                        ulast_readed = (unsigned) (usub.Len/sizeof(USER_LASTREAD_REC));
                        if (FileRead(Fudat,ulast,(unsigned) usub.Len) != (int) usub.Len) _free(ulast);
                    }
                }
                break;
            default:
                FileSeek(Fudat,usub.Len,SEEK_CUR);
                break;
        }
        SubReaded += sizeof(usub)+usub.Len;
    }

    if (!justread)
    {
        _tim = time(NULL);
        tim = localtime(&_tim); memcpy(&ntim,tim,sizeof(ntim));
        ltim = localtime((time_t *) &user.LastTime[0]);

        if (ntim.tm_year != ltim->tm_year || ntim.tm_yday != ltim->tm_yday)
        {
            memmove(&user.LastTime[1],&user.LastTime[0],sizeof(user.LastTime)-sizeof(user.LastTime[0]));
            user.LastTime[0] = _tim;

            user.TodayCalls = 0;
            user.TodayMinutes = 0;
            user.TodayPages = 0;
            user.TodayUploadBytes = 0;
            user.TodayUploadFiles = 0;
            user.TodayDownloadBytes = 0;
            user.TodayDownloadFiles = 0;
            user.TodayMsgPkts = 0;
        }
        if (user.LastFileChk == 0) user.LastFileChk = _tim-7L*3600L*24L;
        if (user.ScreenWidth < 70) user.ScreenWidth = 80;

        if (nodenum == 3)
            time_left = 3600;
        else
            time_left = (unsigned) (user.TodayMinutes > 60 ? 0 : 60-user.TodayMinutes);

        proto = firstproto;
        while (proto != NULL)
        {
            if (toupper(user.Protocol) == toupper(proto->key))
            {
                break;
            }
            proto = proto->next;
        }
        if (proto == NULL)
            memset(&user_proto, 0, sizeof(user_proto));
        else
            memcpy(&user_proto, proto, sizeof(user_proto));

        lang = firstlang;
        while (lang != NULL)
        {
            if (toupper(user.Language) == toupper(lang->key))
            {
                break;
            }
            lang = lang->next;
        }
        if (lang == NULL)
            lang = firstlang;
        else
        {
            close_language();
            if (!read_language(lang->fname))
                read_language(firstlang->fname);
        }
        memcpy(&user_lang, lang, sizeof(user_lang));

        packer = firstpacker;
        while (packer != NULL)
        {
            if (toupper(user.Packer) == toupper(packer->key))
            {
                break;
            }
            packer = packer->next;
        }
        if (packer == NULL)
            memset(&user_packer, 0, sizeof(user_packer));
        else
            memcpy(&user_packer, packer, sizeof(user_packer));

        // Nm pit siirt muualle ja kunnon configit nille ym!
        if (user.CharSet == 'I')
            strcpy(current_charset,"IBM-PC");
        else if (user.CharSet == 'L')
            strcpy(current_charset,"Latin-1");
        else current_charset[0] = '\0';

        if (user.Emulation == USER_EMULATION_ASCII)
            strcpy(current_emulation,"ASCII");
        else if (user.Emulation == USER_EMULATION_ANSI)
            strcpy(current_emulation,"Dummy-ANSI");
        else if (user.Emulation == USER_EMULATION_ANSI_X364)
            strcpy(current_emulation,"ANSI X3.64");
        else current_emulation[0] = '\0';
    }

    FileUnlock(Fuidx);
    return 1;
}

void read_lastread(void)
{
    unsigned long rec,num,CRC,oldarea;

    if (ulast == NULL) return;

    /* Read lastread pointers */
    oldarea = current_marea;
    for (rec=1; rec<=msg_areas; rec++)
    {
        read_marea_record(rec);
        CRC = up_crc32(marea.name);

        marea.lr_ptr = 0;

        /* Search right area */
        for (num=0; num<ulast_readed; num++)
        {
            if (CRC == ulast[num].AreaCRC)
            {
                /* Area found */
                update_pointers(ulast[num].AreaPtr);
                if (ulast[num].Selected) marea_rec->flags |= MAREA_FLAG_SELECTED;
            }
        }
    }
    read_marea_record(oldarea);

    _free(ulast); ulast = NULL;
}

unsigned long write_new_user(void)
{
    typedef struct
    {
        char *name;
        char type;
    }
    TMP_USER_REC;

    TMP_USER_REC subfields[] =
    {
        { usrsub.name, USERSUB_NAME },
        { usrsub.alias, USERSUB_ALIAS },
        { usrsub.password, USERSUB_PASSWORD },
        { usrsub.address1, USERSUB_ADDRESS1 },
        { usrsub.address2, USERSUB_ADDRESS2 },
        { usrsub.address3, USERSUB_ADDRESS3 },
        { usrsub.address4, USERSUB_ADDRESS4 },
        { usrsub.address5, USERSUB_ADDRESS5 },
        { usrsub.sysop_cmnt, USERSUB_SYSOP_CMNT },
        { usrsub.voice, USERSUB_VOICE },
        { usrsub.data, USERSUB_DATA },
        { usrsub.city, USERSUB_CITY },
        { usrsub.birthday, USERSUB_BIRTHDAY },
        { NULL, 0 }
    };

    int sub,slen;
    unsigned long pos;
    unsigned sublen;
    char *data;

    USER_IDX uidx;
    USER_SUBFIELD usub;


    if (!FileLock(Fuidx))
    {
        write_log("write_new_user() : Can't lock userbase.");
        output("\r\nCan't lock userbase! User not saved in userbase!!\r\n");
        return 0;
    }

    /* Make user index record */
    uidx.UserCRC = up_crc32(usrsub.name);
    user.UserCRC = uidx.UserCRC;
    uidx.SeekPos = FileSeek(Fudat,0,SEEK_END);

    /* Calculate how much space subfields take */
    sub = 0; sublen = 0;
    while (subfields[sub].name != NULL)
    {
        slen = strlen(subfields[sub].name);
        if (slen > 0) sublen += sizeof(USER_SUBFIELD)+slen;
        sub++;
    }

    data = (char *) _malloc(sublen);
    if (data == NULL)
    {
        /* Not enough memory */
        FileUnlock(Fuidx);
        write_log("write_new_user() : Not enough memory (%d bytes)", sublen);
        output("\r\nNot enough memory! User not saved in userbase!!\r\n");
        return 0;
    }

    /* Move subfields to 'data' */
    sub = 0; sublen = 0;
    while (subfields[sub].name != NULL)
    {
        usub.ID = subfields[sub].type;
        usub.Len = (char) strlen(subfields[sub].name);
        if (usub.Len > 0)
        {
            memcpy(data+sublen,&usub,sizeof(usub));
            sublen += sizeof(USER_SUBFIELD);
            memcpy(data+sublen,subfields[sub].name,(unsigned) usub.Len);
            sublen += usub.Len;
        }
        sub++;
    }

    user.SubfieldLen = sublen;

    /* Write user record */
    if (FileWrite(Fudat,&user,sizeof(user)) != sizeof(user))
    {
        FileUnlock(Fuidx);
        write_log("write_new_user() : Can't write to userbase .dat file");
        output("\r\nCan't write to userbase! User not saved in userbase!!\r\n");
        return 0;
    }

    /* Write subfields */
    if (FileWrite(Fudat,data,sublen) != (int) sublen)
    {
        FileUnlock(Fuidx);
        _free(data);
        return 0;
    }

    _free(data);

    /* Write user index record */
    pos = FileSeek(Fuidx,0,SEEK_END)/sizeof(USER_IDX)+1;
    if (FileWrite(Fuidx,&uidx,sizeof(uidx)) != sizeof(uidx))
    {
        FileUnlock(Fuidx);
        write_log("write_new_user() : Can't write to userbase .idx file");
        output("\r\nCan't write to userbase! User not saved in userbase!!\r\n");
        return 0;
    }

    FileUnlock(Fuidx);
    return pos;
}

unsigned long write_user(void)
{
    typedef struct
    {
        char *name;
        char type;
    }
    TMP_USER_REC;

    TMP_USER_REC subfields[] =
    {
        { usrsub.name, USERSUB_NAME },
        { usrsub.alias, USERSUB_ALIAS },
        { usrsub.password, USERSUB_PASSWORD },
        { usrsub.address1, USERSUB_ADDRESS1 },
        { usrsub.address2, USERSUB_ADDRESS2 },
        { usrsub.address3, USERSUB_ADDRESS3 },
        { usrsub.address4, USERSUB_ADDRESS4 },
        { usrsub.address5, USERSUB_ADDRESS5 },
        { usrsub.sysop_cmnt, USERSUB_SYSOP_CMNT },
        { usrsub.voice, USERSUB_VOICE },
        { usrsub.data, USERSUB_DATA },
        { usrsub.city, USERSUB_CITY },
        { usrsub.birthday, USERSUB_BIRTHDAY },
        { NULL, 0 }
    };

    int sub,slen;
    unsigned sublen;
    unsigned long pos,usernum,oldarea;
    char *data;

    USER_IDX uidx;
    USER_REC udat;
    USER_SUBFIELD usub;
    USER_LASTREAD_REC ulast;


    if (!FileLock(Fuidx))
    {
        write_log("write_user() : Can't lock userbase");
        return 0;
    }

    usernum = scan_user(usrsub.name);

    /* Calculate how much space subfields take */
    sub = 0; sublen = 0;
    while (subfields[sub].name != NULL)
    {
        slen = strlen(subfields[sub].name);
        if (slen > 0) sublen += sizeof(USER_SUBFIELD)+slen;
        sub++;
    }
    sublen += sizeof(USER_LASTREAD_REC)*msg_areas+sizeof(USER_SUBFIELD);

    data = (char *) _malloc(sublen);
    if (data == NULL)
    {
        /* Not enough memory */
        FileUnlock(Fuidx);
        write_log("write_user() : Not enough memory");
        return 0;
    }

    /* Read old index */
    if (usernum != 0)
    {
        FileSeek(Fuidx,(usernum-1)*sizeof(USER_IDX),SEEK_SET);
        FileRead(Fuidx,&uidx,sizeof(uidx));
    }

    if (sublen != user.SubfieldLen || usernum == 0)
    {
        /* Subfields don't take as much space as before, so delete old fields
           and create new ones to end of the file */

        /* Read user data */
        FileSeek(Fudat,uidx.SeekPos,SEEK_SET);
        FileRead(Fudat,&udat,sizeof(udat));

        /* Delete user */
        udat.Attrib1 |= USER_ATTRIB_DELETED;

        /* Write user data */
        FileSeek(Fudat,uidx.SeekPos,SEEK_SET);
        FileWrite(Fudat,&udat,sizeof(udat));

        /* Make user index record */
        uidx.UserCRC = up_crc32(usrsub.name);
        uidx.SeekPos = FileSeek(Fudat,0,SEEK_END);
    }
    else
    {
        /* Make user index record */
        uidx.UserCRC = up_crc32(usrsub.name);
        FileSeek(Fudat,uidx.SeekPos,SEEK_SET);
    }
    user.UserCRC = uidx.UserCRC;
    user.PassCRC = 0xffffffff;

    if (usernum == 0)
        FileSeek(Fuidx,0,SEEK_END);
    else
        FileSeek(Fuidx,(usernum-1)*sizeof(USER_IDX),SEEK_SET);

    /* Move subfields to 'data' */
    sub = 0; sublen = 0;
    while (subfields[sub].name != NULL)
    {
        usub.ID = subfields[sub].type;
        usub.Len = (char) strlen(subfields[sub].name);
        if (usub.Len > 0)
        {
            memcpy(data+sublen,&usub,sizeof(usub));
            sublen += sizeof(USER_SUBFIELD);
            memcpy(data+sublen,subfields[sub].name,(unsigned) usub.Len);
            sublen += usub.Len;
        }
        sub++;
    }

    usub.ID = USERSUB_LASTREAD;
    usub.Len = msg_areas*sizeof(USER_LASTREAD_REC);
    memcpy(data+sublen,&usub,sizeof(usub));
    sublen += sizeof(USER_SUBFIELD);

    /* Make lastread pointers */
    oldarea = current_marea;
    for (pos=1; pos<=msg_areas; pos++)
    {
        read_marea_record(pos);

        ulast.AreaCRC = up_crc32(marea.name);
        ulast.AreaPtr = marea.lr_ptr;
        ulast.Selected = (marea.flags & MAREA_FLAG_SELECTED) > 0;
        memcpy(data+sublen,&ulast,sizeof(ulast));
        sublen += sizeof(ulast);
    }
    read_marea_record(oldarea);

    user.SubfieldLen = sublen;

    /* Write user record */
    if (FileWrite(Fudat,&user,sizeof(user)) != sizeof(user))
    {
        FileUnlock(Fuidx);
        return 0;
    }

    /* Write subfields */
    if (FileWrite(Fudat,data,sublen) != (int) sublen)
    {
        FileUnlock(Fuidx);
        _free(data);
        return 0;
    }

    _free(data);

    /* Write user index record */
    pos = FileSeek(Fuidx,0,SEEK_CUR)/sizeof(uidx)+1;
    if (FileWrite(Fuidx,&uidx,sizeof(uidx)) != sizeof(uidx))
    {
        FileUnlock(Fuidx);
        return 0;
    }

    FileUnlock(Fuidx);
    return pos;
}

#define copy_pascal_str(pstr, str) memcpy(str,pstr+1,pstr[0]); str[(int) pstr[0]] = '\0'

/* Search user from Concord's userbase */
unsigned long scan_concord_user(char *name)
{
    int Fidx,Fdat;
    unsigned long CRC;
    CONCORD_NAMEIDX_REC nidx;
    CONCORD_USER_REC urec;
    char tmp[256];

    /* Try to upen Concord's userinfo.idx file */
    sprintf(tmp, "%suserinfo.idx", concord_base);
    Fidx = FileOpen(tmp, O_RDONLY | O_BINARY, SH_DENYNO);
    if (Fidx == -1) return 0;

    /* Search user */
    CRC = up_crc32(name);
    while (FileRead(Fidx,&nidx,sizeof(nidx)) == sizeof(nidx))
    {
        if (nidx.NameCRC == CRC)
        {
            /* User found */
            sprintf(tmp, "%suserinfo.dat", concord_base);
            Fdat = FileOpen(tmp, O_RDONLY | O_BINARY, SH_DENYNO);
            if (Fdat == -1) break;

            /* Read user record */
            FileSeek(Fdat,(FileSeek(Fidx,0,SEEK_CUR)/sizeof(nidx)-1)*sizeof(CONCORD_USER_REC)+sizeof(CONCORD_USER_HEADER),SEEK_SET);
            FileRead(Fdat,&urec,sizeof(urec));

            /* Convert to YAK record */
            user.UserCRC = CRC;
            user.ScreenLen = urec.ScreenLen;
            user.ScreenWidth = 80;
            user.FirstTime = urec.FirstTime;
            user.TotalCalls = urec.TimesCalled;
            user.TotalMinutes = urec.TotalMinutes;
            user.TotalPages = (unsigned short) urec.Pages;
            user.UploadBytes = urec.UpK*1024;
            user.UploadFiles = urec.UpTimes;
            user.DownloadBytes = urec.DownK*1024;
            user.DownloadFiles = urec.DownTimes;

            copy_pascal_str(urec.Name,usrsub.name);
            copy_pascal_str(urec.Alias,usrsub.alias);
            copy_pascal_str(urec.City,usrsub.city);
            copy_pascal_str(urec.Voice,usrsub.voice);
            copy_pascal_str(urec.Data,usrsub.data);
            copy_pascal_str(urec.Password,usrsub.password);

            sprintf(usrsub.birthday,"%02d.%02d.%02d",
                    urec.Birthday.Day,urec.Birthday.Month,urec.Birthday.Year);

            /* Close files */
            FileClose(Fdat);
            FileClose(Fidx);

            /* Write user record */
            return write_new_user();
        }
    }

    FileClose(Fidx);
    return 0;
}

int check_flags(char *scan, char *str, int all)
{
    int n1, n2;

    for (n1 = 0; scan[n1] != '\0'; n1++)
    {
        for (n2 = 0; str[n2] != '\0'; n2++)
        {
            if (toupper(scan[n1]) == toupper(str[n2]))
            {
                if (!all) return 0; /* !all = abort if one flag is found */
                break;
            }
        }
        if (all && str[n2] == '\0') return 0; /* all = abort if flag not found */
    }

    return 1;
}

/* Is user in group? */
int in_group(char *group)
{
    char tmp[256], flags[20], *strp, *strp2;
    int len;

    /* Group ALL */
    if (stricmp(group,"ALL") == 0) return 1;

    /* Group user.name */
    strcpy(tmp, group); strp = tmp;
    while (*strp != '\0')
    {
        if (*strp == '.') *strp = ' ';
        strp++;
    }
    if (stricmp(tmp, usrsub.name) == 0) return 1;

    strp = strchr(group, '@');
    if (strp != NULL)
    {
        /* @ found, probably checking msgarea rights.. */
        memcpy(tmp, group, (int) (strp-group)); tmp[(int) (strp-group)] = '\0';
        strp++; strcpy(flags, strp);
    }
    else
    {
        strcpy(tmp, group);
    }

    /* Check from groups */
    group_rec = first_group;
    while (group_rec != NULL)
    {
        /* Group name ends also with @ .. */
        strp2 = strchr(group_rec->name, '@');
        if (strp2 == NULL)
            len = strlen(group_rec->name);
        else
        {
            len = (int) (strp2-group_rec->name);
            strp2++;
        }

        if (strnicmp(group_rec->name, group, len) == 0)
        {
            if (group_rec->user_in)
            {
                /* User in group, great. */
                if (strp == NULL || strp2 == NULL) return 1;

                /* But are flags also ok? */
                if (check_flags(strp, strp2, 1)) return 1;
            }
            else if (strp != NULL && strp2 != NULL)
            {
                /* Not in group .. but @ used .. check flags */
                if (!check_flags(strp, strp2, 0)) return 0;
            }
        }
        group_rec = (GROUP_REC *) group_rec->next;
    }

    return strp != NULL;
}

#define RET_NO          0
#define RET_YES         1
#define RET_NEVER       2
#define RET_OF_COURSE   3

int user_in_groups(char *groupsp)
{
    char *strp,*next,*groups,tmp[256];
    int ok,rev;

    strcpy(tmp,groupsp);
    groups = tmp;

    ok = 0;
    while (groups != NULL)
    {
        next = strchr(groups, ',');
        if (next != NULL)
        {
            *next++ = '\0';
            while (*next == ' ') next++;
        }

        /* Remove spaces from start of line */
        while (*groups == ' ' || *groups == 9) groups++;

        /* Remove trailing spaces */
        strp = groups;
        while (*strp != '\0') strp++;
        while (strp > groups && (*(strp-1) == ' ' || *(strp-1) == 9)) strp--;
        *strp = '\0';

        if (groups[0] == '!')
        {
            groups++;
            rev = 1;
        }
        else
        {
            rev = 0;
        }

        if (stricmp(groups,usrsub.name) == 0) return (!rev)+2;

        if (in_group(groups))
        {
            if (rev) return RET_NEVER;
            ok = 1;
        }

        groups = next;
    }

    return ok;
}

void read_groups(void)
{
    int Fgrp;
    unsigned long lines;
    int newline,already_belongs;
    char str[257],*strp,group_name[36];
    GROUP_REC *tmprec,*last_rec;

    if (first_group != NULL) close_groups();

    first_group = NULL;
    group_rec = NULL;
    last_rec = NULL;

    /* Open group file */
    sprintf(str, "%s"SSLASH"groups.bbs", data_path);
    Fgrp = FileOpen(str, O_RDONLY | O_BINARY, SH_DENYNO);
    if (Fgrp == -1)
    {
        printf("User groups file '%s' not found!\n", str);
        write_log("User groups file '%s' not found", str);
        return;
    }

    /* Read group file */
    lines = 0; newline = 1;
    already_belongs = 0;
    group_name[0] = '\0';
    while (_fgets(str,sizeof(str),Fgrp) != NULL)
    {
        /* Remove trailing spaces */
        strp = str;
        while (*strp) strp++;
        while (strp > str && *strp == ' ') strp--;
        *(strp+1) = '\0';

        /* Comment line or empty line */
        if (str[0] == '\0' || str[0] == ';') continue;

        lines++;
        if (newline)
        {
            /* Get group name */
            strp = strchr(str,':');
            if (strp == NULL)
            {
                printf("Error in \"groups.bbs\" line %lu\n",lines);
                continue;
            }
            *strp = '\0'; strp++;
            if (strlen(str) > 30)
            {
                printf("Too long group name in \"groups.bbs\" line %lu\n",lines);
                group_name[0] = '\0';
            }
            else
            {
                strcpy(group_name,str);
            }
            already_belongs = 0;
        }
        else
        {
            strp = str;
        }

        if (!already_belongs)
        {
            /* Check if user belongs to group */
            if (group_name[0] != '\0')
            {
                if (newline)
                {
                    tmprec = (GROUP_REC *) _malloc(sizeof(GROUP_REC));
                    if (last_rec != NULL) last_rec->next = tmprec;
                    group_rec = tmprec;
                    last_rec = group_rec;
                    if (first_group == NULL) first_group = tmprec;

                    last_rec->next = NULL;
                    last_rec->user_in = 0;
                    strcpy(last_rec->name,group_name);
                }
                else
                {
                    newline = 1;
                }

                already_belongs = user_in_groups(strp);

                if (already_belongs == RET_NEVER)
                {
                    last_rec->user_in = 0;
                    already_belongs = 1;
                }
                else
                {
                    last_rec->user_in |= already_belongs & 1;
                    already_belongs = already_belongs == RET_OF_COURSE;
                }
            }
        }

        newline = strp[strlen(strp)-1] != ',';
    }

    FileClose(Fgrp);
}

void close_groups(void)
{
    /* Release memory used by usergroups */
    while (first_group != NULL)
    {
        group_rec = (GROUP_REC *) first_group->next;
        _free(first_group);
        first_group = group_rec;
    }
}

int pack_userbase(void)
{
    int Fuidx, Fudat, Ftdat, Ftidx;
    USER_IDX uidx;
    USER_REC user;
    char *buf, tmp[256], tmp2[256];
    int memgot;

    /* Make backups */
    sprintf(tmp, "%s"SSLASH"userbase.dat", data_path);
    sprintf(tmp2, "%s"SSLASH"userbase.da~", data_path);
    if (!copyfile(tmp,tmp2))
    {
        printf("Can't copy %s to %s\n", tmp, tmp2);
        return 0;
    }
    sprintf(tmp, "%s"SSLASH"userbase.idx", data_path);
    sprintf(tmp2, "%s"SSLASH"userbase.id~", data_path);
    if (!copyfile(tmp,tmp2))
    {
        printf("Can't copy %s to %s\n", tmp, tmp2);
        return 0;
    }

    sprintf(tmp, "%s"SSLASH"userbase.dat", data_path);
    Fudat = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYRW);
    if (Fudat == -1)
    {
        printf("Can't open %s\n", tmp);
        return 0;
    }

    sprintf(tmp, "%s"SSLASH"userbase.idx", data_path);
    Fuidx = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYRW);
    if (Fuidx == -1)
    {
        FileClose(Fudat);
        printf("Can't open %s\n", tmp);
        return 0;
    }

    sprintf(tmp, "%s"SSLASH"userbase.da~", data_path);
    Ftdat = FileOpen(tmp, O_RDONLY | O_BINARY, SH_DENYRW);
    if (Ftdat == -1)
    {
        FileClose(Fuidx);
        FileClose(Fudat);
        printf("Can't open %s\n", tmp);
        return 0;
    }

    sprintf(tmp, "%s"SSLASH"userbase.id~", data_path);
    Ftidx = FileOpen(tmp, O_RDONLY | O_BINARY, SH_DENYRW);
    if (Ftidx == -1)
    {
        FileClose(Fuidx);
        FileClose(Fudat);
        FileClose(Ftdat);
        printf("Can't open %s\n", tmp);
        return 0;
    }

    if (!FileLock(Fudat))
    {
        printf("Can't lock userbase.dat file");
    __err:
        FileClose(Fuidx);
        FileClose(Fudat);
        FileClose(Ftdat);
        FileClose(Ftidx);
        return 0;
    }
    if (!FileLock(Fuidx))
    {
        printf("Can't lock userbase.idx file");
        goto __err;
    }
    if (!FileLock(Ftdat))
    {
        printf("Can't lock userbase.da~ file");
        goto __err;
    }
    if (!FileLock(Ftidx))
    {
        printf("Can't lock userbase.id~ file");
        goto __err;
    }

    FileTrunc(Fudat, 0);
    FileTrunc(Fuidx, 0);

    FileSeek(Fudat,0,SEEK_SET);
    FileSeek(Fuidx,0,SEEK_SET);

    memgot = 0; buf = NULL;
    FileSeek(Ftidx, 0, SEEK_SET);
    FileSeek(Fuidx, 0, SEEK_SET);
    while (FileRead(Ftidx,&uidx,sizeof(uidx)) == sizeof(uidx))
    {
        /* Read user data */
        FileSeek(Ftdat,uidx.SeekPos,SEEK_SET);
        if (FileRead(Ftdat,&user,sizeof(user)) != sizeof(user)) break;

        if (user.Attrib1 & USER_ATTRIB_DELETED && (user.Attrib1 & USER_ATTRIB_NOREMOVE) == 0)
        {
            /* Delete flag on, noremove flag off. */
            continue;
        }

        /* Read subfields */
        if (memgot < (int) user.SubfieldLen)
        {
            memgot = (int) user.SubfieldLen;
            if (buf != NULL) _free(buf);
            buf = (char *) _malloc(memgot);
        }
        if (FileRead(Ftdat,buf,(unsigned) user.SubfieldLen) != (int) user.SubfieldLen) break;

        /* Write index to temp file */
        /*if (uidx.SeekPos != (unsigned long) FileSeek(Fudat,0,SEEK_CUR))
        {*/
            uidx.SeekPos = FileSeek(Fudat,0,SEEK_CUR);
            if (FileWrite(Fuidx,&uidx,sizeof(uidx)) != sizeof(uidx))
            {
            __error:
                printf("Write error - not enough disk space??\n");
                FileClose(Fudat);
                FileClose(Fuidx);
                FileClose(Ftdat);
                copyfile("userbase.id~","userbase.idx");
                copyfile("userbase.da~","userbase.dat");
                return 0;
            }

            /* Write data and subfields to temp file */
            if (FileWrite(Fudat,&user,sizeof(user)) != sizeof(user)) goto __error;
            if (FileWrite(Fudat,buf,(int) user.SubfieldLen) != (int) user.SubfieldLen) goto __error;
        //}
    }
    _free(buf);

    FileClose(Fudat);
    FileClose(Fuidx);
    FileClose(Ftdat);
    FileClose(Ftidx);

    return 1;
}

int create_useridx(void)
{
    int Fudat, Fuidx;
    USER_IDX uidx, tmpuidx;
    USER_REC user;
    USER_SUBFIELD usub;
    unsigned long fpos, oldfpos, fsize, SubReaded;
    char username[36], tmp[256], tmp2[256];

    /* Make backups */
    sprintf(tmp, "%s"SSLASH"userbase.idx", data_path);
    sprintf(tmp2, "%s"SSLASH"userbase.id~", data_path);
    copyfile(tmp, tmp2);

    /* Open userbase.dat */
    sprintf(tmp, "%s"SSLASH"userbase.dat", data_path);
    Fudat = FileOpen(tmp, O_RDONLY | O_BINARY, SH_DENYRW);
    if (Fudat == -1)
    {
        printf("Can't open %s\n", tmp);
        return 0;
    }

    if (!FileLock(Fudat))
    {
        printf("Can't lock %s\n", tmp);
        FileClose(Fudat);
        return 0;
    }

    sprintf(tmp, "%s"SSLASH"userbase.idx", data_path);
    Fuidx = FileCreate(tmp, CREATE_MODE);
    if (Fuidx == -1)
    {
        printf("Can't create %s\n", tmp);
        FileClose(Fudat);
        return 0;
    }
    FileMode(Fuidx, O_BINARY);

    if (!FileLock(Fuidx))
    {
        printf("Can't lock %s\n", tmp);
        FileClose(Fuidx);
        FileClose(Fudat);
        return 0;
    }

    fsize = FileSeek(Fudat, 0, SEEK_END);
    FileSeek(Fudat, 0, SEEK_SET);
    uidx.SeekPos = 0;
    oldfpos = 0;

    while (FileRead(Fudat,&user,sizeof(user)) == sizeof(user))
    {
        /* Read subfields */
        fpos = 1;
        if (user.SubfieldLen+FileSeek(Fudat,0,SEEK_CUR) > fsize)
        {
            printf("Great. Someone's SubfieldLen is trashed, trying to find next record..\n");
        __back_scanning:
            oldfpos = fpos = FileSeek(Fudat, 0, SEEK_CUR);
            while (FileRead(Fudat,&user,sizeof(user)) == sizeof(user))
            {
                if ((user.PassCRC == 0 || user.PassCRC == 0xffffffff) &&
                    user.SubfieldLen < 100000 && user.Emulation < 4 &&
                    user.Attrib2 == 0 && user.Attrib3 == 0 && user.Attrib4 == 0 &&
                    (user.ScreenLen >= 10 && user.ScreenLen <= 100) &&
                    (user.ScreenWidth >= 70 && user.ScreenWidth <= 200) &&
                    user.TodayMinutes < 3600)
                {
                    /* I think I got one! */
                    uidx.SeekPos = FileSeek(Fudat, 0, SEEK_CUR)-sizeof(user);
                    printf("OK. I think I got one. Had to go %lu bytes away.\n", fpos-oldfpos);
                    oldfpos = fpos;
                    fpos = 0;
                    break;
                }
                fpos++;
                FileSeek(Fudat, fpos, SEEK_SET);
            }
            if (fpos != 0) break;
        }

        /* Read user name from subfields */
        SubReaded = 0; username[0] = '\0';
        while (SubReaded < user.SubfieldLen)
        {
            /* Read subfield ID */
            if (FileRead(Fudat,&usub,sizeof(usub)) != sizeof(usub))
            {
                printf("Can't read subfield!\n");
                break;
            }

            /* Read subfield data */
            switch (usub.ID)
            {
                case USERSUB_NAME:
                    if (usub.Len > sizeof(username)-1)
                    {
                        printf("Error! User name too long!\n");
                        break;
                    }
                    username[FileRead(Fudat,username,(unsigned) usub.Len)] = '\0';
                    break;
                default:
                    FileSeek(Fudat,usub.Len,SEEK_CUR);
                    break;
            }
            SubReaded += sizeof(usub)+usub.Len;
        }

        if (username[0] == '\0')
        {
            if (fpos == 0)
            {
                printf("Ups. It wasn't, user name not found..\n");
                FileSeek(Fudat, oldfpos+1, SEEK_SET);
                goto __back_scanning;
            }
            printf("Found invalid user record - username not found - skipping it..\n");
            uidx.SeekPos = FileSeek(Fudat, 0, SEEK_CUR);
            continue;
        }

        uidx.UserCRC = up_crc32(username);

        /* Check if user is already in .idx file (ie. replace older record) */
        FileSeek(Fuidx, 0, SEEK_SET);
        while (FileRead(Fuidx, &tmpuidx, sizeof(tmpuidx)))
        {
            if (tmpuidx.UserCRC == uidx.UserCRC)
            {
                /* Found one! */
                FileSeek(Fuidx, -sizeof(tmpuidx), SEEK_CUR);
                break;
            }
        }

        if (FileWrite(Fuidx, &uidx, sizeof(uidx)) != sizeof(uidx))
        {
            printf("Write error - not enough disk space??\n");
            FileClose(Fudat);
            FileClose(Fuidx);
            remove("userbase.idx");
            return 0;
        }

        uidx.SeekPos = FileSeek(Fudat, 0, SEEK_CUR);
    }

    FileUnlock(Fudat);
    FileUnlock(Fuidx);

    FileClose(Fuidx);
    FileClose(Fudat);

    return 1;
}

/* Show userbase */
void userlist(char *data)
{
    char scan[36],usr[36],lastuser[36];
    int len,width,emu;
    unsigned long userpos;

    scan[0] = '\0'; if (!ask_string(lang[LANG_USERLIST_ASK], scan, sizeof(scan)-1, 0, &data)) return;
    strupr(scan);
    strcpy(lastuser,usrsub.name);

    if (!write_user())
    {
        write_log("Can't write user to userbase!");
        return;
    }

    output(lang[LANG_USERLIST_HEADER]);
    len = user.ScreenLen; width = user.ScreenWidth; emu = user.Emulation;
    for (userpos = 1; read_user(userpos, 1) != 0; userpos++)
    {
        if (user.Attrib1 & USER_ATTRIB_DELETED) continue; /* Deleted, don't display */

        user.ScreenLen = (unsigned char) len; user.ScreenWidth = (unsigned char) width; user.Emulation = (unsigned char) emu;
        strupr(strcpy(usr,usrsub.name));
        if (scan[0] != '\0' && strstr(usr,scan) == NULL)
        {
            strupr(strcpy(usr,usrsub.alias));
            if (strstr(usr,scan) == NULL) continue;
        }

        if (!output(lang[LANG_USERLIST_ENTRY], userpos)) break;
    }

    read_user(scan_user(lastuser),1);
}
