/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* messages.c - Message handling */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <time.h>

#include "jam.h"
#include "crc32.h"
#include "nodes.h"
#include "output.h"
#include "os.h"
#include "fsed.h"
#include "messages.h"
#include "modem.h"
#include "userbase.h"
#include "lastcall.h"
#include "threads.h"
#include "fareas.h"
#include "mareas.h"
#include "files.h"
#include "quote.h"
#include "ask_str.h"
#include "bbs_func.h"
#include "msg_jam.h"
#include "logfile.h"
#include "timeslic.h"
#include "language.h"
#include "menutype.h"
#include "config.h"

#include "keyb.h"

static int pers_brk,pers_msgs;
static int msgtype = 0;         /* Message base type open */

unsigned long current_msg;      /* Current message */

MAREA_STAT marea_stat;          /* Message area status */

unsigned long msg_num;

/* Get message area status */
int msgarea_stat(char *fname, int type)
{
    char tmp[256];
    unsigned long CRC,pos;
    JAMIDXREC *jamidx;
    struct stat statbuf;

    pers_brk = 0;

    msg_close_area();
    memset(&marea_stat,0,sizeof(marea_stat));

    /* Don't do anything if not JAM area */
    if ((type & 127) != TYPE_JAM) return 0;

    /* Get user's CRC */
    CRC = lo_crc32(usrsub.name);

    sprintf(tmp,"%s.jdx",fname);
    if (stat(tmp,&statbuf) != 0) return 0;
    /* No unread messages, quit */
    if (marea_rec->lr_ptr >= statbuf.st_size / sizeof(JAMIDXREC)) return 0;

    /* Open index file */
    msg_Fjdx = FileBufOpen(tmp, O_RDONLY | O_BINARY, SH_DENYNO);
    if (msg_Fjdx == NULL) return 0;

    msg_Fjhr = NULL;

    /* Get number of unread and personal messages */
    FileBufSeek(msg_Fjdx, marea_rec->lr_ptr*sizeof(JAMIDXREC), SEEK_SET); pos = marea_rec->lr_ptr;
    while (FileBufRead(msg_Fjdx, (void **) &jamidx, sizeof(*jamidx)) == sizeof(*jamidx))
    {
        pos++;
        if (jamidx->HdrOffset == 0xffffffff) continue;
        if (jamidx->UserCRC == CRC)
        {
            marea_stat.personal_msgs++;
            marea_stat.unread_msgs++;
            if (type & TYPE_DISP_PERSONAL)
            {
                if (msg_Fjhr == NULL)
                {
                    sprintf(tmp,"%s.jhr",fname);
                    msg_Fjhr = FileBufOpen(tmp, O_RDONLY | O_BINARY, SH_DENYNO);
                    msg_Fjdt = msg_Fjhr;
                }
                if (msg_Fjhr != NULL)
                {
                    /* Read message record */
                    if (msg_read_message(pos))
                    {
                        if ((msg_flags & MSG_FLAG_RECEIVED) == 0)
                        {
                            /* Yeah, unread personal message found! */
                            strcpy(msg_from,msg_from);
                            strcpy(msg_to,msg_to);
                            strcpy(msg_subj,msg_subj);
                            msg_num = msg_msgnum;
                            pers_msgs++;
                            if (!output("@X0F@MAREA_NAME:25@ @X0A@MSG_FROM:25@ @X07@MSG_SUBJ.27@\r\n"))
                            {
                                pers_brk = 1;
                                break;
                            }
                        }
                    }
                }
            }
        }
        else if ((type & TYPE_DISP_PERSONAL) == 0)
        {
            if ((marea.flags & MAREA_FLAG_PRIVATE_MSGS) == 0)
            {
                marea_stat.unread_msgs++;
            }
            else
            {
                if (msg_Fjhr == NULL)
                {
                    sprintf(tmp,"%s.jhr",fname);
                    msg_Fjhr = FileBufOpen(tmp, O_RDONLY | O_BINARY, SH_DENYNO);
                    msg_Fjdt = msg_Fjhr;
                }
                if (msg_Fjhr != NULL)
                {
                    /* Read message record */
                    if (msg_read_message(pos))
                    {
                        if (((msg_flags & MSG_FLAG_PRIVATE) == 0 && marea.flags & MAREA_FLAG_PUBLIC_MSGS) || stricmp(msg_from,usrsub.name) == 0)
                            marea_stat.unread_msgs++;
                    }
                }
            }
        }
    }

    if (msg_Fjhr != NULL)
    {
        FileBufClose(msg_Fjhr);
        msg_Fjhr = NULL;
        msg_Fjdt = NULL;
    }

    marea_stat.total_msgs = FileBufSeek(msg_Fjdx,0,SEEK_CUR) / sizeof(JAMIDXREC);

    FileBufClose(msg_Fjdx);
    msg_Fjdx = NULL;

    return 1;
}

/* Open new message area */
int open_msgarea(char *fname, int type)
{
    msg_close_area();

    current_msg = marea_rec->lr_ptr;

    msgtype = type;
    return msg_open_area(fname,1);
}

/* Close area */
void close_msgarea(void)
{
    msg_close_area();
}

static int msg_more(void)
{
    char ch,loc;

    for (;;)
    {
        if (!carr_det()) return 0; /* No carrier */

        if (mdm_kbhit() || kbhit())
        {
            loc = (char) kbhit();
            ch = (char) (loc ? getch() : mdm_getch());
            switch (toupper(ch))
            {
                case 0:
                    ch = (char) (loc ? getch() : mdm_getch());
                    switch (ch)
                    {
                        case 'P':
                            /* Down */
                            return 1;
                        case 'H':
                            /* Up */
                            return 2;
                        case 'K':
                            /* Previous message */
                            return 3;
                        case 'M':
                            /* Next message */
                            return 4;
                        case 'Q':
                            /* PgDn */
                            return 5;
                        case 'I':
                            /* PgUp */
                            return 6;
                    }
                    break;
                case 'Q':
                    /* Quit */
                    return 0;
                case 'D':
                    /* Down */
                    return 1;
                case 'U':
                    /* Up */
                    return 2;
                case '-':
                case 'P':
                    /* Previous message */
                    return 3;
                case '+':
                case 'N':
                    /* Next message */
                    return 4;
                case 3:
                    /* PgDn */
                    return 5;
                case 18:
                    /* PgUp */
                    return 6;
                case 'R':
                    /* Reply */
                    return 7;
                case 'C':
                    /* Conf-reply */
                    return 8;
                case 13:
                    /* Enter - next new message */
                    return 9;
            }
        }
        else
        {
            give_timeslice();
        }
    }
}

static void draw_textline(char *txt)
{
    switch (msg_linerec->line_type)
    {
        case LINE_TYPE_QUOTE:
            output("@X%c%c", lang[LANG_MSG_DATA][6], lang[LANG_MSG_DATA][7]);
            break;
        case LINE_TYPE_TEAR:
            output("@X%c%c", lang[LANG_MSG_DATA][9], lang[LANG_MSG_DATA][10]);
            break;
        case LINE_TYPE_TAG:
            output("@X%c%c", lang[LANG_MSG_DATA][12], lang[LANG_MSG_DATA][13]);
            break;
        case LINE_TYPE_ORIGIN:
            output("@X%c%c", lang[LANG_MSG_DATA][15], lang[LANG_MSG_DATA][16]);
            break;
        case LINE_TYPE_KLUDGE:
            output("@X%c%c", lang[LANG_MSG_DATA][18], lang[LANG_MSG_DATA][19]);
            break;
        default:
            output("@X%c%c", lang[LANG_MSG_DATA][21], lang[LANG_MSG_DATA][22]);
            break;
    }

    if (txt[0] == 1)
    {
        output("@X%c%c", lang[LANG_MSG_DATA][18], lang[LANG_MSG_DATA][19]);
        txt[0] = '@';
    }

    outtext(txt);
    outtext("\r\n");
}

/* Draw message to screen */
void draw_msg(unsigned long msgnum)
{
    char pub,pvt,*txt;
    int lines,drawit;
    unsigned long line,linepos,last_marea;
    int msg_header_lines, msg_bottom_lines;

    msg_header_lines = (lang[LANG_MSG_DATA][0]-'0')*10 + lang[LANG_MSG_DATA][1]-'0';
    msg_bottom_lines = (lang[LANG_MSG_DATA][3]-'0')*10 + lang[LANG_MSG_DATA][4]-'0';

    drawit = 1;
    last_marea = current_marea;
__start:
    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    if (msg_area.number != marea.number)
        write_log("draw_msg() found bug : msg_area.number != marea.number - msg_area.name = %s, marea.name = %s", msg_area.name, marea.name);

    if (msg_read_message(msgnum) == 0)
    {
        output(lang[LANG_MSG_NOT_FOUND]);
        return;
    }

    pub = (marea.flags & MAREA_FLAG_PUBLIC_MSGS) > 0;
    pvt = (marea.flags & MAREA_FLAG_PRIVATE_MSGS) > 0;

    if (!((pub && !pvt) || /* Public message area */
          (pvt && pub && ((msg_flags & MSG_FLAG_PRIVATE) == 0)) || /* Public message */
          (in_group("sysop")) ||
          (stricmp(msg_to,usrsub.name) == 0) || /* User is receiver */
          (stricmp(msg_from,usrsub.name) == 0))) /* User is sender */
    {
        output(lang[LANG_NO_READ_RIGHTS]);
        return;
    }

    /* Update lastread pointers to this message */
    if (marea_rec->lr_ptr < msgnum)
    {
        user.MsgsRead++;
        update_pointers(msgnum);
    }

    current_lastrec.done_flags |= DONE_READ;

    noderec.doing = DOING_READ;
    strcpy(noderec.data,marea.name);
    update_nodefile(0);

    msg_num = msg_msgnum;
    if (marea.flags & MAREA_FLAG_NETMAIL)
    {
        sprintf(msg_from+strlen(msg_from), " (%s)", msg_orig_net);
        sprintf(msg_to+strlen(msg_to), " (%s)", msg_dest_net);
    }

    if (msg_init_msgtext() == 0)
    {
        output("@X0FMessage text not found!\r\n");
        noderec.doing = DOING_NOTHING;
        update_nodefile(0);
        return;
    }

    if (drawit)
        output(lang[LANG_MSG_HEADER]);
    else
    {
        if (last_marea != current_marea)
        {
            output(lang[LANG_MSGHDR_AREA_REDRAW]);
            last_marea = current_marea;
        }
        output(lang[LANG_MSGHDR_REDRAW]);
    }
    lines = msg_header_lines;

    for (line=1; line<=msg_txtlines; line++)
    {
        txt = msg_read_textline(line);

        if (!drawit) output("@CLREOL@");
        draw_textline(txt);
        lines++;
        if (lines == user.ScreenLen-(msg_bottom_lines-1)) break;
    }

    if (line > msg_txtlines) linepos = msg_txtlines; else linepos = line;

    if (!drawit)
    {
        while (lines < user.ScreenLen-(msg_bottom_lines-1))
        {
            output("@CLREOL@\r\n");
            lines++;
        }
    }

    /* More prompt */
    if (drawit)
    {
        output("@GOTO:1,%d@",user.ScreenLen-(msg_bottom_lines-1));
        output(lang[LANG_MSG_BOTTOM]);
    }
    while (linepos > 0)
    {
        switch (msg_more())
        {
            case 0:
                /* Quit */
                linepos = 0;
                break;
            case 1:
                /* Down */
                if (user.Emulation == USER_EMULATION_ANSI_X364)
                {
                    if (linepos < msg_txtlines)
                    {
                        linepos++;
                        output("@SCROLL_UP:%d@", msg_header_lines);
                        output("@SCROLL_DOWN:%d@@GOTO:1,%d@", user.ScreenLen-msg_bottom_lines, user.ScreenLen-msg_bottom_lines);
                        draw_textline(msg_read_textline(linepos));
                    }
                    break;
                }
            case 5:
            __pgdn:
                /* PgDn */
                if (linepos < msg_txtlines)
                {
                    lines = msg_header_lines-1;
                    output("@GOTO:1,%d@", msg_header_lines);
                    for (line=0; line<msg_txtlines; line++)
                    {
                        output("@CLREOL@");
                        draw_textline(msg_read_textline(linepos+line));
                        lines++;
                        if (lines == user.ScreenLen-msg_bottom_lines) break;
                    }
                    while (lines < user.ScreenLen-msg_bottom_lines)
                    {
                        output("@CLREOL@\r\n");
                        lines++;
                    }
                    linepos += user.ScreenLen-msg_header_lines-msg_bottom_lines;
                }
                break;
            case 2:
                /* Up */
                if (user.Emulation == USER_EMULATION_ANSI_X364)
                {
                    if ((int) linepos > user.ScreenLen-(msg_header_lines+msg_bottom_lines-1))
                    {
                        linepos--;
                        output("@SCROLL_UP:%d@@GOTO:1,%d@",user.ScreenLen-msg_bottom_lines,user.ScreenLen-msg_bottom_lines);
                        output("@SCROLL_DOWN:%d@", msg_header_lines);
                        draw_textline(msg_read_textline(linepos-(user.ScreenLen-msg_header_lines-msg_bottom_lines)));
                    }
                    break;
                }
            case 6:
                /* PgUp */
                if ((int) linepos > user.ScreenLen-(msg_header_lines+msg_bottom_lines-1))
                {
                    if ((int) linepos > (user.ScreenLen-(msg_header_lines+msg_bottom_lines-1))*2-1)
                        linepos -= user.ScreenLen-(msg_header_lines+msg_bottom_lines);
                    else
                        linepos = user.ScreenLen-(msg_header_lines+msg_bottom_lines-1);
                    lines = msg_header_lines;
                    output("@GOTO:1,%d@", msg_header_lines);
                    for (line=0; line<msg_txtlines; line++)
                    {
                        output("@CLREOL@");
                        draw_textline(msg_read_textline(linepos-(user.ScreenLen-msg_header_lines-msg_bottom_lines)+line));
                        lines++;
                        if (lines == user.ScreenLen-(msg_bottom_lines-1)) break;
                    }
                    while (lines < user.ScreenLen-(msg_bottom_lines-1))
                    {
                        output("@CLREOL@\r\n");
                        lines++;
                    }
                }
                break;
            case 3:
                /* Previous */
                msg_deinit_msgtext();
                prev_message(0);
                msgnum = current_msg;
                if (user.Emulation != USER_EMULATION_ASCII) drawit = 0;
                goto __start;
            case 4:
                /* Next */
                msg_deinit_msgtext();
                next_message(0);
                msgnum = current_msg;
                if (user.Emulation != USER_EMULATION_ASCII) drawit = 0;
                goto __start;
            case 7:
                /* Reply */
                output("@GOTO:1,%d@",user.ScreenLen);
                enter_message(1,0,NULL);
                drawit = 1;
                goto __start;
            case 8:
                /* Conf-reply */
                output("@GOTO:1,%d@",user.ScreenLen);
                enter_message(1,1,NULL);
                drawit = 1;
                goto __start;
            case 9:
                /* Go to next new message */
                if (linepos < msg_txtlines) goto __pgdn;
                msg_deinit_msgtext();
                if (next_new_message(0))
                {
                    msgnum = current_msg;
                    if (user.Emulation != USER_EMULATION_ASCII) drawit = 0;
                    goto __start;
                }
                linepos = 0;
                output("@GOTO:1,%d@",user.ScreenLen);
                output(lang[LANG_NO_UNREAD_LEFT]);
                break;
        }
    }

    output("@GOTO:1,%d@",user.ScreenLen);

    msg_deinit_msgtext();

    noderec.doing = DOING_NOTHING;
    update_nodefile(0);
}

/* List all messages in area to screen */
void list_msgs(char *data)
{
    unsigned long msgnum,lastmsg;
    char str[20], from_col[10], to_col[10], pvt, pub;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    if (msg_area.number != marea.number)
        write_log("list_msgs() found bug : msg_area.number != marea.number - msg_area.name = %s, marea.name = %s", msg_area.name, marea.name);

    for (;;)
    {
        msgnum = marea_rec->lr_ptr+1;
        if (msgnum > msg_messages) msgnum = msg_messages;

        str[0] = '\0'; num1 = msg_firstmsg; num2 = msg_messages; num3 = msgnum;
        if (!ask_string(lang[LANG_LIST_FROM], str, 10, 0, &data)) return;

        if (toupper(str[0]) == 'Q' && str[1] == '\0') return;

        if (str[0] == '\0')
        {
            /* Enter = Next new message (or current...) */
            sprintf(str, "%lu", msgnum);
        }

        if (sscanf(str, "%lu", &msgnum) == 1)
        {
            for (;;)
            {
                str[0] = '\0'; num1 = msgnum; num2 = msg_messages; num3 = msg_messages;
                if (!ask_string(lang[LANG_LIST_TO], str, 10, 0, &data)) return;

                if (toupper(str[0]) == 'Q' && str[1] == '\0') return;

                if (str[0] == '\0')
                {
                    /* Enter = first msg */
                    sprintf(str,"%u",msg_messages);
                }

                if (sscanf(str,"%lu",&lastmsg) == 1)
                    if (lastmsg >= msgnum && lastmsg <= msg_messages) break;
            }
            break;
        }
    }

    output(lang[LANG_LIST_HEADER]);

    pub = (marea.flags & MAREA_FLAG_PUBLIC_MSGS) > 0;
    pvt = (marea.flags & MAREA_FLAG_PRIVATE_MSGS) > 0;

    for (; msgnum<=lastmsg; msgnum++)
    {
        if (msg_read_message(msgnum) != 0)
        {
            if ((pub && !pvt) || /* Public message area */
                (pvt && pub && ((msg_flags & MSG_FLAG_PRIVATE) == 0)) || /* Public message */
                (in_group("sysop")) ||
                (stricmp(msg_to,usrsub.name) == 0) || /* User is receiver */
                (stricmp(msg_from,usrsub.name) == 0)) /* User is sender */
            {
                msg_num = msg_msgnum;
                strcpy(msg_from,msg_from);
                strcpy(msg_to,msg_to);
                strcpy(msg_subj,msg_subj);

                if (stricmp(msg_from,usrsub.name) == 0)
                    sprintf(from_col,"@X%c%c", lang[LANG_LIST_ENTRY][0], lang[LANG_LIST_ENTRY][1]);
                else
                    sprintf(from_col,"@X%c%c", lang[LANG_LIST_ENTRY][3], lang[LANG_LIST_ENTRY][4]);

                if (stricmp(msg_to,usrsub.name) == 0)
                    sprintf(to_col,"@X%c%c", lang[LANG_LIST_ENTRY][6], lang[LANG_LIST_ENTRY][7]);
                else
                    sprintf(to_col,"@X%c%c", lang[LANG_LIST_ENTRY][9], lang[LANG_LIST_ENTRY][10]);

                if (!output(lang[LANG_LIST_ENTRY]+12,from_col,to_col)) break;
            }
        }
    }
    output(lang[LANG_LIST_BOTTOM]);
}

/* Go to previous message */
void prev_message(int display)
{
    unsigned long old_msg;
    char pub,pvt;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    if (current_msg > 0)
    {
        pub = (marea.flags & MAREA_FLAG_PUBLIC_MSGS) > 0;
        pvt = (marea.flags & MAREA_FLAG_PRIVATE_MSGS) > 0;

        old_msg = current_msg;
        current_msg--;
        while (current_msg >= msg_firstmsg)
        {
            if (msg_read_message(current_msg) == 1)
            {
                if ((pub && !pvt) || /* Public message area */
                    (pvt && pub && ((msg_flags & MSG_FLAG_PRIVATE) == 0)) || /* Public message */
                    (in_group("sysop")) ||
                    (stricmp(msg_to,usrsub.name) == 0) || /* User is receiver */
                    (stricmp(msg_from,usrsub.name) == 0)) /* User is sender */
                {
                    if (display) draw_msg(current_msg);
                    return;
                }
            }
            current_msg--;
        }
        current_msg = old_msg;
    }

    if (display) output(lang[LANG_FIRST_MSG]);
}

/* Go to next message */
int next_message(int display)
{
    unsigned long old_msg;
    char pub,pvt;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return 0;
    }

    pub = (marea.flags & MAREA_FLAG_PUBLIC_MSGS) > 0;
    pvt = (marea.flags & MAREA_FLAG_PRIVATE_MSGS) > 0;

    old_msg = current_msg;
    current_msg++;
    while (current_msg <= msg_messages)
    {
        if (msg_read_message(current_msg) == 1)
        {
            if ((pub && !pvt) || /* Public message area */
                (pvt && pub && ((msg_flags & MSG_FLAG_PRIVATE) == 0)) || /* Public message */
                (in_group("sysop")) ||
                (stricmp(msg_to,usrsub.name) == 0) || /* User is receiver */
                (stricmp(msg_from,usrsub.name) == 0)) /* User is sender */
            {
                if (display) draw_msg(current_msg);
                return 1;
            }
        }
        current_msg++;
    }

    current_msg = old_msg;
    if (display == 2) output(lang[LANG_LAST_MSG_HERE]);
    update_pointers(msg_messages);
    return 0;
}

/* Go to next unread message */
int next_new_message(int display)
{
    unsigned long rec,old_area;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return 0;
    }
    for (;;)
    {
        if (marea_rec->lr_ptr >= msg_messages)
        {
            old_area = current_marea;
            for (rec=old_area+1; rec!=old_area; rec = rec>=msg_areas?1:rec+1)
            {
                if (read_marea_record(rec) == 0) continue;
                if ((marea.flags & MAREA_FLAG_SELECTED) == 0) continue;
                if (msgarea_stat(marea.path,TYPE_JAM) == 0) continue;
                if (marea_stat.unread_msgs == 0) continue;

                open_msgarea(marea.path,TYPE_JAM);
                memcpy(&msg_area,&marea,sizeof(msg_area));

                if (next_message(display)) return 1;
            }

            /* Read old area back in memory */
            if (read_marea_record(old_area))
            {
                open_msgarea(marea.path,TYPE_JAM);
                memcpy(&msg_area,&marea,sizeof(msg_area));
            }

            if (display) output(lang[LANG_NO_UNREAD_LEFT]);
            return 0;
        }

        current_msg = marea_rec->lr_ptr;
        if (next_message(display)) break;
    }

    return 1;
}

/* Go to message # */
void goto_message(unsigned long msgnum)
{
    char pub,pvt;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    pub = (marea.flags & MAREA_FLAG_PUBLIC_MSGS) > 0;
    pvt = (marea.flags & MAREA_FLAG_PRIVATE_MSGS) > 0;

    if (msg_read_message(msgnum) == 1)
    {
        if ((pub && !pvt) || /* Public message area */
            (pvt && pub && (msg_flags & MSG_FLAG_PRIVATE) == 0) || /* Public message */
            (in_group("sysop")) ||
            (stricmp(msg_to, usrsub.name) == 0) || /* User is receiver */
            (stricmp(msg_from, usrsub.name) == 0)) /* User is sender */
        {
            current_msg = msgnum;
            draw_msg(current_msg);
            return;
        }
    }

    output(lang[LANG_MSG_NOT_FOUND]);
}

unsigned long get_msgid(void)
{
    int F;
    char tmp[256];
    unsigned long msgtime,lasttime;

    /* Get last used MSGID */
    msgtime = time(NULL);
    sprintf(tmp, "%s"SSLASH"msgid.dat", data_path);
    F = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYRW);

    if (F != -1)
    {
        /* Lock & read last MSGID */
        if (!FileLock(F))
        {
            write_log("get_msgid() : Can't lock file: %s", tmp);
            FileClose(F);
            return 0;
        }
        if (FileRead(F,&lasttime,sizeof(lasttime)) != sizeof(lasttime)) lasttime = 0;
        if (msgtime <= lasttime) msgtime = lasttime+1;

        /* Write new MSGID */
        FileSeek(F,0,SEEK_SET); FileWrite(F,&msgtime,sizeof(msgtime));

        /* Unlock & close */
        FileUnlock(F);
        FileClose(F);
    }
    else
    {
        /* msgid.dat not found, create new one.. */
        F = FileCreate(tmp, CREATE_MODE);
        if (F != -1)
        {
            FileMode(F,O_BINARY);
            if (!FileLock(F))
                write_log("get_msgid() : Can't lock file: %s", tmp);
            else
            {
                FileWrite(F, &msgtime, sizeof(msgtime));
                FileUnlock(F);
                FileClose(F);
            }
        }
    }

    return msgtime;
}

void check_message_text(char *fname)
{
    int F;
    size_t readed;
    unsigned long lnum;
    unsigned last_cr,writeit,num,bufpos;
    char tmp[100],*strp,buf[100];

    /* Open text */
    F = FileOpen(fname, O_RDWR | O_BINARY, SH_DENYNO);
    if (F == -1) return;

    last_cr = 1; bufpos = 0;
    do
    {
        writeit = 0;
        readed = FileRead(F,tmp,sizeof(tmp));
        for (num=0; num<readed; num++)
        {
            switch (tmp[num])
            {
                case 13:
                    buf[bufpos] = '\0';
                    if (strcmp(buf,"---") == 0 || strncmp(buf,"--- ",4) == 0 || strncmp(buf," * Origin:",10) == 0)
                    {
                        /* Tear or origin found, don't allow them! */
                        buf[1] = '!';
                        lnum = FileSeek(F,0,SEEK_CUR);
                        FileSeek(F,-readed+num-bufpos,SEEK_CUR);
                        FileWrite(F,buf,bufpos);
                        FileSeek(F,lnum,SEEK_SET);
                    }
                    last_cr = 1;
                    bufpos = 0;
                    break;
                case 10:
                    break;
                case 1:
                    /* Kludges -> @ */
                    if (last_cr)
                    {
                        tmp[num] = '@';
                        writeit = 1;
                    }
                    break;
                case 26:
                    /* End message to EOT */
                    if (!last_cr)
                    {
                        tmp[num] = ' ';
                        writeit = 1;
                    }
                    else
                    {
                        lnum = FileSeek(F,0,SEEK_CUR)-readed+num;
                        FileSeek(F,0,SEEK_SET);
                        FileTrunc(F,lnum);
                        FileClose(F);
                        return;
                    }
                    break;
                default:
                    if (bufpos == sizeof(buf))
                    {
                        buf[0] = ' ';
                        bufpos = 1;
                    }
                    buf[bufpos++] = tmp[num];
                    last_cr = 0;
                    break;
            }
        }

        if (writeit)
        {
            FileSeek(F,-readed,SEEK_CUR);
            FileWrite(F,tmp,readed);
        }
    }
    while (readed == sizeof(tmp));

    lnum = FileSeek(F,0,SEEK_END);
    if (lnum < sizeof(tmp)) num = (unsigned) lnum; else num = sizeof(tmp);

    /* Strip off CRs and LFs from end of file */
    while (lnum > 0)
    {
        FileSeek(F,-num,SEEK_CUR);
        if (FileRead(F,tmp,num) != (int) num) break;
        FileSeek(F,-num,SEEK_CUR);

        strp = tmp+num-1;
        while (strp > tmp && (*strp == 13 || *strp == 10)) strp--;
        if (strp != tmp)
        {
            lnum = lnum-num + (strp-tmp)+1;
            break;
        }

        if (num > lnum) break;
        lnum -= num;
    }
    FileTrunc(F,lnum);

    FileClose(F);
}

void convert_file(char *fname, unsigned char *table)
{
    int file, readed, num;
    unsigned char buf[256];
    unsigned long pos;

    file = FileOpen(fname, O_RDWR | O_BINARY, SH_DENYRW);
    if (file == -1) return;

    pos = 0;
    for (;;)
    {
        FileSeek(file, pos, SEEK_SET);
        readed = FileRead(file, buf, sizeof(buf));
        if (readed == 0) break;

        num = 0;
        while (num < readed)
        {
            buf[num] = table[buf[num]];
            num++;
        }
        FileSeek(file, pos, SEEK_SET);
        FileWrite(file, buf, readed);
        pos += sizeof(buf);
    }

    FileClose(file);
}

char *get_tagline(char *tagfile, char *output)
{
    int f;
    char tmp[256];
    unsigned long taglines;

    f = FileOpen(tagfile, O_RDONLY | O_BINARY, SH_DENYNO);
    if (f == -1) return NULL;

    /* Get number of taglines */
    taglines = 0;
    while (_fgets(tmp, sizeof(tmp), f) != NULL) taglines++;

    /* Random and read tagline */
    FileSeek(f, 0, SEEK_SET);
    srand(time(NULL)); taglines = rand()%taglines;
    while (_fgets(tmp, sizeof(tmp), f) != NULL && taglines > 0) taglines--;
    FileClose(f);

    return conv_macros(tmp, output);
}

/* reply to message */
int enter_message(int reply, int another, char *data)
{
    unsigned long msgtime,old_area;
    char addr[40],str[256],*strp,datad[256];
    int slen,old_echo,num;
    int F;
    NETADDR_REC *net;

    time_t last_time;
    struct stat statbuf;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return 0;
    }

    sprintf(str,"%s"SSLASH"msgtmp",hold_farea.path);

    old_echo = (marea.flags & MAREA_FLAG_ECHO) > 0;
    if (reply)
    {
        /* Quote message text */
        if (!msg_read_message(current_msg))
        {
            output(lang[LANG_NO_REPLYMSG]);
            return 0;
        }
        quote_message(str);

        strcpy(msg_to,msg_from);
        strcpy(msg_from,usrsub.name);
        msg_replyto = msg_msgnum;
    }
    else
    {
        remove(str);
        msg_to[0] = '\0';
        msg_subj[0] = '\0';
        msg_replyto = 0;
    }

    old_area = current_marea;
    if (another)
    {
        /* Ask new area name */
        strcpy(str, "Netmail");
        if (!ask_string(lang[LANG_ASK_MAREA_NAME], str, 60, 0, &data))
        {
            if (reply)
            {
                sprintf(str, "%s"SSLASH"msgtmp", hold_farea.path);
                remove(str);
            }
            return 0;
        }
        if (!search_areaname(str))
        {
            output(lang[LANG_REP_MAREA_NOT_FOUND]);
            if (reply)
            {
                sprintf(str, "%s"SSLASH"msgtmp", hold_farea.path);
                remove(str);
            }
            return 0;
        }
    }

    if ((marea.flags & MAREA_FLAG_WRITE_RIGHTS) == 0)
    {
        output(lang[LANG_MAREA_NO_RIGHTS]);
        read_marea_record(old_area);
        return 0;
    }

    strcpy(msg_from,usrsub.name);

    /* Ask receiver */
    for (;;)
    {
        if (!ask_string(lang[LANG_ASK_RECEIVER], msg_to, sizeof(msg_to)-1, 0, &data) || msg_to[0] == '\0')
        {
            read_marea_record(old_area);
            return 0;
        }

        slen = strlen(msg_to)-1;
        if (msg_to[slen] == '?')
        {
            /* Show userbase */
            msg_to[slen] = '\0';
            userlist(msg_to);
            continue;
        }

        /* Redirect messages to "sysop" to real sysop */
        if (stricmp(msg_to, "SYSOP") == 0) strcpy(msg_to, sysop_name);

        /* Check if message is to 'All' */
        if (stricmp(msg_to, "All") == 0)
        {
            if ((marea.flags & (MAREA_FLAG_PRIVATE_MSGS|MAREA_FLAG_PUBLIC_MSGS)) == MAREA_FLAG_PRIVATE_MSGS)
            {
                /* No 'All' messages in private areas */
                output(lang[LANG_NO_ALL_MSGS]);
                continue;
            }
        }
        else
        {
            /* Check if user exists when not in echo or netmail area */
            if ((marea.flags & (MAREA_FLAG_ECHO|MAREA_FLAG_NETMAIL)) == 0 && scan_user(msg_to) == 0)
            {
                /* User does not exist */
                output(lang[LANG_USER_NOT_EXIST]);
                continue;
            }
        }
        break;
    }

    /* Ask subject */
    if (!ask_string(lang[LANG_ASK_SUBJECT], msg_subj, sizeof(msg_subj)-1, 0, &data) || msg_subj[0] == '\0')
    {
        read_marea_record(old_area);
        return 0;
    }

    if (marea.flags & MAREA_FLAG_NETMAIL)
    {
        /* Netmail area, ask address */
        str[0] = '\0';
        if (reply)
        {
            if (old_echo)
            {
                /* Orig. message was in echomail area, take address from origin */
                strp = strrchr(msg_origin_line, '(');
                if (strp != NULL)
                {
                    strcpy(str,++strp); strp = str;
                    while (*strp != '\0' && *strp != '@' && *strp != ')') strp++;
                    *strp = '\0';
                }
            }
            else
            {
                strcpy(str,msg_orig_net);
            }
        }
        do
        {
            if (!ask_string(lang[LANG_ASK_NETDEST], str, 23, 0, &data) || str[0] == '\0')
            {
                read_marea_record(old_area);
                return 0;
            }
        }
        while (str2addr(str,&msg_zone,&msg_net,&msg_node,&msg_point) == 0);
        if (msg_zone == 0) return 0;

        /* Netmail destination */
        sprintf(msg_dest_net,"%u:%u/%u",msg_zone,msg_net,msg_node);
        if (msg_point != 0) sprintf(msg_dest_net+strlen(msg_dest_net),".%u",msg_point);

        net = get_nearest_net(msg_zone,msg_net,msg_node,msg_point);
        sprintf(addr, "%u:%u/%u.%u", net->zone, net->net, net->node, net->point);
        strcpy(msg_orig_net, addr);

        /* Make INTL kludge */
        sprintf(msg_intl_kludge,"INTL %s %s",msg_dest_net,msg_orig_net);
    }
    else
    {
        msg_zone = 0;
        msg_orig_net[0] = '\0';
        msg_dest_net[0] = '\0';
    }

    msg_flags &= MSG_FLAG_PRIVATE;
    if (marea.flags & MAREA_FLAG_PRIVATE_MSGS && marea.flags & MAREA_FLAG_PUBLIC_MSGS)
    {
        sprintf(str, lang[LANG_ASK_PRIVATE], (msg_flags & MSG_FLAG_PRIVATE) ? lang[LANG_YES] : lang[LANG_NO]);
        if (yes_no(str, (msg_flags & MSG_FLAG_PRIVATE) > 0, &data))
            msg_flags |= MSG_FLAG_PRIVATE;
    }

    output("\r\n");

    sprintf(str,"%s"SSLASH"msgtmp",hold_farea.path);
    last_time = stat(str,&statbuf) == 0 ? statbuf.st_mtime : 0;

    /* Update node file */
    noderec.doing = DOING_WRITE;
    strcpy(noderec.data,marea.name);
    update_nodefile(0);

    sprintf(str,"%s"SSLASH"msgtmp",hold_farea.path);
    if (!fs_editor(str))
    {
        output(lang[LANG_MSG_ABORTED]);
        read_marea_record(old_area);
        return 0;
    }
    check_message_text(str);

    /* Update node file */
    noderec.doing = DOING_NOTHING;
    update_nodefile(0);

    msgtime = get_msgid();

    /* Add kludges, tear, origin */
    F = FileOpen(str, O_RDWR | O_BINARY, SH_DENYNO);
    if (F == -1)
    {
        output("\r\n@X0FMessage file not found!\r\n");
        write_log("Message file '%s' not found", str);
        read_marea_record(old_area);
        return 0;
    }
    if (fstat(F, &statbuf) == 0)
    {
        if (statbuf.st_mtime == last_time)
        {
            output(lang[LANG_MSG_ABORTED]);
            FileClose(F);
            read_marea_record(old_area);
            return 0;
        }
    }

    if ((marea.flags & MAREA_FLAG_NETMAIL) == 0)
    {
        if (marea.flags & MAREA_FLAG_ECHO)
        {
            /* Echo area, get message area address */
            net = firstnet; num = 1;
            while (net != NULL && num < marea.nodeaddr)
            {
                net = net->next; num++;
            }
            if (net == NULL) net = firstnet;
        }
        else
        {
            /* Local area, use main address */
            net = firstnet;
        }
        if (net->point == 0)
            sprintf(addr, "%u:%u/%u", net->zone, net->net, net->node);
        else
            sprintf(addr, "%u:%u/%u.%u", net->zone, net->net, net->node, net->point);
    }

    /* Make REPLY-kludge */
    if (reply && msg_msgid_kludge[0] != '\0')
        strcpy(msg_reply_kludge,msg_msgid_kludge);
    else
        msg_reply_kludge[0] = '\0';

    /* Make MSGID-kludge */
    sprintf(msg_msgid_kludge,"%s %08lx",addr,msgtime);

    if ((marea.flags & MAREA_FLAG_NO_TAGS) == 0)
    {
        /* Add tagline */
        sprintf(datad, "%s"SSLASH"taglines.dat", data_path);
        if (get_tagline(datad, str) != NULL)
        {
            sprintf(datad, "\r\r... %s", str);
            FileSeek(F, 0, SEEK_END);
            FileWrite(F, datad, strlen(datad));
        }
    }

    if (marea.flags & MAREA_FLAG_ECHO)
    {
        /* Echo area, add tear line and origin line */
        FileSeek(F, 0, SEEK_END);
        conv_macros(tear_line, str);
        sprintf(datad, "\r\r--- %s\r", str);
        strcpy(str," * Origin: ");
        conv_macros(origin_line, str+11);
        str[79-strlen(addr)-3] = '\0';
        sprintf(datad+strlen(datad), "%s (%s)", str, addr);
        FileWrite(F, datad, strlen(datad));
    }
    FileClose(F);

    /* Character set convertion */
    sprintf(msg_charset_kludge, "CHARSET: %s", marea.charset->kludge);

    sprintf(str,"%s"SSLASH"msgtmp",hold_farea.path);
    convert_file(str, marea.charset->outtable);
    convert_string(msg_from, marea.charset->outtable);
    convert_string(msg_to, marea.charset->outtable);
    convert_string(msg_subj, marea.charset->outtable);

    msg_date = time(NULL);

    if (another) open_msgarea(marea.path, TYPE_JAM);

    sprintf(str, "%s"SSLASH"msgtmp", hold_farea.path);
    if (msg_enter_msg(str) == 0)
    {
        output("\r\n\n@X0FCould not write message to message base!\r\n");
        write_log("Could not write message to message base");
        read_marea_record(old_area);
        if (another) open_msgarea(marea.path, TYPE_JAM);
        return 0;
    }

    sprintf(str,"%s"SSLASH"msgtmp",hold_farea.path);
    remove(str);

    memcpy(&msg_area, &marea, sizeof(msg_area));
    output(lang[LANG_MSG_WRITTEN]);
    write_log("Message written in area %s to %s : %s", marea.name, msg_to, msg_subj);

    user.MsgsWritten++;
    current_lastrec.done_flags |= DONE_WRITE;

    str[0] = NODETEXT_MSGWRITTEN;
    sprintf(str+1,"%u;@X0F%s viesti #%u: %s -> %s", current_marea, marea.name, msg_messages, msg_from, msg_to);
    send_nodetext(str,0);

    /* Read back last message area */
    read_marea_record(old_area);
    memcpy(&msg_area, &marea, sizeof(msg_area));
    if (another) open_msgarea(marea.path,TYPE_JAM);

    return 1;
}

/* Check if message is duplicate */
int check_dupe_msg(DUPE_REC *dupe)
{
    FILE_BUF *Fdup;
    char tmp[256];
    DUPE_REC *tmpdupe;

    /* Open dupemsgs.dat */
    sprintf(tmp, "%s"SSLASH"dupemsgs.dat", data_path);
    Fdup = FileBufOpen(tmp, O_RDONLY | O_BINARY, SH_DENYNO);
    if (Fdup == NULL) return 0;

    /* Scan it.. */
    while (FileBufRead(Fdup, (void **) &tmpdupe, sizeof(DUPE_REC)) == sizeof(DUPE_REC))
    {
        if (memcmp(tmpdupe, dupe, sizeof(DUPE_REC)) == 0)
        {
            /* Dupe! */
            FileBufClose(Fdup);
            return 1;
        }
    }

    FileBufClose(Fdup);
    return 0;
}

/* Add CRC to dupe message file */
void add_dupe_msg(DUPE_REC *dupe)
{
    int F;
    char tmp[256];

    /* Append to dupemsgs.dat */
    sprintf(tmp, "%s"SSLASH"dupemsgs.dat", data_path);
    F = FileOpen(tmp, O_WRONLY | O_BINARY, SH_DENYRW);
    if (F == -1)
    {
        F = FileCreate(tmp, CREATE_MODE);
        if (F == -1)
        {
            /* Error!??! */
            write_log("Can't create dupe file '%s'", tmp);
            return;
        }
        FileMode(F,O_BINARY);
    }

    if (!FileLock(F))
    {
        write_log("add_dupe_msg() : Can't lock file: %s", tmp);
    }
    else
    {
        FileSeek(F,0,SEEK_END);

        /* Write dupe.. */
        FileWrite(F,dupe,sizeof(DUPE_REC));

        FileUnlock(F);
        FileClose(F);
    }
}

void kill_message(void)
{
    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    if (current_msg == 0 || current_msg > msg_messages)
    {
        output(lang[LANG_MSG_NOT_FOUND]);
        return;
    }
    msg_read_message(current_msg);
    if (stricmp(msg_from,usrsub.name) != 0 && !in_group("sysop"))
    {
        output(lang[LANG_NO_KILL_RIGHTS]);
        return;
    }
    msg_flags |= MSG_FLAG_DELETED;
    if (msg_update_message(current_msg))
        output(lang[LANG_MSG_KILLED]);
    else
        output("\r\nError! Message not killed!\r\n");
}

void read_personal_msgs(void)
{
    unsigned long rec,old_rec;

    if (restrict_usage & NO_MESSAGES)
    {
        output("\r\nThread #%d restricts usage of message commands!\r\n",restrictive_thread(NO_MESSAGES));
        return;
    }

    output("\r\n@X07Scanning your mail...\r\n@MON@");

    old_rec = current_marea;
    pers_msgs = 0;
    for (rec=1; rec<msg_areas; rec++)
    {
        if (read_marea_record(rec) == 0) continue;
        memcpy(&msg_area,&marea,sizeof(msg_area));
        if ((marea.flags & MAREA_FLAG_SELECTED) == 0) continue;
        msgarea_stat(marea.path,TYPE_JAM|TYPE_DISP_PERSONAL);
        if (pers_brk) break;
    }

    read_marea_record(old_rec);
    memcpy(&msg_area,&marea,sizeof(msg_area));

    output("@MOFF@");
    if (pers_msgs == 0)
    {
        /* No personal messages */
        output(" ... No personal messages found.\r\n");
        return;
    }
    
    output("\r\n");
    if (yes_no("\rWant to read them (Y/n)? ", 1, NULL))
    {
        output("\r\nSorry, not yet possible. :)\r\n");
    }
    output("\r\n");
}
