/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* filelist.c - Display file list, flag files, delete files, etc. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>

#include "os.h"
#include "files.h"
#include "memory.h"
#include "ask_str.h"
#include "fareas.h"
#include "nodes.h"
#include "bbs_func.h"
#include "modem.h"
#include "timeslic.h"
#include "keyb.h"
#include "fsearch.h"
#include "output.h"
#include "filelist.h"
#include "userbase.h"
#include "logfile.h"
#include "language.h"
#include "config.h"

#include "keyb.h"

#define DATA_BUF 8192

int npos,positions[100];
char pos_fnames[100][81];

FILELIST_REC frec;

FLAG_REC *first_flag = NULL, *flagrec = NULL, *last_flag = NULL;

/* Release memory used by flagged files */
void deinit_flags(void)
{
    while (first_flag != NULL)
    {
        flagrec = first_flag->next;
        _free(first_flag->link);
        _free(first_flag->fname);
        _free(first_flag);
        first_flag = flagrec;
    }
    last_flag = NULL;
}

/* Add new record to linked list */
void add_flag_memory(char *fname, char *link, unsigned farea)
{
    flagrec = (FLAG_REC *) _malloc(sizeof(FLAG_REC)+strlen(fname)+1);
    if (last_flag != NULL) last_flag->next = flagrec;
    if (first_flag == NULL) first_flag = flagrec;

    flagrec->next = NULL;
    flagrec->farea = farea;
    flagrec->link = (char *) _malloc(strlen(link)+1);
    strcpy(flagrec->link, link);
    flagrec->fname = (char*) _malloc(strlen(fname)+1);
    strcpy(flagrec->fname, fname);
    strcpy(flagrec->origname, fname);
    last_flag = flagrec;
}

/* Remove record from linked list */
void delete_flag_memory(char *fname)
{
    FLAG_REC *prev;

    flagrec = first_flag; prev = NULL;
    while (flagrec != NULL)
    {
        if (stricmp(flagrec->fname,fname) == 0)
        {
            /* Found it. */
            if (prev == NULL)
                first_flag = flagrec->next;
            else
                prev->next = flagrec->next;
            if (flagrec == last_flag) last_flag = prev;
            _free(flagrec->link);
            _free(flagrec->fname);
            _free(flagrec);
            break;
        }
        prev = flagrec; flagrec = flagrec->next;
    }
}

/* Check if file is flagged */
int file_flagged(char *fname)
{
    /* Scan flagged files from memory */
    flagrec = first_flag;
    while (flagrec != NULL)
    {
        if (stricmp(flagrec->fname,fname) == 0)
        {
            /* Found! */
            return 1;
        }
        flagrec = flagrec->next;
    }
    return 0;
}

FLAG_REC *flagged_record(char *link)
{
    /* Scan flagged files from memory */
    flagrec = first_flag;
    while (flagrec != NULL)
    {
        if (stricmp(flagrec->link,link) == 0)
        {
            /* Found! */
            return flagrec;
        }
        flagrec = flagrec->next;
    }

    return NULL;
}

/* Flag file in area */
int flag_file(char *fname, int Fdes)
{
    char *data;
    char tmp[256],*strp,*strp2,*desc,found;
    unsigned long old_fpos,rec,oldrec;
    int slen, wild;
    struct stat statbuf;

    /* Check if file is already flagged */
    if (file_flagged(fname) || farea.flags & FAREA_FLAG_HOLD_AREA)
    {
        /* Remove flag */
        sprintf(tmp,"/tmp/%s",fname);
        delete_file(tmp);
        return 2;
    }

    /* Check if file contains wildcards */
    strp = fname; wild = 0;
    while (*strp != '\0')
    {
        if (*strp == '*' || *strp == '?')
        {
            wild = 1;
            break;
        }
        strp++;
    }

    /* Strip all extra spaces */
    while (*fname != '\0' && *fname == ' ') fname++;
    slen = strlen(fname)-1;
    while (fname[slen] == ' ' && slen > 0) slen--;
    fname[slen+1] = '\0';

    found = 0;
    if (Fdes != -1)
    {
        data = (char *) _malloc(DATA_BUF);

        /* Search file from descript.ion */
        old_fpos = FileSeek(Fdes,0,SEEK_CUR);
        FileSeek(Fdes,0,SEEK_SET);

        while (_fgets(data,DATA_BUF,Fdes) != NULL)
        {
            /* Get file name */
            strp = strchr(data,' ');
            if (strp != NULL)
            {
                *strp = '\0';
                desc = strp+1;
            }
            else
            {
                desc = NULL;
            }

            if (strp != data && data[0] != '\0')
            {
                /* Don't display private files */
                if (strp != NULL && *(strp+1) == '/' && !in_group("sysop")) continue;

                if (compare_filenames(data,fname) == 1)
                {
                    /* File found! */
                    if (desc != NULL) strp = strstr(desc,"@_LINK=");
                    if (strp == NULL)
                    {
                        /* File name */
                        sprintf(tmp,"%s"SSLASH"%s",farea.path,data);
                    }
                    else
                    {
                        /* Link */
                        strp2 = strchr(strp+7,'@');
                        if (strp2 == NULL || *(strp2-1) != '_')
                        {
                            /* Bad link */
                            sprintf(tmp,"%s"SSLASH"%s",farea.path,data);
                        }
                        else
                        {
                            *(strp2-1) = '\0';
                            strcpy(tmp,strp+7);
                        }
                    }

                    if (stat(tmp,&statbuf) == 0)
                    {
                        add_flag(data,tmp,desc,current_farea);
                        found = 1;
                        if (!wild) break;
                    }
                }
            }
        }
        _free(data);

        FileSeek(Fdes,old_fpos,SEEK_SET);
    }
    else
    {
        /* Search file from filebase */
        oldrec = fbase_pos;
        for (rec=1; read_filebase_record(rec) != 0; rec++)
        {
            /* Don't allow flagging private files */
            if (baserec->Flags & FBASE_FLAGS_PRIVATE && !in_group("sysop")) continue;

            strp = strrchr(fb_fname,SLASH);
            if (strp == NULL) continue;

            if (compare_filenames(strp+1,fname) == 1)
            {
                /* Found */
                old_fpos = FileBufSeek(fs_Fdat,0,SEEK_CUR);
                FileBufSeek(fs_Fdat,baserec->Ptr,SEEK_SET);
                FileBufRead(fs_Fdat,(void **) &data,DATA_BUF);
                FileBufSeek(fs_Fdat,old_fpos,SEEK_SET);

                strp = strstr(data,"@_LINK=");
                if (strp == NULL)
                {
                    /* File name */
                    strcpy(tmp,fb_fname);
                }
                else
                {
                    /* Link */
                    strp2 = strchr(strp+7,'@');
                    if (strp2 == NULL || *(strp2-1) != '_')
                    {
                        /* Bad link */
                        strcpy(tmp,fb_fname);
                    }
                    else
                    {
                        *(strp2-1) = '\0';
                        strcpy(tmp,strp+7);
                        *(strp2-1) = '@';
                    }
                }

#ifdef __FILES_CASE_SENSITIVE__
                search_ign_file(tmp);
#endif
                if (stat(tmp,&statbuf) == 0)
                {
                    strp = strrchr(fb_fname,SLASH);
                    if (strp != NULL) *strp++ = '\0';
                    add_flag(strp,tmp,data,search_path(fb_fname));
                    *(strp-1) = SLASH;
                    found = 1;
                    if (!wild) break;
                }
            }
        }
        if (oldrec != 0) read_filebase_record(oldrec);
    }

    if (!found)
    {
        output(lang[LANG_FILE_FLAG_NOT_FOUND]);
        yes_no("", 1, NULL); /* Wait for enter */
        slen = (lang[LANG_FILE_MORE_CONTROL][29]-'0')*10 + lang[LANG_FILE_MORE_CONTROL][30];
        memset(tmp,' ',slen); tmp[slen] = '\0';
        output("\r%s\r", tmp);
        return 0;
    }

    return 1;
}

int check_more(int *lines, int Fdes, int forced)
{
    char str[81],tmp[101],ch, *ctl;
    int slen,hit,ypos,startpos,morepos;

    ctl = lang[LANG_FILE_MORE_CONTROL];
    if (*lines == user.ScreenLen || forced)
    {
        if (user.Emulation == USER_EMULATION_ASCII) npos = 0;
        ypos = 0;
        morepos = wherey;
        startpos = wherey-*lines;
        *lines = 1;
        for (;;)
        {
            /* Ask question */
            output(lang[LANG_FILE_MORE_MESSAGE]);
            if (ypos < npos)
            {
                if (file_flagged(pos_fnames[ypos])) output("@X%c%c", ctl[5], ctl[6]); else output("@X%c%c", ctl[8], ctl[9]);
                output("@GOTO:1,%d@",startpos+positions[ypos]);
                outtext(pos_fnames[ypos]); output("@X%c%c", ctl[11], ctl[12]);
            }
            for (;;)
            {
                if (!carr_det()) return 0; /* No carrier */

                hit = kbhit();
                if (hit || mdm_kbhit())
                {
                    ch = (char) toupper(hit ? getch() : mdm_getch());
                    if (ch == 0)
                    {
                        ch = (char) (hit ? getch() : mdm_getch());
                        switch (ch)
                        {
                            case 'P':
                                /* Down */
                                if (ypos+1 < npos)
                                {
                                    if (file_flagged(pos_fnames[ypos])) output("\r@X%c%c", ctl[14], ctl[15]); else output("\r@X%c%c", ctl[11], ctl[12]);
                                    outtext(pos_fnames[ypos]);
                                    ypos++;
                                    if (file_flagged(pos_fnames[ypos])) output("@X%c%c", ctl[5], ctl[6]); else output("@X%c%c", ctl[8], ctl[9]);
                                    output("@GOTO:1,%d@",startpos+positions[ypos]);
                                    outtext(pos_fnames[ypos]); output("@X07");
                                }
                                else
                                {
                                    /* Next screen */
                                    if (!forced) ch = 'Y';
                                }
                                break;
                            case 'H':
                                /* Up */
                                if (ypos > 0)
                                {
                                    if (file_flagged(pos_fnames[ypos])) output("\r@X%c%c", ctl[14], ctl[15]); else output("\r@X%c%c", ctl[11], ctl[12]);
                                    outtext(pos_fnames[ypos]);
                                    ypos--;
                                    if (file_flagged(pos_fnames[ypos])) output("@X%c%c", ctl[5], ctl[6]); else output("@X%c%c", ctl[8], ctl[9]);
                                    output("@GOTO:1,%d@",startpos+positions[ypos]);
                                    outtext(pos_fnames[ypos]); output("@X07");
                                }
                                break;
                        }
                        if (ch != 'Y') continue;
                    }

                    if (user.Emulation != USER_EMULATION_ASCII && ch == ' ')
                    {
                        /* Flag file */
                        switch (flag_file(pos_fnames[ypos], Fdes))
                        {
                            case 1:
                                output("\r@X%c%c", ctl[5], ctl[6]);
                                outtext(pos_fnames[ypos]);
                                output("@X07");
                                break;
                            case 2:
                                output("\r@X%c%c", ctl[8], ctl[9]);
                                outtext(pos_fnames[ypos]);
                                output("@X07");
                                break;
                        }
                        continue;
                    }
                    if (ch == ctl[0] || ch == ctl[1] || ch == ctl[2] || ch == 13) break;
                }
                else
                {
                    give_timeslice();
                }
            }

            if (file_flagged(pos_fnames[ypos])) output("\r@X%c%c", ctl[14], ctl[15]); else output("\r@X%c%c", ctl[11], ctl[12]);
            outtext(pos_fnames[ypos]);

            output("@X07@GOTO:1,%d@", morepos);
            slen = (lang[LANG_FILE_MORE_CONTROL][23]-'0')*10 + lang[LANG_FILE_MORE_CONTROL][24];
            memset(tmp,' ',slen); tmp[slen] = '\0';
            output("%s\r",tmp);

            if (ch == 13) { npos = 0; return 1; } /* Enter pressed */
            if (ch == lang[LANG_FILE_MORE_CONTROL][0]) { npos = 0; return 1; } /* Y pressed */
            if (ch == lang[LANG_FILE_MORE_CONTROL][1]) { npos = 0; return 0; } /* N pressed */
            if (ch == lang[LANG_FILE_MORE_CONTROL][2])
            {
                /* Ask what to flag */
                str[0] = '\0';
                if (!ask_string(lang[LANG_FILE_FLAG], str, sizeof(str)-1, 0, NULL)) return 0;
                if (str[0] == '\0')
                {
                    outchr('\r');
                    continue;
                }

                /* Clear "Flag file:" prompt */
                slen = (lang[LANG_FILE_MORE_CONTROL][26]-'0')*10 + lang[LANG_FILE_MORE_CONTROL][27];
                slen += strlen(str)+1;
                memset(tmp,' ',slen); tmp[slen] = '\0';
                output("\r%s\r",tmp);

                /* Flag file */
                flag_file(str,Fdes);
            }
        }
    }
    return 1;
}

/* Compare if file matches search string */
int compare_filenames(char *fname, char *wildcard)
{
    char *strp,*strp2,*wptr;
    int quit,wild;
    
    if (wildcard[0] == '\0') return 1;
    
    while (wildcard != NULL)
    {
        wptr = strchr(wildcard,' ');
        if (wptr != NULL) *wptr = '\0';
        
        strp = fname;
        quit = 0;
        wild = 0;
        while (!quit)
        {
            switch(*wildcard)
            {
                case '\0':
                    if (wptr != NULL) *wptr = ' ';
                    if ((*strp != '\0') && (!wild)) return 0;
                    return 1;
                case '*':
                    strp2 = strrchr(fname,'.');
                    if ((strp2 == NULL) || (strp2 < strp))
                    {
                        strp = &fname[strlen(fname)];
                    }
                    else
                    {
                        strp = strp2;
                    }
                    wild = 1;
                    break;
                case '.':
                    if ((*strp != '.') && (*strp != '\0'))
                    {
                        quit = 1;
                        break;
                    }
                case '?':
                    if (*strp != '\0') strp++;
                    wild = 0;
                    break;
                default:
                    if (toupper(*strp) != toupper(*wildcard))
                    {
                        quit = 1;
                        break;
                    }
                    strp++;
                    wild = 0;
                    break;
            }
            wildcard++;
        }
        if (wptr != NULL)
        {
            *wptr = ' ';
            wildcard = wptr+1;
        }
        else
        {
            wildcard = NULL;
        }
    }
    
    return 0;
}

int disp_desc(char **desc, char *maxp, int *lines, int Fdes)
{
    char tmp[47],*strp;
    int slen,max;

    if (maxp == NULL || *desc+46 < maxp) max = 46; else max = (int) (maxp-*desc);

    if (max == 0)
    {
        *desc += strlen(*desc);
        outchr('\r');
        return 2;
    }

    strncpy(tmp,*((char **) desc),max); tmp[max] = '\0'; slen = strlen(tmp);
    if (slen == 46 && **desc+45 != ' ' && **desc+46 != ' ')
    {
        strp = strrchr(tmp,' ');
        if (strp == NULL)
        {
            *desc += 46;
        }
        else
        {
            *strp = '\0';
            *desc += (int) (strp-tmp)+1;
        }
    }
    else
    {
        if (slen == 46)
        {
            if (tmp[45] == ' ') tmp[45] = '\0';
            if (**desc+46 == ' ') (*desc)++;
        }
        *desc += slen;
    }
    output("@X%c%c%s\r\n",lang[LANG_FILE_MORE_CONTROL][17],lang[LANG_FILE_MORE_CONTROL][18],tmp);
    (*lines)++;
    return check_more(lines,Fdes,0) != 0;
}

int print_file(char *fname, char *data, int *lines, int Fdes, char **desc_out)
{
    char *macro = NULL;
    char *desc,*strp,tmp[256],macrostr[256];
    int ret,inmacro,num,is_line = 0,last_line = 0;
    struct stat statbuf;
    struct tm *tmbuf;

    /* Get file name */
    desc = strchr(data,' ');
    if (desc != NULL && desc != data)
    {
        *desc++ = '\0';
        if (desc[0] == '/' && !in_group("sysop"))
        {
            desc = NULL;
            return 1;
        }
    }

    if (data[0] != '\0')
    {
        if (data[0] == ' ')
        {
            /* Comment */
            desc = NULL;
            strp = data+1;
            while (*strp != '\0' && *strp == ' ') strp++;
            if (*strp != '\0')
            {
                output("@X%c%c%s\r\n",lang[LANG_FILE_MORE_CONTROL][20],
                       lang[LANG_FILE_MORE_CONTROL][21],data+1);
                (*lines)++;
                if (check_more(lines,Fdes,0) == 0) return 0;
            }
        }
        else
        {
            sprintf(tmp,"%s"SSLASH"%s",farea.path,data);

            /* Scan all @_MACROS_@ */
            strp = desc; inmacro = 0; frec.downs = 0;
            while (strp != NULL && *strp != '\0')
            {
                if (inmacro)
                {
                    if (*strp == '@' && (!is_line || last_line))
                    {
                        macro--;
                        if (*macro == '_' && macrostr[0] == '_')
                        {
                            *macro = '\0';
                            if (strnicmp(macrostr+1,"DOWNS=",6) == 0)
                            {
                                sscanf(macrostr+7,"%hu",&frec.downs);
                            }
                            else if (strnicmp(macrostr+1,"LINK=",5) == 0)
                            {
                                strcpy(tmp,macrostr+6);
                            }
                        }
                        inmacro = 0;
                    }
                    else
                    {
                        if (*strp == '_')
                        {
                            last_line = 1;
                            if (macro == macrostr) is_line = 1;
                        }
                        else
                            last_line = 0;

                        *macro++ = *strp;
                    }
                }
                else if (*strp == '@')
                {
                    inmacro = 1; is_line = 0; last_line = 0;
                    macro = macrostr;
                }
                strp++;
            }

            if (compare_filenames(data,fname) == 1)
            {
#ifdef __FILES_CASE_SENSITIVE_
                search_ign_file(tmp);
#endif
                ret = stat(tmp,&statbuf);

                frec.name = data;
                strcpy(pos_fnames[npos],data);

                /* Display flagged files with different color.. */
                if (file_flagged(data))
                    output("@X%c%c",lang[LANG_FILE_MORE_CONTROL][14],lang[LANG_FILE_MORE_CONTROL][15]);
                else
                    output("@X%c%c",lang[LANG_FILE_MORE_CONTROL][11],lang[LANG_FILE_MORE_CONTROL][12]);
                outtext(data);
                output("@X%c%c", lang[LANG_FILE_MORE_CONTROL][17], lang[LANG_FILE_MORE_CONTROL][18]);

                if (ret == 0)
                {
                    /* File exists, display file header */
                    tmbuf = localtime(&statbuf.st_mtime);
                    frec.size = statbuf.st_size;
                    sprintf(frec.date,"%02d.%02d.%02d",
                            tmbuf->tm_mday,tmbuf->tm_mon+1,tmbuf->tm_year);
                    if (strlen(data) > 12)
                    {
                        /* Long file name */
                        outtext("\r\n");
                        positions[npos++] = *lines;
                        (*lines)++;
                        if (check_more(lines,Fdes,0) == 0) return 0;
                        output(lang[LANG_FILE_LIST_ENTRY_LONG]);
                    }
                    else
                    {
                        for (num = strlen(data); num < 13; num++) outchr(' ');
                        output(lang[LANG_FILE_LIST_ENTRY]);
                        positions[npos++] = *lines;
                    }
                }
                else
                {
                    /* File not found - display offline .. */
                    if (strlen(data) > 12)
                    {
                        /* Long file name */
                        outtext("\r\n");
                        positions[npos++] = *lines;
                        (*lines)++;
                        if (check_more(lines,Fdes,0) == 0) return 0;
                        output(lang[LANG_FILE_LIST_OFFLINE_LONG],data);
                    }
                    else
                    {
                        for (num = strlen(data); num < 13; num++) outchr(' ');
                        output(lang[LANG_FILE_LIST_OFFLINE],data);
                        positions[npos++] = *lines;
                    }
                }

                if (desc != NULL)
                {
                    /* Display file description */
                    strp = strstr(desc,"@_");
                    if (strp != NULL) *strp = '\0';
                    if (!disp_desc(&desc,NULL,lines,Fdes)) return 0;
                }
                else
                {
                    output("\r\n");
                    (*lines)++;
                    if (check_more(lines,Fdes,0) == 0) return 0;
                }
            }
            else
            {
                /* Don't display any descriptions */
                desc = NULL;
            }
        }
    }

    *desc_out = desc;
    return 1;
}

/* List files in current file area */
int list_files(char *parms)
{
    char *data;
    char tmp[256],old_history[256],*strp,*parmsp,*desc,file_found;
    unsigned long old_farea,num;
    struct tm *tmbuf;
    int lines, num2, quit;
    int Fdes;

    strp = strchr(parms,' ');
    if (strp != NULL) *strp = '\0';

    conv_macros(parms,tmp);
    strcpy(parms,tmp);

    npos = 0;
    strcpy(old_history,farea_history);
    parmsp = strrchr(parms,'/');
    if (parmsp == NULL)
    {
        parmsp = parms;
    }
    else
    {
        *parmsp = '\0';

        if (parmsp != parms)
        {
            select_farea(parms);
        }
        else
        {
            select_farea("/");
        }
        parmsp++;
    }

    if (select_farea(parmsp) == 1)
    {
        if (strp == NULL)
        {
            *parmsp = '\0';
        }
        else
        {
            parmsp = strp+1;
        }
    }

    output("\r\n");

    /* Update node file */
    noderec.doing = DOING_FILELIST;
    strcpy(noderec.data,farea_history);
    update_nodefile(0);

    if ((farea.flags & FAREA_FLAG_DATABASE_AREA) == 0)
    {
        /* Try to open descript.ion */
        sprintf(tmp,"%s"SSLASH"descript.ion",farea.path);
        Fdes = FileOpenIgn(tmp, O_RDONLY | O_BINARY, SH_DENYNO);
    }
    else
    {
        Fdes = -1;
    }

    lines = 1;
    /* First display file areas in this subarea */
    old_farea = current_farea;
    for (num=1; num<=file_areas; num++)
    {
        read_farea_record(num);
        if (farea.subarea == current_subarea)
        {
            if (compare_filenames(farea.name,parmsp) == 1)
            {
                output(lang[LANG_FILEAREA_ENTRY]);
                lines++;
                if (check_more(&lines,Fdes,0) == 0)
                {
                    parms[0] = '\0';
                    return 1;
                }
            }
        }
    }
    read_farea_record(old_farea);

    if (Fdes == -1)
    {
        if (farea.flags & FAREA_FLAG_DATABASE_AREA)
        {
            /* Database area, display records.. */
            for (num=1; read_filebase_record(num) != 0; num++)
            {
                /* Don't display private files */
                if (baserec->Flags & FBASE_FLAGS_PRIVATE && !in_group("sysop")) continue;

                strp = strrchr(fb_fname,SLASH);
                if (strp != NULL) *strp = '\0';
                if (strp != NULL && strcmp(farea.path,fb_fname) == 0)
                {
                    /* Got one! */
                    *strp = SLASH;
                    tmbuf = localtime(&baserec->FileDate);
                    frec.name = strp+1;
                    frec.size = baserec->Size;
                    frec.downs = baserec->Downloads;
                    sprintf(frec.date,"%02d.%02d.%02d",
                            tmbuf->tm_mday,tmbuf->tm_mon+1,tmbuf->tm_year);
                    strcpy(pos_fnames[npos], frec.name);
                    positions[npos++] = lines;

                    if (file_flagged(frec.name))
                        output("@X%c%c", lang[LANG_FILE_MORE_CONTROL][14], lang[LANG_FILE_MORE_CONTROL][15]);
                    else
                        output("@X%c%c", lang[LANG_FILE_MORE_CONTROL][11], lang[LANG_FILE_MORE_CONTROL][12]);
                    output("%s@X%c%c",frec.name, lang[LANG_FILE_MORE_CONTROL][17], lang[LANG_FILE_MORE_CONTROL][18]);

                    if (strlen(strp+1) > 12)
                    {
                        /* Long file name */
                        outtext("\r\n");
                        lines++;
                        if (check_more(&lines,-1,0) == 0) break;
                        output(lang[LANG_FILE_LIST_ENTRY_LONG]);
                    }
                    else
                    {
                        for (num2 = strlen(frec.name); num2 < 13; num2++) outchr(' ');
                        output(lang[LANG_FILE_LIST_ENTRY]);
                    }

                    /* Display file description */
                    FileBufSeek(fs_Fdat, baserec->Ptr, SEEK_SET);
                    FileBufRead(fs_Fdat, (void **) &data, 8192);
                    desc = data;
                    strp = strstr(desc,"@_");
                    if (strp != NULL) *strp = '\0';
                    quit = 0;
                    for (;;)
                    {
                        if (!disp_desc(&desc,NULL,&lines,-1))
                        {
                            quit = 1;
                            break;
                        }
                        if (desc == NULL || *desc == '\0') break;
                        output(lang[LANG_DESCLINE_START]);
                    }
                    if (quit) break;

                    file_found = 1;
                }
                else
                    if (strp != NULL) *strp = SLASH;
            }
        }

        /* Update node file */
        noderec.doing = DOING_NOTHING;
        update_nodefile(0);

        if (old_history[0] != '\0') select_farea(old_history);
        parms[0] = '\0';
        return 0;
    }

    data = (char *) _malloc(DATA_BUF);
    if (data == NULL)
    {
        output("\r\nNot enough memory!\n");
        write_log("list_files() : Can't allocate %d bytes of memory\n", DATA_BUF);
        FileClose(Fdes);
        Fdes = -1;
    }

    /* Scan files in descript.ion */
    file_found = 0;
    desc = NULL;
    for (;;)
    {
        if (desc != NULL && *desc != '\0')
        {
            output(lang[LANG_DESCLINE_START]);
            if (!disp_desc(&desc,NULL,&lines,Fdes)) break;
            continue;
        }

        if (_fgets(data,DATA_BUF,Fdes) == NULL) break; /* Read file */
        file_found = 1;
        if (!print_file(parmsp,data,&lines,Fdes,&desc)) break;
    }

    if (lines > 1 && file_found)
    {
        check_more(&lines,Fdes,1);
    }

    _free(data);

    if (old_history[0] != '\0') select_farea(old_history);

    /* Update node file */
    noderec.doing = DOING_NOTHING;
    update_nodefile(0);

    FileClose(Fdes);
    parms[0] = '\0';
    return 1;
}

/* Add new file to flagged files */
int add_flag(char *fname, char *linkname, char *desc, unsigned farea)
{
    int Fdes;
    char tmp[256],*strp;

    if (fname == NULL)
    {
        fname = strrchr(linkname,SLASH);
        if (fname == NULL)
        {
            write_log("add_flag() : Invalid linkname '%s'",linkname);
            output("\r\n@X0FCan't find file to be flagged!\r\n");
            return 0;
        }
        fname++;
    }

    sprintf(tmp, "%s"SSLASH"descript.ion",hold_farea.path);
    Fdes = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYRW);
    if (Fdes == -1)
    {
        Fdes = FileCreate(tmp, CREATE_MODE);
        if (Fdes == -1)
        {
            write_log("add_flag() : Could not create file '%s'", tmp);
            output("\r\n@X07ERROR: Could not create file, file not flagged\r\n");
            return 0;
        }
    }
    FileSeek(Fdes, 0, SEEK_END);

    FileWrite(Fdes, fname, strlen(fname));
    if (desc != NULL)
    {
        strp = strstr(desc,"@_LINK=");
        FileWrite(Fdes, " ", 1);
        FileWrite(Fdes, desc, (int) (strp-desc));
    }
    FileWrite(Fdes, tmp, sprintf(tmp, " @_LINK=%s_@\n", linkname));
    FileClose(Fdes);

    add_flag_memory(fname, linkname, farea);

    return 1;
}

int remove_from_description(char *fname, char *scan)
{
    int Fdes;
    char *strp,*data,*desc, tmp[256];
    unsigned long fpos,fpos2;
    int slen;

    /* Open descript.ion */
    Fdes = FileOpen(fname, O_RDWR | O_BINARY, SH_DENYRW);
    if (Fdes == -1) return 0;

    if (!FileLock(Fdes))
    {
        write_log("remove_from_description() : Can't lock file: %s", fname);
        FileClose(Fdes);
        return 0;
    }

    data = (char *) _malloc(DATA_BUF);
    fpos = 0;
    for (; _fgets(data,DATA_BUF,Fdes) != NULL; fpos = FileSeek(Fdes,0,SEEK_CUR))
    {
        /* Get file name */
        strp = strchr(data,' ');
        if (strp == data || data[0] == '\0') continue;

        slen = strlen(data);
        if (strp != NULL) *strp++ = '\0';
        desc = strp;

        if (compare_filenames(data, scan) == 1)
        {
            /* File found! */
            if (farea.flags & FAREA_FLAG_HOLD_AREA) delete_flag_memory(data);

            if (desc != NULL) strp = strstr(desc, "@_LINK="); else strp = NULL;
            if (strp == NULL || strstr(desc, "@_KILL_@") != NULL)
            {
                if (strp == NULL)
                {
                    /* Not a file link, delete real file also */
                    sprintf(tmp, "%s"SSLASH"%s",farea.path,data);
                    if (remove(tmp) != 0)
                        write_log("delete_file() : Can't remove file '%s'", tmp);
                }
                else
                {
                    /* @_KILL_@ found from link, so delete real file also */
                    desc = strp+7;
                    strp = strstr(desc, "_@");
                    if (strp != NULL)
                    {
                        *strp = '\0';
                        if (remove(desc) != 0)
                            write_log("delete_file() : Can't remove file '%s'", desc);

                        /* Remove from original descript.ion */
                        strp = strrchr(desc, SLASH);
                        if (strp != NULL)
                        {
                            strcpy(tmp, strp+1);
                            strcpy(strp, SSLASH"descript.ion");
                            remove_from_description(desc, tmp);
                        }
                    }
                }
            }

            /* Delete it from descript.ion. */
            fpos2 = FileSeek(Fdes,0,SEEK_CUR);
            FileSeek(Fdes,fpos,SEEK_SET);
            memset(data,'\0',slen);
            FileWrite(Fdes,data,slen);
            FileSeek(Fdes,fpos2,SEEK_SET);

        }
    }
    _free(data);
    FileUnlock(Fdes);

    return 1;
}

int delete_file(char *parms)
{
    char tmp[256], old_history[256], *strp, *parmsp;

    strcpy(old_history,farea_history);

    strp = strchr(parms,' ');
    if (strp != NULL) *strp = '\0';

    conv_macros(parms,tmp);
    strcpy(parms,tmp);

    parmsp = strrchr(parms,'/');
    if (parmsp == NULL)
    {
        parmsp = parms;
    }
    else
    {
        /* Path found from names */
        *parmsp = '\0';
        if (parmsp != parms)
        {
            select_farea(parms);
        }
        else
        {
            select_farea("/");
        }
        parmsp++;
    }

    if (select_farea(parmsp) == 1)
    {
        if (strp == NULL)
        {
            *parmsp = '\0';
        }
        else
        {
            parmsp = strp+1;
        }
    }

    if ((farea.flags & FAREA_FLAG_WRITE_RIGHTS) == 0)
    {
        /* No write rights */
        output(lang[LANG_FILEAREA_NO_RIGHTS]);

        parms[0] = '\0';
        if (old_history[0] != '\0') select_farea(old_history);
        return 0;
    }

    sprintf(tmp,"%s"SSLASH"descript.ion",farea.path);
    remove_from_description(tmp, parmsp);

    /* Move back to old file area */
    parms[0] = '\0';
    if (old_history[0] != '\0') select_farea(old_history);

    return 1;
}

int add_dl_counter(char *fname)
{
    int Fdes;
    char tmp[256],*data,*desc,*strp,*orig;
    unsigned downs;
    unsigned long fpos,inspos;

    /* Open descript.ion */
    sprintf(tmp,"%s"SSLASH"descript.ion",farea.path);
    Fdes = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYNO);
    if (Fdes == -1) return 0;

    data = (char *) _malloc(DATA_BUF);
    fpos = 0;
    for (; _fgets(data,DATA_BUF,Fdes) != NULL; fpos = FileSeek(Fdes,0,SEEK_CUR))
    {
        /* Get file name */
        desc = strchr(data,' ');
        if (desc == data || data[0] == '\0') continue;
        if (desc != NULL) *desc++ = '\0';

        if (stricmp(data,fname) == 0)
        {
            /* File found! */
            if (desc != NULL)
            {
                strp = strstr(desc,"@_DOWNS=");
                if (strp != NULL)
                {
                    /* Write new counter */
                    strp += 8; orig = strp; downs = 0;
                    while (*strp >= '0' && *strp <= '9')
                    {
                        downs = downs*10 + (*strp-'0');
                        strp++;
                    }
                    FileSeek(Fdes,(int) (orig-data)+fpos,SEEK_SET);
                    sprintf(tmp,"%04u",downs+1);
                    FileWrite(Fdes,tmp,4);
                    break;
                }
            }

            /* No counter yet, insert some space for it. */
            if (desc == NULL)
                inspos = fpos + strlen(data);
            else
                inspos = fpos + (int) (desc-data) + strlen(desc);

            if (FileLock(Fdes))
            {
                insert_space(Fdes,inspos,14,FileSeek(Fdes,0,SEEK_END));
                FileSeek(Fdes,inspos,SEEK_SET);
                FileWrite(Fdes,"@_DOWNS=0001_@",14);
                FileUnlock(Fdes);
            }
            else
                write_log("add_dl_counter() : Can't lock file: %s"SSLASH"descript.ion",farea.path);
            break;
        }
    }
    _free(data);

    FileClose(Fdes);
    return 0;
}

int rename_file(char *parms)
{
    int Fdes,Fnew;
    char old_history[256],tmp[256],tmp2[256],*data,*desc,*strp,*from,*to;
    unsigned tmparea;
    int diff,ok;

    /* File path */
    strcpy(old_history,farea_history);

    strp = strchr(parms,' ');
    if (strp == NULL)
    {
        output(lang[LANG_FRENAME_NO_DEST]);

        parms[0] = '\0';
        if (old_history[0] != '\0') select_farea(old_history);
        return 0;

    }
    *strp++ = '\0';

    conv_macros(parms,tmp);
    strcpy(parms,tmp);

    from = strrchr(parms,'/');
    if (from == NULL)
    {
        from = parms;
    }
    else
    {
        /* Path found from names */
        *from = '\0';
        if (from != parms)
        {
            select_farea(parms);
        }
        else
        {
            select_farea("/");
        }
        from++;
    }

    if (select_farea(from) == 1)
    {
        /* Can't rename directories.. */
        output(lang[LANG_FRENAME_NO_DIRS]);

        parms[0] = '\0';
        if (old_history[0] != '\0') select_farea(old_history);
        return 0;
    }

    if ((farea.flags & FAREA_FLAG_WRITE_RIGHTS) == 0)
    {
        /* No write rights */
        output(lang[LANG_FILEAREA_NO_RIGHTS]);

        parms[0] = '\0';
        if (old_history[0] != '\0') select_farea(old_history);
        return 0;
    }

    /* Get destination file */
    data = strp;
    strp = strchr(strp,' ');
    if (strp != NULL) *strp = '\0';

    conv_macros(data,tmp2);

    to = strrchr(tmp2,'/');
    if (to == NULL)
    {
        to = tmp2;
    }
    else
    {
        *to++ = '\0';
        tmparea = current_farea;
        if (old_history[0] != '\0') select_farea(old_history);
        select_farea(tmp2);
        if (current_farea != tmparea)
        {
            output("\r\nSource and destination must be in same directory\r\n");

            parms[0] = '\0';
            if (old_history[0] != '\0') select_farea(old_history);
            return 0;
        }
    }

    if (from[0] == '\0' || to == NULL || to[0] == '\0')
    {
        if (from[0] == '\0')
            output(lang[LANG_FRENAME_NO_SOURCE]);
        else
            output(lang[LANG_FRENAME_NO_DEST]);

        parms[0] = '\0';
        if (old_history[0] != '\0') select_farea(old_history);
        return 0;
    }

    /* Open descript.ion */
    sprintf(tmp,"%s"SSLASH"descript.ion",farea.path);
    Fdes = FileOpen(tmp, O_RDWR | O_BINARY, SH_DENYNO);
    if (Fdes == -1)
    {
        output(lang[LANG_FILE_NOT_FOUND]);
        parms[0] = '\0';
        if (old_history[0] != '\0') select_farea(old_history);
        return 0;
    }

    /* Create temp descript.ion */
    sprintf(tmp,"%s"SSLASH"descript.io~",farea.path);
    Fnew = FileCreate(tmp, CREATE_MODE);
    if (Fnew == -1)
    {
        parms[0] = '\0';
        if (old_history[0] != '\0') select_farea(old_history);
        return 0;
    }
    FileMode(Fnew,O_BINARY);

    data = (char *) _malloc(DATA_BUF); ok = 0; diff = 0;
    while (_fgets(data,DATA_BUF,Fdes) != NULL)
    {
        /* Get file name */
        desc = strchr(data,' ');
        if (desc == data || data[0] == '\0') continue;
        if (desc != NULL) *desc = '\0';

        if (stricmp(data,to) == 0)
        {
            /* Destination file already exists, quit.. */
            break;
        }

        if (stricmp(data,from) == 0)
        {
            /* Found! Rename. */
            *desc = ' ';
            diff = strlen(from)-strlen(to);
            if (diff != 0) memmove(data,data+diff,strlen(data)-diff+1);
            memcpy(data,to,strlen(to));
            ok = 1;
        }
        else
        {
            *desc = ' ';
        }

        strcat(data,"\n");
        FileWrite(Fnew,data,strlen(data));
    }
    _free(data);

    FileClose(Fdes);
    FileClose(Fnew);

    if (ok)
    {
        /* If this was hold area, modify also name in memory.. */
        if (farea.flags & FAREA_FLAG_HOLD_AREA)
        {
            flagrec = first_flag;
            while (flagrec != NULL)
            {
                if (stricmp(flagrec->fname,from) == 0)
                {
                    /* Found it! */
                    if (diff < 0)
                    {
                        _free(flagrec->fname);
                        flagrec->fname = (char *) _malloc(strlen(to)+1);
                    }
                    strcpy(flagrec->fname,to);

                    /* New link name.. */
                    _free(flagrec->link);
                    sprintf(tmp,"%s"SSLASH"%s",hold_farea.path,flagrec->fname);
                    flagrec->link = (char *) _malloc(strlen(tmp)+1);
                    strcpy(flagrec->link,tmp);
                    break;
                }
                flagrec = flagrec->next;
            }
        }

        /* File renamed ok. */
        sprintf(tmp,"%s"SSLASH"descript.ion",farea.path);
        sprintf(tmp2,"%s"SSLASH"descript.io~",farea.path);
        remove(tmp);
        rename(tmp2,tmp);
    }
    else
    {
        /* Didn't find source or destination already exists.. */
        sprintf(tmp,"%s"SSLASH"descript.io~",farea.path);
        remove(tmp);
        output(lang[LANG_FILE_NOT_FOUND]);
    }

    parms[0] = '\0';
    if (old_history[0] != '\0') select_farea(old_history);
    return ok;
}

int get_filemail(void)
{
    char str[256], *data, *desc, *strp;
    int h, fdes;

    /* Open descript.ion from filemail path */
    sprintf(str,"%s"SSLASH"descript.ion", filemail_path);
    fdes = FileOpen(str, O_RDONLY, SH_DENYNO);
    if (fdes == -1) return 0;

    data = _malloc(8192); h = -1;
    while (_fgets(data, 8192, fdes) != NULL)
    {
        strp = strchr(data, ' ');
        if (strp == NULL) continue; /* Just file name, no to-field.. */
        *strp++ = '\0'; desc = strp;

        strp = strstr(desc, "@_TO=");
        if (strp == NULL) continue; /* To-field not found */

        sprintf(str, "%s_@", usrsub.name);
        if (strnicmp(strp+5, str, strlen(str)) == 0)
        {
            /* Ok, it's for me. */
            if (h == -1)
            {
                /* Create descript.ion to hold path */
                sprintf(str,"%s"SSLASH"descript.ion", hold_farea.path);
                h = FileOpen(str, O_RDWR | O_BINARY, SH_DENYNO);
                if (h != -1)
                    FileSeek(h, 0, SEEK_END);
                else
                {
                    h = FileCreate(str, CREATE_MODE);
                    if (h == -1)
                    {
                        write_log("Can't create file %s", str);
                        output("\r\n@X0FCan't create descript.ion\r\n");
                        break;
                    }
                }
            }

            strp = strstr(desc, "@_LINK=");
            if (strp == NULL)
                sprintf(str, "@_LINK=%s"SSLASH"%s_@@_KILL_@", filemail_path, data);
            else
                str[0] = '\0';
            *(desc-1) = ' ';
            FileWrite(h, data, strlen(data)); /* Write file+description */
            FileWrite(h, str, strlen(str)); /* Write LINK */
            FileWrite(h, "\n", 1);
        }
    }
    _free(data);

    if (h != -1) FileClose(h);
    FileClose(fdes);

    return h != -1;
}
