/* YAK - Copyright (c) 1997 Timo Sirainen - read license.txt */

/* fareas.c - File area handling */

#include <stdio.h>
#include <string.h>

#include "os.h"
#include "memory.h"
#include "files.h"
#include "output.h"
#include "logfile.h"
#include "userbase.h"
#include "fareas.h"
#include "config.h"

FAREA_REC farea;                /* Current file area record */
FAREA_REC *farea_rec;           /* Current file area record .. pointer */
unsigned file_areas;            /* Number of file areas */
unsigned ul_area;               /* Which area to put uploads */

static FAREA_REC *first_farea;  /* First file area record in linked list */
FAREA_REC hold_farea;           /* Hold file area record */
FAREA_REC ul_farea;             /* Upload file area record */

unsigned current_farea;         /* Current file area number */
unsigned current_subarea;       /* Current subarea number */

char farea_history[256];        /* "/area/area2/..." style area history */

/* Read file areas to memory */
int init_fareas(int all_areas)
{
    int Farea;
    char str[512],rsec[81],wsec[81],flags[11];
    unsigned long line;
    FAREA_REC *tmp_rec,tmp_farea_rec;
#if SLASH == '\\'
    char *strp;
#endif

    file_areas = 0;
    ul_area = 0;
    memset(&hold_farea,0,sizeof(hold_farea));

    /* Open file area configureation file */
    Farea = FileOpen(fareas_bbs, O_RDONLY, SH_DENYNO);
    if (Farea == -1) return 0;

    /* Read it */
    line = 0;
    while (_fgets(str,sizeof(str),Farea) != NULL)
    {
        line++;
        if (str[0] == ';' || str[0] == '\0') continue;

        memset(&tmp_farea_rec,0,sizeof(tmp_farea_rec));
        if (sscanf(str,"%s %s %u %u %s %s %s %[^\n]",
                   tmp_farea_rec.path,
                   tmp_farea_rec.name,
                   &tmp_farea_rec.subarea,
                   &tmp_farea_rec.jumparea,
                   rsec, wsec,
                   flags,
                   tmp_farea_rec.description
                  ) < 7)
        {
            write_log("Error in file area configuration file '%s' line %lu", fareas_bbs, line);
            printf("Error in file area configuration file '%s' line %lu\n", fareas_bbs, line);
            continue;
        }

        if (!all_areas && !in_group(rsec)) continue;

        /* Area name '/' means empty */
        if ((tmp_farea_rec.name[0] == '/') && (tmp_farea_rec.name[1] == '\0'))
        {
            tmp_farea_rec.name[0] = '\0';
        }

        tmp_rec = (FAREA_REC *) _malloc(sizeof(FAREA_REC));
        if (tmp_rec == NULL)
        {
            write_log("Not enough memory");
            printf("Not enough memory!\n");
            deinit_fareas();
            return 0;
        }

        if (file_areas > 0)
        {
            farea_rec->next = tmp_rec;
        }
        else
        {
            first_farea = tmp_rec;
        }
        file_areas++;

        /* Make file area record */
        tmp_rec->prev = farea_rec;
        farea_rec = tmp_rec;
        farea_rec->next = NULL;

        conv_macros(tmp_farea_rec.path,farea_rec->path);
#if SLASH == '\\'
        strp = farea_rec->path;
        while (*strp != '\0')
        {
            if (*strp == '/') *strp = '\\';
            strp++;
        }
#endif
        strcpy(farea_rec->name,tmp_farea_rec.name);
        strcpy(farea_rec->description,tmp_farea_rec.description);
        farea_rec->flags = FAREA_FLAG_READ_RIGHTS;
        farea_rec->subarea = tmp_farea_rec.subarea;
        farea_rec->jumparea = tmp_farea_rec.jumparea;
        if (in_group(wsec)) farea_rec->flags |= FAREA_FLAG_WRITE_RIGHTS;

        /* Get flags */
        strupr(flags);
        if (strchr(flags,'U') != NULL)
        {
            /* Upload area */
            farea_rec->flags |= FAREA_FLAG_UPLOAD_AREA;
            if (ul_area == 0) ul_area = file_areas;
            if (ul_farea.path[0] == '\0')
            {
                memcpy(&ul_farea,farea_rec,sizeof(ul_farea));
            }
        }
        if (strchr(flags,'H') != NULL)
        {
            /* Hold area */
            farea_rec->flags |= FAREA_FLAG_HOLD_AREA;
            if (hold_farea.path[0] == '\0')
            {
                memcpy(&hold_farea,farea_rec,sizeof(hold_farea));
                mkpath(hold_farea.path);

                /* Remove descript.ion from hold path */
                erase_path(hold_farea.path);
            }
        }
        if (strchr(flags,'D') != NULL)
        {
            /* Database area */
            farea_rec->flags |= FAREA_FLAG_DATABASE_AREA;
        }
    }

    /* Close file area configuration file */
    FileClose(Farea);

    current_farea = 1;
    farea_rec = first_farea;

    memcpy(&farea,farea_rec,sizeof(farea));
    current_subarea = farea.subarea;
    farea_history[0] = '\0';

    return 1;
}

/* Release memory taken by file area records */
void deinit_fareas(void)
{
    while (first_farea != NULL)
    {
        farea_rec = first_farea->next;
        _free(first_farea);
        first_farea = farea_rec;
    }
}

/* Select new file area */
int select_farea(char *farea_name_ptr)
{
    FAREA_REC *closest;
    char old_history[256],farea_name_tmp[81],*farea_name;
    char *strp,*strp2;
    int spos;

    strcpy(farea_name_tmp,farea_name_ptr);
    farea_name = farea_name_tmp;

    strcpy(old_history, farea_history);

    strp = farea_name;
    while (farea_name != NULL)
    {
        farea_rec = first_farea;
        current_farea = 1;

        strp = strchr(farea_name,'/');
        if (strp != NULL) *strp = '\0';

        if (strcmp(farea_name,"..") == 0)
        {
            if (strp != NULL) strp++;

            /* Go to previous directory */
            strp2 = strrchr(farea_history,'/');
            if (strp2 != NULL)
            {
                if (strp2 == farea_history) strp2++;
                *strp2 = '\0';
                current_subarea = first_farea->subarea;
                select_farea(farea_history);

                if (strp == NULL || *strp == '\0')
                {
                    memcpy(&farea,farea_rec,sizeof(farea));
                    return 1;
                }
            }
        }
        else
        {
            if (strp == farea_name)
            {
                current_subarea = first_farea->subarea;
                spos = 0;
            }
            else
            {
                spos = strlen(farea_history);
            }
            if (strp != NULL) strp++;

            /* Update file area history */
            if (spos == 0 || farea_history[spos-1] != '/')
            {
                farea_history[spos] = '/';
                farea_history[spos+1] = '\0';
            }

            closest = NULL;
            while (farea_rec != NULL)
            {
                /* Same subarea? */
                if (current_subarea == farea_rec->subarea && farea_rec->flags & FAREA_FLAG_READ_RIGHTS)
                {
                    if (stricmp(farea_name,farea_rec->name) == 0)
                    {
                        /* Found */
                        current_subarea = farea_rec->jumparea;
                        strcat(farea_history, farea_rec->name);
                        if ((strp != NULL) && (*strp != '\0')) break;

                        memcpy(&farea,farea_rec,sizeof(farea));
                        return 1;
                    }
                    else
                    {
                        if (farea_name[0] != '\0' && strstr(farea_rec->name,farea_name) != NULL)
                        {
                            /* This one was close .. */
                            if (closest == NULL || strlen(closest->name) > strlen(farea_rec->name))
                            {
                                /* Whoa, closest one yet! */
                                closest = farea_rec;
                            }
                        }
                    }
                }
                farea_rec = farea_rec->next;
                current_farea++;
            }

            if (farea_rec == NULL && closest != NULL)
            {
                /* Select closest one.. */
                farea_rec = closest;

                current_subarea = farea_rec->jumparea;
                strcat(farea_history, farea_rec->name);
                if ((strp != NULL) && (*strp != '\0')) break;

                memcpy(&farea,farea_rec,sizeof(farea));
                return 1;
            }
        }
        farea_name = strp;
    }

    /* Not found */
    strcpy(farea_history, old_history);
    current_subarea = first_farea->subarea;
    select_farea(farea_history);
    return 0;
}

/* Select new file area */
int read_farea_record(unsigned long farea_num)
{
    /* Too big area number? */
    if (farea_num > file_areas || farea_num == 0) return 0;

    if (current_farea == 0 || farea_rec == NULL)
    {
        current_farea = 1;
        farea_rec = first_farea;
        if (farea_rec == NULL) return 0;
    }

    /* Go to right position */
    while (farea_num > current_farea)
    {
        farea_rec = farea_rec->next;
        if (farea_rec == NULL)
        {
            current_farea = 0;
            return 0;
        }
        current_farea++;
    }
    while (farea_num < current_farea)
    {
        farea_rec = farea_rec->prev;
        if (farea_rec == NULL)
        {
            current_farea = 0;
            return 0;
        }
        current_farea--;
    }

    memcpy(&farea, farea_rec, sizeof(farea));
    return 1;
}

unsigned search_path(char *path)
{
    unsigned rec;

    rec = 1;
    while (read_farea_record(rec) == 1)
    {
        if (stricmp(farea.path,path) == 0 && farea_rec->flags & FAREA_FLAG_READ_RIGHTS) return rec;
        rec++;
    }

    return 0;
}

/* Get path history of file area # */
char *get_history(unsigned long farea_num, char *history)
{
    unsigned long rec,subarea;
    FAREA_REC *fa_recs[50];
    int fa_pos;

    if (!read_farea_record(farea_num)) return NULL;

    history[0] = '\0'; subarea = farea.subarea; rec = 1;
    fa_recs[49] = farea_rec; fa_pos = 49;
    while (read_farea_record(rec) == 1)
    {
        if (farea.jumparea == subarea)
        {
            if (fa_pos == 0)
            {
                printf("get_history() overflow in area %lu\n", farea_num);
                write_log("get_history() overflow in area %lu", farea_num);
                return NULL;
            }
            fa_recs[--fa_pos] = farea_rec;
            if (farea.subarea == 0) break;
            subarea = farea.subarea;
            rec = 0;
        }
        rec++;
    }

    while (fa_pos < 50)
    {
        if (strcmp(fa_recs[fa_pos]->name, "/") != 0) strcat(history,fa_recs[fa_pos]->name);
        strcat(history,"/");
        fa_pos++;
    }

    return history;
}
