// Filename:	msqlmsg.C
// Contents:	the methods for the msqlmsg object
// Author:	Greg Shaw
// Created:	3/6/96

#ifndef _MSQLMSG_C_
#define _MSQLMSG_C_

#include "bbshdr.h"

#ifdef USE_DATABASE

/*
This file is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 2, or (at your option) any
later version.

In addition to the permissions in the GNU General Public License, the
Free Software Foundation gives you unlimited permission to link the
compiled version of this file with other programs, and to distribute
those programs without any restriction coming from the use of this
file.  (The General Public License restrictions do apply in other
respects; for example, they cover modification of the file, and
distribution when not linked into another program.)

This file is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */


// Function:	constructor
// Purpose:	initialize the object variables
// Input:	none
// Output:	object is initialized
// Author:	Greg Shaw
// Created:	3/6/96

msqlmsg::msqlmsg()
{
	msgpos = 0;
	msgresult = NULL;	// no message query yet
	msg_row = NULL;		// no current row
	msgrows = 0;		// no rows yet
	areas = NULL;
	current = NULL;
	dbsock = get_sock(DATABASE_NAME);
}

// Function:	destructor
// Purpose:	clean up the object
// Input:	none
// Output:	data areas are freed, sockets closed
// Author:	Greg Shaw
// Created:	3/6/96

msqlmsg::~msqlmsg()
{
	SectionHdr *here;
	SectionHdr *gone;

	for (here = areas; here != NULL; ) 
	{
		gone = here;
		here = here->next;
		free(gone);
	}
	areas = NULL;
	current = NULL;
}

// Function:	add_hdr
// Purpose:	add a section to the header cache
// Input:	hdr - the header to add
// Output:	0 for success, non zero for failure
// Author:	Greg Shaw
// Created:	3/6/96

int msqlmsg::add_hdr(SectionHdr *hdr)
{
	SectionHdr *ptr;

	clear_hdr(hdr);	// nuke old entry
	if (ptr = (SectionHdr *)malloc(sizeof(SectionHdr)), ptr == NULL)
	{
		ap_log("Unable to malloc new message header cache entry.");
		return(1);
	}
	memcpy(ptr,hdr,sizeof(SectionHdr));
	ptr->next = areas;
	areas = ptr;
	return(0);
}

// Function:	add_header
// Purpose:	add a section to the info table
// Input:	hdr - the header to add
// Output:	0 for success, non zero for failure
// Author:	Greg Shaw
// Created:	3/6/96
// Note:	it is assumed that all pertinent locks have already been
//		set prior to this function being called.

int    msqlmsg::add_header(SectionHdr *hdr)    // add header to info table
{
	char tmpstr[255];

	if (!set_lock(INFO_TABLE_NAME,TABLE_LOCK,time(NULL),username()))
	{
		ap_log("add_header: Unable to set table lock for info table.");
		return(0);
	}
	sprintf(tmpstr,"insert into %s values ('%s',%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%d,%ld,%ld,%d,'%s','%s','%s')",INFO_TABLE_NAME,
	hdr->table_name,hdr->high_message,hdr->high_thread,
	hdr->num_messages,hdr->section_type,hdr->acl,hdr->flags,hdr->modifier,
	hdr->read_only,hdr->anonymous,hdr->group,hdr->days,
	hdr->maxlines, hdr->tagline, hdr->date, hdr->last_import,
	hdr->high_import, hdr->external_name, hdr->moderator,hdr->long_name);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"add of header info record %s failed: %s",hdr->table_name,dberror());
		ap_log(tmpstr);
		return(0);
	}
	clear_lock(INFO_TABLE_NAME,TABLE_LOCK);
	return(1);
}


// Function:	add_msg
// Purpose:	add a message to the database
// Input:	table_name - name of the section to add the message to
//		newmsg - the message (structure)
//		lock - if true, set a lock.  If not, assume a lock has
//			already been set
// Output:	0 for failure, 1 for success
// Author:	Greg Shaw
// Created:	3/13/96

int msqlmsg::add_msg(char *table_name, Msg *newmsg, int lock) 
{
	char *tmpstr;
	char tmpstr2[4000];	// output

	if (lock)	// should we set a table lock?
		set_lock(table_name,TABLE_LOCK,time(NULL),username());
	// figure out how much header
	sprintf(tmpstr2,"insert into %s values (%d,'%s','%s','%s',%ld, '',%d,'%s',%d,%d)",
		table_name,newmsg->message_number, newmsg->from_,
		newmsg->to, newmsg->subject, newmsg->date,
		newmsg->link_number, newmsg->link_section, 
		newmsg->thread_number, newmsg->local_origin);
	// allocate storage for the new message (could be BIG!)
	if (tmpstr = (char *) malloc(strlen(tmpstr2)+strlen(newmsg->text)), tmpstr == NULL)
	{
		sprintf(tmpstr2,"msqlmsg: unable to allocate memory for message of size %d",250+strlen(newmsg->text));
		ap_log(tmpstr2);
		return(0);
	}
	// ok.  got enough room.  Let's do it.	
	sprintf(tmpstr,"insert into %s values (%d,'%s','%s','%s',%ld,'%s',%d,'%s',%d,%d)",
		table_name,newmsg->message_number, newmsg->from_,
		newmsg->to, newmsg->subject, newmsg->date, newmsg->text,
		newmsg->link_number, newmsg->link_section, 
		newmsg->thread_number, newmsg->local_origin);
	if (query(dbsock,tmpstr) == -1)
	{
		ap_log(tmpstr);
		sprintf(tmpstr2,"add of message to %s failed: %s",table_name,dberror());
		free(tmpstr);
		ap_log(tmpstr2);
		if (lock)	// clear lock.  no entry done
			clear_lock(table_name,TABLE_LOCK);
		return(0);
	}
	free(tmpstr);
	if (lock)	// clear lock.  no entry done
		clear_lock(table_name,TABLE_LOCK);
	return(1);
}


// Function:	build_thread_list
// Purpose:	build a list of threads
// Input:	glist - a list of sections to select list from
//		tlist - the thread list
// Output:	the number of threads found
// Author:	Greg Shaw
// Created:	3/6/96

int msqlmsg::build_thread_list(Group *glist, mllist *tlist) // build thread list
{
	Group	*gptr;	// iterator
	m_result *dbresult;	// query result
	m_row	cur_row;	// row from DB
	char 	selstr[255];	// selection string
	char	tmpstr[255];	// tmp string
	int	rows;		// number of threads returned
	Thread	trec;		// end of thread chain

	// figure out length of select for malloc
	rows = 0;
	for (gptr = glist; gptr != NULL; gptr=gptr->next)
	{
		sprintf(selstr,"select * from %s_thread",gptr->section);
		if (query(dbsock,selstr) == -1)
		{
			ap_log(selstr);
			sprintf(tmpstr,"Unable to query threads: %s",dberror());
			ap_log(tmpstr);
			return(0);
		}
		dbresult = storeresult();
		rows += numrows(dbresult);
		if (numrows(dbresult) == 0)	// no rows? skip
		{
			freeresult(dbresult);	// trash selection
			continue;
		}
		for (cur_row = fetchrow(dbresult); cur_row != NULL; cur_row = fetchrow(dbresult))
		{
			strcpy(trec.subject,cur_row[0]);
			strcpy(trec.section,cur_row[1]);
			sscanf(cur_row[2],"%d",&trec.thread_number);
			sscanf(cur_row[3],"%d",&trec.messages);
			// add to end of the list
			tlist->add(&trec);
		}
		freeresult(dbresult);	// trash selection
	}
	return(rows);
}

// Function:	clean_threads
// Purpose:	delete old threads from the threads table, and, update
//		the number of messages in the thread
// Input:	section - the section to update
// Output:	none
// Author:	Greg Shaw
// Created:	4/9/96

int msqlmsg::clean_threads(char *section)
{
	m_result *dbresult,*dbresult2;	// result of query pointer
	char tmpstr[255];	// query string
	char cursection[MAX_SECTION];	// current section
	m_row	cur_row;	// current row
	Thread	trec;		// current thread record

	// select table
	sprintf(cursection,"%s_thread",section);
	if (!set_lock(cursection,TABLE_LOCK,time(NULL),"expire"))
	{
		sprintf(tmpstr,"expire: Unable to set lock for %s table, skipped ...",cursection);
		ap_log(tmpstr);
		return(0);
	}
	sprintf(tmpstr,"select * from %s",cursection);
	// send query off...
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query thread table for expire: %s",dberror());
		ap_log(tmpstr);
		return(0);
	}
	// save it
	dbresult = storeresult();
	for (cur_row = fetchrow(dbresult); cur_row != NULL; cur_row = fetchrow(dbresult))
	{
		strcpy(trec.subject,cur_row[0]);
		strcpy(trec.section,cur_row[1]);
		sscanf(cur_row[2],"%d",&trec.thread_number);
		sscanf(cur_row[3],"%d",&trec.messages);
		// build expire string
		sprintf(tmpstr,"select message_number from %s where thread_number = %d",section,trec.thread_number);
		if (query(dbsock,tmpstr) == -1)
		{
			sprintf(tmpstr,"Unable to expire messages from section %s: %s",cursection,dberror());
			ap_log(tmpstr);
			clear_lock(cursection,TABLE_LOCK);
			continue;
		}
		dbresult2 = storeresult();
		// We've got the list of groups. Let's build the group list
		trec.messages = numrows(dbresult2);
		if (trec.messages == 0)	// no messages?  delete
		{
			delete_thread(cursection,&trec);
		}
		else
			save_thread(section, &trec,0);		// save thread
		freeresult(dbresult2);
		// clean up lock
	}
	// nuke the result.
	freeresult(dbresult);
	clear_lock(cursection,TABLE_LOCK);
	return(0);
}


// Function:	clear_hdr
// Purpose:	delete an entry from the header cache
// Input:	hdr - the header to delete
// Output:	0 for failure, 1 for success
// Author:	Greg Shaw
// Created:	3/6/96

int msqlmsg::clear_hdr(SectionHdr *hdr)
{
	SectionHdr *ptr,*ptr2;

	if (ptr = find_hdr(hdr->table_name), ptr != NULL)
	{
		// found, erase
		if (ptr == areas)	// at head of list
		{
			areas = areas->next;	// skip
			free(ptr);
		}
		else
		{
			for (ptr2 = areas; ptr2->next != ptr; ptr2=ptr2->next);
			ptr2->next = ptr->next;
			free(ptr);
		}
	}
	return(0);
}

// Function:	clear_message
// Purpose:	free the memory from a message
// Input:	msg - the message to free
// Output:	nonzero for failure
// Author:	Greg Shaw
// Created:	4/1/96

int msqlmsg::clear_message(Msg **msg)
{
	free((*msg)->text);	// nuke message text
	free(*msg);		// nuke message itself
	*msg = NULL;	
	return(0);
}

// Function:	clear_messages
// Purpose:	free the result of a message query
// Input:	none
// Output:	nonzero for failure
// Author:	Greg Shaw
// Created:	3/24/96

int msqlmsg::clear_messages(void)
{
	freeresult(msgresult);	// trash selection
	return(0);
}

// Function:	delete_header
// Purpose:	delete a header record from the info table in the database
// Input:	hdr - the header to delete
// Output:	0 for failure, 1 for success
// Author:	Greg Shaw
// Created:	3/6/96
// Note:	it is assumed that all pertinent locks have already been
//		set prior to this function being called.

int    msqlmsg::delete_header(SectionHdr *hdr) // delete header from info table
{
	char tmpstr[255];

	sprintf(tmpstr,"delete from %s where table_name = '%s'",
		INFO_TABLE_NAME,hdr->table_name);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"delete of header info record %s failed: %s",hdr->table_name,dberror());
		ap_log(tmpstr);
		return(0);
	}
	return(1);

}

// Function:	delete_mail
// Purpose:	delete all mail in user's mailbox
// Input:	section - the section to delete the message from
// Output:	0 for normal exit, 1 for failure
// Author:	Greg Shaw
// Created:	8/12/96

int msqlmsg::delete_mail(char *section)
{
	char tmpstr[255];

	sprintf(tmpstr,"delete from %s where to like '%s@%%' or to like '%s'",section,username(),username());
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"delete of mail failed: %s",dberror());
		ap_log(tmpstr);
		return(1);
	}
	return(0);
}

// Function:	delete_message
// Purpose:	delete a message from the database
// Input:	Msg - the message to delete
//		section - the section to delete the message from
// Output:	0 for normal exit, 1 for failure
// Author:	Greg Shaw
// Created:	3/21/96

int    msqlmsg::delete_message(char *section, Msg *msg, int has_thread) 
{
	char tmpstr[255];
	Thread *trec;

	sprintf(tmpstr,"delete from %s where message_number = %d",
		section,msg->message_number);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"delete of message failed: %s",dberror());
		ap_log(tmpstr);
		return(1);
	}
	if (has_thread)
	{
		// get the thread id
		if (trec = get_thread(section,msg->subject), trec == NULL)
		{
			ap_log("Unable to get thread record for message delete.");
			return(1);
		}
		// delete a message from the thread's list
		if (trec->messages)
			trec->messages--;
		if (trec->messages == 0)
			delete_thread(section,trec);
		save_thread(section,trec,1);
	}
	return(0);
}

// Function:	delete_thread
// Purpose:	delete a unused thread from the database
// Input:	t - the thread to delete
// Output:	0 for normal exit, 1 for failure
// Author:	Greg Shaw
// Created:	3/21/96

int    msqlmsg::delete_thread(char *section, Thread *t) 
{
	char tmpstr[255];

	sprintf(tmpstr,"delete from %s where thread_number = %d",
		section,t->thread_number);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"delete of message failed: %s",dberror());
		ap_log(tmpstr);
		return(1);
	}
	return(0);
}



// Function:	expire
// Purpose:	expire messages in all groups
// Input:	none
// Output:	none
// Author:	Greg Shaw
// Created:	4/8/96

void msqlmsg::expire(void)
{
	m_result *dbresult,*dbresult2;	// result of query pointer
	char tmpstr[255];	// query string
	char cursection[MAX_SECTION];	// current section
	SectionHdr hdr;	// found header
	m_row	cur_row;	// current row
	time_t	delday;		// time where articles expire
	int	num_mes;	// number of messages in section prior to delete
	int	deleted;	// number of messages deleted in section

	time(&delday);
	printf("Expire started: %s\n",ctime(&delday));
	printf("Section			Days to Expire    # Messages    # Deleted\n");
	printf("-------			--------------    ----------    ---------\n");
	// select table
	sprintf(tmpstr,"select * from %s",INFO_TABLE_NAME);
	// send query off...
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query info table for expire: %s",dberror());
		ap_log(tmpstr);
		return;
	}
	// save it
	dbresult = storeresult();
	for (cur_row = fetchrow(dbresult); cur_row != NULL; cur_row = fetchrow(dbresult))
	{
		// set lock
		strcpy(cursection,cur_row[0]);
		if (!set_lock(cursection,TABLE_LOCK,time(NULL),username()))
		{
			ap_log("Unable to set lock for expire.");
			continue;
		}
		strcpy(hdr.table_name,cur_row[0]);
		sscanf(cur_row[1],"%d",&hdr.high_message);
		sscanf(cur_row[2],"%d",&hdr.high_thread);
		sscanf(cur_row[3],"%d",&hdr.num_messages);
		sscanf(cur_row[4],"%d",&hdr.section_type);
		sscanf(cur_row[5],"%d",&hdr.acl);
		sscanf(cur_row[6],"%d",&hdr.flags);
		sscanf(cur_row[7],"%d",&hdr.modifier);
		sscanf(cur_row[8],"%d",&hdr.read_only);
		sscanf(cur_row[9],"%d",&hdr.anonymous);
		sscanf(cur_row[10],"%d",&hdr.group);
		sscanf(cur_row[11],"%d",&hdr.days);
		sscanf(cur_row[12],"%d",&hdr.maxlines);
		sscanf(cur_row[13],"%d",&hdr.tagline);
		sscanf(cur_row[14],"%ld",&hdr.date);
		sscanf(cur_row[15],"%ld",&hdr.last_import);
		sscanf(cur_row[16],"%d",&hdr.high_import);
		strcpy(hdr.external_name,cur_row[17]);	
		strcpy(hdr.moderator,cur_row[18]);	
		strcpy(hdr.long_name,cur_row[19]);	
		num_mes = hdr.num_messages;
		// should we expire?
		if (hdr.days > 0)
		{	// don't expire if < 1 day expiration
			delday = (time(NULL)-hdr.days*(60*60*24));	
			// build expire string
			sprintf(tmpstr,"delete from %s where date < %ld",cursection,delday);
			if (query(dbsock,tmpstr) == -1)
			{
				sprintf(tmpstr,"Unable to expire messages from section %s: %s",cursection,dberror());
				ap_log(tmpstr);
				clear_lock(cursection,TABLE_LOCK);
				continue;
			}
		}
		// OK.  messages deleted.  Update the number of messages.
		sprintf(tmpstr,"select message_number from %s",cursection);
		if (query(dbsock,tmpstr) == -1)
		{
			sprintf(tmpstr,"Unable to query messages in section %s: %s",cursection,dberror());
			ap_log(tmpstr);
			clear_lock(cursection,TABLE_LOCK);
			continue;
		}
		dbresult2 = storeresult();
		// We've got the list of groups. Let's build the group list
		hdr.num_messages = numrows(dbresult2);
		save_header(&hdr,0);		// save header information
		freeresult(dbresult2);
		deleted = num_mes - hdr.num_messages;
		if (deleted < 0)
			deleted = 0;	// none deleted
		printf("%-28.28s %2d              %2d             %2d\n",hdr.long_name,hdr.days,hdr.num_messages,deleted);
		// clean up lock
		clear_lock(cursection,TABLE_LOCK);
		// private message bases don't have thread tables
		if (hdr.section_type != PRIVATE_MESSAGE)
			clean_threads(cursection);	// delete unused threads
	}
	// nuke the result.
	freeresult(dbresult);
}


// Function:	find_hdr
// Purpose:	search the object cache for an object
// Input:	section - the name of the section
// Output:	header or NULL for failure
// Author:	Greg Shaw
// Created:	3/6/96

SectionHdr *msqlmsg::find_hdr(char *section)
{
	SectionHdr *ptr;

	for (ptr = areas; ptr != NULL && strcmp(ptr->table_name,section); ptr=ptr->next);
	return(ptr);
}



// Function:	get_bbs_group
// Purpose:	build a list of groups based on the group id.  
// Input:	glist - the list to save into
//		group - the group in the info table
// Output:	the number of groups found
// Author:	Greg Shaw
// Created:	3/22/96

int     msqlmsg::get_bbs_group(Group **glist, int group)        // get bbs group
{
	Group	*gptr;	// iterator
	Group	*gend=NULL;	// iterator
	m_result *dbresult;	// query result
	m_row	cur_row;	// row from DB
	char	tmpstr[255];	// tmp string
	int	groups;

	groups = 0;
	// build select statement
	sprintf(tmpstr,"select * from info where group=%d",group);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query info table: %s",dberror());
		ap_log(tmpstr);
		return(0);
	}
	dbresult = storeresult();
	// We've got the list of groups. Let's build the group list
	if (groups = numrows(dbresult), groups == 0)
	{	// no groups?
		*glist = NULL;
		return(0);
	}
	for (cur_row = fetchrow(dbresult); cur_row != NULL; cur_row = fetchrow(dbresult))
	{
		if (gptr = (Group *)malloc(sizeof(Group)), gptr == NULL)
		{
			ap_log("Unable to malloc group.");
			return(1);
		}
		sscanf(cur_row[0],"%s",gptr->section);
		gptr->group = group;
		gptr->high_message = 0;
		gptr->selected = 0;
		// add to end of the list
		gptr->next = NULL;
		if (*glist == NULL)	// first record?
		{
			*glist = gptr;
			gend = *glist;
		}
		else
		{	// add to end of list
			gend->next = gptr;
			gend = gptr;
		}
	}
	freeresult(dbresult);	// trash selection
	return(groups);

}



// Function:	get_messages
// Purpose:	get messages associated with a particular thread
// Input:	section - the section name
//		thread_number - the thread number to key from
//		high_message - the highest message the user has read from
//			(or 0 for don't use)
// Output:	less than zero for error, number of messages return otherwise
// Author:	Greg Shaw
// Created:	3/24/96

int msqlmsg::get_messages(char *section, int thread_number, int high_message)
{
	char tmpstr[255];


	// build query
	if (high_message != 0)
		sprintf(tmpstr,"select * from %s where message_number > %d and thread_number=%d order by message_number",section,high_message,thread_number);
	else
		sprintf(tmpstr,"select * from %s where thread_number=%d order by message_number ",section,thread_number);
	// do the query
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query table %s: %s",section,dberror());
		ap_log(tmpstr);
		return(-1);
	}
	msgresult = storeresult();
	msgpos = -1;	// at start of list
	msgrows = numrows(msgresult);
	return(msgrows);
}

// Function:	get_private_messages
// Purpose:	get messages addressed to the user
// Input:	section - the section name
//		highest - start at this number at first attempt
//		keep -- keep the database lookup result
// Output:	less than zero for error, number of messages return otherwise
// Author:	Greg Shaw
// Created:	4/14/96

int msqlmsg::get_private_messages(char *section, int keep)
{
	char tmpstr[255];

	sprintf(tmpstr,"select * from %s where to like '%s@%%' or to='%s' order by message_number",section,username(),username());
	// do the query
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query table %s: %s",section,dberror());
		ap_log(tmpstr);
		return(-1);
	}
	msgresult = storeresult();
	msgpos = -1;	// at start of list
	msgrows = numrows(msgresult);
	if (!keep)	// trash result if we don't care
		clear_messages();
	return(msgrows);
}

// Function:	get_tagline
// Purpose:	get a tagline for the end of a newly posted message
// Input:	tag - the tagline group to look for
//		tagline - storage for the tagline
// Output:	a tagline or a zero length string
// Author:	Greg Shaw
// Created:	5/22/96

int msqlmsg::get_tagline(int tag, char *tagline)
{
	m_result *dbresult;	// query result
	m_row	cur_row;	// row from DB
	char	tmpstr[255];	// query string
	int	taglines;	// number of taglines available
	int	thetagline; 	// the randomly selected tagline

	sprintf(tmpstr,"select * from %s where tag_id=%d",TAGLINE_TABLE_NAME,tag);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query tagline table: %s",dberror());
		ap_log(tmpstr);
		tagline[0] = 0;
		return(-1);
	}
	dbresult = storeresult();
	if (taglines = numrows(dbresult), taglines == 0)
	{	
		// none found.  return an empty tagline
		tagline[0] = 0;
		freeresult(dbresult);
		return(-1);
	}
	// OK.  there exists some list of taglines available.  let's pick a random one
	// set random seed
	srand(time(NULL));
	thetagline = rand()%taglines;
	// copy back our tagline
	dataseek(dbresult,thetagline);
	// now get the row
	cur_row = fetchrow(dbresult);
	if (cur_row == NULL)
	{
		sprintf(tmpstr,"Unable to get specific tagline: %s",dberror());
		ap_log(tmpstr);
		tagline[0] = 0;
		freeresult(dbresult);
		return(-1);
	}
	// add bbs name at top if there is enough room
	if (strlen(cur_row[1] + strlen(bbs_name())) < MAX_TAGLINE)
		sprintf(tagline,"%s\n\t%s",bbs_name(),cur_row[1]);
	else
		strcpy(tagline,cur_row[1]);
	freeresult(dbresult);
	return(0);
}

// Function:	get_thread
// Purpose:	get the thread record for a particular subject
// Input:	section - the section name
//		subject - the subject for the thread
// Output:	the thread record or NULL for no record found.
// Author:	Greg Shaw
// Created:	7/15/96

Thread *msqlmsg::get_thread(char *section, char *subject)
{
	char tmpstr[255];
	char	*c;		// counter
	m_result *dbresult;	// query result
	m_row	cur_row;	// row from DB
	char sub[MAX_SUBJECT+1];
	static Thread	trec;		// thread information
	Thread *res;


	// strip case
	for (c=sub; *subject != 0; *c++ = tolower(*subject++));  
	*c = 0;
	// now search for it
	sprintf(tmpstr,"select * from %s_thread where subject='%s'", section,sub);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query thread table %s_thread: %s",section,dberror());
		ap_log(tmpstr);
		return(NULL);
	}
	dbresult = storeresult();
	if (numrows(dbresult) == 1)
	{	// thread found, return
		cur_row = fetchrow(dbresult);
		strcpy(trec.subject,cur_row[0]);
		strcpy(trec.section,cur_row[1]);
		sscanf(cur_row[2],"%d",&trec.thread_number);
		sscanf(cur_row[3],"%d",&trec.messages);
		res = &trec;
	}
	else
		res = NULL;
	freeresult(dbresult);
	return(res);
}

// Function:	get_thread_number
// Purpose:	get the thread number for the subject or create a new
//		thread number.
// Input:	section - the section name
//		subject - the subject for the thread
// Output:	the existing thread number or a new thread number
// Author:	Greg Shaw
// Created:	3/6/96

int msqlmsg::get_thread_number(char *section, char *subject)
{
	char tmpstr[255];
	int	thread_id=0;	// (new) thread id
	SectionHdr hdr;	// section header 
	Thread	*trec;		// thread information


	if (trec = get_thread(section,subject), trec != NULL)
	{
		trec->messages++;
		save_thread(section,trec,1);
		thread_id = trec->thread_number;
	}
	else
	{	// create new thread
		// get section header
		// force it to get it from the database
		// set update lock
		// get header information
		get_header(section,&hdr,0);	// force header reread
		hdr.high_thread++;		// new thread number
		thread_id = hdr.high_thread;
		if (!set_lock(INFO_TABLE_NAME,hashpjw(section),time(NULL),username()))
		{
			ap_log("get_thread_number: Unable to set lock for info table.");
			return(-1);
		}
		save_header(&hdr,0);		// save header information
		clear_lock(INFO_TABLE_NAME,hashpjw(section));
		sprintf(tmpstr,"%s_thread",section);
		if (!set_lock(tmpstr,-1,time(NULL),username()))
		{
			ap_log("get_thread_number: Unable to set table lock for info table.");
			return(-1);
		}
		sprintf(tmpstr,"insert into %s_thread values ('%s','%s',%d,%d)",section,subject,section,thread_id,1);
		if (query(dbsock,tmpstr) == -1)
		{
			// clear db lock
			sprintf(tmpstr,"%s_thread",section);
			clear_lock(tmpstr,TABLE_LOCK);
			// log message
			sprintf(tmpstr,"Unable to insert into %s_thread: %s",section,dberror());
			ap_log(tmpstr);
			return(-1);
		}
		// clear lock
		sprintf(tmpstr,"%s_thread",section);
		clear_lock(tmpstr,TABLE_LOCK);
	}
	return(thread_id);
}


// Function:	get_header
// Purpose:	get the header information from the database
// Input:	section - the section name
// Output:	hdr - the header record
//		0 for success
//		non zero for failure
// Author:	Greg Shaw
// Created:	3/6/96

int msqlmsg::get_header(char *section, SectionHdr *hdr, int use_cache)     // get section information
{
	m_result *dbresult;	// result of query pointer
	char tmpstr[255];	// query string
	SectionHdr *fdr;	// found header
	m_row	cur_row;	// current row
	int	rows;		// number of rows returned

	// has it been cached?
	if (use_cache && (fdr = find_hdr(section), fdr != NULL))
	{
		memcpy(hdr,fdr,sizeof(SectionHdr));
		return(0);
	}
	// check for write lock on header
	if (lock_set(INFO_TABLE_NAME,hashpjw(section)))
	{
		sleep(LOCK_WAIT);
		if (lock_set(INFO_TABLE_NAME,hashpjw(section)))
		{	// sorry....
			ap_log("Unable to set read lock for info table.");

			return(-1);
		}
	}
	// build query
	sprintf(tmpstr,"select * from %s where table_name='%s'",INFO_TABLE_NAME,section);
	// send query off...
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query info table for section %s: %s",section,dberror());
		ap_log(tmpstr);
		return(1);
	}
	// save it
	dbresult = storeresult();
	// Ok.  let's process this beastie
	rows = numrows(dbresult);
	if (rows != 1)
	{
		if (rows == 0)
			sprintf(tmpstr,"unknown section %s referenced",section);
		else
			sprintf(tmpstr,"too many rows returned on info select for section %s",section);
		ap_log(tmpstr);
		return(1);
	}
	// get fields, individually
	// this is ugly, but there is not really any other way to do it.  <sigh>
	strcpy(hdr->table_name,section);	// cheat
	cur_row = fetchrow(dbresult);
	sscanf(cur_row[1],"%d",&hdr->high_message);
	sscanf(cur_row[2],"%d",&hdr->high_thread);
	sscanf(cur_row[3],"%d",&hdr->num_messages);
	sscanf(cur_row[4],"%d",&hdr->section_type);
	sscanf(cur_row[5],"%d",&hdr->acl);
	sscanf(cur_row[6],"%d",&hdr->flags);
	sscanf(cur_row[7],"%d",&hdr->modifier);
	sscanf(cur_row[8],"%d",&hdr->read_only);
	sscanf(cur_row[9],"%d",&hdr->anonymous);
	sscanf(cur_row[10],"%d",&hdr->group);
	sscanf(cur_row[11],"%d",&hdr->days);
	sscanf(cur_row[12],"%d",&hdr->maxlines);
	sscanf(cur_row[13],"%d",&hdr->tagline);
	sscanf(cur_row[14],"%ld",&hdr->date);
	sscanf(cur_row[15],"%ld",&hdr->last_import);
	sscanf(cur_row[16],"%d",&hdr->high_import);
	strcpy(hdr->external_name,cur_row[17]);	
	strcpy(hdr->moderator,cur_row[18]);	
	strcpy(hdr->long_name,cur_row[19]);	
	// nuke the result.
	freeresult(dbresult);
	add_hdr(hdr);
	return(0);
}

// Function:	get_message_number
// Purpose:	return the highest message (and increment) in a table
// Input:	section - the section name
// Output:	highest message + 1
// Author:	Greg Shaw
// Created:	3/6/96

int msqlmsg::get_message_number(char *section)
{
	SectionHdr hdr;

	// get header information
	get_header(section,&hdr,0);	// force header reread
	hdr.high_message++;		// new message number
	// set update lock
	if (!set_lock(INFO_TABLE_NAME,hashpjw(section),time(NULL),username()))
	{
		ap_log("Unable to set lock for info table.");
		return(-1);
	}
	save_header(&hdr,0);		// save header information
	clear_lock(INFO_TABLE_NAME,hashpjw(section));
	return(hdr.high_message);
}

// Function:	head
// Purpose:	return the head message in the query
// Input:	none
// Output:	the message (or NULL for no messages)
// Author:	Greg Shaw
// Created:	8/7/96

Msg *msqlmsg::head(void)
{
	// increase position
	msgpos = 0;
	// set position	 (just to be sure)
	dataseek(msgresult,msgpos);
	// now get the row
	msg_row = fetchrow(msgresult);
	if (msg_row == NULL)
	{
		msgpos--;	// can't go beyond end of list
		return(NULL);
	}
	return(rowtomsg(msg_row));
}

// Function:	iterate_header
// Purpose:	iterate the section headers
// Input:	num - the ordinal number in the list
// Output:	hdr - the 'next' header record
//		0 for normal exit, non zero for end of the list
// Author:	Greg Shaw
// Created:	4/11/96

int     msqlmsg::iterate_header(SectionHdr *hdr, int num, int include_private) // iterate headers
{
	static m_result *dbresult;	// result of query pointer
	m_row	cur_row;	// current row
	int rows;
	char	section[MAX_SECTION+1];
	char	tmpstr[255];	// selection string

	if (num == 0)	// 0 for select
	{
		if (!include_private)
			sprintf(tmpstr,"select table_name from %s where section_type <> %d",INFO_TABLE_NAME,PRIVATE_MESSAGE);
		else
			sprintf(tmpstr,"select table_name from %s",INFO_TABLE_NAME);
		if (query(dbsock,tmpstr) == -1)
		{
			sprintf(tmpstr,"Unable to query info table: %s",dberror());
			ap_log(tmpstr);
			return(0);
		}
		dbresult = storeresult();
		rows = numrows(dbresult);
		if (rows == 0)
			return(-1);
		cur_row = fetchrow(dbresult);
	}
	else
	{
		dataseek(dbresult,num);
		cur_row = fetchrow(dbresult);
		if (cur_row == NULL)	// end of list?
			return(-1);
	}
	strcpy(section,cur_row[0]);
	get_header(section,hdr,1);	// make sure section open
	return(0);
}

// Function:	list_messages
// Purpose:	list the messages in a particular section to the user
// Input:	section - the section to view
//		item - the item number to return
//		row - where to store the information
// Output:	(the list)
// Author:	Greg Shaw
// Created:	4/25/96

int msqlmsg::list_messages(char *section, int item, char row[3][255])
{
	static m_result *dbresult;	// result of query pointer
	m_row	cur_row;	// current row
	int	x;
	int 	rows;
	char	tmpstr[255];	// selection string

	if (item == -1)	// selection for -1
	{
		sprintf(tmpstr,"select from_,date,subject,message_number from %s where to like '%s@%%' or to='%s' order by message_number",section,username(),username());
		if (query(dbsock,tmpstr) == -1)
		{
			ap_log(tmpstr);	// dump query string (for debugging)
			sprintf(tmpstr,"List of messages in section %s failed: %s",section,dberror());
			ap_log(tmpstr);
			return(0);
		}
		dbresult = storeresult();
		rows = numrows(dbresult);
		return(rows);
	}
	dataseek(dbresult,item);
	cur_row = fetchrow(dbresult);
	if (cur_row == NULL)
	{
		freeresult(dbresult);	// trash selection
		return(0);
	}
	for (x=0; x<3; x++)
		strcpy(row[x],cur_row[x]);
	return(1);
}


// Function:	open
// Purpose:	make sure that a particular section is available
// Input:	section - the section name
// Output:	0 for failure, 1 for success
// Author:	Greg Shaw
// Created:	3/6/96
SectionHdr *msqlmsg::open(char *section)    // get section information
{
	SectionHdr	hdr;

	get_header(section,&hdr,1);	// make sure section open
	return(find_hdr(section));
}


// Function:	next
// Purpose:	return the next message in the query
// Input:	none
// Output:	the message (or NULL for no more messages)
// Author:	Greg Shaw
// Created:	3/24/96

Msg *msqlmsg::next(void)
{
	// increase position
	msgpos++;
	// set position	 (just to be sure)
	dataseek(msgresult,msgpos);
	// now get the row
	msg_row = fetchrow(msgresult);
	if (msg_row == NULL)
	{
		msgpos--;	// can't go beyond end of list
		return(NULL);
	}
	return(rowtomsg(msg_row));
	
}

// Function:	num_messages
// Purpose:	return the number of messages in a particular section
// Input:	section - the section name
// Output:	number of messages (or -1)
// Author:	Greg Shaw
// Created:	3/6/96

int msqlmsg::num_messages (char *section)
{
	m_result *dbresult;	// result of query pointer
	char tmpstr[255];	// query string
	int	rows;		// number of rows in table

	// build query
	sprintf(tmpstr,"select * from %s",section);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query table %s: %s",section,dberror());
		ap_log(tmpstr);
		return(-1);
	}
	dbresult = storeresult();
	rows = numrows(dbresult);
	freeresult(dbresult);
	return(rows);
}

// Function:	num_threads
// Purpose:	return the number of threads in a particular section
// Input:	section - the section name
// Output:	number of threads (or -1)
// Author:	Greg Shaw
// Created:	3/17/96

int msqlmsg::num_threads (char *section)
{
	m_result *dbresult;	// result of query pointer
	char tmpstr[255];	// query string
	int	rows;		// number of rows in table

	// build query
	sprintf(tmpstr,"select * from %s_thread",section);
	if (query(dbsock,tmpstr) == -1)
	{
		sprintf(tmpstr,"Unable to query table %s: %s",section,dberror());
		ap_log(tmpstr);
		return(-1);
	}
	dbresult = storeresult();
	rows = numrows(dbresult);
	freeresult(dbresult);
	return(rows);
}

// Function:	post
// Purpose:	post a message to the database
// Input:	section - the section to post to
//		newmsg - the message to post
//		priv - is it a private post?
// Output:	0 for success, non-zero for failure
// Author:	Greg Shaw
// Created:	3/24/96

int msqlmsg::post(char *section, Msg *newmsg, int priv)
{
	char tmpstr[255];

	// Ok.  we've got the message text.  Fill in the rest of it.
	time(&newmsg->date);
	// build Msg header
	if (!priv)
	{
		newmsg->thread_number = get_thread_number(section,newmsg->subject);
		if (newmsg->thread_number < 0)
		{
			ap_log("Unable to get new thread.");
			return(1);
		}
	}
	newmsg->message_number = get_message_number(section);
	if (newmsg->message_number < 0)
	{
		ap_log("Unable to get new message number.");
		return(1);
	}
	// now post this puppy
	if (!add_msg(section,newmsg,1))
	{
		sprintf(tmpstr,"add of new message to section %s failed.",section);
		ap_log(tmpstr);
		return(1);
	}
	return(0);
}


// Function:	previous
// Purpose:	return the previous message in the query
// Input:	none
// Output:	the message (or NULL for no more messages)
// Author:	Greg Shaw
// Created:	3/24/96

Msg *msqlmsg::previous(void)
{
	// go back two (as pointer is generally at start of next record)
	msgpos--;
	// set position	 (just to be sure)
	dataseek(msgresult,msgpos);
	// now get the row
	msg_row = fetchrow(msgresult);
	if (msg_row == NULL)
	{
		msgpos = -1;
		return(NULL);
	}
	return(rowtomsg(msg_row));
	
}


// Function:	rowtomsg
// Purpose:	convert a database row to a message
// Input:	row - the database row
// Output:	a message (or NULL for error)
// Author:	Greg Shaw
// Created:	3/25/96

Msg *msqlmsg::rowtomsg(m_row row)  // convert row to a message
{
	Msg	*newmsg;

	// get message memory
	if (newmsg = (Msg *)malloc(sizeof(Msg)), newmsg == NULL)
	{
		ap_log("Unable to allocate memory for a new message.");
		return(NULL);
	}
	if (newmsg->text = (char *)malloc(strlen(row[5])+1), newmsg->text == NULL)
	{
		ap_log("Unable to allocate memory for a new message text.");
		return(NULL);
	}
	sscanf(row[0],"%d",&newmsg->message_number);
	strcpy(newmsg->from_,row[1]);
	strcpy(newmsg->to,row[2]);
	strcpy(newmsg->subject,row[3]);
	sscanf(row[4],"%ld",&newmsg->date);
	strcpy(newmsg->text,row[5]);
	sscanf(row[6],"%d",&newmsg->link_number);
	strcpy(newmsg->link_section,row[7]);
	sscanf(row[8],"%d",&newmsg->thread_number);
	sscanf(row[9],"%d",&newmsg->local_origin);
	return(newmsg);
}


// Function:	save_header
// Purpose:	update header information in the info table
// Input:	hdr - the section header to update
//		full - update the whole record or only the most probable
//			items?
// Output:	1 if successful, 0 if failure
// Author:	Greg Shaw
// Created:	3/6/96
// Notes:	It is assumed that the calling function has set the
//		appropriate locks prior to updating!

int msqlmsg::save_header(SectionHdr *hdr, int full)
{
	char tmpstr[255];

	time(&hdr->date);	// date is always 'now'
	if (!set_lock(INFO_TABLE_NAME,TABLE_LOCK,time(NULL),username()))
	{
		ap_log("save_header: Unable to set table lock for info table.");
		return(0);
	}
	if (!full)	// full implies delete and re-add
	{
		sprintf(tmpstr,"update %s set high_message=%d, \
high_thread=%d, num_messages=%d, date=%ld where table_name = '%s'",
		INFO_TABLE_NAME,hdr->high_message,hdr->high_thread,
		hdr->num_messages,hdr->date,hdr->table_name);
		if (query(dbsock,tmpstr) == -1)
		{
			ap_log(tmpstr);
			sprintf(tmpstr,"update of header info record %s failed: %s",hdr->table_name,dberror());
			ap_log(tmpstr);
			return(0);
		}
	}
	else
	{
		// delete header record
		if (delete_header(hdr) != 1)
			return(0);
		// now add a new record
		if (add_header(hdr)!= 1)
			return(0);
	}
	clear_lock(INFO_TABLE_NAME,TABLE_LOCK);
	return(1);	
}

// Function:	save_thread
// Purpose:	update a thread record
// Input:	section - the section to save the thread to
//		t - the thread to save
//		lock - whether to set a lock
// Output:	1 if successful, 0 if failure
// Author:	Greg Shaw
// Created:	3/6/96

int msqlmsg::save_thread(char *section, Thread *t, int lock)
{
	char tmpstr[255];

	if (lock)
	{
		if (!set_lock(section,t->thread_number,time(NULL),username()))
		{
			ap_log("save_thread: Unable to set table lock for thread table.");
			return(0);
		}
	}
	sprintf(tmpstr,"update %s_thread set messages = %d where thread_number = %d",
		section,t->messages,t->thread_number);
	if (query(dbsock,tmpstr) == -1)
	{
		ap_log(tmpstr);
		sprintf(tmpstr,"update of thread record %s failed: %s",section,dberror());
		ap_log(tmpstr);
		return(0);
	}
	if (lock)
		clear_lock(section,t->thread_number);
	return(1);	
}

// Function:	scan
// Purpose:	scan a section for messages to the user
// Input:	sec - the section to scan
// Output:	the number of messages found
// Author:	Greg Shaw
// Created:	4/12/96

int msqlmsg::scan(SectionHdr *sec, time_t last_logon)
{
	m_result *dbresult;	// query result
	char	tmpstr[255];
	int	rows;

	// build query
	sprintf(tmpstr,"select message_number from %s where to like '%s@%%' or to='%s' and date > %ld ",sec->table_name,username(),username(),last_logon);
	if (query(dbsock,tmpstr) == -1)
	{
		ap_log(tmpstr);
		sprintf(tmpstr,"Scan of new messages in section %s failed: %s",sec->table_name,dberror());
		ap_log(tmpstr);
		return(0);
	}
	dbresult = storeresult();
	rows = numrows(dbresult);
	freeresult(dbresult);	// trash selection
	return(rows);
}

// Function:	tail
// Purpose:	return the tail message in the query
// Input:	none
// Output:	the message (or NULL for no messages)
// Author:	Greg Shaw
// Created:	8/7/96

Msg *msqlmsg::tail(void)
{
	// increase position
	msgpos = msgrows-1;
	// set position	 (just to be sure)
	dataseek(msgresult,msgpos);
	// now get the row
	msg_row = fetchrow(msgresult);
	if (msg_row == NULL)
	{
		msgpos--;	// can't go beyond end of list
		return(NULL);
	}
	return(rowtomsg(msg_row));
}


#endif // USE_DATABASE

#endif // _MSQLMSG_C_
