// Filename:	bbsmon.C
// Contents:	the methods for the bbsmon object
// Author:	Greg Shaw
// Created:	5/22/95

#ifndef _BBSMON_C_
#define _BBSMON_C_

#include "bbshdr.h"

extern User user;	// 'this' user

// Function:	constructor
// Purpose:	init the bbsmon object
// Author:	Gregory Shaw
// Created:	5/22/95

bbsmon::bbsmon()
{
	userlist = NULL;	
	lnow = 0.0;
	l5min = 0.0;
	l15min = 0.0;
	uptime = 0;
};


// Function:	destructor
// Purpose:	clean up the object
// Author:	Gregory Shaw
// Created:	5/22/95

bbsmon::~bbsmon()
{
	nuke_list();	// empty list
};


// Function:	add_user
// Purpose:	add a user record to the online users list
// Input:	user - user record
// Output:	the record is added to the list
// Author:	Gregory Shaw
// Created:	5/22/95

int bbsmon::add_user(BBSUser *user)
{
	BBSUser *tmp,*ptr;

	if (tmp = (BBSUser *)malloc(sizeof(BBSUser)), tmp == NULL)
	{
		ap_log("bbsmon: out of memory!");
		return(-1);
	}
	// copy it in
	memcpy(tmp,user,sizeof(BBSUser));
	tmp->next = NULL;
	// now add it to the list
	if (userlist == NULL)
	{	// start of list 
		userlist = tmp;
	}
	else
	{	// list has entries -- add to the end
		ptr = userlist;
		while (ptr->next != NULL)
			ptr = ptr->next;
		ptr->next = tmp;
	}
	return(0);
}

// Method: check_broadcast
// Purpose:    poll the broadcast server for any broadcast messages
// Input:  none
// Output: if a message is available, it will be returned
// Author: Gregory Shaw
// Created:    4/10/95

char *bbsmon::check_broadcast(void)
{
        static time_t  last_check = 0;
        static char tmpstr[255];

        // check only every 5 seconds, max
                                // check time expired?
        if (time(NULL) - last_check > CHECK_BROADCAST_TIME)
        {                           // check
                time(&last_check);
                if (broadipc.open_sock(chathost(),CHAT_BROADCAST_PORT) ==
0)
                {
                                // send 'me'
                        broadipc.send(user.logname(),1);
                                // send poll
                        broadipc.send(BROADCAST_POLL,1);
                                // got anything for me?
                        if (broadipc.receive(tmpstr,1) > 0 && strcmp(tmpstr,NO_BROADCAST))
                        {                   // yup.  send it back
                                broadipc.close_sock(0);
                                return(tmpstr);
                        }
                        // nope.  exit.
                        broadipc.close_sock(0);
                }
        }
        return(NULL);
}



// Function:	del_user
// Purpose:	delete a user record from the online users list
// Input:	user - user record
// Output:	the record is deleted from the list
// Author:	Gregory Shaw
// Created:	5/22/95

int bbsmon::del_user(BBSUser *bad)
{
	BBSUser *tmp;

	// find em
	tmp = userlist;
	// look for the record prior to the bad boy
	if (userlist == bad)	// first guy?
		userlist = bad->next;
	else
	{
		while (tmp->next != bad)
			tmp = tmp->next;
		// now update links
		tmp->next = bad->next;
	}
	free(bad);
	return(0);
}

// Function:	get_load
// Purpose:	get the load average figures from the /proc directory
// Input:	(load average figures from system)
// Output:	The object load averages are updated
// Author:	Gregory Shaw
// Created:	5/27/95

int bbsmon::get_load(void)
{
	FILE *loadfile;

	if (loadfile = bopen("/proc/loadavg","r"), loadfile == NULL)
	{
		ap_log("bbsmon: unable to open /proc/loadavg");
		return(1);
	}
        if (fscanf(loadfile,"%lf %lf %lf %*s",&lnow,&l5min,&l15min) != 3)
	{
		ap_log("bbsmon: unable to process /proc/loadavg");
		lnow = 0.0;
		l5min = 0.0;
		l15min = 0.0;
		return(1);
	}
        bclose(loadfile);
	return(0);
}

// Function:	get_uptime
// Purpose:	get the system uptime from the system
// Input:	(uptime from system)
// Output:	The object uptime figures are updated
// Author:	Gregory Shaw
// Created:	5/27/95

int bbsmon::get_uptime(void)
{
	FILE *upfile;
	double up;	// good pun, eh?

	if (upfile = bopen("/proc/uptime","r"), upfile == NULL)
	{
		ap_log("bbsmon: unable to open /proc/uptime");
		return(1);
	}
	if (fscanf(upfile,"%lf %*f",&up) != 1)
	{
		ap_log("bbsmon: unable to process /proc/uptime");
		uptime = 0;
		return(1);
	}
	uptime = (int) up;
	bclose(upfile);
	return(0);
}


// Function:	line_bbsmon
// Purpose:	watch the BBS in operation.  A top level console function.
// Input:	data from the BBS users (clients)
// Output:	the data is displayed on the screen
// Author:	Gregory Shaw
// Created:	5/22/95

int bbsmon::line_bbsmon(void)
{

	bbsipc ipcobj;		// bbsmon IPC object
	char *bcastmsg;             // broadcast
	char c;
	char tmpstr[255];  	// buffer
	char login[25];			// user login name
	char lastmessage[100];	    // last message displayed
	struct utsname unam;	    // uname information
	int  refresh;		    // refresh screen?
	int  lastpriority;	    // priority of current 'last message'
	int  x;
	int  days, hours, mins;	    // uptime calculations
	struct timeval waittime; // buffer
	time_t lasttime;	    // timestamp of 'last message'
	time_t now;                 // date
	time_t then;                 // timeouts
	BBSUser utmp;		// temp record
	BBSUser *ptr;		// iterator
	WINDOW *typing;    	// one window
	struct tm *onsince;	// on since update
	struct tm *lastup;	// last report update

	if (ipcobj.open_sock(NULL,10283) != 0)
	{
		sstrcr("Unable to open server socket.");
		waitcr();
		return(0);
        }

	ipcobj.close_sock(0);	
	return(0);
	
	lastpriority = 0;	// no 'last message' yet
	lastmessage[0] = 0;
	initscr();
	cbreak();                   // set cbreak (no wait) mode
	if (typing = newwin(LINES, COLS, 0, 0 ), typing == NULL)
	{ // create new window
		exit(0);
	}
	// Ok.  We've got windows.  Now set their modes
	clear();                    // clear the screen
	erase();                    // erase all windows
	nonl();                     // no cr+lf (and no cr to nl translation)
	noecho();		    // no echo
	wtimeout(typing,750);       // turn on timeout for getch (350ms)
	// *$#@(%@(#* curses.  Leave them signals ALONE
	signal(SIGINT,SIG_IGN);     // ignore interrupt
	time(&then);                 // get starting time
	refresh = 1;		// refresh the first time
	uname(&unam);	    // uname information
	while (c = wgetch(typing), c != ESC) // exit on ESC
	{
		// check input
		switch (tolower(c))
		{
			case ESC:		// escape?
			case ERR:
			{
				break;          // no input
			}
			case 'l':               // normal input
			{
				// lock screen
				break;
			}
		}
		// check for client messages
		if (ipcobj.do_connect(0) == 0)	// connection?
		{
			// send a poll
			if (ipcobj.send(BBSMON_POLL,1) != 0)
			{
				ipcobj.close_sock(1);
				continue;
			}
			if (ipcobj.receive(tmpstr,1) < 0)
			{	// error
				ipcobj.close_sock(1);	
				continue;
			}

			// temporary ints 
			int msgid;
			int info;
			int good=0;

			// digest message
			if (sscanf(tmpstr,"%d %d",&msgid,&info) != 2 || msgid != 2) 
			{
					ap_log("Bad message from bbsmon client:");
					ap_log(tmpstr);
					ipcobj.close_sock(1);
					continue;
			}
			switch(info)
			{
				case 0:	// nothing for me
					break;
				case 1:	// information follows
					// format: 
					// user host tty timelimit logon_time
					if (ipcobj.receive(tmpstr,1) == 0)
					{
						if (ipcobj.receive(tmpstr,1) == 0)
						{
							if (sscanf(tmpstr,"%s %s %s %d %ld",utmp.user,utmp.host,utmp.ttyname,&utmp.timelimit,&utmp.logon) != 5)
							{
								ap_log("bbsmon: Invalid message received");
							}
							// get 'what' line
							if
								(ipcobj.receive(utmp.doing,1) < 0)
							{	// error
								ipcobj.close_sock(1);	
							}
						}
					}
					break;
				default:	
					ap_log("bbsmon: Invalid message received");
					break;
			}
			// now update the list
			update_rec(&utmp);
			ipcobj.close_sock(1);	
			refresh++;	// refresh screen
		}
		// check for broadcasts
		if (bcastmsg = check_broadcast(), bcastmsg != NULL)
		{
			if (x = check_private(bcastmsg,0), x == 0)
			{
				if ((lastpriority <= 1) || (lastpriority > 1 && time(NULL) - lasttime > BBSMON_LASTTIMEOUT))	
				{	// update 'last message'
					lastpriority = 1;
					strcpy(lastmessage,bcastmsg);
					refresh++;
				}
			}
			else if (x == 2)
			{ // someone wants to chat                  
				if ((lastpriority <= 2) || (lastpriority > 2 && time(NULL) - lasttime > BBSMON_LASTTIMEOUT))	
				{	// update 'last message'
					lastpriority = 2;
					sscanf(bcastmsg,"%*s %s",login);
							// chop off ':'
					login[strlen(login)-1] = 0;
					sprintf(lastmessage,"%s requests chat",login);
					refresh++;
				}
			}
		}
		// refresh screen as necessary
		if (refresh || (time(NULL) - then > BBSMON_REFRESH))
		{	// update screen
			time(&then);
			refresh = 0;
			get_load();	// get load average
			get_uptime();	// get uptime
			wclear(typing);	// erase window
			werase(typing);	// erase window
			wmove(typing,0,0);
			sprintf(tmpstr,"Monitor-v%s for rocat         Press ESC to Exit, L to Lock \n",VERSION);
			waddstr(typing,tmpstr);
			
			days = uptime/86400;
			hours = (uptime - days*86400) / 3600;
			mins = (uptime - (days*86400) - (hours*3600)) / 60;
			time(&now);
			onsince = localtime(&now);
			sprintf(tmpstr,"Time: %.2d:%.2d host %s: Up %2d %s, %.2d:%.2d, Load %.2f, %.2f, %.2f\n",
				onsince->tm_hour,onsince->tm_min,unam.nodename,days,days>1?"days":"day",hours,mins,lnow,l5min,l15min);
			waddstr(typing,tmpstr);
			sprintf(tmpstr,"Last Message: %-62.62s\n",lastmessage);
			waddstr(typing,tmpstr);         // add pointer
			waddstr(typing,"tty  User     Host    On Since  TL    Update  What\n");
			if (userlist == NULL)
				waddstr(typing,"<idle>\n");
			else
			{
				ptr = userlist;
				while (ptr != NULL)
				{
					onsince = localtime(&ptr->logon);
					lastup = localtime(&ptr->last_report);
					sprintf(tmpstr,"%2s   %-8s %-8s %.2d:%.2d   %3d    %.2d:%.2d   %s",
					ptr->ttyname,ptr->user,ptr->host,onsince->tm_hour, onsince->tm_min,ptr->timelimit,lastup->tm_hour,lastup->tm_min,ptr->doing);
					waddstr(typing,tmpstr);         // add pointer
					ptr = ptr->next;
				}
			}
			// now update the window
			wrefresh(typing);
		}
		// linux monkeys with the waittime
		waittime.tv_usec = 950;     // 950msec
		waittime.tv_sec = 0;
		select(0,NULL,NULL,NULL,&waittime);
	}
	endwin();                   // go back to normal
	ipcobj.close_sock(0);	
	return(0);
};

// Function:	expire_users
// Purpose:	scan the user list to look for expired records
// Input:	none
// Output:	A record is deleted if:
//		1.  The current time is beyond their timelimit
//		This allows a removal if the user should exit.
// Author:	Gregory Shaw
// Created:	5/22/95

int bbsmon::expire_users(void)
{
	BBSUser *tmp,*tmp2;
	time_t off;

	tmp = userlist;
	while (tmp != NULL)
	{
		off = tmp->logon + (tmp->timelimit * 60);
		off += (off*5)/100;	// add 5% fudge to make sure
		if (time(NULL) > tmp->logon + (tmp->timelimit * 60))
		{	// he's outta here
			tmp2 = tmp;
			tmp = tmp->next;
			del_user(tmp2);
		}
		else
			tmp = tmp->next;
	}
	return(0);
}


// Function:	nuke_list
// Purpose:	delete all records in the list
// Author:	Gregory Shaw
// Input:	none
// Output:	list is freed
// Created:	5/23/95

int bbsmon::nuke_list(void)
{
	BBSUser *tmp,*tmp2;

	if (userlist != NULL)
	{
		tmp = userlist;
		while (tmp != NULL)
		{
			tmp2 = tmp;
			tmp = tmp->next;
			free(tmp2);
		}
		userlist = NULL;
	}
	return(0);
};

// Method: check_private
// Purpose:    check a message for a private chat request
// Input:  msg - the message to check
//		in_curses  - should I get out of curses?
//		respond - should I respond directly to a chat request?
// Output: 0 if chat not found (and message is normal)
//     1 if chat found and entered
//     2 if chat found and not entered
// Author: Gregory Shaw
// Created:    4/22/95

int bbsmon::check_private(char *msg, int respond)
{
	int chat;

	chat = 0;
	if (strstr(msg,SPECIAL_BROADCAST) != NULL)
	{	// chat request
			return(2);	// not entered (yet)
	}
	return(0);
}


// Function:	update_rec
// Purpose:	update an existing record or add to the list (if new)
// Author:	Gregory Shaw
// Input:	rec - the record to update against
// Output:	none
// Created:	5/23/95

int bbsmon::update_rec(BBSUser *rec)
{
	BBSUser *tmp;

	if (userlist == NULL)
	{
		add_user(rec);
	}
	else
	{	// check against list
		tmp = userlist;
		while (tmp != NULL && strcmp(tmp->user,rec->user))
			tmp = tmp->next;
		if (tmp == NULL)	// new record?
			add_user(rec);	// add
		else 	
		{	// update record
			strcpy(tmp->host,rec->host);	
			strcpy(tmp->user,rec->user);	
			strcpy(tmp->ttyname,rec->ttyname);	
			strcpy(tmp->doing,rec->doing);	
			tmp->logon = rec->logon;
			tmp->timelimit = rec->timelimit;
			tmp->last_report = time(NULL);
		}
	}
	return(0);
}


#endif // _BBSMON_C_
