"""
This module is a menu-door - its purpose is to launch other doors.
For added fanciness, menus can be organized into a HIERARCHY of menus.

TODO: Wait in "line" for a slot on a door
TODO: ANSI configurable
"""
import time
import Global
import os
import DoorDB
import Door

class MenuCommand:
    Logout = "Logout" # Exit the door
    ShowPopular = "Most Popular" # Show top doors
    Whois = "Who is online?" # Show online users
    AllCommands = [Logout, ShowPopular, Whois]
    AllCommandsLower = []
    for Command in AllCommands:
            AllCommandsLower.append(Command.lower())

class EntryType:
    Door = "Door"
    Menu = "Menu"
    Command = "Command"
    
def GetEntryType(Name):
    MappingDict = {"D":EntryType.Door, "M":EntryType.Menu, "C":EntryType.Command}
    if not Name:
        return None
    return MappingDict.get(Name[0].upper(), None)
    
class MenuEntry:
    def __init__(self, Type, Prompt, Target = None, Hotkey = None):
        self.Type = Type
        self.Prompt = Prompt
        self.Target = Target
        if not Target:
            self.Target = Prompt
        self.Hotkey = Hotkey
        
class MenuClass:
    def __init__(self, Name):
        self.Name = Name
        self.Entries = []
        self.KeyToEntry = {} # upper-case keys
    def AddEntry(self, Name):
        self.Entries.append(Name)
    def DebugPrint(self):
        print "-->Menu '%s'"%self.Name
        for Entry in self.Entries:
            print "(%s) %s :: %s :: %s"%(Entry.Hotkey, Entry.Type, Entry.Prompt, Entry.Target)
        print "<--End of menu '%s'"%self.Name
class MenuState:
    Select = 0
    Done = 1

class MenuDoor(Door.Door):
    def __init__(self, Session):
        Door.Door.__init__(self, Session)
        self.State = MenuState.Select
        if Session:
            self.DoorObject = Session.CurrentDoor
        else:
            self.DoorObject = None
        self.User = None
        self.Done = 0
    def LoadMenus(self, Verbose = 0):
        """
        Load menu definitions.  Menus should be .txt files in the "menus" subdirectory.
        """
        self.Menus = {} # lower-case keys
        MenuDir = os.path.join(Global.GetHomeDir(), "Menus")
        if not os.path.exists(MenuDir):
            # No menus are defined at all, so create a simple menu:
            self.BuildDefaultMenu()
        else:
            for FileName in os.listdir(MenuDir):
                (Name, Extension) = os.path.splitext(FileName)
                if Extension.upper() == ".TXT":
                    if Verbose:
                        print "Reading menu '%s'"%FileName
                    Menu = MenuClass(Name)
                    FullPath = os.path.join(MenuDir, FileName)
                    self.LoadMenu(Menu, FullPath, Verbose)
                    self.Menus[Menu.Name.lower()] = Menu
        self.RefineMenus(Verbose)
    def LoadMenu(self, Menu, Path, Verbose):
        """
        Each file line is a menu entry, with these bits:
        Hotkey, Type, Prompt, Target,  (only Type and Prompt are mandatory)
        
        """
        File = open(Path, "r")
        LineNumber = 0
        for FileLine in File.xreadlines():
            LineNumber += 1
            FileLine = FileLine.strip()
            if (not FileLine) or (FileLine[0]=="#"):
                continue
            Bits = FileLine.split("\t")
            if len(Bits)<3:
                if Verbose:
                    print "*"*40
                    print "* Skipping line %s, not enough tab-pieces"%LineNumber
                continue
            Type = GetEntryType(Bits[1])
            if not Type:
                if Verbose:
                    print "*"*40
                    print "** Skipping line %s, bad entry type '%s'"%(LineNumber, Bits[0])
                continue
            Hotkey = Bits[0].strip()
            if len(Bits)>=4:
                Target = Bits[3].strip()
            else:
                Target=None
            Entry = MenuEntry(Type, Bits[2].strip(), Target, Hotkey)
            Menu.AddEntry(Entry)
        File.close()
    def RefineMenus(self, Verbose = 0):
        "Validate our menus, choosing hotkeys as needed and deleting all bad entries"
        DoorList = Global.DoorDatabase.Doors
        for Menu in self.Menus.values():
            BaseLetters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ1234567890"
            Letters = list(BaseLetters)
            UsedLetters = {}
            if Verbose:
                print "Checking menu %s..."%Menu.Name
            for Entry in Menu.Entries[:]:
                if Entry.Type == EntryType.Command:
                    Entry.Target = Entry.Target.lower()
                    if Entry.Target.lower() not in MenuCommand.AllCommandsLower:
                        if Verbose:
                            print "*"*40
                            print "** Removing entry '%s' with bad command '%s'"%(Entry.Prompt, Entry.Target)
                        Menu.Entries.remove(Entry)
                        continue
                elif Entry.Type == EntryType.Door:
                    Door = Global.DoorDatabase.GetByName(Entry.Target)
                    if not Door:
                        if Verbose:
                            print "*"*40
                            print "** Removing entry '%s' with unknown door '%s'"%(Entry.Prompt, Entry.Target)
                        Menu.Entries.remove(Entry)
                        continue
                elif Entry.Type == EntryType.Menu:
                    OtherMenu = self.GetMenu(Entry.Target)
                    if not OtherMenu:
                        if Verbose:
                            print "*"*40
                            print "** Removing entry '%s' with unknown menu '%s'"%(Entry.Prompt, Entry.Target)
                        Menu.Entries.remove(Entry)
                        continue
                # Use the hotkey:
                if Entry.Hotkey:
                    if (Entry.Hotkey not in Letters) and BaseLetters.find(Entry.Hotkey)!=-1:
                        if Verbose:
                            print "*"*40
                            print "** Duplicate hotkey '%s' for entry '%s'"%(Entry.Hotkey, Entry.Prompt)
                        Entry.Hotkey = None
                    else:
                        UsedLetters[Entry.Hotkey] = 1
                        try:
                            Letters.remove(Entry.Hotkey)
                        except:
                            pass # was never there to begin with 
                # Hotkey assignment - Try letters from the prompt
                if Entry.Hotkey == None:
                    for Char in Entry.Prompt.upper():
                        if Char in Letters:
                            Letters.remove(Char)
                            Entry.Hotkey = Char
                            break
                # Ok, try whatever's left:
                if Entry.Hotkey == None:
                    if Letters:
                        Entry.Hotkey = Letters[0]
                        Letters = Letters[1:]
                    else:
                        # Too many damn entries!  Just drop it.
                        if Verbose:
                            print "*"*40
                            print "** Too many menu entries - dropped '%s'"%Entry.Prompt
                        Menu.remove(Entry)
                        continue
                Menu.KeyToEntry[Entry.Hotkey] = Entry
            if Verbose:
                print Verbose, type(Verbose)
                print "Refinement complete!"
                Menu.DebugPrint()
    def GetMenu(self, Name):
        Menu = self.Menus.get(Name.lower(), None)
        return Menu
    def BuildDefaultMenu(self):
        """
        Called if there are no menus defined - builds a (cheesy) menu
        """
        Menu = MenuClass("Main")
        self.Menus[Menu.Name.lower()] = Menu
        for Game in Global.DoorDatabase.Doors:
            if Game in [Global.DoorDatabase.DefaultDoor, Global.DoorDatabase.LoginDoor]:
                continue
            if Game.Nickname:
                Name = Game.Nickname
            else:
                Name = Game.Name
            Entry = MenuEntry(EntryType.Door, Name, Name)
            Menu.AddEntry(Entry)
        Menu.AddEntry(MenuEntry(EntryType.Command, MenuCommand.Logout, Hotkey = "*"))
        Menu.AddEntry(MenuEntry(EntryType.Command, MenuCommand.ShowPopular, Hotkey = "!"))
        Menu.AddEntry(MenuEntry(EntryType.Command, MenuCommand.Whois, Hotkey = "#"))
        self.Menus[Menu.Name] = Menu
        Menu.DebugPrint()
    def ShowWho(self):
        self.Write("\r\n\r\n@yCurrent users and what they're up to:\r\n")
        self.Write("@w-------------------------------------\r\n")
        List = []
        for Session in self.Session.server.Sessions:
            UserName = Session.GetUserName()
            DoorName = Session.GetActionName()
            List.append("@W%-30s @G%s"%(UserName, DoorName))
        for String in List:
            self.Write(String+"\r\n")
        self.Write("\r\n")
    def ShowTopGames(self):
        (TopPlays, TopTime) = Global.BigBrother.GetPopularGames()
        if not TopPlays:
            self.Write("\r\n\r\n@WThere don't seem to be any stats.\r\nMaybe you should play a game and fix that...\r\nCommand: ")
            return
        self.Write("\r\n\r\n@yMost popular games in the last week:\r\n")
        self.Write("@w-----------------------------------\r\n")
        for (Count, DoorName) in TopPlays:
            self.Write("@C%s @w(@Y%d @Wplays@w)\r\n"%(DoorName, Count))
        self.Write("\r\n")
    def Logoff(self):
        self.Done = 1
        self.State = MenuState.Done
        self.Write("\r\n@GThanks for visiting.\r\nCome again soon...@0\r\n")
        self.Session.Logoff()
        self.KeepRunning = 0
    def DisplayMenu(self, Menu):
        for Entry in Menu.Entries:
            if Entry.Type == EntryType.Command:
                Color1 = "@R"
                Color2 = "@r"
            elif Entry.Type == EntryType.Menu:
                Color1 = "@B"
                Color2 = "@b"
            else:
                Color1 = "@G"
                Color2 = "@g"
            Str = "%s[%s%s%s]@w - %s%s"%(Color2, Color1, Entry.Hotkey, Color2, Color1, Entry.Prompt)
            #Str = self.FormatStr(Str)
            self.Write(Str+"\r\n")
    def Flush(self):
        try:
            self.Socket.recv(1024)
        except:
            pass
    def ExecuteEntry(self, Entry):
        if Entry.Type == EntryType.Command:
            self.ExecuteCommandEntry(Entry.Target)
        elif Entry.Type == EntryType.Menu:
            self.ExecuteMenuEntry(Entry.Target)
        elif Entry.Type == EntryType.Door:
            self.ExecuteDoorEntry(Entry.Target)
    def ExecuteMenuEntry(self, Target):
        Menu = self.GetMenu(Target)
        if Menu:
            self.CurrentMenu = Menu
            self.DisplayMenu(self.CurrentMenu)
    def ExecuteDoorEntry(self, Target):
        Door = Global.DoorDatabase.GetByName(Target)
        if Door:
            self.Write("@0")
            self.Session.RunDoor(Door)
            self.Session.CurrentDoor = self.DoorObject
            self.Session.DoorFailed = 0 # Reset
            self.DisplayMenu(self.CurrentMenu)
    def ExecuteCommandEntry(self, Target):
        Target = Target.lower()
        if Target == MenuCommand.Logout.lower():
            self.Logoff()
        elif Target == MenuCommand.ShowPopular.lower():
            self.ShowTopGames()
        elif Target == MenuCommand.Whois.lower():
            self.ShowWho()
    def Run(self):
        # Load menus, and proceed to the first menu:
        self.LoadMenus(0)
        if self.Menus.has_key("main"):
            self.CurrentMenu = self.Menus["main"]
        else:
            self.CurrentMenu = self.Menus[self.Menus.keys()[0]] # arbitrary pick!
        self.DisplayMenu(self.CurrentMenu)
        self.Line = ""
        while (self.KeepRunning):
            self.Write("@WSelection @w(@G?@g for menu@w):@R")
            Char = self.GetKey()
            self.Write("\r\n")
            # Always available:
            if Char == "?":
                self.DisplayMenu(self.CurrentMenu)
                continue
            Entry = self.CurrentMenu.KeyToEntry.get(Char.upper())
            if Entry:
                self.ExecuteEntry(Entry)
            else:
                self.Flush()
##            if self.State == MenuState.Done:
##        
##                return
##            try:
##                Text = self.Socket.recv(1)
##            except:
##                Text = ""
##            if Text:
##                Char = Text[0].upper()
##                Function = self.FunctionMap.get(Char, None)
##                if Function:
##                    apply(Function)
##                elif Char == "\r":
##                    continue
##                else:
##                    Door = self.GameDict.get(Char)
##                    if Door:
##                        self.Socket.send(Char)
##                        self.Session.RunDoor(Door)
##                        self.Session.CurrentDoor = self.DoorObject
##                        self.Session.DoorFailed = 0 # Reset
##                        self.ShowMenu()
##                    else:
##                        self.ShowMenu()
##            else:
##                time.sleep(0.1)


def DumpleMenu(Session):
    Door = MenuDoor(Session)
    Door.Run()

# Command-line usage: Read and validate menus
if __name__ == "__main__":
    print "VERBOSE RUN>>>"
    Door = MenuDoor(None)
    Door.LoadMenus(1)
