<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Vote.inc.php                                                   11/24/2006
 * This script will retrieve all of the informaton related to the configured
 * voting informatoin that are found in VOTE.CFG.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2006 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-Vote.inc.php :: GetUserVote()
*
* This function retrieves the current user's response to the question. It is
* in the form of an integer, which corresponds to the number of the answer.
*
* @param integer $userno
* @param integer $question
* @return integer
*/
function GetUserVote($userno, $question) {
    if (!is_int($userno)) {
        $userno   = (integer) $userno;      // Userno must be an integer.
    }
    if (!is_int($question)) {
        $question = (integer) $question;    // Make sure the question is an integer.
    }
    
    if ($userno == 0) {                     // If a guest then there is no answer.
        return 0;
    }
    
    if ($question < 1) {
        return FALSE;
    }
    
    global $VA_SYSPATHS;                    // Include global variable.
    $VA_VOTE = GetVote();
    
    if ($question > $VA_VOTE[0]) {          // If the question is higher than
        return 0;                           // the number of questions, reutrn zero.
    }
    
    $file = $VA_SYSPATHS['data'] . '/VOTE' . $question . '.DAT';
    if (!is_file($file)) {                  // If the file is not found, then return zero.
        return 0;
    }

    // Determine if the user has a response first...
    if ((filesize($file) / 2) < $userno) {
        return 0;                           // ...if not, then return zero.
    }
    
    $fileString = ReadCfgFile($file);       // Read the file.
    $pos = $userno * 2 - 1;                 // Determine the user's position in the file.

    // Get the user's vote.
    $answer = GetInteger($fileString, $pos);
    
    return $answer;                         // Return the answer.
}


/**
* Va-Vote.inc.php :: GetVote()
*
* Retrieve all of the voting information and store it into a multi-dimensional
* array.
*
* Array format: $vote[1]['setting']
*               $vote[0] = Total number of questions.
*
* @return array
*/
function GetVote() {
    $fileString = ReadVote();
    if ($fileString === FALSE) {
        trigger_error(VA_Error(3));
    }

    $vote   = array();                      // Create the array.
    $offset = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $vote[$x]['question'] = GetString($fileString, 1 + $offset, 64);
        $vote[$x]['respno']   = GetInteger($fileString, 65 + $offset);
        $vote[$x]['resp1']    = GetString($fileString, 67 + $offset, 60);
        $vote[$x]['resp2']    = GetString($fileString, 127 + $offset, 60);
        $vote[$x]['resp3']    = GetString($fileString, 187 + $offset, 60);
        $vote[$x]['resp4']    = GetString($fileString, 247 + $offset, 60);
        $vote[$x]['resp5']    = GetString($fileString, 307 + $offset, 60);
        $vote[$x]['resp6']    = GetString($fileString, 367 + $offset, 60);
        $vote[$x]['resp7']    = GetString($fileString, 427 + $offset, 60);
        $vote[$x]['resp8']    = GetString($fileString, 487 + $offset, 60);
        $offset = 546 * $x;
    }

    $vote[0] = $x - 1;                      // Put the number of voting questions in position 0.

    return $vote;                           // Return the voting information.

}


/**
* Va-Vote.inc.php :: GetVoteStats()
*
* This function will calculate the total number of votes and determine the
* number of votes for each response.
*
* @param integer $question
* @return array
*/
function GetVoteStats($question) {
    if (!is_int($question)) {
        $question = (integer) $question;    // The question must be an integer.
    }
    
    global $VA_SYSPATHS;                    // Include global variable.
    $VA_VOTE = GetVote();
    
    if ($question < 0) {
        return FALSE;
    }
    
    if ($question > $VA_VOTE[0]) {
        return FALSE;
    }
    
    // Setup the array...
    for ($x = 0; $x <= 8; $x++) {
        $stats[$x] = 0;
    }
    
    $file = $VA_SYSPATHS['data'] . '/VOTE' . $question . '.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.
    $offset = 0;

    for ($x = 1; $offset < strlen($fileString); $x++) {
        $resp = GetInteger($fileString, 1 + $offset);    // Get the response for each user.
        if ($resp > 0) {
            $stats[0]++;                    // Increase the total number.
            $stats[$resp]++;                // Increase the response.
        }
        $offset = 2 * $x;                   // Increase the offset.
    }

    return $stats;                          // Return the results.
}


/**
* Va-Vote.inc.php :: PutUserVote()
*
* This function will save the user's response to the proper voteX.dat file.
*
* @param integer $userno
* @param integer $question
* @param integer $answer
* @return mixed
*/
function PutUserVote($userno, $question, $answer) {
    if (!is_int($userno)) {
        $userno   = (integer) $userno;      // Userno must be an integer.
    }
    if (!is_int($question)) {
        $question = (integer) $question;    // Make sure the question is an integer.
    }
    if (!is_int($answer)) {
        $answer   = (integer) $answer;      // Answer must be an integer.
    }
    
    if ($userno == 0) {                     // If a guest, then do not go further.
        return FALSE;
    }
    
    if ($question < 0) {
        return FALSE;
    }
    
    if ($answer < 0) {
        $answer = 0;
    }
    
    global $VA_SYSPATHS;                    // Include global variable.
    $VA_VOTE = GetVote();
    
    if ($question > $VA_VOTE[0]) {
        return FALSE;
    }
    
    // If the passed answer integer is higher than the highest possible, then
    // it will return FALSE and not update the data file.
    if ($answer > $VA_VOTE[$question]['respno']) {
        return FALSE;
    }

    $file = $VA_SYSPATHS['data'] . '/VOTE' . $question . '.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.

    $pos = $userno * 2 - 1;                 // Determine the user's position in the file.
    $fileString = PutInteger($fileString, $answer, $pos);
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the status of the file write.
}


/**
* Va-Vote.inc.php :: PutVote()
*
* This function will change the vote.cfg to match the configuration variable
* passed. It then returns the new file string. The configuration variable MUST
* be in the format of $variable[1]['setting'].
*
* @param string $fileString
* @param array $configVariable
* @return string
*/
function PutVote($fileString, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $length  = count($configVariable);      // Get the number of elements in the config variable.
    $pointer = 0;                           // Reset the pointer.
    $offset  = 0;

    if (isset($configVariable[0])) {
        $length--;
    }

    for ($x = 1; $pointer < $length; $x++) {
        $total = intval(strlen($fileString) / 547);
        if ($x > $total) {
            $fileString = $fileString . str_pad('', 547, chr(0));
        }

        if (isset($configVariable[$x])) {
            $pointer++;
            if (isset($configVariable[$x]['question'])) $fileString = PutString($fileString, $configVariable[$x]['question'], 1 + $offset, 64);
            if (isset($configVariable[$x]['respno']))   $fileString = PutInteger($fileString, $configVariable[$x]['respno'], 65 + $offset);
            if (isset($configVariable[$x]['resp1']))    $fileString = PutString($fileString, $configVariable[$x]['resp1'], 67 + $offset, 60);
            if (isset($configVariable[$x]['resp2']))    $fileString = PutString($fileString, $configVariable[$x]['resp2'], 127 + $offset, 60);
            if (isset($configVariable[$x]['resp3']))    $fileString = PutString($fileString, $configVariable[$x]['resp3'], 187 + $offset, 60);
            if (isset($configVariable[$x]['resp4']))    $fileString = PutString($fileString, $configVariable[$x]['resp4'], 247 + $offset, 60);
            if (isset($configVariable[$x]['resp5']))    $fileString = PutString($fileString, $configVariable[$x]['resp5'], 307 + $offset, 60);
            if (isset($configVariable[$x]['resp6']))    $fileString = PutString($fileString, $configVariable[$x]['resp6'], 367 + $offset, 60);
            if (isset($configVariable[$x]['resp7']))    $fileString = PutString($fileString, $configVariable[$x]['resp7'], 427 + $offset, 60);
            if (isset($configVariable[$x]['resp8']))    $fileString = PutString($fileString, $configVariable[$x]['resp8'], 487 + $offset, 60);
        }
        $offset = 547 * $x;
    }

    return $fileString;                     // Return the new file string.
}


/**
* Va-Vote.inc.php :: ReadVote()
*
* This function will read the vote.cfg file.
*
* @return string
*/
function ReadVote() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/VOTE.CFG';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Vote.inc.php :: WriteVote()
*
* This function will write a new vote.cfg file based on the file string
* passed to it. If the file string is not in the correct format, then your
* vote.cfg file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteVote($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/VOTE.CFG';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}


// Retrieve the voting information automatically.
$VA_VOTE = GetVote();

?>