<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Database.inc.php                                               07/16/2006
 * This script contains a set of functions that may be useful when dealing with
 * Virtual Advanced database groups, databases, and messages.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2005 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.


/**
* Va-Database.inc.php :: DbAccess()
*
* This function will determine if the user has access clearance for the specified
* database.
*
* @param array $dbInfo
* @param string $userNo
* @param mixed $userInfo
* @return boolean
*/
function DbAccess($dbInfo, $userNo = 0, $userInfo = '') {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    if ((!isset($dbInfo)) || (!isset($dbInfo['minreadsl']))) {
        return FALSE;
    }

    if (!is_array($userInfo)) {
        if ($userNo == 0) {
            global $USER;
            $user = $USER;
        } else {
            global $VA_SYSPATHS;            // Include the global variable.
            global $includeDir;
            include_once($includeDir . 'va-userfile.inc.php');
            $user = GetUser($userNo);
        }
    } else {
        $user = $userInfo;
    }
    $access = FALSE;                        // Set the default access value, false.
    
    // If the sysop, give access regardless.
    if ($user['securitylevel'] >= 255) {
        return TRUE;
    }
    
    // Check security level, age and the user access flags.
    if (($user['securitylevel'] >= $dbInfo['minreadsl']) &&
        ($user['age'] >= $dbInfo['agelimit']) &&
        (($dbInfo['accflags'] == '') ||
         (strpos($user['accflags'], $dbInfo['accflags']) !== FALSE))) {
        $access = TRUE;
    }

    return $access;                         // Return the access result.
}


/**
* Va-Database.inc.php :: DeleteMessage()
*
* This function will permanently delete the given message. It will delete any
* attached file as well. BE SURE TO RELOAD THE DATABASE INFO AFTER USING THIS
* FUNCTION SO THE NEW INFORMATION IS UPDATED.
*
* @param array $dbInfo
* @param integer $msgNo
* @return mixed
*/
function DeleteMessage(&$dbInfo, $msgNo) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($msgNo)) {
        $msgNo = (integer) $msgNo;
    }

    // Check to see if the message number is within the proper range.
    if (($msgNo > $dbInfo['msgno']) || ($msgNo < 1)) {
        return FALSE;
    }

    global $CONFIG;
    global $VA_SYSPATHS;                    // Include global variable.
    $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $datFile = $file . '.DAT';
    $binFile = $file . '.BIN';
    $qscFile = $file . '.QSC';

    $fileString = ReadCfgFile($datFile);    // Read the file string.
    $msgInfo = GetMessageInfo($dbInfo, $msgNo, $fileString);

    // If an attached file, then delete it. Non-reversible!
    if ($msgInfo['attfilename'] != '') {
        if ($CONFIG['hostname'] != '') {
            $file = ConvertDir($CONFIG['hostname'], $dbInfo['filepath']) . '/' . $msgInfo['attfilename'];
        } else {
            $file = $dbInfo['filepath'] . '/' . $msgInfo['attfilename'];
        }

        if ($msgInfo['attfilepath'] == '') {
            if (is_file($file)) {
                unlink($file);
            }
        } else {
            if ($CONFIG['hostname'] != '') {
                $userDir = ConvertUnc($VA_SYSPATHS['sysop'] . '/' . $USER['userno'] . '/' . $msgInfo['attfilename']);
            } else {
                $userDir = $VA_SYSPATHS['sysop'] . '/' . $USER['userno'] . '/' . $msgInfo['attfilename'];
            }
            if ($msgInfo['attfilepath'] == $userDir) {
                if ($CONFIG['hostname'] != '') {
                    $file = ConvertDir($CONFIG['hostname'], $msgInfo['attfilepath']);
                } else {
                    $file = $msgInfo['attfilepath'];
                }
            }
            if (is_file($file)) {
                unlink($file);
            }
        }
        $msgInfo['attfilesize'] = 0;
        $msgInfo['attfilename'] = '';
        $msgInfo['attfilepath'] = '';
    }

    // Remove the message from the .BIN file.
    $binFileString = ReadCfgFile($binFile);
    $binFileString = substr_replace($binFileString, '', $msgInfo['msgpos'] - 1, $msgInfo['msglength']);
    $status = WriteCfgFile($binFile, $binFileString);

    // Update all messages with new position information in the .DAT file.
    for ($x = $msgNo + 1; $x <= $dbInfo['msgno']; $x++) {
        $info = GetMessageInfo($dbInfo, $x, $fileString);
        $info['msgpos'] = $info['msgpos'] - $msgInfo['msglength'];
        $fileString = substr_replace($fileString, MakeLongInt($info['msgpos']), (($x - 1) * 582) + 479 - 1, 4);
    }

    // Remove the message information from the .DAT file.
    $fileString = substr_replace($fileString, '', ($msgNo - 1) * 582, 582);
    $status = WriteCfgFile($datFile, $fileString);

    $qscFileString = ReadCfgFile($qscFile);
    if ($qscFileString != FALSE) {
        $qscLength = intval(strlen($qscFileString) / 4);
        $offset = 0;
        // Update all user's pointers if they are now greater than the number of messages.
        for ($x = 1; $x <= $qscLength; $x++) {
            $pointer = GetLongInt($qscFileString, 1 + $offset);
            if (($pointer > $dbInfo['msgno'] - 1) && ($pointer < 32760)) {
                $pointer = $dbInfo['msgno'] - 1;
                $qscFileString = substr_replace($qscFileString, MakeLongInt($pointer), $offset, 4);
            }
            $offset = $offset + 4;
        }
        WriteCfgFile($qscFile, $qscFileString);
    }
    
    return $status;                         // Return the status.
}


/**
* Va-Database.inc.php :: EditMessage()
*
* This function allows you to change the selected message's subject and/or
* body. It will write the new message exactly as it is passed, so routing
* information, color codes, etc. would need to be part of the new message. At
* the end of the new message, it will write a diamond code with date/time
* information of the change.
*
* @param array $dbInfo
* @param integer $msgNo
* @param string $newSubject
* @param string $newBody
* @return mixed
*/
function EditMessage($dbInfo, $msgNo, $newSubject = '', $newBody = '') {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($msgNo)) {
        $msgNo      = (integer) $msgNo;
    }
    if (!is_string($newSubject)) {
        $newSubject = (string)  $newSubject;
    }
    if (!is_string($newBody)) {
        $newBody    = (string)  $newBody;
    }

    // Check to see if the message number is within the proper range.
    if (($msgNo > $dbInfo['msgno']) || ($msgNo < 1)) {
        return FALSE;
    }

    global $VA_SYSPATHS;                    // Include global variable.
    $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $datFile = $file . '.DAT';
    $binFile = $file . '.BIN';

    $fileString = ReadCfgFile($datFile);    // Read the file string.
    $msgInfo = GetMessageInfo($dbInfo, $msgNo, $fileString);

    if ($newSubject != '') {
        // Update the subject of the message.
        $msgSubject = rtrim(strip_tags($newSubject));
        if (strlen($msgSubject) > 64) {
            $msgSubject = substr($msgSubject, 0, 64);
        } else {
            $msgSubject = str_pad($msgSubject, 64);
        }
        $fileString = substr_replace($fileString, $msgSubject, (($msgNo - 1) * 582) + 269 - 1, 64);
    }

    if ($newBody != '') {
        // Get the message from the .BIN file.
        $fp = fopen($binFile, 'r');                         // Open the file.
        fseek($fp, $msgInfo['msgpos'] - 1);
        if ($msgInfo['msglength'] > 0) {                    // If the message length is greater than zero...
            $message = fread($fp, $msgInfo['msglength']);   // ...get the message, otherwise...
        } else {
            $message = '';                                  // ...the message is nothing.
        }
        fclose($fp);                                        // Close the file.

        // Update the body of the message.
        $msgBody = rtrim(wordwrap($newBody, 79, "\r\n", 1));
        $msgBody = $msgBody . "\r\n" . chr(4) . "0\r\n" . chr(4) . '0Last Edited: ' . date("D M d, Y H:i:s") . "\r\n";
        $msgLength  = strlen($msgBody);
        $msgPosDiff = $msgInfo['msglength'] - $msgLength;

        $binFileString = ReadCfgFile($binFile);
        $binFileString = substr_replace($binFileString, $msgBody, $msgInfo['msgpos'] - 1, $msgInfo['msglength']);
        $status = WriteCfgFile($binFile, $binFileString);

        // Update the current message with new length information.
        $fileString = substr_replace($fileString, MakeLongInt($msgLength), (($msgNo - 1) * 582) + 483 - 1, 4);

        // Update all messages with new position information in the .DAT file.
        for ($x = $msgNo + 1; $x <= $dbInfo['msgno']; $x++) {
            $info = GetMessageInfo($dbInfo, $x, $fileString);
            $info['msgpos'] = $info['msgpos'] - $msgPosDiff;
            $fileString = substr_replace($fileString, MakeLongInt($info['msgpos']), (($x - 1) * 582) + 479 - 1, 4);
        }
    }

    // Save the files.
    $status = WriteCfgFile($datFile, $fileString);

    return $status;                         // Return the status.
}


/**
* Va-Database.inc.php :: GetDbCount()
*
* This function will get the number of available databases in the selected
* topic. It will filter out those which the user does not have access to.
*
* @param integer $topicNo
* @return integer
*/
function GetDbCount($topicNo, $filter = 1) {
    if (!is_int($topicNo)) {
        $topicNo = (integer) $topicNo;
    }
    if (!is_bool($filter)) {
        $filter = (boolean) $filter;
    }

    if ($topicNo < 0) {
        trigger_error(VA_Error(2));
    }

    $topic = chr($topicNo + 64);

    global $VA_SYSPATHS;                    // Include the global variable.

    $file = $VA_SYSPATHS['main'] . '/DATABASE.CFG';
    if (!is_file($file)) {
        return 0;
    }
    
    $fileString = ReadCfgFile($file);       // Read the file.

    $position   = 75;                       // Set the starting position.
    $dbNo       = 0;

    for ($x = 0; $position < strlen($fileString); $x++) {
        $string = trim(strtoupper(substr($fileString, $position - 1, 1)));
        $position = $position + 131;
        if ($string == $topic) {
            $offset = 131 * $x;
            if ($filter) {
                $dbInfo['minreadsl'] = GetInteger($fileString, 78 + $offset);
                $dbInfo['agelimit']  = GetInteger($fileString, 82 + $offset);
                $dbInfo['accflags']  = GetAccFlags(GetLongInt($fileString, 84 + $offset));
                if (DbAccess($dbInfo)) {
                    $dbNo++;
                }
            } else {
                $dbNo++;
            }
        }
    }

    return $dbNo;                           // Return the database count.
}


/**
* Va-Database.inc.php :: GetDbInfo()
*
* This function will retrieve the variables for the selected database. You may
* optionally pass the filestring, which would greatly reduce execution time if
* you are to be searching the same database multiple times.
*
* @param integer $topicNo
* @param integer $db
* @param string $fileString
* @return array
*/
function GetDbInfo($topicNo, $db, $fileString = '') {
    if (!is_int($topicNo)) {
        $topicNo = (integer) $topicNo;
    }
    if (!is_int($db)) {
        $db = (integer) $db;
    }
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    if ($topicNo < 1) {
        $topicNo = 1;
    }
    if ($db < 1) {
        $db = 1;
    }

    $topic = chr($topicNo + 64);            // Convert the topic number into the topic letter.

    global $VA_SYSPATHS;                    // Include the global variable.

    if (trim($fileString == '')) {
        // Set the filename.
        $file = $VA_SYSPATHS['main'] . '/DATABASE.CFG';
        if (!is_file($file)) {
            return FALSE;
        }

        $fileString = ReadCfgFile($file);       // Read the file.
    }

    $dbInfo   = array();
    $dbInfo['topicno']    = '';
    $dbInfo['topic']      = '';
    $dbInfo['dbno']       = 0;
    $dbInfo['filename']   = '';
    $dbInfo['filepath']   = '';
    $dbInfo['desc']       = '';
    $dbInfo['private']    = 0;
    $dbInfo['dbgroup']    = '';
    $dbInfo['maxmsg']     = 0;
    $dbInfo['minreadsl']  = 0;
    $dbInfo['minwritesl'] = 0;
    $dbInfo['agelimit']   = 0;
    $dbInfo['accflags']   = 0;
    $dbInfo['tagline']    = 0;
    $dbInfo['dispname']   = 0;
    $dbInfo['randtitle']  = 0;
    $dbInfo['qwkconfno']  = 0;
    $dbInfo['moderator']  = 0;
    $dbInfo['validate']   = 0;
    $dbInfo['freqable']   = 0;
    $dbInfo['devicetype'] = 0;
    $dbInfo['dbtype']     = '';
    $dbInfo['extra']      = '';
    $dbInfo['msgno']      = 0;
    $dbInfo['msgcnt']     = 0;
    $dbInfo['networked']  = FALSE;

    $position = 75;
    $num      = 0;

    for ($x = 0; $position < strlen($fileString); $x++) {
        $string   = trim(strtoupper(substr($fileString, $position - 1, 1)));
        $position = $position + 131;
        if ($string == $topic) {
            $num++;
            if ($num == $db) {
                $offset = 131 * $x;
                $dbInfo['topicno']    = $topicNo;
                $dbInfo['topic']      = $topic;
                $dbInfo['dbno']       = $db;
                $dbInfo['filename']   = GetString($fileString, 1 + $offset, 8);
                $dbInfo['filepath']   = GetString($fileString, 9 + $offset, 32);
                if (dirname($dbInfo['filepath']) == '.') {
                    $dbInfo['filepath'] = $VA_SYSPATHS['main'] . '/' . $dbInfo['filepath'];
                }
                $dbInfo['desc']       = GetString($fileString, 41 + $offset, 32);
                $dbInfo['private']    = GetInteger($fileString, 73 + $offset);
                $dbInfo['dbgroup']    = GetString($fileString, 75 + $offset, 1);
                $dbInfo['maxmsg']     = GetInteger($fileString, 76 + $offset);
                $dbInfo['minreadsl']  = GetInteger($fileString, 78 + $offset);
                $dbInfo['minwritesl'] = GetInteger($fileString, 80 + $offset);
                $dbInfo['agelimit']   = GetInteger($fileString, 82 + $offset);
                $dbInfo['accflags']   = GetAccFlags(GetLongInt($fileString, 84 + $offset));
                $dbInfo['tagline']    = GetInteger($fileString, 88 + $offset);
                $dbInfo['dispname']   = GetInteger($fileString, 90 + $offset);
                $dbInfo['randtitle']  = GetInteger($fileString, 92 + $offset);
                $dbInfo['qwkconfno']  = GetInteger($fileString, 94 + $offset);
                $dbInfo['moderator']  = GetInteger($fileString, 96 + $offset);
                $dbInfo['validate']   = GetInteger($fileString, 98 + $offset);
                $dbInfo['freqable']   = GetInteger($fileString, 100 + $offset);
                $dbInfo['devicetype'] = GetInteger($fileString, 102 + $offset);
                $dbInfo['dbtype']     = GetString($fileString, 104 + $offset, 1);
                $dbInfo['extra']      = GetString($fileString, 105 + $offset, 27);
                $dbInfo['msgno']      = GetMessageCount($dbInfo);
                if ($dbInfo['msgno'] == '') $dbInfo['msgno'] = 0;
                break;
            }
        }
    }

    if ((isset($dbInfo['filename'])) && (trim($dbInfo['filename']) != '')) {
        $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'] . '.NET';
        if ((is_file($file)) && (filesize($file) > 0)) {
            $dbInfo['networked'] = TRUE;
        } else {
            $dbInfo['networked'] = FALSE;
        }
    }

    return $dbInfo;                         // Return the database information.
}


/**
* Va-Database.inc.php :: GetDbNetworks()
*
* This function will get any network information for the database passed to it
* and return the results in an array.
*
* @param array $dbInfo
* @return array
*/
function GetDbNetworks($dbInfo) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }

    global $VA_SYSPATHS;

    $networks[0] = 0;

    $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'] . '.NET';
    if (!is_file($file)) {
        return $networks;
    }
    $fileArray = file($file);

    for ($x = 0; $x < count($fileArray); $x++) {
        $info = explode('=', $fileArray[$x]);
        if ((trim($fileArray[$x]) != '') && (count($info) !== FALSE)) {
            $networks[$x + 1]['network'] = $info[0];
            $networks[$x + 1]['id']      = $info[1];
        }
    }

    $networks[0] = $x;

    return $networks;                       // Return the array results.
}


/**
* Va-Database.inc.php :: GetEmailCount()
*
* This function will determine the number of messages contained within the
* email database for the specified user. It returns an array containing
* the number of messages total (msgno) and the number of messages that have not
* been read (unread) and the number of deleted messages (deleted).
*
* If deleted is set to 0 (false), then it will only return message numbers of
* messages that are not marked for deletion.
*
* It also returns the email message numbers starting with $count[0].
*
* @param integer $userNo
* @return array
*/
function GetEmailCount($userNo, $deleted = 1) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }
    if (!is_bool($deleted)) {
        $deleted = (boolean) $deleted;
    }

    $count['msgno']    = 0;                 // Set the default value.
    $count['msgtotal'] = 0;
    $count['unread']   = 0;
    $count['deleted']  = 0;

    if ($userNo < 1) {
        return $count;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $datFile = $VA_SYSPATHS['db'] . '/EMAIL.DAT';
    if (!is_file($datFile)) {
        return $count;
    }

    $fileString = ReadCfgFile($datFile);    // Read the file.

    $counter = 0;
    $offset = 0;
    for ($x = 0; $offset < filesize($datFile); $x++) {
        $offset = 582 * $x;
        $user = GetLongInt($fileString, 5 + $offset);
        if ($user == $userNo) {
            $delFlag = GetInteger($fileString, 333 + $offset);
            if ($delFlag == 1) {
                $count['deleted']++;
            }
            $readFlag = GetLongInt($fileString, 375 + $offset);
            if ($readFlag == 0) {
                $count['unread']++;
            }
            if ($deleted) {
                $count[$counter] = $x + 1;
                $counter++;
                $count['msgno']++;
            } else {
                if ($delFlag == 0) {
                    $count[$counter] = $x + 1;
                    $counter++;
                    $count['msgno']++;
                }
            }
            $count['msgtotal']++;
        }
    }

    return $count;                          // Return the message count.
}


/**
* Va-Database.inc.php :: GetLastPostInfo()
*
* This function will get the last post's message information of the
* specified database.
*
* @param array $dbInfo
* @return array
*/
function GetLastPostInfo($dbInfo) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }

    $lastPost = GetMessageInfo($dbInfo, $dbInfo['msgno']);

    return $lastPost;                       // Return the lastpost information.
}


/**
* Va-Database.inc.php :: GetMessage()
*
* This function will retrieve the message for the selected information. You may
* set $format to 1 if you would like to strip html tags or 2 for no formatting.
*
* @param array $dbInfo
* @param integer $msgNo
* @param integer $format
* @return string
*/
function GetMessage($dbInfo, $msgNo, $format = 0, $fileString = '', $msgInfo = array()) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($msgNo)) {
        $msgNo = (integer) $msgNo;
    }
    if (!is_int($format)) {
        $format = (integer) $format;
    }
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }
    if (!is_array($msgInfo)) {
        trigger_error(VA_Error(1));
    }

    if ($msgNo < 1) {
        $msgNo = 1;
    }
    if ($msgNo > $dbInfo['msgno']) {
        return '';
    }
    if (intval($format) > 2) {
        $format = 1;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file    = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $binFile = $file . '.BIN';
    if (!is_file($binFile)) {
        return '';
    }

    if (count($msgInfo) == 0) {
        $msgInfo = GetMessageInfo($dbInfo, $msgNo);
    }

    if (trim($fileString) == '') {
        $fp = fopen($binFile, 'r');                         // Open the file.
        fseek($fp, $msgInfo['msgpos'] - 1);
        if ($msgInfo['msglength'] > 0) {                    // If the message length is greater than zero...
            $message = fread($fp, $msgInfo['msglength']);   // get the message, otherwise...
        } else {
            $message = '';                                  // the message is nothing.
        }
        fclose($fp);                                        // Close the file.
    } else {
        if ($msgInfo['msglength'] > 0) {
            $message = substr($fileString, $msgInfo['msgpos'] - 1, $msgInfo['msglength']);
        } else {
            $message = '';
        }
    }

    if ($format == 1) {                             // HTML formatting...
        $message = rtrim(preg_replace("/(=20)||(=09)/", '', strip_tags($message)));
    } else if ($format == 2) {
        $message = rtrim($message);
    } else {
        $message = rtrim(str_replace('&amp;', '&', htmlspecialchars($message)));
    }

    while ((substr($message, 0, 1) == "\n") || (substr($message, 0, 1) == "\r")) {
        $message = substr($message, 1);
    }

    // Detect MIME header and only display text version.
    $pos  = strpos(ltrim($message), "\r\n");
    $line = substr(ltrim($message), 0, $pos);
    if (is_int(strpos($line, 'MIME'))) {
        $tempMessage = ltrim(substr(ltrim($message), $pos + 1));
    } else {
        $tempMessage = ltrim($message);
    }
    $line = substr($tempMessage, 0, 2);
    if ($line == '--') {
        $pos  = strpos($tempMessage, "\r\n");
        $line = ltrim(substr($tempMessage, $pos + 1));
        if (strtolower(substr($line, 0, 8)) == 'content-') {
            $mimehdr = ltrim(substr($tempMessage, 0, $pos));
            $pos = strpos($tempMessage, "\r\n\r\n");
            $message = ltrim(substr($tempMessage, $pos + 1));
            $pos = strpos($message, $mimehdr);
            $message = trim(substr($message, 0, $pos));
        }
    }

    return $message;                        // Return the message.
}


/**
* Va-Database.inc.php :: GetMessageCount()
*
* This function will determine the number of messages contained within the
* specified database. If $includeDeleted is set to FALSE, then any messages
* marked as deleted will not be counted.
*
* @param array $dbInfo
* @param boolean $includeDeleted
* @return integer
*/
function GetMessageCount($dbInfo, $includeDeleted = 1) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_bool($includeDeleted)) {
        $includeDeleted = (boolean) $includeDeleted;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];

    $datFile = $file . '.DAT';          // Add the datafile extension.
    if (!is_file($datFile)) {
        return 0;
    }

    // If $includeDeleted is true, then just calculate the number of messages,
    // otherwise, view each message entry and check the deleted status.
    if ($includeDeleted) {
        $fileSize = filesize($datFile);         // Determine the size of the file.
        $msgNo    = intval($fileSize / 582);    // Determine the number of messages.
    } else {
        $msgNo = 0;
        $fileString = ReadCfgFile($datFile);   // Read the database.
        for ($x = 1; $x <= intval(filesize($datFile) / 582); $x++) {
            $message = GetMessageInfo($dbInfo, $x, $fileString);
            if ($message['deleted'] == 0) {
                $msgNo++;
            }
        }
    }

    return $msgNo;                          // Return the message count.
}


/**
* Va-Database.inc.php :: GetMessageInfo()
*
* This function will retrieve the variables for the specified message. You may
* pass the filestring as well, which would greatly reduce the execution time
* if you do multiple searches.
*
* @param array $dbInfo
* @param integer $msgNo
* @param string $fileString
* @return array
*/
function GetMessageInfo($dbInfo, $msgNo, $fileString = '') {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($msgNo)) {
        $msgNo = (integer) $msgNo;
    }
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    if ($msgNo < 1) {
        $msgno = 1;
    }
    if ($msgNo > $dbInfo['msgno']) {
        return FALSE;
    }

    global $VA_SYSPATHS;                    // Include global variable.
    global $includeDir;
    global $USER;

    if (trim($fileString) == '') {
        // Set the filename.
        $file    = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];

        $datFile = $file . '.DAT';              // Set the database filename.
        if (!is_file($datFile)) {
            return FALSE;
        }

        $fileString = ReadCfgFile($datFile);    // Read the file.
    }

    $msgInfo = array();
    $offset  = 582 * ($msgNo - 1);
    $msgInfo['topicno']     = $dbInfo['topicno'];
    $msgInfo['topic']       = $dbInfo['topic'];
    $msgInfo['dbno']        = $dbInfo['dbno'];
    $msgInfo['fromuserno']  = GetLongInt($fileString, 1 + $offset);
    $msgInfo['touserno']    = GetLongInt($fileString, 5 + $offset);
    $msgInfo['fromhandle']  = GetString($fileString, 9 + $offset, 40);
    $msgInfo['tohandle']    = GetString($fileString, 49 + $offset, 40);
    $msgInfo['fromnetnode'] = GetString($fileString, 89 + $offset, 80);
    $msgInfo['tonetnode']   = GetString($fileString, 169 + $offset, 80);
    $msgInfo['fromnetid']   = GetInteger($fileString, 249 + $offset);
    $msgInfo['tonetid']     = GetInteger($fileString, 251 + $offset);
    $msgInfo['timestamp']   = GetLongInt($fileString, 253 + $offset);
    $msgInfo['daysold']     = intval((time() - $msgInfo['timestamp']) / (60 * 60 * 24));
    $msgInfo['sysmsgno']    = GetLongInt($fileString, 257 + $offset);
    $msgInfo['threadbk']    = GetLongInt($fileString, 261 + $offset);
    $msgInfo['threadbkn']   = GetLongInt($fileString, 265 + $offset);
    $msgInfo['subject']     = GetString($fileString, 269 + $offset, 64);
    $msgInfo['deleted']     = GetInteger($fileString, 333 + $offset);
    $msgInfo['creation']    = GetString($fileString, 335 + $offset, 40);
    $msgInfo['date']        = GetString($fileString, 335 + $offset, 17);
    $msgInfo['time']        = GetString($fileString, 352 + $offset, 9);
    $msgInfo['timezone']    = GetString($fileString, 361 + $offset, 14);
    $msgInfo['offline']     = GetLongInt($fileString, 375 + $offset);
    $msgInfo['attfilesize'] = GetLongInt($fileString, 379 + $offset);
    $msgInfo['attfilename'] = GetString($fileString, 383 + $offset, 12);
    $msgInfo['attfilepath'] = GetString($fileString, 395 + $offset, 64);
    $msgInfo['lib']         = GetLongInt($fileString, 459 + $offset);
    $msgInfo['downloadno']  = GetLongInt($fileString, 463 + $offset);
    $msgInfo['dbid']        = GetLongInt($fileString, 467 + $offset);
    $msgInfo['filename']    = GetString($fileString, 471 + $offset, 8);
    $msgInfo['msgpos']      = GetLongInt($fileString, 479 + $offset);
    $msgInfo['msglength']   = GetLongInt($fileString, 483 + $offset);
    $msgInfo['originnet']   = GetInteger($fileString, 487 + $offset);
    $msgInfo['fidoflags']   = GetInteger($fileString, 489 + $offset);
    $msgInfo['originnode']  = GetLongInt($fileString, 491 + $offset);
    $msgInfo['reserved']    = GetString($fileString, 495 + $offset, 24);
    $msgInfo['extra']       = GetString($fileString, 519 + $offset, 64);

    // Fix any missing information...
    if ($msgInfo['fromhandle'] == '') {
        if ($msgInfo['fromuserno'] == 0) {
            $msgInfo['fromhandle'] = 'Unknown';
        } else {
            $msgInfo['fromhandle'] = $msgInfo['fromuserno'] . '@' . $msgInfo['fromnetnode'];
        }
    }
    if ($msgInfo['tohandle'] == '') {
        if ($msgInfo['touserno'] == 0) {
            $msgInfo['tohandle'] = 'All';
        } else {
            $msgInfo['tohandle'] = $msgInfo['touserno'] . '@' . $msgInfo['tonetnode'];
        }
    }

    return $msgInfo;                        // Return the message information.
}


/**
* Va-Database.inc.php :: GetMessagePointer()
*
* This function will return the specified user's message pointer for the
* current database.
*
* @param array $dbInfo
* @param integer $userNo
* @return integer
*/
function GetMessagePointer($dbInfo, $userNo, $fileString = '') {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    if ($userNo < 1) {
        return 32760;
    }
    
    global $VA_SYSPATHS;                    // Include the global variable.

    $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'] . '.QSC';
    if ($fileString == '') {
        if (!is_file($file)) {
            return 0;
        }
        $fileString = ReadCfgFile($file);
    }

    $offset = ($userNo - 1) * 4;
    $pointer = GetLongInt($fileString, 1 + $offset);
    
    return $pointer;                        // Return the message pointer.
}


/**
* Va-Database.inc.php :: GetOnelinerCount()
*
* This function will determine the number of messages contained within the
* onelner database for the specified user. It returns an array containing
* the number of messages total (msgno) and the number of messages that have not
* been read (unread) and the number of deleted messages (deleted).
*
* @param integer $userNo
* @return integer
*/
function GetOnelinerCount($userNo) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    $count['msgno']   = 0;                  // Set the default value.
    $count['unread']  = 0;
    $count['deleted'] = 0;

    if ($userNo < 1) {
        return $count;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $datFile = $VA_SYSPATHS['db'] . '/ONELINE.DAT';
    if (!is_file($datFile)) {
        return $count;
    }

    $fileString = ReadCfgFile($datFile);    // Read the file.

    $offset = 582;
    for ($x = 0; $offset < filesize($datFile); $x++) {
        $offset = 582 * $x;
        $user = GetLongInt($fileString, 5 + $offset);
        if ($user == $userNo) {
            $delFlag = GetInteger($fileString, 333 + $offset);
            if ($delFlag == 1) {
                $count['deleted']++;
            }
            $readFlag = GetLongInt($fileString, 375 + $offset);
            if ($readFlag == 0) {
                $count['unread']++;
            }
            $count['msgno']++;
        }
    }

    return $count;                          // Return the message count.
}


/**
* Va-Database.inc.php :: GetSentEmailCount()
*
* This function will determine the number of messages contained within the
* sent email database for the specified user. It returns an array containing
* the number of messages total (msgno) and the number of messages that have not
* been read (unread) and the number of deleted messages (deleted).
*
* If deleted is set to 0 (false), then it will only return message numbers of
* messages that are not marked for deletion.
*
* It also returns the email message numbers starting with $count[0].
*
* @param integer $userNo
* @return array
*/
function GetSentEmailCount($userNo, $deleted = 1) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }
    if (!is_bool($deleted)) {
        $deleted = (boolean) $deleted;
    }

    $count['msgno']    = 0;                 // Set the default value.
    $count['msgtotal'] = 0;
    $count['unread']   = 0;
    $count['deleted']  = 0;

    if ($userNo < 1) {
        return $count;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $datFile = $VA_SYSPATHS['db'] . '/SNTEMAIL.DAT';
    if (!is_file($datFile)) {
        return $count;
    }

    $fileString = ReadCfgFile($datFile);    // Read the file.

    $counter = 0;
    $offset = 0;
    for ($x = 0; $offset < filesize($datFile); $x++) {
        $offset = 582 * $x;
        $user = GetLongInt($fileString, 1 + $offset);
        if ($user == $userNo) {
            $delFlag = GetInteger($fileString, 333 + $offset);
            if ($delFlag == 1) {
                $count['deleted']++;
            }
            $readFlag = GetLongInt($fileString, 375 + $offset);
            if ($readFlag == 0) {
                $count['unread']++;
            }
            if ($deleted) {
                $count[$counter] = $x + 1;
                $counter++;
                $count['msgno']++;
            } else {
                if ($delFlag == 0) {
                    $count[$counter] = $x + 1;
                    $counter++;
                    $count['msgno']++;
                }
            }
            $count['msgtotal']++;
        }
    }

    return $count;                          // Return the message count.
}


/**
* Va-Database.inc.php :: GetTopicCount()
*
* This function will determine the total number of topics and return that value.
*
* @param string $type
* @return integer
*/
function GetTopicCount($type = '') {
    if (!is_string($type)) {
        $type = (string) $type;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    $topicNo = 0;

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/DBGROUP.CFG';
    if (!is_file($file)) {
        return $topicNo;
    }

    $fileString = ReadCfgFile($file);       // Read the file.

    $type = trim(strtoupper($type));        // Change the topic type into uppercase.

    $position   = 2;                        // Set the starting position.

    for ($x = 0; $position < strlen($fileString); $x++) {
        $string = trim(strtoupper(substr($fileString, $position - 2, 1)));
        if ((($type == '') || (($type != '') && ($type == $string))) && ($string != '')) {
            $group = trim(strtoupper(substr($fileString, $position - 1, 1)));
            $group = ord($group) - 64;
            if (GetDbCount($group) > 0) {
                $topicNo++;
            }
        }
        $position = $position + 128;
    }

    return $topicNo;                        // Return the topic count.
}


/**
* Va-Database.inc.php :: GetTopicInfo()
*
* This function will retrieve the variables for the selected database topic.
* You can pass the db type to restrict which databases are searched. You may
* also pass the filestring, which will greatly reduce execution time if you
* do multiple searches.
*
* @param integer $topicNo
* @param string $type
* @param string $fileString
* @return array
*/
function GetTopicInfo($topicNo, $type = '', $fileString = '') {
    if (!is_int($topicNo)) {
        $topicNo = (integer) $topicNo;
    }
    if (!is_string($type)) {
        $type = (string) $type;
    }
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    if ($topicNo < 1) {
        $topicNo = 1;
    }

    global $VA_SYSPATHS;                    // Include global variable.

    if (trim($fileString) == '') {
        // Set the filename.
        $file = $VA_SYSPATHS['main'] . '/DBGROUP.CFG';
        if (!is_file($file)) {
            return FALSE;
        }

        $fileString = ReadCfgFile($file);       // Read the file.
    }

    $type  = trim(strtoupper($type));       // Change the type to uppercase.
    $topic = chr($topicNo + 64);            // Change the topic number into the topic letter.

    $topicInfo = array();
    $topicInfo['topicno']     = 0;
    $topicInfo['topic']       = '';
    $topicInfo['dbtype']      = '';
    $topicInfo['topic']       = '';
    $topicInfo['desc']        = '';
    $topicInfo['futuretopic'] = '';
    $topicInfo['marker1']     = 0;
    $topicInfo['marker2']     = 0;
    $topicInfo['extra']       = '';
    $topicInfo['dbno']        = 0;

    $position  = 2;                         // Set the starting position.

    for ($x = 0; $position < strlen($fileString); $x++) {
        $dbType = substr($fileString, $position - 2, 1);
        if (($type == '') || (($type != '') && ($type == $dbType))) {
            $string = trim(strtoupper(substr($fileString, $position - 1, 1)));
            if ($string == $topic) {
                $offset = 128 * $x;
                $topicInfo['topicno']     = $topicNo;
                $topicInfo['topic']       = $topic;
                $topicInfo['dbtype']      = GetString($fileString, 1 + $offset, 1);
                $topicInfo['topic']       = GetString($fileString, 2 + $offset, 1);
                $topicInfo['desc']        = GetString($fileString, 3 + $offset, 65);
                $topicInfo['futuretopic'] = GetString($fileString, 68 + $offset, 2);
                $topicInfo['marker1']     = GetLongInt($fileString, 70 + $offset);
                $topicInfo['marker2']     = GetLongInt($fileString, 74 + $offset);
                $topicInfo['extra']       = GetString($fileString, 78 + $offset, 51);
                $topicInfo['dbno']        = GetDBCount($topicNo);
                break;
            }
        }
        $position = $position + 128;
    }

    return $topicInfo;                      // Return the topic information.
}


/**
* Va-Database.inc.php :: GetTopics()
*
* This function will retrieve the topic numbers and store them into an array.
*
* @param string $type
* @return array
*/
function GetTopics($type = '') {
    if (!is_string($type)) {
        $type = (string) $type;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    $topics[0] = 0;

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/DBGROUP.CFG';
    if (!is_file($file)) {
        return $topics;
    }

    $fileString = ReadCfgFile($file);       // Read the file.

    $type = trim(strtoupper($type));        // Change the topic type into uppercase.

    $position  = 2;                         // Set the starting position.

    for ($x = 0; $position < strlen($fileString); $x++) {
        $string = trim(strtoupper(substr($fileString, $position - 2, 1)));
        if ((($type == '') || (($type != '') && ($type == $string))) && ($string != '')) {
            $group = trim(strtoupper(substr($fileString, $position - 1, 1)));
            $group = ord($group) - 64;
            if (GetDbCount($group) > 0) {
                array_push($topics, $group);
            }
        }
        $position = $position + 128;
    }

    $topics[0] = count($topics) - 1;
    
    return $topics;                        // Return the topic numbers.
}


/**
* Va-Database.inc.php :: GetTotalMessageCount()
*
* This function will return the total number of database entries.
*
* @return integer
*/
function GetTotalMessageCount() {
    global $VA_SYSPATHS;                    // Include the global variable.

    $file = $VA_SYSPATHS['data'] . '/DBSEQ.DAT';
    if (!is_file($file)) {
        return 0;
    }

    $fileString = ReadCfgFile($file);       // Read the file.
    return GetLongInt($fileString, 1);      // Return the total message count.
}


/**
* Va-Database.inc.php :: PutDbInfo()
*
* This function will take the database information that is passed and create
* a new entry into the database.cfg file.
*
* @param array $dbInfo
* @return boolean
*/
function PutDbInfo($dbInfo) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['main'] . '/DATABASE.CFG';
    if (!is_file($file)) {
        return FALSE;
    }

    $newString = str_pad('', 131, chr(0));  // Create the new blank string.

    $newString = PutString($newString, $dbInfo['filename'], 1, 8);
    $newString = PutString($newString, $dbInfo['filepath'], 9, 32);
    $newString = PutString($newString, $dbInfo['desc'], 41, 32);
    $newString = PutInteger($newString, $dbInfo['private'], 73);
    $newString = PutString($newString, $dbInfo['dbgroup'], 75, 1);
    $newString = PutInteger($newString, $dbInfo['maxmsg'], 76);
    $newString = PutInteger($newString, $dbInfo['minreadsl'], 78);
    $newString = PutInteger($newString, $dbInfo['minwritesl'], 80);
    $newString = PutInteger($newString, $dbInfo['agelimit'], 82);
    $newString = PutLongInt($newString, MakeAccFlags($dbInfo['accflags']), 84);
    $newString = PutInteger($newString, $dbInfo['tagline'], 88);
    $newString = PutInteger($newString, $dbInfo['dispname'], 90);
    $newString = PutInteger($newString, $dbInfo['randtitle'], 92);
    $newString = PutInteger($newString, $dbInfo['qwkconfno'], 94);
    $newString = PutInteger($newString, $dbInfo['moderator'], 96);
    $newString = PutInteger($newString, $dbInfo['validate'], 98);
    $newString = PutInteger($newString, $dbInfo['freqable'], 100);
    $newString = PutInteger($newString, $dbInfo['devicetype'], 102);
    $newString = PutString($newString, $dbInfo['dbtype'], 104, 1);
    $newString = PutString($newString, $dbInfo['extra'], 105, 27);

    $fileString = ReadCfgFile($file);           // Read the file.
    $fileString = $fileString . $newString;     // Append the data to the end.
    $status = WriteCfgFile($file, $fileString); // Write the file.

    return $status;                         // Return the write status.
}


/**
* Va-Database.inc.php :: QuoteMessage()
*
* This function will take the passed message and add > to the beginning of each
* line and truncate each line to 78 characters. This is used when quoting a
* message a user is replying to.
*
* @param string $message
* @return string
*/
function QuoteMessage($message) {
    if (!is_string($message)) {
        $message = (string) $message;
    }

    $message = preg_replace("/\r\n|\n|\r/", "\r\n", StripBBSCodes($message));
    $message = explode("\r\n", $message);
    for ($x = 0; $x < count($message); $x++) {
        if (strlen(rtrim($message[$x])) >= 77) {
            $message[$x] = substr(rtrim($message[$x]), 0, 77);
        }
        $message[$x] = rtrim($message[$x]);
    }
    $message = "\r\n\r\n> " . implode("\r\n> ", $message) . "\r\n";
    
    return $message;
}


/**
* Va-Database.inc.php :: SearchMessagesFrom()
*
* This function will scan the current database for all messages from the
* current user and store them into an array.
*
* @param array $dbInfo
* @param integer $userNo
* @return array
*/
function SearchMessagesFrom($dbInfo, $userNo) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    if ($userNo == 0) {
        $msgFrom[1] = '';
        return $msgFrom;
    }

    global $includeDir;
    include_once($includeDir . 'va-networks.inc.php');  // Include network information.

    // This function has been optimized for speed and does not use the pre-
    // written functions above.
    global $VA_SYSPATHS;                    // Include global variable.

    // Set the filename.
    $file    = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $datFile = $file . '.DAT';              // Set the database filename.

    if (!is_file($datFile)) {
        return FALSE;
    }

    $netInfo    = GetNetworkInfo();
    $fileString = ReadCfgFile($datFile);    // Read the database file.

    $cnt     = 1;                           // Counter.
    $msgFrom = array();

    for ($msgNo = 1; $msgNo <= $dbInfo['msgno']; $msgNo++) {
        $offset = 582 * ($msgNo - 1);
        $msgInfo['fromuserno']  = GetLongInt($fileString, 1 + $offset);
        if ($msgInfo['fromuserno'] == $userNo) {
            $msgInfo['fromnetid'] = GetInteger($fileString, 249 + $offset);
            if ((isset($netInfo[$msgInfo['fromnetid']]['netno'])) &&
               ($msgInfo['fromnetid'] == $netInfo[$msgInfo['fromnetid']]['netno'])) {
                $msgInfo['fromnetnode'] = GetString($fileString, 89 + $offset, 80);
                if ($msgInfo['fromnetnode'] == $netInfo[$msgInfo['fromnetid']]['settings']['systemid']) {
                    $msgFrom[$cnt] = $msgNo;    // Add the message number to the array.
                    $cnt++;                     // Increase the counter by one.
                }
            }
        }
    }

    $msgFrom[0] = $cnt - 1;                 // Set the total number of messages.

    return $msgFrom;                        // Return the messages from this user.
}


/**
* Va-Database.inc.php :: SearchMessagesTo()
*
* This function will scan the current database for all messages to the current
* user and store them into an array.
*
* @param array $dbInfo
* @param integer $userNo
* @return array
*/
function SearchMessagesTo($dbInfo, $userNo) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    if ($userNo == 0) {
        $msgTo[1] = '';
        return $msgTo;
    }

    global $includeDir;
    include_once($includeDir . 'va-networks.inc.php');  // Include network information.

    // This function has been optimized for speed and does not use the pre-
    // written functions above.
    global $VA_SYSPATHS;                    // Include global variable.

    // Set the filename.
    $file    = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];

    $datFile = $file . '.DAT';              // Set the database filename.
    if (!is_file($datFile)) {
        return FALSE;
    }

    $netInfo    = GetNetworkInfo();
    $fileString = ReadCfgFile($datFile);    // Read the database file.

    $cnt     = 1;                           // Counter.
    $msgTo   = array();

    for ($msgNo = 1; $msgNo <= $dbInfo['msgno']; $msgNo++) {
        $offset = 582 * ($msgNo - 1);
        $msgInfo['touserno']  = GetLongInt($fileString, 5 + $offset);
        if ($msgInfo['touserno'] == $userNo) {
            $msgInfo['tonetid'] = GetInteger($fileString, 251 + $offset);
            if ((isset($netInfo[$msgInfo['tonetid']]['netno'])) &&
               ($msgInfo['tonetid'] == $netInfo[$msgInfo['tonetid']]['netno'])) {
                $msgInfo['tonetnode'] = GetString($fileString, 169 + $offset, 80);
                if ($msgInfo['tonetnode'] == $netInfo[$msgInfo['tonetid']]['settings']['systemid']) {
                    $msgTo[$cnt] = $msgNo;  // Add the message number to the array.
                    $cnt++;                 // Increase the counter by one.
                }
            }
        }
    }

    $msgTo[0] = $cnt - 1;                   // Set the total number of messages.

    return $msgTo;                          // Return the messages from this user.
}


/**
* Va-Database.inc.php :: SetEmailInfo()
*
* This function will return the email database information in the same format
* as dbInfo.
*
* @param integer $userNo
* @return array
*/
function SetEmailInfo($userNo) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }
    
    global $CONFIG;
    global $VA_SYSPATHS;
    global $VA_MAIN;
    global $includeDir;
    include_once($includeDir . 'va-main.inc.php');

    $emailInfo['topicno']    = '';
    $emailInfo['topic']      = 'Email';
    $emailInfo['dbno']       = 0;
    $emailInfo['filename']   = 'EMAIL';
    if ($CONFIG['filehost'] != '') {
        $emailInfo['filepath'] = ConvertUnc($VA_SYSPATHS['sysop']);
    } else {
        $emailInfo['filepath'] = $VA_SYSPATHS['sysop'];
    }
    $emailInfo['desc']       = 'Personal Email';
    $emailInfo['private']    = 1;
    $emailInfo['dbgroup']    = '';
    $emailInfo['maxmsg']     = $VA_MAIN['mailholdtime'];
    $emailInfo['minreadsl']  = 0;
    $emailInfo['minwritesl'] = $VA_MAIN['emlothers'];
    $emailInfo['agelimit']   = 0;
    $emailInfo['accflags']   = 0;
    $emailInfo['tagline']    = 0;
    $emailInfo['dispname']   = 0;
    $emailInfo['randtitle']  = 0;
    $emailInfo['qwkconfno']  = 0;
    $emailInfo['moderator']  = 0;
    $emailInfo['validate']   = 0;
    $emailInfo['freqable']   = 0;
    $emailInfo['devicetype'] = 0;
    $emailInfo['dbtype']     = 'E';
    $emailInfo['extra']      = '';
    $emailInfo['msgno']      = GetMessageCount($emailInfo);
    $emailInfo['msgcnt']     = GetEmailCount($userNo);

    // If database does not exist, then create it.
    $file = $VA_SYSPATHS['db'] . '/' . $emailInfo['filename'] . '.DAT';
    if (!is_file($file)) {
        WriteCfgFile($file, '');
    }

    return $emailInfo;                      // Return the email database information.
}


/**
* Va-Database.inc.php :: SetOnelinerInfo()
*
* This function will return the oneliner database information in the same format
* as dbInfo.
*
* @param integer $userNo
* @return array
*/
function SetOnelinerInfo($userNo = 0) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    global $VA_SYSPATHS;

    $onelinerInfo['topicno']    = '';
    $onelinerInfo['topic']      = 'Oneliners';
    $onelinerInfo['dbno']       = 0;
    $onelinerInfo['filename']   = 'ONELINE';
    $onelinerInfo['filepath']   = '';
    $onelinerInfo['desc']       = 'Oneliners';
    $onelinerInfo['private']    = 0;
    $onelinerInfo['dbgroup']    = '';
    $onelinerInfo['maxmsg']     = 0;
    $onelinerInfo['minreadsl']  = 0;
    $onelinerInfo['minwritesl'] = 0;
    $onelinerInfo['agelimit']   = 0;
    $onelinerInfo['accflags']   = 0;
    $onelinerInfo['tagline']    = 0;
    $onelinerInfo['dispname']   = 0;
    $onelinerInfo['randtitle']  = 0;
    $onelinerInfo['qwkconfno']  = 0;
    $onelinerInfo['moderator']  = 0;
    $onelinerInfo['validate']   = 0;
    $onelinerInfo['freqable']   = 0;
    $onelinerInfo['devicetype'] = 0;
    $onelinerInfo['dbtype']     = 'E';
    $onelinerInfo['extra']      = '';
    $onelinerInfo['msgno']      = GetMessageCount($onelinerInfo);
    if ($userNo > 0) {
        $onelinerInfo['msgcnt']     = GetOnelinerCount($userNo);
    }

    // If database does not exist, then create it.
    $file = $VA_SYSPATHS['db'] . '/' . $onelinerInfo['filename'] . '.DAT';
    if (!is_file($file)) {
        WriteCfgFile($file, '');
    }

    return $onelinerInfo;                   // Return the oneliner database information.
}


/**
* Va-Database.inc.php :: SetSentEmailInfo()
*
* This function will return the sent email database information in the same
* format as dbInfo.
*
* @param integer $userNo
* @return array
*/
function SetSentEmailInfo($userNo) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }

    global $CONFIG;
    global $VA_SYSPATHS;
    global $VA_MAIN;
    global $includeDir;
    include_once($includeDir . 'va-main.inc.php');

    $emailInfo['topicno']    = '';
    $emailInfo['topic']      = 'SentEmail';
    $emailInfo['dbno']       = 0;
    $emailInfo['filename']   = 'SNTEMAIL';
    if ($CONFIG['filehost'] != '') {
        $emailInfo['filepath'] = ConvertUnc($VA_SYSPATHS['sysop']);
    } else {
        $emailInfo['filepath'] = $VA_SYSPATHS['sysop'];
    }
    $emailInfo['desc']       = 'Sent Email';
    $emailInfo['private']    = 1;
    $emailInfo['dbgroup']    = '';
    $emailInfo['maxmsg']     = $VA_MAIN['mailholdtime'];
    $emailInfo['minreadsl']  = 0;
    $emailInfo['minwritesl'] = $VA_MAIN['emlothers'];
    $emailInfo['agelimit']   = 0;
    $emailInfo['accflags']   = 0;
    $emailInfo['tagline']    = 0;
    $emailInfo['dispname']   = 0;
    $emailInfo['randtitle']  = 0;
    $emailInfo['qwkconfno']  = 0;
    $emailInfo['moderator']  = 0;
    $emailInfo['validate']   = 0;
    $emailInfo['freqable']   = 0;
    $emailInfo['devicetype'] = 0;
    $emailInfo['dbtype']     = 'E';
    $emailInfo['extra']      = '';
    $emailInfo['msgno']      = GetMessageCount($emailInfo);
    $emailInfo['msgcnt']     = GetSentEmailCount($userNo);

    // If database does not exist, then create it.
    $file = $VA_SYSPATHS['db'] . '/' . $emailInfo['filename'] . '.DAT';
    if (!is_file($file)) {
        WriteCfgFile($file, '');
    }

    return $emailInfo;                      // Return the email database information.
}


/**
* Va-Database.inc.php :: SetUploadInfo()
*
* This function will return the upload database information in the same format
* as dbInfo.
*
* @return array
*/
function SetUploadInfo() {
    global $CONFIG;
    global $VA_SYSPATHS;

    $uploadsInfo['topicno']    = '';
    $uploadsInfo['topic']      = 'Uploads';
    $uploadsInfo['dbno']       = 0;
    $uploadsInfo['filename']   = 'UPLOADS';
    if ($CONFIG['filehost'] != '') {
        $uploadsInfo['filepath'] = ConvertUnc($VA_SYSPATHS['sysop']);
    } else {
        $uploadsInfo['filepath'] = $VA_SYSPATHS['sysop'];
    }
    $uploadsInfo['desc']       = 'Temporary Upload DB';
    $uploadsInfo['private']    = 0;
    $uploadsInfo['dbgroup']    = '';
    $uploadsInfo['maxmsg']     = 0;
    $uploadsInfo['minreadsl']  = 250;
    $uploadsInfo['minwritesl'] = 0;
    $uploadsInfo['agelimit']   = 0;
    $uploadsInfo['accflags']   = 0;
    $uploadsInfo['tagline']    = 0;
    $uploadsInfo['dispname']   = 0;
    $uploadsInfo['randtitle']  = 0;
    $uploadsInfo['qwkconfno']  = 0;
    $uploadsInfo['moderator']  = 0;
    $uploadsInfo['validate']   = 0;
    $uploadsInfo['freqable']   = 0;
    $uploadsInfo['devicetype'] = 0;
    $uploadsInfo['dbtype']     = 'F';
    $uploadsInfo['extra']      = '';
    $uploadsInfo['msgno']      = GetMessageCount($uploadsInfo);

    // If database does not exist, then create it.
    $file = $VA_SYSPATHS['db'] . '/' . $uploadsInfo['filename'] . '.DAT';
    if (!is_file($file)) {
        WriteCfgFile($file, '');
    }

    return $uploadsInfo;                   // Return the oneliner database information.
}


/**
* Va-Database.inc.php :: UpdateMessagePointers()
*
* This function will scan ALL message databases and update ALL message pointers
* to the most current message. If $ignore is set to 1, it will obey the
* join/ignore flags set by the user.
*
* @param integer $userNo
* @param boolean $ignore
*/
function UpdateMessagePointers($userNo, $ignore = 1) {
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }
    if (!is_bool($ignore)) {
        $ignore = (boolean) $ignore;
    }

    global $VA_SYSPATHS;
    global $includeDir;
    include_once($includeDir . 'va-userfile.inc.php');
    
    $user = GetUser($userNo);
    $dbString = ReadCfgFile($VA_SYSPATHS['main'] . '/DATABASE.CFG');

    $topics = GetTopics('M');               // Get the accessible topics.
    for ($x = 1; $x <= $topics[0]; $x++) {
        $dbno = GetDbCount($topics[$x]);
        for ($y = 1; $y <= $dbno; $y++) {
            $dbInfo = GetDbInfo($topics[$x], $y, $dbString);
            $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'] . '.QSC';
            $fileString = '';
            if (is_file($file)) {
                $fileString = ReadCfgFile($file);
            }
            if (!$ignore) {
                if (DbAccess($dbInfo, '', $user)) {
                    WriteMessagePointer($dbInfo, $userNo, $dbInfo['msgno'], $fileString);
                }
            } else {
                if ((GetMessagePointer($dbInfo, $userNo, $fileString) < 32760) &&
                    (DbAccess($dbInfo, '', $user))) {
                    WriteMessagePointer($dbInfo, $userNo, $dbInfo['msgno'], $fileString);
                }
            }
        }
    }
}


/**
* Va-Database.inc.php :: WriteEmail()
*
* This function will write the email header information and the message text
* to the proper files based on the passed database information. If a file is
* attached to the email, then the filename must be in the 8.3 format. VADV does
* not support long filenames.
*
* This function is to write only new email messages. For message area or files,
* use the appropriate function. If this is a reply to a message, set $reply to
* the message number of the original message.
*
* The toUserString is in the format of user@node*netnumber, for example:
* to send an email to me, it would be 1@1*80. Local messages are given a network
* number of 0.
*
* @param integer $fromUserNo
* @param string $toUserString
* @param string $msgSubject
* @param string $msgBody
* @param integer $reply
* @param string $attFile
* @param boolean $saveSent
* @param string $attFile
* @return mixed
*/
function WriteEmail($fromUserNo, $toUserString, $msgSubject, $msgBody,
                    $reply = 0, $attFile = '', $saveSent = 1) {
    if (!is_int($fromUserNo)) {
        $fromUserNo   = (integer) $fromUserNo;
    }
    if (!is_string($toUserString)) {
        $toUserString = (string)  $toUserString;
    }
    if (!is_string($msgSubject)) {
        $msgSubject   = (string)  $msgSubject;
    }
    if (!is_string($msgBody)) {
        $msgBody      = (string)  $msgBody;
    }
    if (!is_int($reply)) {
        $reply        = (integer) $reply;
    }
    if (!is_string($attFile)) {
        $attFile      = (string)  $attFile;
    }
    if (!is_bool($saveSent)) {
        $saveSent     = (boolean) $saveSent;
    }

    if ($fromUserNo < 1) {
        return FALSE;
    }

    global $VA_SYSPATHS;                    // Include global variable.
    global $CONFIG;                         // Include VADV-PHP Configuration.
    global $VA_MAIN;                        // Include global variable in case it is already defined.
    global $USER;                           // Include user information.
    global $USER_SETTINGS;                  // Include user php settings.
    global $includeDir;
    include_once($includeDir . 'va-userfile.inc.php');
    include_once($includeDir . 'va-main.inc.php');
    include_once($includeDir . 'va-networks.inc.php');
    include_once($includeDir . 'va-stats.inc.php');

    $toUserString = trim($toUserString);
    $msgSubject   = rtrim(strip_tags($msgSubject));
    if (strlen($msgSubject) > 64) {
        $msgSubject = substr($msgSubject, 0, 64);
    }
    $msgBody = UnHtmlSpecialChars($msgBody);
    $msgBody = rtrim(wordwrap($msgBody, 79, "\r\n", 1));

    if ($USER_SETTINGS['tagline'] > 0) {                                    // If user auto-sig is set...
        if ($USER_SETTINGS['tagline'] == 1) {
            $msgBody = $msgBody . "\r\n\r\n" . rtrim($USER['macro1']);
        } else {
            $msgBody = $msgBody . "\r\n\r\n" . rtrim($USER['macro2']);
        }
    }

    $file = $VA_SYSPATHS['text'] . '/EMAIL.TAG';                // Check for email tagline file.
    if (is_file($file)) {
        $msgBody = $msgBody . "\r\n\r\n" . ReadCfgFile($file);  // Add the tagline if there.
    }

    $networks   = GetNetworkInfo();         // Get the local network info.
    $dbInfo     = SetEmailInfo($fromUserNo);
    $sentdbInfo = SetSentEmailInfo($fromUserNo);
    
    // Setup file variables if a file was attached.
    if ((trim($attFile) != '') && (is_file($attFile))) {
        $attFilesize = filesize($attFile);
        $pos = strrpos($attFile, '/');
        if ($pos == 0) {
            $pos = strrpos($attFile, "\\");
        }
        if ($pos > 0) {
            $attFilename = substr($attFile, $pos + 1);
            if ($CONFIG['hostname'] != '') {
                $attFilepath = ConvertUnc($attFile);
            } else {
                $attFilepath = $attFile;
            }
        } else {
            $attFilesize = 0;
            $attFilename = '';
            $attFilepath = '';
        }
    } else {
        $attFilesize = 0;
        $attFilename = '';
        $attFilepath = '';
    }

    // Break down the toUserString variable.
    $atpos = strpos($toUserString, '@');
    $aspos = strrpos($toUserString, '*');
    $toUserNo  = trim(substr($toUserString, 0, $atpos));
    $toNetNode = trim(substr($toUserString, $atpos + 1, $aspos - $atpos - 1));
    $toNetId   = trim(substr($toUserString, $aspos + 1));

    // Check to see if it a local net id.
    if (($toNetId == 1) && ($toNetNode == $networks[1]['settings']['systemid'])) {
        $toNetNode = 0;
        $toNetId   = 0;
    }

    // If a local email, check to see if the user exists.
    if ($toNetId == 0) {
        if (is_numeric($toUserNo)) {
            $toUser = GetUser($toUserNo);
        } else {
            $toUser = FindHandle($toUserNo);
            if ($toUser == 0) {
                $toUser = FindRealName($toUserNo);
                if ($toUser == 0) {
                    die('<br /><h4>The user specified is invalid. Please press the BACK button on your browser.</h4>');
                }
            }
            $toUserNo = $toUser;
            $toUser = GetUser($toUser);
        }
        // Check if the user has their email forwarded.
        if ($toUser['mailfwdflag'] == 1) {
            // If forwarded to another local user...
            if (($toUser['mailfwdnetno'] == 1) && ($toUser['mailfwdnode'] == $networks[1]['settings']['systemid'])) {
                $toUserNo = $toUser['mailfwduserno'];
                $toUser = GetUser($toUserNo);
            } else {
                // If forwarded to another location...
                if ($toUser['mailfwduserno'] == 0) {    // If the to user no field is blank...
                    $toUserNo = $toUser['mailfwdhandle'];
                } else {                                // If the to user no field is not blank...
                    $toUserNo = $toUser['mailfwduserno'];
                }
                $toNetNode = $toUser['mailfwdnode'];
                $toNetId   = $toUser['mailfwdnetno'];
            }
        }
    }

    // Set the toUserHandle variable.
    if ($toNetId == 0) {
        $toUserHandle = $toUser['handle'];
        $toNetNode    = $networks[1]['settings']['systemid'];
    } else {
        if (is_numeric($toUserNo)) {
            $toUserHandle = '';
        } else {
            $toUserHandle = $toUserNo;
            $toUserNo = 0;
        }
    }

    // Set the filename.
    $file    = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $binFile = $file . '.BIN';              // Set the bin filename.
    $datFile = $file . '.DAT';              // Set the database filename.

    // Set the sent filename.
    $sentfile = $VA_SYSPATHS['db'] . '/' . $sentdbInfo['filename'];
    $sentbinFile = $sentfile . '.BIN';      // Set the bin filename.
    $sentdatFile = $sentfile . '.DAT';      // Set the database filename.

    // Get the 'from' user information and update it.
    $fileString = ReadUserfile();
    $fromUser = GetUser($fromUserNo);
    $fromUser['totalemail']++;
    $fileString = PutUser($fileString, $fromUserNo, $fromUser);
    $status = WriteUserfile($fileString);

    // Set the $fromUserHandle variable.
    $fromUserHandle = $fromUser['handle'];
    if (($toNetId != 0) && ($networks[$toNetId]['nettype'] == 4)) {
        if ($networks[$toNetId]['realname'] == 1) {
            $fromUserHandle = $fromUser['realname'];
        }
    }

    // Update the BBS stats.
    $fileString = ReadStats();
    $stats = GetStats();
    $stats['emailstotal']++;
    $stats['emailstoday']++;
    $fileString = PutStats($fileString, $stats);
    $status = WriteStats($fileString);

    if ($toNetId != 0) {
        $msgpos = 0;
    } else {
        if (!is_file($binFile)) {
            $msgpos = 1;
        } else {
            $msgpos = filesize($binFile) + 1;   // Get the message starting position.
        }
    }
    $msglen = strlen($msgBody) + 2;             // Get the message length.

    // Add spacing of 582 bytes of null chars.
    $newString = str_pad('', 582, chr(0));      // Create the new blank string.

    $newString = PutLongInt($newString, $fromUserNo, 1);
    $newString = PutLongInt($newString, $toUserNo, 5);
    $newString = PutString($newString, $fromUserHandle, 9, 40);
    $newString = PutString($newString, $toUserHandle, 49, 40);
    $newString = PutString($newString, $networks[1]['settings']['systemid'], 89, 80);
    $newString = PutString($newString, $toNetNode, 169, 80);
    $newString = PutInteger($newString, 1, 249);
    if ($toNetId == 0) {
        $newString = PutInteger($newString, 1, 251);
    } else {
        $newString = PutInteger($newString, $toNetId, 251);
    }
    $newString = PutLongInt($newString, time(), 253);
    $newString = PutLongInt($newString, WriteTotalMessageCount(), 257);
    $newString = PutLongInt($newString, 0, 261);
    $newString = PutLongInt($newString, 0, 265);
    $newString = PutString($newString, $msgSubject, 269, 64);
    $newString = PutInteger($newString, 0, 333);
    $newString = PutString($newString, date("D M d, Y H:i:s O"), 335, 40);
    $newString = PutLongInt($newString, 0, 375);
    $newString = PutLongInt($newString, $attFilesize, 379);
    $newString = PutString($newString, strtoupper($attFilename), 383, 12);
    $newString = PutString($newString, strtoupper($attFilepath), 395, 64);
    $newString = PutLongInt($newString, 0, 459);
    $newString = PutLongInt($newString, 0, 463);
    $newString = PutLongInt($newString, 0, 467);
    $newString = PutString($newString, $dbInfo['filename'], 471, 8);
    $newString = PutLongInt($newString, $msgpos, 479);
    $newString = PutLongInt($newString, $msglen, 483);
    $newString = PutInteger($newString, $toNetId, 487);
    $newString = PutInteger($newString, 0, 489);
    $newString = PutLongInt($newString, 0, 491);
    $newString = PutString($newString, '', 495, 24);
    $newString = PutString($newString, '', 519, 64);

    // If a local message, write it to the database, if not, write it to spool.net.
    if ($toNetId == 0) {
        // Update the .DAT file (and update dbseq.dat).
        $fileString = ReadCfgFile($datFile);        // Read the file.
        
        // Error correction...
        $fslen = strlen($fileString);
        $fsnum = $fslen / 582;
        if (is_int($fsnum)) {
            $fileString = $fileString . $newString; // Append the new data to the original file string.
        } else {
            $fileString = substr($fileString, 0, floor($fsnum) * 582) . $newString;
        }
        $status = WriteCfgFile($datFile, $fileString);  // Write the new data file.

        // Add the message to the .BIN file.
        $fileString = ReadCfgFile($binFile);                        // Read the file.
        WriteCfgFile($binFile, $fileString . $msgBody . "\r\n");    // Write the file.

        // Send a one liner to the user if online.
        $oneliner = $fromUser['handle'] . ' just sent you some email!';
        SendOneLiner($oneliner, $toUserNo);
    } else {
        // Create validation file or spool.net file based on db settings.
        $file = $VA_SYSPATHS['data'] . '/SPOOL.NET';            // Create network spool file.

        $fileString = ReadCfgFile($file);                                   // Read the file.
        WriteCfgFile($file, $fileString . $newString . $msgBody . "\r\n");  // Write the file.
    }

    if ($saveSent) {
        // Setup the sent email database entry.
        if (!is_file($sentbinFile)) {
            $msgpos = 1;
        } else {
            $msgpos = filesize($sentbinFile) + 1;   // Get the message starting position.
        }
        $newString = PutString($newString, $sentdbInfo['filename'], 471, 8);
        $newString = PutLongInt($newString, 1, 375);
        $newString = PutLongInt($newString, 0, 379);
        $newString = PutString($newString, '', 383, 12);
        $newString = PutString($newString, '', 395, 64);
        $newString = PutLongInt($newString, $msgpos, 479);

        // Save the email into the sent email database, so the user can
        // retrieve it at a later time.
        $fileString = ReadCfgFile($sentdatFile);    // Read the file.

        // Error correction...
        $fslen = strlen($fileString);
        $fsnum = $fslen / 582;
        if (is_int($fsnum)) {
            $fileString = $fileString . $newString; // Append the new data to the original file string.
        } else {
            $fileString = substr($fileString, 0, intval($fsnum) * 582) . $newString;
        }
        WriteCfgFile($sentdatFile, $fileString);    // Write the new data file.

        // Save the email into the sent email .BIN file.
        $fileString = ReadCfgFile($sentbinFile);                        // Read the file.
        WriteCfgFile($sentbinFile, $fileString . $msgBody . "\r\n");    // Write the file.
    }

    return $status;                         // Return the write status.
}


/**
* Va-Database.inc.php :: WriteFile()
*
* This function will write the file header information and the file description
* to the proper files based on the passed database information. $attFile should
* be a full path to the file and the filename should be in the 8.3 format. VADV
* does not support long filenames.
*
* This function is to write only new file area entries. For emails or messages,
* use the appropriate function.
*
* @param array $dbInfo
* @param integer $fromUserNo
* @param string $fileDesc
* @param string $fileExtDesc
* @param string $attFile
* @return mixed
*/
function WriteFile($dbInfo, $fromUserNo, $fileDesc, $fileExtDesc, $attFile) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($fromUserNo)) {
        $fromUserNo  = (integer) $fromUserNo;
    }
    if (!is_string($fileDesc)) {
        $fileDesc    = (string)  $fileDesc;
    }
    if (!is_string($fileExtDesc)) {
        $fileExtDesc = (string)  $fileExtDesc;
    }
    if (!is_string($attFile)) {
        $attFile     = (string)  $attFile;
    }

    if ($fromUserNo < 1) {
        return FALSE;
    }

    global $VA_SYSPATHS;                    // Include global variable.
    global $CONFIG;                         // Include VADV-PHP configuration.
    global $VA_MAIN;                        // Include global variable in case it is already defined.
    global $includeDir;
    include_once($includeDir . 'va-userfile.inc.php');
    include_once($includeDir . 'va-main.inc.php');
    include_once($includeDir . 'va-networks.inc.php');
    include_once($includeDir . 'va-stats.inc.php');

    $fileDesc = rtrim(strip_tags($fileDesc));
    if (strlen($fileDesc) > 64) {
        $fileDesc = substr($fileDesc, 0, 64);
    }
    $fileExtDesc = UnHtmlSpecialChars($fileExtDesc);
    $fileExtDesc = rtrim(wordwrap($fileExtDesc, 79, "\r\n", 1));
    if ($dbInfo['tagline'] != 0) {                                          // If tagline is set...
        $file = $VA_SYSPATHS['text'] . '/TAGLINE.' . $dbInfo['tagline'];    // Look for the file...
        if (is_file($file)) {
            $fileExtDesc = $fileExtDesc . "\r\n\r\n" . ReadCfgFile($file);  // Append the file if found.
        }
    }

    // Setup file variables if a file was attached.
    if ((trim($attFile) != '') && (is_file($attFile))) {
        $attFilesize = filesize($attFile);
        $pos = strrpos($attFile, '/');
        if ($pos == 0) {
            $pos = strrpos($attFile, "\\");
        }
        if ($pos > 0) {
            $attFilename = substr($attFile, $pos + 1);
        } else {
            $attFilesize = 0;
            $attFilename = '';
        }
    } else {
        $attFilesize = 0;
        $attFilename = '';
    }

    // Set the filename.
    $file    = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $binFile = $file . '.BIN';              // Set the bin filename.
    $datFile = $file . '.DAT';              // Set the database filename.

    // Get the 'from' user information and update it.
    $fileString = ReadUserfile();
    $fromUser = GetUser($fromUserNo);
    $fromUser['uplfiles']++;
    $uplKbCredit = intval($attFilesize / 1024) * $VA_MAIN['credperuplk'];
    $fromUser['credits'] = $fromUser['credits'] + $uplKbCredit; // Award credits for posting.
    $fileString = PutUser($fileString, $fromUserNo, $fromUser);
    $status = WriteUserfile($fileString);

    // Update the BBS stats.
    $fileString = ReadStats();
    $stats = GetStats();
    $stats['uploadstotal']++;
    $stats['uploadstoday']++;
    $fileString = PutStats($fileString, $stats);
    $status = WriteStats($fileString);

    if (!is_file($binFile)) {
        $msgpos = 1;
    } else {
        $msgpos = filesize($binFile) + 1;       // Get the message starting position.
    }
    $msglen = strlen($fileExtDesc) + 2;         // Get the message length.

    // Update the .DAT file (and update dbseq.dat).
    // Add spacing of 582 bytes of null chars.
    $newString = str_pad('', 582, chr(0));  // Create the new blank string.
    $networks = GetNetworkInfo();           // Get the local network info.

    // Check for special database settings.
    if ($dbInfo['dispname'] == 1) {
        $fromHandle = $fromUser['realname'];
    } else if ($dbInfo['dispname'] == 2) {
        $fromHandle = 'Anonymous';
    } else {
        $fromHandle = $fromUser['handle'];
    }

    $newString = PutLongInt($newString, $fromUserNo, 1);
    $newString = PutLongInt($newString, 0, 5);
    $newString = PutString($newString, $fromHandle, 9, 40);
    $newString = PutString($newString, '', 49, 40);
    $newString = PutString($newString, $networks[1]['settings']['systemid'], 89, 80);
    $newString = PutString($newString, '', 169, 80);
    $newString = PutInteger($newString, 1, 249);
    $newString = PutInteger($newString, 1, 251);
    $newString = PutLongInt($newString, time(), 253);
    $newString = PutLongInt($newString, WriteTotalMessageCount(), 257);
    $newString = PutLongInt($newString, 0, 261);
    $newString = PutLongInt($newString, 0, 265);
    $newString = PutString($newString, $fileDesc, 269, 64);
    $newString = PutInteger($newString, 0, 333);
    $newString = PutString($newString, date("D M d, Y H:i:s O"), 335, 40);
    $newString = PutLongInt($newString, 0, 375);
    $newString = PutLongInt($newString, $attFilesize, 379);
    $newString = PutString($newString, strtoupper($attFilename), 383, 12);
    $newString = PutString($newString, '', 395, 64);
    $newString = PutLongInt($newString, 0, 459);
    $newString = PutLongInt($newString, 0, 463);
    $newString = PutLongInt($newString, 0, 467);
    $newString = PutString($newString, $dbInfo['filename'], 471, 8);
    $newString = PutLongInt($newString, $msgpos, 479);
    $newString = PutLongInt($newString, $msglen, 483);
    $newString = PutInteger($newString, 0, 487);
    $newString = PutInteger($newString, 0, 489);
    $newString = PutLongInt($newString, 0, 491);
    $newString = PutString($newString, '', 495, 24);
    $newString = PutString($newString, '', 519, 64);

    $fileString = ReadCfgFile($datFile);            // Read the file.

    // Error correction...
    $fslen = strlen($fileString);
    $fsnum = $fslen / 582;
    if (is_int($fsnum)) {
        $fileString = $fileString . $newString;     // Append the new data to the original file string.
    } else {
        $fileString = substr($fileString, 0, floor($fsnum) * 582) . $newString;
    }
    $status = WriteCfgFile($datFile, $fileString);  // Write the new data file.

    // Add the message to the .BIN file.
    $fileString = ReadCfgFile($binFile);                            // Read the file.
    WriteCfgFile($binFile, $fileString . $fileExtDesc . "\r\n");    // Write the file.

    return $status;                         // Return the write status.
}


/**
* Va-Database.inc.php :: WriteMessage()
*
* This function will write the message header information and the message text
* to the proper files based on the passed database information.
*
* This function is to write only new message area messages. For emails or files,
* use the appropriate function. If this is a reply to a message, set $reply to
* the message number of the original message.
*
* @param array $dbInfo
* @param integer $fromUserNo
* @param string $msgSubject
* @param string $msgBody
* @param integer $reply
* @return mixed
*/
function WriteMessage($dbInfo, $fromUserNo, $msgSubject, $msgBody, $reply = 0) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($fromUserNo)) {
        $fromUserNo = (integer) $fromUserNo;
    }
    if (!is_string($msgSubject)) {
        $msgSubject = (string)  $msgSubject;
    }
    if (!is_string($msgBody)) {
        $msgBody    = (string)  $msgBody;
    }
    if (!is_int($reply)) {
        $reply      = (integer) $reply;
    }

    if ($fromUserNo < 1) {
        return FALSE;
    }
    
    global $VA_SYSPATHS;                    // Include global variable.
    global $VA_MAIN;                        // Include global variable in case it is already defined.
    global $USER;                           // Include user information.
    global $USER_SETTINGS;                  // Include user php settings.
    global $includeDir;
    include_once($includeDir . 'va-userfile.inc.php');
    include_once($includeDir . 'va-main.inc.php');
    include_once($includeDir . 'va-networks.inc.php');
    include_once($includeDir . 'va-stats.inc.php');

    $msgSubject = rtrim(strip_tags($msgSubject));
    if (strlen($msgSubject) > 64) {
        $msgSubject = substr($msgSubject, 0, 64);
    }
    $msgBody = UnHtmlSpecialChars($msgBody);
    $msgBody = rtrim(wordwrap($msgBody, 79, "\r\n", 1));
    if ($USER_SETTINGS['tagline'] > 0) {                                    // If user auto-sig is set...
        if ($USER_SETTINGS['tagline'] == 1) {
            $msgBody = $msgBody . "\r\n\r\n" . rtrim($USER['macro1']);
        } else {
            $msgBody = $msgBody . "\r\n\r\n" . rtrim($USER['macro2']);
        }
    }
    if ($dbInfo['tagline'] != 0) {                                          // If tagline is set...
        $file = $VA_SYSPATHS['text'] . '/TAGLINE.' . $dbInfo['tagline'];    // Look for the file...
        if (is_file($file)) {
            $msgBody = $msgBody . "\r\n\r\n" . ReadCfgFile($file);          // Append the file if found.
        }
    }

    // Set the filename.
    $file    = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $binFile = $file . '.BIN';              // Set the bin filename.
    $datFile = $file . '.DAT';              // Set the database filename.

    // Get the 'from' user information and update it.
    $fileString = ReadUserfile();
    $fromUser = GetUser($fromUserNo);
    $fromUser['totalposts']++;
    $fromUser['credits'] = $fromUser['credits'] + $VA_MAIN['credperpost'];  // Award credits for posting.
    $fileString = PutUser($fileString, $fromUserNo, $fromUser);
    $status = WriteUserfile($fileString);

    // Update the BBS stats.
    $fileString = ReadStats();
    $stats = GetStats();
    $stats['poststotal']++;
    $stats['poststoday']++;
    $fileString = PutStats($fileString, $stats);
    $status = WriteStats($fileString);

    if (!is_file($binFile)) {
        $msgpos = 1;
    } else {
        $msgpos = filesize($binFile) + 1;       // Get the message starting position.
    }
    $msglen = strlen($msgBody) + 2;             // Get the message length.

    // Update the .DAT file (and update dbseq.dat).
    // Add spacing of 582 bytes of null chars.
    $newString = str_pad('', 582, chr(0));  // Create the new blank string.
    $networks = GetNetworkInfo();           // Get the local network info.

    // Check for special database settings.
    if ($dbInfo['dispname'] == 1) {
        $fromHandle = $fromUser['realname'];
    } else if ($dbInfo['dispname'] == 2) {
        $fromHandle = 'Anonymous';
    } else {
        $fromHandle = $fromUser['handle'];
    }

    // Check if a reply and set the variables accordingly.
    if ($reply > 0) {
        $origMessage = GetMessageInfo($dbInfo, $reply);
        $toUserNo    = $origMessage['fromuserno'];
        $toUser      = $origMessage['fromhandle'];
        $toNetNode   = $origMessage['fromnetnode'];
        $toNetId     = $origMessage['fromnetid'];
        $threadid    = $origMessage['sysmsgno'];
        $threadnode  = $origMessage['fromnetnode'];
    } else {
        $toUserNo    = 0;
        $toUser      = '';
        $toNetNode   = '';
        $toNetId     = 0;
        $threadid    = 0;
        $threadnode  = 0;
    }
    $newString = PutLongInt($newString, $fromUserNo, 1);
    $newString = PutLongInt($newString, $toUserNo, 5);
    $newString = PutString($newString, $fromHandle, 9, 40);
    $newString = PutString($newString, $toUser, 49, 40);
    $newString = PutString($newString, $networks[1]['settings']['systemid'], 89, 80);
    $newString = PutString($newString, $toNetNode, 169, 80);
    $newString = PutInteger($newString, 1, 249);
    $newString = PutInteger($newString, $toNetId, 251);
    $newString = PutLongInt($newString, time(), 253);
    $newString = PutLongInt($newString, WriteTotalMessageCount(), 257);
    $newString = PutLongInt($newString, $threadid, 261);
    $newString = PutLongInt($newString, $threadnode, 265);
    $newString = PutString($newString, $msgSubject, 269, 64);
    $newString = PutInteger($newString, 0, 333);
    $newString = PutString($newString, date("D M d, Y H:i:s O"), 335, 40);
    $newString = PutLongInt($newString, 0, 375);
    $newString = PutLongInt($newString, 0, 379);
    $newString = PutString($newString, '', 383, 12);
    $newString = PutString($newString, '', 395, 64);
    $newString = PutLongInt($newString, 0, 459);
    $newString = PutLongInt($newString, 0, 463);
    $newString = PutLongInt($newString, 0, 467);
    $newString = PutString($newString, $dbInfo['filename'], 471, 8);
    $newString = PutLongInt($newString, $msgpos, 479);
    $newString = PutLongInt($newString, $msglen, 483);
    if ($reply > 0) {
        $newString = PutInteger($newString, $toNetId, 487);
    } else {
        $newString = PutInteger($newString, 0, 487);
    }
    $newString = PutInteger($newString, 0, 489);
    $newString = PutLongInt($newString, 0, 491);
    $newString = PutString($newString, '', 495, 24);
    $newString = PutString($newString, '', 519, 64);

    $fileString = ReadCfgFile($datFile);            // Read the file.

    // Error correction...
    $fslen = strlen($fileString);
    $fsnum = $fslen / 582;
    if (is_int($fsnum)) {
        $fileString = $fileString . $newString;     // Append the new data to the original file string.
    } else {
        $fileString = substr($fileString, 0, floor($fsnum) * 582) . $newString;
    }
    $status = WriteCfgFile($datFile, $fileString);  // Write the new data file.

    // Add the message to the .BIN file.
    $fp = fopen($binFile, 'a');                 // Open the file.
    flock($fp, LOCK_EX);                        // Lock the file.
    $status = fwrite($fp, $msgBody . "\r\n");   // Write the message.
    flock($fp, LOCK_UN);                        // Unlock the file.
    fclose($fp);                                // Close the file.

    // Create validation file or spool.net file based on db settings.
    if ($dbInfo['validate'] == 1) {
        $file = $VA_SYSPATHS['data'] . '/VALIDATE.0';   // Create local post validation file.
    } else {
        $file = $VA_SYSPATHS['data'] . '/SPOOL.NET';    // Create network spool file.
    }
    $fp = fopen($file, 'a');                                // Open the file.
    flock($fp, LOCK_EX);                                    // Lock the file.
    $status = fwrite($fp, $newString . $msgBody . "\r\n");  // Write the message.
    flock($fp, LOCK_UN);                                    // Unlock the file.
    fclose($fp);                                            // Close the file.

    return $status;                         // Return the write status.
}


/**
* Va-Database.inc.php :: WriteMessageInfo()
*
* This function will allow you to write new message information into the
* corresponding .DAT file. You cannot change the creation date.
*
* @param array $dbInfo
* @param integer $msgNo
* @param array $configVariable
* @return mixed
*/
function WriteMessageInfo($dbInfo, $msgNo, $configVariable) {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($msgNo)) {
        $msgNo = (integer) $msgNo;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    if ($msgNo < 1) {
        return FALSE;
    }
    if ($msgNo > $dbInfo['msgno']) {
        return FALSE;
    }


    global $VA_SYSPATHS;                    // Include global variable.

    // Set the filename.
    $file    = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $datFile = $file . '.DAT';              // Set the database filename.
    if (!is_file($datFile)) {
        return FALSE;
    }

    $fileString = ReadCfgFile($datFile);    // Read the file.

    $offset = 582 * ($msgNo - 1);
    if (isset($configVariable['fromuserno']))  $fileString = PutLongInt($fileString, $configVariable['fromuserno'], 1 + $offset);
    if (isset($configVariable['touserno']))    $fileString = PutLongInt($fileString, $configVariable['touserno'], 5 + $offset);
    if (isset($configVariable['fromhandle']))  $fileString = PutString($fileString, $configVariable['fromhandle'], 9 + $offset, 40);
    if (isset($configVariable['tohandle']))    $fileString = PutString($fileString, $configVariable['tohandle'], 49 + $offset, 40);
    if (isset($configVariable['fromnetnode'])) $fileString = PutString($fileString, $configVariable['fromnetnode'], 89 + $offset, 80);
    if (isset($configVariable['tonetnode']))   $fileString = PutString($fileString, $configVariable['tonetnode'], 169 + $offset, 80);
    if (isset($configVariable['fromnetid']))   $fileString = PutInteger($fileString, $configVariable['fromnetid'], 249 + $offset);
    if (isset($configVariable['tonetid']))     $fileString = PutInteger($fileString, $configVariable['tonetid'], 251 + $offset);
    if (isset($configVariable['sysmsgno']))    $fileString = PutLongInt($fileString, $configVariable['sysmsgno'], 257 + $offset);
    if (isset($configVariable['threadbk']))    $fileString = PutLongInt($fileString, $configVariable['threadbk'], 261 + $offset);
    if (isset($configVariable['threadbkn']))   $fileString = PutLongInt($fileString, $configVariable['threadbkn'], 265 + $offset);
    if (isset($configVariable['subject']))     $fileString = PutString($fileString, $configVariable['subject'], 269 + $offset, 64);
    if (isset($configVariable['deleted']))     $fileString = PutInteger($fileString, $configVariable['deleted'], 333 + $offset);
    if (isset($configVariable['offline']))     $fileString = PutLongInt($fileString, $configVariable['offline'], 375 + $offset);
    if (isset($configVariable['attfilesize'])) $fileString = PutLongInt($fileString, $configVariable['attfilesize'], 379 + $offset);
    if (isset($configVariable['attfilename'])) $fileString = PutString($fileString, $configVariable['attfilename'], 383 + $offset, 12);
    if (isset($configVariable['attfilepath'])) $fileString = PutString($fileString, $configVariable['attfilepath'], 395 + $offset, 64);
    if (isset($configVariable['lib']))         $fileString = PutLongInt($fileString, $configVariable['lib'], 459 + $offset);
    if (isset($configVariable['downloadno']))  $fileString = PutLongInt($fileString, $configVariable['downloadno'], 463 + $offset);
    if (isset($configVariable['dbid']))        $fileString = PutLongInt($fileString, $configVariable['dbid'], 467 + $offset);
    if (isset($configVariable['filename']))    $fileString = PutString($fileString, $configVariable['filename'], 471 + $offset, 8);
    if (isset($configVariable['msgpos']))      $fileString = PutLongInt($fileString, $configVariable['msgpos'], 479 + $offset);
    if (isset($configVariable['msglength']))   $fileString = PutLongInt($fileString, $configVariable['msglength'], 483 + $offset);
    if (isset($configVariable['originnet']))   $fileString = PutInteger($fileString, $configVariable['originnet'], 487 + $offset);
    if (isset($configVariable['fidoflags']))   $fileString = PutInteger($fileString, $configVariable['fidoflags'], 489 + $offset);
    if (isset($configVariable['originnode']))  $fileString = PutLongInt($fileString, $configVariable['originnode'], 491 + $offset);
    if (isset($configVariable['reserved']))    $fileString = PutString($fileString, $configVariable['reserved'], 495 + $offset, 24);
    if (isset($configVariable['extra']))       $fileString = PutString($fileString, $configVariable['extra'], 519 + $offset, 64);

    $status = WriteCfgFile($datFile, $fileString);

    return $status;                         // Return the result status.

}


/**
* Va-Database.inc.php :: WriteMessagePointer()
*
* This function will update the message pointer file for the specified database
* with either the supplied message number, or if no message number is passed,
* then it will increase the pointer by one.
*
* @param array $dbInfo
* @param integer $userNo
* @param integer $msgNo
* @return integer
*/
function WriteMessagePointer($dbInfo, $userNo, $msgNo = 0, $fileString = '') {
    if (!is_array($dbInfo)) {
        trigger_error(VA_Error(1));
    }
    if (!is_int($userNo)) {
        $userNo = (integer) $userNo;
    }
    if (!is_int($msgNo)) {
        $msgNo  = (integer) $msgNo;
    }
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    if ($userNo < 1) {
        return 32760;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'] . '.QSC';
    if ($fileString == '') {
        if ((!is_file($file)) || (filesize($file) == 0)) {
            $fileString = str_pad('', $userNo * 4, chr(0));
        } else {
            $fileString = ReadCfgFile($file);
        }
    }

    $offset = ($userNo - 1) * 4;
    $pointer = GetLongInt($fileString, 1 + $offset);
    
    if ($msgNo != 0) {
        $newPointer = $msgNo;
    } else {
        $newPointer = $pointer + 1;
    }

    $fileString = PutLongInt($fileString, $newPointer, 1 + $offset);
    $status = WriteCfgFile($file, $fileString);
    if (!$status) {
        return $status;
    }

    return $pointer;                        // Return the message pointer.
}


/**
* Va-Database.inc.php :: WriteOneliner()
*
* This function will create a oneliner message to the specified user. The message
* can be a maximum of 64 characters.
*
* @param integer $toUserNo
* @param string $message
* @return mixed
*/
function WriteOneliner($toUserNo, $message) {
    if (!is_int($toUserNo)) {
        $toUserNo = (integer) $toUserNo;
    }
    if (!is_string($message)) {
        $message  = (string)  $message;
    }
    
    global $VA_SYSPATHS;                    // Include global variable.
    global $VA_MAIN;                        // Include global variable in case it is already defined.
    global $includeDir;
    include_once($includeDir . 'va-userfile.inc.php');
    include_once($includeDir . 'va-main.inc.php');
    include_once($includeDir . 'va-networks.inc.php');

    $dbInfo   = SetOnelinerInfo();
    $toUser   = GetUser($toUserNo);
    $networks = GetVnetInfo();
    
    $file = $VA_SYSPATHS['db'] . '/' . $dbInfo['filename'];
    $datFile = $file . '.DAT';
    $binFile = $file . '.BIN';

    $msgBody = "";

    if (!is_file($binFile)) {
        $msgpos = 1;
    } else {
        $msgpos = filesize($binFile) + 1;       // Get the message starting position.
    }
    $msglen = strlen($msgBody) + 2;             // Get the message length.

    // Write to the DAT file.
    // Add spacing of 582 bytes of null chars.
    $newString = str_pad('', 582, chr(0));  // Create the new blank string.

    $newString = PutLongInt($newString, $toUserNo, 1);
    $newString = PutLongInt($newString, $toUserNo, 5);
    $newString = PutString($newString, $toUser['handle'], 9, 40);
    $newString = PutString($newString, $toUser['handle'], 49, 40);
    $newString = PutString($newString, $networks['systemid'], 89, 80);
    $newString = PutString($newString, $networks['systemid'], 169, 80);
    $newString = PutInteger($newString, 1, 249);
    $newString = PutInteger($newString, 1, 251);
    $newString = PutLongInt($newString, time(), 253);
    $newString = PutLongInt($newString, WriteTotalMessageCount(), 257);
    $newString = PutLongInt($newString, 0, 261);
    $newString = PutLongInt($newString, 0, 265);
    $newString = PutString($newString, $message, 269, 64);
    $newString = PutInteger($newString, 0, 333);
    $newString = PutString($newString, date("D M d, Y H:i:s O"), 335, 40);
    $newString = PutLongInt($newString, 0, 375);
    $newString = PutLongInt($newString, 0, 379);
    $newString = PutString($newString, '', 383, 12);
    $newString = PutString($newString, '', 395, 64);
    $newString = PutLongInt($newString, 0, 459);
    $newString = PutLongInt($newString, 0, 463);
    $newString = PutLongInt($newString, 0, 467);
    $newString = PutString($newString, $dbInfo['filename'], 471, 8);
    $newString = PutLongInt($newString, $msgpos, 479);
    $newString = PutLongInt($newString, $msglen, 483);
    $newString = PutInteger($newString, 0, 487);
    $newString = PutInteger($newString, 0, 489);
    $newString = PutLongInt($newString, 0, 491);
    $newString = PutString($newString, '', 495, 24);
    $newString = PutString($newString, '', 519, 64);

    $fileString = ReadCfgFile($datFile);

    // Error correction...
    $fslen = strlen($fileString);
    $fsnum = $fslen / 582;
    if (is_int($fsnum)) {
        $fileString = $fileString . $newString; // Append the new data to the original file string.
    } else {
        $fileString = substr($fileString, 0, intval($fsnum) * 582) . $newString;
    }
    $status = WriteCfgFile($datFile, $fileString);

    // Add the message to the .BIN file.
    $fileString = ReadCfgFile($binFile);                        // Read the file.
    WriteCfgFile($binFile, $fileString . $msgBody . "\r\n");    // Write the file.

    return $status;                         // Return status.
}


/**
* Va-Database.inc.php :: WriteTotalMessageCount()
*
* This function will add one to the total message count and return that integer.
*
* @return integer
*/
function WriteTotalMessageCount() {
    global $VA_SYSPATHS;                    // Include the global variable.

    $file = $VA_SYSPATHS['data'] . '/DBSEQ.DAT';
    if (!is_file($file)) {
        $fileString = '0';
        $fileString = PutLongInt($fileString, 1, 1);
        $status = WriteCfgFile($file, $fileString);
        if (!$status) {
            return $status;
        }
        return 1;
    }

    $fileString = ReadCfgFile($file);       // Read the file.
    $msgcnt = GetLongInt($fileString, 1);   // Get the total message count.

    if ($msgcnt >= 4000000) {
        $msgcnt = 0;
    }

    $fileString = PutLongInt($fileString, $msgcnt + 1, 1);
    $status = WriteCfgFile($file, $fileString);
    if (!$status) {
        return $status;
    }
    
    return $msgcnt;                         // Return the original value.
}

?>