<?php
/*
 * ----------------------------------------------------------------------------
 * Va-Control.inc.php                                                10/28/2005
 * This script will read the control.dat file and place its information into
 * an array for global use.
 * ----------------------------------------------------------------------------
 * Copyright (C) 2002-2005 Steve Winn. All Rights Reserved.
 *
 * The source code contains a fair amount of comments, which may aid anyone
 * attempting to follow the program or to modify it.
 *
 * For licensing and copyright information, view LICENSE.TXT.
 * ----------------------------------------------------------------------------
 */

require_once('include_dir.php');                    // Get include directory.
require_once($includeDir . 'va-functions.inc.php'); // A VADV-PHP script.
require_once($includeDir . 'va-main.inc.php');      // Include BBS main configuration.


/**
* Va-Control.inc.php :: GetControl()
*
* This function will retrive the node information that is stored in the
* control.dat file. If $node is specified, then it will attempt to get only
* the information for that node, if it is not specified, then it will return
* all node information in the form $control[$node]['setting'].
*
* @param integer $node
* @return array
*/
function GetControl($node = -1) {
    if (!is_int($node)) {
        $node = (integer) $node;
    }

    global $VA_MAIN;                        // Include global variable.
    global $VA_SYSPATHS;                    // Include global variable.
    
    if ($node != -1) {                      // If the node is specified.
        if ($node < 0) {
            $node = 0;
        }
        
        // Set default values.
        $control['baudrate']   = 0;
        $control['action']     = '';
        $control['userno']     = 0;
        $control['handle']     = '';
        $control['logintime']  = 0;
        $control['houron']     = 0;
        $control['minuteon']   = 0;
        $control['teleconflg'] = 0;
        $control['currscript'] = '';
        $control['logindate']  = '';
        $control['blackflag']  = 0;
        $control['filebits']   = 0;
        $control['timeadjust'] = 0;
        $control['language']  = '';
        $control['extra']      = '';
        // Setup default extended values available with VADV32.
        $control['socket']     = '';
        $control['ip']         = '';
        $control['hostname']   = '';

        if ($node > $VA_MAIN['maxchannelno']) {
            return $control;
        }

        $fileString = ReadControl();

        $offset  = 106 * $node;
        $control['baudrate']   = GetLongInt($fileString, 1 + $offset);
        $control['action']     = GetString($fileString, 5 + $offset, 24);
        $control['userno']     = GetLongInt($fileString, 29 + $offset);
        $control['handle']     = GetString($fileString, 33 + $offset, 32);
        $control['logintime']  = GetLongInt($fileString, 65 + $offset);
        $control['houron']     = GetInteger($fileString, 69 + $offset);
        $control['minuteon']   = GetInteger($fileString, 71 + $offset);
        $control['teleconflg'] = GetInteger($fileString, 73 + $offset);
        $control['currscript'] = GetString($fileString, 75 + $offset, 8);
        $control['logindate']  = GetString($fileString, 83 + $offset, 6);
        $control['blackflag']  = GetInteger($fileString, 89 + $offset);
        $control['filebits']   = GetInteger($fileString, 91 + $offset);
        $control['timeadjust'] = GetInteger($fileString, 93 + $offset);
        $control['language']   = GetString($fileString, 95 + $offset, 8);
        $control['extra']      = GetString($fileString, 103 + $offset, 4);

        // Check to see if VADV32 extended values exist and if so assign the variables.
        $va32node = $VA_SYSPATHS['temp'] . '\N' . $node . '\TCPIP.CFG';
        if (is_file($va32node)) {
            $va32info = file($va32node);    // Read the file.
            $control['socket']   = trim($va32info[0]);
            $control['ip']       = trim($va32info[1]);
            $control['hostname'] = trim($va32info[2]);
        }
    } else {                                // If the node is not specified.
        // Set default values.
        $control[0]['baudrate']   = 0;
        $control[0]['action']     = '';
        $control[0]['userno']     = 0;
        $control[0]['handle']     = '';
        $control[0]['logintime']  = 0;
        $control[0]['houron']     = 0;
        $control[0]['minuteon']   = 0;
        $control[0]['teleconflg'] = 0;
        $control[0]['currscript'] = '';
        $control[0]['logindate']  = '';
        $control[0]['blackflag']  = 0;
        $control[0]['filebits']   = 0;
        $control[0]['timeadjust'] = 0;
        $control[0]['language']  = '';
        $control[0]['extra']      = '';
        // Setup default extended values available with VADV32.
        $control[0]['socket']     = '';
        $control[0]['ip']         = '';
        $control[0]['hostname']   = '';

        $fileString = ReadControl();

        $offset      = 0;
        $onlineUsers = 0;

        for ($x = 1; $offset < strlen($fileString); $x++) {
            $control[$x - 1]['baudrate']   = GetLongInt($fileString, 1 + $offset);
            $control[$x - 1]['action']     = GetString($fileString, 5 + $offset, 24);
            $control[$x - 1]['userno']     = GetLongInt($fileString, 29 + $offset);
            $control[$x - 1]['handle']     = GetString($fileString, 33 + $offset, 32);
            $control[$x - 1]['logintime']  = GetLongInt($fileString, 65 + $offset);
            $control[$x - 1]['houron']     = GetInteger($fileString, 69 + $offset);
            $control[$x - 1]['minuteon']   = GetInteger($fileString, 71 + $offset);
            $control[$x - 1]['teleconflg'] = GetInteger($fileString, 73 + $offset);
            $control[$x - 1]['currscript'] = GetString($fileString, 75 + $offset, 8);
            $control[$x - 1]['logindate']  = GetString($fileString, 83 + $offset, 6);
            $control[$x - 1]['blackflag']  = GetInteger($fileString, 89 + $offset);
            $control[$x - 1]['filebits']   = GetInteger($fileString, 91 + $offset);
            $control[$x - 1]['timeadjust'] = GetInteger($fileString, 93 + $offset);
            $control[$x - 1]['language']   = GetString($fileString, 95 + $offset, 8);
            $control[$x - 1]['extra']      = GetString($fileString, 103 + $offset, 4);
            $offset = 106 * $x;
            if ($control[$x - 1]['userno'] > 0) {
                $onlineUsers++;
            }
        }
        $control['nodetotal'] = $x - 1;
        $control['activenodes'] = $onlineUsers;
    }

    return $control;
}


/**
* Va-Control.inc.php :: PutControl()
*
* This function will write updated information to the control.dat file, which
* is used to determine the current users online.
*
* @param string $fileString
* @param integer $node
* @param array $configVariable
* @return string
*/
function PutControl($fileString, $node, $configVariable) {
    if (!is_string($fileString)) {
        $fileString = (string)  $fileString;
    }
    if (!is_int($node)) {
        $node       = (integer) $node;
    }
    if (!is_array($configVariable)) {
        trigger_error(VA_Error(1));
    }

    $total = intval(strlen($fileString) / 106);
    if ($node > $total) {
        $fileString = $fileString . str_pad('', 106, chr(0));
    }

    $offset = 106 * ($node - 1);
    if (isset($configVariable['baudrate']))   $fileString = PutLongInt($fileString, $configVariable['baudrate'], 1 + $offset);
    if (isset($configVariable['action']))     $fileString = PutString($fileString, $configVariable['action'], 5 + $offset, 24);
    if (isset($configVariable['userno']))     $fileString = PutLongInt($fileString, $configVariable['userno'], 29 + $offset);
    if (isset($configVariable['handle']))     $fileString = PutString($fileString, $configVariable['handle'], 33 + $offset, 32);
    if (isset($configVariable['logintime']))  $fileString = PutLongInt($fileString, $configVariable['logintime'], 65 + $offset);
    if (isset($configVariable['houron']))     $fileString = PutInteger($fileString, $configVariable['houron'], 69 + $offset);
    if (isset($configVariable['minuteon']))   $fileString = PutInteger($fileString, $configVariable['minuteon'], 71 + $offset);
    if (isset($configVariable['teleconflg'])) $fileString = PutInteger($fileString, $configVariable['teleconflg'], 73 + $offset);
    if (isset($configVariable['currscript'])) $fileString = PutString($fileString, $configVariable['currscript'], 75 + $offset, 8);
    if (isset($configVariable['logindate']))  $fileString = PutString($fileString, $configVariable['logindate'], 83 + $offset, 6);
    if (isset($configVariable['blackflag']))  $fileString = PutInteger($fileString, $configVariable['blackflag'], 89 + $offset);
    if (isset($configVariable['filebits']))   $fileString = PutInteger($fileString, $configVariable['filebits'], 91 + $offset);
    if (isset($configVariable['timeadjust'])) $fileString = PutInteger($fileString, $configVariable['timeadjust'], 93 + $offset);
    if (isset($configVariable['language']))   $fileString = PutString($fileString, $configVariable['language'], 95 + $offset, 8);
    if (isset($configVariable['extra']))      $fileString = PutString($fileString, $configVariable['extra'], 103 + $offset, 4);
    
    return $fileString;                     // Return the new file string.
}


/**
* Va-Control.inc.php :: ReadControl()
*
* This function will read the control.dat file.
*
* @return string
*/
function ReadControl() {
    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['control'] . '/CONTROL.DAT';
    $fileString = ReadCfgFile($file);       // Read the file.

    return $fileString;                     // Return the file string.
}


/**
* Va-Control.inc.php :: WriteControl()
*
* This function will write a new control.dat file based on the file string
* passed to it. If the file string is not in the correct format, then your
* control.dat file will become corrupt and the BBS could stop functioning!
*
* @param string $fileString
* @return mixed
*/
function WriteControl($fileString) {
    if (!is_string($fileString)) {
        $fileString = (string) $fileString;
    }

    global $VA_SYSPATHS;                    // Include the global variable.

    // Set the filename.
    $file = $VA_SYSPATHS['control'] . '/CONTROL.DAT';
    $status = WriteCfgFile($file, $fileString);

    return $status;                         // Return the result status.
}

?>