/*
 FileNav, File Library Navigator
 Copyright (C) 1995 by Branislav L. Slantchev

 This file is part of FileNav.

 FileNav is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; version 2.

 FileNav is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with FileNav; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#if !defined( __INIFILE_H )
	#include "inifile.h"
#endif

#if !defined( __DIRMANIP_H )
	#include "dirmanip.h"
#endif

/***************************************************************************/
/*************   Reading and writing Windows-style INI files   *************/
/***************************************************************************/

enum { _DUPE, _NODUPE };

int _openfile( const char *name, int mode );
int _closefile( void );
int _locateSec( const char *name );
int _locateVar( const char *name );
int _isSecName( const char *s );

static const char *_inipath;
static char  _tmppath[MAXPATH];
static FILE *_inifile;
static FILE *_tmpfile;
static char  _varvalue[MAX_INI_LINE];
static int   _openmode;

/*
 * prepare ini file for reading, temp file for writing
*/
	int
_openfile( const char *name, int mode )
{
	FILE *fp;

	_openmode = mode;

	_inipath = name;
	fp = fopen( name, "r" );
	if( !fp && _DUPE == mode )
		fp = fopen( name, "w" );
	if( !fp ) return -1;
	else _inifile = fp;

	if( mode == _DUPE ){
		strcpy( _tmppath, tmpname(basedir((char*)name)) );
		fp = fopen( _tmppath, "w" );
		if( !fp ) return -1;
		else _tmpfile = fp;
	}
	return 0;
}

/*
 * flush rest of ini file to tmp, unlink ini and rename the tmp;
*/
	int
_closefile( void )
{
	char buf[MAX_INI_LINE];

	if( _NODUPE == _openmode ){
		fclose( _inifile );
		return 0;
	}

	/* write rest of the ini file if necessary */
	for( ;; ){
		if( NULL == fgets( buf, MAX_INI_LINE, _inifile ) ) break;
		fputs( buf, _tmpfile );
	}

	/* cleanup */
	fclose( _inifile );
	fclose( _tmpfile );
	if( -1 == unlink( _inipath ) ) return -1;
	if( -1 == rename( _tmppath, _inipath ) ) return -1;
	return 0;
}

/*
 * True if buffer resembles a section definition
*/
	int
_isSecName( const char *buf )
{
	const char *p = buf;

	if( '[' != *p++ ) return 0;
	if( !strrchr( p, ']' ) ) return 0;
	return 1;
}


/*
 * locates a section and copies as it goes along, copies the match too!
*/
	int
_locateSec( const char *secName )
{
	char buf[MAX_INI_LINE];
	char name[81];
	int  len;

	sprintf( name, "[%s]", secName );
	len = strlen( name );

	for( ;; ){
		if( NULL == fgets( buf, MAX_INI_LINE, _inifile ) ) break;
		if( _openmode == _DUPE ) fputs( buf, _tmpfile );
		if( !memicmp( buf, name, len ) ) return 0;
	}
	return -1;
}


/*
 * locates a variable in current section only, on match doesn't copy
 * but stores in _varvalue, next sec name is not copied
*/
	int
_locateVar( const char *varName )
{
	char buf[MAX_INI_LINE];
	char *p;
	int  len;

	len = strlen( varName );

	for( ;; ){
		long pos = ftell( _inifile );
		if( NULL == fgets(buf, MAX_INI_LINE, _inifile) ) break;
		/* end of section, push section name back on stream */
		if( _isSecName(buf) ){
			fseek( _inifile, pos, SEEK_SET );
			return -1;
		}

		/* eat whitespace at beginning of variable */
		for( p = buf; isspace(*p); ++p )
			;

		/* see if name matches and if yes, look for '=' char */
		if( !memicmp( p, varName, len ) ){
			for( p = &p[len]; isspace(*p); ++p )
				;

			/* if not '=' but at end of string, then it's a match */
			/* if not at end, then it's not a match, continue     */
			if( '=' != *p ){
				if( EOS == *p ){
					_varvalue[0] = EOS;
					return 0;
				}
			}
			/* ok, we have the '=' sign, store the value after it */
			/* eat leading and trailing whitespace                */
			else{
				for( ++p; isspace(*p); ++p )
					;
				strcpy( _varvalue, p );
				for( p = &_varvalue[strlen(_varvalue)-1]; isspace(*p); --p )
					;
				*(p+1) = EOS;
				return 0;
			}
		}
		/* no match, just dump the buffer to temp file */
		if( _openmode == _DUPE ) fputs( buf, _tmpfile );
	}

	return -1;
}


/*
 * writes/modifies a var in section (if they don't exist), they're created
*/
	int
IniWrite( const char *fname, const char *secName, const char *varName,
		  const char *value )
{
	char buf[MAX_INI_LINE];

	if( -1 == _openfile( fname, _DUPE ) ) return -1;

	if( secName && (-1 == _locateSec(secName)) ){
		sprintf( buf, "\n[%s]\n", secName );
		fputs( buf, _tmpfile );
	}

	/* if no match was found and no data, then it's a new toggle */
	if( -1 == _locateVar(varName) && varName ){
		sprintf( buf, "%s\n", varName );
		fputs( buf, _tmpfile );
	}
	/* if match is found and data, create it, otherwise delete it */
	else if( value ){
		sprintf( buf, "%s=%s\n", varName, value );
		fputs( buf, _tmpfile );
	}

	if( -1 == _closefile() ) return -1;
	return 0;
}


/*
 * Reads a string value
*/
	int
IniReadStr( const char *fname, const char *secName, const char *varName,
			char *value, const char *defaultStr )
{
	int retval = 0;

	strcpy( value, defaultStr );

	if( -1 == _openfile(fname, _NODUPE ) ) return -1;

	/* if section was specified and not located, error */
	if( secName && (-1 == _locateSec( secName )) ){
		retval = -1;
		goto _iniexit;
	}

	/* try to find variable name */
	if( -1 == _locateVar(varName) ){
		retval = -1;
		goto _iniexit;
	}

	/* ok, value is in _varvalue, so that's it */
	strcpy( value, _varvalue );

_iniexit:
	if( -1 == _closefile() ) retval = -1;
	return retval;
}


/*
 * Reads an integer value
*/
	int
IniReadInt( const char *fname, const char *secName, const char *varName,
			int *value, const int defaultInt )
{
	char buf[80]; /* nice large buffer */

	*value = defaultInt;
	if( -1 == IniReadStr( fname, secName, varName, buf, 0 ) )
		return -1;
	*value = atoi( buf );
	return 0;
}


/*
 * Reads a long value
*/
	int
IniReadLong( const char *fname, const char *secName, const char *varName,
			 long *value, const long defaultLong )
{
	char buf[80];

	*value = defaultLong;
	if( -1 == IniReadStr( fname, secName, varName, buf, 0 ) )
		return -1;

	*value = atol( buf );
	return 0;
}


/*
 * Reads a boolean value
*/
	int
IniReadBool( const char *fname, const char *secName, const char *varName,
			 bool *value, const bool defaultBool )
{
	char buf[80];

	*value = defaultBool;
	if( -1 == IniReadStr( fname, secName, varName, buf, 0 ) )
		return -1;

	if( !memicmp( buf, "True", 4 ) || '1' == buf[0] ){
		*value = TRUE;
		return 0;
	}

	if( !memicmp( buf, "False", 5 ) || '0' == buf[0] ){
		*value = FALSE;
		return 0;
	}

	return -1;
}


/***************************************************************************/
/******************** ProBoard's SDK Get/SetIniVar *************************/
/***************************************************************************/
#if !defined( PB_SDK )
	bool
GetIniVar( char *fname, char *var, char *value, int len )
{
	char path[81], buf[MAX_INI_LINE];

#if 0
	char *p;

	p = strrchr( strcpy( path, fname ), '.' );
	if( !p ) strcat( path, ".INI" );
	else strcpy( p, ".INI" );
#endif

	chext( path, fname, ".INI" );

	if( -1 == IniReadStr(path, NULL, var, buf, 0 ) ) return FALSE;
	sprintf( value, "%-*s", len, buf );
	return TRUE;
}

	bool
SetIniVar( char *fname, char *var, char *value )
{
	char path[81];

#if 0
	char *p;

	p = strrchr( strcpy( path, fname ), '.' );
	if( !p ) strcat( path, ".INI" );
	else strcpy( p, ".INI" );
#endif

	chext( path, fname, ".INI" );

	if( -1 == IniWrite(path, NULL, var, value) ) return FALSE;
	return TRUE;
}
#endif
