/*
 fls MS-DOS Standard Executable
 Copyright (C) 1995 by Branislav L. Slantchev

 This file is part of the fls.

 fls is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; version 2.

 fls is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with fls; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#pragma warn -ofp

#include <stdio.h>
#include <stdlib.h>
#include <dir.h>
#include <string.h>
#include <conio.h>
#include <time.h>
#include "structs.h"
#include "getopt.h"
#include "basename.h"
#include "fail.h"
#include "listarea.h"
#include "templ.h"

#define __VERSION__  "1.0.1"

const char version[] =
	"PB-GNU fls, Version " __VERSION__ " (compiled " __DATE__ " " __TIME__ ")\n";
const char copying[] =
	"PB-GNU fls, Version " __VERSION__ "\n\n"
	"Copyright (C) 1995 by Branislav L. Slantchev\n"
	"This is a product of Silicon Creations, Inc.\n"
	"\n"
	"This program is free software; you can redistribute it and/or modify\n"
	"it under the terms of the GNU General Public License as published by\n"
	"the Free Software Foundation; either version 2 of the License or\n"
	"(at your opinion) any later version.\n"
	"\n"
	"This program is distributed in the hope that it will be useful,\n"
	"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
	"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
	"GNU General Public License for more details.\n"
	"\n"
	"You should have received a copy of the GNU General Public License\n"
	"along with the program; see the file COPYING.  If not, write to\n"
	"the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.\n";

void usage_and_die( void );

static struct option long_options[] =
{
	{"group", 1, 0, 'g'},
	{"listing", 1, 0, 'f'},
	{"header", 1, 0, 'r'},
	{"footer", 1, 0, 'o'},
	{"template", 1, 0, 't'},
	{"post", 1, 0, 'p'},
	{"min", 1, 0, 'm'},
	{"max", 1, 0, 'M'},
	{"days", 1, 0, 'n'},
	{"exclude", 0, 0, 'x'},
	{"nocdrom", 0, 0, 'e'},
	{"single", 0, 0, 's'},
	{"copying", 0, 0, 'c'},
	{"version", 0, 0, 'v'},
	{"help", 0, 0, 'h' },
	{0, 0, 0, 0}
};

ushort exclude_lines  = 0; /* exclude lines starting with ';' or '!' */
ushort exclude_cdrom  = 0; /* don't include cdrom file areas         */
ushort single_line    = 0; /* single line descriptions               */
ushort days_old       = 0; /* file newer than days_old only          */
ushort min_level      = 0; /* minimum security level to include area */
ushort max_level = 65535U; /* maximum security level to include area */
char  *listing_file;       /* name of output listing file            */

char *progname;            /* program name                           */

unsigned int area_kbytes;           /* kbytes in current area (after list)    */
unsigned int area_files;            /* files in current area (after list)     */
char area_name[80];        /* area name (before and after list)      */
int area_number;           /* area number (before and after list)    */

int total_areas = 0;       /* total number of areas                  */
long total_files = 0L;     /* total number of files                  */
long total_kbytes = 0L;    /* total kbytes in files                  */


	int
main( argc, argv )
	 int  argc;
	char *argv[];
{
	char *root_dir, areafile_path[MAXPATH];
	char *header_path, *footer_path, *template_header, *template_footer;
	char  groups[255];
	int   group_min = 0, group_max = 255, group_limits = 0;
	int   i;
	FILE *fp_list, *fp_conf;
	FILEAREA fa;

	progname = strlwr(basename( *argv ));

	if( 1 == argc ) usage_and_die();

	for( ;; ){
		int c, option_index = 0;

		c = getopt_long( argc, argv, "g:f:r:o:t:p:m:M:n:xsvceh",
			long_options, &option_index );
		if( EOF == c ) break;

		switch( c ){
			case 'g':
				char *ptr;
				/* groups in range form: [min-max] */
				if( '[' == optarg[0] ){
					ptr = strtok( &optarg[1], " -" );
					if( !ptr ) fail( "%s: syntax error: wrong [] range", progname );
					group_min = atoi( ptr );
					ptr = strtok( NULL, " ]" );
					if( !ptr ) fail( "%s: syntax error: unbalanced []", progname );
					group_max = atoi( ptr );
					group_limits = 1;
					fprintf( stderr, "\nUsing %d <= group <= %d\n",
						group_min, group_max );
				}
				/* listed, comma-separated: 23,44,12 */
				else{
					memset( groups, -1, 255 );
					for( i = 0, ptr = strtok( optarg, "," ); ptr;
						 ptr = strtok( NULL, "," ), ++i ){
						groups[i] = atoi( ptr );
					}
					group_limits = 2;
				}
			break;

			case 'r': header_path = optarg; break;
			case 'o': footer_path = optarg; break;
			case 't': template_header = optarg; break;
			case 'p': template_footer = optarg; break;

			case 'v': fail( version );
			case 'c': fail( copying );
			case 'h': usage_and_die();

			case 'f': listing_file = optarg; break;
			case 'm': min_level = (ushort)atoi(optarg);	break;
			case 'M': max_level = (ushort)atoi(optarg);	break;
			case 'n': days_old = (ushort)atoi(optarg); break;
			case 'x': exclude_lines = 1; break;
			case 'e': exclude_cdrom = 1; break;
			case 's': single_line = 1; break;

			case '?':
				fail( "%s: unrecognized option: '%s'",
					progname, argv[optind - 1] );
		}
	}

	/* see if we can access to file area definition file */
	if( !(root_dir = getenv( "PROBOARD" )) )
		fail( "%s: environment variable PROBOARD is not set.", progname );
	strcat( strcpy(areafile_path, root_dir), "/FILECFG.PRO" );
	if( !(fp_conf = fopen( areafile_path, "rb" )) )
		fail( "%s: could not read from '%s'", progname, areafile_path );

	/* see if we can create the output listing file */
	if( !listing_file )
		fail( "%s: you must use the -f [listing path] to specify output",
			progname );
	if( !(fp_list = fopen( listing_file, "w" )) )
		fail( "%s: couldn't create file '%s'", progname, listing_file );

	/* see if we have the header file */
	if( header_path ) parse_file( header_path, fp_list );

	/* process all areas */
	for( total_areas = 0; ; total_areas++ ){

		int bytes = fread( &fa, 1, sizeof(FILEAREA), fp_conf);

		/* end of file */
		if( !bytes ) break;

		/* file I/O error (read error) */
		if( sizeof(FILEAREA) != bytes )
			fail( "%s: error reading the area definition file.", progname );

		/* if min/max levels, see if we need to list area */
		if( fa.level < min_level || fa.level > max_level ) continue;

		/* see if we need to skip a cdrom area */
		if( exclude_cdrom && fa.cdrom ) continue;

		/* check if belongs to group area specs */
		if( group_limits && !fa.allGroups ){
			int break_on = 0;

			if( 2 == group_limits ){
				for( i = 0; -1 != groups[i]; ++i )
					for( int j = 0; j < 4; ++j )
						if( fa.groups[j] == groups[i] ) goto PROCESS;
				break_on = 1;
			}
			else{
				for( i = 0; i < 4; i++ )
					if( fa.groups[i] < group_min || fa.groups[i] > group_max ){
						break_on = 1;
						break;
				}
			}
			if( break_on ) continue;
		}

	PROCESS:
		/* show indicator */
		putch( '\r' );
		clreol();
		cprintf( "Area: %d  %s", total_areas, fa.name );

		strcpy( area_name, fa.name );
		area_number = total_areas + 1;

		if( template_header ) parse_file( template_header, fp_list );

		total_files += (long)listarea( total_areas + 1, &fa, fp_list );
		total_kbytes += (long)area_kbytes;

		if( template_footer ) parse_file( template_footer, fp_list );
	}

	/* see if we have the footer file */
	if( footer_path ) parse_file( footer_path, fp_list );

	/* print some statistics */
	putch( '\r' );
	clreol();
	cprintf( "Total areas scanned: %d\n\r", total_areas );
	cprintf( "Total files found: %ld\n\r", total_files );

	/* cleanup */
	fclose( fp_list );
	fclose( fp_conf );
	if( !total_files ) unlink( listing_file );
	return 0;
}


/*
** prints usage info and terminates program
*/
	void
usage_and_die( void )
{
	fprintf( stderr, "%s\nUsage: %s [short or long options] -f listing\n\n",
		version, progname );

	fputs(
"Options:           Long options:             Description:\n"
"  -f listing file    +listing=listing file     output file path\n"
"  -g number          +group number             include areas in group(s)\n"
"  -r header file     +header=header file       header file path\n"
"  -o footer file     +footer=footer file       footer file path\n"
"  -t area header     +template=template file   area header template path\n"
"  -p area footer     +post=template file       area footer template path\n"
"  -m level           +min=level                minimum security level\n"
"  -M level           +max=level                maximum security level\n"
"  -n days            +days=number              list file newer than days\n"
"  -x                 +exclude                  exclude lines '!' or ';'\n"
"  -e                 +nocdrom                  exclude cdrom file areas\n"
"  -s                 +single                   single line descriptions\n"
"  -v                 +version                  print version information\n"
"  -c                 +copying                  print copyright info\n"
"  -h                 +help                     print this help screen\n",
		stderr );
	exit( EXIT_SUCCESS );
}
