/*
 * ChkPro v1.0r4 - ProBoard Configuration Checking Utility
 *
 * Copyright (C) 1995 by Branislav L. Slantchev
 * A product of Silicon Creations, part of the PB-GNU Project
 *
 * For details, see the file "copying".
 *
*/
#include <stdio.h>
#include <direct.h>
#include <proutil.h>
#include <fsys.h>

/* these functions are defined in the main module */
extern void _pascal logexit(const char *format, ...);
extern void _pascal log(const char *format, ...);

/* local global variables and functions */
static byte groupDef[32]; /* you can't have more than 255 groups */
static bool _pascal getHudson(char *);

/*************************************************************************\
 ** Local helper functions
\*************************************************************************/
	bool _pascal
getHudson(char *path)
{
	FILE   *fpSrc;
	CONFIG  buf;
	char    curDir[MAXPATH];

	if( NULL == (fpSrc = pbfopen(fnCONFIG, "rb")) )
		logexit("Fatal: could not open config file '%s'\n", fnCONFIG);
	fread(&buf, sizeof(buf), 1, fpSrc);
	fclose(fpSrc);

	strcpy(path, buf.msgpath);
	getcwd(curDir, sizeof(curDir));
	if( -1 == chdir(path) ) return FALSE;
	chdir(curDir);
	return TRUE;
}

/*************************************************************************\
 ** Main processing function
\*************************************************************************/
	int _pascal
do_message(bool repair)
{
	int   retval = 0, i;
	char  curDir[MAXPATH], hudsonPath[MAXPATH];
	bool  hudsonValid;
	FILE *fpSrc;

	/* set the message group bits */
	memset(groupDef, 0, sizeof(groupDef));
	getcwd(curDir, sizeof(curDir));
	hudsonValid = getHudson(hudsonPath);

	/* open the area definition file and assign a large buffer */
	if( NULL == (fpSrc = pbfopen(fnMESSAGES, "rb")) )
		logexit("Fatal: could not open messagebase file '%s'\n", fnMESSAGES);
	setvbuf(fpSrc, NULL, _IOFBF, 0x1000);

	for( ;; ){
		MESSAGES buf;

		/* read a message area record */
		if( 1 != fread(&buf, sizeof(buf), 1, fpSrc) ){
			if( feof(fpSrc) ) break;
			logexit("fread() failed on file '%s'\n", fnMESSAGES);
		}

		if( EOS == buf.name[0] ) continue;

		/* if it's Hudson, check directory */
		if( MSGBASE_HUDSON == buf.msgBaseType ){
			if( 200 < buf.areaNum ){
				log("Area %d: Hudson board exceeds 200\n", buf.areaNum);
				retval = -1;
			}
			if( !hudsonValid ){
				log("Area %d: Hudson path invalid - '%s'\n", buf.areaNum,
					hudsonPath);
				retval = -1;
			}
		}
		/* if it's Fido *.MSG, chek directory too */
		else if( MSGBASE_SDM == buf.msgBaseType ){
			if( -1 == chdir(buf.path) ){
				int stat = 0;

				if( repair ) stat = mkDirTree(buf.path);
				log("Area %d: *.MSG directory does not exist%s\n",
					buf.areaNum, repair ?
					((-1 == stat) ? "...create failed" : "...created") : "");
				retval = -1;
			}
			else chdir(curDir);
		}
		/* for Squish, check presence of files */
		else if( MSGBASE_SQUISH == buf.msgBaseType ){
			char path[MAXPATH];

			strcat( strcpy(path, buf.path), ".SQD" );
			if( -1 == access(path, 0) ){
				log("Area %d: Squish files missing.\n", buf.areaNum);
				retval = -1;
			}
		}
		/* for JAM, check presence of files */
		else /* if( MSGBASE_JAM == buf.msgBaseType ) */{
			char path[MAXPATH];

			strcat( strcpy(path, buf.path), ".JDT" );
			if( -1 == access(path, 0) ){
				log("Area %d: JAM files missing.\n", buf.areaNum);
				retval = -1;
			}
		}

		/* set the group numbers */
		for( i = 0; i < 4; ++i ){
			int n = buf.groups[i] - 1;

			if( 0 != buf.groups[i] ) groupDef[n >> 3] |= 1 << (n & 0x07);
		}
	}
	/* close the area file */
	fclose(fpSrc);

	/* now check for orphaned group entries */
	if( NULL == (fpSrc = pbfopen(fnMGROUPS, "rb")) ){
		log("Could not open message group definition file '%s'\n", fnMGROUPS);
		goto _endCheck;
	}

	for( i = 0; ; i++ ){
		MSGROUP buf;

		if( 1 != fread(&buf, sizeof(buf), 1, fpSrc) ){
			if( feof(fpSrc) ) break;
            logexit("fread() failed on the group file '%s'\n", fnMGROUPS);
		}

		if( EOS == buf.name[0] ) continue;
		if( !(groupDef[i >> 3] & (1 << (i & 0x07))) ){
			log("Message group %d is not used for any area.\n", i + 1);
			retval = -1;
		}
	}

	fclose(fpSrc);
_endCheck:
	return retval;
}

/*************************************************************************\
 ** Packing function
\*************************************************************************/
 	int _pascal
do_packmsg( void )
{
	FILE *fpSrc, *fpDest;
    char  oldPath[MAXPATH], newPath[MAXPATH];

    chext(newPath, fnMESSAGES, ".CHK");
 	if( NULL == (fpSrc = pbfopen(fnMESSAGES, "rb")) )
		logexit("Fatal: could not open messagebase file '%s'\n", fnMESSAGES);
	setvbuf(fpSrc, NULL, _IOFBF, 0x1000);
 	if( NULL == (fpDest = pbfopen(newPath, "wb")) )
		logexit("Fatal: could not create file '%s'\n", newPath);

    for( ;; ){
		MESSAGES buf;

		/* read a message area record */
		if( 1 != fread(&buf, sizeof(buf), 1, fpSrc) ){
			if( feof(fpSrc) ) break;
			logexit("fread() failed on file '%s'\n", fnMESSAGES);
		}

		if( EOS == buf.name[0] ) continue;  /* skip the empty area */

        /* write this one to the new file */
        fwrite(&buf, sizeof(buf), 1, fpDest);
	}

    fclose(fpSrc);
    fclose(fpDest);

    /* now, backup the old file and rename the new one */
    strcpy(oldPath, _mkSysPath(fnMESSAGES));
    chext(newPath, oldPath, ".BAK");
    if( 0 == access(newPath, 0) ) rmfile(newPath);
    rename(oldPath, newPath);
    chext(newPath, oldPath, ".CHK");
    rename(newPath, oldPath);

    return 0;
}
