unit vsop;



{  Calculates the planetary heliocentric coordinates according to the
   VSOP87 theory. Calculations according to chapter 32 (31) of Meeus. }
{$mode objfpc}{$H+}
(* $define meeus *)   { Only use the accuracy as in the Meeus book }

(*$ifdef delphi_1 *)
  (*$define meeus *) { Otherwise the code segment will be too small }
(*$endif *)

(*@/// interface *)
interface

(*@/// uses *)
uses
  ah_math,
  sysutils;
(*@\\\000000020B*)

type
  (*@/// TVSOPEntry=record *)
  TVSOPEntry=record
    A,B,C: extended;
    end;
  (*@\\\*)
  TVSOPCalcFunc = function (nr,index: integer):TVSOPEntry of object;
  (*@/// TVSOP=class(TObject) *)
  TVSOP=class(TObject)
  protected
    FDate: TDateTime;
    function LongitudeFactor(nr,index: integer):TVSOPEntry;  VIRTUAL; abstract;
    function LatitudeFactor(nr,index: integer):TVSOPEntry;  VIRTUAL; abstract;
    function RadiusFactor(nr,index: integer):TVSOPEntry;  VIRTUAL; abstract;
    function CalcLongitude:extended;
    function CalcLatitude:extended;
    function CalcRadius:extended;
    function Calc(factor: TVSOPCalcFunc):extended;
    procedure SetDate(value: TDateTime);
    function Tau:extended;
  public
    procedure DynamicToFK5(var longitude,latitude: extended);
    property Longitude:extended read CalcLongitude;
    property Latitude:extended read CalcLatitude;
    property Radius:extended read CalcRadius;
    property Date:TDateTime write SetDate;
    end;
  (*@\\\0000000E01*)
  TCVSOP=class of TVSOP;
  (*@/// TVSOPEarth=class(TVSOP) *)
  TVSOPEarth=class(TVSOP)
  protected
    function LongitudeFactor(nr,index: integer):TVSOPEntry;  override;
    function LatitudeFactor(nr,index: integer):TVSOPEntry;  override;
    function RadiusFactor(nr,index: integer):TVSOPEntry;  override;
    end;
  (*@\\\0000000607*)
  (*@/// TVSOPJupiter=class(TVSOP) *)
  TVSOPJupiter=class(TVSOP)
  protected
    function LongitudeFactor(nr,index: integer):TVSOPEntry;  override;
    function LatitudeFactor(nr,index: integer):TVSOPEntry;  override;
    function RadiusFactor(nr,index: integer):TVSOPEntry;  override;
    end;
  (*@\\\0000000607*)

procedure earth_coord(date:TdateTime; var l,b,r: extended);
procedure jupiter_coord(date:TdateTime; var l,b,r: extended);
(*@\\\0000000301*)
(*@/// implementation *)
implementation

uses
  moon;

(*$ifdef delphi_ge_3 *)
var
(*$else *)
const
(*$endif *)
  datetime_2000_01_01: extended = 0;

(*@/// procedure calc_coord(date: TDateTime; obj_class: TCVSOP; var l,b,r: extended); *)
procedure calc_coord(date: TDateTime; obj_class: TCVSOP; var l,b,r: extended);
var
  obj: TVSOP;
begin
  obj:=NIL;
  try
    obj:=obj_class.Create;
    obj.date:=date;
    r:=obj.radius;
    l:=obj.longitude;
    b:=obj.latitude;
    obj.DynamicToFK5(l,b);
  finally
    obj.free;
    end;
  l:=put_in_360(rad2deg(l));  (* rad -> degree *)
  b:=rad2deg(b);
  end;
(*@\\\0000001111*)

(*@/// procedure earth_coord(date:TdateTime; var l,b,r: extended); *)
procedure earth_coord(date:TdateTime; var l,b,r: extended);
begin
  calc_coord(date,TVSOPEarth,l,b,r);
  end;
(*@\\\0000000116*)
(*@/// procedure jupiter_coord(date:TdateTime; var l,b,r: extended); *)
procedure jupiter_coord(date:TdateTime; var l,b,r: extended);
begin
  calc_coord(date,TVSOPJupiter,l,b,r);
  end;
(*@\\\000000031C*)

(*@/// class TVSOP *)
(*@/// function TVSOP.CalcLongitude:extended; *)
function TVSOP.CalcLongitude:extended;
begin
  result:=calc(@Longitudefactor);
  end;
(*@\\\0000000401*)
(*@/// function TVSOP.CalcLatitude:extended; *)
function TVSOP.CalcLatitude:extended;
begin
  result:=calc(@Latitudefactor);
  end;
(*@\\\000000031F*)
(*@/// function TVSOP.CalcRadius:extended; *)
function TVSOP.CalcRadius:extended;
begin
  result:=calc(@radiusfactor);
  end;
(*@\\\000000031D*)
(*@/// procedure TVSOP.SetDate(value: TDateTime); *)
procedure TVSOP.SetDate(value: TDateTime);
begin
  FDate:=value;
  end;
(*@\\\*)
(*@/// function TVSOP.Tau:extended; *)
function TVSOP.Tau:extended;
begin
  result:=(FDate-datetime_2000_01_01-0.5)/365250.0;
  end;
(*@\\\0000000301*)
(*@/// function TVSOP.Calc(factor: TVSOPCalcFunc):extended; *)
function TVSOP.Calc(factor: TVSOPCalcFunc):extended;
var
  t: extended;
  current: extended;
  r: array[0..5] of extended;
  i,j: integer;
begin
  t:=Tau;
  for j:=0 to 5 do begin
    r[j]:=0;
    i:=0;
    repeat
      WITH Factor(i,j) do
        current:=a*cos(b+c*t);
      r[j]:=r[j]+current;
      inc(i);
    until current=0;
    end;
  result:=(r[0]+t*(r[1]+t*(r[2]+t*(r[3]+t*(r[4]+t*r[5])))))*1e-8;
  end;
(*@\\\0000000E17*)
(*@/// procedure TVSOP.DynamicToFK5(var longitude,latitude: extended); *)
procedure TVSOP.DynamicToFK5(var longitude,latitude: extended);
var
  lprime,t: extended;
  delta_l, delta_b: extended;
begin
  t:=10*tau;
  lprime:=longitude+deg2rad(-1.397-0.00031*t)*t;
  delta_l:=-deg2rad(0.09033/3600)+deg2rad(0.03916/3600)*(cos(lprime)+sin(lprime))*tan(latitude);
  delta_b:=deg2rad(0.03916/3600)*(cos(lprime)-sin(lprime));
  longitude:=longitude+delta_l;
  latitude:=latitude+delta_b;
  end;
(*@\\\*)
(*@\\\0000000226*)
(*@/// class TVSOPEarth *)
(*@/// function TVSOPEarth.RadiusFactor(nr,index: integer):TVSOPEntry; *)
function TVSOPEarth.RadiusFactor(nr,index: integer):TVSOPEntry;
const
  (*@/// vsop87_ear_r0:array[0..525,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_r0:array[0.. 39,0..2] of extended = (
  (*$else *)
  vsop87_ear_r0:array[0..525,0..2] of extended = (
  (*$endif *)
  { 4330    1 }  ( 100013988.799, 0.00000000000,      0.00000000000 ),
  { 4330    2 }  (   1670699.626, 3.09846350771,   6283.07584999140 ),
  { 4330    3 }  (     13956.023, 3.05524609620,  12566.15169998280 ),
  { 4330    4 }  (      3083.720, 5.19846674381,  77713.77146812050 ),
  { 4330    5 }  (      1628.461, 1.17387749012,   5753.38488489680 ),
  { 4330    6 }  (      1575.568, 2.84685245825,   7860.41939243920 ),
  { 4330    7 }  (       924.799, 5.45292234084,  11506.76976979360 ),
  { 4330    8 }  (       542.444, 4.56409149777,   3930.20969621960 ),
  { 4330    9 }  (       472.110, 3.66100022149,   5884.92684658320 ),
  { 4330   10 }  (       328.780, 5.89983646482,   5223.69391980220 ),
  { 4330   11 }  (       345.983, 0.96368617687,   5507.55323866740 ),
  { 4330   12 }  (       306.784, 0.29867139512,   5573.14280143310 ),
  { 4330   13 }  (       174.844, 3.01193636534,  18849.22754997420 ),
  { 4330   14 }  (       243.189, 4.27349536153,  11790.62908865880 ),
  { 4330   15 }  (       211.829, 5.84714540314,   1577.34354244780 ),
  { 4330   16 }  (       185.752, 5.02194447178,  10977.07880469900 ),
  { 4330   17 }  (       109.835, 5.05510636285,   5486.77784317500 ),
  { 4330   18 }  (        98.316, 0.88681311277,   6069.77675455340 ),
  { 4330   19 }  (        86.499, 5.68959778254,  15720.83878487840 ),
  { 4330   20 }  (        85.825, 1.27083733351, 161000.68573767410 ),
  { 4330   21 }  (        62.916, 0.92177108832,    529.69096509460 ),
  { 4330   22 }  (        57.056, 2.01374292014,  83996.84731811189 ),
  { 4330   23 }  (        64.903, 0.27250613787,  17260.15465469040 ),
  { 4330   24 }  (        49.384, 3.24501240359,   2544.31441988340 ),
  { 4330   25 }  (        55.736, 5.24159798933,  71430.69561812909 ),
  { 4330   26 }  (        42.515, 6.01110242003,   6275.96230299060 ),
  { 4330   27 }  (        46.963, 2.57805070386,    775.52261132400 ),
  { 4330   28 }  (        38.968, 5.36071738169,   4694.00295470760 ),
  { 4330   29 }  (        44.661, 5.53715807302,   9437.76293488700 ),
  { 4330   30 }  (        35.660, 1.67468058995,  12036.46073488820 ),
  { 4330   31 }  (        31.921, 0.18368229781,   5088.62883976680 ),
  { 4330   32 }  (        31.846, 1.77775642085,    398.14900340820 ),
  { 4330   33 }  (        33.193, 0.24370300098,   7084.89678111520 ),
  { 4330   34 }  (        38.245, 2.39255343974,   8827.39026987480 ),
  { 4330   35 }  (        28.464, 1.21344868176,   6286.59896834040 ),
  { 4330   36 }  (        37.490, 0.82952922332,  19651.04848109800 ),
  { 4330   37 }  (        36.957, 4.90107591914,  12139.55350910680 ),
  { 4330   38 }  (        34.537, 1.84270693282,   2942.46342329160 ),
  { 4330   39 }  (        26.275, 4.58896850401,  10447.38783960440 ),
  (*$ifndef meeus *)
  { 4330   40 }  (        24.596, 3.78660875483,   8429.24126646660 ),
  { 4330   41 }  (        23.587, 0.26866117066,    796.29800681640 ),
  (*$endif *)
  { 4330   42 }  (        27.793, 1.89934330904,   6279.55273164240 )
  (*$ifndef meeus *)                                                 ,
  { 4330   43 }  (        23.927, 4.99598548138,   5856.47765911540 ),
  { 4330   44 }  (        20.349, 4.65267995431,   2146.16541647520 ),
  { 4330   45 }  (        23.287, 2.80783650928,  14143.49524243060 ),
  { 4330   46 }  (        22.103, 1.95004702988,   3154.68708489560 ),
  { 4330   47 }  (        19.506, 5.38227371393,   2352.86615377180 ),
  { 4330   48 }  (        17.958, 0.19871379385,   6812.76681508600 ),
  { 4330   49 }  (        17.174, 4.43315560735,  10213.28554621100 ),
  { 4330   50 }  (        16.190, 5.23160507859,  17789.84561978500 ),
  { 4330   51 }  (        17.314, 6.15200787916,  16730.46368959580 ),
  { 4330   52 }  (        13.814, 5.18962074032,   8031.09226305840 ),
  { 4330   53 }  (        18.833, 0.67306674027, 149854.40013480789 ),
  { 4330   54 }  (        18.331, 2.25348733734,  23581.25817731760 ),
  { 4330   55 }  (        13.641, 3.68516118804,   4705.73230754360 ),
  { 4330   56 }  (        13.139, 0.65289581324,  13367.97263110660 ),
  { 4330   57 }  (        10.414, 4.33285688538,  11769.85369316640 ),
  { 4330   58 }  (         9.978, 4.20126336355,   6309.37416979120 ),
  { 4330   59 }  (        10.169, 1.59390681369,   4690.47983635860 ),
  { 4330   60 }  (         7.564, 2.62560597390,   6256.77753019160 ),
  { 4330   61 }  (         9.661, 3.67586791220,  27511.46787353720 ),
  { 4330   62 }  (         6.743, 0.56270332741,   3340.61242669980 ),
  { 4330   63 }  (         8.743, 6.06359123461,   1748.01641306700 ),
  { 4330   64 }  (         7.786, 3.67371235637,  12168.00269657460 ),
  { 4330   65 }  (         6.633, 5.66149277792,  11371.70468975820 ),
  { 4330   66 }  (         7.712, 0.31242577789,   7632.94325965020 ),
  { 4330   67 }  (         6.592, 3.13576266188,    801.82093112380 ),
  { 4330   68 }  (         7.460, 5.64757188143,  11926.25441366880 ),
  { 4330   69 }  (         6.933, 2.92384586400,   6681.22485339960 ),
  { 4330   70 }  (         6.802, 1.42329806420,  23013.53953958720 ),
  { 4330   71 }  (         6.115, 5.13393615454,   1194.44701022460 ),
  { 4330   72 }  (         6.477, 2.64986648492,  19804.82729158280 ),
  { 4330   73 }  (         5.233, 4.62434053374,   6438.49624942560 ),
  { 4330   74 }  (         6.147, 3.02863936662, 233141.31440436149 ),
  { 4330   75 }  (         4.608, 1.72194702724,   7234.79425624200 ),
  { 4330   76 }  (         4.221, 1.55697533729,   7238.67559160000 ),
  { 4330   77 }  (         5.314, 2.40716580847,  11499.65622279280 ),
  { 4330   78 }  (         5.128, 5.32398965690,  11513.88331679440 ),
  { 4330   79 }  (         4.770, 0.25554312006,  11856.21865142450 ),
  { 4330   80 }  (         5.519, 2.09089154502,  17298.18232732620 ),
  { 4330   81 }  (         5.625, 4.34052903053,  90955.55169449610 ),
  { 4330   82 }  (         4.578, 4.46569641570,   5746.27133789600 ),
  { 4330   83 }  (         3.788, 4.90729383510,   4164.31198961300 ),
  { 4330   84 }  (         5.337, 5.09957905104,  31441.67756975680 ),
  { 4330   85 }  (         3.967, 1.20054555174,   1349.86740965880 ),
  { 4330   86 }  (         4.008, 3.03007204392,   1059.38193018920 ),
  { 4330   87 }  (         3.476, 0.76080277030,  10973.55568635000 ),
  { 4330   88 }  (         4.232, 1.05485713117,   5760.49843189760 ),
  { 4330   89 }  (         4.582, 3.76570026763,   6386.16862421000 ),
  { 4330   90 }  (         3.335, 3.13829943354,   6836.64525283380 ),
  { 4330   91 }  (         3.418, 3.00072390334,   4292.33083295040 ),
  { 4330   92 }  (         3.598, 5.70718084323,   5643.17856367740 ),
  { 4330   93 }  (         3.237, 4.16448773994,   9917.69687450980 ),
  { 4330   94 }  (         4.154, 2.59941292162,   7058.59846131540 ),
  { 4330   95 }  (         3.362, 4.54577697964,   4732.03062734340 ),
  { 4330   96 }  (         2.978, 1.30561268820,   6283.14316029419 ),
  { 4330   97 }  (         2.765, 0.51311975679,     26.29831979980 ),
  { 4330   98 }  (         2.802, 5.66263240521,   8635.94200376320 ),
  { 4330   99 }  (         2.927, 5.73787481548,  16200.77272450120 ),
  { 4330  100 }  (         3.164, 1.69140262657,  11015.10647733480 ),
  { 4330  101 }  (         2.598, 2.96244118586,  25132.30339996560 ),
  { 4330  102 }  (         3.519, 3.62639325753, 244287.60000722769 ),
  { 4330  103 }  (         2.676, 4.20725700850,  18073.70493865020 ),
  { 4330  104 }  (         2.978, 1.74971565805,   6283.00853968860 ),
  { 4330  105 }  (         2.287, 1.06975704977,  14314.16811304980 ),
  { 4330  106 }  (         2.863, 5.92838131397,  14712.31711645800 ),
  { 4330  107 }  (         3.071, 0.23793217002,  35371.88726597640 ),
  { 4330  108 }  (         2.656, 0.89959301780,  12352.85260454480 ),
  { 4330  109 }  (         2.415, 2.79975176257,    709.93304855830 ),
  { 4330  110 }  (         2.814, 3.51488206882,  21228.39202354580 ),
  { 4330  111 }  (         1.977, 2.61358297550,    951.71840625060 ),
  { 4330  112 }  (         2.548, 2.47684686575,   6208.29425142410 ),
  { 4330  113 }  (         1.999, 0.56090388160,   7079.37385680780 ),
  { 4330  114 }  (         2.305, 1.05376461628,  22483.84857449259 ),
  { 4330  115 }  (         1.855, 2.86090681163,   5216.58037280140 ),
  { 4330  116 }  (         2.157, 1.31396741861, 154717.60988768269 ),
  { 4330  117 }  (         1.970, 4.36929875289, 167283.76158766549 ),
  { 4330  118 }  (         1.635, 5.85571606764,  10984.19235169980 ),
  { 4330  119 }  (         1.754, 2.14452408833,   6290.18939699220 ),
  { 4330  120 }  (         2.154, 6.03828341543,  10873.98603048040 ),
  { 4330  121 }  (         1.714, 3.70157691113,   1592.59601363280 ),
  { 4330  122 }  (         1.541, 6.21598380732,  23543.23050468179 ),
  { 4330  123 }  (         1.611, 1.99824499377,  10969.96525769820 ),
  { 4330  124 }  (         1.712, 1.34295663542,   3128.38876509580 ),
  { 4330  125 }  (         1.642, 5.55026665339,   6496.37494542940 ),
  { 4330  126 }  (         1.502, 5.43948825854,    155.42039943420 ),
  { 4330  127 }  (         1.827, 5.91227480261,   3738.76143010800 ),
  { 4330  128 }  (         1.726, 2.16764983583,  10575.40668294180 ),
  { 4330  129 }  (         1.532, 5.35683107070,  13521.75144159140 ),
  { 4330  130 }  (         1.829, 1.66006148731,  39302.09696219600 ),
  { 4330  131 }  (         1.605, 1.90928637633,   6133.51265285680 ),
  { 4330  132 }  (         1.282, 2.46014880418,  13916.01910964160 ),
  { 4330  133 }  (         1.211, 4.41360631550,   3894.18182954220 ),
  { 4330  134 }  (         1.394, 1.77801929354,   9225.53927328300 ),
  { 4330  135 }  (         1.571, 4.95512957592,  25158.60171976540 ),
  { 4330  136 }  (         1.205, 1.19212540615,      3.52311834900 ),
  { 4330  137 }  (         1.132, 2.69830084955,   6040.34724601740 ),
  { 4330  138 }  (         1.504, 5.77002730341,  18209.33026366019 ),
  { 4330  139 }  (         1.393, 1.62621805428,   5120.60114558360 ),
  { 4330  140 }  (         1.077, 2.93931554233,  17256.63153634140 ),
  { 4330  141 }  (         1.232, 0.71655165307, 143571.32428481648 ),
  { 4330  142 }  (         1.087, 0.99769687939,    955.59974160860 ),
  { 4330  143 }  (         1.068, 5.28472576231,  65147.61976813770 ),
  { 4330  144 }  (         0.980, 5.10949204607,   6172.86952877200 ),
  { 4330  145 }  (         1.169, 3.11664290862,  14945.31617355440 ),
  { 4330  146 }  (         1.202, 4.02992510402,    553.56940284240 ),
  { 4330  147 }  (         0.979, 2.00000879212,  15110.46611986620 ),
  { 4330  148 }  (         0.962, 4.02380771400,   6282.09552892320 ),
  { 4330  149 }  (         0.999, 3.62643002790,   6262.30045449900 ),
  { 4330  150 }  (         1.030, 5.84989900289,    213.29909543800 ),
  { 4330  151 }  (         1.014, 2.84221578218,   8662.24032356300 ),
  { 4330  152 }  (         1.185, 1.51330541132,  17654.78053974960 ),
  { 4330  153 }  (         0.967, 2.67081017562,   5650.29211067820 ),
  { 4330  154 }  (         1.222, 2.65423784904,  88860.05707098669 ),
  { 4330  155 }  (         0.981, 2.36370360283,   6206.80977871580 ),
  { 4330  156 }  (         1.033, 0.13874927606,  11712.95531823080 ),
  { 4330  157 }  (         1.103, 3.08477302937,  43232.30665841560 ),
  { 4330  158 }  (         0.781, 2.53372735932,  16496.36139620240 ),
  { 4330  159 }  (         1.019, 3.04569392376,   6037.24420376200 ),
  { 4330  160 }  (         0.795, 5.80662989111,   5230.80746680300 ),
  { 4330  161 }  (         0.813, 3.57710279439,  10177.25767953360 ),
  { 4330  162 }  (         0.962, 5.31470594766,   6284.05617105960 ),
  { 4330  163 }  (         0.721, 5.96264301567,  12559.03815298200 ),
  { 4330  164 }  (         0.966, 2.74714939953,   6244.94281435360 ),
  { 4330  165 }  (         0.921, 0.10155275926,  29088.81141598500 ),
  { 4330  166 }  (         0.692, 3.89764447548,   1589.07289528380 ),
  { 4330  167 }  (         0.719, 5.91791450402,   4136.91043351620 ),
  { 4330  168 }  (         0.772, 4.05505682353,   6127.65545055720 ),
  { 4330  169 }  (         0.712, 5.49291532439,  22003.91463486980 ),
  { 4330  170 }  (         0.672, 1.60700490811,  11087.28512591840 ),
  { 4330  171 }  (         0.690, 4.50539825563,    426.59819087600 ),
  { 4330  172 }  (         0.854, 3.26104981596,  20426.57109242200 ),
  { 4330  173 }  (         0.656, 4.32410182940,  16858.48253293320 ),
  { 4330  174 }  (         0.840, 2.59572585222,  28766.92442448400 ),
  { 4330  175 }  (         0.692, 0.61650089011,  11403.67699557500 ),
  { 4330  176 }  (         0.700, 3.40901167143,      7.11354700080 ),
  { 4330  177 }  (         0.726, 0.04243053594,   5481.25491886760 ),
  { 4330  178 }  (         0.557, 4.78317696534,  20199.09495963300 ),
  { 4330  179 }  (         0.649, 1.04027912958,   6062.66320755260 ),
  { 4330  180 }  (         0.633, 5.70229959167,  45892.73043315699 ),
  { 4330  181 }  (         0.592, 6.11836729658,   9623.68827669120 ),
  { 4330  182 }  (         0.523, 3.62840021266,   5333.90024102160 ),
  { 4330  183 }  (         0.604, 5.57734696185,  10344.29506538580 ),
  { 4330  184 }  (         0.496, 2.21023499449,   1990.74501704100 ),
  { 4330  185 }  (         0.691, 1.96071732602,  12416.58850284820 ),
  { 4330  186 }  (         0.640, 1.59074172032,  18319.53658487960 ),
  { 4330  187 }  (         0.625, 3.82362791378,  13517.87010623340 ),
  { 4330  188 }  (         0.663, 5.08444996779,    283.85931886520 ),
  { 4330  189 }  (         0.475, 1.17025894287,  12569.67481833180 ),
  { 4330  190 }  (         0.664, 4.50029469969,  47162.51635463520 ),
  { 4330  191 }  (         0.569, 0.16310365162,  17267.26820169119 ),
  { 4330  192 }  (         0.568, 3.86100969474,   6076.89030155420 ),
  { 4330  193 }  (         0.539, 4.83282276086,  18422.62935909819 ),
  { 4330  194 }  (         0.466, 0.75872342878,   7342.45778018060 ),
  { 4330  195 }  (         0.541, 3.07212190507, 226858.23855437008 ),
  { 4330  196 }  (         0.458, 0.26774483096,   4590.91018048900 ),
  { 4330  197 }  (         0.610, 1.53597051291,  33019.02111220460 ),
  { 4330  198 }  (         0.617, 2.62356328726,  11190.37790013700 ),
  { 4330  199 }  (         0.548, 4.55798855791,  18875.52586977400 ),
  { 4330  200 }  (         0.633, 4.60110281228,  66567.48586525429 ),
  { 4330  201 }  (         0.596, 5.78202396722,    632.78373931320 ),
  { 4330  202 }  (         0.533, 5.01786882904,  12132.43996210600 ),
  { 4330  203 }  (         0.603, 5.38458554802, 316428.22867391503 ),
  { 4330  204 }  (         0.469, 0.59168241917,  21954.15760939799 ),
  { 4330  205 }  (         0.548, 3.50613163558,  17253.04110768959 ),
  { 4330  206 }  (         0.502, 0.98804327589,  11609.86254401220 ),
  { 4330  207 }  (         0.568, 1.98497313089,   7668.63742494250 ),
  { 4330  208 }  (         0.482, 1.62141803864,  12146.66705610760 ),
  { 4330  209 }  (         0.391, 3.68718382989,  18052.92954315780 ),
  { 4330  210 }  (         0.457, 3.77205737340, 156137.47598479928 ),
  { 4330  211 }  (         0.401, 5.28260651958,  15671.08175940660 ),
  { 4330  212 }  (         0.469, 1.80963184268,  12562.62858163380 ),
  { 4330  213 }  (         0.508, 3.36399024699,  20597.24396304120 ),
  { 4330  214 }  (         0.450, 5.66054299250,  10454.50138660520 ),
  { 4330  215 }  (         0.375, 4.98534633105,   9779.10867612540 ),
  { 4330  216 }  (         0.523, 0.97215560834, 155427.54293624099 ),
  { 4330  217 }  (         0.403, 5.13939866506,   1551.04522264800 ),
  { 4330  218 }  (         0.372, 3.69883738807,   9388.00590941520 ),
  { 4330  219 }  (         0.367, 4.43875659716,   4535.05943692440 ),
  { 4330  220 }  (         0.406, 4.20863156600,  12592.45001978260 ),
  { 4330  221 }  (         0.360, 2.53924644657,    242.72860397400 ),
  { 4330  222 }  (         0.471, 4.61907324819,   5436.99301524020 ),
  { 4330  223 }  (         0.441, 5.83872966262,   3496.03282613400 ),
  { 4330  224 }  (         0.385, 4.94496680973,  24356.78078864160 ),
  { 4330  225 }  (         0.349, 6.15018231784,  19800.94595622480 ),
  { 4330  226 }  (         0.355, 0.21895678106,   5429.87946823940 ),
  { 4330  227 }  (         0.344, 5.62993724928,   2379.16447357160 ),
  { 4330  228 }  (         0.380, 2.72105213143,  11933.36796066960 ),
  { 4330  229 }  (         0.432, 0.24221790536,  17996.03116822220 ),
  { 4330  230 }  (         0.378, 5.22517556974,   7477.52286021600 ),
  { 4330  231 }  (         0.337, 5.10888041439,   5849.36411211460 ),
  { 4330  232 }  (         0.315, 0.57827745123,  10557.59416082380 ),
  { 4330  233 }  (         0.318, 4.49953141399,   3634.62102451840 ),
  { 4330  234 }  (         0.323, 1.54274281393,  10440.27429260360 ),
  { 4330  235 }  (         0.309, 5.76839284397,     20.77539549240 ),
  { 4330  236 }  (         0.301, 2.34727604008,   4686.88940770680 ),
  { 4330  237 }  (         0.414, 5.93237602310,  51092.72605085480 ),
  { 4330  238 }  (         0.361, 2.16398609550,  28237.23345938940 ),
  { 4330  239 }  (         0.288, 0.18376252189,  13095.84266507740 ),
  { 4330  240 }  (         0.277, 5.12952205045,  13119.72110282519 ),
  { 4330  241 }  (         0.327, 6.19222146204,   6268.84875598980 ),
  { 4330  242 }  (         0.273, 0.30522428863,  23141.55838292460 ),
  { 4330  243 }  (         0.267, 5.76152585786,   5966.68398033480 ),
  { 4330  244 }  (         0.308, 5.99280509979,  22805.73556599360 ),
  { 4330  245 }  (         0.345, 2.92489919444,  36949.23080842420 ),
  { 4330  246 }  (         0.253, 5.20995219509,  24072.92146977640 ),
  { 4330  247 }  (         0.342, 5.72702586209,  16460.33352952499 ),
  { 4330  248 }  (         0.261, 2.00304796059,   6148.01076995600 ),
  { 4330  249 }  (         0.238, 5.08264392839,   6915.85958930460 ),
  { 4330  250 }  (         0.249, 2.94762789744,    135.06508003540 ),
  { 4330  251 }  (         0.306, 3.89764686987,  10988.80815753500 ),
  { 4330  252 }  (         0.305, 0.05827812117,   4701.11650170840 ),
  { 4330  253 }  (         0.319, 2.95712862064, 163096.18036118349 ),
  { 4330  254 }  (         0.209, 4.43768461442,   6546.15977336420 ),
  { 4330  255 }  (         0.270, 2.06643178717,   4804.20927592700 ),
  { 4330  256 }  (         0.217, 0.73691592312,   6303.85124548380 ),
  { 4330  257 }  (         0.206, 0.32075959415,  25934.12433108940 ),
  { 4330  258 }  (         0.218, 0.18428135264,  28286.99048486120 ),
  { 4330  259 }  (         0.205, 5.21312087405,  20995.39296644940 ),
  { 4330  260 }  (         0.199, 0.44384292491,  16737.57723659660 ),
  { 4330  261 }  (         0.230, 6.06567392849,   6287.00800325450 ),
  { 4330  262 }  (         0.219, 1.29194216300,   5326.78669402080 ),
  { 4330  263 }  (         0.201, 1.74700937253,  22743.40937951640 ),
  { 4330  264 }  (         0.207, 4.45440927276,   6279.48542133960 ),
  { 4330  265 }  (         0.269, 6.05640445030,  64471.99124174489 ),
  { 4330  266 }  (         0.190, 0.99256176518,  29296.61538957860 ),
  { 4330  267 }  (         0.238, 5.42471431221,  39609.65458316560 ),
  { 4330  268 }  (         0.262, 5.26961924198,    522.57741809380 ),
  { 4330  269 }  (         0.210, 4.68618183158,   6254.62666252360 ),
  { 4330  270 }  (         0.197, 2.80624554080,   4933.20844033260 ),
  { 4330  271 }  (         0.252, 4.36220154608,  40879.44050464380 ),
  { 4330  272 }  (         0.261, 1.07241516738,  55022.93574707440 ),
  { 4330  273 }  (         0.189, 3.82966734476,    419.48464387520 ),
  { 4330  274 }  (         0.185, 4.14324541379,   5642.19824260920 ),
  { 4330  275 }  (         0.247, 3.44855612987,   6702.56049386660 ),
  { 4330  276 }  (         0.205, 4.04424043223,    536.80451209540 ),
  { 4330  277 }  (         0.191, 3.14082686083,  16723.35014259500 ),
  { 4330  278 }  (         0.222, 5.16263907319,  23539.70738633280 ),
  { 4330  279 }  (         0.180, 4.56214752149,   6489.26139842860 ),
  { 4330  280 }  (         0.219, 0.80382553358,  16627.37091537720 ),
  { 4330  281 }  (         0.227, 0.60156339452,   5905.70224207560 ),
  { 4330  282 }  (         0.168, 0.88753528161,  16062.18452611680 ),
  { 4330  283 }  (         0.158, 0.92127725775,  23937.85638974100 ),
  { 4330  284 }  (         0.157, 4.69607868164,   6805.65326808520 ),
  { 4330  285 }  (         0.207, 4.88410451334,   6286.66627864320 ),
  { 4330  286 }  (         0.160, 4.95943826846,  10021.83728009940 ),
  { 4330  287 }  (         0.166, 0.97126433565,   3097.88382272579 ),
  { 4330  288 }  (         0.209, 5.75663411805,   3646.35037735440 ),
  { 4330  289 }  (         0.175, 6.12762824412, 239424.39025435288 ),
  { 4330  290 }  (         0.173, 3.13887234973,   6179.98307577280 ),
  { 4330  291 }  (         0.157, 3.62822058179,  18451.07854656599 ),
  { 4330  292 }  (         0.157, 4.67695912235,   6709.67404086740 ),
  { 4330  293 }  (         0.146, 3.09506069735,   4907.30205014560 ),
  { 4330  294 }  (         0.165, 2.27139128760,  10660.68693504240 ),
  { 4330  295 }  (         0.201, 1.67701267433,   2107.03450754240 ),
  { 4330  296 }  (         0.144, 3.96947747592,   6019.99192661860 ),
  { 4330  297 }  (         0.171, 5.91302216729,   6058.73105428950 ),
  { 4330  298 }  (         0.144, 2.13155655120,  26084.02180621620 ),
  { 4330  299 }  (         0.151, 0.67417383554,   2388.89402044920 ),
  { 4330  300 }  (         0.189, 5.07122281033,    263.08392337280 ),
  { 4330  301 }  (         0.146, 5.10373877968,  10770.89325626180 ),
  { 4330  302 }  (         0.187, 1.23915444627,  19402.79695281660 ),
  { 4330  303 }  (         0.174, 0.08407293391,   9380.95967271720 ),
  { 4330  304 }  (         0.137, 1.26247412309,  12566.21901028560 ),
  { 4330  305 }  (         0.137, 3.52826010842,    639.89728631400 ),
  { 4330  306 }  (         0.148, 1.76124372592,   5888.44996493220 ),
  { 4330  307 }  (         0.164, 2.39195095081,   6357.85744855870 ),
  { 4330  308 }  (         0.146, 2.43675816553,   5881.40372823420 ),
  { 4330  309 }  (         0.161, 1.15721259372,  26735.94526221320 ),
  { 4330  310 }  (         0.131, 2.51859277344,   6599.46771964800 ),
  { 4330  311 }  (         0.153, 5.85203687779,   6281.59137728310 ),
  { 4330  312 }  (         0.151, 3.72338532649,  12669.24447420140 ),
  { 4330  313 }  (         0.132, 2.38417741883,   6525.80445396540 ),
  { 4330  314 }  (         0.129, 0.75556744143,   5017.50837136500 ),
  { 4330  315 }  (         0.127, 0.00254936441,  10027.90319572920 ),
  { 4330  316 }  (         0.148, 2.85102145528,   6418.14093002680 ),
  { 4330  317 }  (         0.143, 5.74460279367,  26087.90314157420 ),
  { 4330  318 }  (         0.172, 0.41289962240, 174242.46596404970 ),
  { 4330  319 }  (         0.136, 4.15497742275,   6311.52503745920 ),
  { 4330  320 }  (         0.170, 5.98194913129, 327574.51427678125 ),
  { 4330  321 }  (         0.124, 1.65497607604,  32217.20018108080 ),
  { 4330  322 }  (         0.136, 2.48430783417,  13341.67431130680 ),
  { 4330  323 }  (         0.165, 2.49667924600,  58953.14544329400 ),
  { 4330  324 }  (         0.123, 3.45660563754,   6277.55292568400 ),
  { 4330  325 }  (         0.117, 0.86065134175,   6245.04817735560 ),
  { 4330  326 }  (         0.149, 5.61358280963,   5729.50644714900 ),
  { 4330  327 }  (         0.153, 0.26860029950,    245.83164622940 ),
  { 4330  328 }  (         0.128, 0.71204006588,    103.09277421860 ),
  { 4330  329 }  (         0.159, 2.43166592149, 221995.02880149524 ),
  { 4330  330 }  (         0.130, 2.80707316718,   6016.46880826960 ),
  { 4330  331 }  (         0.137, 1.70657709294,  12566.08438968000 ),
  { 4330  332 }  (         0.111, 1.56305648432,  17782.73207278420 ),
  { 4330  333 }  (         0.113, 3.58302904101,  25685.87280280800 ),
  { 4330  334 }  (         0.109, 3.26403795962,   6819.88036208680 ),
  { 4330  335 }  (         0.122, 0.34120688217,   1162.47470440780 ),
  { 4330  336 }  (         0.119, 5.84644718278,  12721.57209941700 ),
  { 4330  337 }  (         0.144, 2.28899679126,  12489.88562870720 ),
  { 4330  338 }  (         0.137, 5.82029768354,  44809.65020086340 ),
  { 4330  339 }  (         0.107, 2.42818544140,   5547.19933645960 ),
  { 4330  340 }  (         0.134, 1.26539982939,   5331.35744374080 ),
  { 4330  341 }  (         0.103, 5.96518130595,   6321.10352262720 ),
  { 4330  342 }  (         0.109, 0.33808549034,  11300.58422135640 ),
  { 4330  343 }  (         0.129, 5.89187277327,  12029.34718788740 ),
  { 4330  344 }  (         0.122, 5.77325634636,  11919.14086666800 ),
  { 4330  345 }  (         0.107, 6.24998989350,  77690.75950573849 ),
  { 4330  346 }  (         0.107, 1.00535580713,  77736.78343050249 ),
  { 4330  347 }  (         0.143, 0.24122178432,   4214.06901508480 ),
  { 4330  348 }  (         0.143, 0.88529649733,   7576.56007357400 ),
  { 4330  349 }  (         0.107, 2.92124030496,  31415.37924995700 ),
  { 4330  350 }  (         0.099, 5.70862227072,   5540.08578945880 ),
  { 4330  351 }  (         0.110, 0.37528037383,   5863.59120611620 ),
  { 4330  352 }  (         0.104, 4.44107178366,   2118.76386037840 ),
  { 4330  353 }  (         0.098, 5.95877916706,   4061.21921539440 ),
  { 4330  354 }  (         0.113, 1.24206857385,  84672.47584450469 ),
  { 4330  355 }  (         0.124, 2.55619029867,  12539.85338018300 ),
  { 4330  356 }  (         0.110, 3.66952094329, 238004.52415723629 ),
  { 4330  357 }  (         0.112, 4.32512422943,  97238.62754448749 ),
  { 4330  358 }  (         0.097, 3.70151541181,  11720.06886523160 ),
  { 4330  359 }  (         0.120, 1.26895630252,  12043.57428188900 ),
  { 4330  360 }  (         0.094, 2.56461130309,  19004.64794940840 ),
  { 4330  361 }  (         0.117, 3.65425622684,  34520.30930938080 ),
  { 4330  362 }  (         0.098, 0.13589994287,  11080.17157891760 ),
  { 4330  363 }  (         0.097, 5.38330115253,   7834.12107263940 ),
  { 4330  364 }  (         0.097, 2.46722096722,  71980.63357473118 ),
  { 4330  365 }  (         0.095, 5.36958330451,   6288.59877429880 ),
  { 4330  366 }  (         0.111, 5.01961920313,  11823.16163945020 ),
  { 4330  367 }  (         0.090, 2.72299804525,  26880.31981303260 ),
  { 4330  368 }  (         0.099, 0.90164266377,  18635.92845453620 ),
  { 4330  369 }  (         0.126, 4.78722177847, 305281.94307104882 ),
  { 4330  370 }  (         0.093, 0.21240380046,  18139.29450141590 ),
  { 4330  371 }  (         0.124, 5.00979495566, 172146.97134054029 ),
  { 4330  372 }  (         0.099, 5.67090026475,  16522.65971600220 ),
  { 4330  373 }  (         0.092, 2.28180963676,  12491.37010141550 ),
  { 4330  374 }  (         0.090, 4.50544881196,  40077.61957352000 ),
  { 4330  375 }  (         0.100, 2.00639461612,  12323.42309600880 ),
  { 4330  376 }  (         0.095, 5.68801979087,  14919.01785375460 ),
  { 4330  377 }  (         0.087, 1.86043406047,  27707.54249429480 ),
  { 4330  378 }  (         0.105, 3.02903468417,  22345.26037610820 ),
  { 4330  379 }  (         0.087, 5.43970168638,   6272.03014972750 ),
  { 4330  380 }  (         0.089, 1.63389387182,  33326.57873317420 ),
  { 4330  381 }  (         0.082, 5.58298993353,  10241.20229116720 ),
  { 4330  382 }  (         0.094, 5.47749711149,   9924.81042151060 ),
  { 4330  383 }  (         0.082, 4.71988314145,  15141.39079431200 ),
  { 4330  384 }  (         0.097, 5.61458778738,   2787.04302385740 ),
  { 4330  385 }  (         0.096, 3.89073946348,   6379.05507720920 ),
  { 4330  386 }  (         0.081, 3.13038482444,  36147.40987730040 ),
  { 4330  387 }  (         0.110, 4.89978492291,  72140.62866668739 ),
  { 4330  388 }  (         0.097, 5.20764563059,   6303.43116939020 ),
  { 4330  389 }  (         0.082, 5.26342716139,   9814.60410029120 ),
  { 4330  390 }  (         0.109, 2.35555589770,  83286.91426955358 ),
  { 4330  391 }  (         0.097, 2.58492958057,  30666.15495843280 ),
  { 4330  392 }  (         0.093, 1.32651591333,  23020.65308658799 ),
  { 4330  393 }  (         0.078, 3.99588630754,  11293.47067435560 ),
  { 4330  394 }  (         0.090, 0.57771932738,  26482.17080962440 ),
  { 4330  395 }  (         0.106, 3.92012705073,  62883.35513951360 ),
  { 4330  396 }  (         0.098, 2.94397773524,    316.39186965660 ),
  { 4330  397 }  (         0.076, 3.96310417608,  29026.48522950779 ),
  { 4330  398 }  (         0.078, 1.97068529306,  90279.92316810328 ),
  { 4330  399 }  (         0.076, 0.23027966596,  21424.46664430340 ),
  { 4330  400 }  (         0.080, 2.23099742212,    266.60704172180 ),
  { 4330  401 }  (         0.079, 1.46227790922,   8982.81066930900 ),
  { 4330  402 }  (         0.102, 4.92129953565,   5621.84292321040 ),
  { 4330  403 }  (         0.100, 0.39243148321,  24279.10701821359 ),
  { 4330  404 }  (         0.071, 1.52014858474,  33794.54372352860 ),
  { 4330  405 }  (         0.076, 0.22880641443,  57375.80190084620 ),
  { 4330  406 }  (         0.091, 0.96515913904,  48739.85989708300 ),
  { 4330  407 }  (         0.075, 2.77638585157,  12964.30070339100 ),
  { 4330  408 }  (         0.077, 5.18846946344,  11520.99686379520 ),
  { 4330  409 }  (         0.068, 0.50006599129,   4274.51831083240 ),
  { 4330  410 }  (         0.075, 2.07323762803,  15664.03552270859 ),
  { 4330  411 }  (         0.074, 1.01884134928,   6393.28217121080 ),
  { 4330  412 }  (         0.077, 0.46665178780,  16207.88627150200 ),
  { 4330  413 }  (         0.081, 4.10452219483, 161710.61878623239 ),
  { 4330  414 }  (         0.067, 3.83840630887,   6262.72053059260 ),
  { 4330  415 }  (         0.071, 3.91415523291,   7875.67186362420 ),
  { 4330  416 }  (         0.081, 0.91938383237,     74.78159856730 ),
  { 4330  417 }  (         0.083, 4.69916218791,  23006.42599258639 ),
  { 4330  418 }  (         0.063, 2.32556465878,   6279.19451463340 ),
  { 4330  419 }  (         0.065, 5.41938745446,  28628.33622609960 ),
  { 4330  420 }  (         0.065, 3.02336771694,   5959.57043333400 ),
  { 4330  421 }  (         0.064, 3.31033198370,   2636.72547263700 ),
  { 4330  422 }  (         0.064, 0.18375587519,   1066.49547719000 ),
  { 4330  423 }  (         0.080, 5.81239171612,  12341.80690428090 ),
  { 4330  424 }  (         0.066, 2.15105504851,     38.02767263580 ),
  { 4330  425 }  (         0.062, 2.43313614978,  10138.10951694860 ),
  { 4330  426 }  (         0.060, 3.16153906470,   5490.30096152400 ),
  { 4330  427 }  (         0.069, 0.30764736334,   7018.95236352320 ),
  { 4330  428 }  (         0.068, 2.24442548639,  24383.07910844140 ),
  { 4330  429 }  (         0.078, 1.39649386463,   9411.46461508720 ),
  { 4330  430 }  (         0.063, 0.72976362625,   6286.95718534940 ),
  { 4330  431 }  (         0.073, 4.95125917731,   6453.74872061060 ),
  { 4330  432 }  (         0.078, 0.32736023459,   6528.90749622080 ),
  { 4330  433 }  (         0.059, 4.95362151577,  35707.71008290740 ),
  { 4330  434 }  (         0.070, 2.37962727525,  15508.61512327440 ),
  { 4330  435 }  (         0.073, 1.35229143111,   5327.47610838280 ),
  { 4330  436 }  (         0.072, 5.91833527334,  10881.09957748120 ),
  { 4330  437 }  (         0.059, 5.36231868425,  10239.58386601080 ),
  { 4330  438 }  (         0.059, 1.63156134967,  61306.01159706580 ),
  { 4330  439 }  (         0.054, 4.29491690425,  21947.11137270000 ),
  { 4330  440 }  (         0.057, 5.89190132575,  34513.26307268280 ),
  { 4330  441 }  (         0.074, 1.38235845304,   9967.45389998160 ),
  { 4330  442 }  (         0.053, 3.86543309344,  32370.97899156560 ),
  { 4330  443 }  (         0.055, 4.51794544854,  34911.41207609100 ),
  { 4330  444 }  (         0.063, 5.41479412056,  11502.83761653050 ),
  { 4330  445 }  (         0.063, 2.34416220742,  11510.70192305670 ),
  { 4330  446 }  (         0.068, 0.77493931112,  29864.33402730900 ),
  { 4330  447 }  (         0.060, 5.57024703495,   5756.90800324580 ),
  { 4330  448 }  (         0.072, 2.80863088166,  10866.87248347960 ),
  { 4330  449 }  (         0.061, 2.69736991384,  82576.98122099529 ),
  { 4330  450 }  (         0.063, 5.32068807257,   3116.65941225980 ),
  { 4330  451 }  (         0.052, 1.02278758099,   6272.43918464160 ),
  { 4330  452 }  (         0.069, 5.00698550308,  25287.72379939980 ),
  { 4330  453 }  (         0.066, 6.12047940728,  12074.48840752400 ),
  { 4330  454 }  (         0.051, 2.59519527563,  11396.56344857420 ),
  { 4330  455 }  (         0.056, 2.57995973521,  17892.93839400359 ),
  { 4330  456 }  (         0.059, 0.44167237620, 250570.67585721909 ),
  { 4330  457 }  (         0.059, 3.84070143543,   5483.25472482600 ),
  { 4330  458 }  (         0.049, 0.54704693048,  22594.05489571199 ),
  { 4330  459 }  (         0.065, 2.38423614501,  52670.06959330260 ),
  { 4330  460 }  (         0.069, 5.34363738671,  66813.56483573320 ),
  { 4330  461 }  (         0.057, 5.42770501007, 310145.15282392364 ),
  { 4330  462 }  (         0.053, 1.17760296075,    149.56319713460 ),
  { 4330  463 }  (         0.061, 4.02090887211,  34596.36465465240 ),
  { 4330  464 }  (         0.049, 4.18361320516,  18606.49894600020 ),
  { 4330  465 }  (         0.055, 0.83886167974,  20452.86941222180 ),
  { 4330  466 }  (         0.050, 1.46327331958,  37455.72649597440 ),
  { 4330  467 }  (         0.048, 4.53854727167,  29822.78323632420 ),
  { 4330  468 }  (         0.058, 3.34847975377,  33990.61834428620 ),
  { 4330  469 }  (         0.065, 1.45522693982,  76251.32777062019 ),
  { 4330  470 }  (         0.056, 2.35650663692,  37724.75341974820 ),
  { 4330  471 }  (         0.052, 2.61551081496,   5999.21653112620 ),
  { 4330  472 }  (         0.053, 0.17334326094,  77717.29458646949 ),
  { 4330  473 }  (         0.053, 0.79879700631,  77710.24834977149 ),
  { 4330  474 }  (         0.047, 0.43240779709,    735.87651353180 ),
  { 4330  475 }  (         0.053, 4.58763261686,  11616.97609101300 ),
  { 4330  476 }  (         0.048, 6.20230111054,   4171.42553661380 ),
  { 4330  477 }  (         0.052, 1.09723616404,    640.87760738220 ),
  { 4330  478 }  (         0.057, 3.42008310383,  50317.20343953080 ),
  { 4330  479 }  (         0.053, 1.01528448581, 149144.46708624958 ),
  { 4330  480 }  (         0.047, 3.00924906195,  52175.80628314840 ),
  { 4330  481 }  (         0.052, 2.03254070404,   6293.71251534120 ),
  { 4330  482 }  (         0.048, 0.12356889734,  13362.44970679920 ),
  { 4330  483 }  (         0.045, 3.37963782356,  10763.77970926100 ),
  { 4330  484 }  (         0.047, 5.50981287869,  12779.45079542080 ),
  { 4330  485 }  (         0.062, 5.45209070099,    949.17560896980 ),
  { 4330  486 }  (         0.061, 2.93237974631,   5791.41255753260 ),
  { 4330  487 }  (         0.044, 2.87440620802,   8584.66166590080 ),
  { 4330  488 }  (         0.046, 4.03141796560,  10667.80048204320 ),
  { 4330  489 }  (         0.047, 3.89902931422,   3903.91137641980 ),
  { 4330  490 }  (         0.046, 2.75700467329,   6993.00889854970 ),
  { 4330  491 }  (         0.045, 1.93386293300,    206.18554843720 ),
  { 4330  492 }  (         0.047, 2.57670800912,  11492.54267579200 ),
  { 4330  493 }  (         0.044, 3.62570223167,  63658.87775083760 ),
  { 4330  494 }  (         0.051, 0.84536826273,  12345.73905754400 ),
  { 4330  495 }  (         0.043, 0.01524970172,  37853.87549938260 ),
  { 4330  496 }  (         0.041, 3.27146326065,   8858.31494432060 ),
  { 4330  497 }  (         0.045, 3.03765521215,  65236.22129328540 ),
  { 4330  498 }  (         0.047, 1.44447548944,  21393.54196985760 ),
  { 4330  499 }  (         0.058, 5.45843180927,   1975.49254585600 ),
  { 4330  500 }  (         0.050, 2.13285524146,  12573.26524698360 ),
  { 4330  501 }  (         0.041, 1.32190847146,   2547.83753823240 ),
  { 4330  502 }  (         0.047, 3.67579608544,  28313.28880466100 ),
  { 4330  503 }  (         0.041, 2.24013475126,   8273.82086703240 ),
  { 4330  504 }  (         0.047, 6.21438985953,  10991.30589870060 ),
  { 4330  505 }  (         0.042, 3.01631817350,    853.19638175200 ),
  { 4330  506 }  (         0.056, 1.09773690181,  77376.20102240759 ),
  { 4330  507 }  (         0.040, 2.35698541041,   2699.73481931760 ),
  { 4330  508 }  (         0.043, 5.28030898459,  17796.95916678580 ),
  { 4330  509 }  (         0.054, 2.59175932091,  22910.44676536859 ),
  { 4330  510 }  (         0.054, 0.88027764102,  71960.38658322369 ),
  { 4330  511 }  (         0.055, 0.07988899477,  83467.15635301729 ),
  { 4330  512 }  (         0.039, 1.12867321442,   9910.58332750900 ),
  { 4330  513 }  (         0.040, 1.35670430524,  27177.85152920020 ),
  { 4330  514 }  (         0.039, 4.39624220245,   5618.31980486140 ),
  { 4330  515 }  (         0.042, 4.78798367468,   7856.89627409019 ),
  { 4330  516 }  (         0.047, 2.75482175292,  18202.21671665939 ),
  { 4330  517 }  (         0.039, 1.97008298629,  24491.42579258340 ),
  { 4330  518 }  (         0.042, 4.04346599946,   7863.94251078820 ),
  { 4330  519 }  (         0.038, 0.49178679251,  38650.17350619900 ),
  { 4330  520 }  (         0.036, 4.86047906533,   4157.19844261220 ),
  { 4330  521 }  (         0.043, 5.64354880978,   1062.90504853820 ),
  { 4330  522 }  (         0.036, 3.98066313627,  12565.17137891460 ),
  { 4330  523 }  (         0.042, 2.30753932657,   6549.68289171320 ),
  { 4330  524 }  (         0.040, 5.39694918320,   9498.21223063460 ),
  { 4330  525 }  (         0.040, 3.30603243754,  23536.11695768099 ),
  { 4330  526 }  (         0.050, 6.15760345261,  78051.34191383338 )
  (*$endif *)
      );
  (*@\\\0000000601*)
  (*@/// vsop87_ear_r1:array[0..291,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_r1:array[0..  9,0..2] of extended = (
  (*$else *)
  vsop87_ear_r1:array[0..291,0..2] of extended = (
  (*$endif *)
  { 4331    1 }  (    103018.608, 1.10748969588,   6283.07584999140 ),
  { 4331    2 }  (      1721.238, 1.06442301418,  12566.15169998280 ),
  { 4331    3 }  (       702.215, 3.14159265359,      0.00000000000 ),
  { 4331    4 }  (        32.346, 1.02169059149,  18849.22754997420 ),
  { 4331    5 }  (        30.799, 2.84353804832,   5507.55323866740 ),
  { 4331    6 }  (        24.971, 1.31906709482,   5223.69391980220 ),
  { 4331    7 }  (        18.485, 1.42429748614,   1577.34354244780 ),
  { 4331    8 }  (        10.078, 5.91378194648,  10977.07880469900 ),
  { 4331    9 }  (         8.634, 0.27146150602,   5486.77784317500 ),
  { 4331   10 }  (         8.654, 1.42046854427,   6275.96230299060 )
  (*$ifndef meeus *)                                                 ,
  { 4331   11 }  (         5.069, 1.68613426734,   5088.62883976680 ),
  { 4331   12 }  (         4.985, 6.01401770704,   6286.59896834040 ),
  { 4331   13 }  (         4.669, 5.98724494073,    529.69096509460 ),
  { 4331   14 }  (         4.395, 0.51800238019,   4694.00295470760 ),
  { 4331   15 }  (         3.872, 4.74969833437,   2544.31441988340 ),
  { 4331   16 }  (         3.750, 5.07097685568,    796.29800681640 ),
  { 4331   17 }  (         4.100, 1.08424786092,   9437.76293488700 ),
  { 4331   18 }  (         3.518, 0.02290216272,  83996.84731811189 ),
  { 4331   19 }  (         3.436, 0.94937019624,  71430.69561812909 ),
  { 4331   20 }  (         3.221, 6.15628775313,   2146.16541647520 ),
  { 4331   21 }  (         3.414, 5.41218322538,    775.52261132400 ),
  { 4331   22 }  (         2.863, 5.48432847146,  10447.38783960440 ),
  { 4331   23 }  (         2.520, 0.24276941146,    398.14900340820 ),
  { 4331   24 }  (         2.201, 4.95216196651,   6812.76681508600 ),
  { 4331   25 }  (         2.186, 0.41991743105,   8031.09226305840 ),
  { 4331   26 }  (         2.838, 3.42034351366,   2352.86615377180 ),
  { 4331   27 }  (         2.554, 6.13241878525,   6438.49624942560 ),
  { 4331   28 }  (         1.932, 5.31374608366,   8429.24126646660 ),
  { 4331   29 }  (         2.429, 3.09164528262,   4690.47983635860 ),
  { 4331   30 }  (         1.730, 1.53686208550,   4705.73230754360 ),
  { 4331   31 }  (         2.250, 3.68863633842,   7084.89678111520 ),
  { 4331   32 }  (         2.093, 1.28191783032,   1748.01641306700 ),
  { 4331   33 }  (         1.441, 0.81656250862,  14143.49524243060 ),
  { 4331   34 }  (         1.483, 3.22225357771,   7234.79425624200 ),
  { 4331   35 }  (         1.754, 3.22883705112,   6279.55273164240 ),
  { 4331   36 }  (         1.583, 4.09702349428,  11499.65622279280 ),
  { 4331   37 }  (         1.575, 5.53890170575,   3154.68708489560 ),
  { 4331   38 }  (         1.847, 1.82040335363,   7632.94325965020 ),
  { 4331   39 }  (         1.504, 3.63293385726,  11513.88331679440 ),
  { 4331   40 }  (         1.337, 4.64440864339,   6836.64525283380 ),
  { 4331   41 }  (         1.275, 2.69341415363,   1349.86740965880 ),
  { 4331   42 }  (         1.352, 6.15101580257,   5746.27133789600 ),
  { 4331   43 }  (         1.125, 3.35673439497,  17789.84561978500 ),
  { 4331   44 }  (         1.470, 3.65282991755,   1194.44701022460 ),
  { 4331   45 }  (         1.177, 2.57676109092,  13367.97263110660 ),
  { 4331   46 }  (         1.101, 4.49748696552,   4292.33083295040 ),
  { 4331   47 }  (         1.234, 5.65036509521,   5760.49843189760 ),
  { 4331   48 }  (         0.984, 0.65517395136,   5856.47765911540 ),
  { 4331   49 }  (         0.928, 2.32420318751,  10213.28554621100 ),
  { 4331   50 }  (         1.077, 5.82812169132,  12036.46073488820 ),
  { 4331   51 }  (         0.916, 0.76613009583,  16730.46368959580 ),
  { 4331   52 }  (         0.877, 1.50137505051,  11926.25441366880 ),
  { 4331   53 }  (         1.023, 5.62076589825,   6256.77753019160 ),
  { 4331   54 }  (         0.851, 0.65709335533,    155.42039943420 ),
  { 4331   55 }  (         0.802, 4.10519132088,    951.71840625060 ),
  { 4331   56 }  (         0.857, 1.41661697538,   5753.38488489680 ),
  { 4331   57 }  (         0.994, 1.14418521187,   1059.38193018920 ),
  { 4331   58 }  (         0.813, 1.63948433322,   6681.22485339960 ),
  { 4331   59 }  (         0.662, 4.55200452260,   5216.58037280140 ),
  { 4331   60 }  (         0.644, 4.19478168733,   6040.34724601740 ),
  { 4331   61 }  (         0.626, 1.50767713598,   5643.17856367740 ),
  { 4331   62 }  (         0.590, 6.18277145205,   4164.31198961300 ),
  { 4331   63 }  (         0.635, 0.52413263542,   6290.18939699220 ),
  { 4331   64 }  (         0.650, 0.97935690350,  25132.30339996560 ),
  { 4331   65 }  (         0.568, 2.30125315873,  10973.55568635000 ),
  { 4331   66 }  (         0.547, 5.27256412213,   3340.61242669980 ),
  { 4331   67 }  (         0.547, 2.20144422886,   1592.59601363280 ),
  { 4331   68 }  (         0.526, 0.92464258226,  11371.70468975820 ),
  { 4331   69 }  (         0.490, 5.90951388655,   3894.18182954220 ),
  { 4331   70 }  (         0.478, 1.66857963179,  12168.00269657460 ),
  { 4331   71 }  (         0.516, 3.59803483887,  10969.96525769820 ),
  { 4331   72 }  (         0.518, 3.97914412373,  17298.18232732620 ),
  { 4331   73 }  (         0.534, 5.03740926442,   9917.69687450980 ),
  { 4331   74 }  (         0.487, 2.50545369269,   6127.65545055720 ),
  { 4331   75 }  (         0.416, 4.04828175503,  10984.19235169980 ),
  { 4331   76 }  (         0.538, 5.54081539805,    553.56940284240 ),
  { 4331   77 }  (         0.402, 2.16544019233,   7860.41939243920 ),
  { 4331   78 }  (         0.553, 2.32177369366,  11506.76976979360 ),
  { 4331   79 }  (         0.367, 3.39152532250,   6496.37494542940 ),
  { 4331   80 }  (         0.360, 5.34379853282,   7079.37385680780 ),
  { 4331   81 }  (         0.337, 3.61563704045,  11790.62908865880 ),
  { 4331   82 }  (         0.456, 0.30754294809,    801.82093112380 ),
  { 4331   83 }  (         0.417, 3.70009308674,  10575.40668294180 ),
  { 4331   84 }  (         0.381, 5.82033971802,   7058.59846131540 ),
  { 4331   85 }  (         0.321, 0.31988767355,  16200.77272450120 ),
  { 4331   86 }  (         0.364, 1.08414306177,   6309.37416979120 ),
  { 4331   87 }  (         0.294, 4.54798604957,  11856.21865142450 ),
  { 4331   88 }  (         0.290, 1.26473978562,   8635.94200376320 ),
  { 4331   89 }  (         0.399, 4.16998866302,     26.29831979980 ),
  { 4331   90 }  (         0.262, 5.08316906342,  10177.25767953360 ),
  { 4331   91 }  (         0.243, 2.25746091190,  11712.95531823080 ),
  { 4331   92 }  (         0.237, 1.05070575346,    242.72860397400 ),
  { 4331   93 }  (         0.275, 3.45319481756,   5884.92684658320 ),
  { 4331   94 }  (         0.255, 5.38496831087,  21228.39202354580 ),
  { 4331   95 }  (         0.307, 4.24313526604,   3738.76143010800 ),
  { 4331   96 }  (         0.216, 3.46037894728,    213.29909543800 ),
  { 4331   97 }  (         0.196, 0.69029243914,   1990.74501704100 ),
  { 4331   98 }  (         0.198, 5.16301829964,  12352.85260454480 ),
  { 4331   99 }  (         0.214, 3.91876200279,  13916.01910964160 ),
  { 4331  100 }  (         0.212, 4.00861198517,   5230.80746680300 ),
  { 4331  101 }  (         0.184, 5.59805976614,   6283.14316029419 ),
  { 4331  102 }  (         0.184, 2.85275392124,   7238.67559160000 ),
  { 4331  103 }  (         0.179, 2.54259058334,  14314.16811304980 ),
  { 4331  104 }  (         0.225, 1.64458698399,   4732.03062734340 ),
  { 4331  105 }  (         0.236, 5.58826125715,   6069.77675455340 ),
  { 4331  106 }  (         0.187, 2.72805985443,   6062.66320755260 ),
  { 4331  107 }  (         0.184, 6.04216273598,   6283.00853968860 ),
  { 4331  108 }  (         0.230, 3.62591335086,   6284.05617105960 ),
  { 4331  109 }  (         0.163, 2.19117396803,  18073.70493865020 ),
  { 4331  110 }  (         0.172, 0.97612950740,   3930.20969621960 ),
  { 4331  111 }  (         0.215, 1.04672844028,   3496.03282613400 ),
  { 4331  112 }  (         0.169, 4.75084479006,  17267.26820169119 ),
  { 4331  113 }  (         0.152, 0.19390712179,   9779.10867612540 ),
  { 4331  114 }  (         0.182, 5.16288118255,  17253.04110768959 ),
  { 4331  115 }  (         0.149, 0.80944184260,    709.93304855830 ),
  { 4331  116 }  (         0.163, 2.19209570390,   6076.89030155420 ),
  { 4331  117 }  (         0.186, 5.01159497089,  11015.10647733480 ),
  { 4331  118 }  (         0.134, 0.97765485759,  65147.61976813770 ),
  { 4331  119 }  (         0.141, 4.38421981312,   4136.91043351620 ),
  { 4331  120 }  (         0.158, 4.60974280627,   9623.68827669120 ),
  { 4331  121 }  (         0.133, 3.30508592837, 154717.60988768269 ),
  { 4331  122 }  (         0.163, 6.11782626245,      3.52311834900 ),
  { 4331  123 }  (         0.174, 1.58078542187,      7.11354700080 ),
  { 4331  124 }  (         0.141, 0.49976927274,  25158.60171976540 ),
  { 4331  125 }  (         0.124, 6.03440460031,   9225.53927328300 ),
  { 4331  126 }  (         0.150, 5.30166336812,  13517.87010623340 ),
  { 4331  127 }  (         0.127, 1.92389511438,  22483.84857449259 ),
  { 4331  128 }  (         0.121, 2.37813129011, 167283.76158766549 ),
  { 4331  129 }  (         0.120, 3.98423684853,   4686.88940770680 ),
  { 4331  130 }  (         0.117, 5.81072642211,  12569.67481833180 ),
  { 4331  131 }  (         0.122, 5.60973054224,   5642.19824260920 ),
  { 4331  132 }  (         0.157, 3.40236426002,  16496.36139620240 ),
  { 4331  133 }  (         0.129, 2.10705116371,   1589.07289528380 ),
  { 4331  134 }  (         0.116, 0.55839966736,   5849.36411211460 ),
  { 4331  135 }  (         0.123, 1.52961392771,  12559.03815298200 ),
  { 4331  136 }  (         0.111, 0.44848279675,   6172.86952877200 ),
  { 4331  137 }  (         0.123, 5.81645568991,   6282.09552892320 ),
  { 4331  138 }  (         0.150, 4.26278409223,   3128.38876509580 ),
  { 4331  139 }  (         0.106, 2.27437761356,   5429.87946823940 ),
  { 4331  140 }  (         0.104, 4.42743707728,  23543.23050468179 ),
  { 4331  141 }  (         0.121, 0.39459045915,  12132.43996210600 ),
  { 4331  142 }  (         0.104, 2.41842602527,    426.59819087600 ),
  { 4331  143 }  (         0.110, 5.80381480447,  16858.48253293320 ),
  { 4331  144 }  (         0.100, 2.93805577485,   4535.05943692440 ),
  { 4331  145 }  (         0.097, 3.97935904984,   6133.51265285680 ),
  { 4331  146 }  (         0.110, 6.22339014386,  12146.66705610760 ),
  { 4331  147 }  (         0.098, 0.87576563709,   6525.80445396540 ),
  { 4331  148 }  (         0.098, 3.15248421301,  10440.27429260360 ),
  { 4331  149 }  (         0.095, 2.46168411100,   3097.88382272579 ),
  { 4331  150 }  (         0.088, 0.23371480284,  13119.72110282519 ),
  { 4331  151 }  (         0.098, 5.77016493489,   7342.45778018060 ),
  { 4331  152 }  (         0.092, 6.03915555063,  20426.57109242200 ),
  { 4331  153 }  (         0.096, 5.56909292561,   2388.89402044920 ),
  { 4331  154 }  (         0.081, 1.32131147691,   5650.29211067820 ),
  { 4331  155 }  (         0.086, 3.94529200528,  10454.50138660520 ),
  { 4331  156 }  (         0.076, 2.70729716925, 143571.32428481648 ),
  { 4331  157 }  (         0.091, 5.64100034152,   8827.39026987480 ),
  { 4331  158 }  (         0.076, 1.80783856698,  28286.99048486120 ),
  { 4331  159 }  (         0.081, 1.90858992196,  29088.81141598500 ),
  { 4331  160 }  (         0.075, 3.40955892978,   5481.25491886760 ),
  { 4331  161 }  (         0.069, 4.49936170873,  17256.63153634140 ),
  { 4331  162 }  (         0.088, 1.10098454357,  11769.85369316640 ),
  { 4331  163 }  (         0.066, 2.78285801977,    536.80451209540 ),
  { 4331  164 }  (         0.068, 3.88179770758,  17260.15465469040 ),
  { 4331  165 }  (         0.084, 1.59303306354,   9380.95967271720 ),
  { 4331  166 }  (         0.088, 3.88076636762,   7477.52286021600 ),
  { 4331  167 }  (         0.061, 6.17558202197,  11087.28512591840 ),
  { 4331  168 }  (         0.060, 4.34824715818,   6206.80977871580 ),
  { 4331  169 }  (         0.082, 4.59843208943,   9388.00590941520 ),
  { 4331  170 }  (         0.079, 1.63131230601,   4933.20844033260 ),
  { 4331  171 }  (         0.078, 4.20905757484,   5729.50644714900 ),
  { 4331  172 }  (         0.057, 5.48157926651,  18319.53658487960 ),
  { 4331  173 }  (         0.060, 1.01261781084,  12721.57209941700 ),
  { 4331  174 }  (         0.056, 1.63031935692,  15720.83878487840 ),
  { 4331  175 }  (         0.055, 0.24926735018,  15110.46611986620 ),
  { 4331  176 }  (         0.061, 5.93059279661,  12539.85338018300 ),
  { 4331  177 }  (         0.055, 4.84298966314,  13095.84266507740 ),
  { 4331  178 }  (         0.067, 6.11690589247,   8662.24032356300 ),
  { 4331  179 }  (         0.054, 5.73750638571,   3634.62102451840 ),
  { 4331  180 }  (         0.074, 1.05466745829,  16460.33352952499 ),
  { 4331  181 }  (         0.053, 2.29084335688,  16062.18452611680 ),
  { 4331  182 }  (         0.064, 2.13513767927,   7875.67186362420 ),
  { 4331  183 }  (         0.067, 0.07096807518,  14945.31617355440 ),
  { 4331  184 }  (         0.051, 2.31511194429,   6262.72053059260 ),
  { 4331  185 }  (         0.057, 5.77055471237,  12043.57428188900 ),
  { 4331  186 }  (         0.056, 4.41980790431,   4701.11650170840 ),
  { 4331  187 }  (         0.059, 5.87963500073,   5331.35744374080 ),
  { 4331  188 }  (         0.058, 2.30546168628,    955.59974160860 ),
  { 4331  189 }  (         0.049, 1.93839278478,   5333.90024102160 ),
  { 4331  190 }  (         0.048, 2.69973662261,   6709.67404086740 ),
  { 4331  191 }  (         0.064, 1.64379897981,   6262.30045449900 ),
  { 4331  192 }  (         0.046, 3.98449608961,  98068.53671630539 ),
  { 4331  193 }  (         0.050, 3.68875893005,  12323.42309600880 ),
  { 4331  194 }  (         0.045, 3.30068569697,  22003.91463486980 ),
  { 4331  195 }  (         0.047, 1.26317154881,  11919.14086666800 ),
  { 4331  196 }  (         0.045, 0.89150445122,  51868.24866217880 ),
  { 4331  197 }  (         0.043, 1.61526242998,   6277.55292568400 ),
  { 4331  198 }  (         0.043, 5.74295325645,  11403.67699557500 ),
  { 4331  199 }  (         0.044, 3.43070646822,  10021.83728009940 ),
  { 4331  200 }  (         0.056, 0.02481833774,  15671.08175940660 ),
  { 4331  201 }  (         0.055, 3.14274403422,  33019.02111220460 ),
  { 4331  202 }  (         0.045, 3.00877289177,   8982.81066930900 ),
  { 4331  203 }  (         0.046, 0.73303568429,   6303.43116939020 ),
  { 4331  204 }  (         0.049, 1.60455690285,   6303.85124548380 ),
  { 4331  205 }  (         0.045, 0.40210030323,   6805.65326808520 ),
  { 4331  206 }  (         0.053, 0.94869680175,  10988.80815753500 ),
  { 4331  207 }  (         0.041, 1.61122384329,   6819.88036208680 ),
  { 4331  208 }  (         0.055, 0.89439119424,  11933.36796066960 ),
  { 4331  209 }  (         0.045, 3.88495384656,  60530.48898574180 ),
  { 4331  210 }  (         0.040, 4.75740908001,  38526.57435087200 ),
  { 4331  211 }  (         0.040, 1.49921251887,  18451.07854656599 ),
  { 4331  212 }  (         0.040, 3.77498297228,  26087.90314157420 ),
  { 4331  213 }  (         0.051, 1.70258603562,   1551.04522264800 ),
  { 4331  214 }  (         0.039, 2.97100699926,   2118.76386037840 ),
  { 4331  215 }  (         0.053, 5.19854123078,  77713.77146812050 ),
  { 4331  216 }  (         0.047, 4.26356628717,  21424.46664430340 ),
  { 4331  217 }  (         0.037, 0.62902722802,  24356.78078864160 ),
  { 4331  218 }  (         0.036, 0.11087914947,  10344.29506538580 ),
  { 4331  219 }  (         0.036, 0.77037556319,  12029.34718788740 ),
  { 4331  220 }  (         0.035, 3.30933994515,  24072.92146977640 ),
  { 4331  221 }  (         0.035, 5.93650887012,  31570.79964939120 ),
  { 4331  222 }  (         0.036, 2.15108874765,  30774.50164257480 ),
  { 4331  223 }  (         0.036, 1.75078825382,  16207.88627150200 ),
  { 4331  224 }  (         0.033, 5.06264177921, 226858.23855437008 ),
  { 4331  225 }  (         0.034, 6.16891378800,  24491.42579258340 ),
  { 4331  226 }  (         0.035, 3.19120695549,  32217.20018108080 ),
  { 4331  227 }  (         0.034, 2.31528650443,  55798.45835839840 ),
  { 4331  228 }  (         0.032, 4.21446357042,  15664.03552270859 ),
  { 4331  229 }  (         0.039, 1.24979117796,   6418.14093002680 ),
  { 4331  230 }  (         0.037, 4.11943655770,   2787.04302385740 ),
  { 4331  231 }  (         0.032, 1.62887710890,    639.89728631400 ),
  { 4331  232 }  (         0.038, 5.89832942685,    640.87760738220 ),
  { 4331  233 }  (         0.032, 1.72442327688,  27433.88921587499 ),
  { 4331  234 }  (         0.031, 2.78828943753,  12139.55350910680 ),
  { 4331  235 }  (         0.035, 4.44608896525,  18202.21671665939 ),
  { 4331  236 }  (         0.034, 3.96287980676,  18216.44381066100 ),
  { 4331  237 }  (         0.033, 4.73611335874,  16723.35014259500 ),
  { 4331  238 }  (         0.034, 1.43910280005,  49515.38250840700 ),
  { 4331  239 }  (         0.031, 0.23302920161,  23581.25817731760 ),
  { 4331  240 }  (         0.029, 2.02633840220,  11609.86254401220 ),
  { 4331  241 }  (         0.030, 2.54923230240,   9924.81042151060 ),
  { 4331  242 }  (         0.032, 4.91793198558,  11300.58422135640 ),
  { 4331  243 }  (         0.028, 0.26187189577,  13521.75144159140 ),
  { 4331  244 }  (         0.028, 3.84568936822,   2699.73481931760 ),
  { 4331  245 }  (         0.029, 1.83149729794,  29822.78323632420 ),
  { 4331  246 }  (         0.033, 4.60320094415,  19004.64794940840 ),
  { 4331  247 }  (         0.027, 4.46183450287,   6702.56049386660 ),
  { 4331  248 }  (         0.030, 4.46494072240,  36147.40987730040 ),
  { 4331  249 }  (         0.027, 0.03211931363,   6279.78949257360 ),
  { 4331  250 }  (         0.026, 5.46497324333,   6245.04817735560 ),
  { 4331  251 }  (         0.035, 4.52695674113,  36949.23080842420 ),
  { 4331  252 }  (         0.027, 3.52528177609,  10770.89325626180 ),
  { 4331  253 }  (         0.026, 1.48499438453,  11080.17157891760 ),
  { 4331  254 }  (         0.035, 2.82154380962,  19402.79695281660 ),
  { 4331  255 }  (         0.025, 2.46339998836,   6279.48542133960 ),
  { 4331  256 }  (         0.026, 4.97688894643,  16737.57723659660 ),
  { 4331  257 }  (         0.026, 2.36136541526,  17996.03116822220 ),
  { 4331  258 }  (         0.029, 4.15148654061,  45892.73043315699 ),
  { 4331  259 }  (         0.026, 4.50714272714,  17796.95916678580 ),
  { 4331  260 }  (         0.027, 4.72625223674,   1066.49547719000 ),
  { 4331  261 }  (         0.025, 2.89309528854,   6286.66627864320 ),
  { 4331  262 }  (         0.027, 0.37462444357,  12964.30070339100 ),
  { 4331  263 }  (         0.029, 4.94860010533,   5863.59120611620 ),
  { 4331  264 }  (         0.031, 3.93096113577,  29864.33402730900 ),
  { 4331  265 }  (         0.024, 6.14987193584,  18606.49894600020 ),
  { 4331  266 }  (         0.024, 3.74225964547,  29026.48522950779 ),
  { 4331  267 }  (         0.025, 5.70460621565,  27707.54249429480 ),
  { 4331  268 }  (         0.025, 5.33928840652,  15141.39079431200 ),
  { 4331  269 }  (         0.027, 3.02320897140,   6286.36220740920 ),
  { 4331  270 }  (         0.023, 0.28364955406,   5327.47610838280 ),
  { 4331  271 }  (         0.026, 1.34240461687,  18875.52586977400 ),
  { 4331  272 }  (         0.024, 1.33998410121,  19800.94595622480 ),
  { 4331  273 }  (         0.025, 6.00172494004,   6489.26139842860 ),
  { 4331  274 }  (         0.022, 1.81777974484,   6288.59877429880 ),
  { 4331  275 }  (         0.022, 3.58603606640,   6915.85958930460 ),
  { 4331  276 }  (         0.029, 2.09564449439,  15265.88651930040 ),
  { 4331  277 }  (         0.022, 1.02173599251,  11925.27409260060 ),
  { 4331  278 }  (         0.022, 4.74660932338,  28230.18722269139 ),
  { 4331  279 }  (         0.021, 2.30688751432,   5999.21653112620 ),
  { 4331  280 }  (         0.021, 3.22654944430,  25934.12433108940 ),
  { 4331  281 }  (         0.021, 3.04956726238,   6566.93516885660 ),
  { 4331  282 }  (         0.027, 5.35653084499,  33794.54372352860 ),
  { 4331  283 }  (         0.028, 3.91168324815,  18208.34994259200 ),
  { 4331  284 }  (         0.020, 1.52296293311,    135.06508003540 ),
  { 4331  285 }  (         0.022, 4.66462839521,  13362.44970679920 ),
  { 4331  286 }  (         0.019, 1.78121167862, 156137.47598479928 ),
  { 4331  287 }  (         0.019, 2.99969102221,  19651.04848109800 ),
  { 4331  288 }  (         0.019, 2.86664273362,  18422.62935909819 ),
  { 4331  289 }  (         0.025, 0.94995632141,  31415.37924995700 ),
  { 4331  290 }  (         0.019, 4.71432851499,  77690.75950573849 ),
  { 4331  291 }  (         0.019, 2.54227398241,  77736.78343050249 ),
  { 4331  292 }  (         0.020, 5.91915117116,  48739.85989708300 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_ear_r2:array[0..138,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_r2:array[0..  5,0..2] of extended = (
  (*$else *)
  vsop87_ear_r2:array[0..138,0..2] of extended = (
  (*$endif *)
  { 4332    1 }  (      4359.385, 5.78455133738,   6283.07584999140 ),
  { 4332    2 }  (       123.633, 5.57934722157,  12566.15169998280 ),
  { 4332    3 }  (        12.341, 3.14159265359,      0.00000000000 ),
  { 4332    4 }  (         8.792, 3.62777733395,  77713.77146812050 ),
  { 4332    5 }  (         5.689, 1.86958905084,   5573.14280143310 ),
  { 4332    6 }  (         3.301, 5.47027913302,  18849.22754997420 )
  (*$ifndef meeus *)                                                 ,
  { 4332    7 }  (         1.471, 4.48028885617,   5507.55323866740 ),
  { 4332    8 }  (         1.013, 2.81456417694,   5223.69391980220 ),
  { 4332    9 }  (         0.854, 3.10878241236,   1577.34354244780 ),
  { 4332   10 }  (         1.102, 2.84173992403, 161000.68573767410 ),
  { 4332   11 }  (         0.648, 5.47349498544,    775.52261132400 ),
  { 4332   12 }  (         0.609, 1.37969434104,   6438.49624942560 ),
  { 4332   13 }  (         0.499, 4.41649242250,   6286.59896834040 ),
  { 4332   14 }  (         0.417, 0.90242451175,  10977.07880469900 ),
  { 4332   15 }  (         0.402, 3.20376585290,   5088.62883976680 ),
  { 4332   16 }  (         0.351, 1.81079227770,   5486.77784317500 ),
  { 4332   17 }  (         0.467, 3.65753702738,   7084.89678111520 ),
  { 4332   18 }  (         0.458, 5.38585314743, 149854.40013480789 ),
  { 4332   19 }  (         0.304, 3.51701098693,    796.29800681640 ),
  { 4332   20 }  (         0.266, 6.17413982699,   6836.64525283380 ),
  { 4332   21 }  (         0.279, 1.84120501086,   4694.00295470760 ),
  { 4332   22 }  (         0.260, 1.41629543251,   2146.16541647520 ),
  { 4332   23 }  (         0.266, 3.13832905677,  71430.69561812909 ),
  { 4332   24 }  (         0.321, 5.35313367048,   3154.68708489560 ),
  { 4332   25 }  (         0.238, 2.17720020018,    155.42039943420 ),
  { 4332   26 }  (         0.293, 4.61501268144,   4690.47983635860 ),
  { 4332   27 }  (         0.229, 4.75969588070,   7234.79425624200 ),
  { 4332   28 }  (         0.211, 0.21868065485,   4705.73230754360 ),
  { 4332   29 }  (         0.201, 4.21905743357,   1349.86740965880 ),
  { 4332   30 }  (         0.195, 4.57808285364,    529.69096509460 ),
  { 4332   31 }  (         0.253, 2.81496293039,   1748.01641306700 ),
  { 4332   32 }  (         0.182, 5.70454011389,   6040.34724601740 ),
  { 4332   33 }  (         0.179, 6.02897097053,   4292.33083295040 ),
  { 4332   34 }  (         0.186, 1.58690991244,   6309.37416979120 ),
  { 4332   35 }  (         0.170, 2.90220009715,   9437.76293488700 ),
  { 4332   36 }  (         0.166, 1.99984925026,   8031.09226305840 ),
  { 4332   37 }  (         0.158, 0.04783713552,   2544.31441988340 ),
  { 4332   38 }  (         0.197, 2.01083639502,   1194.44701022460 ),
  { 4332   39 }  (         0.165, 5.78372596778,  83996.84731811189 ),
  { 4332   40 }  (         0.214, 3.38285934319,   7632.94325965020 ),
  { 4332   41 }  (         0.140, 0.36401486094,  10447.38783960440 ),
  { 4332   42 }  (         0.151, 0.95153163031,   6127.65545055720 ),
  { 4332   43 }  (         0.136, 1.48426306582,   2352.86615377180 ),
  { 4332   44 }  (         0.127, 5.48475435134,    951.71840625060 ),
  { 4332   45 }  (         0.126, 5.26866506592,   6279.55273164240 ),
  { 4332   46 }  (         0.125, 3.75754889288,   6812.76681508600 ),
  { 4332   47 }  (         0.101, 4.95015746147,    398.14900340820 ),
  { 4332   48 }  (         0.102, 0.68468295277,   1592.59601363280 ),
  { 4332   49 }  (         0.100, 1.14568935785,   3894.18182954220 ),
  { 4332   50 }  (         0.129, 0.76540016965,    553.56940284240 ),
  { 4332   51 }  (         0.109, 5.41063597567,   6256.77753019160 ),
  { 4332   52 }  (         0.075, 5.84804322893,    242.72860397400 ),
  { 4332   53 }  (         0.095, 1.94452244083,  11856.21865142450 ),
  { 4332   54 }  (         0.077, 0.69373708195,   8429.24126646660 ),
  { 4332   55 }  (         0.100, 5.19725292131, 244287.60000722769 ),
  { 4332   56 }  (         0.080, 6.18440483705,   1059.38193018920 ),
  { 4332   57 }  (         0.069, 5.25699888595,  14143.49524243060 ),
  { 4332   58 }  (         0.085, 5.39484725499,  25132.30339996560 ),
  { 4332   59 }  (         0.066, 0.51779993906,    801.82093112380 ),
  { 4332   60 }  (         0.055, 5.16878202461,   7058.59846131540 ),
  { 4332   61 }  (         0.051, 3.88759155247,  12036.46073488820 ),
  { 4332   62 }  (         0.050, 5.57636570536,   6290.18939699220 ),
  { 4332   63 }  (         0.061, 2.24359003264,   8635.94200376320 ),
  { 4332   64 }  (         0.050, 5.54441900966,   1990.74501704100 ),
  { 4332   65 }  (         0.056, 4.00301078040,  13367.97263110660 ),
  { 4332   66 }  (         0.052, 4.13138898038,   7860.41939243920 ),
  { 4332   67 }  (         0.052, 3.90943054011,     26.29831979980 ),
  { 4332   68 }  (         0.041, 3.57128482780,   7079.37385680780 ),
  { 4332   69 }  (         0.056, 2.76959005761,  90955.55169449610 ),
  { 4332   70 }  (         0.042, 1.91461189199,   7477.52286021600 ),
  { 4332   71 }  (         0.042, 0.42728171713,  10213.28554621100 ),
  { 4332   72 }  (         0.042, 1.09413724455,    709.93304855830 ),
  { 4332   73 }  (         0.039, 3.93298068961,  10973.55568635000 ),
  { 4332   74 }  (         0.038, 6.17935925345,   9917.69687450980 ),
  { 4332   75 }  (         0.049, 0.83021145241,  11506.76976979360 ),
  { 4332   76 }  (         0.053, 1.45828359397, 233141.31440436149 ),
  { 4332   77 }  (         0.047, 6.21568666789,   6681.22485339960 ),
  { 4332   78 }  (         0.037, 0.36359309980,  10177.25767953360 ),
  { 4332   79 }  (         0.035, 3.33024911524,   5643.17856367740 ),
  { 4332   80 }  (         0.034, 5.63446915337,   6525.80445396540 ),
  { 4332   81 }  (         0.035, 5.36033855038,  25158.60171976540 ),
  { 4332   82 }  (         0.034, 5.36319798321,   4933.20844033260 ),
  { 4332   83 }  (         0.033, 4.24722336872,  12569.67481833180 ),
  { 4332   84 }  (         0.043, 5.26370903404,  10575.40668294180 ),
  { 4332   85 }  (         0.042, 5.08837645072,  11015.10647733480 ),
  { 4332   86 }  (         0.040, 1.98334703186,   6284.05617105960 ),
  { 4332   87 }  (         0.042, 4.22496037505,  88860.05707098669 ),
  { 4332   88 }  (         0.029, 3.19088628170,  11926.25441366880 ),
  { 4332   89 }  (         0.029, 0.15217616684,  12168.00269657460 ),
  { 4332   90 }  (         0.030, 1.61904744136,   9779.10867612540 ),
  { 4332   91 }  (         0.027, 0.76388991416,   1589.07289528380 ),
  { 4332   92 }  (         0.036, 2.74712003443,   3738.76143010800 ),
  { 4332   93 }  (         0.033, 3.08807829566,   3930.20969621960 ),
  { 4332   94 }  (         0.031, 5.34906619513, 143571.32428481648 ),
  { 4332   95 }  (         0.025, 0.10240267494,  22483.84857449259 ),
  { 4332   96 }  (         0.030, 3.47110495524,  14945.31617355440 ),
  { 4332   97 }  (         0.024, 1.10425016019,   4535.05943692440 ),
  { 4332   98 }  (         0.024, 1.58037259780,   6496.37494542940 ),
  { 4332   99 }  (         0.023, 3.87710321433,   6275.96230299060 ),
  { 4332  100 }  (         0.025, 3.94529778970,   3128.38876509580 ),
  { 4332  101 }  (         0.023, 3.44685609601,   4136.91043351620 ),
  { 4332  102 }  (         0.023, 3.83156029849,   5753.38488489680 ),
  { 4332  103 }  (         0.022, 1.86956128067,  16730.46368959580 ),
  { 4332  104 }  (         0.025, 2.42188933855,   5729.50644714900 ),
  { 4332  105 }  (         0.020, 1.78208352927,  17789.84561978500 ),
  { 4332  106 }  (         0.021, 4.30363087400,  16858.48253293320 ),
  { 4332  107 }  (         0.021, 0.49258939822,  29088.81141598500 ),
  { 4332  108 }  (         0.025, 1.33030250444,   6282.09552892320 ),
  { 4332  109 }  (         0.027, 2.54785812264,   3496.03282613400 ),
  { 4332  110 }  (         0.022, 1.11232521950,  12721.57209941700 ),
  { 4332  111 }  (         0.021, 5.97759081637,      7.11354700080 ),
  { 4332  112 }  (         0.019, 0.80292033311,  16062.18452611680 ),
  { 4332  113 }  (         0.023, 4.12454848769,   2388.89402044920 ),
  { 4332  114 }  (         0.022, 4.92663152168,  18875.52586977400 ),
  { 4332  115 }  (         0.023, 5.68902059771,  16460.33352952499 ),
  { 4332  116 }  (         0.023, 4.97346265647,  17260.15465469040 ),
  { 4332  117 }  (         0.023, 3.03021283729,  66567.48586525429 ),
  { 4332  118 }  (         0.016, 3.89740925257,   5331.35744374080 ),
  { 4332  119 }  (         0.017, 3.08268671348, 154717.60988768269 ),
  { 4332  120 }  (         0.016, 3.95085099736,   3097.88382272579 ),
  { 4332  121 }  (         0.016, 3.99041783945,   6283.14316029419 ),
  { 4332  122 }  (         0.020, 6.10644140189, 167283.76158766549 ),
  { 4332  123 }  (         0.015, 4.09775914607,  11712.95531823080 ),
  { 4332  124 }  (         0.016, 5.71769940700,  17298.18232732620 ),
  { 4332  125 }  (         0.016, 3.28894009404,   5884.92684658320 ),
  { 4332  126 }  (         0.015, 5.64785377164,  12559.03815298200 ),
  { 4332  127 }  (         0.016, 4.43452080930,   6283.00853968860 ),
  { 4332  128 }  (         0.014, 2.31721603062,   5481.25491886760 ),
  { 4332  129 }  (         0.014, 4.43479032305,  13517.87010623340 ),
  { 4332  130 }  (         0.014, 4.73209312936,   7342.45778018060 ),
  { 4332  131 }  (         0.012, 0.64705975463,  18073.70493865020 ),
  { 4332  132 }  (         0.011, 1.51443332200,  16200.77272450120 ),
  { 4332  133 }  (         0.011, 0.88708889185,  21228.39202354580 ),
  { 4332  134 }  (         0.014, 4.50116508534,    640.87760738220 ),
  { 4332  135 }  (         0.011, 4.64339996198,  11790.62908865880 ),
  { 4332  136 }  (         0.011, 1.31064298246,   4164.31198961300 ),
  { 4332  137 }  (         0.009, 3.02238989305,  23543.23050468179 ),
  { 4332  138 }  (         0.009, 2.04999402381,  22003.91463486980 ),
  { 4332  139 }  (         0.009, 4.91488110218,    213.29909543800 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_ear_r3:array[0.. 26,0..2] of extended = (..); *)
  (*$ifdef meeus *)
  vsop87_ear_r3:array[0..  1,0..2] of extended = (
  (*$else *)
  vsop87_ear_r3:array[0.. 26,0..2] of extended = (
  (*$endif *)
  { 4333    1 }  (       144.595, 4.27319435148,   6283.07584999140 ),
  { 4333    2 }  (         6.729, 3.91697608662,  12566.15169998280 )
  (*$ifndef meeus *)                                                 ,
  { 4333    3 }  (         0.774, 0.00000000000,      0.00000000000 ),
  { 4333    4 }  (         0.247, 3.73019298781,  18849.22754997420 ),
  { 4333    5 }  (         0.036, 2.80081409050,   6286.59896834040 ),
  { 4333    6 }  (         0.033, 5.62216602775,   6127.65545055720 ),
  { 4333    7 }  (         0.019, 3.71292621802,   6438.49624942560 ),
  { 4333    8 }  (         0.016, 4.26011484232,   6525.80445396540 ),
  { 4333    9 }  (         0.016, 3.50416887054,   6256.77753019160 ),
  { 4333   10 }  (         0.014, 3.62127621114,  25132.30339996560 ),
  { 4333   11 }  (         0.011, 4.39200958819,   4705.73230754360 ),
  { 4333   12 }  (         0.011, 5.22327127059,   6040.34724601740 ),
  { 4333   13 }  (         0.010, 4.28045254647,  83996.84731811189 ),
  { 4333   14 }  (         0.009, 1.56864096494,   5507.55323866740 ),
  { 4333   15 }  (         0.011, 1.37795688024,   6309.37416979120 ),
  { 4333   16 }  (         0.010, 5.19937959068,  71430.69561812909 ),
  { 4333   17 }  (         0.009, 0.47275199930,   6279.55273164240 ),
  { 4333   18 }  (         0.009, 0.74642756529,   5729.50644714900 ),
  { 4333   19 }  (         0.007, 2.97374891560,    775.52261132400 ),
  { 4333   20 }  (         0.007, 3.28615691021,   7058.59846131540 ),
  { 4333   21 }  (         0.007, 2.19184402142,   6812.76681508600 ),
  { 4333   22 }  (         0.005, 3.15419034438,    529.69096509460 ),
  { 4333   23 }  (         0.006, 4.54725567047,   1059.38193018920 ),
  { 4333   24 }  (         0.005, 1.51104406936,   7079.37385680780 ),
  { 4333   25 }  (         0.007, 2.98052059053,   6681.22485339960 ),
  { 4333   26 }  (         0.005, 2.30961231391,  12036.46073488820 ),
  { 4333   27 }  (         0.005, 3.71102966917,   6290.18939699220 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_ear_r4:array[0..  9,0..2] of extended = (..); *)
  (*$ifdef meeus *)
  vsop87_ear_r4:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_r4:array[0..  9,0..2] of extended = (
  (*$endif *)
  { 4334    1 }  (         3.858, 2.56384387339,   6283.07584999140 )
  (*$ifndef meeus *)                                                 ,
  { 4334    2 }  (         0.306, 2.26769501230,  12566.15169998280 ),
  { 4334    3 }  (         0.053, 3.44031471924,   5573.14280143310 ),
  { 4334    4 }  (         0.015, 2.04794573436,  18849.22754997420 ),
  { 4334    5 }  (         0.013, 2.05688873673,  77713.77146812050 ),
  { 4334    6 }  (         0.007, 4.41218854480, 161000.68573767410 ),
  { 4334    7 }  (         0.005, 5.26154653107,   6438.49624942560 ),
  { 4334    8 }  (         0.005, 4.07695126049,   6127.65545055720 ),
  { 4334    9 }  (         0.006, 3.81514213664, 149854.40013480789 ),
  { 4334   10 }  (         0.003, 1.28175749811,   6286.59896834040 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_ear_r5:array[0..  3,0..2] of extended = (..); *)
  (*$ifdef meeus *)
  vsop87_ear_r5:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_r5:array[0..  2,0..2] of extended = (
  (*$endif *)
  (*$ifdef meeus *)
                 (         0.000, 0.00000000000,      0.00000000000 )
  (*$else *)
  { 4335    1 }  (         0.086, 1.21579741687,   6283.07584999140 ),
  { 4335    2 }  (         0.012, 0.65617264033,  12566.15169998280 ),
  { 4335    3 }  (         0.001, 0.38068797142,  18849.22754997420 )
  (*$endif *)
      );
  (*@\\\0000000601*)
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_ear_r0)) and (nr<=high(vsop87_ear_r0)) then begin
        a:=vsop87_ear_r0[nr,0];  b:=vsop87_ear_r0[nr,1];  c:=vsop87_ear_r0[nr,2];
        end;
      1: if (nr>=low(vsop87_ear_r1)) and (nr<=high(vsop87_ear_r1)) then begin
        a:=vsop87_ear_r1[nr,0];  b:=vsop87_ear_r1[nr,1];  c:=vsop87_ear_r1[nr,2];
        end;
      2: if (nr>=low(vsop87_ear_r2)) and (nr<=high(vsop87_ear_r2)) then begin
        a:=vsop87_ear_r2[nr,0];  b:=vsop87_ear_r2[nr,1];  c:=vsop87_ear_r2[nr,2];
        end;
      3: if (nr>=low(vsop87_ear_r3)) and (nr<=high(vsop87_ear_r3)) then begin
        a:=vsop87_ear_r3[nr,0];  b:=vsop87_ear_r3[nr,1];  c:=vsop87_ear_r3[nr,2];
        end;
      4: if (nr>=low(vsop87_ear_r4)) and (nr<=high(vsop87_ear_r4)) then begin
        a:=vsop87_ear_r4[nr,0];  b:=vsop87_ear_r4[nr,1];  c:=vsop87_ear_r4[nr,2];
        end;
      5: if (nr>=low(vsop87_ear_r5)) and (nr<=high(vsop87_ear_r5)) then begin
        a:=vsop87_ear_r5[nr,0];  b:=vsop87_ear_r5[nr,1];  c:=vsop87_ear_r5[nr,2];
        end;
      end;
    end;
  end;
(*@\\\0000000301*)
(*@/// function TVSOPEarth.LatitudeFactor(nr,index: integer):TVSOPEntry; *)
function TVSOPEarth.LatitudeFactor(nr,index: integer):TVSOPEntry;
const
  (*@/// vsop87_ear_b0:array[0..183,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_b0:array[0..  4,0..2] of extended = (
  (*$else *)
  vsop87_ear_b0:array[0..183,0..2] of extended = (
  (*$endif *)
  { 4320    1 }  (       279.620, 3.19870156017,  84334.66158130829 ),
  { 4320    2 }  (       101.643, 5.42248619256,   5507.55323866740 ),
  { 4320    3 }  (        80.445, 3.88013204458,   5223.69391980220 ),
  { 4320    4 }  (        43.806, 3.70444689758,   2352.86615377180 ),
  { 4320    5 }  (        31.933, 4.00026369781,   1577.34354244780 )
  (*$ifndef meeus *)                                                 ,
  { 4320    6 }  (        22.724, 3.98473831560,   1047.74731175470 ),
  { 4320    7 }  (        16.392, 3.56456119782,   5856.47765911540 ),
  { 4320    8 }  (        18.141, 4.98367470263,   6283.07584999140 ),
  { 4320    9 }  (        14.443, 3.70275614914,   9437.76293488700 ),
  { 4320   10 }  (        14.304, 3.41117857525,  10213.28554621100 ),
  { 4320   11 }  (        11.246, 4.82820690530,  14143.49524243060 ),
  { 4320   12 }  (        10.900, 2.08574562327,   6812.76681508600 ),
  { 4320   13 }  (         9.714, 3.47303947752,   4694.00295470760 ),
  { 4320   14 }  (        10.367, 4.05663927946,  71092.88135493269 ),
  { 4320   15 }  (         8.775, 4.44016515669,   5753.38488489680 ),
  { 4320   16 }  (         8.366, 4.99251512180,   7084.89678111520 ),
  { 4320   17 }  (         6.921, 4.32559054073,   6275.96230299060 ),
  { 4320   18 }  (         9.145, 1.14182646613,   6620.89011318780 ),
  { 4320   19 }  (         7.194, 3.60193205752,    529.69096509460 ),
  { 4320   20 }  (         7.698, 5.55425745881, 167621.57585086189 ),
  { 4320   21 }  (         5.285, 2.48446991566,   4705.73230754360 ),
  { 4320   22 }  (         5.208, 6.24992674537,  18073.70493865020 ),
  { 4320   23 }  (         4.529, 2.33827747356,   6309.37416979120 ),
  { 4320   24 }  (         5.579, 4.41023653738,   7860.41939243920 ),
  { 4320   25 }  (         4.743, 0.70995680136,   5884.92684658320 ),
  { 4320   26 }  (         4.301, 1.10255777773,   6681.22485339960 ),
  { 4320   27 }  (         3.849, 1.82229412531,   5486.77784317500 ),
  { 4320   28 }  (         4.093, 5.11700141207,  13367.97263110660 ),
  { 4320   29 }  (         3.681, 0.43793170356,   3154.68708489560 ),
  { 4320   30 }  (         3.420, 5.42034800952,   6069.77675455340 ),
  { 4320   31 }  (         3.617, 6.04641937526,   3930.20969621960 ),
  { 4320   32 }  (         3.670, 4.58210192227,  12194.03291462090 ),
  { 4320   33 }  (         2.918, 1.95463881126,  10977.07880469900 ),
  { 4320   34 }  (         2.797, 5.61259275048,  11790.62908865880 ),
  { 4320   35 }  (         2.502, 0.60499729367,   6496.37494542940 ),
  { 4320   36 }  (         2.319, 5.01648216014,   1059.38193018920 ),
  { 4320   37 }  (         2.684, 1.39470396488,  22003.91463486980 ),
  { 4320   38 }  (         2.428, 3.24183056052,  78051.58573131690 ),
  { 4320   39 }  (         2.120, 4.30691000285,   5643.17856367740 ),
  { 4320   40 }  (         2.257, 3.15557225618,  90617.73743129970 ),
  { 4320   41 }  (         1.813, 3.75574218285,   3340.61242669980 ),
  { 4320   42 }  (         2.226, 2.79699346659,  12036.46073488820 ),
  { 4320   43 }  (         1.888, 0.86991545823,   8635.94200376320 ),
  { 4320   44 }  (         1.517, 1.95852055701,    398.14900340820 ),
  { 4320   45 }  (         1.581, 3.19976230948,   5088.62883976680 ),
  { 4320   46 }  (         1.421, 6.25530883827,   2544.31441988340 ),
  { 4320   47 }  (         1.595, 0.25619915135,  17298.18232732620 ),
  { 4320   48 }  (         1.391, 4.69964175561,   7058.59846131540 ),
  { 4320   49 }  (         1.478, 2.81808207569,  25934.12433108940 ),
  { 4320   50 }  (         1.481, 3.65823554806,  11506.76976979360 ),
  { 4320   51 }  (         1.693, 4.95689385293, 156475.29024799570 ),
  { 4320   52 }  (         1.183, 1.29343061246,    775.52261132400 ),
  { 4320   53 }  (         1.114, 2.37889311846,   3738.76143010800 ),
  { 4320   54 }  (         0.994, 4.30088900425,   9225.53927328300 ),
  { 4320   55 }  (         0.924, 3.06451026812,   4164.31198961300 ),
  { 4320   56 }  (         0.867, 0.55606931068,   8429.24126646660 ),
  { 4320   57 }  (         0.988, 5.97286104208,   7079.37385680780 ),
  { 4320   58 }  (         0.824, 1.50984806173,  10447.38783960440 ),
  { 4320   59 }  (         0.915, 0.12635654592,  11015.10647733480 ),
  { 4320   60 }  (         0.742, 1.99159139281,  26087.90314157420 ),
  { 4320   61 }  (         1.039, 3.14159265359,      0.00000000000 ),
  { 4320   62 }  (         0.850, 4.24120016095,  29864.33402730900 ),
  { 4320   63 }  (         0.755, 2.89631873320,   4732.03062734340 ),
  { 4320   64 }  (         0.714, 1.37548118603,   2146.16541647520 ),
  { 4320   65 }  (         0.708, 1.91406542362,   8031.09226305840 ),
  { 4320   66 }  (         0.746, 0.57893808616,    796.29800681640 ),
  { 4320   67 }  (         0.802, 5.12339137230,   2942.46342329160 ),
  { 4320   68 }  (         0.751, 1.67479850166,  21228.39202354580 ),
  { 4320   69 }  (         0.602, 4.09976538826,  64809.80550494129 ),
  { 4320   70 }  (         0.594, 3.49580704962,  16496.36139620240 ),
  { 4320   71 }  (         0.592, 4.59481504319,   4690.47983635860 ),
  { 4320   72 }  (         0.530, 5.73979295200,   8827.39026987480 ),
  { 4320   73 }  (         0.503, 5.66433137112,  33794.54372352860 ),
  { 4320   74 }  (         0.483, 1.57106522411,    801.82093112380 ),
  { 4320   75 }  (         0.438, 0.06707733767,   3128.38876509580 ),
  { 4320   76 }  (         0.423, 2.86944595927,  12566.15169998280 ),
  { 4320   77 }  (         0.504, 3.26207669160,   7632.94325965020 ),
  { 4320   78 }  (         0.552, 1.02926440457, 239762.20451754928 ),
  { 4320   79 }  (         0.427, 3.67434378210,    213.29909543800 ),
  { 4320   80 }  (         0.404, 1.46193297142,  15720.83878487840 ),
  { 4320   81 }  (         0.503, 4.85802444134,   6290.18939699220 ),
  { 4320   82 }  (         0.417, 0.81920713533,   5216.58037280140 ),
  { 4320   83 }  (         0.365, 0.01002966162,  12168.00269657460 ),
  { 4320   84 }  (         0.363, 1.28376436579,   6206.80977871580 ),
  { 4320   85 }  (         0.353, 4.70059133110,   7234.79425624200 ),
  { 4320   86 }  (         0.415, 0.96862624175,   4136.91043351620 ),
  { 4320   87 }  (         0.387, 3.09145061418,  25158.60171976540 ),
  { 4320   88 }  (         0.373, 2.65119262792,   7342.45778018060 ),
  { 4320   89 }  (         0.361, 2.97762937739,   9623.68827669120 ),
  { 4320   90 }  (         0.418, 3.75759994446,   5230.80746680300 ),
  { 4320   91 }  (         0.396, 1.22507712354,   6438.49624942560 ),
  { 4320   92 }  (         0.322, 1.21162178805,   8662.24032356300 ),
  { 4320   93 }  (         0.284, 5.64170320068,   1589.07289528380 ),
  { 4320   94 }  (         0.379, 1.72248432748,  14945.31617355440 ),
  { 4320   95 }  (         0.320, 3.94161159962,   7330.82316174610 ),
  { 4320   96 }  (         0.313, 5.47602376446,   1194.44701022460 ),
  { 4320   97 }  (         0.292, 1.38971327603,  11769.85369316640 ),
  { 4320   98 }  (         0.305, 0.80429352049,  37724.75341974820 ),
  { 4320   99 }  (         0.257, 5.81382809757,    426.59819087600 ),
  { 4320  100 }  (         0.265, 6.10358507671,   6836.64525283380 ),
  { 4320  101 }  (         0.250, 4.56452895547,   7477.52286021600 ),
  { 4320  102 }  (         0.266, 2.62926282354,   7238.67559160000 ),
  { 4320  103 }  (         0.263, 6.22089501237,   6133.51265285680 ),
  { 4320  104 }  (         0.306, 2.79682380531,   1748.01641306700 ),
  { 4320  105 }  (         0.236, 2.46093023714,  11371.70468975820 ),
  { 4320  106 }  (         0.316, 1.62662805006, 250908.49012041549 ),
  { 4320  107 }  (         0.216, 3.68721275185,   5849.36411211460 ),
  { 4320  108 }  (         0.230, 0.36165162947,   5863.59120611620 ),
  { 4320  109 }  (         0.233, 5.03509933858,  20426.57109242200 ),
  { 4320  110 }  (         0.200, 5.86073159059,   4535.05943692440 ),
  { 4320  111 }  (         0.277, 4.65400292395,  82239.16695779889 ),
  { 4320  112 }  (         0.209, 3.72323200804,  10973.55568635000 ),
  { 4320  113 }  (         0.199, 5.05186622555,   5429.87946823940 ),
  { 4320  114 }  (         0.256, 2.40923279770,  19651.04848109800 ),
  { 4320  115 }  (         0.210, 4.50691909144,  29088.81141598500 ),
  { 4320  116 }  (         0.181, 6.00294783127,   4292.33083295040 ),
  { 4320  117 }  (         0.249, 0.12900984422, 154379.79562448629 ),
  { 4320  118 }  (         0.209, 3.87759458598,  17789.84561978500 ),
  { 4320  119 }  (         0.225, 3.18339652605,  18875.52586977400 ),
  { 4320  120 }  (         0.191, 4.53897489299,  18477.10876461230 ),
  { 4320  121 }  (         0.172, 2.09694183014,  13095.84266507740 ),
  { 4320  122 }  (         0.182, 3.16107943500,  16730.46368959580 ),
  { 4320  123 }  (         0.188, 2.22746128596,  41654.96311596780 ),
  { 4320  124 }  (         0.164, 5.18686275017,   5481.25491886760 ),
  { 4320  125 }  (         0.160, 2.49298855159,  12592.45001978260 ),
  { 4320  126 }  (         0.155, 1.59595438230,  10021.83728009940 ),
  { 4320  127 }  (         0.135, 0.21349051064,  10988.80815753500 ),
  { 4320  128 }  (         0.178, 3.80375177970,  23581.25817731760 ),
  { 4320  129 }  (         0.123, 1.66800739151,  15110.46611986620 ),
  { 4320  130 }  (         0.122, 2.72678272244,  18849.22754997420 ),
  { 4320  131 }  (         0.126, 1.17675512910,  14919.01785375460 ),
  { 4320  132 }  (         0.142, 3.95053441332,    337.81426319640 ),
  { 4320  133 }  (         0.116, 6.06340906229,   6709.67404086740 ),
  { 4320  134 }  (         0.137, 3.52143246757,  12139.55350910680 ),
  { 4320  135 }  (         0.136, 2.92179113542,  32217.20018108080 ),
  { 4320  136 }  (         0.110, 3.51203379263,  18052.92954315780 ),
  { 4320  137 }  (         0.147, 4.63371971408,  22805.73556599360 ),
  { 4320  138 }  (         0.108, 5.45280814878,      7.11354700080 ),
  { 4320  139 }  (         0.148, 0.65447253687,  95480.94718417450 ),
  { 4320  140 }  (         0.119, 5.92110458985,  33019.02111220460 ),
  { 4320  141 }  (         0.110, 5.34824206306,    639.89728631400 ),
  { 4320  142 }  (         0.106, 3.71081682629,  14314.16811304980 ),
  { 4320  143 }  (         0.139, 6.17607198418,  24356.78078864160 ),
  { 4320  144 }  (         0.118, 5.59738712670, 161338.50000087050 ),
  { 4320  145 }  (         0.117, 3.65065271640,  45585.17281218740 ),
  { 4320  146 }  (         0.127, 4.74596574209,  49515.38250840700 ),
  { 4320  147 }  (         0.120, 1.04211499785,   6915.85958930460 ),
  { 4320  148 }  (         0.120, 5.60638811846,   5650.29211067820 ),
  { 4320  149 }  (         0.115, 3.10668213289,  14712.31711645800 ),
  { 4320  150 }  (         0.099, 0.69018940049,  12779.45079542080 ),
  { 4320  151 }  (         0.097, 1.07908724794,   9917.69687450980 ),
  { 4320  152 }  (         0.093, 2.62295197319,  17260.15465469040 ),
  { 4320  153 }  (         0.099, 4.45774681732,   4933.20844033260 ),
  { 4320  154 }  (         0.123, 1.37488922089,  28286.99048486120 ),
  { 4320  155 }  (         0.121, 5.19767249813,  27511.46787353720 ),
  { 4320  156 }  (         0.105, 0.87192267806,  77375.95720492408 ),
  { 4320  157 }  (         0.087, 3.93637812950,  17654.78053974960 ),
  { 4320  158 }  (         0.122, 2.23956068680,  83997.09113559539 ),
  { 4320  159 }  (         0.087, 4.18201600952,  22779.43724619380 ),
  { 4320  160 }  (         0.104, 4.59580877295,   1349.86740965880 ),
  { 4320  161 }  (         0.102, 2.83545248411,  12352.85260454480 ),
  { 4320  162 }  (         0.102, 3.97386522171,  10818.13528691580 ),
  { 4320  163 }  (         0.101, 4.32892825857,  36147.40987730040 ),
  { 4320  164 }  (         0.094, 5.00001709261, 150192.21439800429 ),
  { 4320  165 }  (         0.077, 3.97199369296,   1592.59601363280 ),
  { 4320  166 }  (         0.100, 6.07733097102,  26735.94526221320 ),
  { 4320  167 }  (         0.086, 5.26029638250,  28313.28880466100 ),
  { 4320  168 }  (         0.093, 4.31900620254,  44809.65020086340 ),
  { 4320  169 }  (         0.076, 6.22743405935,  13521.75144159140 ),
  { 4320  170 }  (         0.072, 1.55820597747,   6256.77753019160 ),
  { 4320  171 }  (         0.082, 4.95202664555,  10575.40668294180 ),
  { 4320  172 }  (         0.082, 1.69647647075,   1990.74501704100 ),
  { 4320  173 }  (         0.075, 2.29836095644,   3634.62102451840 ),
  { 4320  174 }  (         0.075, 2.66367876557,  16200.77272450120 ),
  { 4320  175 }  (         0.087, 0.26630214764,  31441.67756975680 ),
  { 4320  176 }  (         0.077, 2.25530954137,   5235.32853823670 ),
  { 4320  177 }  (         0.076, 1.09869730846,  12903.96596317920 ),
  { 4320  178 }  (         0.058, 4.28246138307,  12559.03815298200 ),
  { 4320  179 }  (         0.064, 5.51112830114, 173904.65170085328 ),
  { 4320  180 }  (         0.056, 2.60133794851,  73188.37597844210 ),
  { 4320  181 }  (         0.055, 5.81483150022, 143233.51002162008 ),
  { 4320  182 }  (         0.054, 3.38482031504, 323049.11878710288 ),
  { 4320  183 }  (         0.039, 3.28500401343,  71768.50988132549 ),
  { 4320  184 }  (         0.039, 3.11239910690,  96900.81328129109 )
  (*$endif *)
      );
  (*@\\\0000000B01*)
  (*@/// vsop87_ear_b1:array[0.. 98,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_b1:array[0..  1,0..2] of extended = (
  (*$else *)
  vsop87_ear_b1:array[0.. 98,0..2] of extended = (
  (*$endif *)
  { 4321    1 }  (         9.030, 3.89729061890,   5507.55323866740 ),
  { 4321    2 }  (         6.177, 1.73038850355,   5223.69391980220 )
  (*$ifndef meeus *)                                                 ,
  { 4321    3 }  (         3.800, 5.24404145734,   2352.86615377180 ),
  { 4321    4 }  (         2.834, 2.47345037450,   1577.34354244780 ),
  { 4321    5 }  (         1.817, 0.41874743765,   6283.07584999140 ),
  { 4321    6 }  (         1.499, 1.83320979291,   5856.47765911540 ),
  { 4321    7 }  (         1.466, 5.69401926017,   5753.38488489680 ),
  { 4321    8 }  (         1.301, 2.18890066314,   9437.76293488700 ),
  { 4321    9 }  (         1.233, 4.95222451476,  10213.28554621100 ),
  { 4321   10 }  (         1.021, 0.12866660208,   7860.41939243920 ),
  { 4321   11 }  (         0.982, 0.09005453285,  14143.49524243060 ),
  { 4321   12 }  (         0.865, 1.73949953555,   3930.20969621960 ),
  { 4321   13 }  (         0.581, 2.26949174067,   5884.92684658320 ),
  { 4321   14 }  (         0.524, 5.65662503159,    529.69096509460 ),
  { 4321   15 }  (         0.473, 6.22750969242,   6309.37416979120 ),
  { 4321   16 }  (         0.451, 1.53288619213,  18073.70493865020 ),
  { 4321   17 }  (         0.364, 3.61614477374,  13367.97263110660 ),
  { 4321   18 }  (         0.372, 3.22470721320,   6275.96230299060 ),
  { 4321   19 }  (         0.268, 2.34341267879,  11790.62908865880 ),
  { 4321   20 }  (         0.322, 0.94084045832,   6069.77675455340 ),
  { 4321   21 }  (         0.232, 0.26781182579,   7058.59846131540 ),
  { 4321   22 }  (         0.216, 6.05952221329,  10977.07880469900 ),
  { 4321   23 }  (         0.232, 2.93325646109,  22003.91463486980 ),
  { 4321   24 }  (         0.204, 3.86264841382,   6496.37494542940 ),
  { 4321   25 }  (         0.202, 2.81892511133,  15720.83878487840 ),
  { 4321   26 }  (         0.185, 4.93512381859,  12036.46073488820 ),
  { 4321   27 }  (         0.220, 3.99305643742,   6812.76681508600 ),
  { 4321   28 }  (         0.166, 1.74970002999,  11506.76976979360 ),
  { 4321   29 }  (         0.212, 1.57166285369,   4694.00295470760 ),
  { 4321   30 }  (         0.157, 1.08259734788,   5643.17856367740 ),
  { 4321   31 }  (         0.154, 5.99434678412,   5486.77784317500 ),
  { 4321   32 }  (         0.144, 5.23285656085,  78051.58573131690 ),
  { 4321   33 }  (         0.144, 1.16454655948,  90617.73743129970 ),
  { 4321   34 }  (         0.137, 2.67760436027,   6290.18939699220 ),
  { 4321   35 }  (         0.180, 2.06509026215,   7084.89678111520 ),
  { 4321   36 }  (         0.121, 5.90212574947,   9225.53927328300 ),
  { 4321   37 }  (         0.150, 2.00175038718,   5230.80746680300 ),
  { 4321   38 }  (         0.149, 5.06157254516,  17298.18232732620 ),
  { 4321   39 }  (         0.118, 5.39979058038,   3340.61242669980 ),
  { 4321   40 }  (         0.161, 3.32421999691,   6283.31966747490 ),
  { 4321   41 }  (         0.121, 4.36722193162,  19651.04848109800 ),
  { 4321   42 }  (         0.116, 5.83462858507,   4705.73230754360 ),
  { 4321   43 }  (         0.128, 4.35489873365,  25934.12433108940 ),
  { 4321   44 }  (         0.143, 0.00000000000,      0.00000000000 ),
  { 4321   45 }  (         0.109, 2.52157834166,   6438.49624942560 ),
  { 4321   46 }  (         0.099, 2.70727488041,   5216.58037280140 ),
  { 4321   47 }  (         0.103, 0.93782340879,   8827.39026987480 ),
  { 4321   48 }  (         0.082, 4.29214680390,   8635.94200376320 ),
  { 4321   49 }  (         0.079, 2.24085737326,   1059.38193018920 ),
  { 4321   50 }  (         0.097, 5.50959692365,  29864.33402730900 ),
  { 4321   51 }  (         0.072, 0.21891639822,  21228.39202354580 ),
  { 4321   52 }  (         0.071, 2.86755026812,   6681.22485339960 ),
  { 4321   53 }  (         0.074, 2.20184828895,  37724.75341974820 ),
  { 4321   54 }  (         0.063, 4.45586625948,   7079.37385680780 ),
  { 4321   55 }  (         0.061, 0.63918772258,  33794.54372352860 ),
  { 4321   56 }  (         0.047, 2.09070235724,   3128.38876509580 ),
  { 4321   57 }  (         0.047, 3.32543843300,  26087.90314157420 ),
  { 4321   58 }  (         0.049, 1.60680905005,   6702.56049386660 ),
  { 4321   59 }  (         0.057, 0.11215813438,  29088.81141598500 ),
  { 4321   60 }  (         0.056, 5.47982934911,    775.52261132400 ),
  { 4321   61 }  (         0.050, 1.89396788463,  12139.55350910680 ),
  { 4321   62 }  (         0.047, 2.97214907240,  20426.57109242200 ),
  { 4321   63 }  (         0.041, 5.55329394890,  11015.10647733480 ),
  { 4321   64 }  (         0.041, 5.91861144924,  23581.25817731760 ),
  { 4321   65 }  (         0.045, 4.95273290181,   5863.59120611620 ),
  { 4321   66 }  (         0.050, 3.62740835096,  41654.96311596780 ),
  { 4321   67 }  (         0.037, 6.09033460601,  64809.80550494129 ),
  { 4321   68 }  (         0.037, 5.86153655431,  12566.15169998280 ),
  { 4321   69 }  (         0.046, 1.65798680284,  25158.60171976540 ),
  { 4321   70 }  (         0.038, 2.00673650251,    426.59819087600 ),
  { 4321   71 }  (         0.036, 6.24373396652,   6283.14316029419 ),
  { 4321   72 }  (         0.036, 0.40465162918,   6283.00853968860 ),
  { 4321   73 }  (         0.032, 6.03707103538,   2942.46342329160 ),
  { 4321   74 }  (         0.041, 4.86809570283,   1592.59601363280 ),
  { 4321   75 }  (         0.028, 4.38359423735,   7632.94325965020 ),
  { 4321   76 }  (         0.028, 6.03334294232,  17789.84561978500 ),
  { 4321   77 }  (         0.026, 3.88971333608,   5331.35744374080 ),
  { 4321   78 }  (         0.026, 5.94932724051,  16496.36139620240 ),
  { 4321   79 }  (         0.031, 1.44666331503,  16730.46368959580 ),
  { 4321   80 }  (         0.026, 6.26376705837,  23543.23050468179 ),
  { 4321   81 }  (         0.033, 0.93797239147,    213.29909543800 ),
  { 4321   82 }  (         0.026, 3.71858432944,  13095.84266507740 ),
  { 4321   83 }  (         0.027, 0.60565274405,  10988.80815753500 ),
  { 4321   84 }  (         0.023, 4.44388985550,  18849.22754997420 ),
  { 4321   85 }  (         0.028, 1.53862289477,   6279.48542133960 ),
  { 4321   86 }  (         0.028, 1.96831814872,   6286.66627864320 ),
  { 4321   87 }  (         0.028, 5.78094918529,  15110.46611986620 ),
  { 4321   88 }  (         0.026, 2.48165809843,   5729.50644714900 ),
  { 4321   89 }  (         0.020, 3.85655029499,   9623.68827669120 ),
  { 4321   90 }  (         0.021, 5.83006047147,   7234.79425624200 ),
  { 4321   91 }  (         0.021, 0.69628570421,    398.14900340820 ),
  { 4321   92 }  (         0.022, 5.02222806555,   6127.65545055720 ),
  { 4321   93 }  (         0.020, 3.47611265290,   6148.01076995600 ),
  { 4321   94 }  (         0.020, 0.90769829044,   5481.25491886760 ),
  { 4321   95 }  (         0.020, 0.03081589303,   6418.14093002680 ),
  { 4321   96 }  (         0.020, 3.74220084927,   1589.07289528380 ),
  { 4321   97 }  (         0.021, 4.00149269576,   3154.68708489560 ),
  { 4321   98 }  (         0.018, 1.58348238359,   2118.76386037840 ),
  { 4321   99 }  (         0.019, 0.85407021371,  14712.31711645800 )
  (*$endif *)
      );
  (*@\\\0000006944*)
  (*@/// vsop87_ear_b2:array[0.. 48,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_b2:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_b2:array[0.. 48,0..2] of extended = (
  (*$endif *)
  (*$ifdef meeus *)
                 (         0.000, 0.00000000000,      0.00000000000 )
  (*$else *)
  { 4322    1 }  (         1.662, 1.62703209173,  84334.66158130829 ),
  { 4322    2 }  (         0.492, 2.41382223971,   1047.74731175470 ),
  { 4322    3 }  (         0.344, 2.24353004539,   5507.55323866740 ),
  { 4322    4 }  (         0.258, 6.00906896311,   5223.69391980220 ),
  { 4322    5 }  (         0.131, 0.95447345240,   6283.07584999140 ),
  { 4322    6 }  (         0.086, 1.67530247303,   7860.41939243920 ),
  { 4322    7 }  (         0.090, 0.97606804452,   1577.34354244780 ),
  { 4322    8 }  (         0.090, 0.37899871725,   2352.86615377180 ),
  { 4322    9 }  (         0.089, 6.25807507963,  10213.28554621100 ),
  { 4322   10 }  (         0.075, 0.84213523741, 167621.57585086189 ),
  { 4322   11 }  (         0.052, 1.70501566089,  14143.49524243060 ),
  { 4322   12 }  (         0.057, 6.15295833679,  12194.03291462090 ),
  { 4322   13 }  (         0.051, 1.27616016740,   5753.38488489680 ),
  { 4322   14 }  (         0.051, 5.37229738682,   6812.76681508600 ),
  { 4322   15 }  (         0.034, 1.73672994279,   7058.59846131540 ),
  { 4322   16 }  (         0.038, 2.77761031485,  10988.80815753500 ),
  { 4322   17 }  (         0.046, 3.38617099014, 156475.29024799570 ),
  { 4322   18 }  (         0.021, 1.95248349228,   8827.39026987480 ),
  { 4322   19 }  (         0.018, 3.33419222028,   8429.24126646660 ),
  { 4322   20 }  (         0.019, 4.32945160287,  17789.84561978500 ),
  { 4322   21 }  (         0.017, 0.66191210656,   6283.00853968860 ),
  { 4322   22 }  (         0.018, 3.74885333072,  11769.85369316640 ),
  { 4322   23 }  (         0.017, 4.23058370776,  10977.07880469900 ),
  { 4322   24 }  (         0.017, 1.78116162721,   5486.77784317500 ),
  { 4322   25 }  (         0.021, 1.36972913918,  12036.46073488820 ),
  { 4322   26 }  (         0.017, 2.79601092529,    796.29800681640 ),
  { 4322   27 }  (         0.015, 0.43087848850,  11790.62908865880 ),
  { 4322   28 }  (         0.017, 1.35132152761,  78051.58573131690 ),
  { 4322   29 }  (         0.015, 1.17032155085,    213.29909543800 ),
  { 4322   30 }  (         0.018, 2.85221514199,   5088.62883976680 ),
  { 4322   31 }  (         0.017, 0.21780913672,   6283.14316029419 ),
  { 4322   32 }  (         0.013, 1.21201504386,  25132.30339996560 ),
  { 4322   33 }  (         0.012, 1.12953712197,  90617.73743129970 ),
  { 4322   34 }  (         0.012, 5.13714452592,   7079.37385680780 ),
  { 4322   35 }  (         0.013, 3.79842135217,   4933.20844033260 ),
  { 4322   36 }  (         0.012, 4.89407978213,   3738.76143010800 ),
  { 4322   37 }  (         0.015, 6.05682328852,    398.14900340820 ),
  { 4322   38 }  (         0.014, 4.81029291856,   4694.00295470760 ),
  { 4322   39 }  (         0.011, 0.61684523405,   3128.38876509580 ),
  { 4322   40 }  (         0.011, 5.32876538500,   6040.34724601740 ),
  { 4322   41 }  (         0.014, 5.27227350286,   4535.05943692440 ),
  { 4322   42 }  (         0.011, 2.39292099451,   5331.35744374080 ),
  { 4322   43 }  (         0.010, 4.45296532710,   6525.80445396540 ),
  { 4322   44 }  (         0.014, 4.66400985037,   8031.09226305840 ),
  { 4322   45 }  (         0.010, 3.22472385926,   9437.76293488700 ),
  { 4322   46 }  (         0.011, 3.80913404437,    801.82093112380 ),
  { 4322   47 }  (         0.010, 5.15032130575,  11371.70468975820 ),
  { 4322   48 }  (         0.013, 0.98720797401,   5729.50644714900 ),
  { 4322   49 }  (         0.009, 5.94191743597,   7632.94325965020 )
  (*$endif *)
      );
  (*@\\\0000000901*)
  (*@/// vsop87_ear_b3:array[0.. 10,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_b3:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_b3:array[0.. 10,0..2] of extended = (
  (*$endif *)
  (*$ifdef meeus *)
                 (         0.000, 0.00000000000,      0.00000000000 )
  (*$else *)
  { 4323    1 }  (         0.011, 0.23877262399,   7860.41939243920 ),
  { 4323    2 }  (         0.009, 1.16069982609,   5507.55323866740 ),
  { 4323    3 }  (         0.008, 1.65357552925,   5884.92684658320 ),
  { 4323    4 }  (         0.008, 2.86720038197,   7058.59846131540 ),
  { 4323    5 }  (         0.007, 3.04818741666,   5486.77784317500 ),
  { 4323    6 }  (         0.007, 2.59437103785,    529.69096509460 ),
  { 4323    7 }  (         0.008, 4.02863090524,   6256.77753019160 ),
  { 4323    8 }  (         0.008, 2.42003508927,   5753.38488489680 ),
  { 4323    9 }  (         0.006, 0.84181087594,   6275.96230299060 ),
  { 4323   10 }  (         0.006, 5.40160929468,   1577.34354244780 ),
  { 4323   11 }  (         0.007, 2.73399865247,   6309.37416979120 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_ear_b4:array[0.. 10,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_b4:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_b4:array[0..  4,0..2] of extended = (
  (*$endif *)
  (*$ifdef meeus *)
                 (         0.000, 0.00000000000,      0.00000000000 )
  (*$else *)
  { 4324    1 }  ( 000000000.004, 0.79662198849,   6438.49624942560 ),
  { 4324    2 }  ( 000000000.005, 0.84308705203,   1047.74731175470 ),
  { 4324    3 }  ( 000000000.005, 0.05711572303,  84334.66158130829 ),
  { 4324    4 }  ( 000000000.003, 3.46779895686,   6279.55273164240 ),
  { 4324    5 }  ( 000000000.003, 2.89822201212,   6127.65545055720 )
  (*$endif *)
      );
  (*@\\\*)
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_ear_b0)) and (nr<=high(vsop87_ear_b0)) then begin
        a:=vsop87_ear_b0[nr,0];  b:=vsop87_ear_b0[nr,1];  c:=vsop87_ear_b0[nr,2];
        end;
      1: if (nr>=low(vsop87_ear_b1)) and (nr<=high(vsop87_ear_b1)) then begin
        a:=vsop87_ear_b1[nr,0];  b:=vsop87_ear_b1[nr,1];  c:=vsop87_ear_b1[nr,2];
        end;
      2: if (nr>=low(vsop87_ear_b2)) and (nr<=high(vsop87_ear_b2)) then begin
        a:=vsop87_ear_b2[nr,0];  b:=vsop87_ear_b2[nr,1];  c:=vsop87_ear_b2[nr,2];
        end;
      3: if (nr>=low(vsop87_ear_b3)) and (nr<=high(vsop87_ear_b3)) then begin
        a:=vsop87_ear_b3[nr,0];  b:=vsop87_ear_b3[nr,1];  c:=vsop87_ear_b3[nr,2];
        end;
      4: if (nr>=low(vsop87_ear_b4)) and (nr<=high(vsop87_ear_b4)) then begin
        a:=vsop87_ear_b4[nr,0];  b:=vsop87_ear_b4[nr,1];  c:=vsop87_ear_b4[nr,2];
        end;
      end;
    end;
  end;
(*@\\\*)
(*@/// function TVSOPEarth.LongitudeFactor(nr,index: integer):TVSOPEntry; *)
function TVSOPEarth.LongitudeFactor(nr,index: integer):TVSOPEntry;
const
  (*@/// vsop87_ear_l0:array[0..558,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_l0:array[0.. 63,0..2] of extended = (
  (*$else *)
  vsop87_ear_l0:array[0..558,0..2] of extended = (
  (*$endif *)
  { 4310    1 }  ( 175347045.673, 0.00000000000,      0.00000000000 ),
  { 4310    2 }  (   3341656.456, 4.66925680417,   6283.07584999140 ),
  { 4310    3 }  (     34894.275, 4.62610241759,  12566.15169998280 ),
  { 4310    4 }  (      3417.571, 2.82886579606,      3.52311834900 ),
  { 4310    5 }  (      3497.056, 2.74411800971,   5753.38488489680 ),
  { 4310    6 }  (      3135.896, 3.62767041758,  77713.77146812050 ),
  { 4310    7 }  (      2676.218, 4.41808351397,   7860.41939243920 ),
  { 4310    8 }  (      2342.687, 6.13516237631,   3930.20969621960 ),
  { 4310    9 }  (      1273.166, 2.03709655772,    529.69096509460 ),
  { 4310   10 }  (      1324.292, 0.74246356352,  11506.76976979360 ),
  { 4310   11 }  (       901.855, 2.04505443513,     26.29831979980 ),
  { 4310   12 }  (      1199.167, 1.10962944315,   1577.34354244780 ),
  { 4310   13 }  (       857.223, 3.50849156957,    398.14900340820 ),
  { 4310   14 }  (       779.786, 1.17882652114,   5223.69391980220 ),
  { 4310   15 }  (       990.250, 5.23268129594,   5884.92684658320 ),
  { 4310   16 }  (       753.141, 2.53339053818,   5507.55323866740 ),
  { 4310   17 }  (       505.264, 4.58292563052,  18849.22754997420 ),
  { 4310   18 }  (       492.379, 4.20506639861,    775.52261132400 ),
  { 4310   19 }  (       356.655, 2.91954116867,      0.06731030280 ),
  { 4310   20 }  (       284.125, 1.89869034186,    796.29800681640 ),
  { 4310   21 }  (       242.810, 0.34481140906,   5486.77784317500 ),
  { 4310   22 }  (       317.087, 5.84901952218,  11790.62908865880 ),
  { 4310   23 }  (       271.039, 0.31488607649,  10977.07880469900 ),
  { 4310   24 }  (       206.160, 4.80646606059,   2544.31441988340 ),
  { 4310   25 }  (       205.385, 1.86947813692,   5573.14280143310 ),
  { 4310   26 }  (       202.261, 2.45767795458,   6069.77675455340 ),
  { 4310   27 }  (       126.184, 1.08302630210,     20.77539549240 ),
  { 4310   28 }  (       155.516, 0.83306073807,    213.29909543800 ),
  { 4310   29 }  (       115.132, 0.64544911683,      0.98032106820 ),
  { 4310   30 }  (       102.851, 0.63599846727,   4694.00295470760 ),
  { 4310   31 }  (       101.724, 4.26679821365,      7.11354700080 ),
  { 4310   32 }  (        99.206, 6.20992940258,   2146.16541647520 ),
  { 4310   33 }  (       132.212, 3.41118275555,   2942.46342329160 ),
  { 4310   34 }  (        97.607, 0.68101272270,    155.42039943420 ),
  { 4310   35 }  (        85.128, 1.29870743025,   6275.96230299060 ),
  { 4310   36 }  (        74.651, 1.75508916159,   5088.62883976680 ),
  { 4310   37 }  (       101.895, 0.97569221824,  15720.83878487840 ),
  { 4310   38 }  (        84.711, 3.67080093025,  71430.69561812909 ),
  { 4310   39 }  (        73.547, 4.67926565481,    801.82093112380 ),
  { 4310   40 }  (        73.874, 3.50319443167,   3154.68708489560 ),
  { 4310   41 }  (        78.756, 3.03698313141,  12036.46073488820 ),
  { 4310   42 }  (        79.637, 1.80791330700,  17260.15465469040 ),
  { 4310   43 }  (        85.803, 5.98322631256, 161000.68573767410 ),
  { 4310   44 }  (        56.963, 2.78430398043,   6286.59896834040 ),
  { 4310   45 }  (        61.148, 1.81839811024,   7084.89678111520 ),
  { 4310   46 }  (        69.627, 0.83297596966,   9437.76293488700 ),
  { 4310   47 }  (        56.116, 4.38694880779,  14143.49524243060 ),
  { 4310   48 }  (        62.449, 3.97763880587,   8827.39026987480 ),
  { 4310   49 }  (        51.145, 0.28306864501,   5856.47765911540 ),
  { 4310   50 }  (        55.577, 3.47006009062,   6279.55273164240 ),
  { 4310   51 }  (        41.036, 5.36817351402,   8429.24126646660 ),
  { 4310   52 }  (        51.605, 1.33282746983,   1748.01641306700 ),
  { 4310   53 }  (        51.992, 0.18914945834,  12139.55350910680 ),
  { 4310   54 }  (        49.000, 0.48735065033,   1194.44701022460 ),
  { 4310   55 }  (        39.200, 6.16832995016,  10447.38783960440 ),
  { 4310   56 }  (        35.566, 1.77597314691,   6812.76681508600 ),
  { 4310   57 }  (        36.770, 6.04133859347,  10213.28554621100 ),
  { 4310   58 }  (        36.596, 2.56955238628,   1059.38193018920 ),
  { 4310   59 }  (        33.291, 0.59309499459,  17789.84561978500 ),
  { 4310   60 }  (        35.954, 1.70876111898,   2352.86615377180 ),
  { 4310   61 }  (        40.938, 2.39850881707,  19651.04848109800 ),
  { 4310   62 }  (        30.047, 2.73975123935,   1349.86740965880 ),
  { 4310   63 }  (        30.412, 0.44294464135,  83996.84731811189 ),
  (*$ifndef meeus *)
  { 4310   64 }  (        23.663, 0.48473567763,   8031.09226305840 ),
  { 4310   65 }  (        23.574, 2.06527720049,   3340.61242669980 ),
  { 4310   66 }  (        21.089, 4.14825464101,    951.71840625060 ),
  { 4310   67 }  (        24.738, 0.21484762138,      3.59042865180 ),
  (*$endif *)
  { 4310   68 }  (        25.352, 3.16470953405,   4690.47983635860 )
  (*$ifndef meeus *)                                                 ,
  { 4310   69 }  (        22.820, 5.22197888032,   4705.73230754360 ),
  { 4310   70 }  (        21.419, 1.42563735525,  16730.46368959580 ),
  { 4310   71 }  (        21.891, 5.55594302562,    553.56940284240 ),
  { 4310   72 }  (        17.481, 4.56052900359,    135.06508003540 ),
  { 4310   73 }  (        19.925, 5.22208471269,  12168.00269657460 ),
  { 4310   74 }  (        19.860, 5.77470167653,   6309.37416979120 ),
  { 4310   75 }  (        20.300, 0.37133792946,    283.85931886520 ),
  { 4310   76 }  (        14.421, 4.19315332546,    242.72860397400 ),
  { 4310   77 }  (        16.225, 5.98837722564,  11769.85369316640 ),
  { 4310   78 }  (        15.077, 4.19567181073,   6256.77753019160 ),
  { 4310   79 }  (        19.124, 3.82219996949,  23581.25817731760 ),
  { 4310   80 }  (        18.888, 5.38626880969, 149854.40013480789 ),
  { 4310   81 }  (        14.346, 3.72355084422,     38.02767263580 ),
  { 4310   82 }  (        17.898, 2.21490735647,  13367.97263110660 ),
  { 4310   83 }  (        12.054, 2.62229588349,    955.59974160860 ),
  { 4310   84 }  (        11.287, 0.17739328092,   4164.31198961300 ),
  { 4310   85 }  (        13.971, 4.40138139996,   6681.22485339960 ),
  { 4310   86 }  (        13.621, 1.88934471407,   7632.94325965020 ),
  { 4310   87 }  (        12.503, 1.13052412208,      5.52292430740 ),
  { 4310   88 }  (        10.498, 5.35909518669,   1592.59601363280 ),
  { 4310   89 }  (         9.803, 0.99947478995,  11371.70468975820 ),
  { 4310   90 }  (         9.220, 4.57138609781,   4292.33083295040 ),
  { 4310   91 }  (        10.327, 6.19982566125,   6438.49624942560 ),
  { 4310   92 }  (        12.003, 1.00351456700,    632.78373931320 ),
  { 4310   93 }  (        10.827, 0.32734520222,    103.09277421860 ),
  { 4310   94 }  (         8.356, 4.53902685948,  25132.30339996560 ),
  { 4310   95 }  (        10.005, 6.02914963280,   5746.27133789600 ),
  { 4310   96 }  (         8.409, 3.29946744189,   7234.79425624200 ),
  { 4310   97 }  (         8.006, 5.82145271907,     28.44918746780 ),
  { 4310   98 }  (        10.523, 0.93871805506,  11926.25441366880 ),
  { 4310   99 }  (         7.686, 3.12142363172,   7238.67559160000 ),
  { 4310  100 }  (         9.378, 2.62414241032,   5760.49843189760 ),
  { 4310  101 }  (         8.127, 6.11228001785,   4732.03062734340 ),
  { 4310  102 }  (         9.232, 0.48343968736,    522.57741809380 ),
  { 4310  103 }  (         9.802, 5.24413991147,  27511.46787353720 ),
  { 4310  104 }  (         7.871, 0.99590177926,   5643.17856367740 ),
  { 4310  105 }  (         8.123, 6.27053013650,    426.59819087600 ),
  { 4310  106 }  (         9.048, 5.33686335897,   6386.16862421000 ),
  { 4310  107 }  (         8.620, 4.16538210888,   7058.59846131540 ),
  { 4310  108 }  (         6.297, 4.71724819317,   6836.64525283380 ),
  { 4310  109 }  (         7.575, 3.97382858911,  11499.65622279280 ),
  { 4310  110 }  (         7.756, 2.95729056763,  23013.53953958720 ),
  { 4310  111 }  (         7.314, 0.60652505806,  11513.88331679440 ),
  { 4310  112 }  (         5.955, 2.87641047971,   6283.14316029419 ),
  { 4310  113 }  (         6.534, 5.79072926033,  18073.70493865020 ),
  { 4310  114 }  (         7.188, 3.99831508699,     74.78159856730 ),
  { 4310  115 }  (         7.346, 4.38582365437,    316.39186965660 ),
  { 4310  116 }  (         5.413, 5.39199024641,    419.48464387520 ),
  { 4310  117 }  (         5.127, 2.36062848786,  10973.55568635000 ),
  { 4310  118 }  (         7.056, 0.32258441903,    263.08392337280 ),
  { 4310  119 }  (         6.625, 3.66475158672,  17298.18232732620 ),
  { 4310  120 }  (         6.762, 5.91132535899,  90955.55169449610 ),
  { 4310  121 }  (         4.938, 5.73672165674,   9917.69687450980 ),
  { 4310  122 }  (         5.547, 2.45152597661,  12352.85260454480 ),
  { 4310  123 }  (         5.958, 3.32051344676,   6283.00853968860 ),
  { 4310  124 }  (         4.471, 2.06385999536,   7079.37385680780 ),
  { 4310  125 }  (         6.153, 1.45823331144, 233141.31440436149 ),
  { 4310  126 }  (         4.348, 4.42342175480,   5216.58037280140 ),
  { 4310  127 }  (         6.123, 1.07494905258,  19804.82729158280 ),
  { 4310  128 }  (         4.488, 3.65285037150,    206.18554843720 ),
  { 4310  129 }  (         4.020, 0.83995823171,     20.35531939880 ),
  { 4310  130 }  (         5.188, 4.06503864016,   6208.29425142410 ),
  { 4310  131 }  (         5.307, 0.38217636096,  31441.67756975680 ),
  { 4310  132 }  (         3.785, 2.34369213733,      3.88133535800 ),
  { 4310  133 }  (         4.497, 3.27230796845,  11015.10647733480 ),
  { 4310  134 }  (         4.132, 0.92128915753,   3738.76143010800 ),
  { 4310  135 }  (         3.521, 5.97844807108,   3894.18182954220 ),
  { 4310  136 }  (         4.215, 1.90601120623,    245.83164622940 ),
  { 4310  137 }  (         3.701, 5.03069397926,    536.80451209540 ),
  { 4310  138 }  (         3.865, 1.82634360607,  11856.21865142450 ),
  { 4310  139 }  (         3.652, 1.01838584934,  16200.77272450120 ),
  { 4310  140 }  (         3.390, 0.97785123922,   8635.94200376320 ),
  { 4310  141 }  (         3.737, 2.95380107829,   3128.38876509580 ),
  { 4310  142 }  (         3.507, 3.71291946325,   6290.18939699220 ),
  { 4310  143 }  (         3.086, 3.64646921512,     10.63666534980 ),
  { 4310  144 }  (         3.397, 1.10590684017,  14712.31711645800 ),
  { 4310  145 }  (         3.334, 0.83684924911,   6496.37494542940 ),
  { 4310  146 }  (         2.805, 2.58504514144,  14314.16811304980 ),
  { 4310  147 }  (         3.650, 1.08344142571,  88860.05707098669 ),
  { 4310  148 }  (         3.388, 3.20185096055,   5120.60114558360 ),
  { 4310  149 }  (         3.252, 3.47859752062,   6133.51265285680 ),
  { 4310  150 }  (         2.553, 3.94869034189,   1990.74501704100 ),
  { 4310  151 }  (         3.520, 2.05559692878, 244287.60000722769 ),
  { 4310  152 }  (         2.565, 1.56071784900,  23543.23050468179 ),
  { 4310  153 }  (         2.621, 3.85639359951,    266.60704172180 ),
  { 4310  154 }  (         2.955, 3.39692949667,   9225.53927328300 ),
  { 4310  155 }  (         2.876, 6.02635617464, 154717.60988768269 ),
  { 4310  156 }  (         2.395, 1.16131956403,  10984.19235169980 ),
  { 4310  157 }  (         3.161, 1.32798718453,  10873.98603048040 ),
  { 4310  158 }  (         3.163, 5.08946464629,  21228.39202354580 ),
  { 4310  159 }  (         2.361, 4.27212906992,   6040.34724601740 ),
  { 4310  160 }  (         3.030, 1.80209931347,  35371.88726597640 ),
  { 4310  161 }  (         2.343, 3.57689860500,  10969.96525769820 ),
  { 4310  162 }  (         2.618, 2.57870156528,  22483.84857449259 ),
  { 4310  163 }  (         2.113, 3.71393780256,  65147.61976813770 ),
  { 4310  164 }  (         2.019, 0.81393923319,    170.67287061920 ),
  { 4310  165 }  (         2.003, 0.38091017375,   6172.86952877200 ),
  { 4310  166 }  (         2.506, 3.74379142438,  10575.40668294180 ),
  { 4310  167 }  (         2.381, 0.10581361289,      7.04623669800 ),
  { 4310  168 }  (         1.949, 4.86892513469,     36.02786667740 ),
  { 4310  169 }  (         2.074, 4.22794774570,   5650.29211067820 ),
  { 4310  170 }  (         1.924, 5.59460549860,   6282.09552892320 ),
  { 4310  171 }  (         1.949, 1.07002512703,   5230.80746680300 ),
  { 4310  172 }  (         1.988, 5.19736046771,   6262.30045449900 ),
  { 4310  173 }  (         1.887, 3.74365662683,     23.87843774780 ),
  { 4310  174 }  (         1.787, 1.25929682929,  12559.03815298200 ),
  { 4310  175 }  (         1.883, 1.90364058477,     15.25247118500 ),
  { 4310  176 }  (         1.816, 3.68083868442,  15110.46611986620 ),
  { 4310  177 }  (         1.701, 4.41105895380,    110.20632121940 ),
  { 4310  178 }  (         1.990, 3.93295788548,   6206.80977871580 ),
  { 4310  179 }  (         2.103, 0.75354917468,  13521.75144159140 ),
  { 4310  180 }  (         1.774, 0.48747535361,   1551.04522264800 ),
  { 4310  181 }  (         1.882, 0.86684493432,  22003.91463486980 ),
  { 4310  182 }  (         1.924, 1.22898324132,    709.93304855830 ),
  { 4310  183 }  (         2.009, 4.62850921980,   6037.24420376200 ),
  { 4310  184 }  (         1.924, 0.60231842508,   6284.05617105960 ),
  { 4310  185 }  (         1.596, 3.98332956992,  13916.01910964160 ),
  { 4310  186 }  (         1.664, 4.41939715469,   8662.24032356300 ),
  { 4310  187 }  (         1.971, 1.04560500503,  18209.33026366019 ),
  { 4310  188 }  (         1.942, 4.31335979989,   6244.94281435360 ),
  { 4310  189 }  (         1.476, 0.93271367331,   2379.16447357160 ),
  { 4310  190 }  (         1.810, 0.49112137707,      1.48447270830 ),
  { 4310  191 }  (         1.346, 1.51574702235,   4136.91043351620 ),
  { 4310  192 }  (         1.528, 5.61835711404,   6127.65545055720 ),
  { 4310  193 }  (         1.791, 3.22187270126,  39302.09696219600 ),
  { 4310  194 }  (         1.747, 3.05638656738,  18319.53658487960 ),
  { 4310  195 }  (         1.431, 4.51153808594,  20426.57109242200 ),
  { 4310  196 }  (         1.695, 0.22047718414,  25158.60171976540 ),
  { 4310  197 }  (         1.242, 4.46665769933,  17256.63153634140 ),
  { 4310  198 }  (         1.463, 4.69242679213,  14945.31617355440 ),
  { 4310  199 }  (         1.205, 1.86912144659,   4590.91018048900 ),
  { 4310  200 }  (         1.192, 2.74227166898,  12569.67481833180 ),
  { 4310  201 }  (         1.222, 5.18120087482,   5333.90024102160 ),
  { 4310  202 }  (         1.390, 5.42894648983, 143571.32428481648 ),
  { 4310  203 }  (         1.473, 1.70479245805,  11712.95531823080 ),
  { 4310  204 }  (         1.362, 2.61069503292,   6062.66320755260 ),
  { 4310  205 }  (         1.148, 6.03001800540,   3634.62102451840 ),
  { 4310  206 }  (         1.198, 5.15294130422,  10177.25767953360 ),
  { 4310  207 }  (         1.266, 0.11421493643,  18422.62935909819 ),
  { 4310  208 }  (         1.411, 1.09908857534,   3496.03282613400 ),
  { 4310  209 }  (         1.349, 2.99805109633,  17654.78053974960 ),
  { 4310  210 }  (         1.253, 2.79850152848, 167283.76158766549 ),
  { 4310  211 }  (         1.311, 1.60942984879,   5481.25491886760 ),
  { 4310  212 }  (         1.079, 6.20304501787,      3.28635741780 ),
  { 4310  213 }  (         1.181, 1.20653776978,    131.54196168640 ),
  { 4310  214 }  (         1.254, 5.45103277798,   6076.89030155420 ),
  { 4310  215 }  (         1.035, 2.32142722747,   7342.45778018060 ),
  { 4310  216 }  (         1.117, 0.38838354256,    949.17560896980 ),
  { 4310  217 }  (         0.966, 3.18341890851,  11087.28512591840 ),
  { 4310  218 }  (         1.171, 3.39635049962,  12562.62858163380 ),
  { 4310  219 }  (         1.121, 0.72627490378,    220.41264243880 ),
  { 4310  220 }  (         1.024, 2.19378315386,  11403.67699557500 ),
  { 4310  221 }  (         0.888, 3.91173199285,   4686.88940770680 ),
  { 4310  222 }  (         0.910, 1.98802695087,    735.87651353180 ),
  { 4310  223 }  (         0.830, 0.48984915507,  24072.92146977640 ),
  { 4310  224 }  (         1.096, 6.17377835617,   5436.99301524020 ),
  { 4310  225 }  (         0.908, 0.44959639433,   7477.52286021600 ),
  { 4310  226 }  (         0.974, 1.52996238356,   9623.68827669120 ),
  { 4310  227 }  (         0.840, 1.79543266333,   5429.87946823940 ),
  { 4310  228 }  (         0.778, 6.17699177946,     38.13303563780 ),
  { 4310  229 }  (         0.776, 4.09855402433,     14.22709400160 ),
  { 4310  230 }  (         1.068, 4.64200173735,  43232.30665841560 ),
  { 4310  231 }  (         0.954, 1.49988435748,   1162.47470440780 ),
  { 4310  232 }  (         0.907, 0.86986870809,  10344.29506538580 ),
  { 4310  233 }  (         0.931, 4.06044689031,  28766.92442448400 ),
  { 4310  234 }  (         0.739, 5.04368197372,    639.89728631400 ),
  { 4310  235 }  (         0.937, 3.46884698960,   1589.07289528380 ),
  { 4310  236 }  (         0.763, 5.86304932998,  16858.48253293320 ),
  { 4310  237 }  (         0.953, 4.20801492835,  11190.37790013700 ),
  { 4310  238 }  (         0.708, 1.72899988940,  13095.84266507740 ),
  { 4310  239 }  (         0.969, 1.64439522215,  29088.81141598500 ),
  { 4310  240 }  (         0.717, 0.16688678895,     11.72935283600 ),
  { 4310  241 }  (         0.962, 3.53092337542,  12416.58850284820 ),
  { 4310  242 }  (         0.747, 5.77866940346,  12592.45001978260 ),
  { 4310  243 }  (         0.672, 1.91095796194,      3.93215326310 ),
  { 4310  244 }  (         0.671, 5.46240843677,  18052.92954315780 ),
  { 4310  245 }  (         0.675, 6.28311558823,   4535.05943692440 ),
  { 4310  246 }  (         0.684, 0.39975012080,   5849.36411211460 ),
  { 4310  247 }  (         0.799, 0.29851185294,  12132.43996210600 ),
  { 4310  248 }  (         0.758, 0.96370823331,   1052.26838318840 ),
  { 4310  249 }  (         0.782, 5.33878339919,  13517.87010623340 ),
  { 4310  250 }  (         0.730, 1.70106160291,  17267.26820169119 ),
  { 4310  251 }  (         0.749, 2.59599901875,  11609.86254401220 ),
  { 4310  252 }  (         0.734, 2.78417782952,    640.87760738220 ),
  { 4310  253 }  (         0.688, 5.15048287468,  16496.36139620240 ),
  { 4310  254 }  (         0.770, 1.62469589333,   4701.11650170840 ),
  { 4310  255 }  (         0.633, 2.20587893893,  25934.12433108940 ),
  { 4310  256 }  (         0.760, 4.21317219403,    377.37360791580 ),
  { 4310  257 }  (         0.584, 2.13420121623,  10557.59416082380 ),
  { 4310  258 }  (         0.574, 0.24250054587,   9779.10867612540 ),
  { 4310  259 }  (         0.573, 3.16435264609,    533.21408344360 ),
  { 4310  260 }  (         0.685, 3.19344289472,  12146.66705610760 ),
  { 4310  261 }  (         0.675, 0.96179233959,  10454.50138660520 ),
  { 4310  262 }  (         0.648, 1.46327342555,   6268.84875598980 ),
  { 4310  263 }  (         0.589, 2.50543543638,   3097.88382272579 ),
  { 4310  264 }  (         0.551, 5.28099026956,   9388.00590941520 ),
  { 4310  265 }  (         0.696, 3.65342150016,   4804.20927592700 ),
  { 4310  266 }  (         0.669, 2.51030077026,   2388.89402044920 ),
  { 4310  267 }  (         0.550, 0.06883864342,  20199.09495963300 ),
  { 4310  268 }  (         0.629, 4.13350995675,  45892.73043315699 ),
  { 4310  269 }  (         0.678, 6.09190163533,    135.62532501000 ),
  { 4310  270 }  (         0.593, 1.50136257618, 226858.23855437008 ),
  { 4310  271 }  (         0.542, 3.58573645173,   6148.01076995600 ),
  { 4310  272 }  (         0.682, 5.02203067788,  17253.04110768959 ),
  { 4310  273 }  (         0.565, 4.29309238610,  11933.36796066960 ),
  { 4310  274 }  (         0.486, 0.77746204893,     27.40155609680 ),
  { 4310  275 }  (         0.503, 0.58963565969,  15671.08175940660 ),
  { 4310  276 }  (         0.616, 4.06539884128,    227.47613278900 ),
  { 4310  277 }  (         0.583, 6.12695541996,  18875.52586977400 ),
  { 4310  278 }  (         0.537, 2.15056440980,  21954.15760939799 ),
  { 4310  279 }  (         0.669, 6.06986269566,  47162.51635463520 ),
  { 4310  280 }  (         0.475, 0.40343842110,   6915.85958930460 ),
  { 4310  281 }  (         0.540, 2.83444222174,   5326.78669402080 ),
  { 4310  282 }  (         0.530, 5.26359885263,  10988.80815753500 ),
  { 4310  283 }  (         0.582, 3.24533095664,    153.77881048480 ),
  { 4310  284 }  (         0.641, 3.24711791371,   2107.03450754240 ),
  { 4310  285 }  (         0.621, 3.09698523779,  33019.02111220460 ),
  { 4310  286 }  (         0.466, 3.14982372198,  10440.27429260360 ),
  { 4310  287 }  (         0.466, 0.90708835657,   5966.68398033480 ),
  { 4310  288 }  (         0.528, 0.81926454470,    813.55028395980 ),
  { 4310  289 }  (         0.603, 3.81378921927, 316428.22867391503 ),
  { 4310  290 }  (         0.559, 1.81894804124,  17996.03116822220 ),
  { 4310  291 }  (         0.437, 2.28625594435,   6303.85124548380 ),
  { 4310  292 }  (         0.518, 4.86069178322,  20597.24396304120 ),
  { 4310  293 }  (         0.424, 6.23520018693,   6489.26139842860 ),
  { 4310  294 }  (         0.518, 6.17617826756,      0.24381748350 ),
  { 4310  295 }  (         0.404, 5.72804304258,   5642.19824260920 ),
  { 4310  296 }  (         0.458, 1.34117773915,   6287.00800325450 ),
  { 4310  297 }  (         0.548, 5.68454458320, 155427.54293624099 ),
  { 4310  298 }  (         0.547, 1.03391472061,   3646.35037735440 ),
  { 4310  299 }  (         0.428, 4.69800981138,    846.08283475120 ),
  { 4310  300 }  (         0.413, 6.02520699406,   6279.48542133960 ),
  { 4310  301 }  (         0.534, 3.03030638223,  66567.48586525429 ),
  { 4310  302 }  (         0.383, 1.49056949125,  19800.94595622480 ),
  { 4310  303 }  (         0.410, 5.28319622279,  18451.07854656599 ),
  { 4310  304 }  (         0.352, 4.68891600359,   4907.30205014560 ),
  { 4310  305 }  (         0.480, 5.36572651091,    348.92442044800 ),
  { 4310  306 }  (         0.344, 5.89157452896,   6546.15977336420 ),
  { 4310  307 }  (         0.340, 0.37557426440,  13119.72110282519 ),
  { 4310  308 }  (         0.434, 4.98417785901,   6702.56049386660 ),
  { 4310  309 }  (         0.332, 2.68902519126,  29296.61538957860 ),
  { 4310  310 }  (         0.448, 2.16478480251,   5905.70224207560 ),
  { 4310  311 }  (         0.344, 2.06546633735,     49.75702547180 ),
  { 4310  312 }  (         0.315, 1.24023811803,   4061.21921539440 ),
  { 4310  313 }  (         0.324, 2.30897526929,   5017.50837136500 ),
  { 4310  314 }  (         0.413, 0.17171692962,   6286.66627864320 ),
  { 4310  315 }  (         0.431, 3.86601101393,  12489.88562870720 ),
  { 4310  316 }  (         0.349, 4.55372342974,   4933.20844033260 ),
  { 4310  317 }  (         0.323, 0.41971136084,  10770.89325626180 ),
  { 4310  318 }  (         0.341, 2.68612860807,     11.04570026390 ),
  { 4310  319 }  (         0.316, 3.52936906658,  17782.73207278420 ),
  { 4310  320 }  (         0.315, 5.63357264999,    568.82187402740 ),
  { 4310  321 }  (         0.340, 3.83571212349,  10660.68693504240 ),
  { 4310  322 }  (         0.297, 0.62691416712,  20995.39296644940 ),
  { 4310  323 }  (         0.405, 1.00085779471,  16460.33352952499 ),
  { 4310  324 }  (         0.414, 1.21998752076,  51092.72605085480 ),
  { 4310  325 }  (         0.336, 4.71465945226,   6179.98307577280 ),
  { 4310  326 }  (         0.361, 3.71227508354,  28237.23345938940 ),
  { 4310  327 }  (         0.385, 6.21925225757,  24356.78078864160 ),
  { 4310  328 }  (         0.327, 1.05606504715,  11919.14086666800 ),
  { 4310  329 }  (         0.327, 6.14222420989,   6254.62666252360 ),
  { 4310  330 }  (         0.268, 2.47224339737,    664.75604513000 ),
  { 4310  331 }  (         0.269, 1.86207884109,  23141.55838292460 ),
  { 4310  332 }  (         0.345, 0.93461290184,   6058.73105428950 ),
  { 4310  333 }  (         0.296, 4.51687557180,   6418.14093002680 ),
  { 4310  334 }  (         0.353, 4.50033653082,  36949.23080842420 ),
  { 4310  335 }  (         0.260, 4.04963546305,   6525.80445396540 ),
  { 4310  336 }  (         0.298, 2.20046722622, 156137.47598479928 ),
  { 4310  337 }  (         0.253, 3.49900838384,  29864.33402730900 ),
  { 4310  338 }  (         0.254, 2.44901693835,   5331.35744374080 ),
  { 4310  339 }  (         0.296, 0.84347588787,   5729.50644714900 ),
  { 4310  340 }  (         0.298, 1.29194706125,  22805.73556599360 ),
  { 4310  341 }  (         0.241, 2.00721280805,  16737.57723659660 ),
  { 4310  342 }  (         0.311, 1.23668016334,   6281.59137728310 ),
  { 4310  343 }  (         0.240, 2.51650377121,   6245.04817735560 ),
  { 4310  344 }  (         0.332, 3.55576945724,   7668.63742494250 ),
  { 4310  345 }  (         0.264, 4.44052061202,  12964.30070339100 ),
  { 4310  346 }  (         0.257, 1.79654471948,  11080.17157891760 ),
  { 4310  347 }  (         0.260, 3.33077598420,   5888.44996493220 ),
  { 4310  348 }  (         0.285, 0.30886361430,  11823.16163945020 ),
  { 4310  349 }  (         0.290, 5.70141882483,     77.67377042800 ),
  { 4310  350 }  (         0.255, 4.00939664440,   5881.40372823420 ),
  { 4310  351 }  (         0.253, 4.73318493678,  16723.35014259500 ),
  { 4310  352 }  (         0.228, 0.95333661324,   5540.08578945880 ),
  { 4310  353 }  (         0.319, 1.38633229189, 163096.18036118349 ),
  { 4310  354 }  (         0.224, 1.65156322696,  10027.90319572920 ),
  { 4310  355 }  (         0.226, 0.34106460604,  17796.95916678580 ),
  { 4310  356 }  (         0.236, 4.19817431922,     19.66976089979 ),
  { 4310  357 }  (         0.280, 4.14080268970,  12539.85338018300 ),
  { 4310  358 }  (         0.275, 5.50306930248,     32.53255079140 ),
  { 4310  359 }  (         0.223, 5.23334210294,     56.89837493560 ),
  { 4310  360 }  (         0.217, 6.08587881787,   6805.65326808520 ),
  { 4310  361 }  (         0.280, 4.52472044653,   6016.46880826960 ),
  { 4310  362 }  (         0.227, 5.06509843737,   6277.55292568400 ),
  { 4310  363 }  (         0.226, 5.17755154305,  11720.06886523160 ),
  { 4310  364 }  (         0.245, 3.96486270306,     22.77520145080 ),
  { 4310  365 }  (         0.220, 4.72078081970,      6.62855890001 ),
  { 4310  366 }  (         0.207, 5.71701403951,     41.55079098480 ),
  { 4310  367 }  (         0.204, 3.91227411250,   2699.73481931760 ),
  { 4310  368 }  (         0.209, 0.86881969011,   6321.10352262720 ),
  { 4310  369 }  (         0.200, 2.11984445273,   4274.51831083240 ),
  { 4310  370 }  (         0.200, 5.39839888163,   6019.99192661860 ),
  { 4310  371 }  (         0.209, 5.67606291663,  11293.47067435560 ),
  { 4310  372 }  (         0.252, 1.64965729351,   9380.95967271720 ),
  { 4310  373 }  (         0.275, 5.04826903506,     73.29712585900 ),
  { 4310  374 }  (         0.208, 1.88207277133,  11300.58422135640 ),
  { 4310  375 }  (         0.272, 0.74640926842,   1975.49254585600 ),
  { 4310  376 }  (         0.199, 3.30836672397,  22743.40937951640 ),
  { 4310  377 }  (         0.269, 4.48560812155,  64471.99124174489 ),
  { 4310  378 }  (         0.192, 2.17464236325,   5863.59120611620 ),
  { 4310  379 }  (         0.228, 5.85373115869,    128.01884333740 ),
  { 4310  380 }  (         0.261, 2.64321183295,  55022.93574707440 ),
  { 4310  381 }  (         0.220, 5.75012110079,     29.42950853600 ),
  { 4310  382 }  (         0.187, 4.03230554718,    467.96499035440 ),
  { 4310  383 }  (         0.200, 5.60556112058,   1066.49547719000 ),
  { 4310  384 }  (         0.231, 1.09802712785,  12341.80690428090 ),
  { 4310  385 }  (         0.199, 0.29500625200,    149.56319713460 ),
  { 4310  386 }  (         0.249, 5.10473210814,   7875.67186362420 ),
  { 4310  387 }  (         0.208, 0.93013835019,  14919.01785375460 ),
  { 4310  388 }  (         0.179, 0.87104393079,  12721.57209941700 ),
  { 4310  389 }  (         0.203, 1.56920753653,  28286.99048486120 ),
  { 4310  390 }  (         0.179, 2.47036386443,  16062.18452611680 ),
  { 4310  391 }  (         0.198, 3.54061588502,     30.91412563500 ),
  { 4310  392 }  (         0.171, 3.45356518113,   5327.47610838280 ),
  { 4310  393 }  (         0.183, 0.72325421604,   6272.03014972750 ),
  { 4310  394 }  (         0.216, 2.97174580686,  19402.79695281660 ),
  { 4310  395 }  (         0.168, 2.51550550242,  23937.85638974100 ),
  { 4310  396 }  (         0.195, 0.09045393425,    156.40072050240 ),
  { 4310  397 }  (         0.179, 4.49471798090,  31415.37924995700 ),
  { 4310  398 }  (         0.216, 0.42177594328,  23539.70738633280 ),
  { 4310  399 }  (         0.189, 0.37542530191,   9814.60410029120 ),
  { 4310  400 }  (         0.218, 2.36835880025,  16627.37091537720 ),
  { 4310  401 }  (         0.166, 4.23182968446,  16840.67001081519 ),
  { 4310  402 }  (         0.200, 2.02153258098,  16097.67995028260 ),
  { 4310  403 }  (         0.169, 0.91318727000,     95.97922721780 ),
  { 4310  404 }  (         0.211, 5.73370637657,    151.89728108520 ),
  { 4310  405 }  (         0.204, 0.42643085174,    515.46387109300 ),
  { 4310  406 }  (         0.212, 3.00233538977,  12043.57428188900 ),
  { 4310  407 }  (         0.192, 5.46153589821,   6379.05507720920 ),
  { 4310  408 }  (         0.165, 1.38698167064,   4171.42553661380 ),
  { 4310  409 }  (         0.160, 6.23798383332,    202.25339517410 ),
  { 4310  410 }  (         0.215, 0.20889073407,   5621.84292321040 ),
  { 4310  411 }  (         0.181, 4.12439203622,  13341.67431130680 ),
  { 4310  412 }  (         0.153, 1.24460848836,  29826.30635467320 ),
  { 4310  413 }  (         0.150, 3.12999753018,    799.82112516540 ),
  { 4310  414 }  (         0.175, 4.55671604437, 239424.39025435288 ),
  { 4310  415 }  (         0.192, 1.33928820063,    394.62588505920 ),
  { 4310  416 }  (         0.149, 2.65697593276,     21.33564046700 ),
  { 4310  417 }  (         0.146, 5.58021191726,    412.37109687440 ),
  { 4310  418 }  (         0.156, 3.75650175503,  12323.42309600880 ),
  { 4310  419 }  (         0.143, 3.75708566606,  58864.54391814630 ),
  { 4310  420 }  (         0.143, 3.28248547724,     29.82143814880 ),
  { 4310  421 }  (         0.144, 1.07862546598,   1265.56747862640 ),
  { 4310  422 }  (         0.148, 0.23389236655,  10021.83728009940 ),
  { 4310  423 }  (         0.193, 5.92751083086,  40879.44050464380 ),
  { 4310  424 }  (         0.140, 4.97612440269,    158.94351778320 ),
  { 4310  425 }  (         0.148, 2.61640453469,  17157.06188047180 ),
  { 4310  426 }  (         0.141, 3.66871308723,  26084.02180621620 ),
  { 4310  427 }  (         0.147, 5.09968173403,    661.23292678100 ),
  { 4310  428 }  (         0.146, 4.96885605695,  57375.80190084620 ),
  { 4310  429 }  (         0.142, 0.78678347839,  12779.45079542080 ),
  { 4310  430 }  (         0.134, 4.79432636012,    111.18664228760 ),
  { 4310  431 }  (         0.140, 1.27748013377,    107.66352393860 ),
  { 4310  432 }  (         0.169, 2.74893543762,  26735.94526221320 ),
  { 4310  433 }  (         0.165, 3.95288000638,   6357.85744855870 ),
  { 4310  434 }  (         0.183, 5.43418358741,    369.69981594040 ),
  { 4310  435 }  (         0.134, 3.09132862833,     17.81252211800 ),
  { 4310  436 }  (         0.132, 3.05633896779,  22490.96212149340 ),
  { 4310  437 }  (         0.134, 4.09472795832,   6599.46771964800 ),
  { 4310  438 }  (         0.181, 4.22950689891,    966.97087743560 ),
  { 4310  439 }  (         0.152, 5.28885894415,  12669.24447420140 ),
  { 4310  440 }  (         0.150, 5.86819430908,  97238.62754448749 ),
  { 4310  441 }  (         0.142, 5.87266532526,  22476.73502749179 ),
  { 4310  442 }  (         0.145, 5.07330784304,     87.30820453981 ),
  { 4310  443 }  (         0.133, 5.65471067133,     31.97230581680 ),
  { 4310  444 }  (         0.124, 2.83326217072,  12566.21901028560 ),
  { 4310  445 }  (         0.135, 3.12861731644,  32217.20018108080 ),
  { 4310  446 }  (         0.137, 0.86487461904,   9924.81042151060 ),
  { 4310  447 }  (         0.172, 1.98369595114, 174242.46596404970 ),
  { 4310  448 }  (         0.170, 4.41115280254, 327574.51427678125 ),
  { 4310  449 }  (         0.151, 0.46542099527,  39609.65458316560 ),
  { 4310  450 }  (         0.148, 2.13439571118,    491.66329245880 ),
  { 4310  451 }  (         0.153, 3.78801830344,  17363.24742890899 ),
  { 4310  452 }  (         0.165, 5.31654110459,  16943.76278503380 ),
  { 4310  453 }  (         0.165, 4.06747587817,  58953.14544329400 ),
  { 4310  454 }  (         0.118, 0.63846333239,      6.06591562980 ),
  { 4310  455 }  (         0.159, 0.86086959274, 221995.02880149524 ),
  { 4310  456 }  (         0.119, 5.96432932413,   1385.89527633620 ),
  { 4310  457 }  (         0.114, 5.16516114595,  25685.87280280800 ),
  { 4310  458 }  (         0.112, 3.39403722178,  21393.54196985760 ),
  { 4310  459 }  (         0.112, 4.92889233335,     56.80326216980 ),
  { 4310  460 }  (         0.119, 2.40637635942,  18635.92845453620 ),
  { 4310  461 }  (         0.115, 0.23374479051,    418.92439890060 ),
  { 4310  462 }  (         0.122, 0.93575234049,  24492.40611365159 ),
  { 4310  463 }  (         0.115, 4.58880032176,  26709.64694241340 ),
  { 4310  464 }  (         0.130, 4.85539251000,  22345.26037610820 ),
  { 4310  465 }  (         0.140, 1.09413073202,  44809.65020086340 ),
  { 4310  466 }  (         0.112, 6.05401806281,    433.71173787680 ),
  { 4310  467 }  (         0.104, 1.54931540602,    127.95153303460 ),
  { 4310  468 }  (         0.105, 4.82620858888,  33794.54372352860 ),
  { 4310  469 }  (         0.102, 4.12448497391,  15664.03552270859 ),
  { 4310  470 }  (         0.107, 4.67919356465,  77690.75950573849 ),
  { 4310  471 }  (         0.118, 4.52320170120,  19004.64794940840 ),
  { 4310  472 }  (         0.107, 5.71774478555,  77736.78343050249 ),
  { 4310  473 }  (         0.143, 1.81201813018,   4214.06901508480 ),
  { 4310  474 }  (         0.125, 1.14419195615,    625.67019231240 ),
  { 4310  475 }  (         0.124, 3.27736514057,  12566.08438968000 ),
  { 4310  476 }  (         0.110, 1.08682570828,   2787.04302385740 ),
  { 4310  477 }  (         0.105, 1.78318141871,  18139.29450141590 ),
  { 4310  478 }  (         0.102, 4.75119578149,  12242.64628332540 ),
  { 4310  479 }  (         0.137, 1.43510636754,  86464.61331683119 ),
  { 4310  480 }  (         0.101, 4.91289409429,    401.67212175720 ),
  { 4310  481 }  (         0.129, 1.23567904485,  12029.34718788740 ),
  { 4310  482 }  (         0.138, 2.45654707999,   7576.56007357400 ),
  { 4310  483 }  (         0.103, 0.40004073416,  90279.92316810328 ),
  { 4310  484 }  (         0.108, 0.98989774940,   5636.06501667660 ),
  { 4310  485 }  (         0.117, 5.17362872063,  34520.30930938080 ),
  { 4310  486 }  (         0.100, 3.95534628189,   5547.19933645960 ),
  { 4310  487 }  (         0.098, 1.28118280598,  21548.96236929180 ),
  { 4310  488 }  (         0.097, 3.34717130592,  16310.97904572060 ),
  { 4310  489 }  (         0.098, 4.37041908717,  34513.26307268280 ),
  { 4310  490 }  (         0.125, 2.72164432960,  24065.80792277559 ),
  { 4310  491 }  (         0.102, 0.66938025772,  10239.58386601080 ),
  { 4310  492 }  (         0.119, 1.21689479331,   1478.86657406440 ),
  { 4310  493 }  (         0.094, 1.99595224256,  13362.44970679920 ),
  { 4310  494 }  (         0.094, 4.30965982872,  26880.31981303260 ),
  { 4310  495 }  (         0.095, 2.89807657534,  34911.41207609100 ),
  { 4310  496 }  (         0.106, 1.00156653590,  16522.65971600220 ),
  { 4310  497 }  (         0.097, 0.89642320201,  71980.63357473118 ),
  { 4310  498 }  (         0.116, 4.19967201116,    206.70073729660 ),
  { 4310  499 }  (         0.099, 1.37437847718,   1039.02661079040 ),
  { 4310  500 }  (         0.126, 3.21642544972, 305281.94307104882 ),
  { 4310  501 }  (         0.094, 0.68997876060,   7834.12107263940 ),
  { 4310  502 }  (         0.094, 5.58132218606,   3104.93005942380 ),
  { 4310  503 }  (         0.095, 3.03823741110,   8982.81066930900 ),
  { 4310  504 }  (         0.108, 0.52696637156,    276.74577186440 ),
  { 4310  505 }  (         0.124, 3.43899862683, 172146.97134054029 ),
  { 4310  506 }  (         0.102, 1.04031728553,  95143.13292097810 ),
  { 4310  507 }  (         0.104, 3.39218586218,    290.97286586600 ),
  { 4310  508 }  (         0.110, 3.68205877433,  22380.75580027400 ),
  { 4310  509 }  (         0.117, 0.78475956902,  83286.91426955358 ),
  { 4310  510 }  (         0.083, 0.18241793425,  15141.39079431200 ),
  { 4310  511 }  (         0.089, 4.45371820659,    792.77488846740 ),
  { 4310  512 }  (         0.082, 4.80703651241,   6819.88036208680 ),
  { 4310  513 }  (         0.087, 3.43122851097,  27707.54249429480 ),
  { 4310  514 }  (         0.101, 5.32081603011,   2301.58581590939 ),
  { 4310  515 }  (         0.082, 0.87060089842,  10241.20229116720 ),
  { 4310  516 }  (         0.086, 4.61919461931,  36147.40987730040 ),
  { 4310  517 }  (         0.095, 2.87032884659,  23020.65308658799 ),
  { 4310  518 }  (         0.088, 3.21133165690,  33326.57873317420 ),
  { 4310  519 }  (         0.080, 1.84900424847,  21424.46664430340 ),
  { 4310  520 }  (         0.101, 4.18796434479,  30666.15495843280 ),
  { 4310  521 }  (         0.107, 5.77864921649,  34115.11406927460 ),
  { 4310  522 }  (         0.104, 1.08739495962,   6288.59877429880 ),
  { 4310  523 }  (         0.110, 3.32898859416,  72140.62866668739 ),
  { 4310  524 }  (         0.087, 4.40657711727,    142.17862703620 ),
  { 4310  525 }  (         0.109, 1.94546030825,  24279.10701821359 ),
  { 4310  526 }  (         0.087, 4.32472045435,    742.99006053260 ),
  { 4310  527 }  (         0.107, 4.91580912547,    277.03499374140 ),
  { 4310  528 }  (         0.088, 2.10180220766,  26482.17080962440 ),
  { 4310  529 }  (         0.086, 4.01887374432,  12491.37010141550 ),
  { 4310  530 }  (         0.106, 5.49092372854,  62883.35513951360 ),
  { 4310  531 }  (         0.080, 6.19781316983,   6709.67404086740 ),
  { 4310  532 }  (         0.088, 2.09872810657, 238004.52415723629 ),
  { 4310  533 }  (         0.083, 4.90662164029,     51.28033786241 ),
  { 4310  534 }  (         0.095, 4.13387406591,  18216.44381066100 ),
  { 4310  535 }  (         0.078, 6.06949391680, 148434.53403769129 ),
  { 4310  536 }  (         0.079, 3.03048221644,    838.96928775040 ),
  { 4310  537 }  (         0.074, 5.49813051211,  29026.48522950779 ),
  { 4310  538 }  (         0.073, 3.05008665738,    567.71863773040 ),
  { 4310  539 }  (         0.084, 0.46604373274,     45.14121963660 ),
  { 4310  540 }  (         0.093, 2.52267536308,  48739.85989708300 ),
  { 4310  541 }  (         0.076, 1.76418124905,  41654.96311596780 ),
  { 4310  542 }  (         0.067, 5.77851227793,   6311.52503745920 ),
  { 4310  543 }  (         0.062, 3.32967880172,  15508.61512327440 ),
  { 4310  544 }  (         0.079, 5.59773841328,  71960.38658322369 ),
  { 4310  545 }  (         0.057, 3.90629505268,   5999.21653112620 ),
  { 4310  546 }  (         0.061, 0.05695043232,   7856.89627409019 ),
  { 4310  547 }  (         0.061, 5.63297958433,   7863.94251078820 ),
  { 4310  548 }  (         0.065, 3.72178394016,  12573.26524698360 ),
  { 4310  549 }  (         0.057, 4.18217219541,  26087.90314157420 ),
  { 4310  550 }  (         0.066, 3.92262333487,  69853.35207568129 ),
  { 4310  551 }  (         0.053, 5.51119362045,  77710.24834977149 ),
  { 4310  552 }  (         0.053, 4.88573986961,  77717.29458646949 ),
  { 4310  553 }  (         0.062, 2.88876342225,   9411.46461508720 ),
  { 4310  554 }  (         0.051, 1.12657183874,  82576.98122099529 ),
  { 4310  555 }  (         0.045, 2.95671076719,  24602.61243487099 ),
  { 4310  556 }  (         0.040, 5.55145719241,  12565.17137891460 ),
  { 4310  557 }  (         0.039, 1.20838190039,  18842.11400297339 ),
  { 4310  558 }  (         0.045, 3.18590558749,  45585.17281218740 ),
  { 4310  559 }  (         0.049, 2.44790934886,  13613.80427733600 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_ear_l1:array[0..340,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_l1:array[0.. 33,0..2] of extended = (
  (*$else *)
  vsop87_ear_l1:array[0..340,0..2] of extended = (
  (*$endif *)
  { 4311    1 }(628331966747.491, 0.00000000000,      0.00000000000 ),
  { 4311    2 }  (    206058.863, 2.67823455584,   6283.07584999140 ),
  { 4311    3 }  (      4303.430, 2.63512650414,  12566.15169998280 ),
  { 4311    4 }  (       425.264, 1.59046980729,      3.52311834900 ),
  { 4311    5 }  (       108.977, 2.96618001993,   1577.34354244780 ),
  { 4311    6 }  (       093.478, 2.59212835365,  18849.22754997420 ),
  { 4311    7 }  (       119.261, 5.79557487799,     26.29831979980 ),
  { 4311    8 }  (        72.122, 1.13846158196,    529.69096509460 ),
  { 4311    9 }  (        67.768, 1.87472304791,    398.14900340820 ),
  { 4311   10 }  (        67.327, 4.40918235168,   5507.55323866740 ),
  { 4311   11 }  (        59.027, 2.88797038460,   5223.69391980220 ),
  { 4311   12 }  (        55.976, 2.17471680261,    155.42039943420 ),
  { 4311   13 }  (        45.407, 0.39803079805,    796.29800681640 ),
  { 4311   14 }  (        36.369, 0.46624739835,    775.52261132400 ),
  { 4311   15 }  (        28.958, 2.64707383882,      7.11354700080 ),
  { 4311   16 }  (        19.097, 1.84628332577,   5486.77784317500 ),
  { 4311   17 }  (        20.844, 5.34138275149,      0.98032106820 ),
  { 4311   18 }  (        18.508, 4.96855124577,    213.29909543800 ),
  { 4311   19 }  (        16.233, 0.03216483047,   2544.31441988340 ),
  { 4311   20 }  (        17.293, 2.99116864949,   6275.96230299060 ),
  { 4311   21 }  (        15.832, 1.43049285325,   2146.16541647520 ),
  { 4311   22 }  (        14.615, 1.20532366323,  10977.07880469900 ),
  { 4311   23 }  (        11.877, 3.25804815607,   5088.62883976680 ),
  { 4311   24 }  (        11.514, 2.07502418155,   4694.00295470760 ),
  { 4311   25 }  (         9.721, 4.23925472239,   1349.86740965880 ),
  { 4311   26 }  (         9.969, 1.30262991097,   6286.59896834040 ),
  { 4311   27 }  (         9.452, 2.69957062864,    242.72860397400 ),
  { 4311   28 }  (        12.461, 2.83432285512,   1748.01641306700 ),
  { 4311   29 }  (        11.808, 5.27379790480,   1194.44701022460 ),
  { 4311   30 }  (         8.577, 5.64475868067,    951.71840625060 ),
  { 4311   31 }  (        10.641, 0.76614199202,    553.56940284240 ),
  { 4311   32 }  (         7.576, 5.30062664886,   2352.86615377180 ),
  (*$ifndef meeus *)
  { 4311   33 }  (         5.834, 1.76649917904,   1059.38193018920 ),
  (*$endif *)
  { 4311   34 }  (         6.385, 2.65033984967,   9437.76293488700 ),
  (*$ifndef meeus *)
  { 4311   35 }  (         5.223, 5.66135767624,  71430.69561812909 ),
  { 4311   36 }  (         5.305, 0.90857521574,   3154.68708489560 ),
  (*$endif *)
  { 4311   37 }  (         6.101, 4.66632584188,   4690.47983635860 )
  (*$ifndef meeus *)                                                 ,
  { 4311   38 }  (         4.330, 0.24102555403,   6812.76681508600 ),
  { 4311   39 }  (         5.041, 1.42490103709,   6438.49624942560 ),
  { 4311   40 }  (         4.259, 0.77355900599,  10447.38783960440 ),
  { 4311   41 }  (         5.198, 1.85353197345,    801.82093112380 ),
  { 4311   42 }  (         3.744, 2.00119516488,   8031.09226305840 ),
  { 4311   43 }  (         3.558, 2.42901552681,  14143.49524243060 ),
  { 4311   44 }  (         3.372, 3.86210700128,   1592.59601363280 ),
  { 4311   45 }  (         3.374, 0.88776219727,  12036.46073488820 ),
  { 4311   46 }  (         3.175, 3.18785710594,   4705.73230754360 ),
  { 4311   47 }  (         3.221, 0.61599835472,   8429.24126646660 ),
  { 4311   48 }  (         4.132, 5.23992859705,   7084.89678111520 ),
  { 4311   49 }  (         2.970, 6.07026318493,   4292.33083295040 ),
  { 4311   50 }  (         2.900, 2.32464208411,     20.35531939880 ),
  { 4311   51 }  (         3.504, 4.79975694359,   6279.55273164240 ),
  { 4311   52 }  (         2.950, 1.43108874817,   5746.27133789600 ),
  { 4311   53 }  (         2.697, 4.80368225199,   7234.79425624200 ),
  { 4311   54 }  (         2.531, 6.22290682655,   6836.64525283380 ),
  { 4311   55 }  (         2.745, 0.93466065396,   5760.49843189760 ),
  { 4311   56 }  (         3.250, 3.39954640038,   7632.94325965020 ),
  { 4311   57 }  (         2.277, 5.00277837672,  17789.84561978500 ),
  { 4311   58 }  (         2.075, 3.95534978634,  10213.28554621100 ),
  { 4311   59 }  (         2.061, 2.22411683077,   5856.47765911540 ),
  { 4311   60 }  (         2.252, 5.67166499885,  11499.65622279280 ),
  { 4311   61 }  (         2.148, 5.20184578235,  11513.88331679440 ),
  { 4311   62 }  (         1.886, 0.53198320577,   3340.61242669980 ),
  { 4311   63 }  (         1.875, 4.73511970207,  83996.84731811189 ),
  { 4311   64 }  (         2.060, 2.54987293999,  25132.30339996560 ),
  { 4311   65 }  (         1.794, 1.47435409831,   4164.31198961300 ),
  { 4311   66 }  (         1.778, 3.02473091781,      5.52292430740 ),
  { 4311   67 }  (         2.029, 0.90960209983,   6256.77753019160 ),
  { 4311   68 }  (         2.075, 2.26767270157,    522.57741809380 ),
  { 4311   69 }  (         1.772, 3.02622802353,   5753.38488489680 ),
  { 4311   70 }  (         1.569, 6.12410242782,   5216.58037280140 ),
  { 4311   71 }  (         1.590, 4.63713748247,      3.28635741780 ),
  { 4311   72 }  (         1.542, 4.20004448567,  13367.97263110660 ),
  { 4311   73 }  (         1.427, 1.19088061711,   3894.18182954220 ),
  { 4311   74 }  (         1.375, 3.09301252193,    135.06508003540 ),
  { 4311   75 }  (         1.359, 4.24532506641,    426.59819087600 ),
  { 4311   76 }  (         1.340, 5.76511818622,   6040.34724601740 ),
  { 4311   77 }  (         1.284, 3.08524663344,   5643.17856367740 ),
  { 4311   78 }  (         1.250, 3.07748157144,  11926.25441366880 ),
  { 4311   79 }  (         1.551, 3.07665451458,   6681.22485339960 ),
  { 4311   80 }  (         1.268, 2.09196018331,   6290.18939699220 ),
  { 4311   81 }  (         1.144, 3.24444699514,  12168.00269657460 ),
  { 4311   82 }  (         1.248, 3.44504937285,    536.80451209540 ),
  { 4311   83 }  (         1.118, 2.31829670425,  16730.46368959580 ),
  { 4311   84 }  (         1.105, 5.31966001019,     23.87843774780 ),
  { 4311   85 }  (         1.051, 3.75015946014,   7860.41939243920 ),
  { 4311   86 }  (         1.025, 2.44688534235,   1990.74501704100 ),
  { 4311   87 }  (         0.962, 0.81771017882,      3.88133535800 ),
  { 4311   88 }  (         0.910, 0.41727865299,   7079.37385680780 ),
  { 4311   89 }  (         0.883, 5.16833917651,  11790.62908865880 ),
  { 4311   90 }  (         0.957, 4.07673573735,   6127.65545055720 ),
  { 4311   91 }  (         1.110, 3.90096793825,  11506.76976979360 ),
  { 4311   92 }  (         0.802, 3.88778875582,  10973.55568635000 ),
  { 4311   93 }  (         0.780, 2.39934293755,   1589.07289528380 ),
  { 4311   94 }  (         0.758, 1.30034364248,    103.09277421860 ),
  { 4311   95 }  (         0.749, 4.96275803300,   6496.37494542940 ),
  { 4311   96 }  (         0.765, 3.36312388424,     36.02786667740 ),
  { 4311   97 }  (         0.915, 5.41543742089,    206.18554843720 ),
  { 4311   98 }  (         0.776, 2.57589093871,  11371.70468975820 ),
  { 4311   99 }  (         0.772, 3.98369209464,    955.59974160860 ),
  { 4311  100 }  (         0.749, 5.17890001805,  10969.96525769820 ),
  { 4311  101 }  (         0.806, 0.34218864254,   9917.69687450980 ),
  { 4311  102 }  (         0.728, 5.20962563787,     38.02767263580 ),
  { 4311  103 }  (         0.685, 2.77592961854,     20.77539549240 ),
  { 4311  104 }  (         0.636, 4.28242193632,     28.44918746780 ),
  { 4311  105 }  (         0.608, 5.63278508906,  10984.19235169980 ),
  { 4311  106 }  (         0.704, 5.60738823665,   3738.76143010800 ),
  { 4311  107 }  (         0.685, 0.38876148682,     15.25247118500 ),
  { 4311  108 }  (         0.601, 0.73489602442,    419.48464387520 ),
  { 4311  109 }  (         0.716, 2.65279791438,   6309.37416979120 ),
  { 4311  110 }  (         0.584, 5.54502568227,  17298.18232732620 ),
  { 4311  111 }  (         0.650, 1.13379656406,   7058.59846131540 ),
  { 4311  112 }  (         0.688, 2.59683891779,   3496.03282613400 ),
  { 4311  113 }  (         0.485, 0.44467180946,  12352.85260454480 ),
  { 4311  114 }  (         0.528, 2.74936967681,   3930.20969621960 ),
  { 4311  115 }  (         0.597, 5.27668281777,  10575.40668294180 ),
  { 4311  116 }  (         0.583, 3.18929067810,   4732.03062734340 ),
  { 4311  117 }  (         0.526, 5.01697321546,   5884.92684658320 ),
  { 4311  118 }  (         0.540, 1.29175137075,    640.87760738220 ),
  { 4311  119 }  (         0.473, 5.49953306970,   5230.80746680300 ),
  { 4311  120 }  (         0.406, 5.21248452189,    220.41264243880 ),
  { 4311  121 }  (         0.395, 1.87474483222,  16200.77272450120 ),
  { 4311  122 }  (         0.370, 3.84921354713,  18073.70493865020 ),
  { 4311  123 }  (         0.367, 0.88533542778,   6283.14316029419 ),
  { 4311  124 }  (         0.379, 0.37983009325,  10177.25767953360 ),
  { 4311  125 }  (         0.356, 3.84145204913,  11712.95531823080 ),
  { 4311  126 }  (         0.374, 5.01577520608,      7.04623669800 ),
  { 4311  127 }  (         0.381, 4.30250406634,   6062.66320755260 ),
  { 4311  128 }  (         0.471, 0.86381834647,   6069.77675455340 ),
  { 4311  129 }  (         0.367, 1.32943839763,   6283.00853968860 ),
  { 4311  130 }  (         0.460, 5.19667219575,   6284.05617105960 ),
  { 4311  131 }  (         0.333, 5.54256205741,   4686.88940770680 ),
  { 4311  132 }  (         0.341, 4.36522989934,   7238.67559160000 ),
  { 4311  133 }  (         0.336, 4.00205876835,   3097.88382272579 ),
  { 4311  134 }  (         0.359, 6.22679790284,    245.83164622940 ),
  { 4311  135 }  (         0.307, 2.35299010924,    170.67287061920 ),
  { 4311  136 }  (         0.343, 3.77164927143,   6076.89030155420 ),
  { 4311  137 }  (         0.296, 5.44152227481,  17260.15465469040 ),
  { 4311  138 }  (         0.328, 0.13837875384,  11015.10647733480 ),
  { 4311  139 }  (         0.268, 1.13904550630,  12569.67481833180 ),
  { 4311  140 }  (         0.263, 0.00538633678,   4136.91043351620 ),
  { 4311  141 }  (         0.282, 5.04399837480,   7477.52286021600 ),
  { 4311  142 }  (         0.288, 3.13401177517,  12559.03815298200 ),
  { 4311  143 }  (         0.259, 0.93882269387,   5642.19824260920 ),
  { 4311  144 }  (         0.292, 1.98420020514,  12132.43996210600 ),
  { 4311  145 }  (         0.247, 3.84244798532,   5429.87946823940 ),
  { 4311  146 }  (         0.245, 5.70467521726,  65147.61976813770 ),
  { 4311  147 }  (         0.241, 0.99480969552,   3634.62102451840 ),
  { 4311  148 }  (         0.246, 3.06168069935,    110.20632121940 ),
  { 4311  149 }  (         0.239, 6.11855909114,  11856.21865142450 ),
  { 4311  150 }  (         0.263, 0.66348415419,  21228.39202354580 ),
  { 4311  151 }  (         0.262, 1.51070507866,  12146.66705610760 ),
  { 4311  152 }  (         0.230, 1.75927314884,   9779.10867612540 ),
  { 4311  153 }  (         0.223, 2.00967043606,   6172.86952877200 ),
  { 4311  154 }  (         0.246, 1.10411690865,   6282.09552892320 ),
  { 4311  155 }  (         0.221, 3.03945240854,   8635.94200376320 ),
  { 4311  156 }  (         0.214, 4.03840869663,  14314.16811304980 ),
  { 4311  157 }  (         0.236, 5.46915070580,  13916.01910964160 ),
  { 4311  158 }  (         0.224, 4.68408089456,  24072.92146977640 ),
  { 4311  159 }  (         0.212, 2.13695625494,   5849.36411211460 ),
  { 4311  160 }  (         0.207, 3.07724246401,     11.72935283600 ),
  { 4311  161 }  (         0.207, 6.10306282747,  23543.23050468179 ),
  { 4311  162 }  (         0.266, 1.00709566823,   2388.89402044920 ),
  { 4311  163 }  (         0.217, 6.27837036335,  17267.26820169119 ),
  { 4311  164 }  (         0.204, 2.34615348695,    266.60704172180 ),
  { 4311  165 }  (         0.195, 5.55015549753,   6133.51265285680 ),
  { 4311  166 }  (         0.188, 2.52667166175,   6525.80445396540 ),
  { 4311  167 }  (         0.185, 0.90960768344,  18319.53658487960 ),
  { 4311  168 }  (         0.177, 1.73429218289, 154717.60988768269 ),
  { 4311  169 }  (         0.187, 4.76483647432,   4535.05943692440 ),
  { 4311  170 }  (         0.186, 4.63080493407,  10440.27429260360 ),
  { 4311  171 }  (         0.215, 2.81255454560,   7342.45778018060 ),
  { 4311  172 }  (         0.172, 1.45551888559,   9225.53927328300 ),
  { 4311  173 }  (         0.162, 3.30661909388,    639.89728631400 ),
  { 4311  174 }  (         0.168, 2.17671416605,     27.40155609680 ),
  { 4311  175 }  (         0.160, 1.68164180475,  15110.46611986620 ),
  { 4311  176 }  (         0.158, 0.13519771874,  13095.84266507740 ),
  { 4311  177 }  (         0.183, 0.56281322071,  13517.87010623340 ),
  { 4311  178 }  (         0.179, 3.58450811616,     87.30820453981 ),
  { 4311  179 }  (         0.152, 2.84070476818,   5650.29211067820 ),
  { 4311  180 }  (         0.182, 0.44065530624,  17253.04110768959 ),
  { 4311  181 }  (         0.160, 5.95767264171,   4701.11650170840 ),
  { 4311  182 }  (         0.142, 1.46290137520,  11087.28512591840 ),
  { 4311  183 }  (         0.142, 2.04464036087,  20426.57109242200 ),
  { 4311  184 }  (         0.131, 5.40912137746,   2699.73481931760 ),
  { 4311  185 }  (         0.144, 2.07312090485,  25158.60171976540 ),
  { 4311  186 }  (         0.147, 6.15106982168,   9623.68827669120 ),
  { 4311  187 }  (         0.141, 5.55739979498,  10454.50138660520 ),
  { 4311  188 }  (         0.135, 0.06098110407,  16723.35014259500 ),
  { 4311  189 }  (         0.124, 5.81218025669,  17256.63153634140 ),
  { 4311  190 }  (         0.124, 2.36293551623,   4933.20844033260 ),
  { 4311  191 }  (         0.126, 3.47435905118,  22483.84857449259 ),
  { 4311  192 }  (         0.159, 5.63954754618,   5729.50644714900 ),
  { 4311  193 }  (         0.123, 3.92815963256,  17996.03116822220 ),
  { 4311  194 }  (         0.148, 3.02509280598,   1551.04522264800 ),
  { 4311  195 }  (         0.120, 5.91904349732,   6206.80977871580 ),
  { 4311  196 }  (         0.134, 3.11122937825,  21954.15760939799 ),
  { 4311  197 }  (         0.119, 5.52141123450,    709.93304855830 ),
  { 4311  198 }  (         0.122, 3.00813429479,  19800.94595622480 ),
  { 4311  199 }  (         0.127, 1.37618620001,  14945.31617355440 ),
  { 4311  200 }  (         0.141, 2.56889468729,   1052.26838318840 ),
  { 4311  201 }  (         0.123, 2.83671175442,  11919.14086666800 ),
  { 4311  202 }  (         0.118, 0.81934438215,   5331.35744374080 ),
  { 4311  203 }  (         0.151, 2.68731829165,  11769.85369316640 ),
  { 4311  204 }  (         0.119, 5.08835797638,   5481.25491886760 ),
  { 4311  205 }  (         0.153, 2.46021790779,  11933.36796066960 ),
  { 4311  206 }  (         0.108, 1.04936452145,  11403.67699557500 ),
  { 4311  207 }  (         0.128, 0.99794735107,   8827.39026987480 ),
  { 4311  208 }  (         0.144, 2.54869747042,    227.47613278900 ),
  { 4311  209 }  (         0.150, 4.50631437136,   2379.16447357160 ),
  { 4311  210 }  (         0.107, 1.79272017026,  13119.72110282519 ),
  { 4311  211 }  (         0.107, 4.43556814486,  18422.62935909819 ),
  { 4311  212 }  (         0.109, 0.29269062317,  16737.57723659660 ),
  { 4311  213 }  (         0.141, 3.18979826258,   6262.30045449900 ),
  { 4311  214 }  (         0.122, 4.23040027813,     29.42950853600 ),
  { 4311  215 }  (         0.111, 5.16954029551,  17782.73207278420 ),
  { 4311  216 }  (         0.100, 3.52213872761,  18052.92954315780 ),
  { 4311  217 }  (         0.108, 1.08514212991,  16858.48253293320 ),
  { 4311  218 }  (         0.106, 1.96085248410,     74.78159856730 ),
  { 4311  219 }  (         0.110, 2.30582372873,  16460.33352952499 ),
  { 4311  220 }  (         0.097, 3.50918940210,   5333.90024102160 ),
  { 4311  221 }  (         0.099, 3.56417337974,    735.87651353180 ),
  { 4311  222 }  (         0.094, 5.01857894228,   3128.38876509580 ),
  { 4311  223 }  (         0.097, 1.65579893894,    533.21408344360 ),
  { 4311  224 }  (         0.092, 0.89217162285,  29296.61538957860 ),
  { 4311  225 }  (         0.123, 3.16062050433,   9380.95967271720 ),
  { 4311  226 }  (         0.102, 1.20493500565,  23020.65308658799 ),
  { 4311  227 }  (         0.088, 2.21296088224,  12721.57209941700 ),
  { 4311  228 }  (         0.089, 1.54264720310,  20199.09495963300 ),
  { 4311  229 }  (         0.113, 4.83320707870,  16496.36139620240 ),
  { 4311  230 }  (         0.121, 6.19860353182,   9388.00590941520 ),
  { 4311  231 }  (         0.089, 4.08082274765,  22805.73556599360 ),
  { 4311  232 }  (         0.098, 1.09181832830,  12043.57428188900 ),
  { 4311  233 }  (         0.086, 1.13655027605, 143571.32428481648 ),
  { 4311  234 }  (         0.088, 5.96980472191,    107.66352393860 ),
  { 4311  235 }  (         0.082, 5.01340404594,  22003.91463486980 ),
  { 4311  236 }  (         0.094, 1.69615700473,  23006.42599258639 ),
  { 4311  237 }  (         0.081, 3.00657814365,   2118.76386037840 ),
  { 4311  238 }  (         0.098, 1.39215287161,   8662.24032356300 ),
  { 4311  239 }  (         0.077, 3.33555190840,  15720.83878487840 ),
  { 4311  240 }  (         0.082, 5.86880116464,   2787.04302385740 ),
  { 4311  241 }  (         0.076, 5.67183650604,     14.22709400160 ),
  { 4311  242 }  (         0.081, 6.16619455699,   1039.02661079040 ),
  { 4311  243 }  (         0.076, 3.21449884756,    111.18664228760 ),
  { 4311  244 }  (         0.078, 1.37531518377,  21947.11137270000 ),
  { 4311  245 }  (         0.074, 3.58814195051,  11609.86254401220 ),
  { 4311  246 }  (         0.077, 4.84846488388,  22743.40937951640 ),
  { 4311  247 }  (         0.090, 1.48869013606,  15671.08175940660 ),
  { 4311  248 }  (         0.082, 3.48618399109,  29088.81141598500 ),
  { 4311  249 }  (         0.069, 3.55746476593,   4590.91018048900 ),
  { 4311  250 }  (         0.069, 1.93625656075,    135.62532501000 ),
  { 4311  251 }  (         0.070, 2.66548322237,  18875.52586977400 ),
  { 4311  252 }  (         0.069, 5.41478093731,  26735.94526221320 ),
  { 4311  253 }  (         0.079, 5.15154513662,  12323.42309600880 ),
  { 4311  254 }  (         0.094, 3.62899392448,  77713.77146812050 ),
  { 4311  255 }  (         0.078, 4.17011182047,   1066.49547719000 ),
  { 4311  256 }  (         0.071, 3.89435637865,  22779.43724619380 ),
  { 4311  257 }  (         0.063, 4.53968787714,   8982.81066930900 ),
  { 4311  258 }  (         0.069, 0.96028230548,  14919.01785375460 ),
  { 4311  259 }  (         0.076, 3.29092216589,   2942.46342329160 ),
  { 4311  260 }  (         0.063, 4.09167842893,  16062.18452611680 ),
  { 4311  261 }  (         0.065, 3.34580407184,     51.28033786241 ),
  { 4311  262 }  (         0.065, 5.75757544877,  52670.06959330260 ),
  { 4311  263 }  (         0.068, 5.75884067555,  21424.46664430340 ),
  { 4311  264 }  (         0.057, 5.45122399850,  12592.45001978260 ),
  { 4311  265 }  (         0.057, 5.25043362558,  20995.39296644940 ),
  { 4311  266 }  (         0.073, 0.53299090807,   2301.58581590939 ),
  { 4311  267 }  (         0.070, 4.31243357502,  19402.79695281660 ),
  { 4311  268 }  (         0.067, 2.53852336668,    377.37360791580 ),
  { 4311  269 }  (         0.056, 3.20816844695,  24889.57479599160 ),
  { 4311  270 }  (         0.053, 3.17816599142,  18451.07854656599 ),
  { 4311  271 }  (         0.053, 3.61529270216,     77.67377042800 ),
  { 4311  272 }  (         0.053, 0.45467549335,  30666.15495843280 ),
  { 4311  273 }  (         0.061, 0.14807288453,  23013.53953958720 ),
  { 4311  274 }  (         0.051, 3.32803972907,     56.89837493560 ),
  { 4311  275 }  (         0.052, 3.41177624177,  23141.55838292460 ),
  { 4311  276 }  (         0.058, 3.13638677202,    309.27832265580 ),
  { 4311  277 }  (         0.070, 2.50592323465,  31415.37924995700 ),
  { 4311  278 }  (         0.052, 5.10673376738,  17796.95916678580 ),
  { 4311  279 }  (         0.067, 6.27917920454,  22345.26037610820 ),
  { 4311  280 }  (         0.050, 0.42577644151,  25685.87280280800 ),
  { 4311  281 }  (         0.048, 0.70204553333,   1162.47470440780 ),
  { 4311  282 }  (         0.066, 3.64350022359,  15265.88651930040 ),
  { 4311  283 }  (         0.050, 5.74382917440,     19.66976089979 ),
  { 4311  284 }  (         0.050, 4.69825387775,  28237.23345938940 ),
  { 4311  285 }  (         0.047, 5.74015846442,  12139.55350910680 ),
  { 4311  286 }  (         0.054, 1.97301333704,  23581.25817731760 ),
  { 4311  287 }  (         0.049, 4.98223579027,  10021.83728009940 ),
  { 4311  288 }  (         0.046, 5.41431705539,  33019.02111220460 ),
  { 4311  289 }  (         0.051, 1.23882053879,  12539.85338018300 ),
  { 4311  290 }  (         0.046, 2.41369976086,  98068.53671630539 ),
  { 4311  291 }  (         0.044, 0.80750593746, 167283.76158766549 ),
  { 4311  292 }  (         0.045, 4.39613584445,    433.71173787680 ),
  { 4311  293 }  (         0.044, 2.57358208785,  12964.30070339100 ),
  { 4311  294 }  (         0.046, 0.26142733448,     11.04570026390 ),
  { 4311  295 }  (         0.045, 2.46230645202,  51868.24866217880 ),
  { 4311  296 }  (         0.048, 0.89551707131,  56600.27928952220 ),
  { 4311  297 }  (         0.057, 1.86416707010,  25287.72379939980 ),
  { 4311  298 }  (         0.042, 5.26377513431,  26084.02180621620 ),
  { 4311  299 }  (         0.049, 3.17757670611,   6303.85124548380 ),
  { 4311  300 }  (         0.052, 3.65266055509,   7872.14874527520 ),
  { 4311  301 }  (         0.040, 1.81891629936,  34596.36465465240 ),
  { 4311  302 }  (         0.043, 1.94164978061,   1903.43681250120 ),
  { 4311  303 }  (         0.041, 0.74461854136,  23937.85638974100 ),
  { 4311  304 }  (         0.048, 6.26034008181,  28286.99048486120 ),
  { 4311  305 }  (         0.045, 5.45575017530,  60530.48898574180 ),
  { 4311  306 }  (         0.040, 2.92105728682,  21548.96236929180 ),
  { 4311  307 }  (         0.040, 0.04502010161,  38526.57435087200 ),
  { 4311  308 }  (         0.053, 3.64791042082,  11925.27409260060 ),
  { 4311  309 }  (         0.041, 5.04048954693,  27832.03821928320 ),
  { 4311  310 }  (         0.042, 5.19292937193,  19004.64794940840 ),
  { 4311  311 }  (         0.040, 2.57120233428,  24356.78078864160 ),
  { 4311  312 }  (         0.038, 3.49190341464, 226858.23855437008 ),
  { 4311  313 }  (         0.039, 4.61184303844,     95.97922721780 ),
  { 4311  314 }  (         0.043, 2.20648228147,  13521.75144159140 ),
  { 4311  315 }  (         0.040, 5.83461945819,  16193.65917750039 ),
  { 4311  316 }  (         0.045, 3.73714372195,   7875.67186362420 ),
  { 4311  317 }  (         0.043, 1.14078465002,     49.75702547180 ),
  { 4311  318 }  (         0.037, 1.29390383811,    310.84079886840 ),
  { 4311  319 }  (         0.038, 0.95970925950,    664.75604513000 ),
  { 4311  320 }  (         0.037, 4.27532649462,   6709.67404086740 ),
  { 4311  321 }  (         0.038, 2.20108541046,  28628.33622609960 ),
  { 4311  322 }  (         0.039, 0.85957361635,  16522.65971600220 ),
  { 4311  323 }  (         0.040, 4.35214003837,  48739.85989708300 ),
  { 4311  324 }  (         0.036, 1.68167662194,  10344.29506538580 ),
  { 4311  325 }  (         0.040, 5.13217319067,  15664.03552270859 ),
  { 4311  326 }  (         0.036, 3.72187132496,  30774.50164257480 ),
  { 4311  327 }  (         0.036, 3.32158458257,  16207.88627150200 ),
  { 4311  328 }  (         0.045, 3.94202418608,  10988.80815753500 ),
  { 4311  329 }  (         0.039, 1.51948786199,  12029.34718788740 ),
  { 4311  330 }  (         0.026, 3.87685883180,   6262.72053059260 ),
  { 4311  331 }  (         0.024, 4.91804163466,  19651.04848109800 ),
  { 4311  332 }  (         0.023, 0.29300197709,  13362.44970679920 ),
  { 4311  333 }  (         0.021, 3.18605672363,   6277.55292568400 ),
  { 4311  334 }  (         0.021, 6.07546891132,  18139.29450141590 ),
  { 4311  335 }  (         0.022, 2.31199937177,   6303.43116939020 ),
  { 4311  336 }  (         0.021, 3.58418394393,  18209.33026366019 ),
  { 4311  337 }  (         0.026, 2.06801296900,  12573.26524698360 ),
  { 4311  338 }  (         0.021, 1.56857722317,  13341.67431130680 ),
  { 4311  339 }  (         0.024, 5.72605158675,  29864.33402730900 ),
  { 4311  340 }  (         0.024, 1.40237993205,  14712.31711645800 ),
  { 4311  341 }  (         0.025, 5.71466092822,  25934.12433108940 )
  (*$endif *)
      );
  (*@\\\0000000601*)
  (*@/// vsop87_ear_l2:array[0..141,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_l2:array[0.. 19,0..2] of extended = (
  (*$else *)
  vsop87_ear_l2:array[0..141,0..2] of extended = (
  (*$endif *)
  { 4312    1 }  (     52918.870, 0.00000000000,      0.00000000000 ),
  { 4312    2 }  (      8719.837, 1.07209665242,   6283.07584999140 ),
  { 4312    3 }  (       309.125, 0.86728818832,  12566.15169998280 ),
  { 4312    4 }  (        27.339, 0.05297871691,      3.52311834900 ),
  { 4312    5 }  (        16.334, 5.18826691036,     26.29831979980 ),
  { 4312    6 }  (        15.752, 3.68457889430,    155.42039943420 ),
  { 4312    7 }  (         9.541, 0.75742297675,  18849.22754997420 ),
  { 4312    8 }  (         8.937, 2.05705419118,  77713.77146812050 ),
  { 4312    9 }  (         6.952, 0.82673305410,    775.52261132400 ),
  { 4312   10 }  (         5.064, 4.66284525271,   1577.34354244780 ),
  { 4312   11 }  (         4.061, 1.03057162962,      7.11354700080 ),
  { 4312   12 }  (         3.463, 5.14074632811,    796.29800681640 ),
  { 4312   13 }  (         3.169, 6.05291851171,   5507.55323866740 ),
  { 4312   14 }  (         3.020, 1.19246506441,    242.72860397400 ),
  { 4312   15 }  (         2.886, 6.11652627155,    529.69096509460 ),
  { 4312   16 }  (         3.810, 3.44050803490,   5573.14280143310 ),
  { 4312   17 }  (         2.714, 0.30637881025,    398.14900340820 ),
  { 4312   18 }  (         2.371, 4.38118838167,   5223.69391980220 ),
  { 4312   19 }  (         2.538, 2.27992810679,    553.56940284240 ),
  { 4312   20 }  (         2.079, 3.75435330484,      0.98032106820 )
  (*$ifndef meeus *)                                                 ,
  { 4312   21 }  (         1.675, 0.90216407959,    951.71840625060 ),
  { 4312   22 }  (         1.534, 5.75900462759,   1349.86740965880 ),
  { 4312   23 }  (         1.224, 2.97328088405,   2146.16541647520 ),
  { 4312   24 }  (         1.449, 4.36415913970,   1748.01641306700 ),
  { 4312   25 }  (         1.341, 3.72061130861,   1194.44701022460 ),
  { 4312   26 }  (         1.254, 2.94846826628,   6438.49624942560 ),
  { 4312   27 }  (         0.999, 5.98640014468,   6286.59896834040 ),
  { 4312   28 }  (         0.917, 4.79788687522,   5088.62883976680 ),
  { 4312   29 }  (         0.828, 3.31321076572,    213.29909543800 ),
  { 4312   30 }  (         1.103, 1.27104454479, 161000.68573767410 ),
  { 4312   31 }  (         0.762, 3.41582762988,   5486.77784317500 ),
  { 4312   32 }  (         1.044, 0.60409577691,   3154.68708489560 ),
  { 4312   33 }  (         0.887, 5.23465144638,   7084.89678111520 ),
  { 4312   34 }  (         0.645, 1.60096192515,   2544.31441988340 ),
  { 4312   35 }  (         0.681, 3.43155669169,   4694.00295470760 ),
  { 4312   36 }  (         0.605, 2.47806340546,  10977.07880469900 ),
  { 4312   37 }  (         0.706, 6.19393222575,   4690.47983635860 ),
  { 4312   38 }  (         0.643, 1.98042503148,    801.82093112380 ),
  { 4312   39 }  (         0.502, 1.44394375363,   6836.64525283380 ),
  { 4312   40 }  (         0.490, 2.34129524194,   1592.59601363280 ),
  { 4312   41 }  (         0.458, 1.30876448575,   4292.33083295040 ),
  { 4312   42 }  (         0.431, 0.03526421494,   7234.79425624200 ),
  { 4312   43 }  (         0.379, 3.17030522615,   6309.37416979120 ),
  { 4312   44 }  (         0.348, 0.99049550009,   6040.34724601740 ),
  { 4312   45 }  (         0.386, 1.57019797263,  71430.69561812909 ),
  { 4312   46 }  (         0.347, 0.67013291338,   1059.38193018920 ),
  { 4312   47 }  (         0.458, 3.81499443681, 149854.40013480789 ),
  { 4312   48 }  (         0.302, 1.91760044838,  10447.38783960440 ),
  { 4312   49 }  (         0.307, 3.55343347416,   8031.09226305840 ),
  { 4312   50 }  (         0.395, 4.93701776616,   7632.94325965020 ),
  { 4312   51 }  (         0.314, 3.18093696547,   2352.86615377180 ),
  { 4312   52 }  (         0.282, 4.41936437052,   9437.76293488700 ),
  { 4312   53 }  (         0.276, 2.71314254553,   3894.18182954220 ),
  { 4312   54 }  (         0.298, 2.52037474210,   6127.65545055720 ),
  { 4312   55 }  (         0.230, 1.37790215549,   4705.73230754360 ),
  { 4312   56 }  (         0.252, 0.55330133471,   6279.55273164240 ),
  { 4312   57 }  (         0.255, 5.26570187369,   6812.76681508600 ),
  { 4312   58 }  (         0.275, 0.67264264272,  25132.30339996560 ),
  { 4312   59 }  (         0.178, 0.92820785174,   1990.74501704100 ),
  { 4312   60 }  (         0.221, 0.63897368842,   6256.77753019160 ),
  { 4312   61 }  (         0.155, 0.77319790838,  14143.49524243060 ),
  { 4312   62 }  (         0.150, 2.40470465561,    426.59819087600 ),
  { 4312   63 }  (         0.196, 6.06877865012,    640.87760738220 ),
  { 4312   64 }  (         0.137, 2.21679460145,   8429.24126646660 ),
  { 4312   65 }  (         0.127, 3.26094223174,  17789.84561978500 ),
  { 4312   66 }  (         0.128, 5.47237279946,  12036.46073488820 ),
  { 4312   67 }  (         0.122, 2.16291082757,  10213.28554621100 ),
  { 4312   68 }  (         0.118, 0.45789822268,   7058.59846131540 ),
  { 4312   69 }  (         0.141, 2.34932647403,  11506.76976979360 ),
  { 4312   70 }  (         0.100, 0.85621569847,   6290.18939699220 ),
  { 4312   71 }  (         0.092, 5.10587476002,   7079.37385680780 ),
  { 4312   72 }  (         0.126, 2.65428307012,  88860.05707098669 ),
  { 4312   73 }  (         0.106, 5.85646710022,   7860.41939243920 ),
  { 4312   74 }  (         0.084, 3.57457554262,  16730.46368959580 ),
  { 4312   75 }  (         0.089, 4.21433259618,  83996.84731811189 ),
  { 4312   76 }  (         0.097, 5.57938280855,  13367.97263110660 ),
  { 4312   77 }  (         0.102, 2.05853060226,     87.30820453981 ),
  { 4312   78 }  (         0.080, 4.73792651816,  11926.25441366880 ),
  { 4312   79 }  (         0.080, 5.41418965044,  10973.55568635000 ),
  { 4312   80 }  (         0.106, 4.10978997399,   3496.03282613400 ),
  { 4312   81 }  (         0.102, 3.62650006043, 244287.60000722769 ),
  { 4312   82 }  (         0.075, 4.89483161769,   5643.17856367740 ),
  { 4312   83 }  (         0.087, 0.42863750683,  11015.10647733480 ),
  { 4312   84 }  (         0.069, 1.88908760720,  10177.25767953360 ),
  { 4312   85 }  (         0.089, 1.35567273119,   6681.22485339960 ),
  { 4312   86 }  (         0.066, 0.99455837265,   6525.80445396540 ),
  { 4312   87 }  (         0.067, 5.51240997070,   3097.88382272579 ),
  { 4312   88 }  (         0.076, 2.72016814799,   4164.31198961300 ),
  { 4312   89 }  (         0.063, 1.44349902540,   9917.69687450980 ),
  { 4312   90 }  (         0.078, 3.51469733747,  11856.21865142450 ),
  { 4312   91 }  (         0.085, 0.50956043858,  10575.40668294180 ),
  { 4312   92 }  (         0.067, 3.62043033405,  16496.36139620240 ),
  { 4312   93 }  (         0.055, 5.24637517308,   3340.61242669980 ),
  { 4312   94 }  (         0.048, 5.43966777314,  20426.57109242200 ),
  { 4312   95 }  (         0.064, 5.79535817813,   2388.89402044920 ),
  { 4312   96 }  (         0.046, 5.43499966519,   6275.96230299060 ),
  { 4312   97 }  (         0.050, 3.86263598617,   5729.50644714900 ),
  { 4312   98 }  (         0.044, 1.52269529228,  12168.00269657460 ),
  { 4312   99 }  (         0.057, 4.96352373486,  14945.31617355440 ),
  { 4312  100 }  (         0.045, 1.00861230160,   8635.94200376320 ),
  { 4312  101 }  (         0.043, 3.30685683359,   9779.10867612540 ),
  { 4312  102 }  (         0.042, 0.63481258930,   2699.73481931760 ),
  { 4312  103 }  (         0.041, 5.67996766641,  11712.95531823080 ),
  { 4312  104 }  (         0.056, 4.34024451468,  90955.55169449610 ),
  { 4312  105 }  (         0.041, 5.81722212845,    709.93304855830 ),
  { 4312  106 }  (         0.053, 6.17052087143, 233141.31440436149 ),
  { 4312  107 }  (         0.037, 3.12495025087,  16200.77272450120 ),
  { 4312  108 }  (         0.035, 5.76973458495,  12569.67481833180 ),
  { 4312  109 }  (         0.037, 0.31656444326,  24356.78078864160 ),
  { 4312  110 }  (         0.035, 0.96229051027,  17298.18232732620 ),
  { 4312  111 }  (         0.033, 5.23130355867,   5331.35744374080 ),
  { 4312  112 }  (         0.035, 0.62517020593,  25158.60171976540 ),
  { 4312  113 }  (         0.035, 0.80004512129,  13916.01910964160 ),
  { 4312  114 }  (         0.037, 2.89336088688,  12721.57209941700 ),
  { 4312  115 }  (         0.030, 4.50198402401,  23543.23050468179 ),
  { 4312  116 }  (         0.030, 5.31355708693,  18319.53658487960 ),
  { 4312  117 }  (         0.029, 3.47275229977,  13119.72110282519 ),
  { 4312  118 }  (         0.029, 3.11002782516,   4136.91043351620 ),
  { 4312  119 }  (         0.032, 5.52273255667,   5753.38488489680 ),
  { 4312  120 }  (         0.035, 3.79699996680, 143571.32428481648 ),
  { 4312  121 }  (         0.026, 1.50634201907, 154717.60988768269 ),
  { 4312  122 }  (         0.030, 3.53519084118,   6284.05617105960 ),
  { 4312  123 }  (         0.023, 4.41808025967,   5884.92684658320 ),
  { 4312  124 }  (         0.025, 1.38477355808,  65147.61976813770 ),
  { 4312  125 }  (         0.023, 3.49782549797,   7477.52286021600 ),
  { 4312  126 }  (         0.019, 3.14329413716,   6496.37494542940 ),
  { 4312  127 }  (         0.019, 2.20135125199,  18073.70493865020 ),
  { 4312  128 }  (         0.019, 4.95020255309,   3930.20969621960 ),
  { 4312  129 }  (         0.019, 0.57998702747,  31415.37924995700 ),
  { 4312  130 }  (         0.021, 1.75474323399,  12139.55350910680 ),
  { 4312  131 }  (         0.019, 3.92233070499,  19651.04848109800 ),
  { 4312  132 }  (         0.014, 0.98131213224,  12559.03815298200 ),
  { 4312  133 }  (         0.019, 4.93309333729,   2942.46342329160 ),
  { 4312  134 }  (         0.016, 5.55997534558,   8827.39026987480 ),
  { 4312  135 }  (         0.013, 1.68808165516,   4535.05943692440 ),
  { 4312  136 }  (         0.013, 0.33982116161,   4933.20844033260 ),
  { 4312  137 }  (         0.012, 1.85426309994,   5856.47765911540 ),
  { 4312  138 }  (         0.010, 4.82763996845,  13095.84266507740 ),
  { 4312  139 }  (         0.011, 5.38005490571,  11790.62908865880 ),
  { 4312  140 }  (         0.010, 1.40815507226,  10988.80815753500 ),
  { 4312  141 }  (         0.011, 3.05005267431,  17260.15465469040 ),
  { 4312  142 }  (         0.010, 4.93364992366,  12352.85260454480 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_ear_l3:array[0.. 21,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_l3:array[0..  6,0..2] of extended = (
  (*$else *)
  vsop87_ear_l3:array[0.. 21,0..2] of extended = (
  (*$endif *)
  { 4313    1 }  (       289.226, 5.84384198723,   6283.07584999140 ),
  { 4313    2 }  (        34.955, 0.00000000000,      0.00000000000 ),
  { 4313    3 }  (        16.819, 5.48766912348,  12566.15169998280 ),
  { 4313    4 }  (         2.962, 5.19577265202,    155.42039943420 ),
  { 4313    5 }  (         1.288, 4.72200252235,      3.52311834900 ),
  { 4313    6 }  (         0.635, 5.96925937141,    242.72860397400 ),
  { 4313    7 }  (         0.714, 5.30045809128,  18849.22754997420 )
  (*$ifndef meeus *)                                                 ,
  { 4313    8 }  (         0.402, 3.78682982419,    553.56940284240 ),
  { 4313    9 }  (         0.072, 4.29768126180,   6286.59896834040 ),
  { 4313   10 }  (         0.067, 0.90721687647,   6127.65545055720 ),
  { 4313   11 }  (         0.036, 5.24029648014,   6438.49624942560 ),
  { 4313   12 }  (         0.024, 5.16003960716,  25132.30339996560 ),
  { 4313   13 }  (         0.023, 3.01921570335,   6309.37416979120 ),
  { 4313   14 }  (         0.017, 5.82863573502,   6525.80445396540 ),
  { 4313   15 }  (         0.017, 3.67772863930,  71430.69561812909 ),
  { 4313   16 }  (         0.009, 4.58467294499,   1577.34354244780 ),
  { 4313   17 }  (         0.008, 1.40626662824,  11856.21865142450 ),
  { 4313   18 }  (         0.008, 5.07561257196,   6256.77753019160 ),
  { 4313   19 }  (         0.007, 2.82473374405,  83996.84731811189 ),
  { 4313   20 }  (         0.005, 2.71488713339,  10977.07880469900 ),
  { 4313   21 }  (         0.005, 3.76879847273,  12036.46073488820 ),
  { 4313   22 }  (         0.005, 4.28412873331,   6275.96230299060 )
  (*$endif *)
      );
  (*@\\\0000000D01*)
  (*@/// vsop87_ear_l4:array[0.. 10,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_ear_l4:array[0..  2,0..2] of extended = (
  (*$else *)
  vsop87_ear_l4:array[0.. 10,0..2] of extended = (
  (*$endif *)
  { 4314    1 }  (       114.084, 3.14159265359,      0.00000000000 ),
  { 4314    2 }  (         7.717, 4.13446589358,   6283.07584999140 ),
  { 4314    3 }  (         0.765, 3.83803776214,  12566.15169998280 )
  (*$ifndef meeus *)                                                 ,
  { 4314    4 }  (         0.420, 0.41925861858,    155.42039943420 ),
  { 4314    5 }  (         0.040, 3.59847585840,  18849.22754997420 ),
  { 4314    6 }  (         0.041, 3.14398414077,      3.52311834900 ),
  { 4314    7 }  (         0.035, 5.00298940826,   5573.14280143310 ),
  { 4314    8 }  (         0.013, 0.48794833701,  77713.77146812050 ),
  { 4314    9 }  (         0.010, 5.64801766350,   6127.65545055720 ),
  { 4314   10 }  (         0.008, 2.84160570605, 161000.68573767410 ),
  { 4314   11 }  (         0.002, 0.54912904658,   6438.49624942560 )
  (*$endif *)
      );
  (*@\\\0000000901*)
  (*@/// vsop87_ear_l5:array[0..  0,0..2] of extended = (..); *)
  (*$ifdef meeus *)
  vsop87_ear_l5:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_ear_l5:array[0..  4,0..2] of extended = (
  (*$endif *)
  { 4315    1 }  (         0.878, 3.14159265359,      0.00000000000 )
  (*$ifndef meeus *)                                                 ,
  { 4315    2 }  (         0.172, 2.76579069510,   6283.07584999140 ),
  { 4315    3 }  (         0.050, 2.01353298182,    155.42039943420 ),
  { 4315    4 }  (         0.028, 2.21496423926,  12566.15169998280 ),
  { 4315    5 }  (         0.005, 1.75600058765,  18849.22754997420 )
  (*$endif *)
      );
  (*@\\\0000000201*)
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_ear_l0)) and (nr<=high(vsop87_ear_l0)) then begin
        a:=vsop87_ear_l0[nr,0];  b:=vsop87_ear_l0[nr,1];  c:=vsop87_ear_l0[nr,2];
        end;
      1: if (nr>=low(vsop87_ear_l1)) and (nr<=high(vsop87_ear_l1)) then begin
        a:=vsop87_ear_l1[nr,0];  b:=vsop87_ear_l1[nr,1];  c:=vsop87_ear_l1[nr,2];
        end;
      2: if (nr>=low(vsop87_ear_l2)) and (nr<=high(vsop87_ear_l2)) then begin
        a:=vsop87_ear_l2[nr,0];  b:=vsop87_ear_l2[nr,1];  c:=vsop87_ear_l2[nr,2];
        end;
      3: if (nr>=low(vsop87_ear_l3)) and (nr<=high(vsop87_ear_l3)) then begin
        a:=vsop87_ear_l3[nr,0];  b:=vsop87_ear_l3[nr,1];  c:=vsop87_ear_l3[nr,2];
        end;
      4: if (nr>=low(vsop87_ear_l4)) and (nr<=high(vsop87_ear_l4)) then begin
        a:=vsop87_ear_l4[nr,0];  b:=vsop87_ear_l4[nr,1];  c:=vsop87_ear_l4[nr,2];
        end;
      5: if (nr>=low(vsop87_ear_l5)) and (nr<=high(vsop87_ear_l5)) then begin
        a:=vsop87_ear_l5[nr,0];  b:=vsop87_ear_l5[nr,1];  c:=vsop87_ear_l5[nr,2];
        end;
      end;
    end;
  end;
(*@\\\000000081C*)
(*@\\\0000000301*)
(*@/// class TVSOPJupiter *)
(*@/// function TVSOPJupiter.RadiusFactor(nr,index: integer):TVSOPEntry; *)
function TVSOPJupiter.RadiusFactor(nr,index: integer):TVSOPEntry;
const
  (*@/// vsop87_jup_r0:array[0..744,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_r0:array[0.. 45,0..2] of extended = (
  (*$else *)
  vsop87_jup_r0:array[0..744,0..2] of extended = (
  (*$endif *)
  { 4530    1 }  ( 520887429.471, 0.00000000000,      0.00000000000 ),
  { 4530    2 }  (  25209327.020, 3.49108640015,    529.69096509460 ),
  { 4530    3 }  (    610599.902, 3.84115365602,   1059.38193018920 ),
  { 4530    4 }  (    282029.465, 2.57419879933,    632.78373931320 ),
  { 4530    5 }  (    187647.391, 2.07590380082,    522.57741809380 ),
  { 4530    6 }  (     86792.941, 0.71001090609,    419.48464387520 ),
  { 4530    7 }  (     72062.869, 0.21465694745,    536.80451209540 ),
  { 4530    8 }  (     65517.227, 5.97995850843,    316.39186965660 ),
  { 4530    9 }  (     29134.620, 1.67759243710,    103.09277421860 ),
  { 4530   10 }  (     30135.275, 2.16132058449,    949.17560896980 ),
  { 4530   11 }  (     23453.209, 3.54023147303,    735.87651353180 ),
  { 4530   12 }  (     22283.710, 4.19362773546,   1589.07289528380 ),
  { 4530   13 }  (     23947.340, 0.27457854894,      7.11354700080 ),
  { 4530   14 }  (     13032.600, 2.96043055741,   1162.47470440780 ),
  { 4530   15 }  (      9703.346, 1.90669572402,    206.18554843720 ),
  { 4530   16 }  (     12749.004, 2.71550102862,   1052.26838318840 ),
  { 4530   17 }  (      9161.431, 4.41352618935,    213.29909543800 ),
  { 4530   18 }  (      7894.539, 2.47907551404,    426.59819087600 ),
  { 4530   19 }  (      7057.978, 2.18184753111,   1265.56747862640 ),
  { 4530   20 }  (      6137.755, 6.26417542514,    846.08283475120 ),
  { 4530   21 }  (      5477.093, 5.65729325169,    639.89728631400 ),
  { 4530   22 }  (      3502.519, 0.56531297394,   1066.49547719000 ),
  { 4530   23 }  (      4136.890, 2.72219979684,    625.67019231240 ),
  { 4530   24 }  (      4170.012, 2.01605033912,    515.46387109300 ),
  { 4530   25 }  (      2499.966, 4.55182055941,    838.96928775040 ),
  { 4530   26 }  (      2616.955, 2.00993967129,   1581.95934828300 ),
  { 4530   27 }  (      1911.876, 0.85621927419,    412.37109687440 ),
  { 4530   28 }  (      2127.644, 6.12751461750,    742.99006053260 ),
  { 4530   29 }  (      1610.549, 3.08867789275,   1368.66025284500 ),
  { 4530   30 }  (      1479.484, 2.68026191372,   1478.86657406440 ),
  { 4530   31 }  (      1230.708, 1.89042979701,    323.50541665740 ),
  { 4530   32 }  (      1216.810, 1.80171561024,    110.20632121940 ),
  { 4530   33 }  (       961.072, 4.54876989805,   2118.76386037840 ),
  { 4530   34 }  (       885.708, 4.14785948471,    533.62311835770 ),
  { 4530   35 }  (       776.700, 3.67696954690,    728.76296653100 ),
  { 4530   36 }  (       998.579, 2.87208940110,    309.27832265580 ),
  { 4530   37 }  (      1014.959, 1.38673237666,    454.90936652730 ),
  { 4530   38 }  (       727.162, 3.98824686402,   1155.36115740700 ),
  { 4530   39 }  (       655.289, 2.79065604219,   1685.05212250160 ),
  { 4530   40 }  (       821.465, 1.59342534396,   1898.35121793960 ),
  { 4530   41 }  (       620.798, 4.82284338962,    956.28915597060 ),
  { 4530   42 }  (       653.981, 3.38150775269,   1692.16566950240 ),
  { 4530   43 }  (       812.036, 5.94091899141,    909.81873305460 ),
  { 4530   44 }  (       562.120, 0.08095987241,    543.91805909620 ),
  { 4530   45 }  (       542.221, 0.28360266386,    525.75881183150 ),
  (*$ifndef meeus *)
  { 4530   46 }  (       457.859, 0.12722694510,   1375.77379984580 ),
  (*$endif *)
  { 4530   47 }  (       614.784, 2.27624915604,    942.06206196900 )
  (*$ifndef meeus *)                                                 ,
  { 4530   48 }  (       435.805, 2.60272129748,     95.97922721780 ),
  { 4530   49 }  (       496.066, 5.53005947761,    380.12776796000 ),
  { 4530   50 }  (       469.965, 2.81896276101,   1795.25844372100 ),
  { 4530   51 }  (       445.003, 0.14623567024,     14.22709400160 ),
  { 4530   52 }  (       290.869, 3.89339143564,   1471.75302706360 ),
  { 4530   53 }  (       276.627, 2.52238450687,   2001.44399215820 ),
  { 4530   54 }  (       275.084, 2.98863518924,    526.50957135690 ),
  { 4530   55 }  (       293.875, 2.04938438861,    199.07200143640 ),
  { 4530   56 }  (       290.985, 6.03131226226,   1169.58825140860 ),
  { 4530   57 }  (       338.342, 2.79873192583,   1045.15483618760 ),
  { 4530   58 }  (       257.482, 6.13395478303,    532.87235883230 ),
  { 4530   59 }  (       319.013, 1.34803130803,   2214.74308759620 ),
  { 4530   60 }  (       309.352, 5.36855804945,   1272.68102562720 ),
  { 4530   61 }  (       345.804, 1.56404293688,    491.55792945680 ),
  { 4530   62 }  (       303.364, 1.15407454372,   5753.38488489680 ),
  { 4530   63 }  (       192.325, 0.91996333387,   1596.18644228460 ),
  { 4530   64 }  (       215.398, 2.63572815848,   2111.65031337760 ),
  { 4530   65 }  (       200.738, 2.37259566683,   1258.45393162560 ),
  { 4530   66 }  (       239.036, 3.57397189838,    835.03713448730 ),
  { 4530   67 }  (       197.073, 5.92859096863,    453.42489381900 ),
  { 4530   68 }  (       139.440, 3.63960322318,   1788.14489672020 ),
  { 4530   69 }  (       191.373, 6.28251311870,    983.11585891360 ),
  { 4530   70 }  (       176.551, 2.57669991654,   9683.59458111640 ),
  { 4530   71 }  (       123.567, 2.26158186345,   2317.83586181480 ),
  { 4530   72 }  (       128.176, 4.66585907670,    831.85574074960 ),
  { 4530   73 }  (       112.430, 0.85604150812,    433.71173787680 ),
  { 4530   74 }  (       128.817, 1.10567106595,   2531.13495725280 ),
  { 4530   75 }  (        99.390, 4.50312054049,    518.64526483070 ),
  { 4530   76 }  (        93.870, 2.72553879990,    853.19638175200 ),
  { 4530   77 }  (       106.481, 5.81462222290,    220.41264243880 ),
  { 4530   78 }  (       120.188, 2.95156363556,      3.93215326310 ),
  { 4530   79 }  (       104.002, 2.22221906187,     74.78159856730 ),
  { 4530   80 }  (        81.655, 3.23481337678,   1361.54670584420 ),
  { 4530   81 }  (       112.513, 4.86216964016,    528.20649238630 ),
  { 4530   82 }  (        79.539, 0.88542246830,    430.53034413910 ),
  { 4530   83 }  (        85.801, 2.11458386763,   1574.84580128220 ),
  { 4530   84 }  (        85.685, 2.33823884827,   2428.04218303420 ),
  { 4530   85 }  (        68.311, 3.35727048905,   2104.53676637680 ),
  { 4530   86 }  (        69.570, 3.04164697156,    302.16477565500 ),
  { 4530   87 }  (        69.775, 3.22402404312,    305.34616939270 ),
  { 4530   88 }  (        69.570, 0.20494979941,    532.13864564940 ),
  { 4530   89 }  (        56.991, 2.00204191909,   2634.22773147140 ),
  { 4530   90 }  (        77.062, 2.09816000231,    508.35032409220 ),
  { 4530   91 }  (        56.716, 3.91743976711,   2221.85663459700 ),
  { 4530   92 }  (        58.325, 5.72360355252,    628.85158605010 ),
  { 4530   93 }  (        52.485, 4.02485010492,    527.24328453980 ),
  { 4530   94 }  (        63.645, 1.09973563964,   1364.72809958190 ),
  { 4530   95 }  (        53.607, 0.87425992614,   2847.52682690940 ),
  { 4530   96 }  (        59.598, 0.95822471775,    494.26624244250 ),
  { 4530   97 }  (        57.960, 3.45779497978,   2008.55753915900 ),
  { 4530   98 }  (        41.512, 3.51955526735,    529.73914920440 ),
  { 4530   99 }  (        44.666, 1.62313786651,    984.60033162190 ),
  { 4530  100 }  (        44.883, 4.90091959557,   2648.45482547300 ),
  { 4530  101 }  (        53.206, 1.19800364308,    760.25553592000 ),
  { 4530  102 }  (        44.393, 4.42623747662,   1063.31408345230 ),
  { 4530  103 }  (        37.566, 2.93021095213,   1677.93857550080 ),
  { 4530  104 }  (        41.516, 0.32174409278,    529.64278098480 ),
  { 4530  105 }  (        42.855, 0.03093594081,   1439.50969814920 ),
  { 4530  106 }  (        45.963, 2.54342106514,    636.71589257630 ),
  { 4530  107 }  (        40.181, 4.39381642864,   1148.24761040620 ),
  { 4530  108 }  (        38.770, 4.31675565025,    149.56319713460 ),
  { 4530  109 }  (        40.348, 2.10140891053,   2744.43405269080 ),
  { 4530  110 }  (        48.851, 5.60297777544,   2810.92146160520 ),
  { 4530  111 }  (        37.085, 5.07828164301,   1905.46476494040 ),
  { 4530  112 }  (        43.875, 1.24536971083,    621.73803904930 ),
  { 4530  113 }  (        34.005, 3.09360167248,   2420.92863603340 ),
  { 4530  114 }  (        36.782, 0.84232174637,    530.65417294110 ),
  { 4530  115 }  (        31.139, 5.35811251334,   1485.98012106520 ),
  { 4530  116 }  (        39.295, 4.70800489067,    569.04784100980 ),
  { 4530  117 }  (        39.700, 2.46163878814,    355.74874557180 ),
  { 4530  118 }  (        31.527, 6.19284070863,      3.18139373770 ),
  { 4530  119 }  (        28.399, 2.48456666067,    519.39602435610 ),
  { 4530  120 }  (        32.432, 2.73281750275,    604.47256366190 ),
  { 4530  121 }  (        27.119, 3.92341697086,   2324.94940881560 ),
  { 4530  122 }  (        26.753, 1.74975198417,   2950.61960112800 ),
  { 4530  123 }  (        28.986, 1.83535862643,   1891.23767093880 ),
  { 4530  124 }  (        26.493, 0.60380196895,   1055.44977692610 ),
  { 4530  125 }  (        33.525, 0.76068430639,    643.82943957710 ),
  { 4530  126 }  (        26.568, 1.03594610835,    405.25754987360 ),
  { 4530  127 }  (        25.534, 3.46320665375,    458.84151979040 ),
  { 4530  128 }  (        24.421, 0.88181836930,    423.41679713830 ),
  { 4530  129 }  (        32.949, 3.18597137308,    528.72775724810 ),
  { 4530  130 }  (        22.456, 0.43129919683,   1073.60902419080 ),
  { 4530  131 }  (        21.599, 1.41820425091,    540.73666535850 ),
  { 4530  132 }  (        25.673, 0.52358194760,    511.53171782990 ),
  { 4530  133 }  (        21.115, 3.08023522766,    629.60234557550 ),
  { 4530  134 }  (        22.713, 0.65234613144,   3163.91869656600 ),
  { 4530  135 }  (        19.189, 5.16589014963,    635.96513305090 ),
  { 4530  136 }  (        26.042, 1.33629471285,    330.61896365820 ),
  { 4530  137 }  (        18.263, 3.59973446951,    746.92221379570 ),
  { 4530  138 }  (        18.210, 2.66819439927,   1994.33044515740 ),
  { 4530  139 }  (        19.724, 4.13552133321,   1464.63948006280 ),
  { 4530  140 }  (        19.480, 1.85656428109,   3060.82592234740 ),
  { 4530  141 }  (        23.927, 4.99826361784,   1289.94650101460 ),
  { 4530  142 }  (        21.886, 5.91718683551,   1802.37199072180 ),
  { 4530  143 }  (        17.482, 2.82161612542,   2737.32050569000 ),
  { 4530  144 }  (        16.608, 5.67394889755,    408.43894361130 ),
  { 4530  145 }  (        22.892, 5.26731352093,    672.14061522840 ),
  { 4530  146 }  (        18.349, 1.89869734949,   1021.24889455140 ),
  { 4530  147 }  (        19.123, 3.65882402977,    415.55249061210 ),
  { 4530  148 }  (        15.735, 3.34772676006,   1056.20053645150 ),
  { 4530  149 }  (        16.373, 0.18094878053,   1699.27921650320 ),
  { 4530  150 }  (        18.899, 3.69120638874,     88.86568021700 ),
  { 4530  151 }  (        18.655, 1.97327300097,     38.13303563780 ),
  { 4530  152 }  (        15.542, 3.82204881010,    721.64941953020 ),
  { 4530  153 }  (        16.780, 1.90976657921,    217.23124870110 ),
  { 4530  154 }  (        15.313, 1.05907174619,    114.13847448250 ),
  { 4530  155 }  (        15.190, 1.32317039042,    117.31986822020 ),
  { 4530  156 }  (        15.080, 3.74469077216,   2641.34127847220 ),
  { 4530  157 }  (        19.836, 2.73184571324,     39.35687591520 ),
  { 4530  158 }  (        14.708, 1.67270454473,    529.16970023280 ),
  { 4530  159 }  (        14.036, 3.54305270022,    142.44965013380 ),
  { 4530  160 }  (        12.931, 1.48829749349,   3267.01147078460 ),
  { 4530  161 }  (        14.924, 1.32546085940,    490.33408917940 ),
  { 4530  162 }  (        14.753, 4.64530618027,   6283.07584999140 ),
  { 4530  163 }  (        14.672, 0.80451954754,   5223.69391980220 ),
  { 4530  164 }  (        12.085, 3.67072510553,    750.10360753340 ),
  { 4530  165 }  (        11.954, 2.97127390765,    505.31194270640 ),
  { 4530  166 }  (        14.650, 2.16792930250,    530.21222995640 ),
  { 4530  167 }  (        11.869, 1.66551754962,   2207.62954059540 ),
  { 4530  168 }  (        12.273, 0.20690014405,   1062.56332392690 ),
  { 4530  169 }  (        11.460, 1.11906683214,    561.93429400900 ),
  { 4530  170 }  (        11.083, 3.22049096074,    535.10759106600 ),
  { 4530  171 }  (        11.567, 5.22625628971,    524.06189080210 ),
  { 4530  172 }  (        11.161, 3.82945634036,     76.26607127560 ),
  { 4530  173 }  (        10.918, 1.27796962818,   2125.87740737920 ),
  { 4530  174 }  (        12.685, 3.96848605476,   2538.24850425360 ),
  { 4530  175 }  (        11.230, 3.23092119889,    422.66603761290 ),
  { 4530  176 }  (        12.645, 0.73670428580,    908.33426034630 ),
  { 4530  177 }  (        11.330, 5.56127247007,    531.17543780290 ),
  { 4530  178 }  (         9.509, 5.00507284204,    597.35901666110 ),
  { 4530  179 }  (        10.291, 3.84159025239,   1781.03134971940 ),
  { 4530  180 }  (        10.762, 4.91380719453,    525.02509864860 ),
  { 4530  181 }  (        11.786, 5.11863653538,    685.47393735270 ),
  { 4530  182 }  (        11.980, 1.72470898635,    911.30320576290 ),
  { 4530  183 }  (         8.937, 2.40338241992,   2310.72231481400 ),
  { 4530  184 }  (         9.253, 2.57670338148,   3053.71237534660 ),
  { 4530  185 }  (         9.488, 2.95089828501,   1382.88734684660 ),
  { 4530  186 }  (         9.889, 0.43758517388,   3480.31056622260 ),
  { 4530  187 }  (         8.781, 3.66562388594,    739.80866679490 ),
  { 4530  188 }  (         8.664, 2.70398612383,    526.77020378780 ),
  { 4530  189 }  (         9.505, 1.61249870019,   3377.21779200400 ),
  { 4530  190 }  (        11.540, 1.59520481029,   1474.67378837040 ),
  { 4530  191 }  (         9.533, 0.35468711552,   1512.80682400820 ),
  { 4530  192 }  (         9.980, 4.80984684596,    558.00214074590 ),
  { 4530  193 }  (         9.014, 1.21458362718,    416.30325013750 ),
  { 4530  194 }  (         7.969, 0.08480602718,    528.94020556920 ),
  { 4530  195 }  (         8.668, 5.29060005706,    945.24345570670 ),
  { 4530  196 }  (         7.851, 1.46751861875,    963.40270297140 ),
  { 4530  197 }  (         8.611, 1.13232641062,    532.61172640140 ),
  { 4530  198 }  (         7.838, 6.26933498027,    647.01083331480 ),
  { 4530  199 }  (         7.581, 2.90608705954,    533.88375078860 ),
  { 4530  200 }  (         8.583, 6.06634530166,  10213.28554621100 ),
  { 4530  201 }  (        10.198, 2.48743123636,   1819.63746610920 ),
  { 4530  202 }  (         8.536, 2.22700701790,   9153.90361602180 ),
  { 4530  203 }  (         9.759, 6.15593336218,    593.42686339800 ),
  { 4530  204 }  (         7.968, 3.75535355212,    530.44172462000 ),
  { 4530  205 }  (         7.142, 3.58836120327,   2957.73314812880 ),
  { 4530  206 }  (         7.122, 0.11970048938,    224.34479570190 ),
  { 4530  207 }  (         8.731, 0.75302913970,    960.22130923370 ),
  { 4530  208 }  (         7.063, 2.16793037690,    724.83081326790 ),
  { 4530  209 }  (         7.263, 2.29499675875,    520.12973753900 ),
  { 4530  210 }  (         6.418, 1.25058991868,   3583.40334044120 ),
  { 4530  211 }  (         8.270, 1.24806288317,    495.75071515080 ),
  { 4530  212 }  (         6.483, 4.74567772640,    202.25339517410 ),
  { 4530  213 }  (         7.197, 3.84169279666,    618.55664531160 ),
  { 4530  214 }  (         8.146, 0.73147060302,    230.56457082540 ),
  { 4530  215 }  (         6.165, 5.50124418381,     11.04570026390 ),
  { 4530  216 }  (         7.946, 2.07754951174,    953.10776223290 ),
  { 4530  217 }  (         7.675, 0.92400307662,    525.49817940060 ),
  { 4530  218 }  (         6.210, 1.45641362115,    483.22054217860 ),
  { 4530  219 }  (         7.359, 0.31355650764,    378.64329525170 ),
  { 4530  220 }  (         6.707, 2.92071167098,   1038.04128918680 ),
  { 4530  221 }  (         7.143, 0.18218134889,    731.94436026870 ),
  { 4530  222 }  (         7.309, 6.27084533477,     21.34064100240 ),
  { 4530  223 }  (         6.135, 2.67651237303,    312.45971639350 ),
  { 4530  224 }  (         5.558, 3.83419160288,    534.35683154060 ),
  { 4530  225 }  (         5.344, 5.25294750019,   1048.33622992530 ),
  { 4530  226 }  (         7.504, 0.74281415471,    457.61767951300 ),
  { 4530  227 }  (         5.335, 6.23059924424,    551.03160609700 ),
  { 4530  228 }  (         5.613, 1.51210605952,    524.27433912320 ),
  { 4530  229 }  (         5.284, 2.18579185671,    280.96714700450 ),
  { 4530  230 }  (         5.475, 5.95864753605,    539.98590583310 ),
  { 4530  231 }  (         5.056, 0.37387972537,    529.53090640020 ),
  { 4530  232 }  (         6.202, 5.53813122743,      2.44768055480 ),
  { 4530  233 }  (         5.490, 5.97692444199,    227.52618943960 ),
  { 4530  234 }  (         6.266, 0.76632858238,    938.12990870590 ),
  { 4530  235 }  (         5.750, 2.13496323512,    191.95845443560 ),
  { 4530  236 }  (         5.218, 4.69335266854,    560.71045373160 ),
  { 4530  237 }  (         5.480, 5.21157595558,   1057.89745748090 ),
  { 4530  238 }  (         5.738, 0.34249718209,    535.91074021810 ),
  { 4530  239 }  (         4.816, 1.51326236835,   2524.02141025200 ),
  { 4530  240 }  (         5.056, 3.46671669992,    529.85102378900 ),
  { 4530  241 }  (         4.710, 2.27813830550,   3370.10424500320 ),
  { 4530  242 }  (         5.228, 3.61776977584,   2097.42321937600 ),
  { 4530  243 }  (         4.878, 1.39829798223,   3693.60966166060 ),
  { 4530  244 }  (         5.727, 4.80120381106,    598.84348936940 ),
  { 4530  245 }  (         5.707, 3.94177950323,   2854.64037391020 ),
  { 4530  246 }  (         4.988, 4.87244187719,      1.48447270830 ),
  { 4530  247 }  (         5.424, 3.53268613904,    456.39383923560 ),
  { 4530  248 }  (         4.288, 4.84438067847,     70.84944530420 ),
  { 4530  249 }  (         5.944, 3.79180483544,  25558.21217647960 ),
  { 4530  250 }  (         4.195, 2.09136830994,   2627.11418447060 ),
  { 4530  251 }  (         4.582, 5.61707254513,   2435.15573003500 ),
  { 4530  252 }  (         4.268, 6.20250525415,    775.23338944700 ),
  { 4530  253 }  (         4.521, 0.20049967962,     92.04707395470 ),
  { 4530  254 }  (         5.405, 4.66492781581,    833.55266177900 ),
  { 4530  255 }  (         5.607, 3.30226645638,    535.32003938710 ),
  { 4530  256 }  (         4.171, 3.14873010832,    944.98282327580 ),
  { 4530  257 }  (         4.108, 5.84489743779,    440.82528487760 ),
  { 4530  258 }  (         4.367, 4.68363584557,    327.43756992050 ),
  { 4530  259 }  (         4.033, 3.30883782817,   3274.12501778540 ),
  { 4530  260 }  (         4.292, 0.20604269202,   3796.70243587920 ),
  { 4530  261 }  (         4.270, 0.98941708997,    387.24131496080 ),
  { 4530  262 }  (         4.259, 3.21120589971,    696.51963761660 ),
  { 4530  263 }  (         4.673, 1.96606729969,    107.02492748170 ),
  { 4530  264 }  (         4.031, 4.62854606236,   2751.54759969160 ),
  { 4530  265 }  (         5.115, 2.66416451377,   1215.16490244730 ),
  { 4530  266 }  (         4.181, 4.74527698816,    988.53248488500 ),
  { 4530  267 }  (         4.374, 1.50010561403,   1894.41906467650 ),
  { 4530  268 }  (         3.803, 3.59911687954,    437.64389113990 ),
  { 4530  269 }  (         3.761, 3.96903199782,    732.69511979410 ),
  { 4530  270 }  (         3.620, 1.57847427805,    381.61224066830 ),
  { 4530  271 }  (         3.490, 0.63097592112,    529.90341341570 ),
  { 4530  272 }  (         4.019, 2.57664165720,    916.93228005540 ),
  { 4530  273 }  (         4.133, 4.78417930217,    824.74219374880 ),
  { 4530  274 }  (         4.411, 3.13179382423,    630.33605875840 ),
  { 4530  275 }  (         4.099, 3.63702212253,    810.65811209910 ),
  { 4530  276 }  (         3.704, 6.17243801274,    537.76771994190 ),
  { 4530  277 }  (         4.124, 2.14248285449,    210.11770170030 ),
  { 4530  278 }  (         3.490, 3.20962050417,    529.47851677350 ),
  { 4530  279 }  (         3.281, 1.53106243317,    547.85021235930 ),
  { 4530  280 }  (         3.554, 6.03787799174,    739.05790726950 ),
  { 4530  281 }  (         4.101, 6.00406226999,    902.70518605380 ),
  { 4530  282 }  (         3.267, 3.49354065789,   1166.40685767090 ),
  { 4530  283 }  (         3.286, 2.55966870530,    945.99421523210 ),
  { 4530  284 }  (         4.041, 4.78735413707,    850.01498801430 ),
  { 4530  285 }  (         4.304, 0.11406117717,   1744.85586754190 ),
  { 4530  286 }  (         4.043, 5.20417093600,    635.23141986800 ),
  { 4530  287 }  (         3.115, 4.61986265585,    952.35700270750 ),
  { 4530  288 }  (         3.016, 0.95126220905,   3899.79521009780 ),
  { 4530  289 }  (         3.017, 2.59699501992,    632.83192342300 ),
  { 4530  290 }  (         3.219, 1.83594791142,     18.15924726470 ),
  { 4530  291 }  (         3.203, 6.12597544496,     10.29494073850 ),
  { 4530  292 }  (         3.220, 6.14213423140,   1158.54255114470 ),
  { 4530  293 }  (         3.000, 5.69509924353,    632.73555520340 ),
  { 4530  294 }  (         3.226, 5.59910267099,    608.40471692500 ),
  { 4530  295 }  (         3.118, 5.64998934505,     99.16062095550 ),
  { 4530  296 }  (         3.745, 2.08111521615,    282.45161971280 ),
  { 4530  297 }  (         2.837, 4.60175594220,    245.54242435240 ),
  { 4530  298 }  (         3.093, 6.02049413961,    633.74694715970 ),
  { 4530  299 }  (         3.120, 2.29047945342,    631.82053146670 ),
  { 4530  300 }  (         2.662, 3.69016679729,    885.43971066640 ),
  { 4530  301 }  (         3.150, 1.79784999553,    521.61421024730 ),
  { 4530  302 }  (         2.822, 3.14927418161,    295.05122865420 ),
  { 4530  303 }  (         2.615, 0.20732170653,     35.42472265210 ),
  { 4530  304 }  (         2.971, 1.28795094653,   1023.95720753710 ),
  { 4530  305 }  (         2.571, 2.01817133502,   1514.29129671650 ),
  { 4530  306 }  (         2.592, 0.48790221200,    195.13984817330 ),
  { 4530  307 }  (         3.263, 2.38820607343,    836.52160719560 ),
  { 4530  308 }  (         2.501, 0.21653750027,    465.95506679120 ),
  { 4530  309 }  (         2.451, 5.58559489768,    544.66881862160 ),
  { 4530  310 }  (         2.535, 1.44414086617,    460.53844081980 ),
  { 4530  311 }  (         2.666, 3.30350145485,   2413.81508903260 ),
  { 4530  312 }  (         2.412, 4.36756580310,   1056.93424963440 ),
  { 4530  313 }  (         2.452, 4.53818816565,    514.71311156760 ),
  { 4530  314 }  (         3.239, 1.17022488774,    177.87437278590 ),
  { 4530  315 }  (         3.218, 0.60551913257,   1061.82961074400 ),
  { 4530  316 }  (         2.408, 0.65423523810,    523.54062594030 ),
  { 4530  317 }  (         2.299, 2.15247752560,    319.57326339430 ),
  { 4530  318 }  (         2.791, 2.71505085086,    610.69233878540 ),
  { 4530  319 }  (         2.729, 1.77685979153,    252.65597135320 ),
  { 4530  320 }  (         2.666, 3.77750458842,   3171.03224356680 ),
  { 4530  321 }  (         2.303, 0.36676453766,   1969.20066324380 ),
  { 4530  322 }  (         2.664, 0.09674841214,    565.11568774670 ),
  { 4530  323 }  (         2.312, 2.07210502831,   3686.49611465980 ),
  { 4530  324 }  (         2.680, 4.94445888050,   1593.00504854690 ),
  { 4530  325 }  (         2.193, 0.55645982205,   2228.97018159780 ),
  { 4530  326 }  (         2.526, 1.07528597373,  12036.46073488820 ),
  { 4530  327 }  (         2.778, 1.48379350517,    447.79581952650 ),
  { 4530  328 }  (         2.235, 5.95475282699,   6151.53388830500 ),
  { 4530  329 }  (         2.759, 4.63976153480,    462.02291352810 ),
  { 4530  330 }  (         2.175, 4.53588570240,    501.37978944330 ),
  { 4530  331 }  (         2.323, 5.93670041006,    611.44309831080 ),
  { 4530  332 }  (         2.384, 2.81746622971,   3340.61242669980 ),
  { 4530  333 }  (         2.087, 3.10716079675,   1049.08698945070 ),
  { 4530  334 }  (         1.994, 2.02500860064,   1058.86066532740 ),
  { 4530  335 }  (         2.199, 2.20937490997,   1269.49963188950 ),
  { 4530  336 }  (         2.705, 1.97665276677,    415.29185818120 ),
  { 4530  337 }  (         2.787, 1.31053438756,   1041.22268292450 ),
  { 4530  338 }  (         2.003, 4.66904374443,    679.25416222920 ),
  { 4530  339 }  (         1.962, 1.82999730674,   2943.50605412720 ),
  { 4530  340 }  (         2.289, 2.96480800939,     69.15252427480 ),
  { 4530  341 }  (         2.192, 4.47837196209,    209.36694217490 ),
  { 4530  342 }  (         2.020, 0.04621364490,   4113.09430553580 ),
  { 4530  343 }  (         2.082, 1.11203059170,   4010.00153131720 ),
  { 4530  344 }  (         1.991, 3.20108648275,   3590.51688744200 ),
  { 4530  345 }  (         1.900, 3.32227077969,    421.93232443000 ),
  { 4530  346 }  (         2.193, 2.82218305362,    292.01284726840 ),
  { 4530  347 }  (         2.288, 1.94695631885,   1279.79457262800 ),
  { 4530  348 }  (         1.843, 5.23293634337,     14.97785352700 ),
  { 4530  349 }  (         1.932, 5.46684252030,   2281.23049651060 ),
  { 4530  350 }  (         2.177, 2.93031976617,    429.04587143080 ),
  { 4530  351 }  (         2.125, 0.06224847826,     24.37902238820 ),
  { 4530  352 }  (         2.464, 5.39581078430,   1261.63532536330 ),
  { 4530  353 }  (         1.938, 3.79908004671,   1059.43011429900 ),
  { 4530  354 }  (         2.029, 3.95461157815,    771.30123618390 ),
  { 4530  355 }  (         1.841, 4.74905354737,     78.71375183040 ),
  { 4530  356 }  (         1.922, 2.21862085389,     99.91138048090 ),
  { 4530  357 }  (         1.836, 5.75449805175,    623.22251175760 ),
  { 4530  358 }  (         2.145, 3.87052575546,    451.94042111070 ),
  { 4530  359 }  (         1.782, 0.40860352236,    754.03576079650 ),
  { 4530  360 }  (         1.784, 1.49468287576,    529.95159752550 ),
  { 4530  361 }  (         1.842, 3.49726261337,   1354.43315884340 ),
  { 4530  362 }  (         1.748, 3.48730020953,    522.62560220360 ),
  { 4530  363 }  (         1.816, 1.24334711210,    417.03696332040 ),
  { 4530  364 }  (         1.752, 1.15500390019,   1060.34513803570 ),
  { 4530  365 }  (         1.729, 2.69831073799,    642.34496686880 ),
  { 4530  366 }  (         1.985, 1.99916658759,    934.94851496820 ),
  { 4530  367 }  (         1.828, 5.44095029767,   1201.83158032300 ),
  { 4530  368 }  (         2.158, 3.45672748590,    827.92358748650 ),
  { 4530  369 }  (         1.959, 1.06033047373,     33.94024994380 ),
  { 4530  370 }  (         1.751, 3.13572498964,    384.05992122310 ),
  { 4530  371 }  (         1.781, 5.02895146997,   1098.73880610440 ),
  { 4530  372 }  (         2.074, 3.18582065441,   1366.21257229020 ),
  { 4530  373 }  (         1.757, 5.02778552877,    586.31331639720 ),
  { 4530  374 }  (         2.045, 3.08816627459,    535.84130424890 ),
  { 4530  375 }  (         2.273, 5.17998505813,   3178.14579056760 ),
  { 4530  376 }  (         1.617, 3.16674916201,     67.66805156650 ),
  { 4530  377 }  (         1.627, 6.10603469594,    432.01481684740 ),
  { 4530  378 }  (         1.930, 1.63968957659,      5.41662597140 ),
  { 4530  379 }  (         1.741, 0.99408274736,   1254.52177836250 ),
  { 4530  380 }  (         1.607, 5.65498642076,   1165.65609814550 ),
  { 4530  381 }  (         1.676, 3.06138410273,   1134.16352875650 ),
  { 4530  382 }  (         1.821, 3.05183555090,    567.82400073240 ),
  { 4530  383 }  (         1.677, 3.09175084930,   1251.34038462480 ),
  { 4530  384 }  (         1.994, 2.52023134712,   1059.90319505100 ),
  { 4530  385 }  (         2.204, 6.15376698510,    563.63121503840 ),
  { 4530  386 }  (         1.692, 4.19142612803,    106.27416795630 ),
  { 4530  387 }  (         1.906, 5.58417395051,     32.24332891440 ),
  { 4530  388 }  (         2.206, 1.75883974012,   1151.42900414390 ),
  { 4530  389 }  (         1.552, 3.04262360186,    385.54439393140 ),
  { 4530  390 }  (         1.508, 0.42002830727,    313.21047591890 ),
  { 4530  391 }  (         1.494, 1.43672345922,   2840.41327990860 ),
  { 4530  392 }  (         1.678, 2.17255433434,    306.83064210100 ),
  { 4530  393 }  (         1.511, 4.44377608685,    395.10562148700 ),
  { 4530  394 }  (         1.958, 0.05215107058,    761.74000862830 ),
  { 4530  395 }  (         1.760, 1.27045286501,   1173.52040467170 ),
  { 4530  396 }  (         1.463, 6.07810373103,      0.96320784650 ),
  { 4530  397 }  (         1.498, 2.79408561759,    277.03499374140 ),
  { 4530  398 }  (         1.636, 0.26199351490,    522.52923398400 ),
  { 4530  399 }  (         1.507, 0.48961801593,   4216.18707975440 ),
  { 4530  400 }  (         1.530, 3.42953827550,   1159.29331067010 ),
  { 4530  401 }  (         1.744, 2.39637837261,    203.00415469950 ),
  { 4530  402 }  (         1.569, 2.55719070621,      4.19278569400 ),
  { 4530  403 }  (         1.576, 3.45039607104,   1058.41872234270 ),
  { 4530  404 }  (         1.466, 2.24427539934,   1550.93985964600 ),
  { 4530  405 }  (         1.784, 2.34591354953,    529.43033266370 ),
  { 4530  406 }  (         1.939, 4.73685428610,   3067.93946934820 ),
  { 4530  407 }  (         1.938, 0.60126164334,   1059.33374607940 ),
  { 4530  408 }  (         1.523, 2.98744673443,   2730.20695868920 ),
  { 4530  409 }  (         1.834, 3.78099298791,    420.96911658350 ),
  { 4530  410 }  (         1.372, 3.53997115825,      5.62907429250 ),
  { 4530  411 }  (         1.361, 0.45533257707,    418.52143602870 ),
  { 4530  412 }  (         1.833, 5.12743628215,   1578.02719501990 ),
  { 4530  413 }  (         1.839, 4.24616044210,    981.63138620530 ),
  { 4530  414 }  (         1.567, 3.32429870195,    532.39927808030 ),
  { 4530  415 }  (         1.340, 1.94668282270,    528.41894070740 ),
  { 4530  416 }  (         1.422, 1.83191577465,   4002.88798431640 ),
  { 4530  417 }  (         1.745, 5.76913240451,    490.07345674850 ),
  { 4530  418 }  (         1.437, 4.19470227783,    420.44785172170 ),
  { 4530  419 }  (         1.419, 0.74849005330,    632.26247445140 ),
  { 4530  420 }  (         1.447, 5.65611888743,    373.01422095920 ),
  { 4530  421 }  (         1.578, 3.90273683089,    602.98809095360 ),
  { 4530  422 }  (         1.385, 3.88479835656,    419.43645976540 ),
  { 4530  423 }  (         1.352, 0.81697905853,   1585.14074202070 ),
  { 4530  424 }  (         1.399, 1.24785452243,    633.30500417500 ),
  { 4530  425 }  (         1.297, 5.57914023189,   1276.61317889030 ),
  { 4530  426 }  (         1.491, 1.66541781223,   2655.56837247380 ),
  { 4530  427 }  (         1.252, 0.72155670765,    173.94221952280 ),
  { 4530  428 }  (         1.658, 5.60924662850,    362.86229257260 ),
  { 4530  429 }  (         1.606, 3.95301396173,   2274.54683263650 ),
  { 4530  430 }  (         1.213, 4.55264289565,    366.79444583570 ),
  { 4530  431 }  (         1.521, 0.55773831071,   1592.25428902150 ),
  { 4530  432 }  (         1.220, 3.63029788040,    497.44763618020 ),
  { 4530  433 }  (         1.215, 4.42854185903,    531.38788612400 ),
  { 4530  434 }  (         1.549, 5.73765962068,    320.32402291970 ),
  { 4530  435 }  (         1.480, 4.29779032931,    303.86169668440 ),
  { 4530  436 }  (         1.507, 2.27998567874,    758.77106321170 ),
  { 4530  437 }  (         1.212, 3.38335836048,    536.85269620520 ),
  { 4530  438 }  (         1.245, 4.21639959154,      4.66586644600 ),
  { 4530  439 }  (         1.507, 3.52136655355,    774.00954916960 ),
  { 4530  440 }  (         1.481, 3.06156044618,   1585.89150154610 ),
  { 4530  441 }  (         1.462, 2.30628702634,   1363.24362687360 ),
  { 4530  442 }  (         1.180, 3.52708055024,   1064.79855616060 ),
  { 4530  443 }  (         1.193, 5.88284733845,   1060.86640289750 ),
  { 4530  444 }  (         1.398, 4.99456521692,    842.90144101350 ),
  { 4530  445 }  (         1.406, 1.53799746944,   1020.02505427400 ),
  { 4530  446 }  (         1.367, 4.10254739443,    799.61241183520 ),
  { 4530  447 }  (         1.336, 1.89387272380,    530.96298948180 ),
  { 4530  448 }  (         1.238, 3.62226383331,   3487.42411322340 ),
  { 4530  449 }  (         1.306, 3.39985119727,    539.25219265020 ),
  { 4530  450 }  (         1.156, 0.77127511567,   1603.29998928540 ),
  { 4530  451 }  (         1.482, 0.48451915093,    493.04240216510 ),
  { 4530  452 }  (         1.247, 5.64344659992,    479.28838891550 ),
  { 4530  453 }  (         1.195, 2.39909893341,    561.18353448360 ),
  { 4530  454 }  (         1.106, 0.89453807282,      2.92076130680 ),
  { 4530  455 }  (         1.227, 2.76231244946,    299.12639426920 ),
  { 4530  456 }  (         1.128, 4.72319873338,    124.43341522100 ),
  { 4530  457 }  (         1.086, 5.66180289525,   1053.75285589670 ),
  { 4530  458 }  (         1.329, 0.16664094530,    536.75632798560 ),
  { 4530  459 }  (         1.082, 4.51407359350,    528.25467649610 ),
  { 4530  460 }  (         1.105, 1.93890691771,    244.31858407500 ),
  { 4530  461 }  (         1.446, 0.65096230619,   1091.62525910360 ),
  { 4530  462 }  (         1.071, 4.67974963103,    521.82665856840 ),
  { 4530  463 }  (         1.413, 4.72936311016,   1141.13406340540 ),
  { 4530  464 }  (         1.086, 2.88721124443,   1262.38608488870 ),
  { 4530  465 }  (         1.254, 5.74156595137,    527.99404406520 ),
  { 4530  466 }  (         1.082, 5.60975006771,    531.12725369310 ),
  { 4530  467 }  (         1.148, 3.27410230525,   1035.00290780100 ),
  { 4530  468 }  (         1.224, 3.68807537150,     81.75213321620 ),
  { 4530  469 }  (         1.072, 0.48068438564,   1058.63117066380 ),
  { 4530  470 }  (         1.036, 1.68789163831,   1070.42763045310 ),
  { 4530  471 }  (         1.052, 4.72763208332,    913.75088631770 ),
  { 4530  472 }  (         1.166, 4.97812626679,    450.97721326420 ),
  { 4530  473 }  (         1.042, 2.90894542321,   3906.90875709860 ),
  { 4530  474 }  (         0.997, 1.65967703856,   3259.89792378380 ),
  { 4530  475 }  (         1.113, 3.06502453809,   1482.79872732750 ),
  { 4530  476 }  (         0.991, 0.91568114148,    576.16138801060 ),
  { 4530  477 }  (         0.987, 0.91349590742,   2332.06295581640 ),
  { 4530  478 }  (         1.003, 6.17381204883,    391.17346822390 ),
  { 4530  479 }  (         1.087, 3.19260020877,    151.04766984290 ),
  { 4530  480 }  (         0.987, 2.48065918834,   1912.57831194120 ),
  { 4530  481 }  (         0.975, 1.55458771092,    536.28324723360 ),
  { 4530  482 }  (         1.193, 2.19383228000,    523.09868295560 ),
  { 4530  483 }  (         0.979, 3.28693620660,   1379.70595310890 ),
  { 4530  484 }  (         0.963, 2.29845109892,   1467.82087380050 ),
  { 4530  485 }  (         1.279, 4.73978455573,    600.54041039880 ),
  { 4530  486 }  (         1.269, 1.77171706595,   5120.60114558360 ),
  { 4530  487 }  (         0.938, 3.13636271584,   1372.59240610810 ),
  { 4530  488 }  (         0.956, 0.94045126791,    429.77958461370 ),
  { 4530  489 }  (         1.130, 4.87259620358,    874.39401040250 ),
  { 4530  490 }  (         1.044, 3.52819283674,    530.58473697190 ),
  { 4530  491 }  (         1.244, 0.80634178279,    419.53282798500 ),
  { 4530  492 }  (         0.914, 4.34324212455,   1127.04998175570 ),
  { 4530  493 }  (         1.095, 3.17513475763,   6681.22485339960 ),
  { 4530  494 }  (         0.926, 5.53099018797,    537.55527162080 ),
  { 4530  495 }  (         1.025, 6.08315999637,    469.88722005430 ),
  { 4530  496 }  (         0.928, 2.64064849636,     31.01948863700 ),
  { 4530  497 }  (         0.887, 5.53922649066,    498.67147645760 ),
  { 4530  498 }  (         1.153, 5.20213407651,    554.06998748280 ),
  { 4530  499 }  (         0.976, 4.26047885490,    806.72595883600 ),
  { 4530  500 }  (         0.871, 5.79751110150,    594.65070367540 ),
  { 4530  501 }  (         1.044, 0.31244551729,    528.79719321730 ),
  { 4530  502 }  (         0.911, 0.94039205468,    337.73251065900 ),
  { 4530  503 }  (         1.197, 3.12884590029,   1966.23171782720 ),
  { 4530  504 }  (         0.930, 2.88178471518,   1056.46116888240 ),
  { 4530  505 }  (         1.052, 1.69484089706,    484.44438245600 ),
  { 4530  506 }  (         0.862, 0.67309397482,  20426.57109242200 ),
  { 4530  507 }  (         1.152, 1.16751621652,   1489.91227432830 ),
  { 4530  508 }  (         0.847, 3.25831322825,   1063.57471588320 ),
  { 4530  509 }  (         0.884, 0.71487680084,   2042.49778910280 ),
  { 4530  510 }  (         0.888, 5.38714907441,   5621.84292321040 ),
  { 4530  511 }  (         1.137, 4.02029739425,   1670.07426897460 ),
  { 4530  512 }  (         0.844, 3.31846798590,    812.14258480740 ),
  { 4530  513 }  (         0.860, 4.78175008217,    530.91480537200 ),
  { 4530  514 }  (         0.835, 3.63117401608,    451.72797278960 ),
  { 4530  515 }  (         0.931, 2.27352189963,    100.64509366380 ),
  { 4530  516 }  (         0.939, 3.51238251326,    523.47118997110 ),
  { 4530  517 }  (         0.860, 5.34207357904,    528.46712481720 ),
  { 4530  518 }  (         0.875, 0.87775537110,   4326.39340097380 ),
  { 4530  519 }  (         0.961, 5.69327275886,    498.19839570560 ),
  { 4530  520 }  (         0.966, 6.25512226434,    700.45179087970 ),
  { 4530  521 }  (         0.842, 3.20535945596,   1670.82502850000 ),
  { 4530  522 }  (         0.808, 1.09148925587,    683.18631549230 ),
  { 4530  523 }  (         0.810, 5.47935192896,    525.54636351040 ),
  { 4530  524 }  (         0.855, 6.06969867736,    446.31134681820 ),
  { 4530  525 }  (         0.989, 1.55623875216,   1493.09366806600 ),
  { 4530  526 }  (         0.837, 1.49510080792,   1025.44168024540 ),
  { 4530  527 }  (         0.974, 3.67667471757,  25565.32572348040 ),
  { 4530  528 }  (         0.788, 0.51622458293,    526.98265210890 ),
  { 4530  529 }  (         0.820, 1.86002542644,    629.86297800640 ),
  { 4530  530 }  (         0.813, 0.45441968195,   4694.00295470760 ),
  { 4530  531 }  (         0.953, 0.58786779132,    627.36711334180 ),
  { 4530  532 }  (         0.908, 2.82093327912,   3046.59882834580 ),
  { 4530  533 }  (         0.912, 2.69124310451,    946.72792841500 ),
  { 4530  534 }  (         0.820, 4.14947931572,   1884.12412393800 ),
  { 4530  535 }  (         0.948, 0.77931728039,  25551.09862947879 ),
  { 4530  536 }  (         0.844, 0.00976249584,    628.59095361920 ),
  { 4530  537 }  (         0.910, 0.99542530366,   5760.49843189760 ),
  { 4530  538 }  (         0.844, 0.22630964490,   1123.11782849260 ),
  { 4530  539 }  (         0.924, 4.41952345708,   5746.27133789600 ),
  { 4530  540 }  (         0.967, 3.20618313117,   9050.81084180320 ),
  { 4530  541 }  (         0.800, 0.10663079153,   4532.57894941100 ),
  { 4530  542 }  (         0.748, 3.01376405927,   5481.75455838080 ),
  { 4530  543 }  (         0.752, 5.82360472890,    701.93626358800 ),
  { 4530  544 }  (         0.771, 0.12101982692,    635.70450062000 ),
  { 4530  545 }  (         0.725, 2.81220410314,   3597.63043444280 ),
  { 4530  546 }  (         0.944, 0.40327408174,   1140.38330388000 ),
  { 4530  547 }  (         0.726, 5.28930472464,   1304.92435454160 ),
  { 4530  548 }  (         0.994, 5.16391370100,  10316.37832042960 ),
  { 4530  549 }  (         0.890, 4.10819809692,   1060.13268971460 ),
  { 4530  550 }  (         0.962, 1.48376004549,   1062.30269149600 ),
  { 4530  551 }  (         0.883, 5.26813169286,   1542.60247236780 ),
  { 4530  552 }  (         0.916, 6.02908368648,      7.86430652620 ),
  { 4530  553 }  (         0.725, 2.18773773010,   1176.70179840940 ),
  { 4530  554 }  (         0.808, 5.81725174908,   1087.69310584050 ),
  { 4530  555 }  (         0.757, 0.77440414330,    977.48678462110 ),
  { 4530  556 }  (         0.838, 3.81585420192,    986.08480433020 ),
  { 4530  557 }  (         0.888, 1.89634795578,    707.56533788050 ),
  { 4530  558 }  (         0.854, 5.47701506544,   2818.03500860600 ),
  { 4530  559 }  (         0.796, 1.08794807212,    987.30864460760 ),
  { 4530  560 }  (         0.856, 2.58042139486,   2803.80791460440 ),
  { 4530  561 }  (         0.708, 1.09492310353,    248.72381809010 ),
  { 4530  562 }  (         0.811, 3.23726191865,    121.25202148330 ),
  { 4530  563 }  (         0.727, 1.56150632966,   4319.27985397300 ),
  { 4530  564 }  (         0.687, 2.65457835371,   1567.73225428140 ),
  { 4530  565 }  (         0.675, 1.78690909614,    103.14095832840 ),
  { 4530  566 }  (         0.853, 4.74476428852,    951.62328952460 ),
  { 4530  567 }  (         0.832, 5.14362789810,   1054.71606374320 ),
  { 4530  568 }  (         0.846, 1.47557828604,    898.77303279070 ),
  { 4530  569 }  (         0.701, 1.72139817505,   5230.80746680300 ),
  { 4530  570 }  (         0.863, 3.98700238575,    686.95841006100 ),
  { 4530  571 }  (         0.703, 2.89202252444,     63.73589830340 ),
  { 4530  572 }  (         0.673, 6.11618580510,    738.32419408660 ),
  { 4530  573 }  (         0.806, 4.64475158248,    533.83556667880 ),
  { 4530  574 }  (         0.670, 2.67625974048,   1012.91150727320 ),
  { 4530  575 }  (         0.668, 4.93815253692,   5172.47623572500 ),
  { 4530  576 }  (         0.818, 1.41973280302,    580.09354127370 ),
  { 4530  577 }  (         0.652, 3.41422919445,    650.94298657790 ),
  { 4530  578 }  (         0.643, 2.46566726278,   1049.82070263360 ),
  { 4530  579 }  (         0.859, 2.50530106631,    782.34693644780 ),
  { 4530  580 }  (         0.662, 4.13533996643,    733.42883297700 ),
  { 4530  581 }  (         0.812, 1.30325352179,   1055.18914449520 ),
  { 4530  582 }  (         0.638, 4.21760246824,   1064.04779663520 ),
  { 4530  583 }  (         0.637, 6.13121700151,   4752.99159184980 ),
  { 4530  584 }  (         0.636, 0.83411828974,    711.49749114360 ),
  { 4530  585 }  (         0.642, 1.86741704507,   1053.96530421780 ),
  { 4530  586 }  (         0.795, 4.54081089118,   1457.52593306200 ),
  { 4530  587 }  (         0.783, 4.37652961667,    105.54045477340 ),
  { 4530  588 }  (         0.640, 5.44039474349,    632.03297978780 ),
  { 4530  589 }  (         0.651, 5.02431301146,    528.04643369190 ),
  { 4530  590 }  (         0.686, 0.27079898498,     11.77941344680 ),
  { 4530  591 }  (         0.644, 5.36935176134,    835.78789401270 ),
  { 4530  592 }  (         0.639, 1.86699974431,   6172.86952877200 ),
  { 4530  593 }  (         0.630, 2.86895754523,    633.53449883860 ),
  { 4530  594 }  (         0.826, 1.46026926041,   2199.76523406920 ),
  { 4530  595 }  (         0.687, 3.81221717134,     73.29712585900 ),
  { 4530  596 }  (         0.697, 4.18082589322,      1.69692102940 ),
  { 4530  597 }  (         0.788, 0.21278801649,    313.94418910180 ),
  { 4530  598 }  (         0.686, 2.51807576494,    638.41281360570 ),
  { 4530  599 }  (         0.847, 5.56263749391,   4429.48617519240 ),
  { 4530  600 }  (         0.673, 4.87494072856,    103.04459010880 ),
  { 4530  601 }  (         0.663, 4.80713895807,    991.71387862270 ),
  { 4530  602 }  (         0.614, 3.87231597482,    767.36908292080 ),
  { 4530  603 }  (         0.666, 5.71697262323,    661.09491496450 ),
  { 4530  604 }  (         0.681, 2.33844767741,    501.23677709140 ),
  { 4530  605 }  (         0.597, 3.03921014345,      6.95348830640 ),
  { 4530  606 }  (         0.777, 3.08786050361,    441.57604440300 ),
  { 4530  607 }  (         0.588, 0.08236113246,   4164.31198961300 ),
  { 4530  608 }  (         0.693, 4.66190836234,   3384.33133900480 ),
  { 4530  609 }  (         0.810, 1.97701084490,    860.30992875280 ),
  { 4530  610 }  (         0.602, 5.56403449542,   1587.58842257550 ),
  { 4530  611 }  (         0.622, 6.11554348965,      7.06536289100 ),
  { 4530  612 }  (         0.592, 3.29013906024,  10103.07922499160 ),
  { 4530  613 }  (         0.692, 6.10931942233,     12.74262129330 ),
  { 4530  614 }  (         0.597, 6.13204711801,      7.27360569520 ),
  { 4530  615 }  (         0.594, 2.58839673551,    849.26422848890 ),
  { 4530  616 }  (         0.728, 2.73732195088,      6.15033915430 ),
  { 4530  617 }  (         0.602, 5.28816527514,    949.12742486000 ),
  { 4530  618 }  (         0.568, 1.75508433865,   1077.54117745390 ),
  { 4530  619 }  (         0.575, 4.50676079721,   1230.14275597430 ),
  { 4530  620 }  (         0.588, 0.65827893998,   4642.78527063040 ),
  { 4530  621 }  (         0.561, 3.87565914360,    135.33610313300 ),
  { 4530  622 }  (         0.558, 3.36094471852,  24498.83024629040 ),
  { 4530  623 }  (         0.557, 3.45629457197,  19896.88012732740 ),
  { 4530  624 }  (         0.558, 1.17103892689,   3576.28979344040 ),
  { 4530  625 }  (         0.574, 5.19235074140,    104.05598206510 ),
  { 4530  626 }  (         0.560, 3.57141429379,   5333.90024102160 ),
  { 4530  627 }  (         0.555, 0.18349908409,    512.42548970720 ),
  { 4530  628 }  (         0.571, 0.83070148820,   1570.91364801910 ),
  { 4530  629 }  (         0.632, 3.67893818442,   1065.01100448170 ),
  { 4530  630 }  (         0.744, 2.33083237537,    620.25356634100 ),
  { 4530  631 }  (         0.540, 5.15775909675,   1751.53953141600 ),
  { 4530  632 }  (         0.592, 3.07238123875,   1446.62324515000 ),
  { 4530  633 }  (         0.537, 1.52803865425,   8094.52168583260 ),
  { 4530  634 }  (         0.550, 5.50701003577,   1432.39615114840 ),
  { 4530  635 }  (         0.546, 2.34388967045,    949.22379307960 ),
  { 4530  636 }  (         0.534, 3.04076654796,      7.16173111060 ),
  { 4530  637 }  (         0.619, 6.07865159203,     46.47042291600 ),
  { 4530  638 }  (         0.562, 0.96641974928,   1438.02522544090 ),
  { 4530  639 }  (         0.531, 1.06695547390,    100.17201291180 ),
  { 4530  640 }  (         0.599, 3.59295739143,   1144.31545714310 ),
  { 4530  641 }  (         0.526, 3.51641923371,      0.75075952540 ),
  { 4530  642 }  (         0.564, 0.72677136494,   1059.22187149480 ),
  { 4530  643 }  (         0.537, 5.72603965787,    513.22863885930 ),
  { 4530  644 }  (         0.630, 2.31183143900,   2729.45619916380 ),
  { 4530  645 }  (         0.530, 4.99510636441,   9264.10993724120 ),
  { 4530  646 }  (         0.649, 0.95666735852,    920.86443331850 ),
  { 4530  647 }  (         0.547, 1.18801926149,  11506.76976979360 ),
  { 4530  648 }  (         0.516, 3.28562070858,    734.91330568530 ),
  { 4530  649 }  (         0.567, 5.13926871155,    288.08069400530 ),
  { 4530  650 }  (         0.538, 0.28159637680,    153.49535039770 ),
  { 4530  651 }  (         0.718, 0.48326672359,    842.15068148810 ),
  { 4530  652 }  (         0.526, 4.39778401928,    546.15329132990 ),
  { 4530  653 }  (         0.695, 2.44235086902,    657.16276170140 ),
  { 4530  654 }  (         0.697, 4.99042365686,     12.53017297220 ),
  { 4530  655 }  (         0.519, 6.27847163164,     59.80374504030 ),
  { 4530  656 }  (         0.504, 2.58550284000,   5378.66178416220 ),
  { 4530  657 }  (         0.496, 2.43659402827,    990.22940591440 ),
  { 4530  658 }  (         0.617, 5.73284985700,    745.43774108740 ),
  { 4530  659 }  (         0.519, 3.10157097770,   9161.01716302260 ),
  { 4530  660 }  (         0.654, 1.31181453784,    878.32616366560 ),
  { 4530  661 }  (         0.619, 3.71554817226,   2090.30967237520 ),
  { 4530  662 }  (         0.500, 4.28937439066,   5216.58037280140 ),
  { 4530  663 }  (         0.621, 3.98893673383,    409.92341631960 ),
  { 4530  664 }  (         0.685, 1.95310431695,   3156.80514956520 ),
  { 4530  665 }  (         0.552, 2.81774132958,    344.70304530790 ),
  { 4530  666 }  (         0.551, 1.91969778405,    113.38771495710 ),
  { 4530  667 }  (         0.682, 0.87321578326,   6069.77675455340 ),
  { 4530  668 }  (         0.651, 5.09951064975,    531.33549649730 ),
  { 4530  669 }  (         0.537, 3.67357440226,    605.95703637020 ),
  { 4530  670 }  (         0.525, 0.74584814988,    736.83972137830 ),
  { 4530  671 }  (         0.505, 3.12494814307,   1475.68518032670 ),
  { 4530  672 }  (         0.622, 3.00013939606,   2349.32843120380 ),
  { 4530  673 }  (         0.644, 3.00156986335,    298.23262239190 ),
  { 4530  674 }  (         0.564, 3.81960833949,   1059.54198888360 ),
  { 4530  675 }  (         0.468, 3.50348554992,   4841.85727206680 ),
  { 4530  676 }  (         0.491, 1.28535573072,    247.23934538180 ),
  { 4530  677 }  (         0.458, 0.45056377876,   1065.60170531270 ),
  { 4530  678 }  (         0.543, 2.39704308320,   9690.70812811720 ),
  { 4530  679 }  (         0.459, 5.29870259698,   1474.93442080130 ),
  { 4530  680 }  (         0.483, 3.63649121244,    131.40394986990 ),
  { 4530  681 }  (         0.632, 2.75028345792,    334.55111692130 ),
  { 4530  682 }  (         0.483, 0.42979609421,    735.82832942200 ),
  { 4530  683 }  (         0.540, 0.54791737146,  51646.11531805379 ),
  { 4530  684 }  (         0.531, 0.30026207053,    912.78767847120 ),
  { 4530  685 }  (         0.449, 3.02583472996,   5901.23920225600 ),
  { 4530  686 }  (         0.544, 2.98747240952,   4223.30062675520 ),
  { 4530  687 }  (         0.557, 5.83542572008,   9676.48103411560 ),
  { 4530  688 }  (         0.501, 0.03408180117,   1080.72257119160 ),
  { 4530  689 }  (         0.517, 4.40400852026,   2545.36205125440 ),
  { 4530  690 }  (         0.481, 3.63292807076,   5584.84733259940 ),
  { 4530  691 }  (         0.557, 6.11443978190,    976.00231191280 ),
  { 4530  692 }  (         0.481, 3.41035583659,   3803.81598288000 ),
  { 4530  693 }  (         0.622, 2.29597570837,   9999.98645077300 ),
  { 4530  694 }  (         0.454, 2.88584538455,   1987.21689815660 ),
  { 4530  695 }  (         0.439, 4.83198101064,     50.40257617910 ),
  { 4530  696 }  (         0.475, 2.69994471394,    491.81856188770 ),
  { 4530  697 }  (         0.618, 0.72471290082,   1291.43097372290 ),
  { 4530  698 }  (         0.503, 0.13449993622,   2015.67108615980 ),
  { 4530  699 }  (         0.551, 2.13418546604,   1440.99417085750 ),
  { 4530  700 }  (         0.595, 3.78181802545,   6386.16862421000 ),
  { 4530  701 }  (         0.434, 2.64411689486,    748.40668650400 ),
  { 4530  702 }  (         0.592, 0.32587740408,    737.36098624010 ),
  { 4530  703 }  (         0.490, 2.37988828800,   2225.78878786010 ),
  { 4530  704 }  (         0.439, 1.33582802018,    995.64603188580 ),
  { 4530  705 }  (         0.543, 2.05067702505,    906.84978763800 ),
  { 4530  706 }  (         0.466, 2.43707405011,   3362.99069800240 ),
  { 4530  707 }  (         0.481, 2.32223226419,   1357.61455258110 ),
  { 4530  708 }  (         0.566, 0.59740900184,    350.33211960040 ),
  { 4530  709 }  (         0.429, 2.46287580628,   3914.02230409940 ),
  { 4530  710 }  (         0.429, 1.01299906509,   4333.50694797460 ),
  { 4530  711 }  (         0.425, 1.67255823369,    148.07872442630 ),
  { 4530  712 }  (         0.412, 3.29630633921,      7.32599532190 ),
  { 4530  713 }  (         0.508, 1.16158524676,      9.56122755560 ),
  { 4530  714 }  (         0.524, 5.02562926120,   1090.40141882620 ),
  { 4530  715 }  (         0.409, 5.80053072411,   9146.79006902100 ),
  { 4530  716 }  (         0.497, 0.01579913593,   1069.67687092770 ),
  { 4530  717 }  (         0.548, 6.03429743373,   9367.20271145980 ),
  { 4530  718 }  (         0.433, 5.93688350840,   1688.23351623930 ),
  { 4530  719 }  (         0.424, 4.18150111530,    550.13783421970 ),
  { 4530  720 }  (         0.401, 0.11519846139,    970.51624997220 ),
  { 4530  721 }  (         0.503, 5.28212300854,    668.20846196530 ),
  { 4530  722 }  (         0.555, 1.00328633255,    141.22580985640 ),
  { 4530  723 }  (         0.404, 2.48633976473,    519.65665678700 ),
  { 4530  724 }  (         0.441, 6.06185501734,     25.12978191360 ),
  { 4530  725 }  (         0.412, 5.87495245826,      6.90109867970 ),
  { 4530  726 }  (         0.478, 0.71264950607,   1094.80665284130 ),
  { 4530  727 }  (         0.446, 2.71248183031,     31.49256938900 ),
  { 4530  728 }  (         0.404, 5.49462012486,    447.93883187840 ),
  { 4530  729 }  (         0.391, 1.26105612700,      8.07675484730 ),
  { 4530  730 }  (         0.463, 1.93535321271,   6275.96230299060 ),
  { 4530  731 }  (         0.507, 3.61089992782,    546.95644048200 ),
  { 4530  732 }  (         0.402, 5.86200127054,    927.83496796740 ),
  { 4530  733 }  (         0.481, 6.21043578332,    683.98946464440 ),
  { 4530  734 }  (         0.483, 5.02142924458,    857.12853501510 ),
  { 4530  735 }  (         0.444, 0.84873092377,   1371.84164658270 ),
  { 4530  736 }  (         0.391, 2.81753436573,   5798.14642803740 ),
  { 4530  737 }  (         0.395, 0.22367886581,  51116.42435295920 ),
  { 4530  738 }  (         0.378, 6.03765733432,   1268.74887236410 ),
  { 4530  739 }  (         0.471, 6.24506463249,    946.46729598410 ),
  { 4530  740 }  (         0.405, 0.57785207581,    107.28555991260 ),
  { 4530  741 }  (         0.371, 6.15750793727,    509.24409596950 ),
  { 4530  742 }  (         0.370, 4.90330687618,   1436.54075273260 ),
  { 4530  743 }  (         0.448, 4.76565111029,    284.14854074220 ),
  { 4530  744 }  (         0.474, 0.71146352197,   2108.46891963990 ),
  { 4530  745 }  (         0.509, 5.53328407404,   1128.53445446400 )
  (*$endif *)
      );
  (*@\\\0000003701*)
  (*@/// vsop87_jup_r1:array[0..380,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_r1:array[0.. 42,0..2] of extended = (
  (*$else *)
  vsop87_jup_r1:array[0..380,0..2] of extended = (
  (*$endif *)
  { 4531    1 }  (   1271801.596, 2.64937511122,    529.69096509460 ),
  { 4531    2 }  (     61661.771, 3.00076251018,   1059.38193018920 ),
  { 4531    3 }  (     53443.592, 3.89717644226,    522.57741809380 ),
  { 4531    4 }  (     31185.167, 4.88276663526,    536.80451209540 ),
  { 4531    5 }  (     41390.257, 0.00000000000,      0.00000000000 ),
  { 4531    6 }  (     11847.190, 2.41329588176,    419.48464387520 ),
  { 4531    7 }  (      9166.360, 4.75979408587,      7.11354700080 ),
  { 4531    8 }  (      3175.763, 2.79297987071,    103.09277421860 ),
  { 4531    9 }  (      3203.446, 5.21083285476,    735.87651353180 ),
  { 4531   10 }  (      3403.605, 3.34688537997,   1589.07289528380 ),
  { 4531   11 }  (      2600.003, 3.63435101622,    206.18554843720 ),
  { 4531   12 }  (      2412.207, 1.46947308304,    426.59819087600 ),
  { 4531   13 }  (      2806.064, 3.74223693580,    515.46387109300 ),
  { 4531   14 }  (      2676.575, 4.33052878699,   1052.26838318840 ),
  { 4531   15 }  (      2100.507, 3.92762682306,    639.89728631400 ),
  { 4531   16 }  (      1646.182, 5.30953510947,   1066.49547719000 ),
  { 4531   17 }  (      1641.257, 4.41628669824,    625.67019231240 ),
  { 4531   18 }  (      1049.866, 3.16113622955,    213.29909543800 ),
  { 4531   19 }  (      1024.802, 2.55432643018,    412.37109687440 ),
  { 4531   20 }  (       740.996, 2.17094630558,   1162.47470440780 ),
  { 4531   21 }  (       806.404, 2.67750801380,    632.78373931320 ),
  { 4531   22 }  (       676.928, 6.24953479790,    838.96928775040 ),
  { 4531   23 }  (       468.895, 4.70973463481,    543.91805909620 ),
  { 4531   24 }  (       444.683, 0.40281181402,    323.50541665740 ),
  { 4531   25 }  (       567.076, 4.57655414712,    742.99006053260 ),
  { 4531   26 }  (       415.894, 5.36836018215,    728.76296653100 ),
  { 4531   27 }  (       484.689, 2.46882793186,    949.17560896980 ),
  { 4531   28 }  (       337.555, 3.16781951120,    956.28915597060 ),
  { 4531   29 }  (       401.738, 4.60528841541,    309.27832265580 ),
  { 4531   30 }  (       347.378, 4.68148808722,     14.22709400160 ),
  { 4531   31 }  (       260.753, 5.34290306101,    846.08283475120 ),
  { 4531   32 }  (       220.084, 4.84210964963,   1368.66025284500 ),
  { 4531   33 }  (       203.217, 5.59995425432,   1155.36115740700 ),
  { 4531   34 }  (       246.603, 3.92313823537,    942.06206196900 ),
  { 4531   35 }  (       183.504, 4.26526769703,     95.97922721780 ),
  { 4531   36 }  (       180.134, 4.40165491159,    532.87235883230 ),
  { 4531   37 }  (       197.134, 3.70551461394,   2118.76386037840 ),
  { 4531   38 }  (       196.005, 3.75877587139,    199.07200143640 ),
  { 4531   39 }  (       200.190, 4.43888814441,   1045.15483618760 ),
  { 4531   40 }  (       170.225, 4.84647488867,    526.50957135690 ),
  { 4531   41 }  (       146.335, 6.12958365535,    533.62311835770 ),
  { 4531   42 }  (       133.483, 1.32245735855,    110.20632121940 ),
  { 4531   43 }  (       132.076, 4.51187950811,    525.75881183150 )
  (*$ifndef meeus *)                                                 ,
  { 4531   44 }  (       123.851, 2.04290370696,   1478.86657406440 ),
  { 4531   45 }  (       121.861, 4.40581788491,   1169.58825140860 ),
  { 4531   46 }  (       115.313, 4.46741278152,   1581.95934828300 ),
  { 4531   47 }  (        98.527, 5.72833991647,   1596.18644228460 ),
  { 4531   48 }  (        91.608, 4.52965592121,   1685.05212250160 ),
  { 4531   49 }  (       110.638, 3.62504147403,   1272.68102562720 ),
  { 4531   50 }  (        80.536, 4.11311699583,   1258.45393162560 ),
  { 4531   51 }  (        79.552, 2.71898473954,   1692.16566950240 ),
  { 4531   52 }  (       100.164, 5.24693885858,   1265.56747862640 ),
  { 4531   53 }  (        77.854, 5.56722651753,   1471.75302706360 ),
  { 4531   54 }  (        85.766, 0.07906707372,    831.85574074960 ),
  { 4531   55 }  (        82.132, 3.80763015979,    508.35032409220 ),
  { 4531   56 }  (        55.319, 0.35180851191,    316.39186965660 ),
  { 4531   57 }  (        52.338, 5.53074272117,    433.71173787680 ),
  { 4531   58 }  (        55.769, 4.75141241141,    302.16477565500 ),
  { 4531   59 }  (        50.597, 4.85603161770,   1375.77379984580 ),
  { 4531   60 }  (        43.554, 4.94441642712,   1361.54670584420 ),
  { 4531   61 }  (        42.172, 1.22404278447,    853.19638175200 ),
  { 4531   62 }  (        37.695, 4.26767539209,   2001.44399215820 ),
  { 4531   63 }  (        49.395, 4.01422828967,    220.41264243880 ),
  { 4531   64 }  (        38.263, 5.33025236797,   1788.14489672020 ),
  { 4531   65 }  (        35.611, 1.76205571128,   1795.25844372100 ),
  { 4531   66 }  (        36.296, 3.84995284393,   1574.84580128220 ),
  { 4531   67 }  (        29.332, 5.16619257786,      3.93215326310 ),
  { 4531   68 }  (        25.180, 4.33777727362,    519.39602435610 ),
  { 4531   69 }  (        24.778, 2.72907897410,    405.25754987360 ),
  { 4531   70 }  (        27.025, 6.09669947903,   1148.24761040620 ),
  { 4531   71 }  (        22.604, 0.19173890105,    380.12776796000 ),
  { 4531   72 }  (        20.499, 4.32881495378,      3.18139373770 ),
  { 4531   73 }  (        19.925, 4.62967500111,   1677.93857550080 ),
  { 4531   74 }  (        19.528, 5.10596326232,   1073.60902419080 ),
  { 4531   75 }  (        18.427, 3.76522178300,   1485.98012106520 ),
  { 4531   76 }  (        18.869, 5.05259402407,   2104.53676637680 ),
  { 4531   77 }  (        17.031, 4.01843356903,   2317.83586181480 ),
  { 4531   78 }  (        16.671, 5.42931676507,     88.86568021700 ),
  { 4531   79 }  (        15.337, 2.92700926091,   2008.55753915900 ),
  { 4531   80 }  (        14.499, 3.63339836845,    628.85158605010 ),
  { 4531   81 }  (        14.575, 5.50832843322,    721.64941953020 ),
  { 4531   82 }  (        13.728, 4.87623389735,    629.60234557550 ),
  { 4531   83 }  (        18.481, 6.03032762264,    330.61896365820 ),
  { 4531   84 }  (        13.499, 1.38539534821,    518.64526483070 ),
  { 4531   85 }  (        15.740, 2.93038271684,   1905.46476494040 ),
  { 4531   86 }  (        12.459, 1.58587053146,   2111.65031337760 ),
  { 4531   87 }  (        12.272, 3.37671053917,    635.96513305090 ),
  { 4531   88 }  (        11.836, 4.08486322993,   2648.45482547300 ),
  { 4531   89 }  (        11.166, 4.62623267608,    636.71589257630 ),
  { 4531   90 }  (        14.348, 2.74177797727,   2221.85663459700 ),
  { 4531   91 }  (        11.221, 3.55311861205,   1891.23767093880 ),
  { 4531   92 }  (        13.121, 5.83845065644,   1464.63948006280 ),
  { 4531   93 }  (        11.351, 2.57606886230,    511.53171782990 ),
  { 4531   94 }  (        10.487, 0.49850799841,    453.42489381900 ),
  { 4531   95 }  (         9.728, 4.38837468002,   1994.33044515740 ),
  { 4531   96 }  (        10.131, 2.76432756215,    423.41679713830 ),
  { 4531   97 }  (         8.620, 5.16374493158,   1056.20053645150 ),
  { 4531   98 }  (         8.952, 4.79407952752,   2420.92863603340 ),
  { 4531   99 }  (         8.126, 3.72977106954,   2634.22773147140 ),
  { 4531  100 }  (         8.078, 1.29246272894,   2428.04218303420 ),
  { 4531  101 }  (         8.867, 1.85684753622,    750.10360753340 ),
  { 4531  102 }  (         8.912, 4.80973516711,   1062.56332392690 ),
  { 4531  103 }  (         8.552, 4.53818617984,     21.34064100240 ),
  { 4531  104 }  (         9.468, 4.33472161983,   1802.37199072180 ),
  { 4531  105 }  (         6.904, 5.96616555709,    540.73666535850 ),
  { 4531  106 }  (         7.293, 4.97763580465,   1699.27921650320 ),
  { 4531  107 }  (         7.083, 4.99096728816,   1055.44977692610 ),
  { 4531  108 }  (         7.226, 4.97823884383,   1898.35121793960 ),
  { 4531  109 }  (         6.464, 1.39173466879,    422.66603761290 ),
  { 4531  110 }  (         6.214, 4.46490158256,    551.03160609700 ),
  { 4531  111 }  (         6.794, 2.90878831415,   2324.94940881560 ),
  { 4531  112 }  (         6.173, 3.65617162985,    621.73803904930 ),
  { 4531  113 }  (         6.243, 6.13691919694,   2125.87740737920 ),
  { 4531  114 }  (         5.936, 2.58312235120,    569.04784100980 ),
  { 4531  115 }  (         6.504, 4.56908431757,   1038.04128918680 ),
  { 4531  116 }  (         7.305, 3.02062127734,    416.30325013750 ),
  { 4531  117 }  (         6.598, 5.55348005731,   1781.03134971940 ),
  { 4531  118 }  (         5.133, 6.21646917980,    963.40270297140 ),
  { 4531  119 }  (         5.876, 4.23153077453,    539.98590583310 ),
  { 4531  120 }  (         5.119, 0.06942832171,   1063.31408345230 ),
  { 4531  121 }  (         5.460, 4.91084384602,    835.03713448730 ),
  { 4531  122 }  (         4.989, 1.35153694680,   1382.88734684660 ),
  { 4531  123 }  (         5.224, 0.18468411116,    117.31986822020 ),
  { 4531  124 }  (         6.187, 3.87193497099,    191.95845443560 ),
  { 4531  125 }  (         4.681, 4.61057119508,    643.82943957710 ),
  { 4531  126 }  (         4.627, 3.34644534691,   2207.62954059540 ),
  { 4531  127 }  (         4.526, 4.07729737127,   2310.72231481400 ),
  { 4531  128 }  (         4.718, 4.55578336947,   2737.32050569000 ),
  { 4531  129 }  (         4.471, 1.47603161897,    408.43894361130 ),
  { 4531  130 }  (         4.073, 1.13014903180,    415.55249061210 ),
  { 4531  131 }  (         5.476, 5.63198569698,    618.55664531160 ),
  { 4531  132 }  (         4.034, 4.09631702747,    430.53034413910 ),
  { 4531  133 }  (         4.304, 4.60536378943,    647.01083331480 ),
  { 4531  134 }  (         3.765, 3.42751259825,   2950.61960112800 ),
  { 4531  135 }  (         4.559, 4.23723998745,    227.52618943960 ),
  { 4531  136 }  (         3.695, 1.03127824978,   2744.43405269080 ),
  { 4531  137 }  (         3.667, 4.12268925541,    440.82528487760 ),
  { 4531  138 }  (         3.677, 2.19480200527,    534.35683154060 ),
  { 4531  139 }  (         3.818, 1.14800596289,     74.78159856730 ),
  { 4531  140 }  (         4.221, 2.37721579949,   2538.24850425360 ),
  { 4531  141 }  (         3.488, 5.33792561596,    458.84151979040 ),
  { 4531  142 }  (         3.437, 4.26164443643,     10.29494073850 ),
  { 4531  143 }  (         4.394, 0.18808423412,    824.74219374880 ),
  { 4531  144 }  (         3.339, 4.85708402591,    295.05122865420 ),
  { 4531  145 }  (         3.329, 5.50043586719,    739.80866679490 ),
  { 4531  146 }  (         3.623, 4.64011531952,   2214.74308759620 ),
  { 4531  147 }  (         3.185, 2.69708590442,    561.93429400900 ),
  { 4531  148 }  (         3.421, 3.38512615384,    149.56319713460 ),
  { 4531  149 }  (         3.442, 4.34217280083,    305.34616939270 ),
  { 4531  150 }  (         3.580, 5.29481665335,   2097.42321937600 ),
  { 4531  151 }  (         3.401, 2.74761862893,   2641.34127847220 ),
  { 4531  152 }  (         2.901, 0.91012525424,    984.60033162190 ),
  { 4531  153 }  (         3.566, 1.63400343968,    525.02509864860 ),
  { 4531  154 }  (         2.869, 1.31799241974,    611.44309831080 ),
  { 4531  155 }  (         2.635, 5.25517910535,    532.13864564940 ),
  { 4531  156 }  (         2.683, 4.24641945773,   3053.71237534660 ),
  { 4531  157 }  (         2.614, 3.17862099921,    527.24328453980 ),
  { 4531  158 }  (         2.251, 4.21598247360,    739.05790726950 ),
  { 4531  159 }  (         2.268, 5.52248110560,    524.27433912320 ),
  { 4531  160 }  (         2.372, 4.19741177512,    217.23124870110 ),
  { 4531  161 }  (         2.623, 5.82647427958,    732.69511979410 ),
  { 4531  162 }  (         2.666, 3.92538056951,    210.11770170030 ),
  { 4531  163 }  (         2.036, 4.84043420813,   1049.08698945070 ),
  { 4531  164 }  (         2.441, 2.63840901843,    760.25553592000 ),
  { 4531  165 }  (         2.095, 5.76269812349,    529.64278098480 ),
  { 4531  166 }  (         2.021, 3.81308146017,   2627.11418447060 ),
  { 4531  167 }  (         2.089, 4.18463193132,    945.99421523210 ),
  { 4531  168 }  (         2.305, 1.61220665690,    604.47256366190 ),
  { 4531  169 }  (         1.969, 5.37427735384,    142.44965013380 ),
  { 4531  170 }  (         1.923, 4.75088270631,    535.10759106600 ),
  { 4531  171 }  (         1.955, 5.49000238006,   1439.50969814920 ),
  { 4531  172 }  (         1.877, 3.26978877187,   3267.01147078460 ),
  { 4531  173 }  (         2.286, 2.93885172004,     76.26607127560 ),
  { 4531  174 }  (         2.074, 5.85386852879,    532.61172640140 ),
  { 4531  175 }  (         2.121, 3.92430797099,   2435.15573003500 ),
  { 4531  176 }  (         1.807, 3.17208959472,   2524.02141025200 ),
  { 4531  177 }  (         1.712, 4.02986641257,    731.94436026870 ),
  { 4531  178 }  (         2.119, 0.41049593984,   1279.79457262800 ),
  { 4531  179 }  (         1.660, 2.34370903423,    528.72775724810 ),
  { 4531  180 }  (         1.655, 0.78809717175,   3060.82592234740 ),
  { 4531  181 }  (         1.729, 4.26127896267,    724.83081326790 ),
  { 4531  182 }  (         2.060, 5.04785330873,   2413.81508903260 ),
  { 4531  183 }  (         2.095, 2.67732367556,    529.73914920440 ),
  { 4531  184 }  (         1.933, 2.49162437046,   2957.73314812880 ),
  { 4531  185 }  (         1.898, 2.71948262975,    952.35700270750 ),
  { 4531  186 }  (         1.634, 2.98113068812,    945.24345570670 ),
  { 4531  187 }  (         1.582, 5.84373095005,    547.85021235930 ),
  { 4531  188 }  (         1.662, 0.27359627181,    454.90936652730 ),
  { 4531  189 }  (         1.595, 1.18530167095,     38.13303563780 ),
  { 4531  190 }  (         1.550, 0.64264572959,    312.45971639350 ),
  { 4531  191 }  (         1.525, 4.08789824989,   1158.54255114470 ),
  { 4531  192 }  (         1.542, 1.12520322326,   1021.24889455140 ),
  { 4531  193 }  (         1.539, 0.37324921979,    319.57326339430 ),
  { 4531  194 }  (         1.628, 5.24285773388,   1354.43315884340 ),
  { 4531  195 }  (         1.897, 3.79973291113,    953.10776223290 ),
  { 4531  196 }  (         1.440, 4.37872256685,   3178.14579056760 ),
  { 4531  197 }  (         1.439, 4.26513521887,    526.77020378780 ),
  { 4531  198 }  (         1.557, 5.43779802371,     81.75213321620 ),
  { 4531  199 }  (         1.656, 6.09667089740,    530.65417294110 ),
  { 4531  200 }  (         1.548, 3.48799710267,    934.94851496820 ),
  { 4531  201 }  (         1.772, 5.82549274759,    909.81873305460 ),
  { 4531  202 }  (         1.615, 1.45018725033,    902.70518605380 ),
  { 4531  203 }  (         1.387, 2.52840497309,    530.44172462000 ),
  { 4531  204 }  (         1.574, 1.89565809136,    437.64389113990 ),
  { 4531  205 }  (         1.459, 3.32546061506,   1041.22268292450 ),
  { 4531  206 }  (         1.377, 0.10015418633,    490.33408917940 ),
  { 4531  207 }  (         1.460, 4.00706825185,   3370.10424500320 ),
  { 4531  208 }  (         1.605, 4.27993020192,   2531.13495725280 ),
  { 4531  209 }  (         1.707, 6.28253681644,     18.15924726470 ),
  { 4531  210 }  (         1.802, 2.23019296374,   2854.64037391020 ),
  { 4531  211 }  (         1.390, 3.76737324192,   1165.65609814550 ),
  { 4531  212 }  (         1.498, 0.17285954362,   1141.13406340540 ),
  { 4531  213 }  (         1.401, 4.81225317549,   1251.34038462480 ),
  { 4531  214 }  (         1.244, 2.83383980283,    124.43341522100 ),
  { 4531  215 }  (         1.320, 5.80675430384,    387.24131496080 ),
  { 4531  216 }  (         1.329, 0.88314574243,    916.93228005540 ),
  { 4531  217 }  (         1.558, 6.17808619637,    983.11585891360 ),
  { 4531  218 }  (         1.243, 0.29239666059,    597.35901666110 ),
  { 4531  219 }  (         1.541, 3.51095241498,   2751.54759969160 ),
  { 4531  220 }  (         1.482, 0.83066678204,    529.16970023280 ),
  { 4531  221 }  (         1.149, 3.91142023857,     99.91138048090 ),
  { 4531  222 }  (         1.114, 3.53339637290,    483.22054217860 ),
  { 4531  223 }  (         1.195, 4.16301075999,    203.00415469950 ),
  { 4531  224 }  (         1.100, 1.74769285223,    497.44763618020 ),
  { 4531  225 }  (         1.458, 5.19315120878,   1592.25428902150 ),
  { 4531  226 }  (         1.123, 1.45270581179,    533.88375078860 ),
  { 4531  227 }  (         1.078, 5.23991792940,   1159.29331067010 ),
  { 4531  228 }  (         1.083, 3.57026506855,   2943.50605412720 ),
  { 4531  229 }  (         1.072, 0.07132659992,   1070.42763045310 ),
  { 4531  230 }  (         1.037, 5.48955598976,   1585.89150154610 ),
  { 4531  231 }  (         1.343, 0.29600445633,    860.30992875280 ),
  { 4531  232 }  (         1.361, 3.46603373194,    107.02492748170 ),
  { 4531  233 }  (         1.061, 2.44580706826,   1048.33622992530 ),
  { 4531  234 }  (         1.002, 5.55216117410,    337.73251065900 ),
  { 4531  235 }  (         0.981, 3.15500987023,     70.84944530420 ),
  { 4531  236 }  (         1.007, 4.11504050436,    501.23677709140 ),
  { 4531  237 }  (         0.965, 5.63719524421,   1603.29998928540 ),
  { 4531  238 }  (         1.083, 4.88373909810,   1166.40685767090 ),
  { 4531  239 }  (         0.953, 2.83352026342,   3583.40334044120 ),
  { 4531  240 }  (         1.060, 3.18542176646,    447.79581952650 ),
  { 4531  241 }  (         1.136, 2.26568590950,    525.49817940060 ),
  { 4531  242 }  (         1.191, 2.25249961404,    106.27416795630 ),
  { 4531  243 }  (         0.884, 4.69777781327,    960.22130923370 ),
  { 4531  244 }  (         1.165, 1.56030440737,    630.33605875840 ),
  { 4531  245 }  (         0.947, 0.50856414717,    842.90144101350 ),
  { 4531  246 }  (         1.011, 0.30814674949,   1593.00504854690 ),
  { 4531  247 }  (         0.924, 2.31939900786,    327.43756992050 ),
  { 4531  248 }  (         0.896, 0.22222521202,    746.92221379570 ),
  { 4531  249 }  (         1.078, 4.78329116086,   2730.20695868920 ),
  { 4531  250 }  (         0.938, 5.42471506763,   1585.14074202070 ),
  { 4531  251 }  (         0.923, 4.44469169065,   9676.48103411560 ),
  { 4531  252 }  (         0.894, 0.26940821870,   2655.56837247380 ),
  { 4531  253 }  (         1.131, 5.46382510304,    224.34479570190 ),
  { 4531  254 }  (         0.808, 0.48295590141,   3377.21779200400 ),
  { 4531  255 }  (         0.809, 4.14122746067,    114.13847448250 ),
  { 4531  256 }  (         0.864, 1.83217006136,      4.66586644600 ),
  { 4531  257 }  (         1.106, 2.60444312553,    209.36694217490 ),
  { 4531  258 }  (         0.790, 0.11493626208,    460.53844081980 ),
  { 4531  259 }  (         0.799, 1.60426497590,   5223.69391980220 ),
  { 4531  260 }  (         0.933, 0.30976125598,    685.47393735270 ),
  { 4531  261 }  (         1.053, 5.23433104008,    842.15068148810 ),
  { 4531  262 }  (         0.846, 3.02878393490,   5746.27133789600 ),
  { 4531  263 }  (         0.799, 2.08457026425,  77734.01845962799 ),
  { 4531  264 }  (         0.820, 0.99821486743,    373.01422095920 ),
  { 4531  265 }  (         0.892, 5.36446426391,    827.92358748650 ),
  { 4531  266 }  (         0.821, 3.53889274951,    498.67147645760 ),
  { 4531  267 }  (         0.741, 1.32379374647,    530.21222995640 ),
  { 4531  268 }  (         0.790, 2.88034567513,    938.12990870590 ),
  { 4531  269 }  (         0.842, 3.39449778904,    484.44438245600 ),
  { 4531  270 }  (         0.785, 0.57841470897,    850.01498801430 ),
  { 4531  271 }  (         0.759, 3.82014112009,   6283.07584999140 ),
  { 4531  272 }  (         0.954, 2.94534072982,    462.02291352810 ),
  { 4531  273 }  (         0.767, 3.33725133157,     99.16062095550 ),
  { 4531  274 }  (         0.810, 4.69425300466,   2228.97018159780 ),
  { 4531  275 }  (         0.700, 1.72050221502,    775.23338944700 ),
  { 4531  276 }  (         0.764, 4.91747674296,   1670.82502850000 ),
  { 4531  277 }  (         0.724, 6.08692841992,   2281.23049651060 ),
  { 4531  278 }  (         0.711, 4.82250918143,     11.77941344680 ),
  { 4531  279 }  (         0.692, 2.63705354662,      6.59228213900 ),
  { 4531  280 }  (         0.771, 3.87410612014,   9690.70812811720 ),
  { 4531  281 }  (         0.906, 2.47189948442,   3274.12501778540 ),
  { 4531  282 }  (         0.781, 1.25357484582,    202.25339517410 ),
  { 4531  283 }  (         0.757, 3.78079814332,   2818.03500860600 ),
  { 4531  284 }  (         0.756, 4.28312053897,   2803.80791460440 ),
  { 4531  285 }  (         0.663, 5.27704405712,   4532.57894941100 ),
  { 4531  286 }  (         0.759, 5.45358686570,   9683.59458111640 ),
  { 4531  287 }  (         0.698, 5.43712520216,    565.11568774670 ),
  { 4531  288 }  (         0.709, 3.71117647887,   3686.49611465980 ),
  { 4531  289 }  (         0.677, 4.27891183416,  25028.52121138500 ),
  { 4531  290 }  (         0.643, 1.40239510103,   9161.01716302260 ),
  { 4531  291 }  (         0.656, 0.60909845504,    835.78789401270 ),
  { 4531  292 }  (         0.635, 5.75373871128,    429.77958461370 ),
  { 4531  293 }  (         0.702, 6.10412979847,   4635.67172362960 ),
  { 4531  294 }  (         0.627, 3.03666956129,   2840.41327990860 ),
  { 4531  295 }  (         0.802, 4.18688054701,   5753.38488489680 ),
  { 4531  296 }  (         0.838, 4.51386507097,   1069.67687092770 ),
  { 4531  297 }  (         0.633, 4.37183361444,      5.41662597140 ),
  { 4531  298 }  (         0.652, 5.79409889124,   1061.82961074400 ),
  { 4531  299 }  (         0.638, 2.18896270346,    313.21047591890 ),
  { 4531  300 }  (         0.827, 5.94231186039,   1457.52593306200 ),
  { 4531  301 }  (         0.678, 2.45013730979,   5760.49843189760 ),
  { 4531  302 }  (         0.814, 4.89578791170,   1567.73225428140 ),
  { 4531  303 }  (         0.624, 0.61631100566,   1176.70179840940 ),
  { 4531  304 }  (         0.600, 3.20918322285,   1098.73880610440 ),
  { 4531  305 }  (         0.717, 1.82349064490,   3171.03224356680 ),
  { 4531  306 }  (         0.651, 4.14419317491,   2847.52682690940 ),
  { 4531  307 }  (         0.629, 1.75272560843,     92.04707395470 ),
  { 4531  308 }  (         0.626, 3.53146082217,   3067.93946934820 ),
  { 4531  309 }  (         0.667, 4.22974611158,   4539.69249641180 ),
  { 4531  310 }  (         0.565, 0.99416346033,   1894.41906467650 ),
  { 4531  311 }  (         0.752, 0.46063700150,    635.23141986800 ),
  { 4531  312 }  (         0.622, 1.98136818407,  25565.32572348040 ),
  { 4531  313 }  (         0.614, 2.48275371627,  25551.09862947879 ),
  { 4531  314 }  (         0.560, 1.40733893388,    446.31134681820 ),
  { 4531  315 }  (         0.558, 4.37217796469,   1057.89745748090 ),
  { 4531  316 }  (         0.628, 4.65037810102,   6275.96230299060 ),
  { 4531  317 }  (         0.659, 2.41470950463,    195.13984817330 ),
  { 4531  318 }  (         0.616, 2.08837621877,     10.03430830760 ),
  { 4531  319 }  (         0.692, 3.13229025530,      7.63481186260 ),
  { 4531  320 }  (         0.685, 4.18539472904,     46.47042291600 ),
  { 4531  321 }  (         0.624, 0.02693303471,   1493.09366806600 ),
  { 4531  322 }  (         0.594, 2.13375704438,    121.25202148330 ),
  { 4531  323 }  (         0.508, 2.13584300710,      1.69692102940 ),
  { 4531  324 }  (         0.674, 1.47570122611,   4694.00295470760 ),
  { 4531  325 }  (         0.559, 4.48852017557,    531.17543780290 ),
  { 4531  326 }  (         0.640, 3.10239233469,     11.04570026390 ),
  { 4531  327 }  (         0.496, 1.29000001439,    927.83496796740 ),
  { 4531  328 }  (         0.587, 3.30651435298,    600.54041039880 ),
  { 4531  329 }  (         0.582, 0.44540948860,    113.38771495710 ),
  { 4531  330 }  (         0.492, 4.83275232000,   9492.14631500480 ),
  { 4531  331 }  (         0.549, 4.34579166146,   3046.59882834580 ),
  { 4531  332 }  (         0.576, 1.22846846364,   1514.29129671650 ),
  { 4531  333 }  (         0.593, 5.86079640612,    524.06189080210 ),
  { 4531  334 }  (         0.510, 2.62557031270,    529.85102378900 ),
  { 4531  335 }  (         0.489, 6.26855707323,   3693.60966166060 ),
  { 4531  336 }  (         0.480, 0.30754294369,    528.94020556920 ),
  { 4531  337 }  (         0.582, 3.51934668795,   1056.93424963440 ),
  { 4531  338 }  (         0.493, 5.52699906925,    512.28247735530 ),
  { 4531  339 }  (         0.481, 2.99681040149,   9153.90361602180 ),
  { 4531  340 }  (         0.562, 3.73437025868,   2015.67108615980 ),
  { 4531  341 }  (         0.458, 3.86646994292,     11.30633269480 ),
  { 4531  342 }  (         0.457, 1.80238019931,   3281.23856478620 ),
  { 4531  343 }  (         0.453, 6.17995938655,   1059.33374607940 ),
  { 4531  344 }  (         0.551, 0.13794958618,   1912.57831194120 ),
  { 4531  345 }  (         0.446, 5.53828660924,   2332.06295581640 ),
  { 4531  346 }  (         0.444, 5.06219342598,      7.86430652620 ),
  { 4531  347 }  (         0.461, 0.16951411708,  26087.90314157420 ),
  { 4531  348 }  (         0.439, 4.14986379679,   1151.42900414390 ),
  { 4531  349 }  (         0.614, 5.42289673768,   2090.30967237520 ),
  { 4531  350 }  (         0.488, 3.71681959056,    447.93883187840 ),
  { 4531  351 }  (         0.592, 2.91424148255,   8624.21265092720 ),
  { 4531  352 }  (         0.433, 2.55336268329,   1064.04779663520 ),
  { 4531  353 }  (         0.449, 5.24955106938,  10213.28554621100 ),
  { 4531  354 }  (         0.510, 5.81591864532,    529.53090640020 ),
  { 4531  355 }  (         0.435, 5.34355963629,    560.71045373160 ),
  { 4531  356 }  (         0.449, 0.72330388784,   2758.66114669240 ),
  { 4531  357 }  (         0.430, 0.94519103478,      6.36278747540 ),
  { 4531  358 }  (         0.563, 6.19175228344,   1884.12412393800 ),
  { 4531  359 }  (         0.443, 3.39246520261,   1152.17976366930 ),
  { 4531  360 }  (         0.430, 1.28652623263,    505.31194270640 ),
  { 4531  361 }  (         0.422, 5.12631540623,    944.98282327580 ),
  { 4531  362 }  (         0.464, 2.90444584145,    398.14400287280 ),
  { 4531  363 }  (         0.410, 1.24248975309,   5069.38346150640 ),
  { 4531  364 }  (         0.411, 2.95117124177,   4326.39340097380 ),
  { 4531  365 }  (         0.418, 5.15499986314,   1173.52040467170 ),
  { 4531  366 }  (         0.412, 2.98125446330,    554.06998748280 ),
  { 4531  367 }  (         0.403, 0.34381388674,     32.24332891440 ),
  { 4531  368 }  (         0.402, 5.88926765351,   1570.91364801910 ),
  { 4531  369 }  (         0.505, 1.49028912471,   3782.47534187760 ),
  { 4531  370 }  (         0.447, 0.03952029309,    245.54242435240 ),
  { 4531  371 }  (         0.453, 3.09458004153,   1059.43011429900 ),
  { 4531  372 }  (         0.411, 3.21727542472,   1475.68518032670 ),
  { 4531  373 }  (         0.426, 3.12237794195,  12566.15169998280 ),
  { 4531  374 }  (         0.434, 3.59362426939,   3259.89792378380 ),
  { 4531  375 }  (         0.398, 4.91510709622,   4120.20785253660 ),
  { 4531  376 }  (         0.399, 4.67075122011,    234.63973644040 ),
  { 4531  377 }  (         0.386, 4.81320787761,    970.51624997220 ),
  { 4531  378 }  (         0.427, 3.21176085113,    977.48678462110 ),
  { 4531  379 }  (         0.411, 4.31566962034,    757.21715453420 ),
  { 4531  380 }  (         0.392, 1.86527946688,    885.43971066640 ),
  { 4531  381 }  (         0.416, 3.81408093105,   3156.80514956520 )
  (*$endif *)
      );
  (*@\\\0000003101*)
  (*@/// vsop87_jup_r2:array[0..189,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_r2:array[0.. 35,0..2] of extended = (
  (*$else *)
  vsop87_jup_r2:array[0..189,0..2] of extended = (
  (*$endif *)
  { 4532    1 }  (     79644.833, 1.35865896596,    529.69096509460 ),
  { 4532    2 }  (      8251.618, 5.77773935444,    522.57741809380 ),
  { 4532    3 }  (      7029.864, 3.27476965833,    536.80451209540 ),
  { 4532    4 }  (      5314.006, 1.83835109712,   1059.38193018920 ),
  { 4532    5 }  (      1860.833, 2.97682139367,      7.11354700080 ),
  { 4532    6 }  (       836.267, 4.19889881718,    419.48464387520 ),
  { 4532    7 }  (       964.466, 5.48031822015,    515.46387109300 ),
  { 4532    8 }  (       406.453, 3.78250730354,   1066.49547719000 ),
  { 4532    9 }  (       426.570, 2.22753101795,    639.89728631400 ),
  { 4532   10 }  (       377.316, 2.24248352873,   1589.07289528380 ),
  { 4532   11 }  (       497.920, 3.14159265359,      0.00000000000 ),
  { 4532   12 }  (       339.043, 6.12690864038,    625.67019231240 ),
  { 4532   13 }  (       362.943, 5.36761847267,    206.18554843720 ),
  { 4532   14 }  (       342.048, 6.09922969324,   1052.26838318840 ),
  { 4532   15 }  (       279.920, 4.26162555827,    412.37109687440 ),
  { 4532   16 }  (       332.578, 0.00328961161,    426.59819087600 ),
  { 4532   17 }  (       229.777, 0.70530766213,    735.87651353180 ),
  { 4532   18 }  (       200.783, 3.06850623368,    543.91805909620 ),
  { 4532   19 }  (       199.807, 4.42884165317,    103.09277421860 ),
  { 4532   20 }  (       257.290, 0.96295364983,    632.78373931320 ),
  { 4532   21 }  (       138.606, 2.93235671606,     14.22709400160 ),
  { 4532   22 }  (       113.535, 0.78713911289,    728.76296653100 ),
  { 4532   23 }  (        86.025, 5.14434751994,    323.50541665740 ),
  { 4532   24 }  (        94.565, 1.70498041073,    838.96928775040 ),
  { 4532   25 }  (        83.469, 0.05834873484,    309.27832265580 ),
  { 4532   26 }  (        75.198, 1.60495195911,    956.28915597060 ),
  { 4532   27 }  (        70.451, 1.50988357484,    213.29909543800 ),
  { 4532   28 }  (        80.328, 2.98122361797,    742.99006053260 ),
  { 4532   29 }  (        56.203, 0.95534810533,   1162.47470440780 ),
  { 4532   30 }  (        61.649, 6.10137889854,   1045.15483618760 ),
  { 4532   31 }  (        66.572, 5.47307178077,    199.07200143640 ),
  { 4532   32 }  (        50.057, 2.72063162317,    532.87235883230 ),
  { 4532   33 }  (        51.904, 5.58435625607,    942.06206196900 ),
  { 4532   34 }  (        39.833, 5.94566506227,     95.97922721780 ),
  { 4532   35 }  (        44.548, 5.52445621411,    508.35032409220 ),
  { 4532   36 }  (        44.282, 0.27118152557,    526.50957135690 )
  (*$ifndef meeus *)                                                 ,
  { 4532   37 }  (        29.944, 0.93641735919,   1155.36115740700 ),
  { 4532   38 }  (        28.412, 2.87835720211,    525.75881183150 ),
  { 4532   39 }  (        26.330, 4.26891877269,   1596.18644228460 ),
  { 4532   40 }  (        27.039, 2.80607741398,   1169.58825140860 ),
  { 4532   41 }  (        27.477, 2.64841266238,   2118.76386037840 ),
  { 4532   42 }  (        22.705, 0.17830004133,    302.16477565500 ),
  { 4532   43 }  (        29.347, 1.78589692350,    831.85574074960 ),
  { 4532   44 }  (        19.991, 0.04328951895,    949.17560896980 ),
  { 4532   45 }  (        19.906, 1.16072627347,    533.62311835770 ),
  { 4532   46 }  (        21.714, 1.88820231818,   1272.68102562720 ),
  { 4532   47 }  (        17.581, 4.14974757919,    846.08283475120 ),
  { 4532   48 }  (        17.085, 5.89188996975,   1258.45393162560 ),
  { 4532   49 }  (        21.407, 4.35468497204,    316.39186965660 ),
  { 4532   50 }  (        21.295, 0.54429472455,   1265.56747862640 ),
  { 4532   51 }  (        19.859, 0.06453825800,   1581.95934828300 ),
  { 4532   52 }  (        17.025, 0.53383755278,   1368.66025284500 ),
  { 4532   53 }  (        12.804, 3.90044242142,    433.71173787680 ),
  { 4532   54 }  (        13.072, 0.79468040717,    110.20632121940 ),
  { 4532   55 }  (        11.945, 0.40671403646,   1361.54670584420 ),
  { 4532   56 }  (        11.695, 4.44394618065,    405.25754987360 ),
  { 4532   57 }  (        11.979, 2.22872778682,    220.41264243880 ),
  { 4532   58 }  (         9.633, 6.01002272123,    853.19638175200 ),
  { 4532   59 }  (        10.163, 0.99504635158,   1471.75302706360 ),
  { 4532   60 }  (         8.977, 1.60328709409,   1692.16566950240 ),
  { 4532   61 }  (         8.701, 3.52167876799,   1073.60902419080 ),
  { 4532   62 }  (         8.314, 5.60169732564,   1574.84580128220 ),
  { 4532   63 }  (         8.958, 6.26708748901,    519.39602435610 ),
  { 4532   64 }  (         7.828, 0.65241611799,   1478.86657406440 ),
  { 4532   65 }  (         7.833, 0.17920601344,   1685.05212250160 ),
  { 4532   66 }  (         7.451, 0.88421084942,     88.86568021700 ),
  { 4532   67 }  (         7.320, 0.89341249264,    721.64941953020 ),
  { 4532   68 }  (         9.135, 1.51210840939,   1148.24761040620 ),
  { 4532   69 }  (         6.110, 2.50080005128,      3.18139373770 ),
  { 4532   70 }  (         7.037, 4.44127496638,    330.61896365820 ),
  { 4532   71 }  (         5.163, 2.79219166952,     21.34064100240 ),
  { 4532   72 }  (         5.079, 2.97991736844,   1375.77379984580 ),
  { 4532   73 }  (         4.930, 0.04683167622,   1677.93857550080 ),
  { 4532   74 }  (         4.664, 2.28007273876,   1485.98012106520 ),
  { 4532   75 }  (         4.692, 0.86220230505,      3.93215326310 ),
  { 4532   76 }  (         5.307, 0.85008578245,   1788.14489672020 ),
  { 4532   77 }  (         4.239, 0.40758287124,    629.60234557550 ),
  { 4532   78 }  (         4.230, 1.61046658091,    635.96513305090 ),
  { 4532   79 }  (         3.627, 2.71151441113,    551.03160609700 ),
  { 4532   80 }  (         3.314, 0.55067236587,   1795.25844372100 ),
  { 4532   81 }  (         4.409, 1.28127751050,   1464.63948006280 ),
  { 4532   82 }  (         3.270, 1.18744032691,   1905.46476494040 ),
  { 4532   83 }  (         3.226, 6.18716071251,   1038.04128918680 ),
  { 4532   84 }  (         3.103, 6.22971614425,   2001.44399215820 ),
  { 4532   85 }  (         3.410, 2.44624067925,    539.98590583310 ),
  { 4532   86 }  (         3.174, 5.54870592599,    191.95845443560 ),
  { 4532   87 }  (         2.590, 3.24430559059,   1062.56332392690 ),
  { 4532   88 }  (         2.614, 0.55149554149,   2104.53676637680 ),
  { 4532   89 }  (         2.174, 5.32613824409,   1891.23767093880 ),
  { 4532   90 }  (         2.659, 4.82459974220,    416.30325013750 ),
  { 4532   91 }  (         2.187, 1.71707514653,    628.85158605010 ),
  { 4532   92 }  (         2.263, 6.19233486371,   1994.33044515740 ),
  { 4532   93 }  (         2.328, 4.28236795066,    963.40270297140 ),
  { 4532   94 }  (         2.579, 0.03256542251,   1898.35121793960 ),
  { 4532   95 }  (         2.077, 3.32602157426,   1699.27921650320 ),
  { 4532   96 }  (         2.529, 2.39697505835,    227.52618943960 ),
  { 4532   97 }  (         2.468, 0.06551346218,    750.10360753340 ),
  { 4532   98 }  (         1.989, 0.29206371261,    636.71589257630 ),
  { 4532   99 }  (         1.927, 0.32286661566,    295.05122865420 ),
  { 4532  100 }  (         1.904, 3.43534792123,    647.01083331480 ),
  { 4532  101 }  (         1.940, 0.29170673525,   2111.65031337760 ),
  { 4532  102 }  (         1.880, 3.14403615586,    611.44309831080 ),
  { 4532  103 }  (         2.324, 1.94960720763,    824.74219374880 ),
  { 4532  104 }  (         1.854, 4.71794950485,   2125.87740737920 ),
  { 4532  105 }  (         2.547, 1.23908353090,   2221.85663459700 ),
  { 4532  106 }  (         1.814, 1.60250861074,   2008.55753915900 ),
  { 4532  107 }  (         1.611, 5.83466560322,    422.66603761290 ),
  { 4532  108 }  (         1.667, 2.32455940876,    440.82528487760 ),
  { 4532  109 }  (         1.622, 0.36650974375,   1056.20053645150 ),
  { 4532  110 }  (         1.624, 2.42139677881,     10.29494073850 ),
  { 4532  111 }  (         1.622, 3.51892791175,   1055.44977692610 ),
  { 4532  112 }  (         1.606, 5.76205763975,    117.31986822020 ),
  { 4532  113 }  (         1.646, 5.88662636573,   2317.83586181480 ),
  { 4532  114 }  (         2.026, 4.61781314145,    423.41679713830 ),
  { 4532  115 }  (         2.098, 1.04559231028,   1781.03134971940 ),
  { 4532  116 }  (         1.868, 1.12487729469,    618.55664531160 ),
  { 4532  117 }  (         1.885, 2.78775930564,   1802.37199072180 ),
  { 4532  118 }  (         1.445, 0.08308050305,   1382.88734684660 ),
  { 4532  119 }  (         1.797, 3.00776822706,   2648.45482547300 ),
  { 4532  120 }  (         1.422, 0.17649746278,   2420.92863603340 ),
  { 4532  121 }  (         1.129, 1.59030291320,    380.12776796000 ),
  { 4532  122 }  (         1.126, 4.19989673600,    547.85021235930 ),
  { 4532  123 }  (         1.186, 5.98943062173,   2310.72231481400 ),
  { 4532  124 }  (         1.108, 4.22655117757,    934.94851496820 ),
  { 4532  125 }  (         1.259, 1.19687222266,   1063.31408345230 ),
  { 4532  126 }  (         1.072, 3.86169004168,   1603.29998928540 ),
  { 4532  127 }  (         0.946, 5.59968097387,     99.91138048090 ),
  { 4532  128 }  (         0.937, 1.03083276760,     81.75213321620 ),
  { 4532  129 }  (         0.938, 6.18136092771,    945.99421523210 ),
  { 4532  130 }  (         0.908, 2.54355964041,   6283.07584999140 ),
  { 4532  131 }  (         0.874, 5.21903196047,   2207.62954059540 ),
  { 4532  132 }  (         0.874, 6.01240284465,    511.53171782990 ),
  { 4532  133 }  (         1.188, 0.75698357968,   2097.42321937600 ),
  { 4532  134 }  (         0.789, 3.91035208173,  10213.28554621100 ),
  { 4532  135 }  (         1.000, 1.34667100304,    732.69511979410 ),
  { 4532  136 }  (         0.952, 1.55355777420,   2324.94940881560 ),
  { 4532  137 }  (         0.811, 5.00475553271,    319.57326339430 ),
  { 4532  138 }  (         0.763, 3.98527559630,    337.73251065900 ),
  { 4532  139 }  (         0.880, 1.14789972199,    952.35700270750 ),
  { 4532  140 }  (         0.780, 4.69463316930,   5746.27133789600 ),
  { 4532  141 }  (         0.910, 0.08774541571,   2737.32050569000 ),
  { 4532  142 }  (         0.773, 0.77131695762,   5760.49843189760 ),
  { 4532  143 }  (         0.764, 6.11686539353,   9676.48103411560 ),
  { 4532  144 }  (         0.758, 2.19350719860,   9690.70812811720 ),
  { 4532  145 }  (         0.671, 1.19532387143,    124.43341522100 ),
  { 4532  146 }  (         0.661, 5.99578306627,    501.23677709140 ),
  { 4532  147 }  (         0.729, 0.65312263578,   2538.24850425360 ),
  { 4532  148 }  (         0.825, 2.70770030205,   3370.10424500320 ),
  { 4532  149 }  (         0.670, 5.44169923277,    107.02492748170 ),
  { 4532  150 }  (         0.739, 1.14609907817,   2641.34127847220 ),
  { 4532  151 }  (         0.866, 3.02831268213,   3046.59882834580 ),
  { 4532  152 }  (         0.718, 4.83684196454,    860.30992875280 ),
  { 4532  153 }  (         0.813, 6.01229270247,   2214.74308759620 ),
  { 4532  154 }  (         0.746, 1.12371143332,    739.80866679490 ),
  { 4532  155 }  (         0.741, 5.93171662010,   2634.22773147140 ),
  { 4532  156 }  (         0.667, 0.89885058003,    106.27416795630 ),
  { 4532  157 }  (         0.573, 2.42701822581,    739.05790726950 ),
  { 4532  158 }  (         0.734, 0.72837704619,   1354.43315884340 ),
  { 4532  159 }  (         0.662, 2.21768976390,   2015.67108615980 ),
  { 4532  160 }  (         0.782, 2.52401202862,   3679.38256765900 ),
  { 4532  161 }  (         0.779, 2.38608991574,   3267.01147078460 ),
  { 4532  162 }  (         0.553, 1.85211127676,    453.42489381900 ),
  { 4532  163 }  (         0.701, 4.23431087374,   9683.59458111640 ),
  { 4532  164 }  (         0.571, 2.98435419019,   1262.38608488870 ),
  { 4532  165 }  (         0.621, 1.24462887440,   3803.81598288000 ),
  { 4532  166 }  (         0.563, 5.99845316446,   1049.08698945070 ),
  { 4532  167 }  (         0.538, 4.92334194042,    447.79581952650 ),
  { 4532  168 }  (         0.534, 0.99911551571,    462.02291352810 ),
  { 4532  169 }  (         0.541, 6.19275150397,   1987.21689815660 ),
  { 4532  170 }  (         0.511, 3.28553278370,      4.66586644600 ),
  { 4532  171 }  (         0.539, 5.33214565622,   2751.54759969160 ),
  { 4532  172 }  (         0.651, 5.12199308959,   3156.80514956520 ),
  { 4532  173 }  (         0.483, 3.03782387056,   3281.23856478620 ),
  { 4532  174 }  (         0.476, 2.17592053936,    149.56319713460 ),
  { 4532  175 }  (         0.510, 5.35664230912,      9.56122755560 ),
  { 4532  176 }  (         0.490, 1.57324553106,   1251.34038462480 ),
  { 4532  177 }  (         0.467, 5.92343423840,    203.00415469950 ),
  { 4532  178 }  (         0.528, 5.81786945766,   2627.11418447060 ),
  { 4532  179 }  (         0.447, 3.51498961805,     18.15924726470 ),
  { 4532  180 }  (         0.429, 0.16627197188,     74.78159856730 ),
  { 4532  181 }  (         0.497, 0.30985248432,   2428.04218303420 ),
  { 4532  182 }  (         0.516, 3.89424540015,   2516.90786325120 ),
  { 4532  183 }  (         0.519, 2.43126348834,   3686.49611465980 ),
  { 4532  184 }  (         0.404, 2.77840802846,      7.16173111060 ),
  { 4532  185 }  (         0.533, 4.77083438961,   3473.19701922180 ),
  { 4532  186 }  (         0.515, 3.54549816613,   3178.14579056760 ),
  { 4532  187 }  (         0.533, 5.61415688189,   2524.02141025200 ),
  { 4532  188 }  (         0.458, 4.91616403047,   3067.93946934820 ),
  { 4532  189 }  (         0.400, 3.13887720912,    540.73666535850 ),
  { 4532  190 }  (         0.378, 0.86122450940,    525.02509864860 )
  (*$endif *)
      );
  (*@\\\0000002A01*)
  (*@/// vsop87_jup_r3:array[0.. 97,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_r3:array[0.. 27,0..2] of extended = (
  (*$else *)
  vsop87_jup_r3:array[0.. 97,0..2] of extended = (
  (*$endif *)
  { 4533    1 }  (      3519.257, 6.05800633846,    529.69096509460 ),
  { 4533    2 }  (      1073.239, 1.67321345760,    536.80451209540 ),
  { 4533    3 }  (       915.666, 1.41329676116,    522.57741809380 ),
  { 4533    4 }  (       341.593, 0.52296542656,   1059.38193018920 ),
  { 4533    5 }  (       254.893, 1.19625473533,      7.11354700080 ),
  { 4533    6 }  (       221.512, 0.95225226237,    515.46387109300 ),
  { 4533    7 }  (        69.078, 2.26885282314,   1066.49547719000 ),
  { 4533    8 }  (        89.729, 3.14159265359,      0.00000000000 ),
  { 4533    9 }  (        57.827, 1.41389745339,    543.91805909620 ),
  { 4533   10 }  (        57.653, 0.52580117593,    639.89728631400 ),
  { 4533   11 }  (        51.079, 5.98016364677,    412.37109687440 ),
  { 4533   12 }  (        46.935, 1.57864237959,    625.67019231240 ),
  { 4533   13 }  (        42.824, 6.11689609099,    419.48464387520 ),
  { 4533   14 }  (        37.477, 1.18262762330,     14.22709400160 ),
  { 4533   15 }  (        33.816, 1.66671706951,   1052.26838318840 ),
  { 4533   16 }  (        31.195, 1.04290245896,   1589.07289528380 ),
  { 4533   17 }  (        30.023, 4.63236245032,    426.59819087600 ),
  { 4533   18 }  (        33.531, 0.84784977903,    206.18554843720 ),
  { 4533   19 }  (        20.804, 2.50071243814,    728.76296653100 ),
  { 4533   20 }  (        14.466, 0.96040197071,    508.35032409220 ),
  { 4533   21 }  (        12.969, 1.50233788550,   1045.15483618760 ),
  { 4533   22 }  (        11.654, 3.55513510121,    323.50541665740 ),
  { 4533   23 }  (        12.319, 2.60952614503,    735.87651353180 ),
  { 4533   24 }  (        15.023, 0.89136998434,    199.07200143640 ),
  { 4533   25 }  (        11.160, 1.79041437555,    309.27832265580 ),
  { 4533   26 }  (        10.554, 6.27845112678,    956.28915597060 ),
  { 4533   27 }  (         9.812, 6.26016859519,    103.09277421860 ),
  { 4533   28 }  (         9.301, 3.45126812476,    838.96928775040 )
  (*$ifndef meeus *)                                                 ,
  { 4533   29 }  (         6.672, 1.87004905364,    302.16477565500 ),
  { 4533   30 }  (         7.442, 1.28047007623,    742.99006053260 ),
  { 4533   31 }  (         7.178, 0.92022189637,    942.06206196900 ),
  { 4533   32 }  (         5.577, 1.37980792905,     95.97922721780 ),
  { 4533   33 }  (         6.834, 3.45228722967,    831.85574074960 ),
  { 4533   34 }  (         4.632, 2.82934545414,   1596.18644228460 ),
  { 4533   35 }  (         3.969, 1.21290005054,   1169.58825140860 ),
  { 4533   36 }  (         3.869, 5.99495313698,    213.29909543800 ),
  { 4533   37 }  (         3.551, 6.10714791535,    405.25754987360 ),
  { 4533   38 }  (         2.943, 2.32831075458,   1155.36115740700 ),
  { 4533   39 }  (         2.442, 1.86965213405,    532.87235883230 ),
  { 4533   40 }  (         2.410, 0.42627205128,    220.41264243880 ),
  { 4533   41 }  (         2.289, 1.94941487274,   1073.60902419080 ),
  { 4533   42 }  (         2.274, 0.09211517505,    632.78373931320 ),
  { 4533   43 }  (         2.189, 1.58907745204,   2118.76386037840 ),
  { 4533   44 }  (         2.387, 5.97080671477,   1162.47470440780 ),
  { 4533   45 }  (         2.104, 1.06751462671,     21.34064100240 ),
  { 4533   46 }  (         2.128, 1.51119399925,   1258.45393162560 ),
  { 4533   47 }  (         2.491, 0.35125020737,   1272.68102562720 ),
  { 4533   48 }  (         2.006, 5.94487388360,    110.20632121940 ),
  { 4533   49 }  (         1.980, 2.54989377864,     88.86568021700 ),
  { 4533   50 }  (         2.040, 2.16463966964,    433.71173787680 ),
  { 4533   51 }  (         1.955, 2.70341589777,    721.64941953020 ),
  { 4533   52 }  (         1.670, 4.46255717328,    853.19638175200 ),
  { 4533   53 }  (         1.910, 2.25964760758,   1361.54670584420 ),
  { 4533   54 }  (         1.710, 1.98372066321,    525.75881183150 ),
  { 4533   55 }  (         1.520, 0.11641358425,    949.17560896980 ),
  { 4533   56 }  (         2.003, 3.16520599208,   1148.24761040620 ),
  { 4533   57 }  (         1.710, 2.70850417287,    330.61896365820 ),
  { 4533   58 }  (         1.629, 0.47376028854,    526.50957135690 ),
  { 4533   59 }  (         1.229, 3.01987279595,    963.40270297140 ),
  { 4533   60 }  (         1.671, 0.44352103086,    533.62311835770 ),
  { 4533   61 }  (         1.207, 1.15774089269,   1574.84580128220 ),
  { 4533   62 }  (         1.146, 2.54505851138,    846.08283475120 ),
  { 4533   63 }  (         1.355, 1.17462112647,   1038.04128918680 ),
  { 4533   64 }  (         1.001, 2.70272799283,    519.39602435610 ),
  { 4533   65 }  (         1.372, 0.67467128629,    551.03160609700 ),
  { 4533   66 }  (         0.983, 4.17198081351,   2627.11418447060 ),
  { 4533   67 }  (         1.084, 1.07011164067,    227.52618943960 ),
  { 4533   68 }  (         0.892, 2.92543286761,   1368.66025284500 ),
  { 4533   69 }  (         0.823, 4.86559196955,    611.44309831080 ),
  { 4533   70 }  (         1.136, 1.78981738432,   1581.95934828300 ),
  { 4533   71 }  (         0.897, 4.91073630270,   1670.82502850000 ),
  { 4533   72 }  (         0.908, 3.68804047330,    824.74219374880 ),
  { 4533   73 }  (         0.789, 3.23380893250,   2125.87740737920 ),
  { 4533   74 }  (         0.771, 2.39070707004,   2317.83586181480 ),
  { 4533   75 }  (         0.891, 0.59692950778,    539.98590583310 ),
  { 4533   76 }  (         0.876, 4.52127091462,    750.10360753340 ),
  { 4533   77 }  (         0.802, 0.20759322884,   1141.13406340540 ),
  { 4533   78 }  (         0.850, 0.94145487094,    191.95845443560 ),
  { 4533   79 }  (         0.762, 2.25149516048,   2538.24850425360 ),
  { 4533   80 }  (         0.694, 0.67080348659,    440.82528487760 ),
  { 4533   81 }  (         0.741, 5.79934203525,   1485.98012106520 ),
  { 4533   82 }  (         0.643, 2.48127580335,   1265.56747862640 ),
  { 4533   83 }  (         0.575, 6.13756590872,   1279.79457262800 ),
  { 4533   84 }  (         0.636, 5.51001645505,   2413.81508903260 ),
  { 4533   85 }  (         0.636, 4.40777238491,   1382.88734684660 ),
  { 4533   86 }  (         0.555, 2.18233983981,   1062.56332392690 ),
  { 4533   87 }  (         0.564, 1.92775967119,   2634.22773147140 ),
  { 4533   88 }  (         0.531, 2.04824376019,    295.05122865420 ),
  { 4533   89 }  (         0.541, 2.32424368689,   1471.75302706360 ),
  { 4533   90 }  (         0.697, 2.27179476322,   1699.27921650320 ),
  { 4533   91 }  (         0.546, 1.95774905730,   1677.93857550080 ),
  { 4533   92 }  (         0.465, 4.35550844067,   1692.16566950240 ),
  { 4533   93 }  (         0.508, 2.50298248836,   2207.62954059540 ),
  { 4533   94 }  (         0.496, 5.77087043616,   1478.86657406440 ),
  { 4533   95 }  (         0.440, 5.98661963879,    934.94851496820 ),
  { 4533   96 }  (         0.424, 2.80194129521,     81.75213321620 ),
  { 4533   97 }  (         0.406, 3.93940190897,    316.39186965660 ),
  { 4533   98 }  (         0.506, 0.18719982992,     10.29494073850 )
  (*$endif *)
      );
  (*@\\\0000002201*)
  (*@/// vsop87_jup_r4:array[0.. 45,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_r4:array[0.. 14,0..2] of extended = (
  (*$else *)
  vsop87_jup_r4:array[0.. 45,0..2] of extended = (
  (*$endif *)
  { 4534    1 }  (       128.628, 0.08419309557,    536.80451209540 ),
  { 4534    2 }  (       113.458, 4.24858855779,    529.69096509460 ),
  { 4534    3 }  (        82.650, 3.29754909408,    522.57741809380 ),
  { 4534    4 }  (        37.883, 2.73326611144,    515.46387109300 ),
  { 4534    5 }  (        26.694, 5.69142588558,      7.11354700080 ),
  { 4534    6 }  (        17.650, 5.40012536918,   1059.38193018920 ),
  { 4534    7 }  (        12.612, 6.01560416057,    543.91805909620 ),
  { 4534    8 }  (         9.287, 0.76813946494,   1066.49547719000 ),
  { 4534    9 }  (         8.107, 5.68228065707,     14.22709400160 ),
  { 4534   10 }  (         6.271, 5.12286932534,    639.89728631400 ),
  { 4534   11 }  (         6.978, 1.42751292055,    412.37109687440 ),
  { 4534   12 }  (         5.377, 3.33501947275,    625.67019231240 ),
  { 4534   13 }  (         2.911, 3.40334805052,   1052.26838318840 ),
  { 4534   14 }  (         2.593, 4.16090412984,    728.76296653100 ),
  { 4534   15 }  (         2.562, 2.89802035072,    426.59819087600 )
  (*$ifndef meeus *)                                                 ,
  { 4534   16 }  (         2.268, 6.22195938856,   1589.07289528380 ),
  { 4534   17 }  (         2.114, 3.11758855774,   1045.15483618760 ),
  { 4534   18 }  (         1.673, 2.81399290364,    206.18554843720 ),
  { 4534   19 }  (         1.805, 2.60030006919,    199.07200143640 ),
  { 4534   20 }  (         1.823, 1.89432426038,    419.48464387520 ),
  { 4534   21 }  (         1.522, 1.33432648232,   1596.18644228460 ),
  { 4534   22 }  (         1.697, 0.00000000000,      0.00000000000 ),
  { 4534   23 }  (         1.039, 4.41904942302,    956.28915597060 ),
  { 4534   24 }  (         1.161, 5.16181311538,    831.85574074960 ),
  { 4534   25 }  (         0.916, 3.17245716108,    508.35032409220 ),
  { 4534   26 }  (         0.870, 5.79387813500,   1169.58825140860 ),
  { 4534   27 }  (         0.916, 1.87129662931,   1148.24761040620 ),
  { 4534   28 }  (         0.955, 0.66801367802,   1361.54670584420 ),
  { 4534   29 }  (         0.788, 1.47515450553,   1272.68102562720 ),
  { 4534   30 }  (         0.966, 5.47457968043,    220.41264243880 ),
  { 4534   31 }  (         0.788, 2.42252866885,    117.31986822020 ),
  { 4534   32 }  (         0.712, 0.49655897030,   1073.60902419080 ),
  { 4534   33 }  (         0.656, 3.53022740783,    302.16477565500 ),
  { 4534   34 }  (         0.681, 2.84507174340,    191.95845443560 ),
  { 4534   35 }  (         0.771, 2.19893222018,    942.06206196900 ),
  { 4534   36 }  (         0.765, 5.31147257700,    551.03160609700 ),
  { 4534   37 }  (         0.667, 3.72432305249,     88.86568021700 ),
  { 4534   38 }  (         0.534, 1.83172084748,    647.01083331480 ),
  { 4534   39 }  (         0.553, 0.85896003802,    330.61896365820 ),
  { 4534   40 }  (         0.543, 5.26057584439,     21.34064100240 ),
  { 4534   41 }  (         0.584, 3.82243061802,    618.55664531160 ),
  { 4534   42 }  (         0.512, 4.44485521707,    110.20632121940 ),
  { 4534   43 }  (         0.612, 1.59320941864,      3.18139373770 ),
  { 4534   44 }  (         0.631, 1.83863158533,     10.29494073850 ),
  { 4534   45 }  (         0.491, 1.52912023181,    405.25754987360 ),
  { 4534   46 }  (         0.521, 0.24011424451,    433.71173787680 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_jup_r5:array[0..  8,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_r5:array[0..  6,0..2] of extended = (
  (*$else *)
  vsop87_jup_r5:array[0..  8,0..2] of extended = (
  (*$endif *)
  { 4535    1 }  (        11.188, 4.75249399945,    536.80451209540 ),
  { 4535    2 }  (         4.255, 5.91516229170,    522.57741809380 ),
  { 4535    3 }  (         2.079, 5.56781555864,    515.46387109300 ),
  { 4535    4 }  (         1.908, 4.29659647286,    543.91805909620 ),
  { 4535    5 }  (         1.875, 3.69357495838,      7.11354700080 ),
  { 4535    6 }  (         1.590, 5.49312796166,   1066.49547719000 ),
  { 4535    7 }  (         1.612, 4.13222808529,   1059.38193018920 )
  (*$ifndef meeus *)                                                 ,
  { 4535    8 }  (         1.240, 3.77981722506,     14.22709400160 ),
  { 4535    9 }  (         1.033, 4.50671820436,    529.69096509460 )
  (*$endif *)
      );
  (*@\\\*)
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_jup_r0)) and (nr<=high(vsop87_jup_r0)) then begin
        a:=vsop87_jup_r0[nr,0];  b:=vsop87_jup_r0[nr,1];  c:=vsop87_jup_r0[nr,2];
        end;
      1: if (nr>=low(vsop87_jup_r1)) and (nr<=high(vsop87_jup_r1)) then begin
        a:=vsop87_jup_r1[nr,0];  b:=vsop87_jup_r1[nr,1];  c:=vsop87_jup_r1[nr,2];
        end;
      2: if (nr>=low(vsop87_jup_r2)) and (nr<=high(vsop87_jup_r2)) then begin
        a:=vsop87_jup_r2[nr,0];  b:=vsop87_jup_r2[nr,1];  c:=vsop87_jup_r2[nr,2];
        end;
      3: if (nr>=low(vsop87_jup_r3)) and (nr<=high(vsop87_jup_r3)) then begin
        a:=vsop87_jup_r3[nr,0];  b:=vsop87_jup_r3[nr,1];  c:=vsop87_jup_r3[nr,2];
        end;
      4: if (nr>=low(vsop87_jup_r4)) and (nr<=high(vsop87_jup_r4)) then begin
        a:=vsop87_jup_r4[nr,0];  b:=vsop87_jup_r4[nr,1];  c:=vsop87_jup_r4[nr,2];
        end;
      5: if (nr>=low(vsop87_jup_r5)) and (nr<=high(vsop87_jup_r5)) then begin
        a:=vsop87_jup_r5[nr,0];  b:=vsop87_jup_r5[nr,1];  c:=vsop87_jup_r5[nr,2];
        end;
      end;
    end;
  end;
(*@\\\0000000801*)
(*@/// function TVSOPJupiter.LatitudeFactor(nr,index: integer):TVSOPEntry; *)
function TVSOPJupiter.LatitudeFactor(nr,index: integer):TVSOPEntry;
const
  (*@/// vsop87_jup_b0:array[0..248,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_b0:array[0.. 25,0..2] of extended = (
  (*$else *)
  vsop87_jup_b0:array[0..248,0..2] of extended = (
  (*$endif *)
  { 4520    1 }  (   2268615.703, 3.55852606718,    529.69096509460 ),
  { 4520    2 }  (    109971.634, 3.90809347389,   1059.38193018920 ),
  { 4520    3 }  (    110090.358, 0.00000000000,      0.00000000000 ),
  { 4520    4 }  (      8101.427, 3.60509573368,    522.57741809380 ),
  { 4520    5 }  (      6043.996, 4.25883108794,   1589.07289528380 ),
  { 4520    6 }  (      6437.782, 0.30627121409,    536.80451209540 ),
  { 4520    7 }  (      1106.880, 2.98534421928,   1162.47470440780 ),
  { 4520    8 }  (       941.651, 2.93619072405,   1052.26838318840 ),
  { 4520    9 }  (       894.088, 1.75447429921,      7.11354700080 ),
  { 4520   10 }  (       767.280, 2.15473594060,    632.78373931320 ),
  { 4520   11 }  (       944.328, 1.67522288396,    426.59819087600 ),
  { 4520   12 }  (       684.220, 3.67808770098,    213.29909543800 ),
  { 4520   13 }  (       629.223, 0.64343282328,   1066.49547719000 ),
  { 4520   14 }  (       835.861, 5.17881973234,    103.09277421860 ),
  { 4520   15 }  (       531.670, 2.70305954352,    110.20632121940 ),
  { 4520   16 }  (       558.524, 0.01354830508,    846.08283475120 ),
  { 4520   17 }  (       464.449, 1.17337249185,    949.17560896980 ),
  { 4520   18 }  (       431.072, 2.60825000494,    419.48464387520 ),
  { 4520   19 }  (       351.433, 4.61062990714,   2118.76386037840 ),
  { 4520   20 }  (       123.148, 3.34968181384,   1692.16566950240 ),
  { 4520   21 }  (       115.038, 5.04892295442,    316.39186965660 ),
  { 4520   22 }  (       132.160, 4.77816990670,    742.99006053260 ),
  { 4520   23 }  (       103.402, 2.31878999565,   1478.86657406440 ),
  { 4520   24 }  (       116.379, 1.38688232033,    323.50541665740 ),
  { 4520   25 }  (       102.420, 3.15293785436,   1581.95934828300 ),
  { 4520   26 }  (       103.762, 3.70103838110,    515.46387109300 )
  (*$ifndef meeus *)                                                 ,
  { 4520   27 }  (        78.650, 3.98318653238,   1265.56747862640 ),
  { 4520   28 }  (        69.935, 2.56006216424,    956.28915597060 ),
  { 4520   29 }  (        55.597, 0.37500753017,   1375.77379984580 ),
  { 4520   30 }  (        51.986, 0.99007119033,   1596.18644228460 ),
  { 4520   31 }  (        55.194, 0.40176412035,    525.75881183150 ),
  { 4520   32 }  (        63.456, 4.50073574333,    735.87651353180 ),
  { 4520   33 }  (        49.691, 0.18649893085,    543.91805909620 ),
  { 4520   34 }  (        48.831, 3.57260550671,    533.62311835770 ),
  { 4520   35 }  (        28.353, 1.53532744749,    625.67019231240 ),
  { 4520   36 }  (        29.209, 5.43145863011,    206.18554843720 ),
  { 4520   37 }  (        23.255, 5.95197992848,    838.96928775040 ),
  { 4520   38 }  (        22.841, 6.19262787685,    532.87235883230 ),
  { 4520   39 }  (        23.202, 4.06473368575,    526.50957135690 ),
  { 4520   40 }  (        24.436, 6.10947656959,   1169.58825140860 ),
  { 4520   41 }  (        21.116, 4.96322972735,   2648.45482547300 ),
  { 4520   42 }  (        17.879, 3.08704395969,   1795.25844372100 ),
  { 4520   43 }  (        16.234, 4.83515727869,   1368.66025284500 ),
  { 4520   44 }  (        21.314, 2.69476951059,   1045.15483618760 ),
  { 4520   45 }  (        15.740, 1.15130330106,    942.06206196900 ),
  { 4520   46 }  (        17.325, 1.61550009206,     14.22709400160 ),
  { 4520   47 }  (        13.396, 2.30539585502,    853.19638175200 ),
  { 4520   48 }  (        11.904, 3.09811974536,   2111.65031337760 ),
  { 4520   49 }  (        11.734, 2.83006431723,   2008.55753915900 ),
  { 4520   50 }  (        11.291, 0.98957560201,    433.71173787680 ),
  { 4520   51 }  (        11.830, 4.76527836803,    309.27832265580 ),
  { 4520   52 }  (        10.702, 3.70181397065,   2221.85663459700 ),
  { 4520   53 }  (        10.815, 5.81958878617,   1272.68102562720 ),
  { 4520   54 }  (        13.505, 3.28126975760,   1155.36115740700 ),
  { 4520   55 }  (        10.179, 2.58691128827,    117.31986822020 ),
  { 4520   56 }  (        10.632, 5.23487936086,     95.97922721780 ),
  { 4520   57 }  (         8.771, 0.40456546655,    220.41264243880 ),
  { 4520   58 }  (         7.439, 2.94638292086,    412.37109687440 ),
  { 4520   59 }  (         6.151, 2.69100382247,    380.12776796000 ),
  { 4520   60 }  (         5.028, 0.72750312028,   1055.44977692610 ),
  { 4520   61 }  (         4.939, 0.73756716762,   1905.46476494040 ),
  { 4520   62 }  (         5.421, 4.08612438558,   1685.05212250160 ),
  { 4520   63 }  (         5.936, 4.32059910537,   1063.31408345230 ),
  { 4520   64 }  (         4.737, 4.09303016850,    527.24328453980 ),
  { 4520   65 }  (         4.010, 0.51530008355,   1073.60902419080 ),
  { 4520   66 }  (         4.709, 1.84067645204,    984.60033162190 ),
  { 4520   67 }  (         3.974, 1.33608029246,   2125.87740737920 ),
  { 4520   68 }  (         3.762, 3.58647039394,    529.73914920440 ),
  { 4520   69 }  (         4.731, 6.16377350841,    532.13864564940 ),
  { 4520   70 }  (         4.666, 5.88762905802,    639.89728631400 ),
  { 4520   71 }  (         3.763, 0.38865925413,    529.64278098480 ),
  { 4520   72 }  (         3.409, 4.05398247269,   1898.35121793960 ),
  { 4520   73 }  (         3.457, 3.43865563497,   1485.98012106520 ),
  { 4520   74 }  (         4.229, 2.23767157901,     74.78159856730 ),
  { 4520   75 }  (         3.091, 0.16470256025,   1699.27921650320 ),
  { 4520   76 }  (         2.975, 0.72268908074,    530.65417294110 ),
  { 4520   77 }  (         3.162, 1.25048416420,    330.61896365820 ),
  { 4520   78 }  (         2.727, 4.37679213321,    149.56319713460 ),
  { 4520   79 }  (         2.837, 0.05987107395,   1439.50969814920 ),
  { 4520   80 }  (         2.983, 3.25251207220,    528.72775724810 ),
  { 4520   81 }  (         2.232, 0.26149880534,   1062.56332392690 ),
  { 4520   82 }  (         2.464, 1.16913304420,    453.42489381900 ),
  { 4520   83 }  (         2.596, 3.30510149086,   2324.94940881560 ),
  { 4520   84 }  (         1.988, 2.85269577619,   1574.84580128220 ),
  { 4520   85 }  (         2.527, 5.94458202950,    909.81873305460 ),
  { 4520   86 }  (         2.269, 1.30379329597,      3.93215326310 ),
  { 4520   87 }  (         1.742, 4.49909767044,   1258.45393162560 ),
  { 4520   88 }  (         1.714, 4.12945878208,   2001.44399215820 ),
  { 4520   89 }  (         2.029, 3.97938086639,   1056.20053645150 ),
  { 4520   90 }  (         1.667, 0.36037092553,  10213.28554621100 ),
  { 4520   91 }  (         1.579, 6.11640144795,   1802.37199072180 ),
  { 4520   92 }  (         1.393, 3.69324470827,   2214.74308759620 ),
  { 4520   93 }  (         1.604, 1.98841031703,     38.13303563780 ),
  { 4520   94 }  (         1.325, 1.74025919863,    529.16970023280 ),
  { 4520   95 }  (         1.451, 2.39804501178,   2428.04218303420 ),
  { 4520   96 }  (         1.594, 2.07556780757,   1021.24889455140 ),
  { 4520   97 }  (         1.320, 1.33770977126,    618.55664531160 ),
  { 4520   98 }  (         1.346, 3.27591492540,   2641.34127847220 ),
  { 4520   99 }  (         1.230, 0.19552728220,    305.34616939270 ),
  { 4520  100 }  (         1.223, 2.86681556337,   1382.88734684660 ),
  { 4520  101 }  (         1.324, 2.23549334986,    530.21222995640 ),
  { 4520  102 }  (         1.056, 3.80579750957,     76.26607127560 ),
  { 4520  103 }  (         1.050, 4.68011652614,   1788.14489672020 ),
  { 4520  104 }  (         1.226, 5.34003255221,   3178.14579056760 ),
  { 4520  105 }  (         1.009, 3.19608028376,   2538.24850425360 ),
  { 4520  106 }  (         1.266, 3.04704446731,    604.47256366190 ),
  { 4520  107 }  (         0.954, 3.86932544808,    728.76296653100 ),
  { 4520  108 }  (         1.124, 1.59560367480,      3.18139373770 ),
  { 4520  109 }  (         0.978, 0.25223689838,    983.11585891360 ),
  { 4520  110 }  (         0.948, 0.21552742733,    750.10360753340 ),
  { 4520  111 }  (         0.946, 3.93927748120,    508.35032409220 ),
  { 4520  112 }  (         0.920, 1.14672086939,    963.40270297140 ),
  { 4520  113 }  (         0.817, 5.93809619876,    831.85574074960 ),
  { 4520  114 }  (         0.770, 2.96062737592,    526.77020378780 ),
  { 4520  115 }  (         1.017, 5.55711112145,    199.07200143640 ),
  { 4520  116 }  (         0.761, 1.38163787157,    532.61172640140 ),
  { 4520  117 }  (         0.726, 3.98337964395,   2317.83586181480 ),
  { 4520  118 }  (         0.862, 0.87975657414,    490.33408917940 ),
  { 4520  119 }  (         0.868, 3.44331872364,    569.04784100980 ),
  { 4520  120 }  (         0.711, 4.11107052823,   2751.54759969160 ),
  { 4520  121 }  (         0.708, 0.33555577415,    528.94020556920 ),
  { 4520  122 }  (         0.708, 4.00539820601,    530.44172462000 ),
  { 4520  123 }  (         0.656, 4.39568451439,    519.39602435610 ),
  { 4520  124 }  (         0.801, 4.03984430862,   1364.72809958190 ),
  { 4520  125 }  (         0.679, 1.18645749024,    525.49817940060 ),
  { 4520  126 }  (         0.645, 5.10510349996,   1361.54670584420 ),
  { 4520  127 }  (         0.668, 3.15607509055,    533.88375078860 ),
  { 4520  128 }  (         0.663, 0.73722024843,   5223.69391980220 ),
  { 4520  129 }  (         0.663, 1.57092786811,   6283.07584999140 ),
  { 4520  130 }  (         0.543, 0.26376529935,    227.52618943960 ),
  { 4520  131 }  (         0.525, 6.22318693939,    539.98590583310 ),
  { 4520  132 }  (         0.513, 4.98337900151,    302.16477565500 ),
  { 4520  133 }  (         0.544, 2.22227019273,   2744.43405269080 ),
  { 4520  134 }  (         0.532, 2.62425372687,     99.16062095550 ),
  { 4520  135 }  (         0.602, 1.56074089013,    454.90936652730 ),
  { 4520  136 }  (         0.518, 0.26343805959,    551.03160609700 ),
  { 4520  137 }  (         0.516, 1.09376390349,    934.94851496820 ),
  { 4520  138 }  (         0.659, 0.62560671589,   1512.80682400820 ),
  { 4520  139 }  (         0.524, 0.64710955846,    524.06189080210 ),
  { 4520  140 }  (         0.516, 3.69478866795,    535.32003938710 ),
  { 4520  141 }  (         0.491, 3.63039940597,   2531.13495725280 ),
  { 4520  142 }  (         0.570, 0.61976758791,    540.73666535850 ),
  { 4520  143 }  (         0.496, 2.19398015038,   1514.29129671650 ),
  { 4520  144 }  (         0.532, 0.20040217534,    525.02509864860 ),
  { 4520  145 }  (         0.493, 0.39160693598,    224.34479570190 ),
  { 4520  146 }  (         0.449, 0.62392433691,    529.53090640020 ),
  { 4520  147 }  (         0.449, 3.71676131146,    529.85102378900 ),
  { 4520  148 }  (         0.450, 5.02467015031,   1048.33622992530 ),
  { 4520  149 }  (         0.428, 5.44804660290,     11.04570026390 ),
  { 4520  150 }  (         0.499, 4.13924061941,    534.35683154060 ),
  { 4520  151 }  (         0.528, 1.76471074936,    524.27433912320 ),
  { 4520  152 }  (         0.454, 4.53321742354,   1056.93424963440 ),
  { 4520  153 }  (         0.520, 2.57406093768,    535.10759106600 ),
  { 4520  154 }  (         0.398, 1.40345870113,    960.22130923370 ),
  { 4520  155 }  (         0.457, 4.17708652827,   2104.53676637680 ),
  { 4520  156 }  (         0.505, 5.36536256321,   1057.89745748090 ),
  { 4520  157 }  (         0.535, 4.80455380313,   1593.00504854690 ),
  { 4520  158 }  (         0.415, 0.96548127237,   2435.15573003500 ),
  { 4520  159 }  (         0.519, 0.54543519483,   1061.82961074400 ),
  { 4520  160 }  (         0.359, 4.02704454075,   1059.43011429900 ),
  { 4520  161 }  (         0.356, 2.66818105522,    835.03713448730 ),
  { 4520  162 }  (         0.443, 5.27513700376,      1.48447270830 ),
  { 4520  163 }  (         0.358, 5.94423960514,    440.82528487760 ),
  { 4520  164 }  (         0.471, 6.05791940453,   1471.75302706360 ),
  { 4520  165 }  (         0.386, 2.15984900214,   9153.90361602180 ),
  { 4520  166 }  (         0.424, 2.70929670030,   1038.04128918680 ),
  { 4520  167 }  (         0.359, 0.82922836987,   1059.33374607940 ),
  { 4520  168 }  (         0.310, 0.88102053266,    529.90341341570 ),
  { 4520  169 }  (         0.310, 3.45966511571,    529.47851677350 ),
  { 4520  170 }  (         0.300, 3.70331799503,   2634.22773147140 ),
  { 4520  171 }  (         0.292, 2.63594456361,    415.55249061210 ),
  { 4520  172 }  (         0.279, 1.60669121578,    643.82943957710 ),
  { 4520  173 }  (         0.291, 5.83134071820,   1148.24761040620 ),
  { 4520  174 }  (         0.370, 5.71572992274,    531.17543780290 ),
  { 4520  175 }  (         0.268, 5.39275891813,   1891.23767093880 ),
  { 4520  176 }  (         0.275, 3.34108666036,    518.64526483070 ),
  { 4520  177 }  (         0.269, 1.06051406954,   1585.14074202070 ),
  { 4520  178 }  (         0.306, 2.50289017370,    511.53171782990 ),
  { 4520  179 }  (         0.295, 1.84394223501,    547.85021235930 ),
  { 4520  180 }  (         0.254, 2.98312992496,   1134.16352875650 ),
  { 4520  181 }  (         0.289, 1.86070918711,     21.34064100240 ),
  { 4520  182 }  (         0.265, 4.93075479744,    679.25416222920 ),
  { 4520  183 }  (         0.250, 0.42860925124,   1969.20066324380 ),
  { 4520  184 }  (         0.308, 2.67237933272,   2957.73314812880 ),
  { 4520  185 }  (         0.313, 4.88085697819,    528.20649238630 ),
  { 4520  186 }  (         0.222, 4.78828764413,    514.71311156760 ),
  { 4520  187 }  (         0.221, 4.32763468981,   1677.93857550080 ),
  { 4520  188 }  (         0.217, 3.46278526461,   2950.61960112800 ),
  { 4520  189 }  (         0.216, 0.52207667980,   2228.97018159780 ),
  { 4520  190 }  (         0.214, 5.83569926578,    544.66881862160 ),
  { 4520  191 }  (         0.283, 2.88709716090,     35.42472265210 ),
  { 4520  192 }  (         0.272, 1.65708415457,   3060.82592234740 ),
  { 4520  193 }  (         0.234, 1.68821537711,   2655.56837247380 ),
  { 4520  194 }  (         0.205, 3.36186888290,   2847.52682690940 ),
  { 4520  195 }  (         0.264, 3.62722625694,   2420.92863603340 ),
  { 4520  196 }  (         0.191, 4.26821147044,    430.53034413910 ),
  { 4520  197 }  (         0.179, 3.91470663005,   3340.61242669980 ),
  { 4520  198 }  (         0.180, 0.04531671003,    387.24131496080 ),
  { 4520  199 }  (         0.241, 4.03927631611,    494.26624244250 ),
  { 4520  200 }  (         0.176, 4.26298906325,    672.14061522840 ),
  { 4520  201 }  (         0.187, 2.72587420586,    299.12639426920 ),
  { 4520  202 }  (         0.234, 1.34474827450,    173.94221952280 ),
  { 4520  203 }  (         0.171, 0.85473611718,   1603.29998928540 ),
  { 4520  204 }  (         0.224, 0.33130232434,    565.11568774670 ),
  { 4520  205 }  (         0.200, 1.27632489123,     39.35687591520 ),
  { 4520  206 }  (         0.170, 4.96479470273,   1464.63948006280 ),
  { 4520  207 }  (         0.211, 1.00937080256,    523.54062594030 ),
  { 4520  208 }  (         0.210, 3.75793720248,   2854.64037391020 ),
  { 4520  209 }  (         0.162, 5.87784787295,   3480.31056622260 ),
  { 4520  210 }  (         0.163, 4.62850343495,   2015.67108615980 ),
  { 4520  211 }  (         0.191, 3.33159283750,    535.84130424890 ),
  { 4520  212 }  (         0.151, 1.17096741034,   1060.34513803570 ),
  { 4520  213 }  (         0.160, 1.81852636004,    312.45971639350 ),
  { 4520  214 }  (         0.158, 2.59595816107,    529.43033266370 ),
  { 4520  215 }  (         0.158, 1.74472748730,    529.95159752550 ),
  { 4520  216 }  (         0.173, 3.62399350412,    230.56457082540 ),
  { 4520  217 }  (         0.142, 0.70435921398,    522.52923398400 ),
  { 4520  218 }  (         0.144, 5.35763122430,    107.02492748170 ),
  { 4520  219 }  (         0.144, 6.13954848857,   1158.54255114470 ),
  { 4520  220 }  (         0.178, 0.27566275049,   3906.90875709860 ),
  { 4520  221 }  (         0.126, 5.14832919826,   2207.62954059540 ),
  { 4520  222 }  (         0.126, 3.41994798109,      2.44768055480 ),
  { 4520  223 }  (         0.127, 0.39825164051,     70.84944530420 ),
  { 4520  224 }  (         0.123, 4.77865550523,   2524.02141025200 ),
  { 4520  225 }  (         0.123, 0.46184813516,    647.01083331480 ),
  { 4520  226 }  (         0.144, 3.60261852727,   1058.41872234270 ),
  { 4520  227 }  (         0.158, 3.76231915252,     92.04707395470 ),
  { 4520  228 }  (         0.119, 4.08266911415,   1585.89150154610 ),
  { 4520  229 }  (         0.125, 2.35496721797,   3163.91869656600 ),
  { 4520  230 }  (         0.122, 3.21027426317,   3377.21779200400 ),
  { 4520  231 }  (         0.121, 3.39770381916,     18.15924726470 ),
  { 4520  232 }  (         0.131, 1.67926417552,   1289.94650101460 ),
  { 4520  233 }  (         0.115, 2.35735471566,   1550.93985964600 ),
  { 4520  234 }  (         0.126, 2.40833814513,    106.27416795630 ),
  { 4520  235 }  (         0.131, 1.37610474529,   1023.95720753710 ),
  { 4520  236 }  (         0.121, 1.60252617273,     10.29494073850 ),
  { 4520  237 }  (         0.121, 0.61420823557,   1592.25428902150 ),
  { 4520  238 }  (         0.135, 3.60177675518,    124.43341522100 ),
  { 4520  239 }  (         0.137, 2.41724947062,   3274.12501778540 ),
  { 4520  240 }  (         0.129, 0.09702914345,   2332.06295581640 ),
  { 4520  241 }  (         0.093, 4.88949890397,   1098.73880610440 ),
  { 4520  242 }  (         0.106, 5.18592950792,   2281.23049651060 ),
  { 4520  243 }  (         0.114, 2.96523316419,   1166.40685767090 ),
  { 4520  244 }  (         0.092, 1.65166124027,    860.30992875280 ),
  { 4520  245 }  (         0.102, 3.64093193142,   3171.03224356680 ),
  { 4520  246 }  (         0.103, 1.63066232967,   1894.41906467650 ),
  { 4520  247 }  (         0.080, 0.38766601876,   4694.00295470760 ),
  { 4520  248 }  (         0.074, 3.86865238736,   3067.93946934820 ),
  { 4520  249 }  (         0.095, 1.66362447044,   1151.42900414390 )
  (*$endif *)
      );
  (*@\\\0000002001*)
  (*@/// vsop87_jup_b1:array[0..140,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_b1:array[0.. 21,0..2] of extended = (
  (*$else *)
  vsop87_jup_b1:array[0..140,0..2] of extended = (
  (*$endif *)
  { 4521    1 }  (    177351.787, 5.70166488486,    529.69096509460 ),
  { 4521    2 }  (      3230.171, 5.77941619340,   1059.38193018920 ),
  { 4521    3 }  (      3081.364, 5.47464296527,    522.57741809380 ),
  { 4521    4 }  (      2211.914, 4.73477480209,    536.80451209540 ),
  { 4521    5 }  (      1694.232, 3.14159265359,      0.00000000000 ),
  { 4521    6 }  (       346.445, 4.74595174109,   1052.26838318840 ),
  { 4521    7 }  (       234.264, 5.18856099929,   1066.49547719000 ),
  { 4521    8 }  (       196.154, 6.18554286642,      7.11354700080 ),
  { 4521    9 }  (       150.468, 3.92721226087,   1589.07289528380 ),
  { 4521   10 }  (       114.128, 3.43897271830,    632.78373931320 ),
  { 4521   11 }  (        96.667, 2.91426304090,    949.17560896980 ),
  { 4521   12 }  (        76.599, 2.50522188662,    103.09277421860 ),
  { 4521   13 }  (        81.671, 5.07666097497,   1162.47470440780 ),
  { 4521   14 }  (        76.572, 0.61288981445,    419.48464387520 ),
  { 4521   15 }  (        73.875, 5.49958292155,    515.46387109300 ),
  { 4521   16 }  (        49.915, 3.94799616572,    735.87651353180 ),
  { 4521   17 }  (        60.544, 5.44740084359,    213.29909543800 ),
  { 4521   18 }  (        36.561, 4.69828392839,    543.91805909620 ),
  { 4521   19 }  (        46.032, 0.53850360901,    110.20632121940 ),
  { 4521   20 }  (        45.123, 1.89516645239,    846.08283475120 ),
  { 4521   21 }  (        36.019, 6.10952578764,    316.39186965660 ),
  { 4521   22 }  (        31.975, 4.92452714629,   1581.95934828300 )
  (*$ifndef meeus *)                                                 ,
  { 4521   23 }  (        21.015, 5.62957731410,   1596.18644228460 ),
  { 4521   24 }  (        23.156, 5.84829490183,    323.50541665740 ),
  { 4521   25 }  (        24.719, 3.94107395247,   2118.76386037840 ),
  { 4521   26 }  (        17.274, 5.65310656429,    533.62311835770 ),
  { 4521   27 }  (        16.521, 5.89840100621,    526.50957135690 ),
  { 4521   28 }  (        16.698, 5.66663034948,   1265.56747862640 ),
  { 4521   29 }  (        15.815, 4.43314786393,   1045.15483618760 ),
  { 4521   30 }  (        13.398, 4.30179033605,    532.87235883230 ),
  { 4521   31 }  (        11.744, 1.80990486955,    956.28915597060 ),
  { 4521   32 }  (        11.925, 4.30094564154,    525.75881183150 ),
  { 4521   33 }  (         9.514, 2.02589667166,    206.18554843720 ),
  { 4521   34 }  (        10.542, 6.15533910933,     14.22709400160 ),
  { 4521   35 }  (         8.414, 3.92910450340,   1478.86657406440 ),
  { 4521   36 }  (         8.099, 4.20152809071,   1169.58825140860 ),
  { 4521   37 }  (         7.712, 2.99160389601,    942.06206196900 ),
  { 4521   38 }  (         8.825, 1.55897920307,    426.59819087600 ),
  { 4521   39 }  (         8.884, 4.87430124264,   1155.36115740700 ),
  { 4521   40 }  (         7.793, 3.84684930196,    625.67019231240 ),
  { 4521   41 }  (         5.646, 3.40915964493,    639.89728631400 ),
  { 4521   42 }  (         4.615, 0.83374662294,    117.31986822020 ),
  { 4521   43 }  (         4.020, 5.50502127885,    433.71173787680 ),
  { 4521   44 }  (         3.704, 0.90226777963,     95.97922721780 ),
  { 4521   45 }  (         3.859, 0.69640284662,    853.19638175200 ),
  { 4521   46 }  (         3.091, 5.09115860882,   1073.60902419080 ),
  { 4521   47 }  (         3.360, 5.10133284081,   1692.16566950240 ),
  { 4521   48 }  (         2.892, 4.90418916660,    220.41264243880 ),
  { 4521   49 }  (         2.772, 5.09066125724,   2111.65031337760 ),
  { 4521   50 }  (         2.425, 3.74438653232,    742.99006053260 ),
  { 4521   51 }  (         2.558, 5.46955948791,   1795.25844372100 ),
  { 4521   52 }  (         2.466, 4.22278355430,   2648.45482547300 ),
  { 4521   53 }  (         1.968, 0.57192251841,    309.27832265580 ),
  { 4521   54 }  (         1.794, 4.60765219417,   1272.68102562720 ),
  { 4521   55 }  (         1.822, 1.98842964323,   1375.77379984580 ),
  { 4521   56 }  (         1.703, 6.12660562937,   2125.87740737920 ),
  { 4521   57 }  (         2.011, 5.00936865256,    412.37109687440 ),
  { 4521   58 }  (         1.645, 0.08830372958,   1063.31408345230 ),
  { 4521   59 }  (         1.875, 5.81006158403,    330.61896365820 ),
  { 4521   60 }  (         1.741, 4.58650290431,   1574.84580128220 ),
  { 4521   61 }  (         1.529, 5.81660291389,   1258.45393162560 ),
  { 4521   62 }  (         1.916, 0.85150399517,   1368.66025284500 ),
  { 4521   63 }  (         1.614, 4.36839107221,    728.76296653100 ),
  { 4521   64 }  (         1.510, 2.79374165455,   1485.98012106520 ),
  { 4521   65 }  (         1.333, 4.84260898693,   1062.56332392690 ),
  { 4521   66 }  (         1.359, 5.16511980864,    838.96928775040 ),
  { 4521   67 }  (         1.165, 5.66275740881,    508.35032409220 ),
  { 4521   68 }  (         1.092, 4.68797557406,   1699.27921650320 ),
  { 4521   69 }  (         1.438, 5.78105679279,   1056.20053645150 ),
  { 4521   70 }  (         1.083, 3.99886917926,   1471.75302706360 ),
  { 4521   71 }  (         1.002, 4.79949608524,   1055.44977692610 ),
  { 4521   72 }  (         0.749, 6.14400862030,    519.39602435610 ),
  { 4521   73 }  (         0.657, 5.63765568876,   1898.35121793960 ),
  { 4521   74 }  (         0.702, 5.04126574492,   1685.05212250160 ),
  { 4521   75 }  (         0.607, 3.15707515246,    618.55664531160 ),
  { 4521   76 }  (         0.587, 1.37658820775,    199.07200143640 ),
  { 4521   77 }  (         0.552, 4.80657729450,    551.03160609700 ),
  { 4521   78 }  (         0.494, 4.43417307482,    539.98590583310 ),
  { 4521   79 }  (         0.517, 0.05161181997,      3.18139373770 ),
  { 4521   80 }  (         0.469, 3.81715950042,   2008.55753915900 ),
  { 4521   81 }  (         0.415, 1.34693184108,   1382.88734684660 ),
  { 4521   82 }  (         0.382, 4.86764073919,    227.52618943960 ),
  { 4521   83 }  (         0.473, 1.72405831407,    532.13864564940 ),
  { 4521   84 }  (         0.458, 4.44604993015,   1038.04128918680 ),
  { 4521   85 }  (         0.376, 2.23190744786,    529.64278098480 ),
  { 4521   86 }  (         0.451, 3.75869883836,    984.60033162190 ),
  { 4521   87 }  (         0.376, 5.42971857629,    529.73914920440 ),
  { 4521   88 }  (         0.389, 1.92698506631,    525.02509864860 ),
  { 4521   89 }  (         0.364, 3.35456685746,   2221.85663459700 ),
  { 4521   90 }  (         0.476, 5.93625415892,    527.24328453980 ),
  { 4521   91 }  (         0.383, 6.12255867339,    149.56319713460 ),
  { 4521   92 }  (         0.301, 4.09378934049,    440.82528487760 ),
  { 4521   93 }  (         0.310, 5.58150418981,   2428.04218303420 ),
  { 4521   94 }  (         0.282, 4.85996662231,   1788.14489672020 ),
  { 4521   95 }  (         0.298, 5.09589374634,    528.72775724810 ),
  { 4521   96 }  (         0.340, 4.56537070220,    750.10360753340 ),
  { 4521   97 }  (         0.272, 2.35346960340,    534.35683154060 ),
  { 4521   98 }  (         0.360, 3.91050161665,     74.78159856730 ),
  { 4521   99 }  (         0.299, 1.43093538841,    909.81873305460 ),
  { 4521  100 }  (         0.297, 2.56584512211,    530.65417294110 ),
  { 4521  101 }  (         0.235, 4.81644489422,    535.10759106600 ),
  { 4521  102 }  (         0.306, 0.68420442848,    380.12776796000 ),
  { 4521  103 }  (         0.236, 4.63162956792,    526.77020378780 ),
  { 4521  104 }  (         0.270, 0.18549916939,     21.34064100240 ),
  { 4521  105 }  (         0.288, 4.26655874393,   1802.37199072180 ),
  { 4521  106 }  (         0.196, 5.35950443033,   2214.74308759620 ),
  { 4521  107 }  (         0.190, 4.54615193260,   2104.53676637680 ),
  { 4521  108 }  (         0.193, 4.35426216497,    511.53171782990 ),
  { 4521  109 }  (         0.178, 4.51895208036,   3178.14579056760 ),
  { 4521  110 }  (         0.194, 0.57050756837,   1361.54670584420 ),
  { 4521  111 }  (         0.200, 1.48040474749,    302.16477565500 ),
  { 4521  112 }  (         0.168, 5.40141749419,    524.27433912320 ),
  { 4521  113 }  (         0.152, 0.68077486546,   1905.46476494040 ),
  { 4521  114 }  (         0.149, 1.06678990744,    831.85574074960 ),
  { 4521  115 }  (         0.182, 3.62401009613,     38.13303563780 ),
  { 4521  116 }  (         0.176, 5.64331384323,    963.40270297140 ),
  { 4521  117 }  (         0.184, 4.48850356629,    604.47256366190 ),
  { 4521  118 }  (         0.133, 5.45026366125,   2641.34127847220 ),
  { 4521  119 }  (         0.143, 2.21577268292,   1439.50969814920 ),
  { 4521  120 }  (         0.130, 4.88155705493,   2531.13495725280 ),
  { 4521  121 }  (         0.129, 6.15206333598,    547.85021235930 ),
  { 4521  122 }  (         0.133, 5.43193972385,   1603.29998928540 ),
  { 4521  123 }  (         0.133, 3.49297492409,    529.16970023280 ),
  { 4521  124 }  (         0.132, 3.98820790955,    530.21222995640 ),
  { 4521  125 }  (         0.118, 5.38352943814,   1891.23767093880 ),
  { 4521  126 }  (         0.133, 5.65694269884,     76.26607127560 ),
  { 4521  127 }  (         0.145, 2.94976686191,    454.90936652730 ),
  { 4521  128 }  (         0.115, 3.29206553804,      3.93215326310 ),
  { 4521  129 }  (         0.102, 4.48856749557,   2001.44399215820 ),
  { 4521  130 }  (         0.106, 6.08434275898,     10.29494073850 ),
  { 4521  131 }  (         0.093, 5.84737771840,   2324.94940881560 ),
  { 4521  132 }  (         0.101, 0.15815934254,   2655.56837247380 ),
  { 4521  133 }  (         0.115, 3.59221021604,   2015.67108615980 ),
  { 4521  134 }  (         0.103, 4.70399583323,    305.34616939270 ),
  { 4521  135 }  (         0.084, 0.44180206332,   1593.00504854690 ),
  { 4521  136 }  (         0.092, 2.44863388631,    490.33408917940 ),
  { 4521  137 }  (         0.087, 6.23817512863,   6283.07584999140 ),
  { 4521  138 }  (         0.095, 3.30154605532,   2317.83586181480 ),
  { 4521  139 }  (         0.072, 1.90578907085,    528.94020556920 ),
  { 4521  140 }  (         0.072, 5.57619428876,    530.44172462000 ),
  { 4521  141 }  (         0.078, 5.97323507836,   1585.89150154610 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_jup_b2:array[0.. 80,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_b2:array[0.. 13,0..2] of extended = (
  (*$else *)
  vsop87_jup_b2:array[0.. 80,0..2] of extended = (
  (*$endif *)
  { 4522    1 }  (      8094.051, 1.46322843658,    529.69096509460 ),
  { 4522    2 }  (       742.415, 0.95691639003,    522.57741809380 ),
  { 4522    3 }  (       813.244, 3.14159265359,      0.00000000000 ),
  { 4522    4 }  (       398.951, 2.89888666447,    536.80451209540 ),
  { 4522    5 }  (       342.226, 1.44683789727,   1059.38193018920 ),
  { 4522    6 }  (        73.948, 0.40724675866,   1052.26838318840 ),
  { 4522    7 }  (        46.151, 3.48036895772,   1066.49547719000 ),
  { 4522    8 }  (        29.314, 0.99088831805,    515.46387109300 ),
  { 4522    9 }  (        29.717, 1.92504171329,   1589.07289528380 ),
  { 4522   10 }  (        22.753, 4.27124052435,      7.11354700080 ),
  { 4522   11 }  (        13.916, 2.92242387338,    543.91805909620 ),
  { 4522   12 }  (        12.067, 5.22168932482,    632.78373931320 ),
  { 4522   13 }  (        10.703, 4.88024222475,    949.17560896980 ),
  { 4522   14 }  (         6.078, 6.21089108431,   1045.15483618760 )
  (*$ifndef meeus *)                                                 ,
  { 4522   15 }  (         5.935, 0.52977760072,   1581.95934828300 ),
  { 4522   16 }  (         5.037, 1.43444929374,    526.50957135690 ),
  { 4522   17 }  (         4.564, 0.91811732585,   1162.47470440780 ),
  { 4522   18 }  (         4.547, 4.01953745202,   1596.18644228460 ),
  { 4522   19 }  (         5.098, 6.03169795231,    735.87651353180 ),
  { 4522   20 }  (         3.593, 4.54080164408,    110.20632121940 ),
  { 4522   21 }  (         3.443, 1.38618954572,    533.62311835770 ),
  { 4522   22 }  (         3.277, 4.39650286553,     14.22709400160 ),
  { 4522   23 }  (         3.407, 0.42275631534,    419.48464387520 ),
  { 4522   24 }  (         2.904, 2.06041641723,    316.39186965660 ),
  { 4522   25 }  (         2.541, 3.98323842017,    323.50541665740 ),
  { 4522   26 }  (         3.113, 2.48079280193,   2118.76386037840 ),
  { 4522   27 }  (         3.061, 2.39880866911,    532.87235883230 ),
  { 4522   28 }  (         2.155, 4.77990637140,    942.06206196900 ),
  { 4522   29 }  (         2.143, 3.88727338786,    426.59819087600 ),
  { 4522   30 }  (         2.252, 0.37196434120,   1155.36115740700 ),
  { 4522   31 }  (         2.019, 3.89985000464,    846.08283475120 ),
  { 4522   32 }  (         1.857, 1.19658907851,    103.09277421860 ),
  { 4522   33 }  (         1.683, 1.42264195434,   1265.56747862640 ),
  { 4522   34 }  (         2.313, 0.87671613055,    213.29909543800 ),
  { 4522   35 }  (         1.443, 2.38565505909,   1169.58825140860 ),
  { 4522   36 }  (         1.823, 5.80106463776,    625.67019231240 ),
  { 4522   37 }  (         1.728, 2.24114678267,    525.75881183150 ),
  { 4522   38 }  (         1.198, 0.03252059731,    956.28915597060 ),
  { 4522   39 }  (         1.138, 3.46420904745,   1073.60902419080 ),
  { 4522   40 }  (         1.086, 5.35279146700,    117.31986822020 ),
  { 4522   41 }  (         0.840, 2.89946334223,     95.97922721780 ),
  { 4522   42 }  (         0.746, 5.53017890231,   1478.86657406440 ),
  { 4522   43 }  (         0.944, 4.05587053500,    206.18554843720 ),
  { 4522   44 }  (         0.758, 3.74770617289,    433.71173787680 ),
  { 4522   45 }  (         0.673, 1.26396626349,    508.35032409220 ),
  { 4522   46 }  (         0.889, 6.07878453176,    728.76296653100 ),
  { 4522   47 }  (         0.600, 1.82954494089,    639.89728631400 ),
  { 4522   48 }  (         0.589, 1.23625943417,   1258.45393162560 ),
  { 4522   49 }  (         0.619, 0.67923057477,    838.96928775040 ),
  { 4522   50 }  (         0.566, 5.36336098734,    742.99006053260 ),
  { 4522   51 }  (         0.648, 5.32990375008,    853.19638175200 ),
  { 4522   52 }  (         0.553, 3.15511946637,    220.41264243880 ),
  { 4522   53 }  (         0.432, 1.03719283016,   1692.16566950240 ),
  { 4522   54 }  (         0.435, 1.65056479007,    519.39602435610 ),
  { 4522   55 }  (         0.430, 1.41830384501,    412.37109687440 ),
  { 4522   56 }  (         0.431, 2.20986254651,   1368.66025284500 ),
  { 4522   57 }  (         0.415, 4.35372561905,    330.61896365820 ),
  { 4522   58 }  (         0.438, 0.16552277290,   1574.84580128220 ),
  { 4522   59 }  (         0.312, 4.50639455819,   2125.87740737920 ),
  { 4522   60 }  (         0.280, 3.01441283033,    551.03160609700 ),
  { 4522   61 }  (         0.309, 0.67399908949,   2111.65031337760 ),
  { 4522   62 }  (         0.301, 3.06868080871,   1062.56332392690 ),
  { 4522   63 }  (         0.236, 1.94696842200,   1485.98012106520 ),
  { 4522   64 }  (         0.235, 3.41850395941,    199.07200143640 ),
  { 4522   65 }  (         0.246, 2.61803442505,    309.27832265580 ),
  { 4522   66 }  (         0.238, 2.56643737684,    539.98590583310 ),
  { 4522   67 }  (         0.248, 2.96997778167,   2648.45482547300 ),
  { 4522   68 }  (         0.209, 5.82481690851,   1471.75302706360 ),
  { 4522   69 }  (         0.205, 1.20202002469,   1056.20053645150 ),
  { 4522   70 }  (         0.188, 0.97113663101,   1685.05212250160 ),
  { 4522   71 }  (         0.137, 2.91203499563,   1699.27921650320 ),
  { 4522   72 }  (         0.131, 1.79274504072,   1063.31408345230 ),
  { 4522   73 }  (         0.161, 1.05926568614,   1795.25844372100 ),
  { 4522   74 }  (         0.112, 2.62660288825,    440.82528487760 ),
  { 4522   75 }  (         0.110, 3.56263668146,    227.52618943960 ),
  { 4522   76 }  (         0.114, 6.13907482464,   1038.04128918680 ),
  { 4522   77 }  (         0.103, 4.64287101040,      3.18139373770 ),
  { 4522   78 }  (         0.123, 4.81268110532,     21.34064100240 ),
  { 4522   79 }  (         0.102, 4.27603827970,   1375.77379984580 ),
  { 4522   80 }  (         0.089, 1.22926014128,   1898.35121793960 ),
  { 4522   81 }  (         0.080, 0.62129648755,    831.85574074960 )
  (*$endif *)
      );
  (*@\\\0000001401*)
  (*@/// vsop87_jup_b3:array[0.. 41,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_b3:array[0..  8,0..2] of extended = (
  (*$else *)
  vsop87_jup_b3:array[0.. 41,0..2] of extended = (
  (*$endif *)
  { 4523    1 }  (       251.624, 3.38087923084,    529.69096509460 ),
  { 4523    2 }  (       121.738, 2.73311837200,    522.57741809380 ),
  { 4523    3 }  (        48.694, 1.03689996685,    536.80451209540 ),
  { 4523    4 }  (        10.988, 2.31463561347,   1052.26838318840 ),
  { 4523    5 }  (         8.067, 2.76729757621,    515.46387109300 ),
  { 4523    6 }  (         6.205, 1.78115827370,   1066.49547719000 ),
  { 4523    7 }  (         7.287, 4.25268318975,   1059.38193018920 ),
  { 4523    8 }  (         3.627, 1.13028917221,    543.91805909620 ),
  { 4523    9 }  (         2.798, 3.14159265359,      0.00000000000 )
  (*$ifndef meeus *)                                                 ,
  { 4523   10 }  (         1.898, 2.28934054087,      7.11354700080 ),
  { 4523   11 }  (         1.643, 1.77507208483,   1045.15483618760 ),
  { 4523   12 }  (         0.945, 0.45261136388,    632.78373931320 ),
  { 4523   13 }  (         0.758, 0.30577920142,    949.17560896980 ),
  { 4523   14 }  (         0.731, 2.63748223583,     14.22709400160 ),
  { 4523   15 }  (         0.876, 0.32927768725,   1589.07289528380 ),
  { 4523   16 }  (         0.678, 2.36909615348,   1581.95934828300 ),
  { 4523   17 }  (         0.623, 2.48056213600,   1596.18644228460 ),
  { 4523   18 }  (         0.736, 1.52532370632,    735.87651353180 ),
  { 4523   19 }  (         0.499, 3.67985494258,    419.48464387520 ),
  { 4523   20 }  (         0.454, 0.26977404624,    942.06206196900 ),
  { 4523   21 }  (         0.453, 3.18232334886,    526.50957135690 ),
  { 4523   22 }  (         0.409, 2.88147337106,    110.20632121940 ),
  { 4523   23 }  (         0.347, 5.76244285870,    103.09277421860 ),
  { 4523   24 }  (         0.310, 2.98017326384,    508.35032409220 ),
  { 4523   25 }  (         0.321, 4.40642025933,    532.87235883230 ),
  { 4523   26 }  (         0.300, 1.66936571536,    625.67019231240 ),
  { 4523   27 }  (         0.295, 1.75924202728,   1073.60902419080 ),
  { 4523   28 }  (         0.282, 3.11087801399,    533.62311835770 ),
  { 4523   29 }  (         0.263, 0.55255030187,    426.59819087600 ),
  { 4523   30 }  (         0.208, 2.17540496886,   1155.36115740700 ),
  { 4523   31 }  (         0.183, 4.34670868038,    525.75881183150 ),
  { 4523   32 }  (         0.180, 6.07777744541,    639.89728631400 ),
  { 4523   33 }  (         0.159, 2.60843864402,   1162.47470440780 ),
  { 4523   34 }  (         0.117, 4.70141431381,     95.97922721780 ),
  { 4523   35 }  (         0.107, 5.48942805114,    433.71173787680 ),
  { 4523   36 }  (         0.105, 3.75192101775,    316.39186965660 ),
  { 4523   37 }  (         0.130, 1.37897716939,    323.50541665740 ),
  { 4523   38 }  (         0.094, 3.05797832024,   1265.56747862640 ),
  { 4523   39 }  (         0.114, 3.75170981478,    117.31986822020 ),
  { 4523   40 }  (         0.095, 0.54905691533,   1169.58825140860 ),
  { 4523   41 }  (         0.088, 3.26874502411,    213.29909543800 ),
  { 4523   42 }  (         0.098, 2.00704668688,   1574.84580128220 )
  (*$endif *)
      );
  (*@\\\0000000F01*)
  (*@/// vsop87_jup_b4:array[0.. 5,0..2] of extended = (..); *)
  (*$ifdef meeus *)
  vsop87_jup_b4:array[0..  5,0..2] of extended = (
  (*$else *)
  vsop87_jup_b4:array[0.. 11,0..2] of extended = (
  (*$endif *)
  { 4524    1 }  (        15.050, 4.52956999637,    522.57741809380 ),
  { 4524    2 }  (         5.370, 4.47427159142,    529.69096509460 ),
  { 4524    3 }  (         4.456, 5.43908581047,    536.80451209540 ),
  { 4524    4 }  (         3.422, 0.00000000000,      0.00000000000 ),
  { 4524    5 }  (         1.833, 4.51807036227,    515.46387109300 ),
  { 4524    6 }  (         1.322, 4.20117611581,   1052.26838318840 )
  (*$ifndef meeus *)                                                 ,
  { 4524    7 }  (         0.755, 5.59451554966,    543.91805909620 ),
  { 4524    8 }  (         0.512, 0.05803177475,   1066.49547719000 ),
  { 4524    9 }  (         0.282, 3.66807771223,   1059.38193018920 ),
  { 4524   10 }  (         0.147, 3.56490986181,   1045.15483618760 ),
  { 4524   11 }  (         0.142, 5.69936472988,      7.11354700080 ),
  { 4524   12 }  (         0.112, 1.16718383135,     14.22709400160 )
  (*$endif *)
      );
  (*@\\\0000000C01*)
  (*@/// vsop87_jup_b5:array[0.. 0,0..2] of extended = (..); *)
  (*$ifdef meeus *)
  vsop87_jup_b5:array[0..  0,0..2] of extended = (
  (*$else *)
  vsop87_jup_b5:array[0..  4,0..2] of extended = (
  (*$endif *)
  { 4525    1 }  (         1.445, 0.09198554072,    522.57741809380 )
  (*$ifndef meeus *)                                                 ,
  { 4525    2 }  (         0.368, 0.00874408003,    515.46387109300 ),
  { 4525    3 }  (         0.304, 3.27902945138,    536.80451209540 ),
  { 4525    4 }  (         0.129, 0.33959775247,    529.69096509460 ),
  { 4525    5 }  (         0.095, 1.29305954542,    543.91805909620 )
  (*$endif *)
      );
  (*@\\\*)
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_jup_b0)) and (nr<=high(vsop87_jup_b0)) then begin
        a:=vsop87_jup_b0[nr,0];  b:=vsop87_jup_b0[nr,1];  c:=vsop87_jup_b0[nr,2];
        end;
      1: if (nr>=low(vsop87_jup_b1)) and (nr<=high(vsop87_jup_b1)) then begin
        a:=vsop87_jup_b1[nr,0];  b:=vsop87_jup_b1[nr,1];  c:=vsop87_jup_b1[nr,2];
        end;
      2: if (nr>=low(vsop87_jup_b2)) and (nr<=high(vsop87_jup_b2)) then begin
        a:=vsop87_jup_b2[nr,0];  b:=vsop87_jup_b2[nr,1];  c:=vsop87_jup_b2[nr,2];
        end;
      3: if (nr>=low(vsop87_jup_b3)) and (nr<=high(vsop87_jup_b3)) then begin
        a:=vsop87_jup_b3[nr,0];  b:=vsop87_jup_b3[nr,1];  c:=vsop87_jup_b3[nr,2];
        end;
      4: if (nr>=low(vsop87_jup_b4)) and (nr<=high(vsop87_jup_b4)) then begin
        a:=vsop87_jup_b4[nr,0];  b:=vsop87_jup_b4[nr,1];  c:=vsop87_jup_b4[nr,2];
        end;
      5: if (nr>=low(vsop87_jup_b5)) and (nr<=high(vsop87_jup_b5)) then begin
        a:=vsop87_jup_b5[nr,0];  b:=vsop87_jup_b5[nr,1];  c:=vsop87_jup_b5[nr,2];
        end;
      end;
    end;
  end;
(*@\\\0000000301*)
(*@/// function TVSOPJupiter.LongitudeFactor(nr,index: integer):TVSOPEntry; *)
function TVSOPJupiter.LongitudeFactor(nr,index: integer):TVSOPEntry;
const
  (*@/// vsop87_jup_l0:array[0..759,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_l0:array[0.. 63,0..2] of extended = (
  (*$else *)
  vsop87_jup_l0:array[0..759,0..2] of extended = (
  (*$endif *)
  { 4510    1 }  (  59954691.495, 0.00000000000,      0.00000000000 ),
  { 4510    2 }  (   9695898.711, 5.06191793105,    529.69096509460 ),
  { 4510    3 }  (    573610.145, 1.44406205976,      7.11354700080 ),
  { 4510    4 }  (    306389.180, 5.41734729976,   1059.38193018920 ),
  { 4510    5 }  (     97178.280, 4.14264708819,    632.78373931320 ),
  { 4510    6 }  (     72903.096, 3.64042909255,    522.57741809380 ),
  { 4510    7 }  (     64263.986, 3.41145185203,    103.09277421860 ),
  { 4510    8 }  (     39806.051, 2.29376744855,    419.48464387520 ),
  { 4510    9 }  (     38857.780, 1.27231724860,    316.39186965660 ),
  { 4510   10 }  (     27964.622, 1.78454589485,    536.80451209540 ),
  { 4510   11 }  (     13589.738, 5.77481031590,   1589.07289528380 ),
  { 4510   12 }  (      8246.362, 3.58227961655,    206.18554843720 ),
  { 4510   13 }  (      8768.686, 3.63000324417,    949.17560896980 ),
  { 4510   14 }  (      7368.057, 5.08101125612,    735.87651353180 ),
  { 4510   15 }  (      6263.171, 0.02497643742,    213.29909543800 ),
  { 4510   16 }  (      6114.050, 4.51319531666,   1162.47470440780 ),
  { 4510   17 }  (      4905.419, 1.32084631684,    110.20632121940 ),
  { 4510   18 }  (      5305.283, 1.30671236848,     14.22709400160 ),
  { 4510   19 }  (      5305.457, 4.18625053495,   1052.26838318840 ),
  { 4510   20 }  (      4647.249, 4.69958109497,      3.93215326310 ),
  { 4510   21 }  (      3045.009, 4.31675960318,    426.59819087600 ),
  { 4510   22 }  (      2610.001, 1.56667594850,    846.08283475120 ),
  { 4510   23 }  (      2028.191, 1.06376547379,      3.18139373770 ),
  { 4510   24 }  (      1764.768, 2.14148077766,   1066.49547719000 ),
  { 4510   25 }  (      1722.983, 3.88036008872,   1265.56747862640 ),
  { 4510   26 }  (      1920.959, 0.97168928755,    639.89728631400 ),
  { 4510   27 }  (      1633.217, 3.58201089758,    515.46387109300 ),
  { 4510   28 }  (      1431.997, 4.29683690269,    625.67019231240 ),
  { 4510   29 }  (       973.278, 4.09764957065,     95.97922721780 ),
  { 4510   30 }  (       884.439, 2.43701426123,    412.37109687440 ),
  { 4510   31 }  (       732.875, 6.08534113239,    838.96928775040 ),
  { 4510   32 }  (       731.072, 3.80591233956,   1581.95934828300 ),
  { 4510   33 }  (       691.928, 6.13368222939,   2118.76386037840 ),
  { 4510   34 }  (       709.190, 1.29272573658,    742.99006053260 ),
  { 4510   35 }  (       614.464, 4.10853496756,   1478.86657406440 ),
  { 4510   36 }  (       495.224, 3.75567461379,    323.50541665740 ),
  { 4510   37 }  (       581.902, 4.53967717552,    309.27832265580 ),
  { 4510   38 }  (       375.657, 4.70299124833,   1368.66025284500 ),
  { 4510   39 }  (       389.864, 4.89716105852,   1692.16566950240 ),
  { 4510   40 }  (       341.006, 5.71452525783,    533.62311835770 ),
  { 4510   41 }  (       330.458, 4.74049819491,      0.04818410980 ),
  { 4510   42 }  (       440.854, 2.95818460943,    454.90936652730 ),
  { 4510   43 }  (       417.266, 1.03554430161,      2.44768055480 ),
  { 4510   44 }  (       244.170, 5.22020878900,    728.76296653100 ),
  { 4510   45 }  (       261.540, 1.87652461032,      0.96320784650 ),
  { 4510   46 }  (       256.568, 3.72410724159,    199.07200143640 ),
  { 4510   47 }  (       261.009, 0.82047246448,    380.12776796000 ),
  { 4510   48 }  (       220.382, 1.65115015995,    543.91805909620 ),
  { 4510   49 }  (       201.996, 1.80684574186,   1375.77379984580 ),
  { 4510   50 }  (       207.327, 1.85461666594,    525.75881183150 ),
  { 4510   51 }  (       197.046, 5.29252149016,   1155.36115740700 ),
  { 4510   52 }  (       235.141, 1.22693908124,    909.81873305460 ),
  { 4510   53 }  (       174.809, 5.90973505276,    956.28915597060 ),
  { 4510   54 }  (       149.368, 4.37745104275,   1685.05212250160 ),
  { 4510   55 }  (       175.184, 3.22634903433,   1898.35121793960 ),
  { 4510   56 }  (       175.191, 3.72966554761,    942.06206196900 ),
  { 4510   57 }  (       157.909, 4.36483921766,   1795.25844372100 ),
  { 4510   58 }  (       137.871, 1.31797920785,   1169.58825140860 ),
  { 4510   59 }  (       117.495, 2.50022140890,   1596.18644228460 ),
  { 4510   60 }  (       150.502, 3.90625022622,     74.78159856730 ),
  { 4510   61 }  (       116.757, 3.38920921041,      0.52126486180 ),
  { 4510   62 }  (       105.895, 4.55439798236,    526.50957135690 ),
  { 4510   63 }  (       130.531, 4.16867945489,   1045.15483618760 ),
  { 4510   64 }  (       141.445, 3.13568357861,    491.55792945680 )
  (*$ifndef meeus *)                                                 ,
  { 4510   65 }  (        99.511, 1.42117395747,    532.87235883230 ),
  { 4510   66 }  (        96.137, 1.18156870005,    117.31986822020 ),
  { 4510   67 }  (        91.758, 0.85756633461,   1272.68102562720 ),
  { 4510   68 }  (        87.695, 1.21738140813,    453.42489381900 ),
  { 4510   69 }  (        68.507, 2.35242959478,      2.92076130680 ),
  { 4510   70 }  (        66.098, 5.34386149468,   1471.75302706360 ),
  { 4510   71 }  (        77.401, 4.42676337124,     39.35687591520 ),
  { 4510   72 }  (        72.006, 4.23834923691,   2111.65031337760 ),
  { 4510   73 }  (        63.406, 4.97665525033,      0.75075952540 ),
  { 4510   74 }  (        59.427, 4.11130498612,   2001.44399215820 ),
  { 4510   75 }  (        62.481, 0.51211384012,    220.41264243880 ),
  { 4510   76 }  (        66.532, 2.98864358135,   2214.74308759620 ),
  { 4510   77 }  (        60.194, 4.12628179571,      4.19278569400 ),
  { 4510   78 }  (        56.012, 1.15493222602,     21.34064100240 ),
  { 4510   79 }  (        52.854, 0.91207215543,     10.29494073850 ),
  { 4510   80 }  (        70.297, 5.14180555282,    835.03713448730 ),
  { 4510   81 }  (        51.916, 4.10048180020,   1258.45393162560 ),
  { 4510   82 }  (        46.442, 4.66531163524,      5.62907429250 ),
  { 4510   83 }  (        58.190, 5.86646380344,   5753.38488489680 ),
  { 4510   84 }  (        40.103, 4.68801114087,      0.16005869440 ),
  { 4510   85 }  (        46.654, 4.79394835282,    305.34616939270 ),
  { 4510   86 }  (        39.298, 4.25448423697,    853.19638175200 ),
  { 4510   87 }  (        46.042, 5.10983515150,      4.66586644600 ),
  { 4510   88 }  (        54.459, 1.57072704127,    983.11585891360 ),
  { 4510   89 }  (        38.920, 6.07592905580,    518.64526483070 ),
  { 4510   90 }  (        38.450, 2.43836870888,    433.71173787680 ),
  { 4510   91 }  (        46.800, 3.54640538283,      5.41662597140 ),
  { 4510   92 }  (        41.830, 4.67982493646,    302.16477565500 ),
  { 4510   93 }  (        35.920, 2.45088036239,    430.53034413910 ),
  { 4510   94 }  (        37.888, 0.21127448431,   2648.45482547300 ),
  { 4510   95 }  (        39.190, 1.71835571629,     11.04570026390 ),
  { 4510   96 }  (        37.567, 6.19481310233,    831.85574074960 ),
  { 4510   97 }  (        35.828, 4.61459907698,   2008.55753915900 ),
  { 4510   98 }  (        43.402, 0.14992289081,    528.20649238630 ),
  { 4510   99 }  (        31.598, 5.14073450755,   1788.14489672020 ),
  { 4510  100 }  (        29.849, 5.34441117167,   2221.85663459700 ),
  { 4510  101 }  (        32.811, 5.28907118836,     88.86568021700 ),
  { 4510  102 }  (        27.686, 1.85227036207,      0.21244832110 ),
  { 4510  103 }  (        25.820, 3.85920882494,   2317.83586181480 ),
  { 4510  104 }  (        24.705, 2.63495214991,    114.13847448250 ),
  { 4510  105 }  (        33.844, 1.00563073268,   9683.59458111640 ),
  { 4510  106 }  (        24.266, 3.82355417268,   1574.84580128220 ),
  { 4510  107 }  (        27.111, 2.80845435102,     18.15924726470 ),
  { 4510  108 }  (        26.837, 1.77586123775,    532.13864564940 ),
  { 4510  109 }  (        26.064, 2.74361318804,   2531.13495725280 ),
  { 4510  110 }  (        30.765, 0.42330537728,      1.48447270830 ),
  { 4510  111 }  (        30.476, 3.66677894407,    508.35032409220 ),
  { 4510  112 }  (        23.282, 3.24372142416,    984.60033162190 ),
  { 4510  113 }  (        19.445, 0.52370214471,     14.97785352700 ),
  { 4510  114 }  (        19.332, 4.86314494382,   1361.54670584420 ),
  { 4510  115 }  (        22.910, 3.84914895064,   2428.04218303420 ),
  { 4510  116 }  (        21.617, 6.01696940024,   1063.31408345230 ),
  { 4510  117 }  (        20.155, 5.59582008789,    527.24328453980 ),
  { 4510  118 }  (        23.732, 2.52766031921,    494.26624244250 ),
  { 4510  119 }  (        20.189, 1.01560227681,    628.85158605010 ),
  { 4510  120 }  (        15.994, 5.09003530653,    529.73914920440 ),
  { 4510  121 }  (        16.134, 5.27095037302,    142.44965013380 ),
  { 4510  122 }  (        20.697, 4.03443281612,    355.74874557180 ),
  { 4510  123 }  (        21.479, 1.28668134295,     35.42472265210 ),
  { 4510  124 }  (        14.964, 4.86039684390,   2104.53676637680 ),
  { 4510  125 }  (        17.242, 1.59187913206,   1439.50969814920 ),
  { 4510  126 }  (        15.994, 1.89222417794,    529.64278098480 ),
  { 4510  127 }  (        17.958, 4.30178016003,      6.15033915430 ),
  { 4510  128 }  (        13.279, 2.18943981644,   1055.44977692610 ),
  { 4510  129 }  (        14.148, 2.71597731671,      0.26063243090 ),
  { 4510  130 }  (        14.689, 0.87944553412,     99.16062095550 ),
  { 4510  131 }  (        14.202, 2.41335693735,    530.65417294110 ),
  { 4510  132 }  (        15.320, 6.07703092728,    149.56319713460 ),
  { 4510  133 }  (        15.832, 4.11682440678,    636.71589257630 ),
  { 4510  134 }  (        12.398, 2.61042299578,    405.25754987360 ),
  { 4510  135 }  (        16.199, 2.77035044582,    760.25553592000 ),
  { 4510  136 }  (        13.665, 3.56039678310,    217.23124870110 ),
  { 4510  137 }  (        15.261, 2.81824770887,    621.73803904930 ),
  { 4510  138 }  (        14.681, 6.26423732742,    569.04784100980 ),
  { 4510  139 }  (        12.529, 1.39077179081,      7.06536289100 ),
  { 4510  140 }  (        11.677, 3.60447374272,   2634.22773147140 ),
  { 4510  141 }  (        11.603, 4.60461756191,      7.16173111060 ),
  { 4510  142 }  (        12.152, 0.24540531919,   1485.98012106520 ),
  { 4510  143 }  (        11.347, 2.00818458261,   1073.60902419080 ),
  { 4510  144 }  (        11.242, 2.48000947870,    423.41679713830 ),
  { 4510  145 }  (        10.942, 5.03602448252,    458.84151979040 ),
  { 4510  146 }  (        11.117, 4.04973271023,    519.39602435610 ),
  { 4510  147 }  (        12.256, 4.30153222783,    604.47256366190 ),
  { 4510  148 }  (        13.149, 2.72189077702,   1364.72809958190 ),
  { 4510  149 }  (        10.604, 3.11518747072,      1.27202438720 ),
  { 4510  150 }  (         9.874, 1.70200068743,   1699.27921650320 ),
  { 4510  151 }  (        10.851, 5.08554552028,   2324.94940881560 ),
  { 4510  152 }  (        10.692, 2.51401681528,   2847.52682690940 ),
  { 4510  153 }  (        12.640, 4.75572797691,    528.72775724810 ),
  { 4510  154 }  (        10.084, 4.05599810206,     38.13303563780 ),
  { 4510  155 }  (        11.536, 2.35034215745,    643.82943957710 ),
  { 4510  156 }  (        10.247, 3.63479911496,   2744.43405269080 ),
  { 4510  157 }  (        10.105, 3.65845333837,    107.02492748170 ),
  { 4510  158 }  (        10.121, 1.31482648275,   1905.46476494040 ),
  { 4510  159 }  (         9.341, 5.92176693887,   1148.24761040620 ),
  { 4510  160 }  (         8.796, 2.77421822809,      6.59228213900 ),
  { 4510  161 }  (         8.420, 4.52537756809,   1677.93857550080 ),
  { 4510  162 }  (        10.128, 2.09034472544,    511.53171782990 ),
  { 4510  163 }  (         8.272, 2.98682673354,    540.73666535850 ),
  { 4510  164 }  (         9.753, 1.22438911827,     32.24332891440 ),
  { 4510  165 }  (        10.630, 2.07777800288,     92.04707395470 ),
  { 4510  166 }  (         7.850, 0.98996894618,    408.43894361130 ),
  { 4510  167 }  (         8.811, 3.46911754939,   1021.24889455140 ),
  { 4510  168 }  (         7.946, 2.86682926070,   2125.87740737920 ),
  { 4510  169 }  (         8.575, 5.29590411702,    415.55249061210 ),
  { 4510  170 }  (         7.841, 6.08025056721,     70.84944530420 ),
  { 4510  171 }  (         7.706, 1.69832954219,      8.07675484730 ),
  { 4510  172 }  (         7.265, 4.65503563919,    629.60234557550 ),
  { 4510  173 }  (         7.164, 4.93400217968,   1056.20053645150 ),
  { 4510  174 }  (         7.247, 4.61607677560,   2420.92863603340 ),
  { 4510  175 }  (         7.753, 2.12871653382,     33.94024994380 ),
  { 4510  176 }  (         6.645, 0.45647460873,    635.96513305090 ),
  { 4510  177 }  (         9.377, 4.03158388202,   2810.92146160520 ),
  { 4510  178 }  (         8.263, 1.23558676139,   1802.37199072180 ),
  { 4510  179 }  (         6.341, 0.07278001580,    202.25339517410 ),
  { 4510  180 }  (         6.383, 3.54310669809,   1891.23767093880 ),
  { 4510  181 }  (         7.902, 2.32510002614,    230.56457082540 ),
  { 4510  182 }  (         6.214, 4.54560345237,      2.70831298570 ),
  { 4510  183 }  (         7.347, 1.24457237337,     24.37902238820 ),
  { 4510  184 }  (         7.451, 3.02719199239,    330.61896365820 ),
  { 4510  185 }  (         6.220, 1.77687561489,   1062.56332392690 ),
  { 4510  186 }  (         5.674, 5.14132196367,    746.92221379570 ),
  { 4510  187 }  (         5.855, 5.42130172896,     28.31117565130 ),
  { 4510  188 }  (         5.629, 3.24348217277,    529.16970023280 ),
  { 4510  189 }  (         7.652, 0.52813391052,    672.14061522840 ),
  { 4510  190 }  (         5.456, 3.34716871364,   2950.61960112800 ),
  { 4510  191 }  (         7.127, 1.43497795005,      6.21977512350 ),
  { 4510  192 }  (         5.388, 4.90175095580,     69.15252427480 ),
  { 4510  193 }  (         5.618, 4.97903783721,   2641.34127847220 ),
  { 4510  194 }  (         5.844, 2.95364118152,    490.33408917940 ),
  { 4510  195 }  (         4.943, 5.37597740579,    721.64941953020 ),
  { 4510  196 }  (         5.062, 4.84282906467,     31.01948863700 ),
  { 4510  197 }  (         5.163, 5.07410777073,     67.66805156650 ),
  { 4510  198 }  (         4.739, 6.10248862834,    106.27416795630 ),
  { 4510  199 }  (         4.879, 0.07095292379,     78.71375183040 ),
  { 4510  200 }  (         4.854, 5.63875661096,      1.69692102940 ),
  { 4510  201 }  (         5.629, 3.73871604865,    530.21222995640 ),
  { 4510  202 }  (         4.471, 4.49152882547,    505.31194270640 ),
  { 4510  203 }  (         4.313, 4.79369370451,    535.10759106600 ),
  { 4510  204 }  (         4.280, 0.54783823710,      1.43628859850 ),
  { 4510  205 }  (         4.453, 0.50551854591,    524.06189080210 ),
  { 4510  206 }  (         4.936, 4.82992988255,    422.66603761290 ),
  { 4510  207 }  (         4.701, 3.41634869046,   3060.82592234740 ),
  { 4510  208 }  (         4.261, 2.67044686458,    561.93429400900 ),
  { 4510  209 }  (         4.156, 4.00660658688,     99.91138048090 ),
  { 4510  210 }  (         4.561, 2.29650164054,   3163.91869656600 ),
  { 4510  211 }  (         4.414, 5.67224020329,   1464.63948006280 ),
  { 4510  212 }  (         5.345, 0.31513851830,   1289.94650101460 ),
  { 4510  213 }  (         5.269, 3.89116469022,    191.95845443560 ),
  { 4510  214 }  (         3.855, 4.28942301453,   1994.33044515740 ),
  { 4510  215 }  (         4.210, 5.32763589447,   2538.24850425360 ),
  { 4510  216 }  (         3.949, 4.56507101172,   1382.88734684660 ),
  { 4510  217 }  (         3.885, 1.56778786810,    647.01083331480 ),
  { 4510  218 }  (         4.227, 5.51697599030,   5223.69391980220 ),
  { 4510  219 }  (         4.129, 2.81119457666,    416.30325013750 ),
  { 4510  220 }  (         3.663, 4.35187510477,   2737.32050569000 ),
  { 4510  221 }  (         3.566, 5.48243943375,    750.10360753340 ),
  { 4510  222 }  (         4.330, 0.84941756640,    531.17543780290 ),
  { 4510  223 }  (         4.093, 0.19980340452,    525.02509864860 ),
  { 4510  224 }  (         4.022, 1.92293311337,   1512.80682400820 ),
  { 4510  225 }  (         3.400, 6.00302355875,      1.22384027740 ),
  { 4510  226 }  (         3.496, 0.31252921473,    597.35901666110 ),
  { 4510  227 }  (         3.299, 4.27596694481,    526.77020378780 ),
  { 4510  228 }  (         3.226, 2.90455264496,    963.40270297140 ),
  { 4510  229 }  (         3.150, 3.81061764181,    280.96714700450 ),
  { 4510  230 }  (         4.129, 4.74946631331,      0.89377187730 ),
  { 4510  231 }  (         3.840, 1.91064405186,    378.64329525170 ),
  { 4510  232 }  (         3.057, 1.65589659685,    528.94020556920 ),
  { 4510  233 }  (         3.011, 1.59276337369,    224.34479570190 ),
  { 4510  234 }  (         3.196, 5.86588452873,      4.14460158420 ),
  { 4510  235 }  (         3.628, 0.07930225897,    558.00214074590 ),
  { 4510  236 }  (         2.932, 0.41424445089,      7.86430652620 ),
  { 4510  237 }  (         3.316, 2.70211697795,    532.61172640140 ),
  { 4510  238 }  (         2.925, 4.47580363425,    533.88375078860 ),
  { 4510  239 }  (         3.690, 0.39897023849,    685.47393735270 ),
  { 4510  240 }  (         3.223, 2.45833032883,    960.22130923370 ),
  { 4510  241 }  (         3.059, 5.32616140812,    530.44172462000 ),
  { 4510  242 }  (         3.383, 4.42170370028,    312.45971639350 ),
  { 4510  243 }  (         3.320, 2.71417812514,    495.75071515080 ),
  { 4510  244 }  (         2.697, 5.23146633437,    739.80866679490 ),
  { 4510  245 }  (         3.590, 2.30999595873,    908.33426034630 ),
  { 4510  246 }  (         3.677, 5.07337955976,     73.29712585900 ),
  { 4510  247 }  (         2.618, 3.09118499149,   3267.01147078460 ),
  { 4510  248 }  (         2.796, 2.98942316119,    483.22054217860 ),
  { 4510  249 }  (         3.398, 3.29598270278,    911.30320576290 ),
  { 4510  250 }  (         3.352, 1.44391979336,    593.42686339800 ),
  { 4510  251 }  (         2.563, 3.35080110279,   2207.62954059540 ),
  { 4510  252 }  (         2.553, 0.36892288645,   1048.33622992530 ),
  { 4510  253 }  (         2.620, 3.82769874340,    520.12973753900 ),
  { 4510  254 }  (         3.356, 1.08315053878,     46.47042291600 ),
  { 4510  255 }  (         3.035, 5.52230028113,    618.55664531160 ),
  { 4510  256 }  (         3.397, 3.83084746522,    210.11770170030 ),
  { 4510  257 }  (         2.497, 0.47917884538,    945.24345570670 ),
  { 4510  258 }  (         2.341, 5.87941292649,   2751.54759969160 ),
  { 4510  259 }  (         2.656, 0.49713061045,   1057.89745748090 ),
  { 4510  260 }  (         2.581, 0.03759881914,      1.64453140270 ),
  { 4510  261 }  (         2.900, 2.50019054587,    525.49817940060 ),
  { 4510  262 }  (         3.153, 2.30900986177,    457.61767951300 ),
  { 4510  263 }  (         2.201, 3.94367109739,     31.49256938900 ),
  { 4510  264 }  (         2.381, 6.19252134885,    327.43756992050 ),
  { 4510  265 }  (         2.458, 0.65614291954,   9153.90361602180 ),
  { 4510  266 }  (         2.111, 5.61905648764,     16.46232623530 ),
  { 4510  267 }  (         2.130, 3.75880734109,    724.83081326790 ),
  { 4510  268 }  (         2.406, 2.29315649755,    195.13984817330 ),
  { 4510  269 }  (         2.166, 5.43262641046,    534.35683154060 ),
  { 4510  270 }  (         2.057, 1.49875151278,    551.03160609700 ),
  { 4510  271 }  (         2.676, 5.06374981112,    456.39383923560 ),
  { 4510  272 }  (         2.078, 5.28920097886,     76.26607127560 ),
  { 4510  273 }  (         2.261, 5.38117230692,   1781.03134971940 ),
  { 4510  274 }  (         2.356, 0.67392574097,    227.52618943960 ),
  { 4510  275 }  (         2.240, 3.18006978517,   3377.21779200400 ),
  { 4510  276 }  (         2.183, 3.08384250950,    524.27433912320 ),
  { 4510  277 }  (         2.119, 2.70107659927,    387.24131496080 ),
  { 4510  278 }  (         2.056, 4.82779196994,   2957.73314812880 ),
  { 4510  279 }  (         2.116, 6.20263841494,    209.36694217490 ),
  { 4510  280 }  (         2.712, 3.18157754631,   1474.67378837040 ),
  { 4510  281 }  (         2.127, 1.24424012514,    539.98590583310 ),
  { 4510  282 }  (         2.424, 3.57595925853,    953.10776223290 ),
  { 4510  283 }  (         1.947, 1.94468082546,    529.53090640020 ),
  { 4510  284 }  (         1.896, 4.01406242800,   2310.72231481400 ),
  { 4510  285 }  (         1.935, 4.10051493950,   3053.71237534660 ),
  { 4510  286 }  (         2.056, 6.27074148550,    245.54242435240 ),
  { 4510  287 }  (         2.108, 3.22886474225,    252.65597135320 ),
  { 4510  288 }  (         2.596, 2.77467278614,    177.87437278590 ),
  { 4510  289 }  (         1.919, 3.14834694111,    381.61224066830 ),
  { 4510  290 }  (         2.217, 1.92368906925,    535.91074021810 ),
  { 4510  291 }  (         1.947, 5.03751780002,    529.85102378900 ),
  { 4510  292 }  (         2.025, 4.82814272957,     17.26547538740 ),
  { 4510  293 }  (         1.945, 2.10611582568,   3480.31056622260 ),
  { 4510  294 }  (         1.899, 0.05104263891,    560.71045373160 ),
  { 4510  295 }  (         2.221, 0.58365090630,   3178.14579056760 ),
  { 4510  296 }  (         2.271, 1.67360565619,    731.94436026870 ),
  { 4510  297 }  (         1.706, 5.40277333462,     20.44686912510 ),
  { 4510  298 }  (         2.295, 4.20863103004,   1038.04128918680 ),
  { 4510  299 }  (         2.218, 3.65982280555,    282.45161971280 ),
  { 4510  300 }  (         2.181, 4.87369503022,    535.32003938710 ),
  { 4510  301 }  (         1.745, 1.34021867874,     25.12978191360 ),
  { 4510  302 }  (         1.601, 3.92730015840,     17.52610781830 ),
  { 4510  303 }  (         1.651, 0.63598292839,     17.40848773930 ),
  { 4510  304 }  (         1.826, 0.31592311031,    124.43341522100 ),
  { 4510  305 }  (         2.041, 0.15617294873,    598.84348936940 ),
  { 4510  306 }  (         1.494, 3.81418025130,    319.57326339430 ),
  { 4510  307 }  (         1.551, 5.25201528605,    437.64389113990 ),
  { 4510  308 }  (         1.852, 2.36130812462,     37.87240320690 ),
  { 4510  309 }  (         1.466, 1.72926380881,     59.80374504030 ),
  { 4510  310 }  (         1.417, 5.82273267086,     81.75213321620 ),
  { 4510  311 }  (         1.430, 1.17528806260,    440.82528487760 ),
  { 4510  312 }  (         1.906, 4.06896022692,   1819.63746610920 ),
  { 4510  313 }  (         1.397, 0.26383366743,     50.40257617910 ),
  { 4510  314 }  (         1.756, 2.32977483716,    938.12990870590 ),
  { 4510  315 }  (         1.487, 2.24866746540,     10.03430830760 ),
  { 4510  316 }  (         1.368, 3.56691602771,   1514.29129671650 ),
  { 4510  317 }  (         1.400, 4.84502200703,    295.05122865420 ),
  { 4510  318 }  (         1.344, 2.20177702122,    529.90341341570 ),
  { 4510  319 }  (         1.464, 1.42648716568,   1158.54255114470 ),
  { 4510  320 }  (         1.341, 1.15693423225,   2435.15573003500 ),
  { 4510  321 }  (         1.786, 5.44716330146,   2854.64037391020 ),
  { 4510  322 }  (         1.677, 6.22875777048,    833.55266177900 ),
  { 4510  323 }  (         1.471, 4.80574535807,    696.51963761660 ),
  { 4510  324 }  (         1.436, 1.45810957330,    537.76771994190 ),
  { 4510  325 }  (         1.657, 0.02890651793,    138.51749687070 ),
  { 4510  326 }  (         1.300, 3.14074420421,    547.85021235930 ),
  { 4510  327 }  (         1.343, 6.14827138025,    988.53248488500 ),
  { 4510  328 }  (         1.344, 4.78042160426,    529.47851677350 ),
  { 4510  329 }  (         1.234, 2.83294330979,   3583.40334044120 ),
  { 4510  330 }  (         1.651, 2.12056447005,   1061.82961074400 ),
  { 4510  331 }  (         1.479, 0.24646493075,   1593.00504854690 ),
  { 4510  332 }  (         1.413, 3.07444632745,   6283.07584999140 ),
  { 4510  333 }  (         1.246, 5.94882321661,   1056.93424963440 ),
  { 4510  334 }  (         1.225, 1.95642397635,   1969.20066324380 ),
  { 4510  335 }  (         1.388, 2.87749576073,   1023.95720753710 ),
  { 4510  336 }  (         1.263, 3.46181945031,     40.84134862350 ),
  { 4510  337 }  (         1.325, 4.15429781246,    916.93228005540 ),
  { 4510  338 }  (         1.477, 5.26691818477,    810.65811209910 ),
  { 4510  339 }  (         1.165, 4.65528125418,    944.98282327580 ),
  { 4510  340 }  (         1.137, 2.48561382158,      2.00573757010 ),
  { 4510  341 }  (         1.118, 3.80747957482,      7.00167241620 ),
  { 4510  342 }  (         1.138, 5.11611532241,    885.43971066640 ),
  { 4510  343 }  (         1.131, 1.54599459004,    775.23338944700 ),
  { 4510  344 }  (         1.477, 4.69742954455,    630.33605875840 ),
  { 4510  345 }  (         1.252, 1.34316620527,    739.05790726950 ),
  { 4510  346 }  (         1.273, 5.19070939905,   2097.42321937600 ),
  { 4510  347 }  (         1.446, 5.54999644374,     43.28902917830 ),
  { 4510  348 }  (         1.344, 4.75897665313,   1166.40685767090 ),
  { 4510  349 }  (         1.101, 4.56997613488,   3274.12501778540 ),
  { 4510  350 }  (         1.376, 3.60998729004,    415.29185818120 ),
  { 4510  351 }  (         1.437, 6.22410093972,    155.78297225810 ),
  { 4510  352 }  (         1.167, 4.09497264272,    203.00415469950 ),
  { 4510  353 }  (         1.237, 4.41132627005,    292.01284726840 ),
  { 4510  354 }  (         1.077, 2.57045229823,     25.27279426550 ),
  { 4510  355 }  (         1.341, 0.49262296655,    635.23141986800 ),
  { 4510  356 }  (         1.209, 3.36289125536,    521.61421024730 ),
  { 4510  357 }  (         1.030, 1.81822316284,    465.95506679120 ),
  { 4510  358 }  (         1.002, 3.21720955284,   2524.02141025200 ),
  { 4510  359 }  (         1.338, 1.26054917773,    902.70518605380 ),
  { 4510  360 }  (         1.037, 3.87858871885,   3370.10424500320 ),
  { 4510  361 }  (         1.224, 0.09219976028,    824.74219374880 ),
  { 4510  362 }  (         1.255, 3.04675952762,    447.79581952650 ),
  { 4510  363 }  (         0.991, 4.16587903812,    632.83192342300 ),
  { 4510  364 }  (         0.975, 3.80216680539,   2627.11418447060 ),
  { 4510  365 }  (         1.061, 5.60184374277,    732.69511979410 ),
  { 4510  366 }  (         1.049, 2.94931080683,   3693.60966166060 ),
  { 4510  367 }  (         0.984, 0.98260254313,    632.73555520340 ),
  { 4510  368 }  (         1.050, 2.20935815967,      7.22542158540 ),
  { 4510  369 }  (         0.996, 5.41921062583,   1059.43011429900 ),
  { 4510  370 }  (         0.961, 0.87315283361,    544.66881862160 ),
  { 4510  371 }  (         1.175, 3.09093466406,   1894.41906467650 ),
  { 4510  372 }  (         1.049, 5.81616384906,     26.82670294300 ),
  { 4510  373 }  (         1.161, 0.01274801567,    850.01498801430 ),
  { 4510  374 }  (         1.109, 3.63294273717,    306.83064210100 ),
  { 4510  375 }  (         1.077, 0.95716576092,    608.40471692500 ),
  { 4510  376 }  (         1.288, 4.23019288942,   1215.16490244730 ),
  { 4510  377 }  (         1.060, 3.85856787901,    631.82053146670 ),
  { 4510  378 }  (         1.251, 6.15889818604,    462.02291352810 ),
  { 4510  379 }  (         1.165, 3.50653563773,      8.59801970910 ),
  { 4510  380 }  (         0.933, 4.62559759882,   1049.08698945070 ),
  { 4510  381 }  (         1.035, 1.30805283339,    633.74694715970 ),
  { 4510  382 }  (         1.238, 2.21195391602,  25558.21217647960 ),
  { 4510  383 }  (         1.240, 2.27960685992,      6.90109867970 ),
  { 4510  384 }  (         0.942, 4.14526324371,    945.99421523210 ),
  { 4510  385 }  (         0.927, 6.10893117637,    514.71311156760 ),
  { 4510  386 }  (         0.914, 6.17656044376,    952.35700270750 ),
  { 4510  387 }  (         0.893, 4.27448748055,      0.63313944640 ),
  { 4510  388 }  (         1.045, 1.64682770236,    565.11568774670 ),
  { 4510  389 }  (         0.903, 1.94250156640,   3796.70243587920 ),
  { 4510  390 }  (         1.162, 5.51229668479,      2.96894541660 ),
  { 4510  391 }  (         0.901, 3.03568112112,    460.53844081980 ),
  { 4510  392 }  (         0.903, 2.24012822393,    523.54062594030 ),
  { 4510  393 }  (         1.060, 5.28027224466,   3171.03224356680 ),
  { 4510  394 }  (         1.064, 0.99330150801,    320.32402291970 ),
  { 4510  395 }  (         0.970, 4.56607888439,    429.04587143080 ),
  { 4510  396 }  (         1.071, 4.33203090957,    610.69233878540 ),
  { 4510  397 }  (         0.865, 0.21831429230,   1098.73880610440 ),
  { 4510  398 }  (         0.865, 2.82123742108,   1060.34513803570 ),
  { 4510  399 }  (         0.882, 4.80076824948,    384.05992122310 ),
  { 4510  400 }  (         0.959, 5.45468005818,    451.94042111070 ),
  { 4510  401 }  (         1.042, 5.79270325150,    303.86169668440 ),
  { 4510  402 }  (         0.784, 1.85150700827,    313.21047591890 ),
  { 4510  403 }  (         1.083, 1.40526460812,     72.07328558160 ),
  { 4510  404 }  (         0.782, 3.03559242565,      5.84152261360 ),
  { 4510  405 }  (         0.854, 1.22236205478,    611.44309831080 ),
  { 4510  406 }  (         0.996, 2.22139794743,   1059.33374607940 ),
  { 4510  407 }  (         0.719, 4.92550252164,    421.93232443000 ),
  { 4510  408 }  (         0.953, 3.98347050083,    836.52160719560 ),
  { 4510  409 }  (         0.822, 4.49679856387,  10213.28554621100 ),
  { 4510  410 }  (         0.707, 2.16473400319,   2228.97018159780 ),
  { 4510  411 }  (         0.715, 4.62515255534,    385.54439393140 ),
  { 4510  412 }  (         0.737, 4.63776694324,   1134.16352875650 ),
  { 4510  413 }  (         0.730, 1.87179326186,    153.49535039770 ),
  { 4510  414 }  (         0.709, 2.93132115910,    417.03696332040 ),
  { 4510  415 }  (         0.926, 1.77006317007,   2332.06295581640 ),
  { 4510  416 }  (         0.864, 3.03246275970,   1041.22268292450 ),
  { 4510  417 }  (         0.708, 6.01601101389,    395.10562148700 ),
  { 4510  418 }  (         0.935, 6.01864676296,    173.94221952280 ),
  { 4510  419 }  (         0.695, 1.39408383356,    432.01481684740 ),
  { 4510  420 }  (         0.687, 3.06548397586,    529.95159752550 ),
  { 4510  421 }  (         0.677, 3.58357527210,    244.31858407500 ),
  { 4510  422 }  (         0.850, 5.46114025921,     41.05379694460 ),
  { 4510  423 }  (         0.817, 4.65315342412,    535.84130424890 ),
  { 4510  424 }  (         0.652, 0.44173759183,   1201.83158032300 ),
  { 4510  425 }  (         0.711, 0.96283289310,    373.01422095920 ),
  { 4510  426 }  (         0.665, 1.03244633471,    623.22251175760 ),
  { 4510  427 }  (         0.643, 5.05335060049,    522.62560220360 ),
  { 4510  428 }  (         0.639, 4.22718483639,     25.86349509650 ),
  { 4510  429 }  (         0.718, 5.07576900710,   1058.41872234270 ),
  { 4510  430 }  (         0.664, 2.43728454444,   1585.14074202070 ),
  { 4510  431 }  (         0.833, 1.49468440213,    563.63121503840 ),
  { 4510  432 }  (         0.760, 4.34849823663,    100.64509366380 ),
  { 4510  433 }  (         0.633, 4.31796718640,   3590.51688744200 ),
  { 4510  434 }  (         0.629, 6.23431126402,    679.25416222920 ),
  { 4510  435 }  (         0.617, 2.68075016456,   3899.79521009780 ),
  { 4510  436 }  (         0.646, 2.88581188015,     13.49338081870 ),
  { 4510  437 }  (         0.768, 3.18498076120,   1151.42900414390 ),
  { 4510  438 }  (         0.731, 5.86653168561,    501.37978944330 ),
  { 4510  439 }  (         0.652, 0.82865771780,   2015.67108615980 ),
  { 4510  440 }  (         0.796, 5.36663489938,    420.96911658350 ),
  { 4510  441 }  (         0.647, 4.74965662438,    567.82400073240 ),
  { 4510  442 }  (         0.845, 1.69406147722,   1744.85586754190 ),
  { 4510  443 }  (         0.802, 5.79824707751,    981.63138620530 ),
  { 4510  444 }  (         0.764, 5.05232933368,    827.92358748650 ),
  { 4510  445 }  (         0.604, 5.11265182908,   1159.29331067010 ),
  { 4510  446 }  (         0.682, 3.68248136835,   2281.23049651060 ),
  { 4510  447 }  (         0.740, 0.74512356954,   1261.63532536330 ),
  { 4510  448 }  (         0.666, 2.06624389616,     27.08733537390 ),
  { 4510  449 }  (         0.652, 4.92932795958,   2413.81508903260 ),
  { 4510  450 }  (         0.559, 0.17558868481,     63.73589830340 ),
  { 4510  451 }  (         0.577, 3.82752312276,   1550.93985964600 ),
  { 4510  452 }  (         0.727, 1.05835550856,    490.07345674850 ),
  { 4510  453 }  (         0.574, 3.61492119092,   3686.49611465980 ),
  { 4510  454 }  (         0.732, 5.93179840659,     42.53826965290 ),
  { 4510  455 }  (         0.606, 2.71411884300,   1173.52040467170 ),
  { 4510  456 }  (         0.633, 4.21720828607,    166.82867252200 ),
  { 4510  457 }  (         0.687, 3.91671464962,    529.43033266370 ),
  { 4510  458 }  (         0.570, 2.73551750122,   4010.00153131720 ),
  { 4510  459 }  (         0.552, 2.36967119362,   1603.29998928540 ),
  { 4510  460 }  (         0.600, 1.82659364395,    522.52923398400 ),
  { 4510  461 }  (         0.558, 5.09099246601,   1354.43315884340 ),
  { 4510  462 }  (         0.519, 6.11952999304,    366.79444583570 ),
  { 4510  463 }  (         0.719, 0.85722557905,    362.86229257260 ),
  { 4510  464 }  (         0.518, 2.03954064144,    418.52143602870 ),
  { 4510  465 }  (         0.515, 3.51750445111,    528.41894070740 ),
  { 4510  466 }  (         0.515, 3.47930063838,    103.14095832840 ),
  { 4510  467 }  (         0.550, 5.77676837730,    420.44785172170 ),
  { 4510  468 }  (         0.702, 3.67952126446,   1279.79457262800 ),
  { 4510  469 }  (         0.550, 0.61451088395,    104.05598206510 ),
  { 4510  470 }  (         0.495, 2.41738205536,    179.35884549420 ),
  { 4510  471 }  (         0.513, 0.29823688044,    103.04459010880 ),
  { 4510  472 }  (         0.537, 5.47946238724,    771.30123618390 ),
  { 4510  473 }  (         0.507, 3.08777345288,   1357.61455258110 ),
  { 4510  474 }  (         0.495, 4.95362659160,    536.85269620520 ),
  { 4510  475 }  (         0.681, 4.56294416261,    112.65400177420 ),
  { 4510  476 }  (         0.500, 3.15631977489,   1070.42763045310 ),
  { 4510  477 }  (         0.484, 0.79038835602,     28.45418800320 ),
  { 4510  478 }  (         0.529, 5.46978501034,    419.43645976540 ),
  { 4510  479 }  (         0.597, 4.98058295172,   1251.34038462480 ),
  { 4510  480 }  (         0.492, 3.96066546484,   1269.49963188950 ),
  { 4510  481 }  (         0.482, 3.60167662490,   2943.50605412720 ),
  { 4510  482 }  (         0.630, 6.16496640092,    105.54045477340 ),
  { 4510  483 }  (         0.480, 0.86786400621,     35.21227433100 ),
  { 4510  484 }  (         0.516, 5.97528782923,   3067.93946934820 ),
  { 4510  485 }  (         0.586, 5.48467997697,     56.62235130260 ),
  { 4510  486 }  (         0.502, 1.43671788959,    469.88722005430 ),
  { 4510  487 }  (         0.473, 2.28007170041,   2042.49778910280 ),
  { 4510  488 }  (         0.565, 1.90952569252,    107.28555991260 ),
  { 4510  489 }  (         0.452, 3.13938145287,    934.94851496820 ),
  { 4510  490 }  (         0.605, 1.65413715574,    761.74000862830 ),
  { 4510  491 }  (         0.443, 5.46282223686,    135.33610313300 ),
  { 4510  492 }  (         0.580, 2.06327501551,    493.04240216510 ),
  { 4510  493 }  (         0.540, 1.73777995970,    536.75632798560 ),
  { 4510  494 }  (         0.432, 0.27167052107,     93.53154666300 ),
  { 4510  495 }  (         0.515, 3.46469417437,    530.96298948180 ),
  { 4510  496 }  (         0.440, 5.28884782489,    497.44763618020 ),
  { 4510  497 }  (         0.487, 5.78767525063,  12036.46073488820 ),
  { 4510  498 }  (         0.452, 2.57855172248,   1254.52177836250 ),
  { 4510  499 }  (         0.427, 3.21032629463,   2840.41327990860 ),
  { 4510  500 }  (         0.414, 1.54298025443,    115.62294719080 ),
  { 4510  501 }  (         0.424, 0.12699448931,   1268.74887236410 ),
  { 4510  502 }  (         0.411, 3.12424023238,    536.28324723360 ),
  { 4510  503 }  (         0.452, 1.00194596383,    113.38771495710 ),
  { 4510  504 }  (         0.419, 0.81834479225,   1165.65609814550 ),
  { 4510  505 }  (         0.490, 4.72785081986,    277.03499374140 ),
  { 4510  506 }  (         0.434, 0.36146539146,   1304.92435454160 ),
  { 4510  507 }  (         0.401, 5.70326543719,   1127.04998175570 ),
  { 4510  508 }  (         0.461, 3.26462894820,    102.12956637210 ),
  { 4510  509 }  (         0.533, 2.54951615753,    141.22580985640 ),
  { 4510  510 }  (         0.413, 4.38801694479,   6151.53388830500 ),
  { 4510  511 }  (         0.415, 1.68861617902,    391.17346822390 ),
  { 4510  512 }  (         0.385, 1.69092319074,   4113.09430553580 ),
  { 4510  513 }  (         0.450, 5.49339192735,    602.98809095360 ),
  { 4510  514 }  (         0.499, 3.80738617353,     81.00137369080 ),
  { 4510  515 }  (         0.454, 0.10952919733,    600.54041039880 ),
  { 4510  516 }  (         0.377, 6.25375060718,    913.75088631770 ),
  { 4510  517 }  (         0.453, 3.86104865567,    758.77106321170 ),
  { 4510  518 }  (         0.401, 4.44475618337,    990.22940591440 ),
  { 4510  519 }  (         0.407, 5.13442416563,   3487.42411322340 ),
  { 4510  520 }  (         0.435, 3.76103358490,    523.09868295560 ),
  { 4510  521 }  (         0.425, 3.22287851959,   2655.56837247380 ),
  { 4510  522 }  (         0.365, 5.16456645463,   4694.00295470760 ),
  { 4510  523 }  (         0.454, 1.63325197950,    976.00231191280 ),
  { 4510  524 }  (         0.406, 2.72102389267,   1438.02522544090 ),
  { 4510  525 }  (         0.349, 3.59598366422,   1058.86066532740 ),
  { 4510  526 }  (         0.354, 0.62136331420,    498.67147645760 ),
  { 4510  527 }  (         0.383, 5.09229089574,    539.25219265020 ),
  { 4510  528 }  (         0.380, 3.92653231573,    561.18353448360 ),
  { 4510  529 }  (         0.339, 4.12175871949,   3906.90875709860 ),
  { 4510  530 }  (         0.458, 3.42556794767,    121.25202148330 ),
  { 4510  531 }  (         0.427, 3.61285264910,    860.30992875280 ),
  { 4510  532 }  (         0.424, 4.72757252331,   1366.21257229020 ),
  { 4510  533 }  (         0.328, 4.55286002816,   1696.09782276550 ),
  { 4510  534 }  (         0.324, 4.23685005210,    642.34496686880 ),
  { 4510  535 }  (         0.395, 3.26282558955,    484.44438245600 ),
  { 4510  536 }  (         0.330, 6.05223507989,    215.74677599280 ),
  { 4510  537 }  (         0.318, 2.02072800070,   2964.84669512960 ),
  { 4510  538 }  (         0.417, 0.20173093597,    842.90144101350 ),
  { 4510  539 }  (         0.408, 0.45800247268,   1578.02719501990 ),
  { 4510  540 }  (         0.342, 6.15347077985,   1371.84164658270 ),
  { 4510  541 }  (         0.310, 1.97259286255,    754.03576079650 ),
  { 4510  542 }  (         0.340, 2.77813018312,      3.52311834900 ),
  { 4510  543 }  (         0.333, 2.91352254678,    576.16138801060 ),
  { 4510  544 }  (         0.324, 0.32544817254,    586.31331639720 ),
  { 4510  545 }  (         0.302, 2.08708848849,    526.98265210890 ),
  { 4510  546 }  (         0.363, 4.70567113230,   2730.20695868920 ),
  { 4510  547 }  (         0.300, 0.94464473068,   1432.39615114840 ),
  { 4510  548 }  (         0.352, 5.75013621801,    806.72595883600 ),
  { 4510  549 }  (         0.296, 3.97807312133,   2043.98226181110 ),
  { 4510  550 }  (         0.295, 2.35257797599,   4216.18707975440 ),
  { 4510  551 }  (         0.309, 2.49768755925,   4326.39340097380 ),
  { 4510  552 }  (         0.306, 3.35876843257,   2424.11002977110 ),
  { 4510  553 }  (         0.300, 4.94288858368,   1379.70595310890 ),
  { 4510  554 }  (         0.336, 4.49193455535,   1585.89150154610 ),
  { 4510  555 }  (         0.402, 2.04684001796,    842.15068148810 ),
  { 4510  556 }  (         0.312, 4.59043534747,    188.92007304980 ),
  { 4510  557 }  (         0.346, 5.19792097706,    523.47118997110 ),
  { 4510  558 }  (         0.380, 1.67961600066,     36.64856292950 ),
  { 4510  559 }  (         0.338, 1.32014513725,    148.07872442630 ),
  { 4510  560 }  (         0.391, 4.82224015188,   1012.91150727320 ),
  { 4510  561 }  (         0.285, 3.43655052437,   1053.96530421780 ),
  { 4510  562 }  (         0.332, 2.02575636311,   1091.62525910360 ),
  { 4510  563 }  (         0.282, 5.78865321890,   1064.04779663520 ),
  { 4510  564 }  (         0.282, 0.39153852422,    207.67002114550 ),
  { 4510  565 }  (         0.280, 3.80196391678,    298.23262239190 ),
  { 4510  566 }  (         0.387, 6.26819309990,   1141.13406340540 ),
  { 4510  567 }  (         0.349, 4.09121908199,   1059.90319505100 ),
  { 4510  568 }  (         0.320, 0.39871942000,   2122.69601364150 ),
  { 4510  569 }  (         0.327, 4.76503823073,    134.58534360760 ),
  { 4510  570 }  (         0.283, 3.90409016441,    127.47179660680 ),
  { 4510  571 }  (         0.301, 4.30291951219,    299.12639426920 ),
  { 4510  572 }  (         0.322, 2.48251052680,   1065.60170531270 ),
  { 4510  573 }  (         0.297, 2.40814103509,   1591.52057583860 ),
  { 4510  574 }  (         0.286, 5.85849626574,    172.24529849340 ),
  { 4510  575 }  (         0.285, 4.55845472479,   1446.62324515000 ),
  { 4510  576 }  (         0.270, 4.08342186112,   1578.77795454530 ),
  { 4510  577 }  (         0.362, 1.06148806683,    181.80652604900 ),
  { 4510  578 }  (         0.335, 4.51094500655,   2349.32843120380 ),
  { 4510  579 }  (         0.347, 0.62281394535,   1542.60247236780 ),
  { 4510  580 }  (         0.275, 3.38473403113,   4002.88798431640 ),
  { 4510  581 }  (         0.255, 1.52357936497,   1688.23351623930 ),
  { 4510  582 }  (         0.276, 4.32192160071,   1912.57831194120 ),
  { 4510  583 }  (         0.253, 2.40482338279,     97.67614824720 ),
  { 4510  584 }  (         0.248, 4.45058246237,   1688.98427576470 ),
  { 4510  585 }  (         0.300, 3.07435583442,   1902.28337120270 ),
  { 4510  586 }  (         0.257, 4.79180478086,   1670.82502850000 ),
  { 4510  587 }  (         0.319, 1.34244222683,   1288.46202830630 ),
  { 4510  588 }  (         0.245, 4.01852686769,   1567.73225428140 ),
  { 4510  589 }  (         0.278, 0.25406312148,    874.39401040250 ),
  { 4510  590 }  (         0.324, 5.57824969423,   1670.07426897460 ),
  { 4510  591 }  (         0.300, 4.67161812947,   1329.30337692980 ),
  { 4510  592 }  (         0.241, 0.01789818312,   1586.62521472900 ),
  { 4510  593 }  (         0.295, 5.86996114913,   2804.23779773110 ),
  { 4510  594 }  (         0.317, 3.17967272487,   1020.02505427400 ),
  { 4510  595 }  (         0.238, 4.97765946754,    351.81659230870 ),
  { 4510  596 }  (         0.302, 1.20236375616,    232.04904353370 ),
  { 4510  597 }  (         0.301, 5.53432687957,   2274.54683263650 ),
  { 4510  598 }  (         0.286, 2.41008592059,   2545.36205125440 ),
  { 4510  599 }  (         0.294, 2.01783542485,    313.94418910180 ),
  { 4510  600 }  (         0.292, 2.12690999284,   1592.25428902150 ),
  { 4510  601 }  (         0.250, 2.31712163679,    632.26247445140 ),
  { 4510  602 }  (         0.238, 5.06557054569,   3803.81598288000 ),
  { 4510  603 }  (         0.226, 0.05916712753,   1518.22344997960 ),
  { 4510  604 }  (         0.235, 0.16574304942,    137.03302416240 ),
  { 4510  605 }  (         0.298, 2.99720233431,   1467.82087380050 ),
  { 4510  606 }  (         0.286, 5.08357076653,    774.00954916960 ),
  { 4510  607 }  (         0.246, 2.81685822336,    633.30500417500 ),
  { 4510  608 }  (         0.269, 4.93023426152,    151.04766984290 ),
  { 4510  609 }  (         0.228, 6.13118739321,   3281.23856478620 ),
  { 4510  610 }  (         0.228, 1.22066024988,    700.45179087970 ),
  { 4510  611 }  (         0.239, 0.71695698501,   1276.61317889030 ),
  { 4510  612 }  (         0.289, 6.08263862565,   3384.33133900480 ),
  { 4510  613 }  (         0.218, 2.90308501961,     85.82729883120 ),
  { 4510  614 }  (         0.283, 6.28058228271,     71.81265315070 ),
  { 4510  615 }  (         0.271, 6.01605074549,    170.76082578510 ),
  { 4510  616 }  (         0.221, 0.99914179141,   1053.75285589670 ),
  { 4510  617 }  (         0.218, 1.50681393471,   1087.69310584050 ),
  { 4510  618 }  (         0.223, 3.39126063354,   3259.89792378380 ),
  { 4510  619 }  (         0.229, 1.19373202707,   1060.86640289750 ),
  { 4510  620 }  (         0.264, 3.93467945263,   1363.24362687360 ),
  { 4510  621 }  (         0.228, 5.04188376116,   1064.79855616060 ),
  { 4510  622 }  (         0.295, 2.15253086390,   6386.16862421000 ),
  { 4510  623 }  (         0.214, 3.85961180377,   4223.30062675520 ),
  { 4510  624 }  (         0.218, 0.79681703388,   1909.39691820350 ),
  { 4510  625 }  (         0.212, 4.11706418218,    269.92144674060 ),
  { 4510  626 }  (         0.264, 5.81676406517,     77.96299230500 ),
  { 4510  627 }  (         0.256, 5.65978708108,    799.61241183520 ),
  { 4510  628 }  (         0.242, 6.25078283449,   1621.31622419820 ),
  { 4510  629 }  (         0.235, 2.20668997852,   1570.91364801910 ),
  { 4510  630 }  (         0.212, 2.88214546012,   1674.00642223770 ),
  { 4510  631 }  (         0.206, 1.59586787037,   4429.48617519240 ),
  { 4510  632 }  (         0.208, 2.31366614282,    878.32616366560 ),
  { 4510  633 }  (         0.213, 0.30373338388,   8624.21265092720 ),
  { 4510  634 }  (         0.223, 4.88419887133,   1035.00290780100 ),
  { 4510  635 }  (         0.279, 3.65173543621,     84.93352695390 ),
  { 4510  636 }  (         0.210, 4.08825553401,    203.73786788240 ),
  { 4510  637 }  (         0.214, 4.63498396475,    812.14258480740 ),
  { 4510  638 }  (         0.258, 1.73501688450,   1887.30551767570 ),
  { 4510  639 }  (         0.210, 4.51798082710,   1262.38608488870 ),
  { 4510  640 }  (         0.252, 5.69246905091,    104.57724692690 ),
  { 4510  641 }  (         0.205, 4.62946016431,   1056.46116888240 ),
  { 4510  642 }  (         0.263, 3.04951219565,   1493.09366806600 ),
  { 4510  643 }  (         0.222, 5.54424082649,   5216.58037280140 ),
  { 4510  644 }  (         0.244, 0.91026645686,   3707.83675566220 ),
  { 4510  645 }  (         0.204, 0.90117975859,   1408.01712876020 ),
  { 4510  646 }  (         0.225, 1.23997048012,   3340.61242669980 ),
  { 4510  647 }  (         0.258, 2.35906183505,   2861.75392091100 ),
  { 4510  648 }  (         0.267, 3.27705002283,   5120.60114558360 ),
  { 4510  649 }  (         0.214, 0.66988779149,   9146.79006902100 ),
  { 4510  650 }  (         0.235, 4.93761209111,   1443.44185141230 ),
  { 4510  651 }  (         0.194, 1.60798828275,    102.57150935680 ),
  { 4510  652 }  (         0.215, 0.97603524747,    479.28838891550 ),
  { 4510  653 }  (         0.205, 5.23642605904,   4649.89881763120 ),
  { 4510  654 }  (         0.257, 4.70227260707,   9050.81084180320 ),
  { 4510  655 }  (         0.228, 6.23410921116,     64.95973858080 ),
  { 4510  656 }  (         0.180, 4.21309134581,    143.93412284210 ),
  { 4510  657 }  (         0.180, 4.82870451226,   1063.57471588320 ),
  { 4510  658 }  (         0.180, 5.06126965624,     52.69019803950 ),
  { 4510  659 }  (         0.226, 0.55334952097,    554.06998748280 ),
  { 4510  660 }  (         0.209, 5.67975843693,     48.75804477640 ),
  { 4510  661 }  (         0.186, 3.66368928017,    108.72184851110 ),
  { 4510  662 }  (         0.190, 2.00852986549,   1058.63117066380 ),
  { 4510  663 }  (         0.183, 3.17358464220,    140.96517742550 ),
  { 4510  664 }  (         0.198, 5.49816579454,   4333.50694797460 ),
  { 4510  665 }  (         0.240, 6.06602357868,   1821.12193881750 ),
  { 4510  666 }  (         0.172, 3.04802064781,     54.33472944220 ),
  { 4510  667 }  (         0.170, 4.66520291204,   1372.59240610810 ),
  { 4510  668 }  (         0.173, 4.72884056307,  77204.32749453338 ),
  { 4510  669 }  (         0.174, 0.85370421252,   1587.58842257550 ),
  { 4510  670 }  (         0.215, 0.68219980704,   1054.71606374320 ),
  { 4510  671 }  (         0.170, 1.52204803308,   5591.96087960020 ),
  { 4510  672 }  (         0.200, 1.60275092073,   6681.22485339960 ),
  { 4510  673 }  (         0.193, 2.13003479280,    103.61403908040 ),
  { 4510  674 }  (         0.231, 4.69962389031,   1966.23171782720 ),
  { 4510  675 }  (         0.179, 5.57395905447,   1457.52593306200 ),
  { 4510  676 }  (         0.205, 3.65507571128,    906.84978763800 ),
  { 4510  677 }  (         0.181, 4.52272934666,  24498.83024629040 ),
  { 4510  678 }  (         0.223, 0.11650319998,     67.88049988760 ),
  { 4510  679 }  (         0.172, 5.68083885227,   1884.12412393800 ),
  { 4510  680 }  (         0.219, 0.60964963735,   2729.45619916380 ),
  { 4510  681 }  (         0.164, 1.06675279755,    594.65070367540 ),
  { 4510  682 }  (         0.176, 2.36848603898,    977.48678462110 ),
  { 4510  683 }  (         0.170, 2.43036684800,   4532.57894941100 ),
  { 4510  684 }  (         0.191, 3.64255924842,   1440.99417085750 ),
  { 4510  685 }  (         0.207, 0.49276008455,     71.60020482960 ),
  { 4510  686 }  (         0.157, 4.26888100582,   5069.38346150640 ),
  { 4510  687 }  (         0.157, 5.14847227422,    451.72797278960 ),
  { 4510  688 }  (         0.158, 5.00063628575,    650.94298657790 ),
  { 4510  689 }  (         0.159, 5.37530499642,  20426.57109242200 ),
  { 4510  690 }  (         0.218, 0.27875408082,    175.16605980020 ),
  { 4510  691 }  (         0.155, 0.83696849428,   1474.93442080130 ),
  { 4510  692 }  (         0.154, 2.62839957291,    683.18631549230 ),
  { 4510  693 }  (         0.171, 1.79511736017,   1123.11782849260 ),
  { 4510  694 }  (         0.188, 5.24747110812,  25565.32572348040 ),
  { 4510  695 }  (         0.168, 4.14907553818,    946.72792841500 ),
  { 4510  696 }  (         0.203, 2.83699715530,   1489.91227432830 ),
  { 4510  697 }  (         0.173, 4.34546063838,   3046.59882834580 ),
  { 4510  698 }  (         0.190, 5.67865607835,   1060.13268971460 ),
  { 4510  699 }  (         0.201, 2.38524182920,    419.53282798500 ),
  { 4510  700 }  (         0.152, 5.89088685790,    208.63322899200 ),
  { 4510  701 }  (         0.206, 4.46933127349,   2654.67460059650 ),
  { 4510  702 }  (         0.156, 2.37819796438,   2758.66114669240 ),
  { 4510  703 }  (         0.203, 0.70565514297,    498.19839570560 ),
  { 4510  704 }  (         0.205, 3.05468636546,   1062.30269149600 ),
  { 4510  705 }  (         0.174, 3.50824761708,   2004.36475346500 ),
  { 4510  706 }  (         0.148, 4.73961194393,   1799.19059698410 ),
  { 4510  707 }  (         0.188, 3.62315953725,   3156.80514956520 ),
  { 4510  708 }  (         0.183, 2.35011338194,  25551.09862947879 ),
  { 4510  709 }  (         0.162, 1.58053710589,    628.59095361920 ),
  { 4510  710 }  (         0.162, 3.99983876824,   1482.79872732750 ),
  { 4510  711 }  (         0.181, 2.85489861839,   1055.18914449520 ),
  { 4510  712 }  (         0.151, 3.43198157222,    629.86297800640 ),
  { 4510  713 }  (         0.157, 3.15195826490,   1025.44168024540 ),
  { 4510  714 }  (         0.194, 5.13049187783,   1818.15299340090 ),
  { 4510  715 }  (         0.193, 1.92287052164,   1140.38330388000 ),
  { 4510  716 }  (         0.137, 4.22335221970,   1049.82070263360 ),
  { 4510  717 }  (         0.167, 2.85163087563,   5746.27133789600 ),
  { 4510  718 }  (         0.167, 5.73970282991,   5760.49843189760 ),
  { 4510  719 }  (         0.138, 2.23519776527,   1176.70179840940 ),
  { 4510  720 }  (         0.151, 4.89507270899,    532.39927808030 ),
  { 4510  721 }  (         0.147, 2.65931838448,    987.30864460760 ),
  { 4510  722 }  (         0.135, 0.12836417770,    991.71387862270 ),
  { 4510  723 }  (         0.166, 3.12682515439,    580.09354127370 ),
  { 4510  724 }  (         0.118, 5.98810576300,    531.38788612400 ),
  { 4510  725 }  (         0.135, 5.26601313643,   1065.01100448170 ),
  { 4510  726 }  (         0.138, 3.18511244397,    707.56533788050 ),
  { 4510  727 }  (         0.122, 1.34377059565,    446.31134681820 ),
  { 4510  728 }  (         0.120, 2.29717714347,   1059.22187149480 ),
  { 4510  729 }  (         0.121, 0.58145552537,   5621.84292321040 ),
  { 4510  730 }  (         0.103, 4.75645235023,   1226.21060271120 ),
  { 4510  731 }  (         0.104, 6.08481630139,    528.25467649610 ),
  { 4510  732 }  (         0.119, 1.06475523307,    527.99404406520 ),
  { 4510  733 }  (         0.104, 0.89730746841,    531.12725369310 ),
  { 4510  734 }  (         0.120, 5.39001411803,   1059.54198888360 ),
  { 4510  735 }  (         0.104, 0.44849170648,   1128.53445446400 ),
  { 4510  736 }  (         0.117, 5.42449214711,    986.08480433020 ),
  { 4510  737 }  (         0.101, 5.09893554462,    530.58473697190 ),
  { 4510  738 }  (         0.102, 0.26948040239,    450.97721326420 ),
  { 4510  739 }  (         0.107, 1.58724086516,   1069.67687092770 ),
  { 4510  740 }  (         0.086, 2.28711702506,   2498.89162833840 ),
  { 4510  741 }  (         0.101, 1.88318822518,    528.79719321730 ),
  { 4510  742 }  (         0.086, 1.37568728263,    970.51624997220 ),
  { 4510  743 }  (         0.083, 0.06930748288,    530.91480537200 ),
  { 4510  744 }  (         0.085, 3.22094000094,   1553.64817263170 ),
  { 4510  745 }  (         0.083, 0.62963097974,    528.46712481720 ),
  { 4510  746 }  (         0.083, 4.16314675511,    849.26422848890 ),
  { 4510  747 }  (         0.079, 3.46688102340,   1077.54117745390 ),
  { 4510  748 }  (         0.097, 0.87886975916,   9690.70812811720 ),
  { 4510  749 }  (         0.097, 4.27398311206,   9676.48103411560 ),
  { 4510  750 }  (         0.101, 0.29639798579,    857.12853501510 ),
  { 4510  751 }  (         0.083, 2.55427333923,   1059.59437851030 ),
  { 4510  752 }  (         0.078, 0.06461496210,    521.82665856840 ),
  { 4510  753 }  (         0.078, 0.76677000862,    525.54636351040 ),
  { 4510  754 }  (         0.096, 0.33631035749,   1090.40141882620 ),
  { 4510  755 }  (         0.098, 1.42815294497,    757.21715453420 ),
  { 4510  756 }  (         0.077, 0.85066773729,    537.55527162080 ),
  { 4510  757 }  (         0.084, 5.04765104413,   1160.02702385300 ),
  { 4510  758 }  (         0.076, 3.62264327413,    782.34693644780 ),
  { 4510  759 }  (         0.085, 1.86831145784,  25028.52121138500 ),
  { 4510  760 }  (         0.079, 2.90602202890,   2114.83170711530 )
  (*$endif *)
      );
  (*@\\\*)
  (*@/// vsop87_jup_l1:array[0..368,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_l1:array[0.. 60,0..2] of extended = (
  (*$else *)
  vsop87_jup_l1:array[0..368,0..2] of extended = (
  (*$endif *)
  { 4511    1 } (52993480757.497, 0.00000000000,      0.00000000000 ),
  { 4511    2 }  (    489741.194, 4.22066689928,    529.69096509460 ),
  { 4511    3 }  (    228918.538, 6.02647464016,      7.11354700080 ),
  { 4511    4 }  (     27655.380, 4.57265956824,   1059.38193018920 ),
  { 4511    5 }  (     20720.943, 5.45938936295,    522.57741809380 ),
  { 4511    6 }  (     12105.732, 0.16985765041,    536.80451209540 ),
  { 4511    7 }  (      6068.051, 4.42419502005,    103.09277421860 ),
  { 4511    8 }  (      5433.924, 3.98478382565,    419.48464387520 ),
  { 4511    9 }  (      4237.795, 5.89009351271,     14.22709400160 ),
  { 4511   10 }  (      2211.854, 5.26771446618,    206.18554843720 ),
  { 4511   11 }  (      1295.769, 5.55132765087,      3.18139373770 ),
  { 4511   12 }  (      1745.919, 4.92669378486,   1589.07289528380 ),
  { 4511   13 }  (      1163.411, 0.51450895328,      3.93215326310 ),
  { 4511   14 }  (      1007.216, 0.46478398551,    735.87651353180 ),
  { 4511   15 }  (      1173.129, 5.85647304350,   1052.26838318840 ),
  { 4511   16 }  (       847.678, 5.75805850450,    110.20632121940 ),
  { 4511   17 }  (       827.329, 4.80312015734,    213.29909543800 ),
  { 4511   18 }  (      1003.574, 3.15040301822,    426.59819087600 ),
  { 4511   19 }  (      1098.735, 5.30704981594,    515.46387109300 ),
  { 4511   20 }  (       816.397, 0.58643054886,   1066.49547719000 ),
  { 4511   21 }  (       725.447, 5.51827471473,    639.89728631400 ),
  { 4511   22 }  (       567.845, 5.98867049451,    625.67019231240 ),
  { 4511   23 }  (       474.181, 4.13245269168,    412.37109687440 ),
  { 4511   24 }  (       412.930, 5.73652891261,     95.97922721780 ),
  { 4511   25 }  (       335.817, 3.73248749046,   1162.47470440780 ),
  { 4511   26 }  (       345.249, 4.24159565410,    632.78373931320 ),
  { 4511   27 }  (       234.066, 6.24302226646,    309.27832265580 ),
  { 4511   28 }  (       194.784, 2.21879010911,    323.50541665740 ),
  { 4511   29 }  (       234.340, 4.03469970332,    949.17560896980 ),
  { 4511   30 }  (       183.938, 6.27963588822,    543.91805909620 ),
  { 4511   31 }  (       198.525, 1.50458442825,    838.96928775040 ),
  { 4511   32 }  (       186.899, 6.08620565908,    742.99006053260 ),
  { 4511   33 }  (       171.380, 5.41655983845,    199.07200143640 ),
  { 4511   34 }  (       130.771, 0.62643377351,    728.76296653100 ),
  { 4511   35 }  (       107.575, 4.49282760117,    956.28915597060 ),
  { 4511   36 }  (       115.393, 0.68019050174,    846.08283475120 ),
  { 4511   37 }  (       115.047, 5.28641699144,   2118.76386037840 ),
  { 4511   38 }  (        66.824, 5.73365126533,     21.34064100240 ),
  { 4511   39 }  (        69.618, 5.97263450278,    532.87235883230 ),
  { 4511   40 }  (        64.850, 6.08803490288,   1581.95934828300 ),
  { 4511   41 }  (        79.686, 5.82412400273,   1045.15483618760 ),
  { 4511   42 }  (        57.939, 0.99453087342,   1596.18644228460 ),
  { 4511   43 }  (        65.635, 0.12924191430,    526.50957135690 ),
  { 4511   44 }  (        58.509, 0.58626971028,   1155.36115740700 ),
  { 4511   45 }  (        56.600, 1.41198438841,    533.62311835770 ),
  { 4511   46 }  (        71.643, 5.34162650321,    942.06206196900 ),
  { 4511   47 }  (        57.368, 5.96851304799,   1169.58825140860 ),
  { 4511   48 }  (        54.935, 5.42806383723,     10.29494073850 ),
  { 4511   49 }  (        52.016, 0.22981299129,   1368.66025284500 ),
  { 4511   50 }  (        52.309, 5.72661448388,    117.31986822020 ),
  { 4511   51 }  (        50.418, 6.08075147811,    525.75881183150 ),
  { 4511   52 }  (        47.418, 3.62611843241,   1478.86657406440 ),
  { 4511   53 }  (        39.888, 4.16158013600,   1692.16566950240 ),
  { 4511   54 }  (        46.678, 0.51144073175,   1265.56747862640 ),
  { 4511   55 }  (        32.827, 5.03596689455,    220.41264243880 ),
  { 4511   56 }  (        33.558, 0.09913904872,    302.16477565500 ),
  { 4511   57 }  (        29.379, 3.35927241533,      4.66586644600 ),
  { 4511   58 }  (        29.307, 0.75907909735,     88.86568021700 ),
  { 4511   59 }  (        32.449, 5.37492530697,    508.35032409220 ),
  { 4511   60 }  (        29.483, 5.42208897099,   1272.68102562720 ),
  (*$ifndef meeus *)
  { 4511   61 }  (        21.802, 6.15054054070,   1685.05212250160 ),
  (*$endif *)
  { 4511   62 }  (        25.195, 1.60723063387,    831.85574074960 )
  (*$ifndef meeus *)                                                 ,
  { 4511   63 }  (        21.133, 5.86346824200,   1258.45393162560 ),
  { 4511   64 }  (        19.747, 2.17205957814,    316.39186965660 ),
  { 4511   65 }  (        17.871, 0.82841413516,    433.71173787680 ),
  { 4511   66 }  (        17.703, 5.95527049039,      5.41662597140 ),
  { 4511   67 }  (        17.230, 2.76395560958,    853.19638175200 ),
  { 4511   68 }  (        17.453, 0.70749901224,   1471.75302706360 ),
  { 4511   69 }  (        17.508, 0.49799925173,   1375.77379984580 ),
  { 4511   70 }  (        14.368, 0.91459831140,     18.15924726470 ),
  { 4511   71 }  (        14.107, 0.63031082833,      2.92076130680 ),
  { 4511   72 }  (        11.559, 4.30379009964,    405.25754987360 ),
  { 4511   73 }  (        11.728, 1.76426582357,    380.12776796000 ),
  { 4511   74 }  (        11.054, 5.56735602213,   1574.84580128220 ),
  { 4511   75 }  (        10.425, 0.31355034390,   1361.54670584420 ),
  { 4511   76 }  (         9.804, 5.90363777277,    519.39602435610 ),
  { 4511   77 }  (         9.805, 0.38648727979,   1073.60902419080 ),
  { 4511   78 }  (         9.285, 3.21842287530,   1795.25844372100 ),
  { 4511   79 }  (         8.864, 0.53776257958,   1788.14489672020 ),
  { 4511   80 }  (         8.370, 5.88484552222,   2001.44399215820 ),
  { 4511   81 }  (         8.148, 5.10162311410,   1485.98012106520 ),
  { 4511   82 }  (         7.658, 5.64890060131,   2648.45482547300 ),
  { 4511   83 }  (         6.690, 2.41093459420,      4.19278569400 ),
  { 4511   84 }  (         5.840, 4.22347896053,   2008.55753915900 ),
  { 4511   85 }  (         7.256, 6.19384525651,     11.04570026390 ),
  { 4511   86 }  (         6.266, 1.36137786945,   1148.24761040620 ),
  { 4511   87 }  (         5.141, 5.23083932012,    628.85158605010 ),
  { 4511   88 }  (         5.140, 2.92955981951,    518.64526483070 ),
  { 4511   89 }  (         4.765, 0.16838181862,    629.60234557550 ),
  { 4511   90 }  (         4.603, 0.78529559911,    721.64941953020 ),
  { 4511   91 }  (         4.575, 6.24794935732,   1677.93857550080 ),
  { 4511   92 }  (         4.537, 4.95096707833,    635.96513305090 ),
  { 4511   93 }  (         4.518, 2.06523915453,    453.42489381900 ),
  { 4511   94 }  (         4.414, 0.15381186059,   1699.27921650320 ),
  { 4511   95 }  (         5.593, 5.57489981207,    191.95845443560 ),
  { 4511   96 }  (         5.403, 1.46004886198,    330.61896365820 ),
  { 4511   97 }  (         4.285, 0.23949868127,   2104.53676637680 ),
  { 4511   98 }  (         4.223, 1.44087555881,   2125.87740737920 ),
  { 4511   99 }  (         4.101, 6.19274358942,    636.71589257630 ),
  { 4511  100 }  (         4.432, 4.35811524051,    423.41679713830 ),
  { 4511  101 }  (         4.132, 0.50170694173,   1056.20053645150 ),
  { 4511  102 }  (         4.398, 4.14280286969,    511.53171782990 ),
  { 4511  103 }  (         5.406, 4.40429493698,   2221.85663459700 ),
  { 4511  104 }  (         4.467, 0.08534650684,   1062.56332392690 ),
  { 4511  105 }  (         3.569, 5.66540477010,   2317.83586181480 ),
  { 4511  106 }  (         4.007, 2.54845549248,     74.78159856730 ),
  { 4511  107 }  (         3.515, 0.25495124831,   1055.44977692610 ),
  { 4511  108 }  (         3.687, 2.93378008847,     32.24332891440 ),
  { 4511  109 }  (         2.883, 5.72793010505,     99.91138048090 ),
  { 4511  110 }  (         2.969, 5.50054720569,    107.02492748170 ),
  { 4511  111 }  (         2.720, 1.25222590925,    540.73666535850 ),
  { 4511  112 }  (         2.808, 3.30714813896,      0.75075952540 ),
  { 4511  113 }  (         2.768, 1.61339487804,   1063.31408345230 ),
  { 4511  114 }  (         2.666, 4.28662288102,    106.27416795630 ),
  { 4511  115 }  (         2.704, 3.03615556153,    422.66603761290 ),
  { 4511  116 }  (         3.290, 5.89081682150,   1802.37199072180 ),
  { 4511  117 }  (         2.578, 3.60390367979,    750.10360753340 ),
  { 4511  118 }  (         2.661, 0.35249312659,   1898.35121793960 ),
  { 4511  119 }  (         2.486, 5.28950877719,   1891.23767093880 ),
  { 4511  120 }  (         2.936, 1.09052029450,   1464.63948006280 ),
  { 4511  121 }  (         3.190, 4.60740643547,    416.30325013750 ),
  { 4511  122 }  (         2.390, 6.01779736611,    551.03160609700 ),
  { 4511  123 }  (         2.214, 5.24450923180,    621.73803904930 ),
  { 4511  124 }  (         2.319, 5.82920300130,    305.34616939270 ),
  { 4511  125 }  (         2.089, 5.99310370434,   1994.33044515740 ),
  { 4511  126 }  (         2.042, 0.75008788531,    142.44965013380 ),
  { 4511  127 }  (         2.121, 0.01537599023,   2420.92863603340 ),
  { 4511  128 }  (         2.114, 6.25308371567,    647.01083331480 ),
  { 4511  129 }  (         2.020, 4.17560390841,    569.04784100980 ),
  { 4511  130 }  (         2.109, 5.18682321403,    227.52618943960 ),
  { 4511  131 }  (         2.283, 5.80043809222,    539.98590583310 ),
  { 4511  132 }  (         1.977, 3.99197009651,     24.37902238820 ),
  { 4511  133 }  (         1.960, 1.35288793079,    963.40270297140 ),
  { 4511  134 }  (         1.903, 2.78349628184,   2428.04218303420 ),
  { 4511  135 }  (         1.915, 4.22134509685,   2324.94940881560 ),
  { 4511  136 }  (         1.971, 5.88715684267,    217.23124870110 ),
  { 4511  137 }  (         1.917, 3.03728154374,   1382.88734684660 ),
  { 4511  138 }  (         2.026, 3.08606488714,    408.43894361130 ),
  { 4511  139 }  (         1.834, 5.61474110217,    430.53034413910 ),
  { 4511  140 }  (         1.838, 1.25467410218,     81.75213321620 ),
  { 4511  141 }  (         2.460, 4.63268678998,   1905.46476494040 ),
  { 4511  142 }  (         1.820, 5.97497926120,    114.13847448250 ),
  { 4511  143 }  (         2.043, 4.34047514845,     70.84944530420 ),
  { 4511  144 }  (         1.959, 4.03116026306,     92.04707395470 ),
  { 4511  145 }  (         1.768, 0.33097462499,     35.42472265210 ),
  { 4511  146 }  (         2.334, 5.87042638470,   1038.04128918680 ),
  { 4511  147 }  (         1.835, 4.81326127892,    124.43341522100 ),
  { 4511  148 }  (         2.269, 1.02549350754,    618.55664531160 ),
  { 4511  149 }  (         1.919, 5.01297395549,     99.16062095550 ),
  { 4511  150 }  (         1.923, 0.28688549585,     31.01948863700 ),
  { 4511  151 }  (         1.878, 5.69299116574,    210.11770170030 ),
  { 4511  152 }  (         1.679, 0.25635730278,    295.05122865420 ),
  { 4511  153 }  (         1.656, 5.46039280732,   2634.22773147140 ),
  { 4511  154 }  (         1.675, 6.15609073315,    643.82943957710 ),
  { 4511  155 }  (         1.953, 5.09846435548,     17.40848773930 ),
  { 4511  156 }  (         1.539, 2.75316078346,    415.55249061210 ),
  { 4511  157 }  (         1.467, 0.54812675158,    458.84151979040 ),
  { 4511  158 }  (         1.482, 3.76736278426,    534.35683154060 ),
  { 4511  159 }  (         1.446, 3.15802770791,     25.12978191360 ),
  { 4511  160 }  (         1.667, 0.26406950755,    835.03713448730 ),
  { 4511  161 }  (         1.472, 0.83054329617,     28.31117565130 ),
  { 4511  162 }  (         1.655, 0.88908548504,   1781.03134971940 ),
  { 4511  163 }  (         1.294, 5.76241191046,    440.82528487760 ),
  { 4511  164 }  (         1.348, 2.49823510924,    984.60033162190 ),
  { 4511  165 }  (         1.352, 5.10869562455,    149.56319713460 ),
  { 4511  166 }  (         1.344, 0.01942249067,   2214.74308759620 ),
  { 4511  167 }  (         1.188, 2.24279457878,     31.49256938900 ),
  { 4511  168 }  (         1.166, 0.80686346228,    739.80866679490 ),
  { 4511  169 }  (         1.322, 4.25691184168,   2538.24850425360 ),
  { 4511  170 }  (         1.094, 6.02985819406,   2737.32050569000 ),
  { 4511  171 }  (         1.112, 4.38204360670,    561.93429400900 ),
  { 4511  172 }  (         1.346, 3.20575848870,    525.02509864860 ),
  { 4511  173 }  (         1.056, 5.76507115032,   2310.72231481400 ),
  { 4511  174 }  (         1.159, 0.46189564970,     67.66805156650 ),
  { 4511  175 }  (         1.027, 0.20709586018,      7.86430652620 ),
  { 4511  176 }  (         1.143, 5.56626418636,     46.47042291600 ),
  { 4511  177 }  (         1.012, 0.54293005597,    532.13864564940 ),
  { 4511  178 }  (         0.978, 5.13939194101,   2207.62954059540 ),
  { 4511  179 }  (         0.993, 2.03698185233,    319.57326339430 ),
  { 4511  180 }  (         1.035, 2.90231353535,    611.44309831080 ),
  { 4511  181 }  (         1.021, 4.75651217048,    527.24328453980 ),
  { 4511  182 }  (         1.308, 1.78809336431,    824.74219374880 ),
  { 4511  183 }  (         0.964, 2.82269601958,   2111.65031337760 ),
  { 4511  184 }  (         0.896, 2.54505998806,   2744.43405269080 ),
  { 4511  185 }  (         0.890, 5.41036782817,     28.45418800320 ),
  { 4511  186 }  (         0.906, 0.76565238554,   1439.50969814920 ),
  { 4511  187 }  (         0.985, 0.88687623770,   5760.49843189760 ),
  { 4511  188 }  (         0.983, 1.42102343372,   5746.27133789600 ),
  { 4511  189 }  (         0.892, 5.87250060663,    203.00415469950 ),
  { 4511  190 }  (         0.942, 2.31049430734,   9690.70812811720 ),
  { 4511  191 }  (         0.941, 2.84331157527,   9676.48103411560 ),
  { 4511  192 }  (         0.867, 0.81020362547,    524.27433912320 ),
  { 4511  193 }  (         0.829, 2.35178495412,    312.45971639350 ),
  { 4511  194 }  (         0.912, 2.80494184378,      6.21977512350 ),
  { 4511  195 }  (         0.809, 1.05148218513,    529.64278098480 ),
  { 4511  196 }  (         0.779, 4.80009242059,    945.24345570670 ),
  { 4511  197 }  (         0.878, 5.76532521399,      1.64453140270 ),
  { 4511  198 }  (         0.953, 4.30945738629,    209.36694217490 ),
  { 4511  199 }  (         0.772, 5.25607113566,   2950.61960112800 ),
  { 4511  200 }  (         0.745, 0.03810558502,    535.10759106600 ),
  { 4511  201 }  (         0.744, 0.58381523987,     25.27279426550 ),
  { 4511  202 }  (         0.734, 0.20800485100,   1049.08698945070 ),
  { 4511  203 }  (         0.747, 2.71772840871,     38.13303563780 ),
  { 4511  204 }  (         0.728, 5.97210358938,    945.99421523210 ),
  { 4511  205 }  (         0.769, 4.51394016967,    952.35700270750 ),
  { 4511  206 }  (         0.710, 0.38016353553,     69.15252427480 ),
  { 4511  207 }  (         0.760, 3.07033779824,     39.35687591520 ),
  { 4511  208 }  (         0.802, 1.14191463412,    532.61172640140 ),
  { 4511  209 }  (         0.704, 1.25447308120,    547.85021235930 ),
  { 4511  210 }  (         0.721, 0.73855379162,   2228.97018159780 ),
  { 4511  211 }  (         0.794, 4.25051539085,   2641.34127847220 ),
  { 4511  212 }  (         0.795, 3.20588363820,    604.47256366190 ),
  { 4511  213 }  (         0.818, 1.05229815343,    909.81873305460 ),
  { 4511  214 }  (         0.724, 5.68281830264,    953.10776223290 ),
  { 4511  215 }  (         0.836, 0.60410469174,   2097.42321937600 ),
  { 4511  216 }  (         0.669, 5.75757140051,   2015.67108615980 ),
  { 4511  217 }  (         0.682, 1.19994890339,    387.24131496080 ),
  { 4511  218 }  (         0.640, 3.91546675664,    528.72775724810 ),
  { 4511  219 }  (         0.809, 4.24929331276,    529.73914920440 ),
  { 4511  220 }  (         0.819, 4.91540072376,   2751.54759969160 ),
  { 4511  221 }  (         0.692, 2.51162384766,    916.93228005540 ),
  { 4511  222 }  (         0.784, 4.23651511312,    195.13984817330 ),
  { 4511  223 }  (         0.762, 1.12201139619,    732.69511979410 ),
  { 4511  224 }  (         0.617, 5.80920925081,    739.05790726950 ),
  { 4511  225 }  (         0.727, 4.24401822698,    760.25553592000 ),
  { 4511  226 }  (         0.591, 3.26075006572,    202.25339517410 ),
  { 4511  227 }  (         0.552, 5.83533550039,    526.77020378780 ),
  { 4511  228 }  (         0.640, 1.38530872949,    530.65417294110 ),
  { 4511  229 }  (         0.577, 6.09100925678,   2531.13495725280 ),
  { 4511  230 }  (         0.620, 3.01917904435,    902.70518605380 ),
  { 4511  231 }  (         0.722, 5.18171159557,      1.48447270830 ),
  { 4511  232 }  (         0.540, 3.78809230820,   2957.73314812880 ),
  { 4511  233 }  (         0.523, 3.63882376000,    437.64389113990 ),
  { 4511  234 }  (         0.527, 5.80796427555,   3053.71237534660 ),
  { 4511  235 }  (         0.488, 4.99103190309,    483.22054217860 ),
  { 4511  236 }  (         0.557, 4.11381202161,   2854.64037391020 ),
  { 4511  237 }  (         0.492, 0.76371083106,   1603.29998928540 ),
  { 4511  238 }  (         0.487, 5.55383951779,   2627.11418447060 ),
  { 4511  239 }  (         0.487, 5.86510858429,    724.83081326790 ),
  { 4511  240 }  (         0.453, 0.61375011101,   1159.29331067010 ),
  { 4511  241 }  (         0.450, 2.28121042355,   3060.82592234740 ),
  { 4511  242 }  (         0.515, 4.78126059280,    447.79581952650 ),
  { 4511  243 }  (         0.449, 4.70231576312,    934.94851496820 ),
  { 4511  244 }  (         0.450, 1.91049508739,    597.35901666110 ),
  { 4511  245 }  (         0.438, 6.01178917646,   3178.14579056760 ),
  { 4511  246 }  (         0.494, 0.53844942275,   1354.43315884340 ),
  { 4511  247 }  (         0.501, 5.51752195462,   2435.15573003500 ),
  { 4511  248 }  (         0.432, 3.64903264921,    313.21047591890 ),
  { 4511  249 }  (         0.435, 3.02449828967,    533.88375078860 ),
  { 4511  250 }  (         0.426, 5.07945534339,   2524.02141025200 ),
  { 4511  251 }  (         0.491, 3.59286364200,    230.56457082540 ),
  { 4511  252 }  (         0.547, 0.34432090949,   1251.34038462480 ),
  { 4511  253 }  (         0.503, 1.57454509207,    454.90936652730 ),
  { 4511  254 }  (         0.486, 4.39351469958,    462.02291352810 ),
  { 4511  255 }  (         0.524, 2.03003740296,   1279.79457262800 ),
  { 4511  256 }  (         0.388, 5.58318013074,    731.94436026870 ),
  { 4511  257 }  (         0.449, 1.11025492739,     56.62235130260 ),
  { 4511  258 }  (         0.398, 5.19943284273,   3267.01147078460 ),
  { 4511  259 }  (         0.416, 1.70821917336,    245.54242435240 ),
  { 4511  260 }  (         0.379, 1.80234948769,   2655.56837247380 ),
  { 4511  261 }  (         0.355, 1.65214516751,     78.71375183040 ),
  { 4511  262 }  (         0.404, 1.72647262603,   1141.13406340540 ),
  { 4511  263 }  (         0.335, 6.01254286794,    960.22130923370 ),
  { 4511  264 }  (         0.331, 1.74086938716,    490.33408917940 ),
  { 4511  265 }  (         0.401, 0.30034336462,   2332.06295581640 ),
  { 4511  266 }  (         0.336, 2.64385574909,   1021.24889455140 ),
  { 4511  267 }  (         0.389, 0.31259289221,   2413.81508903260 ),
  { 4511  268 }  (         0.314, 5.73833529708,   1158.54255114470 ),
  { 4511  269 }  (         0.313, 4.74363791106,    938.12990870590 ),
  { 4511  270 }  (         0.333, 0.80112437148,   1585.14074202070 ),
  { 4511  271 }  (         0.323, 3.52656245280,   3274.12501778540 ),
  { 4511  272 }  (         0.395, 1.73181407631,   1593.00504854690 ),
  { 4511  273 }  (         0.302, 4.64184749164,   1261.63532536330 ),
  { 4511  274 }  (         0.325, 0.54991590409,     43.28902917830 ),
  { 4511  275 }  (         0.293, 0.97977818746,   1585.89150154610 ),
  { 4511  276 }  (         0.341, 2.80833606944,   1514.29129671650 ),
  { 4511  277 }  (         0.304, 6.12522825214,   1262.38608488870 ),
  { 4511  278 }  (         0.286, 2.89800423081,    530.21222995640 ),
  { 4511  279 }  (         0.387, 0.46648572639,   1592.25428902150 ),
  { 4511  280 }  (         0.285, 4.56394598052,   1268.74887236410 ),
  { 4511  281 }  (         0.310, 4.69102289591,     76.26607127560 ),
  { 4511  282 }  (         0.278, 5.49867187248,    280.96714700450 ),
  { 4511  283 }  (         0.358, 5.45926487831,    113.38771495710 ),
  { 4511  284 }  (         0.283, 1.09230506350,   1061.82961074400 ),
  { 4511  285 }  (         0.326, 0.60265259639,    827.92358748650 ),
  { 4511  286 }  (         0.284, 5.36580034539,   1165.65609814550 ),
  { 4511  287 }  (         0.281, 5.54635461050,   3370.10424500320 ),
  { 4511  288 }  (         0.269, 3.92616563946,     42.53826965290 ),
  { 4511  289 }  (         0.275, 2.58465453365,    373.01422095920 ),
  { 4511  290 }  (         0.357, 1.39391983207,   1493.09366806600 ),
  { 4511  291 }  (         0.258, 5.96670694140,   1269.49963188950 ),
  { 4511  292 }  (         0.259, 2.56026216784,   9146.79006902100 ),
  { 4511  293 }  (         0.281, 2.74823090198,   4694.00295470760 ),
  { 4511  294 }  (         0.281, 3.01324655940,    320.32402291970 ),
  { 4511  295 }  (         0.272, 4.18504958920,   8624.21265092720 ),
  { 4511  296 }  (         0.245, 1.24462798353,    252.65597135320 ),
  { 4511  297 }  (         0.244, 2.02892764690,   3377.21779200400 ),
  { 4511  298 }  (         0.324, 1.84851618413,   1289.94650101460 ),
  { 4511  299 }  (         0.221, 6.22167997496,   3281.23856478620 ),
  { 4511  300 }  (         0.238, 3.93371505401,   3171.03224356680 ),
  { 4511  301 }  (         0.226, 5.94296271326,    224.34479570190 ),
  { 4511  302 }  (         0.213, 3.68264234750,   1048.33622992530 ),
  { 4511  303 }  (         0.216, 5.82941334164,   1567.73225428140 ),
  { 4511  304 }  (         0.295, 4.70194747095,   3067.93946934820 ),
  { 4511  305 }  (         0.206, 4.98184230959,   1357.61455258110 ),
  { 4511  306 }  (         0.202, 1.32439444045,   4326.39340097380 ),
  { 4511  307 }  (         0.227, 0.78540105705,     59.80374504030 ),
  { 4511  308 }  (         0.237, 5.56926897693,   2943.50605412720 ),
  { 4511  309 }  (         0.207, 0.07907015398,   5223.69391980220 ),
  { 4511  310 }  (         0.199, 3.30501818656,   4120.20785253660 ),
  { 4511  311 }  (         0.194, 5.95526916809,     84.93352695390 ),
  { 4511  312 }  (         0.266, 1.58032565718,    983.11585891360 ),
  { 4511  313 }  (         0.198, 4.31078641704,   4017.11507831800 ),
  { 4511  314 }  (         0.198, 0.30166351366,   1166.40685767090 ),
  { 4511  315 }  (         0.188, 0.90738705875,    135.33610313300 ),
  { 4511  316 }  (         0.186, 0.69289672485,     92.79783348010 ),
  { 4511  317 }  (         0.182, 1.18931462257,   1512.80682400820 ),
  { 4511  318 }  (         0.191, 1.04146023518,   1884.12412393800 ),
  { 4511  319 }  (         0.174, 6.13734594396,   3597.63043444280 ),
  { 4511  320 }  (         0.189, 0.35191512844,   1372.59240610810 ),
  { 4511  321 }  (         0.172, 4.35250972697,   1578.02719501990 ),
  { 4511  322 }  (         0.173, 2.30241719278,   1176.70179840940 ),
  { 4511  323 }  (         0.220, 1.06991056825,   2200.51599359460 ),
  { 4511  324 }  (         0.186, 4.90511103807,   3583.40334044120 ),
  { 4511  325 }  (         0.189, 0.24160744024,   1670.82502850000 ),
  { 4511  326 }  (         0.206, 0.01485146863,   2730.20695868920 ),
  { 4511  327 }  (         0.174, 1.83997277029,    746.92221379570 ),
  { 4511  328 }  (         0.225, 3.13108099660,    630.33605875840 ),
  { 4511  329 }  (         0.206, 5.22730929781,   3995.77443731560 ),
  { 4511  330 }  (         0.169, 2.57956682688,   9161.01716302260 ),
  { 4511  331 }  (         0.165, 1.51795928301,   4010.00153131720 ),
  { 4511  332 }  (         0.181, 2.05055200822,    842.90144101350 ),
  { 4511  333 }  (         0.181, 5.96554625357,   1578.77795454530 ),
  { 4511  334 }  (         0.166, 1.55114863100,   1070.42763045310 ),
  { 4511  335 }  (         0.157, 5.87839958880,   3914.02230409940 ),
  { 4511  336 }  (         0.160, 0.43729819176,   2545.36205125440 ),
  { 4511  337 }  (         0.168, 5.73975661792,   2847.52682690940 ),
  { 4511  338 }  (         0.157, 2.25764581068,    850.01498801430 ),
  { 4511  339 }  (         0.187, 0.64918748618,    842.15068148810 ),
  { 4511  340 }  (         0.180, 1.88055488803,    685.47393735270 ),
  { 4511  341 }  (         0.153, 4.15259684562,   4333.50694797460 ),
  { 4511  342 }  (         0.154, 3.65536637158,  77734.01845962799 ),
  { 4511  343 }  (         0.151, 3.17795437121,   3590.51688744200 ),
  { 4511  344 }  (         0.155, 3.87623547990,    327.43756992050 ),
  { 4511  345 }  (         0.171, 3.33647878498,   1912.57831194120 ),
  { 4511  346 }  (         0.188, 4.53005359421,   1041.22268292450 ),
  { 4511  347 }  (         0.134, 4.09921613445,    530.44172462000 ),
  { 4511  348 }  (         0.123, 4.79543460218,   1098.73880610440 ),
  { 4511  349 }  (         0.161, 2.02006564218,    860.30992875280 ),
  { 4511  350 }  (         0.143, 2.40197278329,    529.16970023280 ),
  { 4511  351 }  (         0.115, 1.55831212007,   9153.90361602180 ),
  { 4511  352 }  (         0.106, 5.94313244357,   1057.89745748090 ),
  { 4511  353 }  (         0.119, 5.10578428676,   1056.93424963440 ),
  { 4511  354 }  (         0.100, 5.74974781049,    501.23677709140 ),
  { 4511  355 }  (         0.094, 1.40134175492,   1059.33374607940 ),
  { 4511  356 }  (         0.098, 3.79115318281,    497.44763618020 ),
  { 4511  357 }  (         0.090, 4.09610113044,   1064.04779663520 ),
  { 4511  358 }  (         0.102, 1.10442899544,   1969.20066324380 ),
  { 4511  359 }  (         0.087, 0.58218477838,   1173.52040467170 ),
  { 4511  360 }  (         0.109, 3.83745968299,    525.49817940060 ),
  { 4511  361 }  (         0.094, 4.59915291355,   1059.43011429900 ),
  { 4511  362 }  (         0.118, 6.11701561559,   1069.67687092770 ),
  { 4511  363 }  (         0.107, 5.40509332689,    679.25416222920 ),
  { 4511  364 }  (         0.089, 5.90037690244,    757.21715453420 ),
  { 4511  365 }  (         0.078, 6.06217863109,    970.51624997220 ),
  { 4511  366 }  (         0.080, 5.45470236239,   3163.91869656600 ),
  { 4511  367 }  (         0.072, 5.65789862232,   1151.42900414390 ),
  { 4511  368 }  (         0.080, 0.04539720100,   1080.72257119160 ),
  { 4511  369 }  (         0.075, 4.26526686574,   1058.41872234270 )
  (*$endif *)
      );
  (*@\\\0000000401*)
  (*@/// vsop87_jup_l2:array[0..190,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_l2:array[0.. 56,0..2] of extended = (
  (*$else *)
  vsop87_jup_l2:array[0..190,0..2] of extended = (
  (*$endif *)
  { 4512    1 }  (     47233.598, 4.32148323554,      7.11354700080 ),
  { 4512    2 }  (     30629.053, 2.93021440216,    529.69096509460 ),
  { 4512    3 }  (     38965.550, 0.00000000000,      0.00000000000 ),
  { 4512    4 }  (      3189.317, 1.05504615595,    522.57741809380 ),
  { 4512    5 }  (      2723.358, 3.41411526638,   1059.38193018920 ),
  { 4512    6 }  (      2729.292, 4.84545481351,    536.80451209540 ),
  { 4512    7 }  (      1721.069, 4.18734385158,     14.22709400160 ),
  { 4512    8 }  (       383.258, 5.76790714387,    419.48464387520 ),
  { 4512    9 }  (       367.498, 6.05509120409,    103.09277421860 ),
  { 4512   10 }  (       377.524, 0.76048964872,    515.46387109300 ),
  { 4512   11 }  (       337.386, 3.78644384244,      3.18139373770 ),
  { 4512   12 }  (       308.200, 0.69356654052,    206.18554843720 ),
  { 4512   13 }  (       218.408, 3.81389191353,   1589.07289528380 ),
  { 4512   14 }  (       198.883, 5.33996443444,   1066.49547719000 ),
  { 4512   15 }  (       197.445, 2.48356402053,      3.93215326310 ),
  { 4512   16 }  (       146.230, 3.81373196838,    639.89728631400 ),
  { 4512   17 }  (       155.862, 1.40642426467,   1052.26838318840 ),
  { 4512   18 }  (       129.570, 5.83738872525,    412.37109687440 ),
  { 4512   19 }  (       141.932, 1.63435169016,    426.59819087600 ),
  { 4512   20 }  (       117.327, 1.41435462588,    625.67019231240 ),
  { 4512   21 }  (        96.733, 4.03383427887,    110.20632121940 ),
  { 4512   22 }  (        90.823, 1.10630629042,     95.97922721780 ),
  { 4512   23 }  (        78.769, 4.63726131329,    543.91805909620 ),
  { 4512   24 }  (        72.392, 2.21716670026,    735.87651353180 ),
  { 4512   25 }  (        87.292, 2.52235174825,    632.78373931320 ),
  { 4512   26 }  (        56.910, 3.12292059854,    213.29909543800 ),
  { 4512   27 }  (        48.622, 1.67283791618,    309.27832265580 ),
  { 4512   28 }  (        58.475, 0.83216317444,    199.07200143640 ),
  { 4512   29 }  (        40.150, 4.02485444740,     21.34064100240 ),
  { 4512   30 }  (        39.784, 0.62416945827,    323.50541665740 ),
  { 4512   31 }  (        35.718, 2.32581247002,    728.76296653100 ),
  { 4512   32 }  (        25.620, 2.51240623862,   1162.47470440780 ),
  { 4512   33 }  (        29.255, 3.60838327799,     10.29494073850 ),
  { 4512   34 }  (        23.591, 3.00532139306,    956.28915597060 ),
  { 4512   35 }  (        27.814, 3.23992013743,    838.96928775040 ),
  { 4512   36 }  (        25.993, 4.50118298290,    742.99006053260 ),
  { 4512   37 }  (        25.194, 1.21868110687,   1045.15483618760 ),
  { 4512   38 }  (        19.458, 4.29028644674,    532.87235883230 ),
  { 4512   39 }  (        17.660, 0.80953941560,    508.35032409220 ),
  { 4512   40 }  (        15.355, 5.81037986941,   1596.18644228460 ),
  { 4512   41 }  (        17.058, 4.20001977723,   2118.76386037840 ),
  { 4512   42 }  (        17.040, 1.83402146640,    526.50957135690 ),
  { 4512   43 }  (        14.661, 3.99989622586,    117.31986822020 ),
  { 4512   44 }  (        13.639, 1.80336677963,    302.16477565500 ),
  { 4512   45 }  (        13.230, 2.51856643603,     88.86568021700 ),
  { 4512   46 }  (        12.756, 4.36856232414,   1169.58825140860 ),
  { 4512   47 }  (        15.292, 0.68174165476,    942.06206196900 ),
  { 4512   48 }  (        10.986, 4.43586634639,    525.75881183150 ),
  { 4512   49 }  (        13.920, 5.95169568482,    316.39186965660 ),
  { 4512   50 }  (         9.437, 2.17684563456,   1155.36115740700 ),
  { 4512   51 }  (         8.812, 3.29452783338,    220.41264243880 ),
  { 4512   52 }  (         7.823, 5.75672228354,    846.08283475120 ),
  { 4512   53 }  (         7.549, 2.70955516779,    533.62311835770 ),
  { 4512   54 }  (         9.681, 1.71563161051,   1581.95934828300 ),
  { 4512   55 }  (         8.690, 3.31924493607,    831.85574074960 ),
  { 4512   56 }  (         6.285, 0.49939863541,    949.17560896980 ),
  { 4512   57 }  (         6.685, 2.17560093281,   1265.56747862640 )
  (*$ifndef meeus *)                                                 ,
  { 4512   58 }  (         5.381, 6.00510875948,    405.25754987360 ),
  { 4512   59 }  (         4.676, 1.40846192799,   1258.45393162560 ),
  { 4512   60 }  (         4.421, 3.02360159274,   1692.16566950240 ),
  { 4512   61 }  (         4.403, 5.47737266160,    433.71173787680 ),
  { 4512   62 }  (         4.286, 5.07139951645,   1073.60902419080 ),
  { 4512   63 }  (         4.201, 5.28560721767,     18.15924726470 ),
  { 4512   64 }  (         3.933, 1.26665387164,    853.19638175200 ),
  { 4512   65 }  (         5.351, 3.65320121089,   1272.68102562720 ),
  { 4512   66 }  (         4.392, 2.27325303667,   1368.66025284500 ),
  { 4512   67 }  (         3.482, 1.53983001273,    519.39602435610 ),
  { 4512   68 }  (         2.745, 2.09685315627,   1478.86657406440 ),
  { 4512   69 }  (         2.737, 1.06017230524,   1574.84580128220 ),
  { 4512   70 }  (         2.897, 2.05128453665,   1361.54670584420 ),
  { 4512   71 }  (         3.075, 0.99085727534,    191.95845443560 ),
  { 4512   72 }  (         2.462, 2.37173605635,   1471.75302706360 ),
  { 4512   73 }  (         2.203, 2.47960567714,    721.64941953020 ),
  { 4512   74 }  (         2.096, 3.71482580504,   1485.98012106520 ),
  { 4512   75 }  (         1.984, 1.88475229557,   1685.05212250160 ),
  { 4512   76 }  (         2.274, 3.03360234351,   1148.24761040620 ),
  { 4512   77 }  (         2.041, 6.17114556019,    330.61896365820 ),
  { 4512   78 }  (         1.451, 4.72055072637,     32.24332891440 ),
  { 4512   79 }  (         1.454, 5.14703918585,   1375.77379984580 ),
  { 4512   80 }  (         1.447, 3.18833439444,    635.96513305090 ),
  { 4512   81 }  (         1.403, 4.26712075104,    551.03160609700 ),
  { 4512   82 }  (         1.420, 1.99288040133,    629.60234557550 ),
  { 4512   83 }  (         1.269, 0.03300387779,   2125.87740737920 ),
  { 4512   84 }  (         1.276, 2.26356919237,   1788.14489672020 ),
  { 4512   85 }  (         1.189, 1.70223550488,   1677.93857550080 ),
  { 4512   86 }  (         1.182, 2.18142313946,   1795.25844372100 ),
  { 4512   87 }  (         1.366, 1.27629917215,   1038.04128918680 ),
  { 4512   88 }  (         1.306, 4.76302079847,   1062.56332392690 ),
  { 4512   89 }  (         1.109, 2.97787130235,     81.75213321620 ),
  { 4512   90 }  (         1.027, 1.99236027398,    295.05122865420 ),
  { 4512   91 }  (         1.349, 4.01621534182,    539.98590583310 ),
  { 4512   92 }  (         1.025, 3.75336759986,     28.45418800320 ),
  { 4512   93 }  (         0.977, 3.01355125761,    124.43341522100 ),
  { 4512   94 }  (         1.290, 4.62594234857,   2648.45482547300 ),
  { 4512   95 }  (         1.065, 5.06153058155,   1699.27921650320 ),
  { 4512   96 }  (         0.965, 1.17716405513,     99.91138048090 ),
  { 4512   97 }  (         1.021, 1.90712102660,    750.10360753340 ),
  { 4512   98 }  (         0.923, 3.53450109212,    227.52618943960 ),
  { 4512   99 }  (         1.059, 0.13532061468,    416.30325013750 ),
  { 4512  100 }  (         0.836, 2.07492422755,   1056.20053645150 ),
  { 4512  101 }  (         0.889, 1.75177808106,   1898.35121793960 ),
  { 4512  102 }  (         0.772, 2.89217715561,   2008.55753915900 ),
  { 4512  103 }  (         1.014, 2.80847772922,   1464.63948006280 ),
  { 4512  104 }  (         0.820, 1.99735697577,   2111.65031337760 ),
  { 4512  105 }  (         0.787, 4.91912237671,   1055.44977692610 ),
  { 4512  106 }  (         0.743, 2.65209650690,    106.27416795630 ),
  { 4512  107 }  (         0.705, 0.08006443278,    963.40270297140 ),
  { 4512  108 }  (         0.724, 3.29664246938,    628.85158605010 ),
  { 4512  109 }  (         0.791, 1.64655202110,   2001.44399215820 ),
  { 4512  110 }  (         0.822, 2.74067639972,    618.55664531160 ),
  { 4512  111 }  (         0.761, 1.26393500358,   1382.88734684660 ),
  { 4512  112 }  (         0.650, 1.19590511216,    422.66603761290 ),
  { 4512  113 }  (         0.677, 1.88476058357,   2104.53676637680 ),
  { 4512  114 }  (         0.681, 5.47481665606,   5760.49843189760 ),
  { 4512  115 }  (         0.681, 3.11621209674,   5746.27133789600 ),
  { 4512  116 }  (         0.644, 4.68385640894,    611.44309831080 ),
  { 4512  117 }  (         0.752, 3.03497138894,   2221.85663459700 ),
  { 4512  118 }  (         0.641, 1.86274530783,    636.71589257630 ),
  { 4512  119 }  (         0.614, 3.07677356670,    380.12776796000 ),
  { 4512  120 }  (         0.635, 4.53916684689,   9676.48103411560 ),
  { 4512  121 }  (         0.635, 0.61458805483,   9690.70812811720 ),
  { 4512  122 }  (         0.822, 6.25170365084,    423.41679713830 ),
  { 4512  123 }  (         0.762, 4.32362906505,   1802.37199072180 ),
  { 4512  124 }  (         0.582, 0.84137872868,   1891.23767093880 ),
  { 4512  125 }  (         0.558, 3.96171840325,    440.82528487760 ),
  { 4512  126 }  (         0.624, 2.83657771014,   1905.46476494040 ),
  { 4512  127 }  (         0.711, 3.43538032357,    824.74219374880 ),
  { 4512  128 }  (         0.517, 1.10660016329,    107.02492748170 ),
  { 4512  129 }  (         0.535, 1.55761050176,   1994.33044515740 ),
  { 4512  130 }  (         0.501, 4.44389802599,    647.01083331480 ),
  { 4512  131 }  (         0.414, 5.37130370397,   2228.97018159780 ),
  { 4512  132 }  (         0.533, 2.54756313371,   1781.03134971940 ),
  { 4512  133 }  (         0.393, 1.26351262287,    210.11770170030 ),
  { 4512  134 }  (         0.433, 2.90103969634,   1063.31408345230 ),
  { 4512  135 }  (         0.384, 1.36194621083,    203.00415469950 ),
  { 4512  136 }  (         0.440, 1.46934545869,   2214.74308759620 ),
  { 4512  137 }  (         0.424, 4.98974282486,   3178.14579056760 ),
  { 4512  138 }  (         0.338, 2.72210106345,   2324.94940881560 ),
  { 4512  139 }  (         0.332, 0.37505564414,   2655.56837247380 ),
  { 4512  140 }  (         0.318, 6.11024720065,    934.94851496820 ),
  { 4512  141 }  (         0.405, 3.51005860013,   2751.54759969160 ),
  { 4512  142 }  (         0.388, 5.00609647265,   2015.67108615980 ),
  { 4512  143 }  (         0.424, 4.29668654117,   5753.38488489680 ),
  { 4512  144 }  (         0.328, 2.35571531981,   1251.34038462480 ),
  { 4512  145 }  (         0.316, 0.16949503062,   1279.79457262800 ),
  { 4512  146 }  (         0.345, 2.89328206121,   2957.73314812880 ),
  { 4512  147 }  (         0.303, 1.63964826684,   2428.04218303420 ),
  { 4512  148 }  (         0.328, 3.36132375845,   1141.13406340540 ),
  { 4512  149 }  (         0.294, 2.48947693371,   2641.34127847220 ),
  { 4512  150 }  (         0.350, 1.50537240918,   2317.83586181480 ),
  { 4512  151 }  (         0.287, 1.69638214958,   2420.92863603340 ),
  { 4512  152 }  (         0.272, 0.27466529753,    319.57326339430 ),
  { 4512  153 }  (         0.303, 2.43034117616,     70.84944530420 ),
  { 4512  154 }  (         0.251, 0.43544711316,   3259.89792378380 ),
  { 4512  155 }  (         0.224, 4.49752269293,   5223.69391980220 ),
  { 4512  156 }  (         0.272, 2.98590404673,   1457.52593306200 ),
  { 4512  157 }  (         0.228, 5.47896916415,   1603.29998928540 ),
  { 4512  158 }  (         0.288, 2.30146999217,   2854.64037391020 ),
  { 4512  159 }  (         0.207, 5.94297320087,   9153.90361602180 ),
  { 4512  160 }  (         0.243, 1.58604251447,   2744.43405269080 ),
  { 4512  161 }  (         0.228, 1.28182702946,   2310.72231481400 ),
  { 4512  162 }  (         0.224, 1.28623905132,   3060.82592234740 ),
  { 4512  163 }  (         0.222, 0.63265553397,   3163.91869656600 ),
  { 4512  164 }  (         0.242, 2.52382905368,   3274.12501778540 ),
  { 4512  165 }  (         0.188, 6.00513627145,     92.04707395470 ),
  { 4512  166 }  (         0.239, 1.93897157244,   2413.81508903260 ),
  { 4512  167 }  (         0.214, 1.14529237568,   2531.13495725280 ),
  { 4512  168 }  (         0.200, 3.42280996072,     99.16062095550 ),
  { 4512  169 }  (         0.179, 0.53892926207,   2207.62954059540 ),
  { 4512  170 }  (         0.177, 5.56545270243,   2332.06295581640 ),
  { 4512  171 }  (         0.172, 1.38604067808,    945.99421523210 ),
  { 4512  172 }  (         0.203, 0.41899069603,   2840.41327990860 ),
  { 4512  173 }  (         0.231, 2.26353330460,   2097.42321937600 ),
  { 4512  174 }  (         0.228, 3.82701076821,    113.38771495710 ),
  { 4512  175 }  (         0.165, 4.08776703733,   6283.07584999140 ),
  { 4512  176 }  (         0.202, 3.30429764992,   3067.93946934820 ),
  { 4512  177 }  (         0.224, 3.69285208525,   2435.15573003500 ),
  { 4512  178 }  (         0.214, 2.55756944911,   2538.24850425360 ),
  { 4512  179 }  (         0.203, 2.24205059922,     67.66805156650 ),
  { 4512  180 }  (         0.152, 5.48122906518,  10213.28554621100 ),
  { 4512  181 }  (         0.191, 2.68685722531,   1773.91780271860 ),
  { 4512  182 }  (         0.189, 2.95184620359,    732.69511979410 ),
  { 4512  183 }  (         0.149, 1.98737542735,   1049.08698945070 ),
  { 4512  184 }  (         0.163, 1.24084734609,   3053.71237534660 ),
  { 4512  185 }  (         0.171, 2.34210749987,   1354.43315884340 ),
  { 4512  186 }  (         0.112, 5.77407285790,    547.85021235930 ),
  { 4512  187 }  (         0.124, 0.14001204498,    860.30992875280 ),
  { 4512  188 }  (         0.086, 1.26924601636,    511.53171782990 ),
  { 4512  189 }  (         0.114, 5.15982838070,   1592.25428902150 ),
  { 4512  190 }  (         0.091, 1.48896790758,   1567.73225428140 ),
  { 4512  191 }  (         0.086, 4.34444949905,   1069.67687092770 )
  (*$endif *)
      );
  (*@\\\000000041B*)
  (*@/// vsop87_jup_l3:array[0..108,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_l3:array[0.. 38,0..2] of extended = (
  (*$else *)
  vsop87_jup_l3:array[0..108,0..2] of extended = (
  (*$endif *)
  { 4513    1 }  (      6501.665, 2.59862880482,      7.11354700080 ),
  { 4513    2 }  (      1356.524, 1.34635886411,    529.69096509460 ),
  { 4513    3 }  (       470.716, 2.47503977883,     14.22709400160 ),
  { 4513    4 }  (       416.960, 3.24451243214,    536.80451209540 ),
  { 4513    5 }  (       352.851, 2.97360159003,    522.57741809380 ),
  { 4513    6 }  (       154.880, 2.07565585817,   1059.38193018920 ),
  { 4513    7 }  (        86.771, 2.51431584316,    515.46387109300 ),
  { 4513    8 }  (        33.538, 3.82633794497,   1066.49547719000 ),
  { 4513    9 }  (        44.378, 0.00000000000,      0.00000000000 ),
  { 4513   10 }  (        22.644, 2.98231326774,    543.91805909620 ),
  { 4513   11 }  (        23.737, 1.27667172313,    412.37109687440 ),
  { 4513   12 }  (        28.457, 2.44754756058,    206.18554843720 ),
  { 4513   13 }  (        19.798, 2.10099934005,    639.89728631400 ),
  { 4513   14 }  (        19.740, 1.40255938973,    419.48464387520 ),
  { 4513   15 }  (        18.768, 1.59368403500,    103.09277421860 ),
  { 4513   16 }  (        17.033, 2.30214681202,     21.34064100240 ),
  { 4513   17 }  (        16.774, 2.59821460673,   1589.07289528380 ),
  { 4513   18 }  (        16.214, 3.14521117299,    625.67019231240 ),
  { 4513   19 }  (        16.055, 3.36030126297,   1052.26838318840 ),
  { 4513   20 }  (        13.392, 2.75973892202,     95.97922721780 ),
  { 4513   21 }  (        13.234, 2.53862244340,    199.07200143640 ),
  { 4513   22 }  (        12.611, 6.26578110400,    426.59819087600 ),
  { 4513   23 }  (         8.637, 2.26563256289,    110.20632121940 ),
  { 4513   24 }  (         6.725, 3.42566433316,    309.27832265580 ),
  { 4513   25 }  (         8.701, 1.76334960737,     10.29494073850 ),
  { 4513   26 }  (         6.527, 4.03869562907,    728.76296653100 ),
  { 4513   27 }  (         5.368, 5.25196153539,    323.50541665740 ),
  { 4513   28 }  (         5.675, 2.52096417685,    508.35032409220 ),
  { 4513   29 }  (         5.399, 2.91184687105,   1045.15483618760 ),
  { 4513   30 }  (         3.996, 4.30290261177,     88.86568021700 ),
  { 4513   31 }  (         3.857, 3.52381361552,    302.16477565500 ),
  { 4513   32 }  (         3.774, 4.09125315146,    735.87651353180 ),
  { 4513   33 }  (         3.269, 1.43175991274,    956.28915597060 ),
  { 4513   34 }  (         2.783, 4.35817507670,   1596.18644228460 ),
  { 4513   35 }  (         2.661, 1.25276590759,    213.29909543800 ),
  { 4513   36 }  (         2.553, 2.23785673285,    117.31986822020 ),
  { 4513   37 }  (         2.371, 2.89662409244,    742.99006053260 ),
  { 4513   38 }  (         2.656, 5.01505839848,    838.96928775040 ),
  (*$ifndef meeus *)
  { 4513   39 }  (         1.948, 2.77248294666,   1169.58825140860 ),
  (*$endif *)
  { 4513   40 }  (         2.279, 2.35581871230,    942.06206196900 )
  (*$ifndef meeus *)                                                 ,
  { 4513   41 }  (         1.474, 1.61011468581,    220.41264243880 ),
  { 4513   42 }  (         1.457, 3.09381959396,   2118.76386037840 ),
  { 4513   43 }  (         1.937, 5.01388256693,    831.85574074960 ),
  { 4513   44 }  (         1.585, 1.40097680805,    405.25754987360 ),
  { 4513   45 }  (         1.257, 3.97811260358,   1155.36115740700 ),
  { 4513   46 }  (         1.227, 3.45959919972,   1073.60902419080 ),
  { 4513   47 }  (         0.986, 3.39209446167,    532.87235883230 ),
  { 4513   48 }  (         0.942, 2.70200385825,    191.95845443560 ),
  { 4513   49 }  (         0.828, 1.48348768286,    632.78373931320 ),
  { 4513   50 }  (         0.797, 1.10706688850,   1162.47470440780 ),
  { 4513   51 }  (         0.822, 3.30295824153,   1258.45393162560 ),
  { 4513   52 }  (         0.710, 5.89798771980,    853.19638175200 ),
  { 4513   53 }  (         0.766, 3.66351539483,   1581.95934828300 ),
  { 4513   54 }  (         0.722, 3.74673245797,    433.71173787680 ),
  { 4513   55 }  (         0.663, 2.93063953915,   1574.84580128220 ),
  { 4513   56 }  (         0.658, 3.52797311863,    525.75881183150 ),
  { 4513   57 }  (         0.609, 4.14881313523,    721.64941953020 ),
  { 4513   58 }  (         0.598, 4.69454609357,     81.75213321620 ),
  { 4513   59 }  (         0.668, 1.96442971289,   1272.68102562720 ),
  { 4513   60 }  (         0.515, 1.57251270902,    949.17560896980 ),
  { 4513   61 }  (         0.658, 2.02329201466,    526.50957135690 ),
  { 4513   62 }  (         0.517, 4.35827478516,   1368.66025284500 ),
  { 4513   63 }  (         0.510, 4.95846155301,   1148.24761040620 ),
  { 4513   64 }  (         0.507, 4.31396370095,    330.61896365820 ),
  { 4513   65 }  (         0.567, 2.27813343743,    551.03160609700 ),
  { 4513   66 }  (         0.480, 3.86758235988,   1361.54670584420 ),
  { 4513   67 }  (         0.383, 0.24287136454,    611.44309831080 ),
  { 4513   68 }  (         0.434, 2.95461755540,   1038.04128918680 ),
  { 4513   69 }  (         0.377, 1.42957648215,    124.43341522100 ),
  { 4513   70 }  (         0.391, 4.07770324592,   1471.75302706360 ),
  { 4513   71 }  (         0.385, 4.70295179800,    519.39602435610 ),
  { 4513   72 }  (         0.428, 2.22472522305,    539.98590583310 ),
  { 4513   73 }  (         0.343, 4.83463725823,   2125.87740737920 ),
  { 4513   74 }  (         0.394, 4.52891996323,   1464.63948006280 ),
  { 4513   75 }  (         0.305, 2.02797683648,   1485.98012106520 ),
  { 4513   76 }  (         0.283, 0.97461612169,   1905.46476494040 ),
  { 4513   77 }  (         0.276, 3.83552772064,   1062.56332392690 ),
  { 4513   78 }  (         0.351, 2.06334334462,    533.62311835770 ),
  { 4513   79 }  (         0.304, 3.93228052293,   1685.05212250160 ),
  { 4513   80 }  (         0.322, 3.54763044791,    846.08283475120 ),
  { 4513   81 }  (         0.345, 4.18332148409,   1788.14489672020 ),
  { 4513   82 }  (         0.253, 3.12703531516,   1994.33044515740 ),
  { 4513   83 }  (         0.257, 1.05361498985,   1478.86657406440 ),
  { 4513   84 }  (         0.232, 1.69999081817,   1692.16566950240 ),
  { 4513   85 }  (         0.225, 2.51624149780,   1891.23767093880 ),
  { 4513   86 }  (         0.217, 4.58512911216,    963.40270297140 ),
  { 4513   87 }  (         0.277, 3.63353707701,   1677.93857550080 ),
  { 4513   88 }  (         0.242, 2.90163762388,   2310.72231481400 ),
  { 4513   89 }  (         0.211, 3.96419403991,    295.05122865420 ),
  { 4513   90 }  (         0.199, 5.17046500750,    618.55664531160 ),
  { 4513   91 }  (         0.256, 4.19052619061,   1781.03134971940 ),
  { 4513   92 }  (         0.192, 0.81556540966,   2221.85663459700 ),
  { 4513   93 }  (         0.187, 3.49895198981,   2648.45482547300 ),
  { 4513   94 }  (         0.208, 4.11838429822,   2097.42321937600 ),
  { 4513   95 }  (         0.183, 3.30680692414,   1699.27921650320 ),
  { 4513   96 }  (         0.231, 2.54516792766,   1375.77379984580 ),
  { 4513   97 }  (         0.189, 5.74277274755,   2627.11418447060 ),
  { 4513   98 }  (         0.214, 5.48031974537,   1354.43315884340 ),
  { 4513   99 }  (         0.220, 3.87471989410,   2104.53676637680 ),
  { 4513  100 }  (         0.171, 6.10827209399,   1382.88734684660 ),
  { 4513  101 }  (         0.184, 5.98415847544,    750.10360753340 ),
  { 4513  102 }  (         0.171, 5.25744961028,    824.74219374880 ),
  { 4513  103 }  (         0.151, 4.30799091626,   2001.44399215820 ),
  { 4513  104 }  (         0.140, 4.27089466070,   1265.56747862640 ),
  { 4513  105 }  (         0.097, 4.67188056608,    647.01083331480 ),
  { 4513  106 }  (         0.088, 2.43775210355,    440.82528487760 ),
  { 4513  107 }  (         0.075, 3.93105183253,   1055.44977692610 ),
  { 4513  108 }  (         0.079, 1.88533153220,    934.94851496820 ),
  { 4513  109 }  (         0.077, 3.80503143236,   1603.29998928540 )
  (*$endif *)
      );
  (*@\\\0000002F44*)
  (*@/// vsop87_jup_l4:array[0.. 44,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_l4:array[0.. 18,0..2] of extended = (
  (*$else *)
  vsop87_jup_l4:array[0.. 44,0..2] of extended = (
  (*$endif *)
  { 4514    1 }  (       669.483, 0.85282421090,      7.11354700080 ),
  { 4514    2 }  (        99.961, 0.74258947751,     14.22709400160 ),
  { 4514    3 }  (       114.019, 3.14159265359,      0.00000000000 ),
  { 4514    4 }  (        50.024, 1.65346208248,    536.80451209540 ),
  { 4514    5 }  (        43.585, 5.82026386621,    529.69096509460 ),
  { 4514    6 }  (        31.813, 4.85829986650,    522.57741809380 ),
  { 4514    7 }  (        14.742, 4.29061635784,    515.46387109300 ),
  { 4514    8 }  (         8.899, 0.71478520741,   1059.38193018920 ),
  { 4514    9 }  (         4.957, 1.29502259434,    543.91805909620 ),
  { 4514   10 }  (         4.484, 2.31715516627,   1066.49547719000 ),
  { 4514   11 }  (         4.251, 0.48326797501,     21.34064100240 ),
  { 4514   12 }  (         3.100, 3.00245542678,    412.37109687440 ),
  { 4514   13 }  (         2.055, 0.39858940218,    639.89728631400 ),
  { 4514   14 }  (         1.762, 4.90536207307,    625.67019231240 ),
  { 4514   15 }  (         1.902, 4.25925620271,    199.07200143640 ),
  { 4514   16 }  (         1.695, 4.26147580803,    206.18554843720 ),
  { 4514   17 }  (         1.375, 5.25546955667,   1052.26838318840 ),
  { 4514   18 }  (         1.203, 4.71614633845,     95.97922721780 ),
  { 4514   19 }  (         1.086, 1.28604571172,   1589.07289528380 )
  (*$ifndef meeus *)                                                 ,
  { 4514   20 }  (         0.982, 4.77990073662,   1045.15483618760 ),
  { 4514   21 }  (         0.935, 6.05847062188,     88.86568021700 ),
  { 4514   22 }  (         0.916, 5.77537499431,    728.76296653100 ),
  { 4514   23 }  (         0.890, 4.55299189579,    426.59819087600 ),
  { 4514   24 }  (         0.784, 3.40161567950,    419.48464387520 ),
  { 4514   25 }  (         0.768, 3.54672049322,    103.09277421860 ),
  { 4514   26 }  (         0.670, 0.52223307700,    110.20632121940 ),
  { 4514   27 }  (         0.415, 5.22809480633,    302.16477565500 ),
  { 4514   28 }  (         0.393, 6.24184621807,    956.28915597060 ),
  { 4514   29 }  (         0.381, 5.25466966040,    309.27832265580 ),
  { 4514   30 }  (         0.421, 0.59561318533,    117.31986822020 ),
  { 4514   31 }  (         0.346, 4.78348312106,    508.35032409220 ),
  { 4514   32 }  (         0.319, 3.47979828725,    323.50541665740 ),
  { 4514   33 }  (         0.331, 2.95893485883,   1596.18644228460 ),
  { 4514   34 }  (         0.295, 4.32713459459,    942.06206196900 ),
  { 4514   35 }  (         0.319, 0.47990052824,    831.85574074960 ),
  { 4514   36 }  (         0.251, 1.79898001222,   1073.60902419080 ),
  { 4514   37 }  (         0.212, 0.43917684084,    220.41264243880 ),
  { 4514   38 }  (         0.188, 1.12654974776,   1169.58825140860 ),
  { 4514   39 }  (         0.188, 2.16135407548,   1361.54670584420 ),
  { 4514   40 }  (         0.180, 3.43266428069,   1148.24761040620 ),
  { 4514   41 }  (         0.164, 1.92864127211,   2118.76386037840 ),
  { 4514   42 }  (         0.157, 3.02963907392,   1272.68102562720 ),
  { 4514   43 }  (         0.093, 5.60436000012,   1581.95934828300 ),
  { 4514   44 }  (         0.085, 5.02317256200,   1155.36115740700 ),
  { 4514   45 }  (         0.075, 3.13198879608,    632.78373931320 )
  (*$endif *)
      );
  (*@\\\0000000601*)
  (*@/// vsop87_jup_l5:array[0..  9,0..2] of extended = (...); *)
  (*$ifdef meeus *)
  vsop87_jup_l5:array[0..  4,0..2] of extended = (
  (*$else *)
  vsop87_jup_l5:array[0..  9,0..2] of extended = (
  (*$endif *)
  { 4515    1 }  (        49.577, 5.25658966184,      7.11354700080 ),
  { 4515    2 }  (        15.761, 5.25126837478,     14.22709400160 ),
  { 4515    3 }  (         4.343, 0.01461869263,    536.80451209540 ),
  { 4515    4 }  (         1.526, 1.09739911439,    522.57741809380 ),
  (*$ifndef meeus *)
  { 4515    5 }  (         0.728, 5.85949047619,    543.91805909620 ),
  { 4515    6 }  (         0.694, 0.87382487754,    515.46387109300 ),
  (*$endif *)
  { 4515    7 }  (         0.845, 3.14159265359,      0.00000000000 )
  (*$ifndef meeus *)                                                 ,
  { 4515    8 }  (         0.456, 0.81521692852,   1066.49547719000 ),
  { 4515    9 }  (         0.293, 5.62909357048,   1059.38193018920 ),
  { 4515   10 }  (         0.090, 0.21178119710,    529.69096509460 )
  (*$endif *)
      );
  (*@\\\000000020F*)
begin
  WITH result do begin
    a:=0; b:=0; c:=0;
    case index of
      0: if (nr>=low(vsop87_jup_l0)) and (nr<=high(vsop87_jup_l0)) then begin
        a:=vsop87_jup_l0[nr,0];  b:=vsop87_jup_l0[nr,1];  c:=vsop87_jup_l0[nr,2];
        end;
      1: if (nr>=low(vsop87_jup_l1)) and (nr<=high(vsop87_jup_l1)) then begin
        a:=vsop87_jup_l1[nr,0];  b:=vsop87_jup_l1[nr,1];  c:=vsop87_jup_l1[nr,2];
        end;
      2: if (nr>=low(vsop87_jup_l2)) and (nr<=high(vsop87_jup_l2)) then begin
        a:=vsop87_jup_l2[nr,0];  b:=vsop87_jup_l2[nr,1];  c:=vsop87_jup_l2[nr,2];
        end;
      3: if (nr>=low(vsop87_jup_l3)) and (nr<=high(vsop87_jup_l3)) then begin
        a:=vsop87_jup_l3[nr,0];  b:=vsop87_jup_l3[nr,1];  c:=vsop87_jup_l3[nr,2];
        end;
      4: if (nr>=low(vsop87_jup_l4)) and (nr<=high(vsop87_jup_l4)) then begin
        a:=vsop87_jup_l4[nr,0];  b:=vsop87_jup_l4[nr,1];  c:=vsop87_jup_l4[nr,2];
        end;
      5: if (nr>=low(vsop87_jup_l5)) and (nr<=high(vsop87_jup_l5)) then begin
        a:=vsop87_jup_l5[nr,0];  b:=vsop87_jup_l5[nr,1];  c:=vsop87_jup_l5[nr,2];
        end;
      end;
    end;
  end;
(*@\\\0000000801*)
(*@\\\0000000345*)
(*@\\\0002000A0C000A0C*)
(*@/// initialization *)
(*$ifdef delphi_1 *)
begin
(*$else *)
initialization
(*$endif *)
  datetime_2000_01_01:=Encodedate(2000,1,1);
(*@\\\*)
(*$ifdef delphi_ge_2 *) (*$warnings off *) (*$endif *)
end.
(*@\\\003D001205001205001101001201000011000F05*)
