/*****************************************************************************/
/*                                                                           */
/*                 (C) Copyright 1991-1997  Alberto Pasquale                 */
/*                 (C) Copyright 1999       Paul Kourochka                   */
/*                                                                           */
/*                   A L L   R I G H T S   R E S E R V E D                   */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   How to contact the author:  Paul Kourochka of 2:464/52@fidonet          */
/*                               nefpk@nefpk.8m.com                          */
/*                               Murmanskaja str, 2-28                       */
/*                               49098 Dniepropetrovsk                       */
/*                               Ukraine                                     */
/*                                                                           */
/*****************************************************************************/

// NeFeat.H

// This include file is for use by the developers who want to write a
// Feature DLL for NEF/pk.

// In the case you _really_ need some "wider" interface to NEF/pk,
// please feel free to ask the author.


#ifndef	_NEFEAT_H_
#define	_NEFEAT_H_

#if	defined (__OS2__)
#include <os2def.h>
typedef	CHAR const *	PCCH;
#elif	defined (__NT__)
#include <windef.h>
typedef	LPCSTR		PCSZ;
typedef	ULONG		APIRET;
#else
#error Unsupported platform for NeFeat.H, define __OS2__ or __NT__
#endif

// Exported functions qualifier.  You may redefine one from command line
#ifdef	FEATEXPORT
#define	FEATAPIENTRY	APIENTRY FEATEXPORT
#else
#define	FEATAPIENTRY	APIENTRY
#endif

#pragma	pack (1)

#ifdef	__cplusplus
extern	"C" {
#endif


		// for dest of FeatOut, bitwise

#define	FO_LOG	0x01	// send output to the log
#define	FO_CRT	0x02	// send output to the screen
#define	FO_BOTH	0x03	// send output to both log and screen
#define	FO_EXT	0x04	// featureDLL extensions (see below)

typedef	VOID (*APIENTRY FeatOut)(UCHAR dest, CHAR prty, PCSZ format, ...);

// Function to write output to screen and/or log.
// prty is the log priority char, usually one of "!*#: ".
// prty is ignored for output to screen.
// format and ... are the formatting string and parameters as in printf.
// Heading and trailing \n are ignored for log entry.


typedef struct _tagADR {
	USHORT	zone;
	USHORT	net;
	USHORT	node;
	USHORT	point;
} _ADR;


APIRET FEATAPIENTRY Init(FeatOut prnf);

#define	NFI_OK	0x00000000	// Normal termination
#define	NFI_ERR	0x00000001	// Initialization error

// Mandatory.
// This function is called during cfg file parsing, when the
// FeatureLoad statement is encountered and the DLL loaded.
// The prnf function pointer can be used to output messages to
// screen and log file.
// Must return NFI_OK on successful initialization.


APIRET FEATAPIENTRY ParseCfg(PCSZ clnline);

#define	NFP_OK	0x00000000	// Normal termination
#define	NFP_ERR	0x00000001	// Fatal Cfg Error

// This function is called by NEF when it finds a "Feature" configuration
// statement.  The entire line, after the "Feature" statement, is passed
// via the clnline pointer.
// This string is guaranteed without tabs, comments,
// trailing space and newline.
// This function must normally return NFP_OK.
// If NEF must exit with a "Cfg Error" errorlevel, this function
// must return NFP_ERR.
// In any case this function might use the prnf function to write
// something to the screen and log file.


// structure with TIC data to be used (not modified) by FeatureDLL.

// Please note that all the strings containing lines from TICs (desc,
// ldesc etc.) are a simple concatenation of the TIC lines, including
// the heading keyword and terminating '\n'.

// The string pointers are never NULL: if there is no entry, they point
// to an empty string.

		// definitions for the bits of the flags field.

#define	TDF_CRCused		0x0001	// crc field is valid
#define	TDF_PassThru		0x0002	// The area is Pass Through
#define	TDF_NewArea		0x0004	// The area has just been created
#define	TDF_DateUsed		0x0008	// date field is valid
#define	TDF_ReceiptRequest	0x0010	// TIC has `ReceiptRequest' line

typedef struct _tagTICDATA {
	USHORT	struct_size;	// Size of this structure.
	USHORT	flags;		// Misc flags.
	PCSZ	areatag;	// Area TAG.
	PCSZ	areapath;	// Area path, terminated by '\'.
	PCSZ	file;		// File name (no path).
	PCSZ	replaces;	// File(s) to be replaced, separated by ' '.
	LONG	size;		// Size of file.
	ULONG	crc;		// CRC32 of file.
	_ADR	from;		// System we received the file from.
	_ADR	origin;		// System that hatched the file.
	PCSZ	pwd;		// Password.
	PCSZ	adesc;		// Description for announcement,
				//   lines separated by '\r'.
	PCSZ	desc;		// "Desc" lines.
	PCSZ	ldesc;		// "LDesc" lines.
	PCSZ	path;		// "Path" lines.
	PCSZ	seenby;		// "Seenby" lines.
	PCSZ	unknown;	// Unrecognized lines.  This field may be
				//   scanned only if struct_size equals
				//   to sizeof (_TICDATA)
	ULONG	date;		// True file modification time
	PCSZ	magics;		// Magic names, separated by ' '.
	PCSZ	areadesc;	// Area description (or empty) line.
} _TICDATA;


APIRET FEATAPIENTRY BeforeNefToss(PCSZ fullname, PCSZ Tag, PCSZ Desc);

#define	BNT_OK		0x00000000	// Normal termination
#define	BNT_BAD		0x00000001	// TIC must be renamed BAD (BeforeNefToss)
					// File will not be hatched (BeforeNefHatch)

#define	BNT_NoForward	0x00000010	// do not forward this file
					//   to other systems
#define	BNT_NoMagic	0x00000020	// do not import magic names
#define	BNT_NoRRQ	0x00000040	// do not generate RRQ answer (BeforeNefToss)
					// do not add ReceiptRequest in TIC (BeforeNefHatch)
#define	BNT_NoReplace	0x00000080	// do not replace files
#define	BNT_NoAnnounce	0x00000100	// do not announce this file
#define	BNT_NoFilesBbs	0x00000200	// do not update Files.Bbs

#define	BNT_LEAVE	0x80000000	// Leave file as is but delete TIC (BeforeNefToss)
					// File will not be hatched (BeforeNefHatch)

// This function is called after NEF has parsed the TIC file, before
// it starts tossing the associated file.

// PCSZ	fullname	Complete name (with path) of the file.
// PCSZ	Tag		Area Tag.
// PCSZ	Desc		Description for the file, as for announcements.
//			Multiple lines are separated by '\r'.
// This function should normally return BNT_OK, with optional
// bitwise BNT_* flags;
// if the file must be processed as BAD, return BNT_BAD.


APIRET FEATAPIENTRY BeforeNefHatch(PCSZ fullname, const _TICDATA *td);

// This function is called after NEF has prepared to hatch the TIC file.

// PCSZ			fullname	Complete name (with path) of the file.
// const _TICDATA *	td		prepared .TIC data
//
// This function should normally return BNT_OK, with optional
// bitwise BNT_* flags;
// if the file must not be processed, return BNT_BAD.


APIRET FEATAPIENTRY AfterNefToss (const _TICDATA *td);

#define	ANT_OK		0x00000000	// normal termination.
#define	ANT_NoFilesBbs	0x00000001	// do not update files.bbs.
#define	ANT_NoAnnounce	0x00000002	// do not announce this file.
#define	ANT_NoMagic	0x00000004	// do not import magic names.

// This function is called after NEF has tossed the file, before
// it updates the files.bbs entry.

// The value returned by this function is to be processed as a group of
// bitwise flags.


///////////////////////////////////////////////////////////////////////
//
// Extensions to the NEF/pk FeatureDLLs
//

		// Commands for feature extensions

#define	FEC_GetExtensionList	0x01
#define	FEC_GetWSelList		0x02
#define	FEC_InWSelList		0x03
#define	FEC_GetFileList		0x04
#define	FEC_InFileList		0x05
#define	FEC_FileCRC32		0x06
#define	FEC_SimpleAnnounce	0x07


		// Extension#1: get extension list

#define	FE_TagMatch	0x00000001	// WSEL/Filespec matching
#define	FE_FileCRC	0x00000002	// calculating of file's CRC32
#define	FE_Announce	0x00000004	// simple announces
#define	FE_BntData	0x00000008	// get _TICDATA from BeforeNefToss

#define	FeatGetExtensionList(prnf,pexts)	\
	prnf(FO_EXT, FEC_GetExtensionList, NULL, pexts)

// Usage:
//
// VOID FEATAPIENTRY FeatGetExtensionList (FeatOut prnf, PULONG pexts);
//
//   Clears in *pexts bits for supported extensions.
//   In: prnf -- callback function passed in Init()
//       pexts -- pointer to ULONG with required FE_* flags is switched on
//   Out: *pexts -- clear FE_* flags for supported features
//
// Example:
//
// APIRET FEATAPIENTRY Init(FeatOut prnf)
// {
//     ULONG reqext = FE_TagMatch | FE_Announce;	// required extensions
//     FeatGetExtensionList(prnf, &reqext);
//     if (reqext) {		// some of features do not supported by NEF
//         prnf(FO_BOTH, '!', "FeatDLL: Use newer version of NEF "
//                            "(extensions %#08lX not supported)\n", reqext);
//         return NFI_ERR;
//     }
//     return NFI_OK;
// }


		// Extension#2: tag matching

typedef	PVOID		FeatTagList;
typedef	FeatTagList *	PFeatTagList;

#define	FeatGetWSelList(prnf,ptaglist,first,last)	\
	prnf(FO_EXT, FEC_GetWSelList, NULL, ptaglist, first, last)
#define	FeatInWSelList(prnf,taglist,tag,result)		\
	prnf(FO_EXT, FEC_InWSelList, tag, taglist, result)
#define	FeatGetFileList(prnf,ptaglist,first,last)	\
	prnf(FO_EXT, FEC_GetFileList, NULL, ptaglist, first, last)
#define	FeatInFileList(prnf,taglist,tag,result)		\
	prnf(FO_EXT, FEC_InFileList, tag, taglist, result)

// Usage:
//
// VOID FEATAPIENTRY FeatGetWSelList(FeatOut prnf, PFeatTagList ptaglist,
//                                   PCCH first, PCCH last);
//
//   Reads <WSEL-list> from [first;last) and stores handle in *ptaglist.
//   In: prnf -- callback function passed in Init()
//       ptaglist -- pointer to the handle, *ptaglist=NULL for first call
//       first -- address of first character in parsed string
//       last -- address of next-after-last character in parsed string
//   Out: *ptaglist -- modified handle
//
// VOID FEATAPIENTRY FeatInWSelList(FeatOut prnf, FeatTagList taglist,
//                                  PCSZ tag, PBOOL result);
//
//   Match area in taglist.
//   In: prnf -- callback function passed in Init()
//       taglist -- created by FeatGetWSelList()
//       tag -- checked area tag
//       result -- default result
//   Out: *result -- TRUE if tag matches taglist, FALSE otherwise
//
// VOID FEATAPIENTRY FeatGetFileList(FeatOut prnf, PFeatTagList ptaglist,
//                                   PCCH first, PCCH last);
//
//   Reads <Filespec-list> from [first;last) and stores handle in *ptaglist.
//   In: prnf -- callback function passed in Init()
//       ptaglist -- pointer to the handle, *ptaglist=NULL for first call
//       first -- address of first character in parsed string
//       last -- address of next-after-last character in parsed string
//   Out: *ptaglist -- modified handle
//
// VOID FEATAPIENTRY FeatInFileList(FeatOut prnf, FeatTagList taglist,
//                                  PCSZ name, PBOOL result);
//
//   Match filename in taglist.
//   In: prnf -- callback function passed in Init()
//       taglist -- created by FeatGetFileList()
//       name -- checked file name
//       result -- default result
//   Out: *result -- TRUE if name matches taglist, FALSE otherwise
//
// Example: disable files.bbs updating in specified areas
//
// #include <string.h>
//
// static FeatOut pf;
// static FeatTagList taglist;
//
// APIRET FEATAPIENTRY Init(FeatOut prnf)
// {
//     pf = prnf;
//     return NFI_OK;
// }
//
// APIRET FEATAPIENTRY ParseCfg(PCSZ clnline)
// {
//     if (strnicmp(clnline, "Areas ", 6) == 0) {
//         clnline += 6;
//         FeatGetWSelList(pf, &taglist, clnline, clnline + strlen(clnline));
//         return NFP_OK;
//     }
//     return NFP_ERR;
// }
//
// APIRET FEATAPIENTRY AfterNefToss(const _TICDATA *td)
// {
//     BOOL res = FALSE;
//     FeatInWSelList(pf, taglist, td->area, &res);
//     if (res) {
//         pf(FO_BOTH, ':', "Do not update Files.Bbs for %s (Area %s)\n",
//            td->file, td->area);
//         return ANT_NoFilesBbs;
//     }
//     return ANT_OK;
// }


		// Extension#3: calculating of file's CRC32

#define	FeatFileCRC32(prnf,fname,crc,result)	\
	prnf(FO_EXT, FEC_FileCRC32, fname, crc, result)

// Usage:
//
// VOID FEATAPIENTRY FeatFileCRC32(FeatOut prnf, PCSZ fname,
//                                 PULONG crc, PBOOL result);
//
//   Calculates CRC32 for specified file.
//   In: prnf -- callback function passed in Init()
//       fname -- file name
//       result -- pointer where result is stored, *result=FALSE
//   Out: *result -- TRUE if CRC32 successfully calculated, FALSE otherwise
//        [only if *result==TRUE] crc -- file's CRC32
//
// Example:
//
// static FeatOut pf;
//
// APIRET FEATAPIENTRY Init(FeatOut prnf)
// {
//     pf = prnf;
//     return NFI_OK;
// }
//
// APIRET FEATAPIENTRY BeforeNefToss(PCSZ fullname, PCSZ Tag, PCSZ Desc)
// {
//     BOOL res = FALSE;
//     ULONG crc;
//     FeatFileCRC32(pf, fullname, &crc, &res);
//     if (!res) {
//         pf(FO_BOTH, '!', "Cannot calculate CRC32, possible file read error"
//                          " or feature extension not supported\n");
//     } else {
//         pf(FO_LOG, '*', "CRC32 %s: %lX\n", fullname, crc);
//     }
//     return BNT_OK;
// }


		// Extension#4: simple announces

#define	FeatAnnounce(prnf,tag,text)	\
	prnf(FO_EXT, FEC_SimpleAnnounce, tag, text)

// Usage:
//
// VOID FEATAPIENTRY FeatAnnounce(FeatOut prnf, PCSZ tag, PCSZ text);
//
//   Make announce in specified area.
//   Announces allowed from BeforeNefToss/BeforeNefHatch/AfterNefToss only.
//   In: prnf -- callback function passed in Init()
//       tag -- announceId.  Use `<name>' with angle brackets for special
//              or user-defined tags.  Always allow to redefine announceId
//       text -- announce text.  Use '\r' for separate different lines
//
// Example: show AreaDesc lines for new areas in <NEW>
//
// #include <stdio.h>
//
// static FeatOut pf;
//
// APIRET FEATAPIENTRY Init(FeatOut prnf)
// {
//     pf = prnf;
//     return NFI_OK;
// }
//
// APIRET FEATAPIENTRY AfterNefToss(const _TICDATA *td)
// {
//     if ((td->flags & TDF_NewArea) && *td->areadesc) {
//         char buf[256];
//         _snprintf(buf, sizeof(buf), "%s: %s\r", td->area, td->areadesc);
//         FeatAnnounce(pf, "<NEW>", buf);
//     }
//     return ANT_OK;
// }


		// Extension#5: get _TICDATA from BeforeNefToss

#define	BeforeNefToss_TICDATA(fullname)	\
	(*(const _TICDATA **)((fullname) - sizeof(void *)))

// Get pointer to _TICDATA structure from BeforeNefToss
//
// Example:
//
// APIRET FEATAPIENTRY Init(FeatOut prnf)
// {
//     ULONG reqext = FE_BntData;
//     FeatGetExtensionList(prnf, &reqext);
//     if (reqext){
//         prnf(FO_BOTH, '!', "FeatDLL: Use newer version of NEF "
//                            "(extensions %#08lX not supported)\n", reqext);
//         return NFI_ERR;
//     }
//     return NFI_OK;
// }
//
// APIRET FEATAPIENTRY BeforeNefToss(PCSZ fullname, PCSZ Tag, PCSZ Desc)
// {
//     const _TICDATA *td = BeforeNefToss_TICDATA(fullname);
//     // use _TICDATA ...
// }


#ifdef	__cplusplus
};
#endif

#pragma	pack ()

#endif
