#ifdef WIN32
    constant
        STDERR = _STDERR_,
        MAXPATH = _MAXPATH_


DLL "<kernel32.dll>"
    integer proc GetShortPathName(string long_fn:cstrval, var string short_fn:strptr, integer n) : "GetShortPathNameA"
    integer proc CreateDirectory(string dir:cstrval, integer sa) : "CreateDirectoryA"
    integer proc RemoveDirectory(string dir:cstrval) : "RemoveDirectoryA"
    integer proc GetLastError() : "GetLastError"
    integer proc WinSearchPath(
            integer lpPath,             // address of search path - set this to 0
            string lpFileName:cstrval,  // address of filename
            integer lpExtension,        // address of extension - set this to 0, make sure lpPath includes extension
            integer nBufferLength,      // size, in characters, of buffer
            var string lpBuffer:strptr, // address of buffer for found filename
            var integer lpFilePart      // address of pointer to file component
   ) : "SearchPathA"
end

string view_cmd[] = " -view "
string delete_cmd[] = " -delete "

/**************************************************************************
  Search the PATH environment variable for a file using the Win32 API
 **************************************************************************/
string proc SearchPathForFile(string fn)
    string path[MAXPATH] = Format("":MAXPATH)
    integer len, ptr_to_fn = 0
    len = WinSearchPath(0, fn, 0, SizeOf(path), path, ptr_to_fn)
    if (len > MAXPATH)
        Warn("Path of ", fn, " is larger than can currently be handled.")
        return ("")
    endif
    return (path[1:len])
end
/*****************************************************************************
  Add this function to the editor
 *****************************************************************************/
integer proc MakeDir(string s)
    string path[_MAXPATH_] = RemoveTrailingSlash(s)
    integer attribute, error

    attribute = FileExists(path)
    if attribute
        return (iif((attribute & _DIRECTORY_), TRUE, MsgBox("Error", Format("Cannot create: ", s, " directory because a file exists with this name."))))
    endif

    if CreateDirectory(path, 0)
        return (TRUE)
    endif
    error = GetLastError()
    return (MsgBox("Error", Format("Error ", error, ". Can not create directory: ", path)))
end

/*****************************************************************************
  Add this function to the editor
 *****************************************************************************/
integer proc RemoveDir(string s)
    string path[_MAXPATH_] = RemoveTrailingSlash(s)
    integer error

    if not FileExists(path) or RemoveDirectory(path)
        return (TRUE)
    endif

    error = GetLastError()
    return (MsgBox("Error", Format("Error ", error, " removing: ", path)))
end

/**************************************************************************
  GetShortPathName only works for existing path's..., so create inpath if
  it does not already exist.
 **************************************************************************/
string proc GetShortFilename(string inpath)
    integer h, len
    string short_path[_MAXPATH_] = Format("":_MAXPATH_:Chr(0))

    h = iif(FileExists(inpath), -1, fCreate(inpath))
    len = GetShortPathName(inpath, short_path, _MAXPATH_)
    if h <> -1
        fClose(h)
        EraseDiskFile(inpath)
    endif
    return (iif(len > 0, short_path[1:Pos(Chr(0), short_path) - 1], inpath))
end


#else
    constant
        STDERR = 2,
        MAXPATH = 80

integer proc fDup2(integer oldhandle, integer newhandle)
    register r

    r.ax = 0x4600
    r.bx = oldhandle
    r.cx = newhandle
    intr(0x21, r)
    return (0)
end

integer proc fDup(integer han)
    register r

    r.ax = 0x4500
    r.bx = han
    intr(0x21, r)
    return (iif(r.flags & _FLCARRY_, -1, r.ax))
end

/****************************************************************************
  Empties the current buffer
  sets the binary mode to reclen
  Loads fn into it
  returns true(non-zero)/false(zero)

  No hooks are called
 ****************************************************************************/
integer proc LoadBuffer(string fn, integer reclen)
    integer ok

    EmptyBuffer()
    BinaryMode(reclen)
    PushBlock()
    ok = InsertFile(fn, _DONT_PROMPT_)
    PopBlock()
    return (ok)
end

/****************************************************************************
  Creates a new temporary buffer of type flags (_SYSTEM_ if not passed)
  sets the binary mode to reclen
  Loads fn into it
  makes it the current buffer
  returns id

  No hooks are called
 ****************************************************************************/
integer proc EditBuffer(string fn, integer flags, integer reclen)
    integer id

    id = CreateBuffer("", iif(flags == 0, _SYSTEM_, flags))
    if id
        LoadBuffer(fn, reclen)
    endif
    return (id)
end

integer proc MsgBox(string title, string msg)
    return (Warn(title, msg))
end

integer proc LongestLineInBuffer()
    integer len = 0

    PushPosition()
    BegFile()
    repeat
        len = Max(len, CurrLineLen())
    until not Down()
    PopPosition()
    return (len)
end

string proc AddTrailingSlash(string path)
    if path[Length(path)] == '\'
        return (path)
    endif
    return (path + '\')
end

#endif

string proc TempPath()
    string path[MAXPATH]

    path = GetEnvStr("TEMP")
    if path == ""
        path = GetEnvStr("TMP")
        if path == ""
            path = CurrDir()
        endif
    endif
    return (path)
end

proc mRepeatFind()
    string find_str[128]

    find_str = GetHistoryStr(_FIND_HISTORY_, 1)
    if find_str == ""
        Find()
    else
        Find(find_str, GetHistoryStr(_FIND_OPTIONS_HISTORY_, 1) + '+')
    endif
end

// application code
constant
    NAME_START = 63

string
    unzip[MAXPATH],
    zip[MAXPATH],
    zipfn[MAXPATH],
    zipview_footer[] = "{Enter}-View {Ctrl U}-Unzip {Del}-Delete",
    viewfile_footer[] = "{Ctrl F}-Find {Ctrl L}-Again {Ctrl S}-SaveAs"

keydef viewfile_keys
<Ctrl F>            Find()
<Ctrl L>            mRepeatFind()
<Ctrl S>            SaveAs()
end

forward keydef zipview_keys
forward proc AfterNoneditCommand()

integer proc DoZipCommand(string cmd, string cmdline, var string tempfn, integer keep_fn)
    integer ok, rc

    tempfn = GetShortFilename(MakeTempname(TempPath()))
    ok = Dos(Format(cmd; cmdline, ">", tempfn), iif(WhichOS() == _WINDOWS_, _RUN_DETACHED_|_START_HIDDEN_, _CREATE_NEW_CONSOLE_|_START_HIDDEN_))
    rc = DOSIOResult()

    if not keep_fn
        EraseDiskFile(tempfn)
    endif
    if not ok or rc <> 0
        return (MsgBox("", "Cannot unzip file"))
    endif
    return (TRUE)
end

integer proc EraseExistingFile(string fn)
    if FileExists(fn)
#ifdef WIN32
        if MsgBox("", "Ok to erase " + ExpandPath(fn), _YES_NO_CANCEL_) <> 1
#else
        if YesNo("Ok to erase " + fn) <> 1
#endif
            return (FALSE)
        elseif not EraseDiskFile(fn)
            MsgBox("", "Cannot erase " + ExpandPath(fn))
            return (FALSE)
        endif
    endif
    return (TRUE)
end

proc GetPathFn(var string path, var string fn)
    string save_wordset[32]
    integer n

    path = ""
    fn = ""
    PushPosition()
    if lFind("Name", "gw$") and Down() and GetText(CurrPos(), 4) == "----"
        n = NumTokens(GetText(1, 255), ' ')
        PopPosition()

        PushPosition()
        BegLine()

        save_wordset = Set(WordSet, ChrSet("[~ ]"))
        WordRight(n - 1)
        Set(WordSet, save_wordset)

        path = Trim(GetText(CurrPos(), 255))
        fn = SplitPath(path, _NAME_|_EXT_)

        PopPosition()
    else
        PopPosition()
    endif
end

#if 0
proc GetPathFn(var string path, var string fn)
    integer start, p, compressfn_col = NAME_START

    p = compressfn_col
    loop
        if p > 0 and CurrChar(p) in 33..41, 43..254 // leave out '*'
            p = p - 1
        else
            p = p + 1
            break
        endif
    endloop

    start = p

    loop
        if CurrChar(p) in 32..254
            p = p + 1
        else
            p = p - 1
            break
        endif
    endloop

    path = Trim(GetText(start, p - start + 1))
    fn = SplitPath(path, _NAME_|_EXT_)
end
#endif

proc ViewFileStartup()
    Unhook(ViewFileStartup)
    Enable(viewfile_keys)
    ListFooter(viewfile_footer)
end

proc ViewFileInZip()
    string temp_dir[MAXPATH], dir[MAXPATH]
    string path[MAXPATH] = "", fn[MAXPATH] = "", temp[MAXPATH] = ""
    integer cur_id, id, tid

    id = 0
    cur_id = GetBufferId()
    dir = CurrDir()
    GetPathFn(path, fn)

    temp_dir = TempPath()
    temp_dir = MakeTempName(temp_dir)

    if temp_dir == "" or not MakeDir(temp_dir) or not ChDir(temp_dir)
        MsgBox("", "Cannot get TEMP path")
        return ()
    endif

    if DoZipCommand(unzip, zipfn + " " + path, temp, TRUE)
        id = EditBuffer(fn, _NORMAL_, 0)
        if NumLines() == 0
            tid = EditBuffer(temp, _NORMAL_, 0)
            EraseDiskFile(temp)
            MsgBoxBuff("Error unzipping file")
            GotoBufferId(id)
            AbandonFile(tid)
        else
            Hook(_LIST_STARTUP_, ViewFileStartup)
            List(path, LongestLineInBuffer() + 4)
            Disable(viewfile_keys)
        endif
        EraseDiskFile(fn)
        ChDir("..")
        RemoveDir(temp_dir)
    endif

    GotoBufferId(cur_id)
    AbandonFile(id)
    ChDir(dir)
end

proc UnzipFile()
    string dest_dir[MAXPATH], dir[MAXPATH]
    string path[MAXPATH] = "", fn[MAXPATH] = "", temp[MAXPATH] = ""
    integer attribute

    dir = CurrDir()
    GetPathFn(path, fn)

    dest_dir = dir
    while Ask("Destination path:", dest_dir)
        attribute = FileExists(dest_dir)
        if attribute == 0
            MsgBox("", "path " + dest_dir + " does not exist")
        elseif (attribute & _DIRECTORY_) == 0
            MsgBox("", "path " + dest_dir + " is not a directory")
        elseif EraseExistingFile(AddTrailingSlash(dest_dir) + fn)
            ChDir(dest_dir)
            DoZipCommand(unzip, zipfn + " " + path, temp, 0)
            ChDir(dir)
            break
        endif
    endwhile
end

proc DelFileFromZip()
    string path[MAXPATH] = "", fn[MAXPATH] = "", temp[MAXPATH] = ""

    GetPathFn(path, fn)
#ifdef WIN32
    if MsgBox("", "Delete " + path + " from archive?", _YES_NO_CANCEL_) == 1
#else
    if YesNo("Delete " + path + " from archive?") == 1
#endif
        if DoZipCommand(zip, delete_cmd + zipfn + " " + path, temp, 0)
            KillLine()
        endif
    endif
end

proc HouseKeep()
    Disable(zipview_keys)
    Unhook(AfterNoneditCommand)
end

proc CleanUp()
    Enable(zipview_keys)
    Hook(_AFTER_NONEDIT_COMMAND_, AfterNoneditCommand)
end

keydef zipview_keys
<Enter>             HouseKeep()     ViewFileInZip()     CleanUp()
<Ctrl V>            HouseKeep()     ViewFileInZip()     CleanUp()

<Ctrl U>            HouseKeep()     UnzipFile()         CleanUp()

<Del>               HouseKeep()     DelFileFromZip()    CleanUp()
<Ctrl D>            HouseKeep()     DelFileFromZip()    CleanUp()
end

proc AfterNoneditCommand()
    if CurrLine() < 3
        GotoLine(3)
    elseif CurrLine() >= NumLines() - 1
        GotoLine(NumLines() - 2)
    endif
end

proc ListStartup()
    Unhook(ListStartup)
    Enable(zipview_keys)
    GotoLine(3)
    ListFooter(zipview_footer)
end

proc main()
    integer attribute
    string fn[MAXPATH] = ""

    if unzip == ""
        MsgBox("", "(pk)unzip.exe not found!")
        return ()
    endif

    zipfn = Trim(Query(MacroCmdLine))
    attribute = FileExists(zipfn)
    if attribute == 0 or (attribute & (_VOLUME_|_DIRECTORY_))
        return ()
    endif
    zipfn = GetShortFileName(ExpandPath(zipfn))

    if not DoZipCommand(unzip, view_cmd + zipfn, fn, TRUE)
        EraseDiskFile(fn)
        return ()
    endif

    EditBuffer(fn, _NORMAL_, 0)
    /*
    PushBlock()
    MarkLine()
    ok = lFind(" Length ", "^")
    if ok
        Up()
        KillBlock()
        FileChanged(FALSE)
    endif
    PopBlock()
    */
    EraseDiskFile(fn)
    /*
    if not ok
        AbandonFile()
        MsgBox(Format(unzip; view_cmd), "Invalid zip format")
        return ()
    endif
    */

    Hook(_LIST_STARTUP_, ListStartup)
    Hook(_AFTER_NONEDIT_COMMAND_, AfterNoneditCommand)
    List(zipfn, LongestLineInBuffer())
    Disable(zipview_keys)
    UnHook(AfterNoneditCommand)

    AbandonFile()
end

proc WhenLoaded()
    // zip32 handles both zip and unzip
    unzip = SearchPathForFile("zip32.exe")
    if unzip <> ""
        zip = unzip
    else
        if unzip == ""
            view_cmd = " -v "
            delete_cmd = " -v "
        endif
        if unzip == ""
            unzip = SearchPathForFile("unzip.exe")
        endif
        if unzip == ""
            unzip = SearchPathForFile("pkunzip.exe")
        endif

        zip = SearchPathForFile("zip.exe")
        if zip == ""
            zip = SearchPathForFile("pkzip.exe")
        endif
    endif
end
