/*
  Version: 31 Dec 2023 (2)


  DOCUMENTATION

  This tool sorts those lines in the current buffer that overlap with a marked
  block.

  If the block is a column block, then the column is the sort key, otherwise
  the whole line is the sort key.

  Column sorts are stable, meaning that for lines with the same sort key their
  relative positions remain the same.
  This allows us to sort on multiple columns, one at a time.

  This tool can be called manually from TSE's Macro Execute menu with
    sort [option] ...
  or from a macro with
    ExecMacro("sort [option] ...")

  Options:
    -i          Ignore case. A letter being upper or lower case sorts the same.
    -d          Descending. Sort on the sort key in reverse order.
    -k          Kill duplicates. Remove duplicate lines after sorting.

    --internal  Requires the sort tool to use TSE's internal Sort() command.
    --external  requires the sort tool to use TSE's external tsort.com tool.

  Options are case-insensitive.

  The "--internal" and "--external" options are for testing the sort tool.
  They should not be used for regular use.

  Example:
    sort -i -k  Sort the marked lines, ignoring case and removing duplicates.

*/





/****************************************************************************

 HISTORY

 Sort a marked block, calling either the internal (slow) sort on small
 blocks, and the external (fast) sort on large blocks.

 08 May 2019 SEM - check MsgLevel before outputting "Finished" message.

 04 Nov 2011 SEM - bump sort threshold to 6000 (machines are faster now).

 06 Dec 2010 SEM - bump sort threshold to 4000 (machines are faster now).

 28 Feb 2009 SEM - Fix problem of not finding tsort.com in certain cases where
           TSELOADDIR was used. Reported (along with a suggested fix) by Ross
           Barnes.

 Oct 24 2006 SEM: better error message on failure.

 2-04-2000 SEM: tsort.com is an old dos-based program.  Maximum command line
           that it supports is 128 characters.  This can be a problem, since
           tsort requires an input and output filename.  A solution is to
           switch to the drive/directory where the sorting will occur, and
           then use relative filenames.

11-16-99   SEM: Use either temp drive or drive file to be sorted is on as
           sort work drive, depending on which one has the most free disk
           space.

 2-10-99   SEM: Replaced OS checking with new built-in command. Made error
           messages a little prettier.

 8-21-98   SEM: Set the sorts' temporary file drive to the same as the drive
           where the editor writes the file to be sorted.  This is the
           SwapPath editor variable for the DOS version, and the TEMP
           environment variable path for the Win32 version.

 8-17-98   SEM: Bug!  The ignore-case option was being set the opposite of
           what it should be.

 3-27-98   SEM: Quote the path of the sort executable, if needed.

 2-05-98   SEM: Add "kill duplicates" option, and accept two different
           command line formats.  New format is:

           sort -i[gnorecase] -d[escending] -k[illdups]

                    or

           sort integer_sort_flags
                where flags can be _DESCENDING_ and/or _IGNORE_CASE_

           If no options passed, then sort will be ascending, respecting case.

 1-08-98   SEM: Tsort fails when the temp path is a long filename. Since
           tsort is a 16 program that does not handle long filenames, use
           GetShortPathName() to convert long names to short names before
           passing them to tsort.

 9-39-97   SEM: check to see if we are Windows NT or Windows 95.  If NT,
           call the sort via Dos(), if 95 call the sort via lDos(). Note
           that the options for the Dos() command to keep the window from
           showing are very particular, it took quite a bit of testing to
           come up with the right combination.

 7-30-97   SEM: win32 CreateProcess() doesn't always work with 16-bit
           executables.  In lieu of that, go back to calling Dos.  However,
           using the proper options, the screen doesn't get resized anymore!

 1-09-96   SEM: Always erase the block from disk (even if sort fails)

12-13-96   SEM: Use lDos in lieu of Dos to keep screen from resizing

12-18-95   SEM: Reset blocktype even if an error occurs.

           Call internal (slow) sort if:

           -lines to sort is < sort_thresh
           -file is in binary mode
           -block contains tabs or cr/lf characters
           -external sort module can not be found

04-27-20   SEM: Remove explicit Win95/98 support

09-30-2022 SEM: using CompareLines() from the ignore-case on kill dupes.
            Pass the flags in, and respect ignore case.

31 Dec 2023, Carlo:
    Note:
        I did use Semware's programming standards and left the old source code
        (structure) as much the same as possible, to make it both as easy as
        possible to compare to the old source code, as well as and to be
        maintained by Semware.
        I necessarily forewent opportunities to optimize the source code.
        That said, there were a lot of changes.
    Deleted debug messages.
        For example, users do not need a "progress" message for a find
        command that takes a fraction of a second.
    No longer perform known-to-fail internal sorts.
        Both the internal and external sort have limitations, and if the
        limitations of one are exceeded this macro can try to switch to the
        other.
        However, if the limitations of both are exceeded, then the macro no
        longer asks a question to proceed anyway, but stops with an error and
        reports the crossed boundaries of both.
        The decision for this change was easy to make because:
        - The old sort did sorts that failed without reporting an error.
        - The internal sort no longer has a hard number-of-lines limit,
          though it does become unbearably slow at some point.
        - In TSE 4.50 rc 15 Semware increased the maximum sort key size of the
          internal sort to 16,000.
    Fixed the macro's inconsistency in handling horizontal tab characters:
        Both the internal Sort() function and the external sort programs sort
        buffers with horizontal tab characters without expanding the tabs.
        However, to use an external sort program, the macro must save and load
        the buffer to and from a file, for which TSE's tab settings did apply.
        The sort macro was ignoring this impact of TSE's tab settings.
        Hidden from the user a buffer could be sorted differently based on
        whether it exceeded the internal Sort() function's limitations or not.
        The chosen solution is to temporarily set TSE's configuration options
        EntabOnSave and DetabOnLoad to Off around an external sort.
        Now the internal and external sorts function the same, not expanding
        tab characters for sorting.
    No longer provide a "/D" parameter to the external sort.
        This was done when the caller requested to remove duplicate lines.
        This parameter was probably for future tsort.com use, but currently
        tsort.com cannot remove duplicate lines.
        Duplicate lines are removed by this sort.s macro
    Removed the _RUN_DETACHED_ option from lDos().
        Maybe this stems from a long since resolved bug in lDos().
        Logically it makes no sense to start a sort in TSE's background
        and immediately try to load its result.
        Likewise it does not make sense to expect a useful return code
        from a detached process.
        Unsurprisingly sort.s therefore went out of its way to detect
        errors itself and partially make up its own return codes, leaving
        us with a bit of a mess.
    Rewrote the error handling around the external sort.
        For example, sort.s overused the same field to store errors from
        many error sources, making the code hard to read.
        Now there are 6 separate error fields, one for each error source,
        making it clear which error sources are used at each step.
    Bug fixed:
        If initially no block is defined but the user confirmed "Sort
        entire file?", then no longer call the internal sort for a
        buffer that exceeds the internal sort's known thresholds.
    Bug in tsort.com worked around for TSE versions < v4.50 rc 14 (23 Nov 2023):
        tsort.com had an undocumented default sort key length of 4095
        bytes, which made sort.s fail if it called tsort.com for a line
        block with longer lines.
        This tsort.com flaw was fixed in TSE 4.50 rc 14.
    Fixed inconsistently not calling PurgeMacro():
        Added the inconsistently missing PurgeMacro() statements.
    Removed using tsort.com's obsolete /T and /Z parameters.
    Parameters "--internal" and "--external" were added for testing.
        For testing it is necessary to be able to explicitly call the internal
        or external sort.
        These two parameters are interpreted as a requirement: No fallback to
        the other sort tool happens, but an error is reported if the required
        tool has or would have a problem.
    Sorting an entire file now uses internal sort when possible.
        A MarkAll()'s block's BlockEndCol is MAXLINELEN (30,000 since TSE
        v4.40.29).
        The internal sort's maximum column width capability is 283.
        This meant the internal sort was never used for sorting a whole buffer,
        even if it was a tiny buffer.
        Now the minimum of BlockEndCol and LongestLineInBuffer() is used
        instead to determine the right-most column.
    In Linux this sort macro now calls tse/tsort.com if present.
        Otherwise it calls the internal Sort(), even when this will be slower.
        In sorting "stability" is a term for a defined capability:
            https://en.wikipedia.org/wiki/Sorting_algorithm#Stability
        The latest tsort.com in Linux TSE 4.50 rc 16 is stable!
    For tsort.com now "-" options instead of "/" options are used.
        The Linux tsort.com does not understand "/" options, and it made sense
        to make calls to the Linux and Windows tsort.com the same.
    Hard-coded the requirement for TSE v4.50 rc 16 upwards.
        Because the beta is not distributed with TSE, a careless user could
        download it and try it with lower TSE version that it is not compatible
        with, possibly leading to an irrelevant bug report.
    Documented the sort tool.
      See the top of this file.


    TODO
        Beta testing.

 ****************************************************************************/





/*

T E C H N I C A L   B A C K G R O U N D   I N F O


LINUX OWN SORT

  At one point the command line sort of Linux was used in Debian and Centos.
  This worked well except for the sort being "unstable".
    https://en.wikipedia.org/wiki/Sorting_algorithm#Stability
  As Semware released a stable Linux tsort.com with TSE v4.50 rc 16,
  using the Linux sort became moot.

*/





// Compatibility check.

#ifndef INTERNAL_VERSION
    #define INTERNAL_VERSION 0
#endif
#if INTERNAL_VERSION < 12378

    This Version of the macro requires TSE v4.50 rc 16 or higher.

#endif



// return codes:

constant
    ERROR_SAVEBLOCK = 0ff00h,   // SaveBlock fails
    ERROR_SPAWN     = 0fe00h,   // if the Dos/lDos command fails
    ERROR_GONE      = 0fc00h,   // everything looks ok, but sorted file does not exist
    ERROR_DISK_FULL = 00036h    // out of disk space

#define MAXPATH 255

integer external_sort_required    = FALSE
integer internal_sort_required    = FALSE
string  MACRO_NAME [MAXSTRINGLEN] = ''


#ifndef LINUX
    dll "<kernel32.dll>"
        integer proc GetLastError() : "GetLastError"
        integer proc GetDiskFreeSpace(string root_dir:cstrval,
                        var integer SectorsPerCluster,
                        var integer BytesPerSector,
                        var integer NumberOfFreeClusters,
                        var integer TotalNumberOfClusters) : "GetDiskFreeSpaceA"
    end

    integer proc mGetDiskFreeSpace(string p)
        integer sectors_per_cluster, bytes_per_sector, free_clusters, total_clusters

        if GetDiskFreeSpace(p[1:3], sectors_per_cluster, bytes_per_sector, free_clusters, total_clusters)
            return (bytes_per_sector * sectors_per_cluster * free_clusters)
        endif
        return (0)
    end

    string proc MostFreeSpace(string p1, string p2)
        integer
            F1 = mGetDiskFreeSpace(p1),
            F2 = mGetDiskFreeSpace(p2)

        //  both >= 0
        if F1 >= 0 and F2 >= 0
            return (iif(F1 >= F2, p1, p2))
        endif

        //  both < 0,
        if F1 < 0 and F2 < 0
            return (iif(F1 <= F2, p1, p2))
        endif

        // 1 is < 0
        return (iif(F1 < 0, p1, p2))
    end
#endif


proc mWarn(string msg)
    MsgBox("Error!", msg)
end

proc mMessage(string msg)
    if Query(MsgLevel) == _ALL_MESSAGES_
        Message(msg)
    endif
end

integer proc remove_dups(integer Flags, integer doit, integer line1, integer line2)
    integer n = 0, deleted = 0, ignore_case = 0

    if not doit or line1 >= line2
        return (0)
    endif

    PushPosition()

    GotoLine(line1)
    if Flags & _IGNORE_CASE_
        ignore_case = 1
    endif

    mMessage("Removing duplicates")
    while CurrLine() < line2 and CurrLine() < NumLines()
        if CompareLines(CurrLine(), CurrLine() + 1, ignore_case) == 0
            Down()
            deleted = deleted + 1
            if not KillLine()
                break
            endif
            Up()
        elseif not Down()
            break
        endif
        n = n + 1
        if n mod 100000 == 0
            KeyPressed()
            Message("Removing dupes: ", CurrLine(), ' of ', NumLines())
        endif
    endwhile

    PopPosition()
    return (deleted)
end

/**************************************************************************
  Decode the passed options - can either be a binary integer, or a flag in
  the format -option

  Return TRUE if the -killdups option is present
 **************************************************************************/
integer proc GetOptions(string cmdline, var integer Flags, var string options)
    integer i, n, z, kill_dups, descending, ignore_case
    string s[32]

    kill_dups = FALSE
    descending = FALSE
    ignore_case = FALSE
    n = NumTokens(cmdline, ' ')
    for i = 1 to n
        s = GetToken(cmdline, ' ', i)
        case Lower(s[1:2])
            when '-i'
                ignore_case = TRUE
            when '-d'
                descending = TRUE
            when '-k'
                kill_dups = TRUE
            otherwise
                if     EquiStr(s, '--internal')
                  internal_sort_required = TRUE
                elseif EquiStr(s, '--external')
                  external_sort_required = TRUE
                else
                  z = Val(s)
                  if z & _IGNORE_CASE_
                      ignore_case = TRUE
                  endif
                  if z & _DESCENDING_
                      descending = TRUE
                  endif
                endif
        endcase
    endfor

    if ignore_case
        Flags = Flags | _IGNORE_CASE_
    else
        options = options + "-a "
    endif

    if descending
        options = options + "-r "
        Flags = Flags | _DESCENDING_
    endif

    return (kill_dups)
end

proc mGotoLineCol(integer line, integer col)
    GotoLine(line)
    GotoColumn(col)
end

integer proc ChangeDriveDir(string dir)
    #ifdef LINUX
        return (ChDir(dir))
    #else
        return (LogDrive(dir[1]) and Upper(GetDrive()) == Upper(dir[1]) and ChDir(dir))
    #endif
end

proc mMarkColumn(integer Y1, integer X1, integer y2, integer x2)
    UnMarkBlock()
    mGotoLineCol(Y1, X1)
    MarkColumn()
    mGotoLineCol(y2, x2)
    MarkColumn()
end

/**************************************************************************
  Call the internal (slow) sort.

  Do not call it if the kill_dups option is set.
 **************************************************************************/
proc BuiltInSort(integer Flags, integer kill_dups, string msg, integer Y1, integer y2)
    integer result, deleted

    if msg == ""
        result = 1
    else
        result = MsgBox("", msg + " Use slow sort?", _YES_NO_CANCEL_)
    endif

    if result == 1
        Sort(Flags)
        deleted = remove_dups(Flags, kill_dups, Y1, y2)
        if deleted > 0
            mMessage(Format(deleted; "duplicates removed"))
        else
            mMessage("Finished")
        endif
    else
        Message("Sort terminated")
    endif
end

#ifndef INTERNAL_VERSION
    #define INTERNAL_VERSION 0
#endif

#if INTERNAL_VERSION > 12375  // TSE 4.50 rc 14, 23 Nov 2023.
    #define SORT_COLUMNS_THRESHOLD 16000
#else
    #define SORT_COLUMNS_THRESHOLD   283
#endif

//  Note:
//    This is not a hard threshold.
//    The internal sort works for all numbers of lines that TSE can hold,
//    (source: Semware) but becomes unbearably slow for large buffers.
//    This threshold makes TSE switch to the external sort for large buffers.
#define SORT_LINES_THRESHOLD 6000

#define INTERNAL_SORT        1
#define EXTERNAL_SORT        2

string  sort_path[MAXPATH]

string fn[MAXPATH], efn[MAXPATH], ofn[MAXPATH], rfn[MAXPATH], options[255], sort_cmdline[255]
string dir[MAXPATH], save_dir[MAXPATH], drive[1]

#ifndef LINUX
    string t_dir[MAXPATH]
#endif

proc Main()
    integer X1, Y1, x2, y2, b, line, col, row, Marking, save_eoltype,
        Flags, save_eoftype, kill_dups, deleted

    integer tsort_error_buf           = 0
    integer stderr_buf                = 0
    string  external_sort_reason [20] = ''
    string  internal_sort_reason [20] = ''
    integer old_DetabOnLoad           = 0
    integer old_EntabOnSave           = 0
    integer sort_tool                 = _NONE_

    // Error source variables
    integer dos_io_result                = 0
    integer dos_ok                       = TRUE
    integer macro_error_code             = 0
    integer os_error_code                = 0
    integer tsort_error_buffer_not_empty = FALSE
    integer stderr_buffer_not_empty      = FALSE

    #ifdef LINUX
      string os [5] = 'Linux'
    #else
      string os [7] = 'Windows'
    #endif

    options = ""
    Flags = 0

    PushBlock()

    kill_dups = GetOptions(Trim(Query(MacroCmdLine)), Flags, options)
    b = isBlockInCurrFile()
    if b == 0
        if MsgBox("", "Sort entire file?", _YES_NO_CANCEL_) == 1
            MarkAll()
            b = isBlockInCurrFile()
        endif
    endif
    Y1 = Query(BlockBegLine)
    y2 = Query(BlockEndLine)
    X1 = Query(BlockBegCol)
    x2 = Min(Query(BlockEndCol), LongestLineInBuffer())

    // Make initial sort tool choice
    if external_sort_required
        external_sort_reason = '--external parameter'
        if internal_sort_required
            internal_sort_reason = '--internal parameter'
            sort_tool            = _NONE_
        else
            sort_tool = EXTERNAL_SORT
        endif
    else
        if internal_sort_required
            internal_sort_reason = '--internal parameter'
            sort_tool            = INTERNAL_SORT
        else
            PushPosition()
            if BinaryMode()
                internal_sort_required = TRUE
                internal_sort_reason   = 'binary buffer'
                sort_tool              = INTERNAL_SORT
            elseif lFind("[\n\r]", "glx")
                internal_sort_required = TRUE
                internal_sort_reason   = 'control characters'
                sort_tool              = INTERNAL_SORT
            elseif b
            and    y2 - Y1 + 1 <  SORT_LINES_THRESHOLD
            and    x2 - X1 + 1 <= SORT_COLUMNS_THRESHOLD
                sort_tool              = INTERNAL_SORT
            else
                sort_tool              = EXTERNAL_SORT
            endif
            PopPosition()
        endif
    endif

    // Does the external sort need to fall back to the internal sort?
    if  not internal_sort_required
    and sort_tool         == EXTERNAL_SORT
    and Length(sort_path) == 0
        internal_sort_required = TRUE
        internal_sort_reason   = 'no tsort.com'
        sort_tool              = INTERNAL_SORT
    endif

    // Does the internal sort need to fall back to the external sort?
    if  not external_sort_required
    and sort_tool   == INTERNAL_SORT
    and x2 - X1 + 1 >  SORT_COLUMNS_THRESHOLD
        external_sort_required = TRUE
        external_sort_reason   = 'key length > ' + Str(SORT_COLUMNS_THRESHOLD)
        sort_tool = EXTERNAL_SORT
    endif

    // Final sort tool decision.
    if  internal_sort_required
    and external_sort_required
        PopBlock()
        Warn(MACRO_NAME;
             'conflict: Neither internal nor external sort is possible',
             Chr(13), 'because of "',
             internal_sort_reason , '" versus "', external_sort_reason, '".')
        PurgeMacro(CurrMacroFilename())
        return ()
    elseif sort_tool == INTERNAL_SORT
        BuiltInSort(Flags, kill_dups, "", Y1, y2)
        PopBlock()
        PurgeMacro(CurrMacroFilename())
        return ()
    endif

    // Do external sort.

    // force eoltype
    #ifdef LINUX
        // This was necessary for Linux sort.
        // I'll leave it in for tsort.com for slightly smaller temp files.
        save_eoltype = Set(EOLType, 2)
    #else
        save_eoltype = Set(EOLType, 3)
    #endif
    save_eoftype = Set(EOFType, 2)

    Marking = Query(Marking)
    line = CurrLine()
    col = CurrCol()
    row = CurrRow()

    if b == _LINE_
        #ifndef LINUX
            #if INTERNAL_VERSION < 12375
                //  Bug in TSE versions < 4.50 rc 14, 23 Nov 2023:
                //    tsort limited its sort to 4095 characters if a line block
                //    was specified and it received no column parameters.
                options = Format(options, "-+", 1, ":", LongestLineInBuffer())
            #endif
        #endif
    else
        if b == _COLUMN_
            options = Format(options, "-+", X1, ":", x2 - X1 + 1)
        endif
        MarkLine(Y1, y2)
    endif
    GotoBlockBegin()
    BegLine()

    dir = GetTempPath()

    dir = SplitPath(ExpandPath(dir), _DRIVE_|_PATH_)
    #ifndef LINUX
        t_dir = dir
        dir = MostFreeSpace(t_dir, SplitPath(CurrFilename(), _DRIVE_|_PATH_))
        drive = dir[1]
    #endif

    save_dir = CurrDir()
    #ifdef LINUX
        ChangeDriveDir(dir)
    #else
        if (Upper(dir[1]) in 'A'..'Z') and ChangeDriveDir(dir)
            dir = ".\"
        else
            save_dir = ""
        endif
    #endif

    fn  = SplitPath(MakeTempName(dir), _NAME_|_EXT_)
    efn = SplitPath(MakeTempName(dir), _NAME_|_EXT_)
    ofn = SplitPath(MakeTempName(dir), _NAME_|_EXT_)
    rfn = SplitPath(MakeTempName(dir), _NAME_|_EXT_)

    fn  = GetShortPath(fn)
    efn = GetShortPath(efn)
    ofn = GetShortPath(ofn)
    rfn = GetShortPath(rfn)

    /**************************************************************************
      Save the block to be sorted in the temp directory
     **************************************************************************/
    if SaveBlock(fn, _OVERWRITE_)
        // get the free space again, after the file has been saved
        #ifndef LINUX
            dir = MostFreeSpace(t_dir, SplitPath(CurrFilename(), _DRIVE_|_PATH_))
            drive = dir[1]
        #endif

        options = Trim(options)

        #ifdef LINUX
            sort_cmdline = Format(sort_path; fn; fn; '-q -e', efn; options;
                                  '>', ofn; '2>', rfn)
        #else
            sort_cmdline = Format("-q -e", efn; fn; fn; options)
        #endif

        old_DetabOnLoad = Set(DetabOnLoad, OFF)
        old_EntabOnSave = Set(EntabOnSave, OFF)

        mMessage("Calling external sort...")

        #ifdef LINUX
            dos_ok = Dos(sort_cmdline,
                         _DONT_CLEAR_|_DONT_PROMPT_|_RETURN_CODE_|_START_HIDDEN_)
        #else
            dos_ok = lDos(sort_path, sort_cmdline,
                          _DONT_CLEAR_|_DONT_PROMPT_|_RETURN_CODE_|_START_HIDDEN_)
        #endif

        dos_io_result = DosIOResult()

        Set(DetabOnLoad, old_DetabOnLoad)
        Set(EntabOnSave, old_EntabOnSave)

        #ifndef LINUX
            if not dos_ok
            or     dos_io_result <> 0
                os_error_code = GetLastError()
            endif
        #endif

        if not FileExists(fn)
            macro_error_code = ERROR_GONE
        endif

        if  dos_ok
        and dos_io_result    == 0
        and macro_error_code == 0
        and os_error_code    == 0
            KillBlock()
            InsertFile(fn)
        endif

        PushPosition()
        tsort_error_buf = EditBuffer(efn)
        if NumLines() > 1 or CurrLineLen() > 0
            tsort_error_buffer_not_empty = TRUE
        endif
        stderr_buf = EditBuffer(rfn)
        if NumLines() > 1 // tsort.com always writes a version line to STDERR.
            stderr_buffer_not_empty = TRUE
        endif
        PopPosition()

        EraseDiskFile(ofn)
        EraseDiskFile(efn)
        EraseDiskFile(rfn)
        EraseDiskFile(fn)

        if save_dir <> ""
            ChangeDriveDir(save_dir)
        endif
    else
        macro_error_code = ERROR_SAVEBLOCK
    endif

    if b <> _COLUMN_
        MarkLine(Y1, y2)
    elseif not Marking
        MarkColumn(Y1, X1, y2, x2)
    elseif line == Y1 and col == x2
        mMarkColumn(y2, X1, Y1, x2)
    elseif line == y2 and col == X1
        mMarkColumn(Y1, x2, y2, X1)
    else
        MarkColumn(Y1, X1, y2, x2)
    endif

    GotoLine(line)
    GotoColumn(col)
    ScrollToRow(row)
    Set(Marking, Marking)

    if      macro_error_code == ERROR_SAVEBLOCK
        mWarn(Format("Error on SaveBlock(", fn, ")"))
    elseif  macro_error_code == ERROR_GONE
        mWarn(Format("Error, sorted file disappeared!"))
    elseif  dos_io_result    == ERROR_DISK_FULL
        mWarn(Format("Sort ran out of disk space on drive ", drive, ":"))
    elseif not dos_ok
    or         dos_io_result    <> 0
    or         macro_error_code <> 0
    or         os_error_code    <> 0
    or         tsort_error_buffer_not_empty
        mWarn(Format('External sort error(code)s:';
                     'Dos():'; iif(dos_ok, 'OK', 'NOT OK'),
                     ', program:'; HiByte(dos_io_result),
                     ', errorlevel:'; LoByte(dos_io_result),
                     ', ', os,': 0x', os_error_code:4:'0':16,
                     ', tsort error buffer:'; iif(tsort_error_buffer_not_empty,
                                                  'NOT EMPTY', 'EMPTY'),
                     ', STDERR:'; iif(tsort_error_buffer_not_empty,
                                      'NOT EMPTY', 'EMPTY'),
                     '.'))
        #ifdef LINUX
            mWarn(Format('CmdLine:'; sort_cmdline))
        #else
            mWarn(Format("CmdLine:"; sort_path; sort_cmdline))
        #endif
    endif

    if tsort_error_buffer_not_empty
        MsgBoxBuff("Sort error", _OK_, tsort_error_buf)
    endif

    if stderr_buffer_not_empty
        MsgBoxBuff("Sort error", _OK_, stderr_buf)
    endif

    if      dos_ok
    and     dos_io_result    == 0
    and     macro_error_code == 0
    and     os_error_code    == 0
    and not tsort_error_buffer_not_empty
    and not stderr_buffer_not_empty
        deleted = remove_dups(Flags, kill_dups, Y1, y2)
        if deleted > 0
            mMessage(Format(deleted; "duplicates removed"))
        else
            mMessage("Finished")
        endif
    endif

    AbandonFile(tsort_error_buf)
    Set(EOLType, save_eoltype)
    Set(EOFType, save_eoftype)
    PurgeMacro(CurrMacroFilename())
    PopBlock()
end Main

/**************************************************************************
  See if the specified pgm exists in the actual (LoadDir(1)) or the
  (possibly redirected) LoadDir().
 **************************************************************************/
proc find_editor_program(string fn, var string path)
    path = SplitPath(LoadDir(1), _DRIVE_|_PATH_) + fn
    if not FileExists(path)
        path = LoadDir() + fn
        if not FileExists(path)
            path = ""
        endif
    endif
    path = QuotePath(path)
end

proc WhenLoaded()
    MACRO_NAME = SplitPath(CurrMacroFilename(), _NAME_)
    find_editor_program("tsort.com", sort_path)
end

