/*------------------------------------------------------------------------
  13 Mar 2021 Bug: OEM chars in the pallette list are not displayed correctly
        when using ANSI fonts.  Fixed.  Reported by Carlo Hogeveen
 ------------------------------------------------------------------------*/

/*
https://alumni.media.mit.edu/~wad/color/numbers.html

Black         RGB(  0,   0,   0)
Blue          RGB( 42,  75, 215)
Green         RGB( 29, 105,  20)
Cyan          RGB( 41, 208, 208)
Red           RGB(173,  35,  35)
Purple x      RGB( 129, 38, 192)
Brown         RGB(129,  74,  25)
Lt. Gray      RGB(160, 160, 160)
Dk. Gray      RGB( 87,  87,  87)
Lt. Blue      RGB(157, 175, 255)
Lt. Green     RGB(129, 197, 122)
Lt. Cyan      RGB(233, 222, 187) (tan)
Lt. Red       RGB(255, 205, 243) (pink)
Lt. Magenta   RGB(255, 146,  51) (orange)
Yellow        RGB(255, 238,  51)
White         RGB(255, 255, 255)

Orange x
Tan
Pink   x

Magenta       purple    RGB(129,  38, 192)
Light Cyan    tan       RGB(233, 222, 187)
Light Red     pink      RGB(255, 205, 243)
Light Magenta orange    RGB(255, 146,  51)

https://sashamaps.net/docs/resources/20-colors/

Black    0    0    0
Navy     0    0    128
Green    60   180  75
Teal     0    128  128
Maroon   128  0    0
Purple   145  30   180
Olive    128  128  0

Blue     0    130  200
Lime     210  245  60
Cyan     70   240  240
Red      230  25   75
Magenta  240  50   230
Yellow   255  225  25
White    255  255  255

Brown    170  110  40
Orange   245  130  48
Pink     250  190  212
Apricot  255  215  180
Beige    255  250  200
Mint     170  255  195
Lavender 220  190  255

https://davidmathlogic.com/colorblind/#%23D81B60-%231E88E5-%23FFC107-%23004D40
HEX #FFC20A R 255  G 194  B 10
HEX #0C7BDC R 12  G 123  B 220
HEX #994F00 R 153  G 79  B 0
HEX #006CD1 R 0  G 108  B 209
HEX #E1BE6A R 225  G 190  B 106
HEX #40B0A6 R 64  G 176  B 166
HEX #E66100 R 230  G 97  B 0
HEX #5D3A9B R 93  G 58  B 155
HEX #1AFF1A R 26  G 255  B 26
HEX #4B0092 R 75  G 0  B 146
HEX #FEFE62 R 254  G 254  B 98
HEX #D35FB7 R 211  G 95  B 183
HEX #005AB5 R 0  G 90  B 181
HEX #DC3220 R 220  G 50  B 32
HEX #1A85FF R 26  G 133  B 255
HEX #D41159 R 212  G 17  B 89
A number of color palettes have been developed with the intention of being accessible to people who are colorblind. Three of them appear below, from

https://lospec.com/palette-list/tag/16

From:https://en.wikipedia.org/wiki/ANSI_escape_code
                    vga                 xp console          powershell          vscode              win 10 console      terminal            putty               mirc                xterm               ubuntu              eclipse
Black               0,0,0                                                                         12,12,12          0,0,0                                                                         1,1,1             0,0,0
Red                 170,0,0           128,0,0                               205, 49, 49         197,15,31         194,54,33         187,0,0           127,0,0           205,0,0           222,56,43         205,0,0
Green               0,170,0           0,128,0                               13, 188, 121        19,161,14         37,188,36         0,187,0           0,147,0           0,205,0           57,181,74         0,205,0
Yellow              170,85,0[h]       128,128,0         238,237,240       229, 229, 16        193,156,0         173,173,39        187,187,0         252,127,0         205,205,0         255,199,6         205,205,0
Blue                0,0,170           0,0,128                               36, 114, 200        0,55,218          73,46,225         0,0,187           0,0,127           0,0,238[47]       0,111,184         0,0,238
Magenta             170,0,170         128,0,128         1,36,86           188, 63, 188        136,23,152        211,56,211        187,0,187         156,0,156         205,0,205         118,38,113        205,0,205
Cyan                0,170,170         0,128,128                             17, 168, 205        58,150,221        51,187,200        0,187,187         0,147,147         0,205,205         44,181,233        0,205,205
White               170,170,170       192,192,192                           229, 229, 229       204,204,204       203,204,205       187,187,187       210,210,210       229,229,229       204,204,204       229,229,229
Bright Black (Gray) 85,85,85          128,128,128                           102, 102, 102       118,118,118       129,131,131       85,85,85          127,127,127       127,127,127       128,128,128       0,0,0
Bright Red          255,85,85         255,0,0                               241, 76, 76         231,72,86         252,57,31         255,85,85         255,0,0
Bright Green        85,255,85         0,255,0                               35, 209, 139        22,198,12         49,231,34         85,255,85         0,252,0           0,255,0
Bright Yellow       255,255,85        255,255,0                             245, 245, 67        249,241,165       234,236,35        255,255,85        255,255,0
Bright Blue         85,85,255         0,0,255                               59, 142, 234        59,120,255        88,51,255         85,85,255         0,0,252           92,92,255[48]     0,0,255           92,92,255
Bright Magenta      255,85,255        255,0,255                             214, 112, 214       180,0,158         249,53,248        255,85,255        255,0,255
Bright Cyan         85,255,255        0,255,255                             41, 184, 219        97,214,214        20,240,240        85,255,255        0,255,255
Bright White        255,255,255                                               229, 229, 229       242,242,242       233,235,235       255,255,255

From: https://chrisyeh96.github.io/2020/03/28/terminal-colors.html
Color	RGB	Hexadecimal
 black*	0, 0, 0	000000
 red	204, 0, 0	cc0000
 green	78, 154, 6	4e9a06
 yellow	196, 160, 0	c4a000
 blue*	114, 159, 207	729fcf
 magenta	117, 80, 123	75507b
 cyan	6, 152, 154	06989a
 white	211, 215, 207	d3d7cf
 bright black	85, 87, 83	555753
 bright red	239, 41, 41	ef2929
 bright green	138, 226, 52	8ae234
 bright yellow	252, 233, 79	fce94f
 bright blue*	50, 175, 255	32afff
 bright magenta	173, 127, 168	ad7fa8
 bright cyan	52, 226, 226	34e2e2
 bright white*	255, 255, 255	ffffff

*/

constant SHOW_FG_TEXT = 0

integer proc rgb(integer r, integer g, integer b)
 return ((r shl 16) | (g shl 8) | b )
end

#if 0
proc soft_cga3()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , rgb(  0,   0,   0) ) // "black":
    SetColorTableValue(_FOREGROUND_,_BLUE_       , rgb( 34,   0, 119) ) // "blue":
    SetColorTableValue(_FOREGROUND_,_GREEN_      , rgb(  0, 119,  34) ) // "green":
    SetColorTableValue(_FOREGROUND_,_CYAN_       , rgb( 34, 119, 136) ) // "cyan":
    SetColorTableValue(_FOREGROUND_,_RED_        , rgb(102,   0,  51) ) // "red":
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , rgb(119,   0, 136) ) // "purple":
    SetColorTableValue(_FOREGROUND_,_BROWN_      , rgb(153, 102,   0) ) // "yellow":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , rgb( 68,  68,  85) ) // "white":
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , rgb(119, 119, 153) ) // "brightBlack":
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , rgb( 51,  34, 255) ) // "brightBlue":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, rgb(  0, 255,  51) ) // "brightGreen":
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , rgb( 34, 238, 255) ) // "brightCyan":
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , rgb(255,   0,  51) ) // "brightRed":
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , rgb(255,  51, 221) ) // "brightPurple":
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , rgb(255, 238,  51) ) // "brightYellow":
    SetColorTableValue(_FOREGROUND_,_WHITE_      , rgb(255, 255, 255) ) // "brightWhite":

    SetColorTableValue(_BACKGROUND_,_BLACK_      , rgb(  0,   0,   0) ) // "black":
    SetColorTableValue(_BACKGROUND_,_BLUE_       , rgb( 34,   0, 119) ) // "blue":
    SetColorTableValue(_BACKGROUND_,_GREEN_      , rgb(  0, 119,  34) ) // "green":
    SetColorTableValue(_BACKGROUND_,_CYAN_       , rgb( 34, 119, 136) ) // "cyan":
    SetColorTableValue(_BACKGROUND_,_RED_        , rgb(102,   0,  51) ) // "red":
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , rgb(119,   0, 136) ) // "purple":
    SetColorTableValue(_BACKGROUND_,_BROWN_      , rgb(153, 102,   0) ) // "yellow":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , rgb( 68,  68,  85) ) // "white":
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , rgb(119, 119, 153) ) // "brightBlack":
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , rgb( 51,  34, 255) ) // "brightBlue":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, rgb(  0, 255,  51) ) // "brightGreen":
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , rgb( 34, 238, 255) ) // "brightCyan":
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , rgb(255,   0,  51) ) // "brightRed":
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , rgb(255,  51, 221) ) // "brightPurple":
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , rgb(255, 238,  51) ) // "brightYellow":
    SetColorTableValue(_BACKGROUND_,_WHITE_      , rgb(255, 255, 255) ) // "brightWhite":
end
#endif

// from minibasic
proc MiniBasic()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000) // "black":
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000aa) // "blue":
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00cc55) // "green":
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0xaaffee) // "cyan":
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x880000) // "red":
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xcc44cc) // "purple":
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x664400) // "yellow":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xbbbbbb) // "white":
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x333333) // "brightBlack":
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0088ff) // "brightBlue":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0xaaff66) // "brightGreen":
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x777777) // "brightCyan":
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff7777) // "brightRed":
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xdd8855) // "brightPurple":
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xeeee77) // "brightYellow":
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff) // "brightWhite":

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000) // "black":
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000aa) // "blue":
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00cc55) // "green":
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0xaaffee) // "cyan":
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x880000) // "red":
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xcc44cc) // "purple":
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x664400) // "yellow":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xbbbbbb) // "white":
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x333333) // "brightBlack":
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0088ff) // "brightBlue":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0xaaff66) // "brightGreen":
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x777777) // "brightCyan":
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff7777) // "brightRed":
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xdd8855) // "brightPurple":
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xeeee77) // "brightYellow":
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff) // "brightWhite":
end

//https://renenyffenegger.ch/notes/Linux/shell/ANSI-escape-sequences
proc LinuxShell()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , rgb(  0, 39, 49)) // "black":
    SetColorTableValue(_FOREGROUND_,_BLUE_       , rgb( 32,117,199)) // "blue":
    SetColorTableValue(_FOREGROUND_,_GREEN_      , rgb(114,137,  5)) // "green":
    SetColorTableValue(_FOREGROUND_,_CYAN_       , rgb( 37,145,133)) // "cyan":
    SetColorTableValue(_FOREGROUND_,_RED_        , rgb(208, 27, 36)) // "red":
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , rgb(198, 27,110)) // "purple":
    SetColorTableValue(_FOREGROUND_,_BROWN_      , rgb(165,119,  5)) // "yellow":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , rgb(233,226,203)) // "white":
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , rgb(  0, 77,100)) // "brightBlack":
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , rgb(112,129,131)) // "brightBlue":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, rgb( 70, 90, 97)) // "brightGreen":
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , rgb(129,144,143)) // "brightCyan":
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , rgb(189, 54, 18)) // "brightRed":
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , rgb( 88, 86,185)) // "brightPurple":
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , rgb( 82,103,111)) // "brightYellow":
    SetColorTableValue(_FOREGROUND_,_WHITE_      , rgb(252,244,220)) // "brightWhite":

    SetColorTableValue(_BACKGROUND_,_BLACK_      , rgb(  0, 39, 49)) // "black":
    SetColorTableValue(_BACKGROUND_,_BLUE_       , rgb( 32,117,199)) // "blue":
    SetColorTableValue(_BACKGROUND_,_GREEN_      , rgb(114,137,  5)) // "green":
    SetColorTableValue(_BACKGROUND_,_CYAN_       , rgb( 37,145,133)) // "cyan":
    SetColorTableValue(_BACKGROUND_,_RED_        , rgb(208, 27, 36)) // "red":
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , rgb(198, 27,110)) // "purple":
    SetColorTableValue(_BACKGROUND_,_BROWN_      , rgb(165,119,  5)) // "yellow":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , rgb(233,226,203)) // "white":
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , rgb(  0, 77,100)) // "brightBlack":
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , rgb(112,129,131)) // "brightBlue":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, rgb( 70, 90, 97)) // "brightGreen":
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , rgb(129,144,143)) // "brightCyan":
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , rgb(189, 54, 18)) // "brightRed":
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , rgb( 88, 86,185)) // "brightPurple":
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , rgb( 82,103,111)) // "brightYellow":
    SetColorTableValue(_BACKGROUND_,_WHITE_      , rgb(252,244,220)) // "brightWhite":
end

//https://thestarman.pcministry.com/RGB/16WinColorT.html
proc Starman()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000) // "black":
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x000099) // "blue":         substituted safe color
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x009900) // "green":        substituted safe color
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x009999) // "cyan":         substituted safe color
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x000000) // "red":          substituted safe color
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x990099) // "purple":       substituted safe color
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x999900) // "yellow":       substituted safe color
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xcccccc) // "white":        substituted safe color
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x999999) // "brightBlack":  substituted safe color
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000FF) // "brightBlue":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x00FF00) // "brightGreen":
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x00FFFF) // "brightCyan":
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xFF0000) // "brightRed":
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xFF00FF) // "brightPurple":
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xFFFF00) // "brightYellow":
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xFFFFFF) // "brightWhite":

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000) // "black":
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x000099) // "blue":
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x009900) // "green":
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x009999) // "cyan":
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x000000) // "red":
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x990099) // "purple":
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x999900) // "yellow":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xcccccc) // "white":
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x999999) // "brightBlack":
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000FF) // "brightBlue":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00FF00) // "brightGreen":
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00FFFF) // "brightCyan":
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xFF0000) // "brightRed":
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xFF00FF) // "brightPurple":
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xFFFF00) // "brightYellow":
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xFFFFFF) // "brightWhite":
end

// from WSL
proc Campbell()
    SetColorTableValue(_FOREGROUND_,_BLACK_      ,    0x0C0C0C) // "black":
    SetColorTableValue(_FOREGROUND_,_BLUE_       ,    0x0037DA) // "blue":
    SetColorTableValue(_FOREGROUND_,_GREEN_      ,    0x13A10E) // "green":
    SetColorTableValue(_FOREGROUND_,_CYAN_       ,    0x3A96DD) // "cyan":
    SetColorTableValue(_FOREGROUND_,_RED_        ,    0xC50F1F) // "red":
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    ,    0x881798) // "purple":
    SetColorTableValue(_FOREGROUND_,_BROWN_      ,    0xC19C00) // "yellow":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ ,    0xCCCCCC) // "white":
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  ,    0x767676) // "brightBlack":
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ ,    0x3B78FF) // "brightBlue":
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_,    0x16C60C) // "brightGreen":
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ ,    0x61D6D6) // "brightCyan":
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  ,    0xE74856) // "brightRed":
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  ,    0xB4009E) // "brightPurple":
    SetColorTableValue(_FOREGROUND_,_YELLOW_     ,    0xF9F1A5) // "brightYellow":
    SetColorTableValue(_FOREGROUND_,_WHITE_      ,    0xF2F2F2) // "brightWhite":

    SetColorTableValue(_BACKGROUND_,_BLACK_      ,    0x0C0C0C) // "black":
    SetColorTableValue(_BACKGROUND_,_BLUE_       ,    0x0037DA) // "blue":
    SetColorTableValue(_BACKGROUND_,_GREEN_      ,    0x13A10E) // "green":
    SetColorTableValue(_BACKGROUND_,_CYAN_       ,    0x3A96DD) // "cyan":
    SetColorTableValue(_BACKGROUND_,_RED_        ,    0xC50F1F) // "red":
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    ,    0x881798) // "purple":
    SetColorTableValue(_BACKGROUND_,_BROWN_      ,    0xC19C00) // "yellow":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ ,    0xCCCCCC) // "white":
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  ,    0x767676) // "brightBlack":
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ ,    0x3B78FF) // "brightBlue":
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_,    0x16C60C) // "brightGreen":
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ ,    0x61D6D6) // "brightCyan":
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  ,    0xE74856) // "brightRed":
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  ,    0xB4009E) // "brightPurple":
    SetColorTableValue(_BACKGROUND_,_YELLOW_     ,    0xF9F1A5) // "brightYellow":
    SetColorTableValue(_BACKGROUND_,_WHITE_      ,    0xF2F2F2) // "brightWhite":
end

//https://github.com/dosbox-staging/dosbox-staging/issues/2014
//I also created one myself with a boosted contrast because mine always had such vivid colors. It's like maxing out your contrast knob.
//*I call this Default colors with increased contrast. (vivid might be okay too) Very colorful.
proc cga3()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000A8)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x008C00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x008686)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x9D0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x970097)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x8C3600)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0x939393)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x545454)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x4B4BFF)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x23FF23)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x1AFFFF)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xFF3B3B)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xFF32FF)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xFFFF0A)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000A8)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x008C00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x008686)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x9D0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x970097)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x8C3600)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0x939393)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x545454)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x4B4BFF)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x23FF23)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x1AFFFF)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xFF3B3B)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xFF32FF)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xFFFF0A)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

//PCJr Palette (from Pcjr Magazine) I cloned the colors from PCJr magazine screenshots!
//#000000 #0D1B9E #146626 #639F7A #AD1F13 #7A0EA1 #FFB139 #94B1C1 #4E4075 #3756E2 #8BAF65 #9CC4E3 #D13321 #DE24AD #F7E36F #EFF1ED
//
//CGA Palette (from pcjr Magazine) Same as above but for CGA machines.
//#000000 #0D1B9E #146626 #3B8CC4 #AD1F13 #7A0EA1 #8F8034 #94B1C1 #4E4075 #3756E2 #8BAF65 #9CC4E3 #D13321 #DE24AD #F7E36F #EFF1ED

//https://int10h.org/blog/2022/06/ibm-5153-color-true-cga-palette/
proc cga2()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000C4)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00C400)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x00C4C4)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xC40000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xC400C4)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xC47E00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xC4C4C4)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x4E4E4E)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x4E4EDC)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x4EDC4E)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x4EF3F3)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xDC4E4E)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xF34EF3)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xF3F34E)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xFFFFFF)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000C4)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00C400)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x00C4C4)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xC40000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xC400C4)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xC47E00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xC4C4C4)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x4E4E4E)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x4E4EDC)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x4EDC4E)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x4EF3F3)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xDC4E4E)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xF34EF3)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xF3F34E)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xFFFFFF)
end

//https://lospec.com/palette-list/soft-cga
proc soft_cga()
//    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
//    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x444455)
//    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x220077)
//    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x3322ff)
//    SetColorTableValue(_FOREGROUND_,_RED_        , 0x660033)
//    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xff0033)
//    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x770088)
//    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xff33dd)
//    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x007722)
//    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x00ff33)
//    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x227788)
//    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x22eeff)
//    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0x996600)
//    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xffee33)
//    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0x777799)
//    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)
//
//    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
//    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x444455)
//    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x220077)
//    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x3322ff)
//    SetColorTableValue(_BACKGROUND_,_RED_        , 0x660033)
//    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xff0033)
//    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x770088)
//    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xff33dd)
//    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x007722)
//    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x00ff33)
//    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x227788)
//    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x22eeff)
//    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0x996600)
//    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xffee33)
//    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0x777799)
//    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)

SetColorTableValue(_BACKGROUND_, _BLACK_, 0x000000)
SetColorTableValue(_BACKGROUND_, _BLUE_, 0x220077)
SetColorTableValue(_BACKGROUND_, _GREEN_, 0x007722)
SetColorTableValue(_BACKGROUND_, _CYAN_, 0x227788)
SetColorTableValue(_BACKGROUND_, _RED_, 0x660033)
SetColorTableValue(_BACKGROUND_, _MAGENTA_, 0x770088)
SetColorTableValue(_BACKGROUND_, _BROWN_, 0x996600)
SetColorTableValue(_BACKGROUND_, _LIGHT_GRAY_, 0x777799)
SetColorTableValue(_BACKGROUND_, _DARK_GRAY_, 0x444455)
SetColorTableValue(_BACKGROUND_, _LIGHT_BLUE_, 0x3322ff)
SetColorTableValue(_BACKGROUND_, _LIGHT_GREEN_, 0x00ff33)
SetColorTableValue(_BACKGROUND_, _LIGHT_CYAN_, 0x22eeff)
SetColorTableValue(_BACKGROUND_, _LIGHT_RED_, 0xff0033)
SetColorTableValue(_BACKGROUND_, _LIGHT_MAG_,     0xff33dd)
SetColorTableValue(_BACKGROUND_, _YELLOW_, 0xffee33)
SetColorTableValue(_BACKGROUND_, _WHITE_, 0xffffff)

SetColorTableValue(_FOREGROUND_, _BLACK_, 0x000000)
SetColorTableValue(_FOREGROUND_, _BLUE_, 0x220077)
SetColorTableValue(_FOREGROUND_, _GREEN_, 0x007722)
SetColorTableValue(_FOREGROUND_, _CYAN_, 0x227788)
SetColorTableValue(_FOREGROUND_, _RED_, 0x660033)
SetColorTableValue(_FOREGROUND_, _MAGENTA_, 0x770088)
SetColorTableValue(_FOREGROUND_, _BROWN_, 0x996600)
SetColorTableValue(_FOREGROUND_, _LIGHT_GRAY_, 0x777799)
SetColorTableValue(_FOREGROUND_, _DARK_GRAY_, 0x444455)
SetColorTableValue(_FOREGROUND_, _LIGHT_BLUE_, 0x3322ff)
SetColorTableValue(_FOREGROUND_, _LIGHT_GREEN_, 0x00ff33)
SetColorTableValue(_FOREGROUND_, _LIGHT_CYAN_, 0x22eeff)
SetColorTableValue(_FOREGROUND_, _LIGHT_RED_, 0xff0033)
SetColorTableValue(_FOREGROUND_, _LIGHT_MAG_,     0xff33dd)
SetColorTableValue(_FOREGROUND_, _YELLOW_, 0xffee33)
SetColorTableValue(_FOREGROUND_, _WHITE_, 0xffffff)

//SetColorTableValue(_FOREGROUND_, _BLACK_        , 0x000000)
//SetColorTableValue(_FOREGROUND_, _DARK_GRAY_    , 0x444455)
//SetColorTableValue(_FOREGROUND_, _BLUE_         , 0x220077)
//SetColorTableValue(_FOREGROUND_, _LIGHT_BLUE_   , 0x3322ff)
//SetColorTableValue(_FOREGROUND_, _CYAN_         , 0x660033)
//SetColorTableValue(_FOREGROUND_, _RED_          , 0xff0033)
//SetColorTableValue(_FOREGROUND_, _MAGENTA_      , 0x770088)
//SetColorTableValue(_FOREGROUND_, _LIGHT_MAG_    , 0xff33dd)
//SetColorTableValue(_FOREGROUND_, _GREEN_        , 0x007722)
//SetColorTableValue(_FOREGROUND_, _LIGHT_GREEN_  , 0x00ff33)
//SetColorTableValue(_FOREGROUND_, _LIGHT_CYAN_   , 0x227788)
//SetColorTableValue(_FOREGROUND_, _CYAN_         , 0x22eeff)
//SetColorTableValue(_FOREGROUND_, _LIGHT_RED_    , 0x996600)
//SetColorTableValue(_FOREGROUND_, _YELLOW_       , 0xffee33)
//SetColorTableValue(_FOREGROUND_, _LIGHT_GRAY_   , 0x777799)
//SetColorTableValue(_FOREGROUND_, _WHITE_        , 0xffffff)
//
//SetColorTableValue(_BACKGROUND_, _BLACK_        , 0x000000)
//SetColorTableValue(_BACKGROUND_, _DARK_GRAY_    , 0x444455)
//SetColorTableValue(_BACKGROUND_, _BLUE_         , 0x220077)
//SetColorTableValue(_BACKGROUND_, _LIGHT_BLUE_   , 0x3322ff)
//SetColorTableValue(_BACKGROUND_, _CYAN_         , 0x660033)
//SetColorTableValue(_BACKGROUND_, _RED_          , 0xff0033)
//SetColorTableValue(_BACKGROUND_, _MAGENTA_      , 0x770088)
//SetColorTableValue(_BACKGROUND_, _LIGHT_MAG_    , 0xff33dd)
//SetColorTableValue(_BACKGROUND_, _GREEN_        , 0x007722)
//SetColorTableValue(_BACKGROUND_, _LIGHT_GREEN_  , 0x00ff33)
//SetColorTableValue(_BACKGROUND_, _LIGHT_CYAN_   , 0x227788)
//SetColorTableValue(_BACKGROUND_, _CYAN_         , 0x22eeff)
//SetColorTableValue(_BACKGROUND_, _LIGHT_RED_    , 0x996600)
//SetColorTableValue(_BACKGROUND_, _YELLOW_       , 0xffee33)
//SetColorTableValue(_BACKGROUND_, _LIGHT_GRAY_   , 0x777799)
//SetColorTableValue(_BACKGROUND_, _WHITE_        , 0xffffff)
end

proc soft_cga2()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)  // black
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x220077)  // blue
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x007722)  // green
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x22eeff)  // cyan
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xff0033)  // red
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x770088)  // magenta
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x660033)  // dark red
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0x777799)  // light gray
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x444455)  // dark gray
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x3322ff)  // light blue
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x00ff33)  // light green
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x227788)  // light cyan
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0x996600)  // light red
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff33dd)  // light magenta
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffee33)  // light yellow
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)  // white

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)  // black
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x220077)  // blue
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x007722)  // green
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x22eeff)  // cyan
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xff0033)  // red
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x770088)  // magenta
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x660033)  // dark red
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0x777799)  // light gray
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x444455)  // dark gray
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x3322ff)  // light blue
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff33)  // light green
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x227788)  // light cyan
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0x996600)  // light red
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff33dd)  // light magenta
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffee33)  // light yellow
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)  // white
end

//https://lospec.com/palette-list/aap-dga16
proc aap_dga16()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x010101)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x031b75)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x108c00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x17bbd3)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x720c0a)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x6c1c9e)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xb25116)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xb8b0a8)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x4a4842)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0b63c4)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x9bce00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x73f5d5)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xe89e00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff7bdb)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xfef255)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xfffffe)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x010101)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x031b75)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x108c00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x17bbd3)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x720c0a)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x6c1c9e)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xb25116)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xb8b0a8)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x4a4842)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0b63c4)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x9bce00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x73f5d5)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xe89e00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff7bdb)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xfef255)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xfffffe)
end

// https://androidarts.com/palette/16pal.htm
// new rgb(), removed toggle.  Hmmm. Looked better before?
proc perfect4()
    SetColorTableValue(_FOREGROUND_,_BLACK_      ,RGB(  0,   0,   0))    // black
    SetColorTableValue(_FOREGROUND_,_BLUE_       ,RGB(  0,  87, 132))    // sea blue
    SetColorTableValue(_FOREGROUND_,_GREEN_      ,RGB( 47,  72,  78))    // dark green
    SetColorTableValue(_FOREGROUND_,_CYAN_       ,RGB(190,  38,  51))    // red
    SetColorTableValue(_FOREGROUND_,_RED_        ,RGB(224, 111, 139))    // meat
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    ,RGB( 73,  60,  43))    // dark brown
    SetColorTableValue(_FOREGROUND_,_BROWN_      ,RGB(164, 100,  34))    // brown
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ ,RGB(157, 157, 157))    // gray
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  ,RGB( 27,  38,  50))    // night blue
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ ,RGB(163, 206,  39))    // slime green
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_,RGB( 68, 137,  26))    // green
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ ,RGB( 49, 162, 242))    // sky blue
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  ,RGB(235, 137,  49))    // orange
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  ,RGB(178, 220, 239))    // cloud blue
    SetColorTableValue(_FOREGROUND_,_YELLOW_     ,RGB(247, 226, 107))    // yellow
    SetColorTableValue(_FOREGROUND_,_WHITE_      ,RGB(255, 255, 255))    // white

    SetColorTableValue(_BACKGROUND_,_BLACK_      ,RGB(  0,   0,   0))    // black
    SetColorTableValue(_BACKGROUND_,_BLUE_       ,RGB(  0,  87, 132))    // sea blue
    SetColorTableValue(_BACKGROUND_,_GREEN_      ,RGB( 47,  72,  78))    // dark green
    SetColorTableValue(_BACKGROUND_,_CYAN_       ,RGB(190,  38,  51))    // red
    SetColorTableValue(_BACKGROUND_,_RED_        ,RGB(224, 111, 139))    // meat
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    ,RGB( 73,  60,  43))    // dark brown
    SetColorTableValue(_BACKGROUND_,_BROWN_      ,RGB(164, 100,  34))    // brown
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ ,RGB(157, 157, 157))    // gray
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  ,RGB( 27,  38,  50))    // night blue
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ ,RGB(163, 206,  39))    // slime gree
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_,RGB( 68, 137,  26))    // green     n
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ ,RGB( 49, 162, 242))    // sky blue
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  ,RGB(235, 137,  49))    // orange
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  ,RGB(178, 220, 239))    // cloud bluenta
    SetColorTableValue(_BACKGROUND_,_YELLOW_     ,RGB(247, 226, 107))    // yellow
    SetColorTableValue(_BACKGROUND_,_WHITE_      ,RGB(255, 255, 255))    // white
end

// https://alumni.media.mit.edu/~wad/color/numbers.html
// new rgb(), removed toggle.  Hmmm. Looks ok.
proc perfect3()
    SetColorTableValue(_FOREGROUND_,_BLACK_      ,RGB(  0,   0,   0))    // black
    SetColorTableValue(_FOREGROUND_,_BLUE_       ,RGB( 42,  75, 215))    // blue
    SetColorTableValue(_FOREGROUND_,_GREEN_      ,RGB( 29, 105,  20))    // green
    SetColorTableValue(_FOREGROUND_,_CYAN_       ,RGB( 41, 208, 208))    // cyan
    SetColorTableValue(_FOREGROUND_,_RED_        ,RGB(173,  35,  35))    // red
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    ,RGB(129,  38, 192))    // purple
    SetColorTableValue(_FOREGROUND_,_BROWN_      ,RGB(129,  74,  25))    // brown
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ ,RGB(160, 160, 160))    // lt. gray
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  ,RGB( 87,  87,  87))    // dk. gray
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ ,RGB(157, 175, 255))    // lt. blue
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_,RGB(129, 197, 122))    // lt. green
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ ,RGB(233, 222, 187))    // tan
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  ,RGB(255, 205, 243))    // pink
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  ,RGB(255, 146,  51))    // orange
    SetColorTableValue(_FOREGROUND_,_YELLOW_     ,RGB(255, 238,  51))    // yellow
    SetColorTableValue(_FOREGROUND_,_WHITE_      ,RGB(255, 255, 255))    // black

    SetColorTableValue(_BACKGROUND_,_BLACK_      ,RGB(  0,   0,   0))    //
    SetColorTableValue(_BACKGROUND_,_BLUE_       ,RGB( 42,  75, 215))    //
    SetColorTableValue(_BACKGROUND_,_GREEN_      ,RGB( 29, 105,  20))    //
    SetColorTableValue(_BACKGROUND_,_CYAN_       ,RGB( 41, 208, 208))    //
    SetColorTableValue(_BACKGROUND_,_RED_        ,RGB(173,  35,  35))    //
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    ,RGB(129,  38, 192))    //
    SetColorTableValue(_BACKGROUND_,_BROWN_      ,RGB(129,  74,  25))    //
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ ,RGB(160, 160, 160))    //
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  ,RGB( 87,  87,  87))    //
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ ,RGB(157, 175, 255))    //
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_,RGB(129, 197, 122))    //
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ ,RGB(233, 222, 187))    //
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  ,RGB(255, 205, 243))    //
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  ,RGB(255, 146,  51))    //
    SetColorTableValue(_BACKGROUND_,_YELLOW_     ,RGB(255, 238,  51))    //
    SetColorTableValue(_BACKGROUND_,_WHITE_      ,RGB(255, 255, 255))    //
end

// first is foreground color - 2nd is background
proc TSE_default()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)    // black   rgb(  0,   0,   0)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x000080)    // navy    rgb(  0,   0, 128)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x008000)    // green   rgb(  0, 128,   0)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x008080)    // teal    rgb(  0, 128, 128)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x800000)    // maroon  rgb(128,   0,   0)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x800080)    // purple  rgb(128,   0, 128)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x808000)    // olive   rgb(128, 128,   0)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xc0c0c0)    // silver  rgb(192, 192, 192)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x808080)    // gray    rgb(128, 128, 128)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)    // blue    rgb(  0,   0, 255)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x00ff00)    // lime    rgb(  0, 255,   0)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x00ffff)    // aqua    rgb(  0, 255, 255)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff0000)    // red     rgb(255,   0,   0)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)    // fuchsia rgb(255,   0, 255)   violet
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff00)    // yellow  rgb(255, 255,   0)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)    // white   rgb(255, 255, 255)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)   //
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x000080)   //
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x008000)   //
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x008080)   //
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x800000)   //
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x800080)   //
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x808000)   //
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xc0c0c0)   //
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x808080)   //
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)   //
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff00)   //
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)   //
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff0000)   //
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)   //
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)   //
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)   //
end

// first is foreground color - 2nd is background
proc retro()
//    SetColorTableValue(_FOREGROUND_,_BLACK_      ,ToggleRGB(0x201914))    // black
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x262626)    // black
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x000080)    // blue
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x008000)    // green
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x008080)    // cyan
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x800000)    // red
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x800080)    // magenta
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x808000)    // brown
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xc0c0c0)    // light gray
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x808080)    // dark gray
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)    // light blue
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x00ff00)    // light green
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x00ffff)    // light cyan
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff0000)    // light red
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)    // light magenta
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff00)    // yellow
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)    // white

//    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x141920)    // black
    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x262626)    // black
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x000080)    // blue
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x008000)    // green
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x008080)    // cyan
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x800000)    // red
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x800080)    // magenta
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x808000)    // brown
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xc0c0c0)    // light gray
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x808080)    // dark gray
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)    // light blue
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff00)    // light green
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)    // light cyan
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff0000)    // light red
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)    // light magenta
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)    // yellow
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)    // white
end

// first is foreground color - 2nd is background
proc solorized1()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x002b36)    // black
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x073642)    // blue
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x586e75)    // green
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x657b83)    // cyan
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x839496)    // red
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x93a1a1)    // magenta
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xeee8d5)    // brown
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xfdf6e3)    // light gray
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0xb58900)    // dark gray
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0xcb4b16)    // light blue
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0xdc322f)    // light green
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0xd33682)    // light cyan
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0x6c71c4)    // light red
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0x268bd2)    // light magenta
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0x2aa198)    // yellow
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0x859900)    // white

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x002b36)    // black
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x073642)    // blue
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x586e75)    // green
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x657b83)    // cyan
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x839496)    // red
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x93a1a1)    // magenta
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xeee8d5)    // brown
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xfdf6e3)    // light gray
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0xb58900)    // dark gray
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0xcb4b16)    // light blue
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0xdc322f)    // light green
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0xd33682)    // light cyan
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0x6c71c4)    // light red
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0x268bd2)    // light magenta
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0x2aa198)    // yellow
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0x859900)    // white
end

// first is foreground color - 2nd is background
proc solorized2()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x1c1c1c)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x262626)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x585858)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x626262)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x808080)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x8a8a8a)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xe4e4e4)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xffffd7)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0xaf8700)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0xd75f00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0xd70000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0xaf005f)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0x5f5faf)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0x0087ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0x00afaf)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0x5f8700)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x1c1c1c)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x262626)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x585858)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x626262)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x808080)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x8a8a8a)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xe4e4e4)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xffffd7)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0xaf8700)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0xd75f00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0xd70000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0xaf005f)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0x5f5faf)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0x0087ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0x00afaf)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0x5f8700)
end

// first is foreground color - 2nd is background
proc c_HTML()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x00008b)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x006400)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x008080)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x8b0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x8b008b)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x8b8b00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xd3d3d3)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0xa9a9a9)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x008000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x00008b)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x006400)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x008080)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x8b0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x8b008b)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x8b8b00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xd3d3d3)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0xa9a9a9)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x008000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc c_Brite() //@80x25
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000a0)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00a000)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0xa0a000)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xa00000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xa000a0)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xa05000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x505050)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x5050ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x50ff50)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x50ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff5050)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff50ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff50)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000a0)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00a000)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0xa0a000)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xa00000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xa000a0)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xa05000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x505050)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x5050ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x50ff50)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x50ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff5050)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff50ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff50)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc c_Dark() // >80x25
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x00007f)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x007f00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x007f7f)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x7f0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x7f007f)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x7f7f00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xbfbfbf)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x7f7f7f)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x00007f)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x007f00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x007f7f)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x7f0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x7f007f)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x7f7f00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xbfbfbf)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x7f7f7f)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc c_Brite2() // 80x25#2
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000a8)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00a800)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x00a8a8)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xa80000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xa800a8)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xa85800)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xa8a8a8)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x585858)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x5858ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x58ff58)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x58ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff5858)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff58ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff58)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000a8)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00a800)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x00a8a8)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xa80000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xa800a8)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xa85800)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xa8a8a8)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x585858)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x5858ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x58ff58)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x58ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff5858)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff58ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff58)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc FED()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000aa)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00aa00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x2255aa)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xaa0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xaa00aa)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xaaaa00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xaaaaaa)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x444444)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x4444ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x44ff44)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x44ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff4444)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff44ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff44)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000aa)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00aa00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x2255aa)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xaa0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xaa00aa)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xaaaa00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xaaaaaa)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x444444)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x4444ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x44ff44)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x44ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff4444)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff44ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff44)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc IBM_CGA()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000aa)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00aa00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x00aaaa)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xaa0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xaa00aa)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xaa5500)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xaaaaaa)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x555555)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x5555ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x55ff55)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x55ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff5555)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff55ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff55)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000aa)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00aa00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x00aaaa)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xaa0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xaa00aa)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xaa5500)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xaaaaaa)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x555555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x5555ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x55ff55)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x55ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff5555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff55ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff55)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc _24_BIT()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x000084)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x008400)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x008484)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x840000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x840084)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x848400)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xc6c6c6)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x848484)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x000084)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x008400)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x008484)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x840000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x840084)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x848400)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xc6c6c6)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x848484)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc _1() //Perfect
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x2a4bd7)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x1d6914)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x29d0d0)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xad2323)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x8126c0)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x814a19)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x575757)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x9dafff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x81c57a)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0xe9debb)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff9233)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xffcdf3)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffee33)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x2a4bd7)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x1d6914)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x29d0d0)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xad2323)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x8126c0)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x814a19)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x575757)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x9dafff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x81c57a)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0xe9debb)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff9233)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xffcdf3)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffee33)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc perfect2() //Perfect
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x2323ad)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x14691d)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x194a81)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xd74b2a)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xc02681)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xd0d029)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x575757)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x3392ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x7ac581)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x33eeff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xffaf9d)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xf3cdff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xbbdee9)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x2323ad)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x14691d)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x194a81)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xd74b2a)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xc02681)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xd0d029)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x575757)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x3392ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x7ac581)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x33eeff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xffaf9d)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xf3cdff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xbbdee9)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc _2() // xp boot
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x2d3ed2)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x536501)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x5e7ffc)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x899200)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x201a15)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xf76b20)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0x7e7e7e)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x464646)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x8edc04)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x8da6ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0xfcfcfc)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xb23505)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xbcbcbc)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xf3bc1b)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x2d3ed2)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x536501)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x5e7ffc)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xb23505)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x201a15)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x899200)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0x7e7e7e)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x464646)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x8da6ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x8edc04)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0xfcfcfc)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xf76b20)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xbcbcbc)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xf3bc1b)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc ANSIPLUS()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x000068)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x006833)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x006666)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x680000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x680068)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x663300)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xcccccc)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x666666)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0099ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff99cc)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x000068)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x006833)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x006666)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x680000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x680068)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x663300)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xcccccc)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x666666)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0099ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff99cc)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc IBM_OEM()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x000099)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x009900)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x009999)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x990000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x990099)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xcc6600)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xcccccc)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x666666)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x6666ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x66ff66)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x66ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff6666)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff66ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff66)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x000099)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x009900)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x009999)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x990000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x990099)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xcc6600)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xcccccc)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x666666)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x6666ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x66ff66)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x66ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff6666)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff66ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff66)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc vga()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000aa)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00aa00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x0055aa)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xaa0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xaa00aa)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xaaaa00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xaaaaaa)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x555555)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x5555ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x55ff55)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x55ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff5555)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff55ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff55)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000aa)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00aa00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x0055aa)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xaa0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xaa00aa)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xaaaa00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xaaaaaa)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x555555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x5555ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x55ff55)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x55ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff5555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff55ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff55)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc efte()
    SetColorTableValue(_FOREGROUND_,_BLACK_      ,  0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       ,  0x002454)
    SetColorTableValue(_FOREGROUND_,_GREEN_      ,  0x047c34)
    SetColorTableValue(_FOREGROUND_,_CYAN_       ,  0x007c9c)
    SetColorTableValue(_FOREGROUND_,_RED_        ,  0x980000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    ,  0x00a8fc)
    SetColorTableValue(_FOREGROUND_,_BROWN_      ,  0x888800)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ ,  0xa8a8a8)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  ,  0x404040)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ ,  0x547cac)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_,  0x54f490)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ ,  0x54e0fc)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  ,  0xfc6854)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  ,  0xfc84b0)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     ,  0xfcfc54)
    SetColorTableValue(_FOREGROUND_,_WHITE_      ,  0xfcfce4)

    SetColorTableValue(_BACKGROUND_,_BLACK_      ,  0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       ,  0x002454)
    SetColorTableValue(_BACKGROUND_,_GREEN_      ,  0x047c34)
    SetColorTableValue(_BACKGROUND_,_CYAN_       ,  0x007c9c)
    SetColorTableValue(_BACKGROUND_,_RED_        ,  0x980000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    ,  0x00a8fc)
    SetColorTableValue(_BACKGROUND_,_BROWN_      ,  0x888800)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ ,  0xa8a8a8)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  ,  0x404040)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ ,  0x547cac)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_,  0x54f490)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ ,  0x54e0fc)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  ,  0xfc6854)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  ,  0xfc84b0)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     ,  0xfcfc54)
    SetColorTableValue(_BACKGROUND_,_WHITE_      ,  0xfcfce4)
end

// first is foreground color - 2nd is background
proc web1()
    SetColorTableValue(_BACKGROUND_,_BLACK_      ,  0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       ,  0x0000aa)
    SetColorTableValue(_BACKGROUND_,_GREEN_      ,  0x00aa00)
    SetColorTableValue(_BACKGROUND_,_BROWN_      ,  0x00aaff)
    SetColorTableValue(_BACKGROUND_,_RED_        ,  0xaa0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    ,  0xaa00aa)
    SetColorTableValue(_BACKGROUND_,_CYAN_       ,  0xaaaa00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ ,  0xaaaaaa)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  ,  0x555555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  ,  0x5555ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_,  0x55ff55)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     ,  0x55ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ ,  0xff5555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  ,  0xff55ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ ,  0xffff55)
    SetColorTableValue(_BACKGROUND_,_WHITE_      ,  0xffffff)

    SetColorTableValue(_FOREGROUND_,_BLACK_      ,  0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       ,  0x00002a)
    SetColorTableValue(_FOREGROUND_,_GREEN_      ,  0x002a00)
    SetColorTableValue(_FOREGROUND_,_BROWN_      ,  0x002a2a)
    SetColorTableValue(_FOREGROUND_,_RED_        ,  0x2a0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    ,  0x2a002a)
    SetColorTableValue(_FOREGROUND_,_CYAN_       ,  0x2a2a00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ ,  0x2a2a2a)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  ,  0x151515)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  ,  0x15153f)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_,  0x153f15)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     ,  0x153f3f)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ ,  0x3f1515)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  ,  0x3f153f)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ ,  0x3f3f15)
    SetColorTableValue(_FOREGROUND_,_WHITE_      ,  0x3f3f3f)
end

//  9 becomes 12
// 12 to 13
// 11 to  9
// 13 to 11
// 3 to 6
// first is foreground color - 2nd is background
proc web2()
    SetColorTableValue(_FOREGROUND_,_BLACK_      ,  0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       ,  0x0000aa)
    SetColorTableValue(_FOREGROUND_,_GREEN_      ,  0x00aa00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       ,  0x00aaff)
    SetColorTableValue(_FOREGROUND_,_RED_        ,  0xaa0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    ,  0xaa00aa)
    SetColorTableValue(_FOREGROUND_,_BROWN_      ,  0xaaaa00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ ,  0xaaaaaa)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  ,  0x555555)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ ,  0x5555ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_,  0x55ff55)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ ,  0x55ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  ,  0xff5555)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  ,  0xff55ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     ,  0xffff55)
    SetColorTableValue(_FOREGROUND_,_WHITE_      ,  0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      ,  0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       ,  0x0000aa)
    SetColorTableValue(_BACKGROUND_,_GREEN_      ,  0x00aa00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       ,  0x00aaff)
    SetColorTableValue(_BACKGROUND_,_RED_        ,  0xaa0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    ,  0xaa00aa)
    SetColorTableValue(_BACKGROUND_,_BROWN_      ,  0xaaaa00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ ,  0xaaaaaa)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  ,  0x555555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ ,  0x5555ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_,  0x55ff55)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ ,  0x55ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  ,  0xff5555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  ,  0xff55ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     ,  0xffff55)
    SetColorTableValue(_BACKGROUND_,_WHITE_      ,  0xffffff)
end

// first is foreground color - 2nd is background
proc light_color()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x008000)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x008080)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0x800000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x800080)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x808000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x808080)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x395ff4)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x008000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x66ccff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xeed202)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xf0ffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x008000)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x008080)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0x800000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x800080)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x808000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x808080)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x395ff4)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x008000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x66ccff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff0000)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xeed202)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xf0ffff)
end

// first is foreground color - 2nd is background, using standard xterm
proc xterm_opiz()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0xb2b2b2)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x2828cd)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x28cd28)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x47cdcd)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xed6b6b)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xcd28cd)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xcdcd28)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xe5e5e5)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0xb2b2b2)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x5959ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x59ff59)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x7fffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff7f7f)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff59ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff59)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000cd)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00cd00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x00cdcd)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xee0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xcd00cd)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xcdcd00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xe5e5e5)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x7f7f7f)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff5c5c)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc term_app()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x2136c2)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x26bc25)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x27adad)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xe12e49)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xd338d3)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xc8bb33)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xcdcccb)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x838381)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x1f39fc)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x22e731)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x23ecea)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff3358)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xf835f9)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xf0f014)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xebebe9)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x2136c2)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x26bc25)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x27adad)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xe12e49)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xd338d3)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xc8bb33)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xcdcccb)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x838381)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x1f39fc)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x22e731)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x23ecea)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff3358)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xf835f9)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xf0f014)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xebebe9)
end

// first is foreground color - 2nd is background
proc putty()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000bb)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00bb00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x00bbbb)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xbb0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xbb00bb)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xbbbb00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xbbbbbb)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x555555)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x5555ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x55ff55)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x55ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff5555)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff55ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff55)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000bb)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00bb00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x00bbbb)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xbb0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xbb00bb)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xbbbb00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xbbbbbb)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x555555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x5555ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x55ff55)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x55ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff5555)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff55ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff55)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc xterm()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000cd)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00cd00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x00cdcd)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xee0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xcd00cd)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xcdcd00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xe5e5e5)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x7f7f7f)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff5c5c)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000cd)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00cd00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x00cdcd)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xee0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xcd00cd)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xcdcd00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xe5e5e5)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x7f7f7f)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff5c5c)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc css_html()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00ff00)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x00ffff)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xff0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xffff00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xffffff)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0xa9a9a9)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x90ee90)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0xe0ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xe6d8ac)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffffe0)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00ff00)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x00ffff)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xff0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xffff00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xffffff)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0xa9a9a9)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x90ee90)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0xe0ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xe6d8ac)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffffe0)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc x_windows()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x008000)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x00ffff)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xff0000)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xffff00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xffffff)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0xa9a9a9)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x90ee90)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0xe0ffff)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xe6d8ac)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffffe0)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x008000)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x00ffff)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xff0000)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xffff00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xffffff)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0xa9a9a9)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x90ee90)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0xe0ffff)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xe6d8ac)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffffe0)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)
end

// first is foreground color - 2nd is background
proc solorized5()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x423607)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x2f32dc)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x009985)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x0089b5)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xd28b26)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x8236d3)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x98a12a)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xd5e8ee)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x362b00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x164bcb)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x756e58)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x837b65)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0x969483)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xc4716c)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xa1a193)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xe3f6fd)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x423607)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x2f32dc)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x009985)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x0089b5)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xd28b26)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x8236d3)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x98a12a)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xd5e8ee)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x362b00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x164bcb)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x756e58)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x837b65)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0x969483)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xc4716c)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xa1a193)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xe3f6fd)
end

// first is foreground color - 2nd is background
proc solorized6()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x262626)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x0000d7)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x00875f)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x0087af)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xff8700)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x5f00af)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0xafaf00)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xe4e4e4)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x1c1c1c)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x005fd7)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x585858)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x626262)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0x808080)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xaf5f5f)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0x8a8a8a)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xd7ffff)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x262626)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x0000d7)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x00875f)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x0087af)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xff8700)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x5f00af)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0xafaf00)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xe4e4e4)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x1c1c1c)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x005fd7)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x585858)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x626262)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0x808080)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xaf5f5f)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0x8a8a8a)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xd7ffff)
end

// first is foreground color - 2nd is background
proc from_stackoverflow()
    SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x0c0c0c)
    SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x1f0fc5)
    SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x0ea113)
    SetColorTableValue(_FOREGROUND_,_CYAN_       , 0xdd963a)
    SetColorTableValue(_FOREGROUND_,_RED_        , 0xda3700)
    SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x981788)
    SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x009cc1)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xcccccc)
    SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x767676)
    SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x5648e7)
    SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x0cc616)
    SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0xd6d661)
    SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xff783b)
    SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0x9e00b4)
    SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xa5f1f9)
    SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xf2f2f2)

    SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x0c0c0c)
    SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x1f0fc5)
    SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x0ea113)
    SetColorTableValue(_BACKGROUND_,_CYAN_       , 0xdd963a)
    SetColorTableValue(_BACKGROUND_,_RED_        , 0xda3700)
    SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x981788)
    SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x009cc1)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xcccccc)
    SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x767676)
    SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x5648e7)
    SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x0cc616)
    SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0xd6d661)
    SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff783b)
    SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0x9e00b4)
    SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xa5f1f9)
    SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xf2f2f2)
end

proc ZhongZhao()
  SetColorTableValue(_BACKGROUND_,_BLACK_      , 0x000000)    // black          0x000000
  SetColorTableValue(_BACKGROUND_,_BLUE_       , 0x000080)    // blue           0x000080
  SetColorTableValue(_BACKGROUND_,_GREEN_      , 0x548c54)    // green          0x008000
  SetColorTableValue(_BACKGROUND_,_CYAN_       , 0x548c87)    // cyan           0x008080
  SetColorTableValue(_BACKGROUND_,_RED_        , 0x800000)    // red            0x800000
  SetColorTableValue(_BACKGROUND_,_MAGENTA_    , 0x800080)    // magenta        0x800080
  SetColorTableValue(_BACKGROUND_,_BROWN_      , 0x4c8065)    // brown          0x808000
  SetColorTableValue(_BACKGROUND_,_LIGHT_GRAY_ , 0xa0a0a0)    // light gray     0xc0c0c0
  SetColorTableValue(_BACKGROUND_,_DARK_GRAY_  , 0x808080)    // dark gray      0x808080
  SetColorTableValue(_BACKGROUND_,_LIGHT_BLUE_ , 0x0000ff)    // light blue     0x0000ff
  SetColorTableValue(_BACKGROUND_,_LIGHT_GREEN_, 0x00ff00)    // light green    0x00ff00
  SetColorTableValue(_BACKGROUND_,_LIGHT_CYAN_ , 0x00ffff)    // light cyan     0x00ffff
  SetColorTableValue(_BACKGROUND_,_LIGHT_RED_  , 0xff0000)    // light red      0xff0000
  SetColorTableValue(_BACKGROUND_,_LIGHT_MAG_  , 0xff00ff)    // light magenta  0xff00ff
  SetColorTableValue(_BACKGROUND_,_YELLOW_     , 0xffff00)    // yellow         0xffff00
  SetColorTableValue(_BACKGROUND_,_WHITE_      , 0xffffff)    // white          0xffffff

  SetColorTableValue(_FOREGROUND_,_BLACK_      , 0x000000)    // black          0x000000
  SetColorTableValue(_FOREGROUND_,_BLUE_       , 0x000080)    // blue           0x000080
  SetColorTableValue(_FOREGROUND_,_GREEN_      , 0x004000)    // green          0x008000
  SetColorTableValue(_FOREGROUND_,_CYAN_       , 0x004040)    // cyan           0x008080
  SetColorTableValue(_FOREGROUND_,_RED_        , 0x800000)    // red            0x800000
  SetColorTableValue(_FOREGROUND_,_MAGENTA_    , 0x800080)    // magenta        0x800080
  SetColorTableValue(_FOREGROUND_,_BROWN_      , 0x804000)    // brown          0x808000
  SetColorTableValue(_FOREGROUND_,_LIGHT_GRAY_ , 0xfeddbc)    // light gray     0xc0c0c0
  SetColorTableValue(_FOREGROUND_,_DARK_GRAY_  , 0x404040)    // dark gray      0x808080
  SetColorTableValue(_FOREGROUND_,_LIGHT_BLUE_ , 0x0000ff)    // light blue     0x0000ff
  SetColorTableValue(_FOREGROUND_,_LIGHT_GREEN_, 0x86fc86)    // light green    0x00ff00
  SetColorTableValue(_FOREGROUND_,_LIGHT_CYAN_ , 0x9ffafa)    // light cyan     0x00ffff
  SetColorTableValue(_FOREGROUND_,_LIGHT_RED_  , 0xffbfbf)    // light red      0xff0000
  SetColorTableValue(_FOREGROUND_,_LIGHT_MAG_  , 0xfecbfe)    // light magenta  0xff00ff
  SetColorTableValue(_FOREGROUND_,_YELLOW_     , 0xffffa0)    // yellow         0xffff00
  SetColorTableValue(_FOREGROUND_,_WHITE_      , 0xffffff)    // white          0xffffff
end

proc MathLogic()
    SetColorTableValue(_FOREGROUND_, _BLACK_      ,(RGB(255, 194,  10)))
    SetColorTableValue(_FOREGROUND_, _BLUE_       ,(RGB( 12, 123, 220)))
    SetColorTableValue(_FOREGROUND_, _GREEN_      ,(RGB(153,  79,   0)))
    SetColorTableValue(_FOREGROUND_, _CYAN_       ,(RGB(  0, 108, 209)))
    SetColorTableValue(_FOREGROUND_, _RED_        ,(RGB(225, 190, 106)))
    SetColorTableValue(_FOREGROUND_, _MAGENTA_    ,(RGB( 64, 176, 166)))
    SetColorTableValue(_FOREGROUND_, _BROWN_      ,(RGB(230,  97,   0)))
    SetColorTableValue(_FOREGROUND_, _LIGHT_GRAY_ ,(RGB( 93,  58, 155)))
    SetColorTableValue(_FOREGROUND_, _DARK_GRAY_  ,(RGB( 26, 255,  26)))
    SetColorTableValue(_FOREGROUND_, _LIGHT_BLUE_ ,(RGB( 75,   0, 146)))
    SetColorTableValue(_FOREGROUND_, _LIGHT_GREEN_,(RGB(254, 254,  98)))
    SetColorTableValue(_FOREGROUND_, _LIGHT_CYAN_ ,(RGB(211,  95, 183)))
    SetColorTableValue(_FOREGROUND_, _LIGHT_RED_  ,(RGB(  0,  90, 181)))
    SetColorTableValue(_FOREGROUND_, _LIGHT_MAG_  ,(RGB(220,  50,  32)))
    SetColorTableValue(_FOREGROUND_, _YELLOW_     ,(RGB( 26, 133, 255)))
    SetColorTableValue(_FOREGROUND_, _WHITE_      ,(RGB(212,  17,  89)))

    SetColorTableValue(_BACKGROUND_, _BLACK_      ,(RGB(255, 194,  10)))
    SetColorTableValue(_BACKGROUND_, _BLUE_       ,(RGB( 12, 123, 220)))
    SetColorTableValue(_BACKGROUND_, _GREEN_      ,(RGB(153,  79,   0)))
    SetColorTableValue(_BACKGROUND_, _CYAN_       ,(RGB(  0, 108, 209)))
    SetColorTableValue(_BACKGROUND_, _RED_        ,(RGB(225, 190, 106)))
    SetColorTableValue(_BACKGROUND_, _MAGENTA_    ,(RGB( 64, 176, 166)))
    SetColorTableValue(_BACKGROUND_, _BROWN_      ,(RGB(230,  97,   0)))
    SetColorTableValue(_BACKGROUND_, _LIGHT_GRAY_ ,(RGB( 93,  58, 155)))
    SetColorTableValue(_BACKGROUND_, _DARK_GRAY_  ,(RGB( 26, 255,  26)))
    SetColorTableValue(_BACKGROUND_, _LIGHT_BLUE_ ,(RGB( 75,   0, 146)))
    SetColorTableValue(_BACKGROUND_, _LIGHT_GREEN_,(RGB(254, 254,  98)))
    SetColorTableValue(_BACKGROUND_, _LIGHT_CYAN_ ,(RGB(211,  95, 183)))
    SetColorTableValue(_BACKGROUND_, _LIGHT_RED_  ,(RGB(  0,  90, 181)))
    SetColorTableValue(_BACKGROUND_, _LIGHT_MAG_  ,(RGB(220,  50,  32)))
    SetColorTableValue(_BACKGROUND_, _YELLOW_     ,(RGB( 26, 133, 255)))
    SetColorTableValue(_BACKGROUND_, _WHITE_      ,(RGB(212,  17,  89)))
end

// Display a color chart, just below the top solid bars
proc chart2()
  integer xoff = 1, yoff = vWhereY() + 1
  integer x, y, attr, kwidth
  string junk[1], attr_str[1]

  junk = ""
  attr_str = ""
  GetStrAttrXY(WhereXAbs(), WhereYAbs(), junk, attr_str, 1)

  kwidth = (query(ScreenCols) - 2) / 16

  Set(Attr, 15)
  Window(xoff, yoff, 2 + kwidth * 16, yoff + 16 + 1)
  ClrScr()
  DrawBox(4, 7)
  FullWindow()
  vGotoxy(1 + xoff, 1 + yoff)
  for y = 0 to 15
    for x = 0 to 15
      attr = (x * 16) + y
      Set(attr, attr)
      PutStr(Format(attr:kwidth:'0':16))
    endfor
    vGotoxy(1 + xoff, y + 2 + yoff)
  endfor
end

// the top solid bars
proc ColorChart()
    integer xoff = 1, yoff = 1
    integer x, y, attr, kwidth

    if SHOW_FG_TEXT
        xoff = 1 yoff = 1
    else
        xoff = 0 yoff = 0
    endif
    kwidth = 4
    Set(Attr, 0)
    ClrScr()
    if SHOW_FG_TEXT
        Window(xoff, yoff, 2 + kwidth * 16, yoff + 16 + 1)
        DrawBox(4, 7)
    endif
    FullWindow()
    vGotoxy(1 + xoff, 1 + yoff)
    for y = 0 to 15
        for x = 0 to 15
            attr = (x * 16) + y
            Set(attr, attr)
            if SHOW_FG_TEXT
                PutStr(Format(attr:kwidth:'0':16))
            else
                PutStr(Format(' ':kwidth:' '))
            endif
        endfor
        vGotoxy(1 + xoff, y + 2 + yoff)
    endfor
end

// show the values
proc chart3(integer y)
    integer n, i, j

    n = 0
    y = 1
    for j = 1 to 16
        vgotoxy(68, y)
        for i = 1 to 16
            if i in 1
                set(Attr, (n & 0x0f) | (0x0ff & 0xf0))
            else
                Set(Attr, n & 0x0f)
            endif
            PutStr(Format(n:3:'0':16, ' '))
            n = n + 1
        endfor
        y = y + 1
    endfor

    Set(Attr, 0x9e)
end

proc chart4()
    integer x, y

    x = 32
    y = 37
    PutStrXY(x, y, "Hello, World! - BlockAttr", Query(BlockAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - CurrWinBorderAttr", Query(CurrWinBorderAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - CursorAttr", Query(CursorAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - CursorInBlockAttr", Query(CursorInBlockAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - EOFMarkerAttr", Query(EOFMarkerAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - HiliteAttr", Query(HiliteAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - LineNumbersAttr", Query(LineNumbersAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MenuBorderAttr", Query(MenuBorderAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MenuGrayAttr", Query(MenuGrayAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MenuSelectAttr", Query(MenuSelectAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MenuSelectGrayAttr", Query(MenuSelectGrayAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MenuSelectLtrAttr", Query(MenuSelectLtrAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MenuTextAttr", Query(MenuTextAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MenuTextLtrAttr", Query(MenuTextLtrAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MsgAttr", Query(MsgAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - OtherWinBorderAttr", Query(OtherWinBorderAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - StatusLineAttr", Query(StatusLineAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - TextAttr", Query(TextAttr))
    y = y + 1

    x = 66
    y = 37
    PutStrXY(x, y, "Hello, World! - IncompleteQuoteAttr", Query(IncompleteQuoteAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - NumberAttr", Query(NumberAttr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - Directive1Attr", Query(Directive1Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - Directive2Attr", Query(Directive2Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - Directive3Attr", Query(Directive3Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MultiLnDlmt1Attr", Query(MultiLnDlmt1Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MultiLnDlmt2Attr", Query(MultiLnDlmt2Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - MultiLnDlmt3Attr", Query(MultiLnDlmt3Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - SingleLnDlmt1Attr", Query(SingleLnDlmt1Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - SingleLnDlmt2Attr", Query(SingleLnDlmt2Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - SingleLnDlmt3Attr", Query(SingleLnDlmt3Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - Quote1Attr", Query(Quote1Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - Quote2Attr", Query(Quote2Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - Quote3Attr", Query(Quote3Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - ToEOL1Attr", Query(ToEOL1Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - ToEOL2Attr", Query(ToEOL2Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - ToEOL3Attr", Query(ToEOL3Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords1Attr", Query(KeyWords1Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords2Attr", Query(KeyWords2Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords3Attr", Query(KeyWords3Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords4Attr", Query(KeyWords4Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords5Attr", Query(KeyWords5Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords6Attr", Query(KeyWords6Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords7Attr", Query(KeyWords7Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords8Attr", Query(KeyWords8Attr))
    y = y + 1

    PutStrXY(x, y, "Hello, World! - KeyWords9Attr", Query(KeyWords9Attr))
    y = y + 1
end

datadef pallettes
/* 1*/    "Brite"
/* 2*/    "Brite2 (console colors)"
/* 3*/    "FED"
/* 4*/    "IBM CGA"
/* 5*/    "IBM OEM"
/* 6*/    "VGA"
/* 7*/    "EFTE"
/* 8*/    "Web2"
/* 9*/    "Light color"
/*10*/    "Putty"
/*11*/    ""
/*12*/    "(1)"
/*13*/    "AnsiPlus"
/*14*/    "Perfect 2"
/*15*/    "Term app"
/*16*/    ""
/*17*/    "TSE v4 default colors"
/*18*/    "HTML (from HTML book)"
/*19*/    "Dark"
/*20*/    "24 BIT"
/*21*/    "Retro"
/*22*/    "Xterm"
/*23*/    ""
/*24*/    "(2)"
/*25*/    "Web1"
/*26*/    "Xterm optiz"
/*27*/    "Css html"
/*28*/    "X windows"
/*29*/    "Solorized1"
/*30*/    "Solorized2"
/*31*/    "Solorized5"
/*32*/    "Solorized6"
/*33*/    "From StackOverflow"
/*34*/    "Perfect 3"
/*35*/    "Perfect 4"
/*36*/    "aap_dga16"
/*37*/    "soft_cfg"
/*38*/    "cga2"
/*39*/    "cga3"
/*40*/    "ZhongZhao"
/*41*/    "MathLogic"
/*42*/    "Campbell"
/*43*/    "StarMan"
/*44*/    "LinuxShell"
/*45*/    "MiniBasic"
/*46*/    "soft_cga2"
end

integer proc build_buffer(var integer pallet_buffer)
    integer nlines

    PushPosition()

    if pallet_buffer == 0
       pallet_buffer = CreateTempBuffer()
    else
       GotoBufferId(pallet_buffer)
       EmptyBuffer()
    endif

    PushBlock()
    InsertData(pallettes)
    PopBlock()

    nlines = NumLines() + 1
    PopPosition()
    return (nlines)
end

proc load_pallette(integer n)
    case n
        when  1 c_Brite()
        when  2 c_Brite2()
        when  3 FED()
        when  4 IBM_CGA()
        when  5 IBM_OEM()
        when  6 vga()
        when  7 efte()
        when  8 web2()
        when  9 light_color()
        when 10 putty()
        when 11
        when 12 _1()
        when 13 ANSIPLUS()
        when 14 perfect2()
        when 15 term_app()
        when 16
        when 17 TSE_default()
        when 18 c_HTML()
        when 19 c_Dark()
        when 20 _24_BIT()
        when 21 retro()
        when 22 xterm()
        when 23
        when 24 _2()
        when 25 web1()
        when 26 xterm_opiz()
        when 27 css_html()
        when 28 x_windows()
        when 29 solorized1()
        when 30 solorized2()
        when 31 solorized5()
        when 32 solorized6()
        when 33 from_stackoverflow()
        when 34 perfect3()
        when 35 perfect4()
        when 36 aap_dga16()
        when 37 soft_cga()
        when 38 cga2()
        when 39 cga3()
        when 40 ZhongZhao()
        when 41 MathLogic()
        when 42 Campbell()
        when 43 Starman()
        when 44 LinuxShell()
        when 45 MiniBasic()
        when 46 soft_cga2()
    endcase
end

constant kSchemes = -1

keydef list_keys
<cursorup>    if not up() endfile() begline() endif
<cursordown>  if not down() begfile() endif
<Alt s>       EndProcess(kSchemes)
end

string list_footer[] = "{Enter}-Load {Alt-S}-Schemes {Esc}-Exit"

proc list_startup()
    UnHook(list_startup)
    enable(list_keys)
    ListFooter(list_footer)
end

integer pallette_buffer

proc main()
    integer y, menu_only, cur_id, first_time, line, result, menu_rows, schemes, esc_pressed, special_effects

    special_effects = Query(SpecialEffects)
    menu_only = trim(Query(MacroCmdLine)) <> ""

    if not menu_only
        PopWinOpen(1,_BLUE_       , Query(ScreenCols), Query(ScreenRows),_BLACK_      , "", 0)
    endif

    cur_id = GetBufferId()
    menu_rows = build_buffer(pallette_buffer)
    first_time = true

    schemes = false
    esc_pressed = false
    repeat
        if schemes
            ExecMacro("colors")
            schemes = false
        endif
        y = 0
        if not menu_only
            BufferVideo()
            ColorChart()
            chart2()
            y = vWhereY() + 2
            chart3(y)
            chart4()
            UnBufferVideo()
        endif

        GotoBufferId(pallette_buffer)
        if first_time
            BegFile()
            first_time = false
        endif

        if menu_only
            if menu_rows > Query(ScreenRows)
                y = 1
            else
                y = Query(ScreenRows) / 2 - (menu_rows / 2 + 3)
            endif
        elseif y == 0
            y = 4
        endif

        Hook(_LIST_STARTUP_, list_startup)
        Set(X1, 4)
        Set(Y1, y)
        BufferVideo()

        Set(SpecialEffects, special_effects | _SHOW_OEM_CHARS_)
        result = lList("Palette Schemes", Length(list_footer), Query(ScreenRows) - y, 0)
        Set(SpecialEffects, special_effects)

        line = CurrLine()
        GotoBufferId(cur_id)
        case result
            when kSchemes
                schemes = true
            when TRUE
                BufferVideo()
                load_pallette(line)
                GotoBufferId(pallette_buffer)
                if not Down()
                   BegFile()
                endif
                GotoBufferId(cur_id)
                UpdateDisplay(_ALL_WINDOWS_REFRESH_)
                UnBufferVideo()
                VirtualToRealScreen()
            // escape pressed
            when FALSE
                esc_pressed = true
        endcase
        UnBufferVideo()

    until esc_pressed

    if not menu_only
        PopWinClose()   // be sure to close popwin!
    endif
end

integer palette
public proc setpalette()
    palette = palette + 1
    while palette in 11,16,23, /*no good*/ 1,2,3,4,5,6,7,8,9,_LIGHT_GREEN_,_LIGHT_RED_  ,_LIGHT_MAG_  ,_YELLOW_     ,_WHITE_      , 18, 22, 24,25, 26, 27, 28, 29, 30, 31, 32, 33, 34, 36, 37, 39
        palette = palette + 1
        if palette > 39
            palette = 1
        endif
    endwhile
    if palette > 39
        palette = 1
    endif

    BufferVideo()
    load_pallette(palette)
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)
    UnBufferVideo()
    VirtualToRealScreen()
    Message(palette)
end
<f3> setpalette()
