/****************************************************************************
  List Files placed in the editor's internal ring of files.

  Change Log

    29 May 2023 SEM - removed the buffervideo()/unbuffervideo() statements, so that it
        works with Carlo Hogeveen's wonderful picklist macro.

    07 Oct 2020 SEM - Show filenames in compact (_USE_HOME_PATH_) form.
                    - Allow the entry key to also exit the list.

     9 Dec 2006 SEM - Added some help text.

     8 Dec 2006 Rob den Heijer - Added "Save Filelist" feature.

    20 Oct 2006 SEM - Add CopyToWinClip [ctrl c], [ctrl ins] and Copy to
            editor clipboard [grey+] the current filename.

    June 10, 1998 SEM - Add "sorted filelist feature".  Change the
            SORT_FILE_LIST constant (below) to get a sorted or unsorted
            list.

    Feb   6, 1997 SEM - Use #ifdef's so it will compile with DOS and win32
            versions.

    Oct   8, 1996 SEM - win32 version

    Sep   5, 1995 SEM - Initial standalone version - was previously a part
            of the .ui file.

  Implementation notes:

    We depend on <Escape> returning 0, <Enter> returning > 0, and additional
    commands returning < 0 (kQUITFILE, kSAVEFILE, currently)

 ***************************************************************************/

constant SORT_FILE_LIST = TRUE

integer max_width

constant MAXPATH = 255

helpdef listopen_help
    Title="List Open Help"
    x=5
    y=5
   ""
   " [Alt E]     Make the selected file the current file             "
   " [Del]       Quit the selected file                              "
   " [Alt R]     Rename the selected file                            "
   " [Alt S]     Sort the File List                                  "
   " [Alt W]     Save(Write) the current file                        "
   " [Ctrl C]    Copy the selected filename to the Windows clipboard "
   " [Ctrl Ins]  Copy the selected filename to the Windows clipboard "
   " [Ctrl V]    View current file                                   "
   " [Grey +]    Copy the selected filename to the Editors clipboard "
   " [Alt L]     Save the list of files to listopen.txt              "
   " [Esc]       Return to editing                                   "
   ""
end

constant
    kQUITFILE       = -1,
    kSAVEFILE       = -2,
    kTOGGLESORT     = -3,
    kCOPYTOWINCLIP  = -4,
    kCOPY           = -5,
    kRENAME         = -6

string
    TITLE[] = "Buffer List",
    FILELIST_FOOTER[] = "{F1}-Help {Enter}-EditFile {Del}-QuitFile {Alt-S}-SortList {Alt-W}-WriteFile {Escape}-Cancel"

string unloaded_char[1]
integer entry_key

forward keydef FileListKeys

proc SmartViewFile()
    integer id
    string fn[_MAX_PATH_]

    fn = GetText(2, _MAX_PATH_)

    id = GetBufferId()

    Disable(FileListKeys)
    ExecMacro("f -z " + QuotePath(fn))
    Enable(FileListKeys)

    GotoBufferId(id)
end

menu sort_menu()
    "Sort by &Path"
    "Sort by &Filename + Extension"
    "Sort by &Extension"
    "&Cancel"
end

proc sort_by(integer sortby, integer max_width)
    integer msglevel
    string orig_filename[MAXPATH], filename[MAXPATH]

    orig_filename = GetText(2, max_width)
    // sort the list of files
    msglevel = Set(MsgLevel, _WARNINGS_ONLY_)

    if sortby in 2,3
        begfile()
        repeat
            if sortby == 2
                filename = SplitPath(GetText(2, max_width), _NAME_|_EXT_)
            else
                filename = SplitPath(GetText(2, max_width), _EXT_)
            endif
            gotocolumn(max_width + 1)
            inserttext(filename)
        until not down()
    endif

    PushBlock()
    if sortby == 1
        MarkColumn(1, 2, NumLines(), max_width + 1)
    else
        MarkColumn(1, max_width + 1, NumLines(), max_width + 1 + MAXPATH)
    endif
    Sort(_IGNORE_CASE_)

    if sortby in 2,3
        KillBlock()
    endif

    PopBlock()

    Set(MsgLevel, msglevel)
    lFind(orig_filename, "g")
end

proc sort_list()
    Disable(FileListKeys)
    Set(X1, Query(PopWinX1))
    case sort_menu()
        when 0, 4
        when 1    sort_by(1, max_width)
        when 2    sort_by(2, max_width)
        when 3    sort_by(3, max_width)
    endcase
    Enable(FileListKeys)
end

proc FileListHelper()
    if Enable(FileListKeys)
        ListFooter(FILELIST_FOOTER)
    endif
    Unhook(FileListHelper)
    BreakHookChain()
end

proc GetFile(var string fn, var integer id)
    id = GetBufferId()
    fn = SqueezePath(CurrFilename(), _USE_HOME_PATH_)
end

proc mCopy(string fn)
    integer id

    PushBlock()
    PushPosition()
    id = GetWorkBuffer()
    InsertText(fn)
    BegLine()
    MarkChar()
    EndLine()
    MarkChar()
    Copy()
    PopPosition()
    PopBlock()
    FreeWorkBuffer(id)
end

/*------------------------------------------------------------------------
  First character of list buffer does not belong with the filename
  Remove it before saving the list buffer.
 ------------------------------------------------------------------------*/
proc SaveListOpen()
    integer id

    PushBlock()
    MarkColumn(1, 2, NumLines(), 255)
    PushLocation()
    id = CreateTempBuffer()
    CopyBlock()
    PopBlock()
    SaveAs()
    PopLocation()
    AbandonFile(id)
end

keydef FileListKeys
    <Alt e>     EndProcess(TRUE)
    <Del>       EndProcess(kQUITFILE)
    <Alt r>     EndProcess(kRENAME)
    <Alt w>     EndProcess(kSAVEFILE)
    <Alt s>     sort_list()
    <Ctrl c>    EndProcess(kCOPYTOWINCLIP)
    <Ctrl Ins>  EndProcess(kCOPYTOWINCLIP)
    <Ctrl v>    SmartViewFile()
    <Grey+>     EndProcess(kCOPY)
    <alt l>     AddHistoryStr("listopen.txt", _EDIT_HISTORY_) SaveListOpen()
    <F1>        QuickHelp(listopen_help)
end

proc AfterGetKey()
    if Query(key) == entry_key
        Set(Key, <Escape>)
    endif
end

/**************************************************************************
  Assumes we are in the 'starting' buffer, *not* the filelist buffer
 **************************************************************************/
integer proc BuildFileList(integer filelist, var integer n_changed, var integer n_loaded)
    string fn[MAXPATH], indicator[1]
    integer maxl

    maxl = 0
    n_changed = 0
    n_loaded = 0
    EmptyBuffer(filelist)
    // build the list of files
    do NumFiles() + (BufferType() <> _NORMAL_) times
        fn = SqueezePath(CurrFilename(), _USE_HOME_PATH_)
        if length(fn)
            maxl = Max(maxl, Length(fn))
            indicator = ' '
            if (Query(BufferFlags) & _LOADED_) == 0
                indicator = unloaded_char
            else
                n_loaded = n_loaded + 1
                if FileChanged()
                    indicator = '*'
                    n_changed = n_changed + 1
                endif
            endif
            AddLine(indicator + fn, filelist)
        endif
        NextFile(_DONT_LOAD_)
    enddo
    GotoBufferId(filelist)
    BegFile()
    maxl = Max(maxl + 2, Length(TITLE) + 6)
    maxl = Max(maxl, Length(FILELIST_FOOTER) - 8)
    return (maxl)
end

string proc DisplayFileList(integer filelist, var integer start_file, var string start_fn)
    integer ok, quit, saved, maxl, n_changed, n_loaded, not_loaded, ren_file_changed
    string fn[MAXPATH], new_fn[MAXPATH]

    maxl = BuildFileList(filelist, n_changed, n_loaded)
    max_width = maxl + 1
    if SORT_FILE_LIST
        sort_by(1, max_width)
    endif

    // display the list, process options
    BufferType(_NORMAL_)
    //BufferVideo()
    lFind(start_fn, "gw")
    repeat
        Hook(_LIST_STARTUP_, FileListHelper)
        if entry_key <> -1
            Hook(_AFTER_GETKEY_, AfterGetKey)
        endif
        not_loaded = NumLines() - n_loaded
        FileChanged(False)
        maxl = max(maxl, length(FILELIST_FOOTER) - 12)
//        Set(X1, 1)
        ok = List(Format(TITLE; "-"; NumLines(); iif(NumLines() == 1, "file", "file(s)"); "[",

            iif(n_loaded, Str(n_loaded) + " loaded ", ""),
            iif(n_changed, Str(n_changed) + " changed ", ""),
            iif(not_loaded, Str(not_loaded) + " not loaded", "")
            ), maxl)

        if entry_key <> -1
            UnHook(AfterGetKey)
        endif
        quit = FALSE
        saved = FALSE

        fn = GetText(2, CurrLineLen() - 1)
        case ok
            when kQUITFILE
                if GetText(1, 1) == '*'
                    if EditThisFile(QuotePath(fn)) and QuitFile()
                        quit = TRUE
                    endif
                elseif AbandonFile((GetBufferId(fn)))
                    quit = TRUE
                endif

                GotoBufferId(filelist)      // back to the filelist buffer
                if quit
                    KillLine()
                    if start_fn == fn
                        start_fn = ""
                        start_file = 0
                        if NumLines()
                            GotoBufferId(start_file)
                            PrevFile(_DONT_LOAD_)
                            GetFile(start_fn, start_file)
                            GotoBufferId(filelist)
                        endif
                    endif
                endif

            when kSAVEFILE
                if EditThisFile(QuotePath(fn))
                    if (FileChanged() or not FileExists(QuotePath(fn))) and SaveFile()
                        saved = TRUE
                    endif
                endif

                GotoBufferId(filelist)      // back to the filelist buffer
                if saved
                    BegLine()
                    InsertText(' ', _OVERWRITE_)
                endif

            when kCOPYTOWINCLIP
                CopyToWinClip(QuotePath(ExpandPath(fn)))
            when kCOPY
                mCopy(QuotePath(ExpandPath(fn)))
            when 0
                fn = ""
            when kRENAME
                if EditThisFile(QuotePath(fn))
                    if ChangeCurrFilename()
                        new_fn = SqueezePath(CurrFilename(), _USE_HOME_PATH_)
                        ren_file_changed = FileChanged()
                        if fn == start_fn
                            start_fn = new_fn
                            start_file = GetBufferId()
                        endif
                        GotoBufferId(filelist)
                        lFind(fn, "gw")
                        BegLine()
                        KillToEOL()
                        InsertText(iif(ren_file_changed, "*", " "))
                        InsertText(new_fn)
                    endif
                endif

                GotoBufferId(filelist)      // back to the filelist buffer
        endcase
    until NumLines() == 0 or ok >= 0
    //UnBufferVideo()
    BufferType(_SYSTEM_)

    return (fn)
end

integer filelist

string proc get_unloaded_char()
    string fnt[80] = ""
    integer ps, flags

    GetFont(fnt, ps, flags)
    return (iif((flags & _FONT_OEM_), Chr(250), Chr(183)))
end

proc main()
    integer start_file
    string fn[MAXPATH], start_fn[MAXPATH]

    entry_key = Query(key)
    if isTypeAbleKey(entry_key) or entry_key == <Enter>
        entry_key = -1
    endif

    unloaded_char = get_unloaded_char()
    start_fn = ""
    GetFile(start_fn, start_file)

    if filelist == 0
        filelist = CreateTempBuffer()
        if filelist == 0
            warn("Cannot create filelist")
            return ()
        endif
    endif

    GotoBufferId(start_file)

    fn = DisplayFileList(filelist, start_file, start_fn)

    if fn <> ""
        EditThisFile(QuotePath(fn))
    elseif start_fn <> ""
        EditThisFile(QuotePath(start_fn))
    endif

    EmptyBuffer(filelist)
end

