/***********************************************************************

    FileMgr     File manager macro for TSE Pro

    Author:     SemWare

    Date:       0.01    01/12/95    Richard Blackburn
                                    Initial version (worked in PickList)

                0.02                Cleaned up for release

                0.03    09/10/95    Richard Blackburn
                                    Added command line support to allow just
                                    the zip manager to be called

                0.04    11/29/95    Force Shell to change to viewed
                                    directory.

                0.05    06/27/96    Various bug fixes

                0.06    10/08/96    Convert to WIN32/CONSOLE
                                    Special thanks to Chris Antos for lots
                                    of help in converting over to WIN32.
                                    f.dat is now stored in the editors
                                    load directory.

                0.07    10/17/96    Allow directories to be deleted via
                                    <del> instead of having to select rmdir
                                    from menu.  If the dir is not empty, it
                                    is not deleted.

                                    zip/unzip fail on files without at least
                                    a '.' for the extension.  Force '.' if
                                    no extension.

                0.08    11/05/96    rewrite LoadIt() rtn to take advantage
                                    of 2.5+ features, and fix long-filename
                                    loading bug.

                0.09    11/15/96    Allow starting where left-off if "-r"
                                    switch passed

                0.10    11/21/96    Use short path for compression utilities
                                    Thanks to Chris Antos for the fix.

                0.11    12/20/96    Handle additional zip file formats.

                0.12     1/09/97    NT uses 128 as NORMAL attribute (w95
                                    uses 0).  We should account for this in
                                    RemoveUnwantedFiles().

                0.13     1/30/97    For some reason, passing "copy src dst
                                    >temp" does not always work under
                                    WIN-NT.  Rework the copyfile logic to
                                    use the builtin WIN32 CopyFile() API.

                0.14     2/19/97    Show current path in rename/move commands.

                0.15     4/08/97    Fix remove directories.  Add remove tagged
                                    files from list.  Clean up code so it
                                    will still compile and run with the DOS
                                    version.

                0.16    10/22/97
                                    - Bug fix to remove directory.
                                    - Changed some menu items to be more descriptive
                                    - Reworked copyto
                                    - Added 'start' current file

                0.17     3/26/98    Add "Set As Current Working Dir" under Dir menu.

                0.18    12/02/98    Refresh list when changing directories.

                0.19     1/04/99    Lots of work getting it to work better on NT

                0.20     2/17/99    Fix spaces-in-filename problem in SetupPicklist()
                                    (reported by Guy Rouillier - thanks!)

                0.21  July 27, 2000 For .zip file management: Attempt to find where
                                    the filename starts, based on the "Name" column-headings.
                                                                      "----"
                0.22  Sep 15, 2000  SEM - fix long standing speed-search bug.

                0.23  May  2, 2001  SEM - if TEMP path has spaces, compressed file handling
                                    fails - fixed.

                0.24  May 23, 2001  SEM - when using the -r option, if the path no longer
                                    exists, instead of exiting, use the current dir.

                0.25  Feb 2004 Ross Boyd - Fix the diskspace routines to correctly work on
                                    really large drives.

                0.26  Nov 21 2006 SEM - more work recognizing filenames in various zipper
                                    listings.

                0.27  April 2007  SEM - add the ability to:
                                    Move files over existing files
                                    Delete non-empty directories

                0.28  Nov 21 2007 SEM - add Where command to Execute menu.

                0,29  Mar 04 2008 SEM - add "Copy Path to Windows
                                    Clipboard" to Execute menu.

                0.30  28 Feb 2009 SEM - Fix problem of not finding tee32.exe in
                                    certain cases where TSELOADDIR was used.
                                    Reported (along with a suggested fix) by
                                    Ross Barnes.

                0.31 06 Dec 2010  SEM - Use builtin StartPgm instead of DLL ShellExecute.

                0.32 22 Feb 2011  SEM - Remove <.> assignment, so that the '.' can be used
                                    in speed search.  Thanks to Sjoerd Rienstra for the
                                    suggestion.  <alt cursorleft> can be used instead.

                0.33 31 Dec 2012  SEM - info-zip, when unzipping, files inside zips with '['
                                    look like regular expressions.  Escape them, by changing
                                    "[" to "[[]".

                0.34 30 Jun 2015  SEM - update internal viewer.

                0.35 14 May 2016  SEM - show correct filesize for large files when creating
                                      filelist.tse
                0.36 27 Oct 2018  SEM - add -c option
                0.37 06 Mar 2020  SEM - remove ViewFile(), user ViewIt() instead, cause it works better.
                0.38 22 Jun 2021  SEM - fix bug in copy/moving tagged files. It was trying to copy the
                                    same file more than once.
                0.39 22 Oct 2021  SEM - add "Change to current working directory" menu item
                0.40 29 Aug 2022  SEM - add expand ~  to -z option
                0.41 19 Nov 2022  SEM - quote names for zips
                0.42 10 Dec 2022  SEM - add a zip header; more quoting for zips; add ctrl-space for tagging
                0.43 25 Jan 2023  SEM - fixed -z to work on <unnamed-> buffers.
                0.44 08 Feb 2023  SEM - fix bug in ListFile(): wasn't displaying unloaded files.
                0.45 14 Aug 2023  SEM - when editing a file from a .zip, append original name to the temp name



        Overview:

        A simple file manager for TSE Pro.  Main features include Copying,
        deleting, moving, and renaming files. Included are commands to set
        the time and date of a file, and a ZIP file manager.

    Usage:

        f [-z zipfile]
        f [-r]
        f [filespecs]
        f [-c]

        where:

            -z zipfile causes zipfile to be viewed; if not a zip, the passed file is viewed
            -r causes the last directory viewed to be used (if f has not
               been purged since the last invocation).
            filespecs causes f to work on that specification
            -c causes the directory of the currently loaded file
               to be used.

            Otherwise, the user is prompted for the directory to view.

    Assumptions:

        The entire macro assumes that the buffers are created in the
        WhenLoaded(), and no one else touches them.

        Handling of filenames with spaces:

        All user generated filenames must quote filenames with spaces.

    How to add a new compressor:

      1) add a new signature string - search for:
        // compressor signature strings

      2) add exception code as needed to GetInZipName()
        see the GetInZipName() function for details

      3) add exception code as needed to ViewFileInZip()
        see the ViewFileInZip() function for details

      4) add new compressor code to isCompressFile()
      5) add new compressor code to CompresserMenu()
      6) add new compressor code to Compresser()
      7) add new compressor code to ConfigurationMenu()
      8) edit f.dat as: g32 -b-1 f.dat
         make a copy of last compressor entry, appending it to the
         end of the file
         save and quit the file

      9) use File, Update Configuration to add the compressor program

    Copyright 1995-2003 SemWare Corporation.  All Rights Reserved Worldwide.

***********************************************************************/

constant FN_COMP = 0x4

#define MAXPATH 255

dll "<kernel32.dll>"
    integer proc GetLastError() : "GetLastError"
    integer proc GetDiskFreeSpace(string root_dir:cstrval,
                    var integer SectorsPerCluster,
                    var integer BytesPerSector,
                    var integer NumberOfFreeClusters,
                    var integer TotalNumberOfClusters) : "GetDiskFreeSpaceA"
    integer proc GetDiskFreeSpaceEx(
             string      lpDirectoryName:cstrval,
             var integer lpFreeBytesAvailableToCaller,
             var integer lpTotalNumberOfBytes,
             var integer lpTotalNumberOfFreeBytes) : "GetDiskFreeSpaceExA"

    integer proc LoadLibrary(string libraryname:cstrval) :"LoadLibraryA"

    integer proc FreeLibrary(integer dllhandle) :"FreeLibrary"

    integer proc GetProcAddress(integer dllhandle, string procname:cstrval) :"GetProcAddress"

    integer proc CreateFile(string fn:cstrval, integer mode, integer sec, integer attr, integer flags, integer junk1, integer junk2) : "CreateFileA"
    integer proc CloseHandle(integer handle) : "CloseHandle"
    integer proc DosDateTimeToFileTime(integer date:word,
            integer time:word, string FILETIME:StrPtr) : "DosDateTimeToFileTime"
    integer proc LocalFileTimeToFileTime(string in_time:StrPtr, string out_time:StrPtr) : "LocalFileTimeToFileTime"
    integer proc SetFileTime(integer handle,
//            string FILETIME_Creation:StrPtr,
            integer FILETIME_Creation,
//            string FILETIME_LastAccess:StrPtr,
            integer FILETIME_LastAccess,
            string FILETIME_LastWrite:StrPtr) : "SetFileTime"
end

#define SW_HIDE             0
#define SW_SHOWNORMAL       1
#define SW_NORMAL           1
#define SW_SHOWMINIMIZED    2
#define SW_SHOWMAXIMIZED    3
#define SW_MAXIMIZE         3
#define SW_SHOWNOACTIVATE   4
#define SW_SHOW             5
#define SW_MINIMIZE         6
#define SW_SHOWMINNOACTIVE  7
#define SW_SHOWNA           8
#define SW_RESTORE          9
#define SW_SHOWDEFAULT      10
#define SW_MAX              10

#define FILE_FLAG_BACKUP_SEMANTICS  33554432
#define OPEN_EXISTING               3
#define GENERIC_WRITE	            0x40000000

// compressor signature strings
string
        arcsig[] =  "",
        arjsig[] =  "`",
        lhasig[] =  "-lh",
        rarsig[] =  "Rar!" + Chr(26) + Chr(7) + Chr(0),
        s7zsig[] =  Chr(0x37) + Chr(0x7A) + Chr(0xBC) + Chr(0xAF) + Chr(0x27) + Chr(0x1C),
        tgzsig[] =  "",
        zipsig[] =  "PK",
        zipsig2[]=  "PK00PK",         // alternate zip signature
        zipsig3[]=  "PK",             // alternate zip signature
        zoosig[] =  "ܧ"

constant
        TAG_COL = 1,                   // column in pick list of tag

         MARK_CHAR = 251,               // character to use for marking
         INZIP = 0x01,                  // view inside of zip file
         ZIPFILE = 0x10,                // viewing zip file
         FORCE = 0x01,                  // flag for AddFiles()
         _MARK = 0x01,                  // Force mark
         _UNMARK = 0x02,                // Force un mark
         _TOGGLE = 0x10,                // toggleing mark
        BIGN = 20

// ticks for dblclk expiration
constant DBLCLK_THRESHHOLD = 4

integer cd_his, mvcpy_his,
    lastLeft = 0                    // last click (dblclk processing)

/***********************************************************************
    compression information
***********************************************************************/

//
// ******************************* Compress *******************************
//

string  compress[8],                    // compression program
        add_comp[10],                   // additional command to compress files
        uncompress[8],                  // decompression program
        add_uncomp[10],                 // additional command to decompress files
        compressview[10],               // view contents of compress file
        ext_to_con[10],                 // extract file to console
        comp_over[10],                  // extract with overwite
        del_file[10],                   // del from compress file
        compressorname[3]               // Name of compressor for title

integer compressfn_col,                 // file name column in ZIP list
        compressfn_endcol,              // file name column in ZIP list
        compress_sig_offset             // offset into file of signature

string  flist_path[MAXPATH],            // path we are working with
        fileslisted[80] = "*.*",        // file specs
        binaryload[60],                 // load files in binary mode
        hexload[60],                    // load files in bin/hex mode
        exe[40],                        // Files that can be executed
        datfile[128],                   // name of .DAT file
        lpickfilesortorder[8],          // sort order
        tee32[_MAX_PATH_]               // so we can run tee32.exe
//        lasttree[3]                     // last tree viewed

integer viewid,                         // id of filelist
        viewinzipid,                    // view inside of compress file
        copyid,                         // id of copy temp file
        listid,                         // id of list buffer
        storeid,                        // id of tracking file
        scrapbuffer,                    // misc functions use this
        validdrives,                    // list of valid drives
        configurationvars,              // Compression strings
        firstfileid,                    // id of firstfile loaded
        searchattribs = -1,             // attributes to display - need to be init in case no config file
        taggedfiles,                    // number of tagged files
        realtextattr,                   // Save the TextAttr
        pickfileflags,                  // flags for pick file
        skipdrive                       // last drive to skip in GetVolume()
string  taggedsize_st[BIGN]

forward proc WriteLineInt(integer variable)
forward proc AskDate(var integer month, var integer day, var integer year)
forward keydef FileManagerKeys
forward helpdef FileManagerHelp

string view_file_footer[] = "{<Ctrl V>}-View file {<Ctrl U>}-Unzip file {<Alt E>}-Edit file {<Del>}-Delete file  {<F1>}-Help"

//************ WIN32 specific functions ****************************

string proc mQuotePath(string path)
    string s[MAXPATH]
    integer p

    s = path
    repeat
        p = Pos('"', s)
        if p
            s = DelStr(s, p, 1)
        endif
    until p == 0
    return (QuotePath(s))
end

string proc TempFile()
    string dir[MAXPATH]

    dir = GetEnvStr("TEMP")
    if dir == ""
        dir = GetEnvStr("TMP")
        if dir == ""
            dir = CurrDir()
        endif
    endif
    return (MakeTempName(dir))
end

/*****************************************************************************
  Add this function to the editor
 *****************************************************************************/
integer proc MakeDirectory(string s)
    integer attribute
    string dir[MAXPATH]

    dir = RemoveTrailingSlash(s)
    attribute = FileExists(dir)

    if attribute & _DIRECTORY_
        return (TRUE)
    endif

    if attribute == 0
        if MkDir(dir)
            return (TRUE)
        else
            return (Warn("Can't mkdir:", s))
        endif
    endif

    return (Warn("Can't mkdir over existing file:", s))
end

/*****************************************************************************
  Add this function to the editor
 *****************************************************************************/
forward integer proc remove_dir(string s, var integer remove_all_dir, var integer remove_all_read_only)

// assumed dir is a directory when called
// assumed we can't have a dir that ends in ".", except for "." and ".." entries
integer proc isDotEntry(string dir)
    return (RightStr(dir, 1) == ".")
end

integer proc find_any_files(string dir)
    integer h, found = False

    h = FindFirstFile(AddTrailingSlash(dir) + "*.*", -1)
    if h <> -1
        repeat
            if (FFAttribute() & _DIRECTORY_) == 0
                found = TRUE
            elseif not isDotEntry(FFName())
                found = True
            endif
        until found or not FindNextFile(h, -1)
        FindFileClose(h)
    endif
    return (found)
end

integer proc recursively_remove_dir(string dir, var integer level, var integer remove_all_dir, var integer remove_all_read_only)
    integer h, attribute, ok = True
    string fn[_MAX_PATH_]

    level = level + 1
    h = FindFirstFile(AddTrailingSlash(dir) + "*.*", -1)
    if h <> -1
        repeat
            fn = AddTrailingSlash(dir) + FFName()
            attribute = FFAttribute()
            if (attribute & _DIRECTORY_) == 0
                if not EraseDiskFile(fn)
                    if (attribute & _READ_ONLY_) == 0
                        ok = Warn("Unknown error removing:", fn)
                    elseif not remove_all_read_only
                        case MsgBoxEx("File is READ_ONLY",
                                "Delete `" + fn + "' anyway?",
                                "[&Yes];[&No];[Yes to &All];[&Cancel]")
                            when 0, 4   ok = False
                            when 1
                            when 2
                                ok = False
                            when 3
                                remove_all_read_only = True
                        endcase
                    endif
                    if ok and not SetFileAttr(fn, attribute & ~ _READ_ONLY_)
                        ok = Warn("Unknown error removing READ-ONLY attribute:", fn)
                    endif
                    if ok and not EraseDiskFile(fn)
                        ok = Warn("Unknown error removing:", fn)
                    endif
                endif
            elseif not isDotEntry(fn)
                ok = remove_dir(fn, remove_all_dir, remove_all_read_only)
            endif
        until not ok or not FindNextFile(h, -1)
        FindFileClose(h)
    endif
    level = level - 1
    return (ok and remove_dir(dir, remove_all_dir, remove_all_read_only))
end

integer proc remove_dir(string s, var integer remove_all_dir, var integer remove_all_read_only)
    integer attribute, level = 0
    string dir[_MAX_PATH_]

    dir = RemoveTrailingSlash(s)
    attribute = FileExists(dir)
    if (attribute & _DIRECTORY_) == 0
        return (Warn("Not a directory:", s))
    endif

    // can't remove the current directory!
    if EquiStr(dir, RemoveTrailingSlash(CurrDir()))
        ChDir("..")
    endif
    if RmDir(dir)
        return (TRUE)
    endif

    if not find_any_files(dir)
        return (Warn("Unknown error on rmdir ", s))
    endif

    if not remove_all_dir
        case MsgBoxEx("Directory is not empty",
                "Delete `" + dir + "' anyway?",
                "[&Yes];[&No];[Yes to &All];[&Cancel]")
            when 0, 4   return (False)
            when 1      // just continue
            when 2
                return (False)
            when 3
                remove_all_dir = True
        endcase
    endif

    return (recursively_remove_dir(dir, level, remove_all_dir, remove_all_read_only))
end

integer proc RemoveDirectory(string dir)
    integer remove_all_dir = False, remove_all_read_only = False
    return (remove_dir(dir, remove_all_dir, remove_all_read_only))
end

string proc ExpandPathNoWild(string path0, integer respect_case)
    string path[_MAX_PATH_] = ExpandPath(path0, respect_case)
    if isWildPath(path)
        path = SplitPath(path, _DRIVE_|_PATH_)
    endif
    return (path)
end

/**************************************************************************

 **************************************************************************/

// Do NOT change these declarations
// as each pair is reserving space for a 64bit integer
integer  lpFreeBytesAvailableToCallerLo
        ,lpFreeBytesAvailableToCallerHi
        ,lpTotalNumberOfBytesLo
        ,lpTotalNumberOfBytesHi
        ,lpTotalNumberOfFreeBytesLo
        ,lpTotalNumberOfFreeBytesHi

// Shift a 64bit unsigned string right
string proc ShiftRightU64(string U64,integer shift)
    string   a[BIGN],result[BIGN] = Format("":BIGN:"0")
    integer  remainder,idx

    // Pad with leading zeroes
    a = Format(U64:BIGN:"0")

    // We do the shift by scanning the string from left to right
    // dividing by 2 and carrying leftovers.
    do shift times
        remainder = 0
        for idx = 1 to length(a)
            result[idx] = Str( (Val(a[idx])+remainder*10)/2  )
            remainder   = Val(a[idx]) & 0x1
        endfor
        a = result
    enddo

    if not shift   // In case nothing is shifted
        result = a
    endif

    // Get the start of the actual number (skip leading zeroes)
    for idx = 1 to length(result)
        if result[idx] <> "0"
            break
        endif
    endfor

    // Handle case where number is zero
    if idx > length(result)
        return("0")
    endif

    return(result[idx..length(result)])
end

// Do simple addition of two numeric strings
string proc AddUnsignedStrings(string a, string b)
    integer  idx,carry,work
    string   a2[BIGN],b2[BIGN],result[BIGN]

    // First, pad strings with leading zeroes
    a2     = Format(a:sizeof(a2):"0")
    b2     = Format(b:sizeof(b2):"0")
    result = Format("":sizeof(result):"0")

    // Point idx to the rightmost element and
    // work backwards doing the addition and
    // carrying the overflow...
    idx   = length(a2)
    carry = 0
    while idx
        work        = Val(a2[idx]) + Val(b2[idx]) + carry
        carry       = work > 9         // Boolean result is either 1 or 0
        result[idx] = Str(work mod 10)
        idx         = idx - 1
    endwhile

    // Get the start of the actual number (skip leading zeroes)
    for idx = 1 to length(result)
        if result[idx] <> "0"
            break
        endif
    endfor

    // Handle case where number is zero
    if idx > length(result)
        return("0")
    endif

    return(result[idx..length(result)])
end

// works - but only if a > b
string proc SubUnsignedStrings(string a, string b)
    integer idx, borrow, difference
    string a2[BIGN], b2[BIGN], result[BIGN]

    // First, pad strings with leading zeroes
    a2     = Format(a:sizeof(a2):"0")
    b2     = Format(b:sizeof(b2):"0")
    result = Format("":sizeof(result):"0")

    borrow = 0
    for idx = length(a2) downto 1
        difference = val(a2[idx]) - val(b2[idx]) - borrow
        borrow = 0
        if difference < 0
            difference = difference + 10
            borrow = 1
        endif
        result[idx] = str(difference)
    endfor

    // Get the start of the actual number (skip leading zeroes)
    for idx = 1 to length(result)
        if result[idx] <> "0"
            break
        endif
    endfor

    // Handle case where number is zero
    if idx > length(result)
        return("0")
    endif

    return(result[idx..length(result)])
end

// Returns a string value representation of an
// unsigned 32 bit integer.
string proc UnsignedValue(integer i)
    // 10 is sufficient for (2^32)-1
    string   t[10]

    t = Str(i & 0x7FFFFFFF)            // Mask off sign bit
    if i & 0x80000000                  // If signed bit was set then
        t = AddUnsignedStrings(t,"2147483648") // add unsigned value 2^31
    endif
    return(t)
end

// Returns a string value representation of the
// HIGH 32 bits of a 64 bit unsigned integer
string proc UnsignedHiValue(integer i)
    // BIGN is sufficient for a 64 bit number
    string   t[BIGN] = UnsignedValue(i)

    // Adding the number to itself 32 times
    // will convert it to a Hi DWORD value
    do 32 times
        t = AddUnsignedStrings(t,t)
    enddo
    return(t)
end

// Detect if a DLL library contains a given ProcName
integer proc DLLProcExists(string dllname,string procname)
    integer  dllhandle,ProcExists
    ProcExists = 0
    dllhandle = LoadLibrary(dllname)
    if dllhandle
        ProcExists = GetProcAddress(dllhandle,procname)
        FreeLibrary(dllhandle)
    endif
    return(ProcExists)
end

// This wrapper routes the call to the correct API call
// depending on the version of the Win32 platform
// The results are returned as strings to overcome
// the 32bit integer limitation.
// NB: GetDiskFreeSpaceA does not support drives > 2Gig, however,
//     GetDiskFreeSpaceExA does, but isn't available on older OS's.
proc loDiskSpace(string drive, var string size, var string free)
    integer  sectors, bytes, num_free, total

    // Default return values
    size = "0"
    free = "0"

    // Detect which API call to use...
    if DllProcExists("kernel32.dll","GetDiskFreeSpaceExA")
        if GetDiskFreeSpaceEx(AddTrailingSlash(drive),lpFreeBytesAvailableToCallerLo,lpTotalNumberOfBytesLo,lpTotalNumberOfFreeBytesLo)

            size = AddUnsignedStrings( UnsignedValue(lpTotalNumberOfBytesLo)
                    ,UnsignedHiValue(lpTotalNumberOfBytesHi))

            // Return the actual free space
            free = AddUnsignedStrings( UnsignedValue(lpTotalNumberOfFreeBytesLo)
                    ,UnsignedHiValue(lpTotalNumberOfFreeBytesHi))

            // free = AddUnsignedStrings( UnsignedValue(lpFreeBytesAvailableToCallerLo),UnsignedHiValue(lpFreeBytesAvailableToCallerHi))
        endif
    else
        if GetDiskFreeSpace(AddTrailingSlash(drive), sectors, bytes, num_free, total)
            size = UnsignedValue(sectors * bytes * total)
            free = UnsignedValue(sectors * bytes * num_free)
        endif
    endif
end

/**************************************************************************

 **************************************************************************/

/***********************************************************************
    Low level DOS call to actually set the date and time
***********************************************************************/
integer proc LoSetDateTime(string fn, integer month, integer day, integer year,
                            integer hours, integer mins, integer secs)
    integer handle, time, date, ok
    string FILETIME[8] = "        "    // needs to be at least 64 bits long

    time = (hours shl 11) | (mins shl 5) | (secs / 2)
                                        //  Bitfields for file time:
                                        //   15-11  hours (0-23)
                                        //   10-5   minutes
                                        //   4-0    seconds/2
    date = ((year - 1980) shl 9) | (month shl 5) | day
                                        //  Bitfields for file date:
                                        //   15-9   year - 1980
                                        //   8-5    month
                                        //   4-0    day
    if not DosDateTimeToFileTime(date, time, FILETIME)
        return (Warn("Error on DosDateTimeToFileTime()"))
    endif
    // the two 0's are FILETIME_Creation and FILETIME_LastAccess, but to pass
    // 0 they have to have been declared as integers.
//    handle = fOpen(fn, _OPEN_WRITEONLY_|_OPEN_DENYALL_)// open handle
    handle = CreateFile(fn,
            GENERIC_WRITE,
            0,
            0,
            OPEN_EXISTING,
            FILE_FLAG_BACKUP_SEMANTICS,
            0
    )      // GENERIC_WRITE and OPEN_EXISTING
    if handle == -1
        return (Warn("Error ", GetLastError(), " opening file ", fn))
    endif
    LocalFileTimeToFileTime(FILETIME, FILETIME)
    ok = SetFileTime(handle, 0, 0, FILETIME)
    if not ok
        Warn("Error ", GetLastError(), " setting time, handle:", handle)
    endif
//    fClose(handle)
    CloseHandle(handle)
    return (ok)
end

// The documentation person demanded a name change in 2.6.  Oh well, what can I say!
integer proc MatchFilenames(string fn, string mask)
    return (MatchFilename(fn, mask))
end

/**************************************************************************
  Convert a binary WORD-length string to an integer
 **************************************************************************/
integer proc StrWord(string s)
    return (Asc(s[2]) * 256 + Asc(s[1]))
end

integer proc SetCurrDriveDir(string path)
    return (ChDir(path))
end

//************ End of system specific code ************************

integer proc ChDriveDir(string dir, integer verbose)
    if dir[2] == ':' and not LogDrive(dir[1])
        return (iif(verbose, Warn("Cannot log drive ", dir[1]), FALSE))
    endif
    if ChDir(dir)
        return (TRUE)
    endif
    return (iif(verbose, Warn("Cannot CD to ", dir), FALSE))
end

/***********************************************************************
    Is the current file a directory?  Return 0 for FALSE, or 1 for
    TRUE.  The return value is used in GetFullPath() to adjust the
    starting point of file name.
***********************************************************************/
integer proc isDir()
    return (PBAttribute() & _DIRECTORY_)
end

string proc PBExt()
    return (Lower(SplitPath(PBName(), _EXT_)))
end

/***********************************************************************
    Low level routines.  These use intr() to get/pass information.
    There are also several "helper" routines to work with the intr()
    calls.
***********************************************************************/

/***********************************************************************
    Dos() command that clears the screen first, while saving and
    restoring the screen.
***********************************************************************/
integer proc _Dos(string dos_cmdline, integer flags)
    integer ok
    // save the current screen
    PopWinOpen(1,1,Query(ScreenCols), Query(ScreenRows), 0, "", 0)
    ClrScr()                    // clear it
    ok = Dos(dos_cmdline, flags)
    PopWinClose()
    return (ok)
end

/**************************************************************************
  Remove files not matching users specified attributes.
  Do not remove NORMAL files.

  NORMAL files have no attribute under w95.
  Under NT, NORMAL files can also have an attribute of 128
 **************************************************************************/
proc RemoveUnWantedFiles(integer attribs)
    if (attribs & (_READONLY_|_HIDDEN_|_SYSTEM_|_VOLUME_|_DIRECTORY_|_ARCHIVE_)) == (_READONLY_|_HIDDEN_|_SYSTEM_|_VOLUME_|_DIRECTORY_|_ARCHIVE_)
        return ()
    endif
    loop
        if (PBAttribute() & attribs) == 0 and (PBAttribute() & ~128) //  <> 0 and PBAttribute() <> 128
            KillLine()              // remove unwanted files
        elseif not Down()
            break
        endif
    endloop
end

/***********************************************************************
    Add the files to the pick list.

    Assumes
        1) you are in the viewid buffer when this routine is called.
        2) global variable path has been set
        3) always displays dirs

    returns whether or not it adds files to list
***********************************************************************/
integer proc AddFileSpec(string files, integer attribs)
    integer ok, empty

    empty = (CurrLine() == 1) AND (CurrLineLen() == 0) // is file empty?

    EndFile()
    PushPosition()
    ok = BuildPickBufferEx(flist_path + files,
            _READONLY_|_HIDDEN_|_SYSTEM_|_ARCHIVE_|_DIRECTORY_|_VOLUME_)
    PopPosition()
    if NOT empty                     // already had something in the file
        if NOT Down()                // and we did not add anything else
            return(ok)               // return
        endif
    endif
    RemoveUnWantedFiles(attribs)
    return(ok)
end

/***********************************************************************
    Remove the directory listings from the viewer
***********************************************************************/
proc HideDirs()
    PushPosition()
    BegFile()
    RemoveUnWantedFiles(_READONLY_|_HIDDEN_|_SYSTEM_|_VOLUME_|_ARCHIVE_)
    PopPosition()
end

/***********************************************************************
    Put the "menu" on the top of the screen
***********************************************************************/
proc PutTopLine()
    ListHeader("{F}ile  {D}ir  {T}ag  {P}rint  {S}ort  {E}xecute  {U}til  {H}elp " + Format("["+SqueezePath(flist_path, _USE_HOME_PATH_)+"] Files: "+ Str(NumLines()):Query(ScreenCols)-60))
end

/***********************************************************************
    Return everything beyond the last \ in the string passed.

    60:261
***********************************************************************/
string proc GetNameAndExt(string s)
    return (SplitPath(s, _NAME_|_EXT_))
end

proc FixUpSavedPath(var integer cl, var integer cr, var string dirorder,
                    string dir, integer getting)
    integer cid

    cid = GetBufferId()
    GotoBufferId(storeid)                        // go to storage file

    if lFind(dir, "^wg")                        // look up old dir
        if NOT getting
            BegLine()
            InsertText(Format(dir:-80, cl:-10, cr:-10), _OVERWRITE_) //update it
        else
            cl = Val(GetText(81,10))            // get line
            cr = Val(GetText(91,10))            // get row
            dirorder = GetText(101,3)           // get sort order
        endif
    else                                        // not found??
        EndFile()
        if getting
            cl = 1                                  // set line to 1
            cr = 1                                  // set row to 1
        endif
        AddLine(Format(dir:-80, cl:-10, cr:-10, dirorder))       // add it
    endif
    GotoBufferId(cid)
end

/***********************************************************************
    Set up pick list that is viewed

    Set up the pick list.  Look up the directory in the storage buffer,
    place the cursor ont the same line, and row, and same sort order.

    ??? Far too long/complex a routine for the comments here ???
***********************************************************************/
integer proc SetUpPickList(string fspec)
    integer rc, cr,  cl, i, pickfileflags
    string  files[13], pickfilesortorder[8], dirorder[8],
            filespecs[MAXPATH], tf[13]

//    pickfileflags = Set(PickFileFlags, _ADD_DIRS_|_ADD_SLASH_|_DIRS_AT_TOP_)
    pickfileflags = Set(PickFileFlags, Query(PickFileFlags) | _ADD_DIRS_)
    filespecs = fspec
    if Length(GetNameAndExt(filespecs)) == 0
        filespecs = mQuotePath(fspec + fileslisted)
    endif
    filespecs[Length(filespecs) + 1] = " "
    cr = CurrRow()                               // save current row
    cl = CurrLine()                              // save current line
    files = SplitPath(ExpandPath(GetFileToken(fspec, 1)), _NAME_|_EXT_) // get just the filespec
    dirorder = Query(PickFileSortOrder)              // save sort order

    FixUpSavedPath(cl, cr, dirorder, flist_path, FALSE) // FALSE put info
    flist_path = SplitPath(ExpandPath(GetFileToken(fspec, 1)), _DRIVE_|_PATH_) // find new path
    FixUpSavedPath(cl, cr, dirorder, flist_path, TRUE)  // TRUE Get info

    GotoBufferId(viewid)                        // go to view file
    EmptyBuffer()                               // clear it out

    tf = GetNameAndExt(filespecs)               // get just the filespec

    i = Pos(" ", tf)
    tf =  SubStr(GetNameAndExt(ExpandPath(GetFileToken(filespecs, 1))), 1, iif(i, i - 1, Length(tf)))
    if tf <> files
        tf = GetNameAndExt(fileslisted)
        files = GetFileToken(tf, 1)            // get just the filespec
        filespecs = fileslisted
    endif

    rc = AddFileSpec(files, searchattribs | _DIRECTORY_)

    Set(PickFileFlags, 0)
    fileslisted = files
    for i = 2 to NumFileTokens(filespecs)
        if Length(GetFileToken(filespecs, i))
            files = GetNameAndExt(GetFileToken(filespecs, i))
            fileslisted = fileslisted + " " + files
            rc = AddFileSpec(files, searchattribs) OR rc
        endif
    endfor
    fileslisted[Length(fileslisted)+1] = " "

    Set(PickFileFlags, pickfileflags)           // Restore flags
    pickfilesortorder = Set(PickFileSortOrder, lpickfilesortorder)// Set the sort order
    Sort(_PICK_SORT_)                           // sort it
    GotoLine(cl)                                // goto saved line
    ScrollToRow(cr)                             // goto saved row
    Set(PickFileSortOrder, pickfilesortorder)   // restore PickFileSortOrder
    PutTopLine()
    MarkColumn(1, _PICK_BUFFER_FN_COL_, NumLines(), _PICK_BUFFER_FN_COL_ + 255)
    return(rc)
end

integer proc RebuildPickListThisPath(string path)
    return (SetupPickList(QuotePath(path)))
end

integer proc RebuildPickList()
    return (RebuildPickListThisPath(flist_path))
end

/***********************************************************************
    Return the expanded file name from the list, appending the current
    path to the beginning of the file name.
***********************************************************************/
string proc GetFullPath()
    return (flist_path + PBName())
end

/***********************************************************************
  The following menus and proc's are to allow you to change the file
  attributes.
***********************************************************************/

/***********************************************************************
    Set the attribute byte in the buffer, by oring in the new
    attribute.
***********************************************************************/
proc mSet(integer attribute, integer search)
    integer t

    if search
        searchattribs = searchattribs ^ attribute
    else
        PushPosition()
        attribute = PBAttribute() ^ attribute
        SetFileAttr(GetFullPath(), attribute)  // Set the attribute
        t = Set(PickFileFlags, _NONE_)
        BuildPickBufferEx(GetFullPath(), _READONLY_|_ARCHIVE_|_SYSTEM_|_HIDDEN_)
        Set(PickFileFlags, t)
        RollUp()
        KillLine()
        PopPosition()
    endif
end

/***********************************************************************
    Check for the attribute, and return either a check mark, or a
    nul string.
***********************************************************************/
string proc CheckAttr(integer i, integer search)
    if search
        return(iif(searchattribs & i, Chr(MARK_CHAR), ""))
    endif
    return(iif(PBAttribute() & i, Chr(MARK_CHAR), ""))
end

menu AttrMenu()
    '&Archive'   [CheckAttr(_ARCHIVE_, FALSE):2]   ,mset(_ARCHIVE_  , FALSE)  , DontClose
    '&Hidden'    [CheckAttr(_HIDDEN_, FALSE):2]    ,mset(_HIDDEN_, FALSE)     , DontClose
    '&Read Only' [CheckAttr(_READ_ONLY_, FALSE):2] ,mset(_READ_ONLY_, FALSE)  , DontClose
    '&System'    [CheckAttr(_SYSTEM_, FALSE):2]    ,mset(_SYSTEM_, FALSE)     , DontClose
end

/***********************************************************************
    Set attributes of the current file.  If you pass in an attribute
    set it, and return.  Else prompt for the attribute to set.
***********************************************************************/
proc ChangeAttr(integer attrib)
    if isDir()                          // if directory
        return ()                       // do nothing
    endif

    if attrib                           // if passed an attribute
        mSet(attrib, FALSE)             // Set it
    else
        AttrMenu()
    endif
end

/***********************************************************************
    These routines work on the tagged status of files
***********************************************************************/

/***********************************************************************
    Return the size of all the files in the current display list
***********************************************************************/

string proc TotalSize()
    string s[BIGN]
    string stot[BIGN] = "0"

    PushPosition()
    BegFile()

    repeat
        s = AddUnsignedStrings(UnsignedValue(PBSize()), UnsignedHiValue(PBSizeHigh()))
        stot = AddUnsignedStrings(s, stot)
    until not Down()
    PopPosition()
    return (stot)
end

/***********************************************************************
    Tag/UnTag all of the files by filling the tag column with the
    character passed.
***********************************************************************/
proc TagAll(string fillwith)
    PushBlock()                         // save any block
    UnmarkBlock()                       // clear blocks
    MarkColumn(1, 1, NumLines(), 1)     // mark column 1 (tag column)
    FillBlock(fillwith)                 // fill with character passed
    PopBlock()                          // restore blocks
    if fillwith == " "
        taggedfiles = 0
        taggedsize_st = "0"
    else
        taggedfiles = NumLines()
        taggedsize_st = TotalSize()
    endif
end

/***********************************************************************
    Check to see if any of the files are marked.
***********************************************************************/
integer proc Tagged()
    return(lFind(Chr(MARK_CHAR), "g^"))
end

integer proc isMarked()
    return(CurrChar() == MARK_CHAR)
end

/***********************************************************************
    Toggle Mark on Current file.
***********************************************************************/
proc MarkIt(integer flag)
    string s[BIGN]

    BegLine()

    if isMarked() AND ((flag == _TOGGLE) OR (flag == _UNMARK))
        taggedfiles = taggedfiles - 1
        s = AddUnsignedStrings(UnsignedValue(PBSize()), UnsignedHiValue(PBSizeHigh()))
        taggedsize_st = SubUnsignedStrings(taggedsize_st, s)
    elseif (NOT isMarked()) AND ((flag == _MARK) OR (flag == _TOGGLE))
        taggedfiles = taggedfiles + 1
        s = AddUnsignedStrings(UnsignedValue(PBSize()), UnsignedHiValue(PBSizeHigh()))
        taggedsize_st = AddUnsignedStrings(taggedsize_st, s)
    endif
    InsertText(iif((isMarked() AND flag == _TOGGLE) OR flag == _UNMARK, " ",
                        Chr(MARK_CHAR)), _OVERWRITE_)     // tag the file
    BegLine()
end

proc UnMarkFile()
    MarkIt(_UNMARK)
end

integer proc MarkIfNotTagged(integer chkdir)
    if NOT Tagged()
        if chkdir AND isDir()
            return(FALSE)
        endif
        MarkIt(_MARK)
    endif
    return(TRUE)
end

/***********************************************************************
    Invert the tags
***********************************************************************/
proc InvertTags()
    PushPosition()
    BegFile()
    repeat
        MarkIt(_TOGGLE)
    until NOT Down()
    PopPosition()
end

/***********************************************************************
    Menu to find attribute to tag by
***********************************************************************/
menu TagAttrMenu()
    title = "Tag file with:"
    '&Read Only'
    '&Archive'
    '&System'
    '&Hidden'
end

/***********************************************************************
    Call TagAttrMenu() to get the attribute to use in tagging the
    files.  Go to the begining of the file, and check the attribute
    of each file.
***********************************************************************/
proc TagAttributes()
    integer tagattr
    case TagAttrMenu()                  // find the attribute to search for
        when 1      tagattr =  _READ_ONLY_
        when 2      tagattr =  _ARCHIVE_
        when 3      tagattr =  _SYSTEM_
        when 4      tagattr =  _HIDDEN_
        otherwise   return()
    endcase
    PushPosition()                      // save positiion
    BegFile()                           // goto begining of the file
    repeat
        if PBAttribute() & tagattr // if the current file has attribute
            MarkIt(_MARK)                    // tag it
        endif
    until NOT Down()
    PopPosition()
end

/***********************************************************************
  Tag by either name or ext, as specified by flag
***********************************************************************/
proc SearchAndTag(string name, integer flag)
    if NumLines() == 0
        return ()
    endif

    PushPosition()
    BegFile()
    repeat
        if MatchFilenames(SplitPath(PBName(), flag), name)
            MarkIt(_MARK)
        endif
    until not Down()
    PopPosition()
end

/***********************************************************************
    Tag files by name
***********************************************************************/
proc TagName()
    string name[80] = ""                 // name to look for

    if Ask("Name to tag by:", name) AND Length(name) // ask for name
        SearchAndTag(name, _NAME_)
    endif
end

/***********************************************************************
    Tag files by extension
***********************************************************************/
proc TagExtension()
    string ext[32] = ""                  // name to look for

    if Ask("Ext to tag by: ['.' ok]", ext) AND Length(ext) // ask for ext
        if ext[1] <> '.'
            ext = '.' + ext
        endif
        SearchAndTag(ext, _EXT_)
    endif
end

/***********************************************************************
    Tag files by the date you enter.  Uses AskDate() to get the date
    you are searching for.
***********************************************************************/
proc TagDate()
    integer month, day, year, date

    AskDate(month, day, year)
    date = ((year - 1980) shl 9) | (month shl 5) | (day)
    PushPosition()
    repeat
        if PBDate() == date
            MarkIt(_MARK)
        endif
    until NOT Down()
    PopPosition()
end

/***********************************************************************
    Tag files by size.  If you enter a negative number, tags files
    <= Abs(size), positive number tags files >= size
***********************************************************************/
proc TagSize()
    integer size, s
    string ssize[10]

    ssize = ""
    size = 0
    if AskNumeric("Tag by size [neg means files <= size, pos means files >= size] ", ssize) AND Length(ssize)
        size = Val(ssize)
        PushPosition()
        BegFile()
        repeat
            if NOT isDir()
                s = PBSize()
                if (size > 0 AND s >= size) OR (size < 0 AND s <= Abs(size))
                    MarkIt(_MARK)
                endif
            endif
        until NOT Down()
        PopPosition()
    endif
end

/***********************************************************************
    Delete the file.  First check to see if it is a read only file.
    If it is, prompt to delete it, or unmark it.  Then, if you can
    erase it, delete the current line in the file.
***********************************************************************/
proc DelOneFile()
    integer atrib
    string fn[MAXPATH]

    UnMarkFile()
    atrib = PBAttribute()
    fn = GetFullPath()

    if atrib & _VOLUME_
        Warn("Can't remove file: ", fn)
    elseif atrib & _DIRECTORY_
        if RemoveDirectory(fn)
            KillLine()
        endif
    else
        if atrib & _READ_ONLY_
            case YesNo("File is _READ_ONLY_ delete anyway?")
                when 1      ChangeAttr(_READ_ONLY_)     // turn off read only
                otherwise   return()                    // unmark and return
            endcase
        endif
        if EraseDiskFile(fn)
            KillLine()
        endif
    endif
end

/***********************************************************************
    Delete all marked files.  If there are not any marked file, mark
    the current file.
***********************************************************************/
proc DelIt()
    integer cr

    cr = CurrRow()
    PushPosition()
    if YesNo(iif(Tagged(), "Delete marked files?",
                            Format("Delete : ", SqueezePath(GetFullPath(),
                            Query(ScreenCols) - 13)))) == 1
        MarkIfNotTagged(FALSE)
        while Tagged()                      // while you find marked files
            DelOneFile()                    // delete the current file
        endwhile
    endif
    PopPosition()
    if CurrLine() > NumLines()          // if on <*** End of File ***>
        Up()
    endif
    GotoRow(cr)
    PutTopLine()
end

/***********************************************************************
    Is the file extension in the binary load list?
***********************************************************************/
integer proc isBinaryFile(string ext)
    return(Length(ext) AND Pos(ext + ".", binaryload))
end

/***********************************************************************
    Is the file extension in the hex load list?
***********************************************************************/
integer proc isHexFile(string ext)
    return(Length(ext) AND Pos(ext + ".", hexload))
end

/***********************************************************************
    Load the current file or the marked files, if it is a dir,
    change to it.
***********************************************************************/
proc LoadIt()
    string fn[MAXPATH], dir[MAXPATH]
    integer id,         // id of original current file
        tid,            // id of last file added to ring
        recsize,        // binary recsize for hex/binary files
        display_mode    // display mode for hex files

    dir = ""
    tid = 0
    id = GetBufferId()
    MarkIfNotTagged(FALSE)
    while Tagged()              // while tagged files
        UnMarkFile()
        if isDir()
            if Length(dir) == 0
                dir = GetFullPath() // remember found dir
            endif
        else
            recsize = 0
            display_mode = _DISPLAY_TEXT_
            if isBinaryFile(PBExt())
                if isHexFile(PBExt())
                    display_mode = _DISPLAY_HEX_
                    recsize = 16
                else
                    recsize = 64
                endif
            endif
            fn = GetFullPath()          // we must save path, since we change buffers

            GotoBufferId(tid)           // go to last file added to ring
            tid = AddFileToRing(fn, recsize, display_mode)  // add this file to ring
            GotoBufferId(id)            // go back to file list

            if tid and firstfileid == 0
                firstfileid = tid
            endif
        endif
    endwhile
    if Length(dir)
        RebuildPickListThisPath(dir)
    elseif firstfileid
        EndProcess(1)
    endif
end

proc edit_file_here()
    string fn[MAXPATH] = flist_path
    PushKey(<End>)
    if AskFilename("Create New File here:", fn, 0, _EDIT_HISTORY_)
        EditThisFile(fn)
        EndProcess(-1)
    endif
end

/****************************************************************************
   Helper routines for Sort on the Options menu.
 ***************************************************************************/
integer sort_flags      // used for Sorting

string proc ShowSortFlag()
    return (iif(sort_flags & 1, "Descending", "Ascending"))
end

proc ToggleSortFlag(integer which)
    if sort_flags & which
        sort_flags = sort_flags & ~ which
    else
        sort_flags = sort_flags | which
    endif
end

/***********************************************************************
    Sort the pick list.  This routine uses the SortMenu() to determine
    what keys to sort on:

    Option      Primary key             Secondary key           Third key
    ------      -----------             -------------           ---------
    NAME        Name                    Extension
    EXTENSION   Extension               Name
    SIZE        Size                    Name
    DATE        Date                    Time                    Name
    ATTRIBUTE   Attribute               Name
    Also uses the sort_flags variable to determine order.
***********************************************************************/
proc SortIt(integer sortby)
    string  pickfilesortorder[8],
            flags[3]

    pickfilesortorder = Query(PickFileSortOrder)
    flags = ""
    case sortby
        when 0  return()
        when 1  flags = "NE"   // Name
        when 2  flags = "EN"   // Ext
        when 3  flags = "SN"   // Size
        when 4  flags = "DTN"  // Date
        when 5  flags = "AN"   // Attribute
    endcase
    if NOT sort_flags          // Decending order?
        flags = lower(flags)   // if so, lower case our flags
    endif
    lpickfilesortorder = flags
    Set(PickFileSortOrder, flags)  // set the sort order
    Sort(_PICK_SORT_)
    BegFile()
//    GotoBufferId(storeid)
//    if lFind(path, "^wg")
//        GotoColumn(101)
//        InsertText(Format(Query(PickFileSortOrder):-3), _OVERWRITE_)
//    endif
    GotoBufferId(viewid)
    Set(PickFileSortOrder, pickfilesortorder)
end

/***********************************************************************
    How do you want to sort the list?
***********************************************************************/
menu SortMenu()
    History
    Command = SortIt(MenuOption())
    Title = 'Sort by:'
    "&Name"
    "&Extension"
    "&Size"
    "&Date"
    "&Attribute"
    "", , Divide
    "Sort &Order"   [ShowSortFlag() : 10], ToggleSortFlag(1), DontClose
end

/***********************************************************************
    These routines control the handling of compress files.
***********************************************************************/
string zipfilename[MAXPATH]                  // file name for use in zip routines

proc ShowNumber()
    vGotoxy(6, 0)
    PutStr(Format(CurrLine():6))
end

proc ListFile2(string title, integer flags)
    integer btype, ncols

    if flags <> ZIPFILE
        Hook(_AFTER_NONEDIT_COMMAND_, ShowNumber)
    endif
    btype = BufferType(_NORMAL_)        // normal -- turn tab expanding on :)

    ncols = Max(LongestLineInBuffer() + 2, length(view_file_footer))
    ncols = Min(ncols, Query(ScreenCols))

    lList(title, ncols, Query(ScreenRows), _ENABLE_SEARCH_|_ENABLE_HSCROLL_)
    BufferType(btype)
    if flags <> ZIPFILE
        Unhook(ShowNumber)
    endif
end

/***********************************************************************
    List the file passed, using the title passed
    Only load the file into the temp buffer if needed
***********************************************************************/
proc ListFile(string title, string fn, integer flags)
    integer cid, currwinborderattr, buf_id, bmode, dmode

    currwinborderattr = Set(CurrWinBorderAttr, Query(MenuBorderAttr))
    cid = GotoBufferId(iif(flags & INZIP, viewinzipid, listid))
    if cid
        buf_id = 0
        EmptyBuffer()
        bmode = BinaryMode()
        dmode = DisplayMode()
        if isHexFile(SplitPath(fn, _EXT_))
            bmode = BinaryMode(16)
            dmode = DisplayMode(_DISPLAY_HEX_)
        elseif isBinaryFile(SplitPath(fn, _EXT_))
            bmode = BinaryMode(64)
        else
            // don't load the file if already loaded
            buf_id = GetBufferId(fn)
            GotoBufferId(buf_id)
            if (Query(BufferFlags) & _LOADED_) == 0
                GotoBufferId(cid)
                buf_id = 0
            endif
        endif

        if buf_id or InsertFile(fn, _DONT_PROMPT_)
            ListFile2(title, flags)
        else
            Warn("Error viewing file")
        endif
        if buf_id == 0
            BinaryMode(bmode)
            DisplayMode(dmode)
        endif
        GotoBufferId(cid)               // return to pick list
    endif
    if flags & ZIPFILE
        EraseDiskFile(fn)               // delete temp file
    endif
    Set(CurrWinBorderAttr, currwinborderattr)
end

integer proc CompressOk(string cmd, var string tmp_fn, integer is_compress, integer maybe_prompt)
    integer ok, rc, flags
    string msg[80], s[255]

    if tmp_fn == ""
        tmp_fn = GetShortPath(TempFile())
    endif
    flags = iif(WhichOS() == _WINDOWS_, _RUN_DETACHED_|_START_HIDDEN_, _CREATE_NEW_CONSOLE_|_START_HIDDEN_)
    if maybe_prompt
        s = ""
        flags = _DONT_PROMPT_
    else
        s = ">" + tmp_fn
    endif
    msg = iif(is_compress, compress, uncompress)
    if is32BitApp(GetToken(cmd, ' ', 1))
        ok = lDos(tee32, cmd + s, flags)
    else
        ok = Dos(cmd + s, flags)
    endif

    rc = DosIOResult()
    if not ok or rc <> 0
        if not EquiStr(tmp_fn, "nul")
            EraseDiskFile(tmp_fn)
        endif
        return (Warn("Error ", rc, " running: ", msg))
    endif
    return (TRUE)
end

/***********************************************************************
  List the directory of a compressed file.
***********************************************************************/
proc ViewZipFile()
    string tf[MAXPATH] = ""

    if CompressOk(Format(uncompress, " ", compressview, " ", GetShortPath(zipfilename)), tf, FALSE, FALSE)
        ListFile(zipfilename, tf, ZIPFILE)
    endif
end

/***********************************************************************
    return the file name from the zip view file
***********************************************************************/
string proc GetInZipName()
    integer p, line
    string fn[_MAX_PATH_]
    if compressfn_col < 0
        // if compressfn_col is negative, use absolute value as index for
        // GetToken.  various ZIP managers have slightly different column
        // formats, and this provides a workaround.
        return (GetToken(GetText(1, 255), " ", -compressfn_col))
    endif

/**************************************************************************
  Handle this guy somehow...

    pkzip 32 bit version:

  Length  Method    Size  Ratio    Date    Time    CRC-32   Attr    Name
  ------  ------    ----- -----    ----    ----   --------  ----    ----
       0 Stored         0  0.0% 10/19/2004  2:33p 00000000 ----  bat/

    info-zip:

 Length   Method    Size  Ratio   Date   Time   CRC-32    Name
--------  ------  ------- -----   ----   ----   ------    ----
   73649  Defl:X    17358  76%  11-15-06 09:58  e337d8f6  hash2.c
 **************************************************************************/
    if compressorname == "ZIP"
        PushPosition()
        line = CurrLine()
        if lFind("Length[ \t]*Method[ \t]*Size[ \t]*.*\cName", "gx") and Down()
            repeat until CurrChar() == Asc(' ') or not Left()
            repeat until CurrChar() == Asc('-') or not Left()
            GotoLine(line)
            repeat until CurrChar() == Asc(' ') or not Right()
            repeat until CurrChar() <> Asc(' ') or not Right()
            p = CurrPos()
            PopPosition()
            fn = (Trim(GetText(p, 255)))
            fn = StrReplace("[", fn, "[[]")
            return (QuotePath(fn))
        endif
        PopPosition()
    elseif compressorname == "TGZ"
        return (GetText(1, 255))
    endif

//    return (Trim(GetText(compressfn_col, compressfn_endcol - compressfn_col + 1)))
    return (GetToken(GetText(compressfn_col, 255), " ", 1))
end

/**************************************************************************
  List a file in a compressed file
 **************************************************************************/
proc ViewFileInZip(string fn)
    string tf[MAXPATH] = ""

    if compressorname == "ZIP" and ((CurrLine() <= 2) OR (CurrLine() >= (NumLines() - 1)))
        return()
    endif
// 07/09/98 SEM: I don't think we need to get the short name of the file in the zip???
    if CompressOk(Format(uncompress, " ", ext_to_con, " ", GetShortPath(zipfilename)," ", QuotePath(fn)), tf, FALSE, FALSE)
        ListFile(fn, tf, ZIPFILE | INZIP)
    endif
end

/**************************************************************************
  List a file in a compressed file
 **************************************************************************/
proc EditFileInZip(string fn)
    string tf[MAXPATH] = ""

    if compressorname == "ZIP" and
                ((CurrLine() <= 2) OR (CurrLine() >= (NumLines() - 1)))
        return()
    endif
// 07/09/98 SEM: I don't think we need to get the short name of the file in the zip???
    if CompressOk(Format(uncompress, " ", ext_to_con, " ", GetShortPath(zipfilename)," ", QuotePath(fn)), tf, FALSE, FALSE)
        PushPosition()
        firstfileid = EditFile(tf)
        EraseDiskFile(tf)
        if lFind("Inflating: .* \<to console\>", "x")
            MarkLine(1, CurrLine() + 1)
            KillBlock()
            BegLine()
        endif
        // change \temp\12345.txt to \temp\12345-fn.fn+fn.ext
        tf = SplitPath(tf, _DRIVE_|_NAME_) + "-" + SplitPath(fn, _NAME_|_EXT_)
        ChangeCurrFilename(tf, _DONT_EXPAND_)
        FileChanged(FALSE)
        PopPosition()
        EndProcess(-1)
    endif
end

/**************************************************************************
  Extract a file from a compressed file
 **************************************************************************/
proc UnZipFileFromZip(string fn)
    integer ow
    string savepath[MAXPATH] = CurrDir()
    string tf[MAXPATH] = "nul"

    if (CurrLine() <= 2) OR (CurrLine() >= (NumLines() - 1))
        return()
    endif

    ow = FALSE

    if FileExists(flist_path + fn)
        case YesNo("File Exists, overwrite?")
            when 0, 3   return()
            when 1      ow = TRUE
        endcase
    endif

    ChDriveDir(flist_path, 0)

    if ow
        CompressOk(Format(uncompress, " ", add_uncomp, " ", comp_over, " ", GetShortPath(zipfilename), " ", GetShortPath(fn), " ", flist_path), tf, FALSE, FALSE)
    else
        // unzip and let it prompt, restore screen - should call _dos
        CompressOk(Format(uncompress, " ", add_uncomp, " ", GetShortPath(zipfilename), " ", GetShortPath(fn), " ", flist_path), tf, FALSE, TRUE)
    endif

    ChDriveDir(savepath, 0)
end

helpdef ZipHelp
    "<Del>      Delete from compress file"
    "<Enter>    View File"
    "<Ctrl v>   View File"
    "<Ctrl u>   Unzip file"
end

proc mZipLeftBtn()
    case MouseHotSpot()
        when _MOUSE_MARKING_
            GotoMouseCursor()
            // dblclk at midnight will fail
            if GetClockTicks() - lastLeft <= DBLCLK_THRESHHOLD
                // view it
                ViewFileInZip(GetInZipName())
                return()
            endif
            lastLeft = GetClockTicks()
        otherwise
            ProcessHotSpot()
    endcase
end

proc DeleteFileInZip()
    string tf[MAXPATH] = "nul"
    if CompressOk(Format(compress, " ", del_file, " ", zipfilename, " ", GetInZipName()), tf, TRUE, FALSE)
        KillLine()
    endif
end

keydef ZipKeys
    <F1>        QuickHelp(ZipHelp)
    <Del>       DeleteFileInZip()
    <Enter>     ViewFileInZip(GetInZipName())
    <Ctrl v>    ViewFileInZip(GetInZipName())
    <Ctrl u>    UnZipFileFromZip(GetInZipName())
    <Alt e>     EditFileInZip(GetInZipName())

    <LeftBtn>   mZipLeftBtn()
end

proc ZipListStartUp()
    UnHook(ZipListStartUp)
    Enable(ZipKeys)
    ListFooter(view_file_footer)
end

// zip fails if passed a file without at least a dot for extension
// Force a dot if no extension
string proc ForceDot(string fn)
    return (iif(Length(SplitPath(fn, _EXT_)), fn, fn + '.'))
end

proc ViewModifyZipFile(string fn)
    zipfilename = fn
    zipfilename = ForceDot(zipfilename)
    zipfilename = QuotePath(zipfilename)
    Hook(_LIST_STARTUP_, ZipListStartUp)
    ViewZipFile()

    UnHook(ZipListStartUp)

    if firstfileid
        EndProcess(-1)
    elseif NOT Length(fn)
        RebuildPickList()
    endif
end

proc ZipFiles()
    string tf[MAXPATH] = "nul"
    zipfilename = flist_path
    PushKey(<End>)
    if AskFilename("compress file:", zipfilename, 0) AND Length(zipfilename)
        MarkIfNotTagged(FALSE)
        while Tagged()                      // while you find marked files
            UnMarkFile()                        // Unmark the file
            CompressOk(Format(compress," ", add_comp, " ", GetShortPath(zipfilename), " ", GetShortPath(GetFullPath())), tf, TRUE, FALSE)
        endwhile
    endif
    RebuildPickList()
end

proc UnZipFile()
    string tf[MAXPATH] = "nul"
    zipfilename = GetShortPath(ForceDot(GetFullPath()))
    // unzip and let it prompt
    // restore screen
    CompressOk(Format(uncompress, " ", add_uncomp, " ", zipfilename, " ", flist_path), tf, FALSE, TRUE)
    RebuildPickList()
end

proc ReadLineStr(var string variable)
    variable = Trim(GetText(5,CurrLineLen()))
    Down()
end

proc ReadLineInt(var integer variable)
    // gracefully degrade to read old format
    PushPosition()
    GotoColumn(5)
    if lFind("[~0-9a-fA-F]", "xc")
//        variable = PeekWord(AdjPtr(CurrLinePtr(), 5)) --CurrLinePtr() no longer supported
        variable = StrWord(GetText(5, 2))
        if variable > 32767
            // make it work for signed numbers; it GPFs if just change it to
            // PeekLong/PokeLong.
            variable = variable - 65536
        endif
        // write it back out in new format
        WriteLineInt(variable)
    endif
    PopPosition()

    variable = Val(Trim(GetText(5,CurrLineLen())), 16)
    Down()
end

integer proc SetupCompressVars(string sig)
    integer cid, rc

    cid = GetBufferId()
    GotoBufferId(configurationvars)
    rc = false
    if lFind(sig, "g^")
        rc = true
        compressorname = GetText(13, CurrLineLen())
        Down()
        ReadLineStr(compress)
        ReadLineStr(add_comp)
        ReadLineStr(uncompress)
        ReadLineStr(add_uncomp)
        ReadLineStr(compressview)
        ReadLineStr(ext_to_con)
        ReadLineStr(comp_over)
        ReadLineStr(del_file)
        ReadLineInt(compressfn_col)
        ReadLineInt(compressfn_endcol)
        ReadLineInt(compress_sig_offset)
    endif
    GotoBufferId(cid)
    return (rc)
end

integer proc isCompressFile(string fn)
    integer handle, rc
    string data[40] = ""

    handle = fOpen(fn + Chr(0), _OPEN_READONLY_)
    rc = fRead(handle, data, sizeof(data)) <> -1
    fClose(handle)
    if not rc
        return (false)
    endif

    rc = false
    if data[1:length(zipsig)] == zipsig
        rc = SetupCompressVars(zipsig)
    elseif data[1:length(zipsig2)] == zipsig2
        rc = SetupCompressVars(zipsig)
    elseif data[1:length(zipsig3)] == zipsig3
        rc = SetupCompressVars(zipsig)
    elseif data[1:length(tgzsig) ] == tgzsig
        rc = SetupCompressVars(tgzsig)
    elseif data[1:length(rarsig) ] == rarsig
        rc = SetupCompressVars(rarsig)
    elseif data[1:length(arjsig) ] == arjsig
        rc = SetupCompressVars(arjsig)
    elseif data[1:length(arcsig) ] == arcsig
        rc = SetupCompressVars(arcsig)
    elseif data[20:length(zoosig)] ==  zoosig
        rc = SetupCompressVars(zoosig)
    elseif data[2:length(lhasig) ] == lhasig
        rc = SetupCompressVars(lhasig)
    elseif data[1:length(s7zsig) ] == s7zsig
        rc = SetupCompressVars(s7zsig)
    endif
    return (rc)

end

constant kARC = 1, kARJ, kLZH, kRAR, kTGZ, kZIP, kZOO, k7ZIP

Menu CompresserMenu()
    "&ARC"      ,, CloseBefore
    "AR&J"      ,, CloseBefore
    "&LZH"      ,, CloseBefore
    "&RAR"      ,, CloseBefore
    "&TGZ"      ,, CloseBefore
    "&ZIP"      ,, CloseBefore
    "Z&OO"      ,, CloseBefore
    "&7ZIP"     ,, CloseBefore
end

proc Compresser()
    case CompresserMenu()
        when kARC   SetupCompressVars(arcsig)
        when kARJ   SetupCompressVars(arjsig)
        when kLZH   SetupCompressVars(lhasig)
        when kRAR   SetupCompressVars(rarsig)
        when kTGZ   SetupCompressVars(tgzsig)
        when kZIP   SetupCompressVars(zipsig)
        when kZOO   SetupCompressVars(zoosig)
        when k7ZIP  SetupCompressVars(s7zsig)
    endcase
end

Menu ZipManagerMenu()
    "&Unzip Tagged Files"
    "&View/Modify Contents"
    "&Zip Tagged Files"
    "&Change Compressor"  [compress:10], Compresser(), DontClose
end

proc ZipManager()
    case ZipManagerMenu()
        when 0      return()
        when 1      if isCompressFile(GetFullPath())
                        UnZipFile()
                    endif
        when 2      if isCompressFile(GetFullPath())
                        ViewModifyZipFile(GetFullPath())
                    endif
        when 3      ZipFiles()
    endcase
end

/***********************************************************************
    View the current file in a list box.  If it is a directory, go to
    the dir and exit proc, if it is a file, insert it into a temp
    buffer, if it is a compress file ViewZipFile()
***********************************************************************/
proc ViewIt(string fn0)
    string fn[MAXPATH] = fn0

    // Directory ?
    if (fn == "" and isDir()) or (FileExists(fn) & _DIRECTORY_)
        RebuildPickListThisPath(GetFullPath())  // set up list
    else
        if fn == ""
            fn = GetFullPath()
        endif
        if isCompressFile(fn)
            ViewModifyZipFile(fn)
        else
            ListFile(fn, fn, 0)
        endif
    endif
end

/***********************************************************************
    Print the file and return to the original file
***********************************************************************/
proc _PrintFile()
    integer id, msglevel, tid

    id = GetBufferId()
    msglevel = Set(MsgLevel, _WARNINGS_ONLY_) // turn off updating
    tid = EditFile(GetFullPath())   // edit the file
    Set(MsgLevel, msglevel)         // restore updating
    if tid
        PrintFile()                 // print the file
    endif
    GotoBufferId(id)                // go back to pick list
    AbandonFile(tid)                // Abandon the printed file
end

/***********************************************************************
    Print the current file or all files tagged.
***********************************************************************/
proc PrintIt(integer marked)
    if NOT marked
        _PrintFile()
    else
        MarkIfNotTagged(FALSE)
        while Tagged()                      // while you find marked files
            UnMarkFile()                        // Unmark the file
            _PrintFile()
        endwhile
    endif
end

/***********************************************************************
    Run the current file.  Check to make sure the _EXT_ is in our
    list of runable extensions.
***********************************************************************/
proc Run(integer pause)
    string fn[MAXPATH], ext[32]

    ext = PBExt()

    fn = GetFullPath()

    if Pos(ext + ".", exe)      // if it is an executable
        _Dos(fn, pause)         // run it
    elseif ext == ".mac"        // if it is a macro
        ExecMacro(fn)           // exec the macro
    endif
end

proc change_dir(string path)
    if path[2] == ':'
        LogDrive(path[1])
    endif
    ChDir(path)
end

proc mWhereIs()
    string save_dir[MAXPATH]

    save_dir = CurrDir()
    change_dir(flist_path)
    ExecMacro("where")
    change_dir(save_dir)
end

proc mStart()
    string save_dir[_MAX_PATH_]

    save_dir = CurrDir()
    change_dir(SplitPath(GetFullPath(), _DRIVE_|_PATH_))

    if Lower(SplitPath(GetFullPath(), _EXT_)) == ".exe"
        lDos(GetFullPath(), "", _RUN_DETACHED_|_DONT_WAIT_|_DONT_PROMPT_|_DONT_CLEAR_)
    else
        StartPgm(GetFullPath())
    endif

    change_dir(save_dir)
end

/***********************************************************************
    Copy a single file, and warn if any problems.
/***********************************************************************/
integer proc CopyMoveOneFile(string src, string dst_path, var integer overwrite, integer moving)
    string dst[MAXPATH]

    dst = dst_path
    if Length(dst) == 0
        Warn("Bogus destination path")
        return (FALSE)
    endif

    if isTrailingSlash(dst)
        dst = dst + SplitPath(src, _NAME_|_EXT_)
    endif

    if not overwrite and FileExists(dst)
        case MsgBoxEx("Destination file exists",
                "Overwrite " + dst + " ?",
                "[&Yes];[&No];[Yes to &All];[&Cancel]")
            when 0, 4   return (False)
            when 1
            when 2
                return (True)
            when 3
                overwrite = True
        endcase
    endif

    if moving
        if MoveFile(src, dst, True)
            return (True)
        endif
    elseif CopyFile(src, dst, True)
        return (True)
    endif
    return (Warn("Error copying/moving ", src, " to: ", dst))
end

/***********************************************************************
    Copy/move the current file or all files tagged.
    If destinateion directory does not exist, create it.
***********************************************************************/
proc CopyMoveIt(integer is_moving)
    string dest[MAXPATH], dest_dir[MAXPATH]
    integer overwrite = False, is_tagged = False
    string action[15] = iif(is_moving, "Move", "Copy")
    string actioning[15] = iif(is_moving, "moving", "copying")
#ifdef COPYTO_CURR
    string save_dir[MAXPATH]

    save_dir = CurrDir()
    dest = flist_path
    ChDir(dest)
#endif

    if not tagged()
        dest = GetFullPath()
    else
        is_tagged = True
    endif
    loop
        if tagged()
            if not Ask(action + " tagged files to: [" + SqueezePath(CurrDir(), _USE_HOME_PATH_) + "]:", dest, mvcpy_his)
                return ()
            endif
        else
            if not Ask(Format(action; PBName(); "to: [" + SqueezePath(CurrDir(), _USE_HOME_PATH_) + "]"), dest, mvcpy_his)
                return ()
            endif
        endif

        if trim(dest) == ""
            dest = CurrDir() + PBName()
        endif

        dest = ExpandPath(dest, 1)
        if Length(dest) == 0
            Warn("Bogus destination")
        else
            if isWildPath(dest)
                dest = SplitPath(dest, _DRIVE_|_PATH_)
            endif
            break
        endif
    endloop

    dest_dir = SplitPath(ExpandPath(dest), _DRIVE_|_PATH_)
    if not MakeDirectory(dest_dir)
        return()
    endif

    PushLocation()
    PushPosition()

    if not Tagged()
        MarkIt(_MARK)
    endif

    if is_tagged
        dest = dest_dir
    endif

    while Tagged()                          // while you find marked files
        UnMarkFile()                        // Unmark the file
        if isDir()
            Warn(GetFullPath(), " is a directory, can not ", action)
        elseif NOT CopyMoveOneFile(GetFullPath(), dest, overwrite, is_moving)
            Warn("Error ", actioning, " file: ", GetFullPath(); "to"; dest)
        elseif is_moving
            KillLine()
        endif
    endwhile

    RebuildPickList()
#ifdef COPYTO_CURR
    ChDir(save_dir)
#endif
    PopPosition()
    PopLocation()
end

proc CopyIt()
    CopyMoveIt(false)
end

proc MoveIt()
    CopyMoveIt(true)
end

/*------------------------------------------------------------------------
  if dst does not have a path, use current
 ------------------------------------------------------------------------*/
integer proc rename_dir(string src)
    string dst0[_MAX_PATH_] = src, dst[_MAX_PATH_], src2[_MAX_PATH_]

    if not Ask("New name for " + SqueezePath(src, _USE_HOME_PATH_), dst0, _EDIT_HISTORY_) or trim(dst0) == ""
        return (false)
    endif

    dst = dst0
    if SplitPath(dst, _PATH_) == ""
        dst = SplitPath(src, _DRIVE_|_PATH_) + dst
    endif
    dst = ExpandPathNoWild(dst, true)
    if dst == ""
        return (Warn("Bogus path: ", dst0))
    endif
    if FileExists(dst)
        src2 = ExpandPathNoWild(src, true)
        if not EquiStr(src2, dst)
            return (Warn("Cannot rename to existing file: ", dst))
        endif
    endif

    if not MoveFile(src, dst)
        return (Warn("Could not rename: ", src, " to: ", dst))
    endif
    return (true)
end

/***********************************************************************
    Rename the current file, or all tagged files.
***********************************************************************/
proc RenameIt()
    string nfn[MAXPATH], fn[MAXPATH]
    integer cr, overwrite = False

    cr = CurrRow()

    if NOT MarkIfNotTagged(FALSE)
        return()                    // exit
    endif

    while Tagged()                      // while marked files
        if isDir()
            UnMarkFile()
            rename_dir(GetFullPath())
        else
            fn = SqueezePath(GetFullPath(), _USE_HOME_PATH_)
            nfn = fn
            UnMarkFile()
            if Ask('New filename for ' + fn, nfn) AND Length(nfn) // ask for new name
                if nfn == ".."
                    nfn = ExpandPath(format(SplitPath(fn,_DRIVE_|_PATH_), nfn, GetDirSeparator(), SplitPath(fn, _NAME_|_EXT_)))
                endif
                if Lower(SplitPath(ExpandPath(nfn), _DRIVE_)) == Lower(SplitPath(flist_path, _DRIVE_))
                    if FileExists(nfn) & _DIRECTORY_
                        nfn = Format(nfn, GetDirSeparator(), SplitPath(fn, _NAME_|_EXT_))
                    endif
                    RenameDiskFile(fn, nfn)      // rename the file
                else
                    CopyMoveOneFile(fn, nfn, overwrite, true)      // else copy it
                endif
            endif
        endif
    endwhile
    GotoRow(cr)
    RebuildPickList()                 // reset the picklist
end

/***********************************************************************
    The following routines are to set the date and time of the files
    you want to TOUCH.
***********************************************************************/

/***********************************************************************
    Prompt for the date to set.  If no date, use current.
***********************************************************************/
proc AskDate(var integer month, var integer day, var integer year)
    string date[10]
    integer dow

    date = ""
    if NOT (Ask("Date: [mm/dd/yyyy]", date) AND Length(date))
        GetDate(month, day, year, dow)      // get the current date
    else
        month = Val(date[1:2])
        day = Val(date[4:2])
        year = Val(date[7:4])
        if (year - 1900) < 0
            year = year + 1900
        endif
    endif
end

/***********************************************************************
    Prompt for the time to set files to.  Use current time if none
    given.
***********************************************************************/
proc AskTime(var integer hours, var integer mins, var integer secs)
    string time[8]
    integer hunds

    time = ""
    if NOT (Ask("Time: [hh:mm:ss]", time) AND Length(time))
        GetTime(hours, mins, secs, hunds)      // get the current time
    else
        hours = Val(time[1:2])
        mins = Val(time[4:2])
        secs = Val(time[7:2])
    endif
end

/***********************************************************************
    This is the routine to set the date and time of the files you want
    to TOUCH.
***********************************************************************/
proc SetDateTime()
    integer month, day, year, dow,
            hours, mins, secs, hunds

    if NOT MarkIfNotTagged(TRUE)
        return()                    // exit
    endif

    case YesNo("Set to current date, and time?")
        when 1      GetDate(month, day, year, dow)      // get the date
                    GetTime(hours, mins, secs, hunds)   // get the time
        when 2      AskDate(month, day, year)           // get the date
                    AskTime(hours, mins, secs)          // get the time
        when 0, 3   return()
    endcase

    PushPosition()

    while Tagged()
        UnMarkFile()                            // unmark the file
        if not LoSetDateTime(GetFullPath(), month, day, year, hours, mins, secs)
            Warn("Cannot 'touch' ", GetFullPath())
        endif
    endwhile

    PopPosition()
    RebuildPickList()             // if created update list
end

/***********************************************************************
    The following routines work with the directories
***********************************************************************/

/***********************************************************************
    Make a directory.  Prompt for a dir to make.
***********************************************************************/
proc mMakeDirectory()
    string  dir[MAXPATH],                         // store dir name
            currpath[MAXPATH]

    dir = ""
    if Ask("Directory to make:", dir) AND Length(dir)   // Get dir name
        currpath = CurrDir()
        if ChDriveDir(flist_path, 1)
            if MakeDirectory(dir)
                RebuildPickList()
            endif
        endif
        ChDriveDir(currpath, 0)
    endif
end

/***********************************************************************
    Remove the directory you are setting on, or prompt for a directory
    to remove.
***********************************************************************/
proc mRemoveDirectory2(string dir)
    string currpath[MAXPATH]

    if YesNo(Format("Remove : ", dir)) == 1
        currpath = CurrDir()
        if ChDriveDir(flist_path, 1) and RemoveDirectory(dir)
            RebuildPickList()
        endif
        ChDriveDir(currpath, 0)
    endif
end

proc mRemoveDirectory()
    string dir[MAXPATH]

    dir = ""
    if isDir()
        mRemoveDirectory2(GetFullPath())
    elseif Ask("Directory to remove:", dir) and Length(dir)
        mRemoveDirectory2(dir)
    endif
end

/***********************************************************************
    Change the drive you are working with, and use the current directory
    on that drive.
***********************************************************************/
proc ChangeDrv()
    string drvstr[255] = PickDrive()
    if drvstr <> ""
        flist_path = ExpandPath(drvstr)
        RebuildPickList()
    endif
end

/***********************************************************************
    Change directory, but prompting for the path, and switching
    to it.
***********************************************************************/
integer proc GetNewDir(var string dir)
    return (Ask("New Directory [" + fileslisted + "]", dir, cd_his))
end

proc ChangeDir()
    string cur_dir[MAXPATH], new_dir[MAXPATH]

    cur_dir = CurrDir()
    ExecMacro("cd")
    new_dir = CurrDir()
    SetCurrDriveDir(cur_dir)

    SetUpPickList(QuotePath(new_dir))
end

/***********************************************************************
    Format output numbers, into comma seperated numbers, and right
    justify them to "width"
***********************************************************************/
string proc nFormat(integer n, integer width)
    string s[32], comma[1], digits[4]
    integer modulus

    s = ''
    comma = ''

    repeat
        modulus = Abs(n mod 1000)
        n = n / 1000
        digits = iif(n, format(modulus:3:'0'), format(modulus))
        s = format(digits,comma,s)
        comma = Chr(Query(ThousandsSeparator))
    until n == 0

    return (format(s:width))
end

// Assumes input string has no leading zeroes...
// Returns string with formatting commas inserted
string proc InsertCommas(string s)
    string   ret[26] = s
    integer  idx = (length(s) mod 3) + 1
    if length(s) > 3
        while idx < length(ret)
            if idx > 1
                ret = InsStr(Chr(Query(ThousandsSeparator)),ret,idx)
                idx = idx + 4
            else
                idx = idx + 3
            endif
        endwhile
    endif
    return( ret)
end

// A Macro to display the diskspace on the current drive.
// Handles drives >2Gig and should work on any Win32 platform
// until drives exceed ~16,777,215 Terabytes...

/***********************************************************************
    Pop up a box that displays the information about the current
    drive, the listed files, and the tagged files.
***********************************************************************/
proc DiskSpace()
    integer attr, cursor, x, y, displaylength
    string free_st[30]="", size_st[30]="", s_units[10]

    attr = Set(Attr, realtextattr)
    cursor = Set(Cursor, OFF)
    x = Query(ScreenCols) / 2
    y = Query(ScreenRows) / 2
    PopWinOpen(x - 25, y - 6, x + 25, y + 5, 1, SplitPath(flist_path, _DRIVE_), Query(CurrWinBorderAttr))
    ClrScr()
    WriteLine("")

    loDiskSpace(SplitPath(flist_path, _DRIVE_), size_st, free_st)

    // Optionally, the results can be truncated
    // by passing one of the following to ShiftRightU64()

    #define SCALE_TO_GBYTES 30
    #define SCALE_TO_MBYTES 20
    #define SCALE_TO_KBYTES 10
    #define SCALE_TO_BYTES   0  // No truncation

    size_st = ShiftRightU64(size_st,SCALE_TO_BYTES)
    free_st = ShiftRightU64(free_st,SCALE_TO_BYTES)
    s_units = "bytes"

    size_st = InsertCommas(size_st)
    free_st = InsertCommas(free_st)

    displaylength = max(length(size_st),length(free_st))

    Write("   Total Diskspace   ")
    WriteLine(size_st:displaylength," ",s_units)

    Write("   Free Diskspace    ")
    WriteLine(free_st:displaylength," ",s_units)

    WriteLine("")
    WriteLine("   Files and Dirs    ", nFormat(NumLines(), 15))

    size_st = TotalSize()
    size_st = InsertCommas(size_st)

    WriteLine("   Diskspace         ", Format(size_st:15), " bytes")
    WriteLine("")
    WriteLine("   Tagged Files      ", nFormat(taggedfiles, 15))
    size_st = InsertCommas(taggedsize_st)
    Write    ("   Diskspace         ", Format(size_st:15), " bytes")
    WindowFooter(" {Press any key} ")
    GetKey()
    PopWinClose()
    Set(Attr, attr)
    Set(Cursor, cursor)
end

/***********************************************************************
    Add files to view list.

    Assumes they are in the same directory.  For example, view *.s,
    and then add *.h *.c.

    flags allows you to force the list to only the files you enter
***********************************************************************/
integer proc AddFiles(integer flags)
    string files[MAXPATH]
    integer rc

    rc = FALSE

    files = ""
    if Ask(iif(flags & FORCE, "New File spec", "File(s) to add:"), files) AND Length(files)
        if files == "*.*"
            fileslisted = "*.*"
        else
            if flags & FORCE
                fileslisted = files + " "
            else
                fileslisted = fileslisted + files + " "
                if Pos(" *.*", fileslisted)
                    fileslisted = files + " "
                endif
            endif
        endif
        rc = RebuildPickList()
    endif
    return(rc)
end

/***********************************************************************
    The following routines are to select the attributes to display.
    Note that if the file has at least one of the attributes, it
    will be displayed.
***********************************************************************/

menu SelectAttribsMenu()
    '&Archive'   [CheckAttr(_ARCHIVE_, TRUE):2]   ,mset(_ARCHIVE_  , TRUE)  , DontClose
    '&Hidden'    [CheckAttr(_HIDDEN_, TRUE):2]    ,mset(_HIDDEN_, TRUE)     , DontClose
    '&Read Only' [CheckAttr(_READ_ONLY_, TRUE):2] ,mset(_READ_ONLY_, TRUE)  , DontClose
    '&System'    [CheckAttr(_SYSTEM_, TRUE):2]    ,mset(_SYSTEM_, TRUE)     , DontClose
    '&Volume'    [CheckAttr(_VOLUME_, TRUE):2]    ,mset(_VOLUME_, TRUE)     , DontClose
end

proc SelectAttribs(integer cfig)
    SelectAttribsMenu()
    if NOT cfig
        RebuildPickList()
    endif
end

/**************************************************************************
  Hide tagged files from the list
 **************************************************************************/
proc HideTaggedFiles()
    MarkIfNotTagged(FALSE)
    if YesNo("Remove file(s) from list?") == 1     // confirm??
        while Tagged()
            KillLine()
        endwhile
    endif
    PutTopLine()
end

/***********************************************************************
    Narrow display to flagged files.

    Assumes that you have files flaged, and removes all un-tagged
    files from view buffer.
***********************************************************************/
proc Narrow()
    BegFile()
    repeat
        while CurrChar() == 32                 // not marked
            KillLine()
        endwhile
    until NOT Down()
    TagAll(" ")                                // Clear tags
    BegFile()
    PutTopLine()
end

proc HereDos(string s, integer flags)
    string save_dir[MAXPATH] = CurrDir()

    ChDriveDir(flist_path, 0)

    _Dos(s, flags)

    ChDriveDir(save_dir, 0)
end

proc mShell()
    HereDos("", 0)
end

proc mDos()
    string dosstr[MAXPATH] = GetHistoryStr(_DOS_HISTORY_, 1)

    if Ask("DOS command:", dosstr, _DOS_HISTORY_)
        HereDos(dosstr, 0)                 // execute DOS command
    endif
end
/***********************************************************************
    Routines to update the .DAT file
***********************************************************************/

proc WriteLineStr(string variable)
    GotoColumn(5)
    KillToEOL()
    InsertText(variable)
    Down()
end

proc WriteLineInt(integer variable)
    GotoColumn(5)
    // width is 8 instead of 3 so negative numbers work, used so
    // compressfn_col can be negative to denote index for GetToken.
    InsertText(Format(variable:8:"0":16), _OVERWRITE_)
    Down()
end

integer proc SaveCompressVars(string sig)
    integer cid, rc

    cid = GetBufferId()
    rc = TRUE
    GotoBufferId(configurationvars)
    if lFind(sig, "g^")
        Down()
        WriteLineStr(compress)
        WriteLineStr(add_comp)
        WriteLineStr(uncompress)
        WriteLineStr(add_uncomp)
        WriteLineStr(compressview)
        WriteLineStr(ext_to_con)
        WriteLineStr(comp_over)
        WriteLineStr(del_file)
        WriteLineInt(compressfn_col)
        WriteLineInt(compressfn_endcol)
        WriteLineInt(compress_sig_offset)
    endif
    GotoBufferId(cid)
    return(rc)
end

proc ReadInt(var integer variable)
    string temp[10]

    temp = Str(variable)
    if lReadNumeric(temp, 10)
        variable = Val(temp)
    endif
end

menu CompressEdit()
    "Compressor"                        [compress:sizeof(compress)],
                                        Read(compress),             DontClose
    "Decompressor"                      [uncompress:sizeof(uncompress)],
                                        Read(uncompress),           DontClose
    "Command Line Options"              ,,Divide
    "Compress Option"                   [add_comp:sizeof(add_comp)],
                                        Read(add_comp),             DontClose
    "Decompress Option"                 [add_uncomp:sizeof(add_uncomp)],
                                        Read(add_uncomp),           DontClose
    "View Option"                       [compressview:sizeof(compressview)],
                                        Read(compressview),         DontClose
    "Extract to Screen Option"          [ext_to_con:sizeof(ext_to_con)],
                                        Read(ext_to_con),           DontClose
    "Write Over Option"                 [comp_over:sizeof(comp_over)],
                                        Read(comp_over),            DontClose
    "Delete File Option"                [del_file:sizeof(del_file)],
                                        Read(del_file),             DontClose
    ""                                  ,,Divide
    "Start Column of Filename in View"  [compressfn_col:10],
                                        ReadInt(compressfn_col),    DontClose, "Use a negative number to signify a token index."
    "End Column of Filename in View"    [compressfn_endcol:10],
                                        ReadInt(compressfn_endcol), DontClose
end

proc EditCompressors(string sig)
    SetupCompressVars(sig)
    CompressEdit(compressorname)
    SaveCompressVars(sig)
end

proc ForceDotRead(var string variable)
    Read(variable)
    if variable[Length(variable)] <> "."
        variable[Length(variable)+1] = "."
    endif
end

proc GetSortOrder(var string sortorder)
    if Read(sortorder)
        Set(PickFileSortOrder, sortorder)
    endif
end

proc ReadCharInt(var integer variable)
    string temp[1]
    temp = ""
    read(temp)
    variable = Asc(temp)
end

menu EditOptionsMenu()
    "&Binary Ext"           [binaryload:60]
                            , ForceDotRead(binaryload)          , DontClose
    "&Hex Ext"              [hexload:60]
                            , ForceDotRead(hexload)             , DontClose
    "&Execute Ext"          [exe:40]
                            , ForceDotRead(exe)                 , DontClose
    "&Default Sort Order"   [lpickfilesortorder:8]
                            , GetSortOrder(lpickfilesortorder)   , DontClose
    "&Search Attributes"    , SelectAttribs(TRUE)               , DontClose
    "&First Drive to Display Volume Label" [Chr(skipdrive):1]
                            , ReadCharInt(skipdrive)            , DontClose
end

proc EditOptions()
    integer cid

    cid = GetBufferId()
    GotoBufferId(configurationvars)
    EditOptionsMenu()
    if lFind("config", "g^")
        Down()
        WriteLineStr(binaryload)
        WriteLineStr(hexload)
        WriteLineStr(exe)
        WriteLineStr(lpickfilesortorder)
        WriteLineInt(searchattribs)
        WriteLineInt(skipdrive)
    endif
    GotoBufferId(cid)
end

menu ConfigurationMenu()
    title = "Update"
    width = 20

    "O&ptions"      , EditOptions()             , DontClose
    "Compressors"   ,                           , Divide
    "&ARC"          , EditCompressors(arcsig)   , DontClose
    "AR&J"          , EditCompressors(arjsig)   , DontClose
    "&LZH"          , EditCompressors(lhasig)   , DontClose
    "&RAR"          , EditCompressors(rarsig)   , DontClose
    "&TGZ"          , EditCompressors(tgzsig)   , DontClose
    "&ZIP"          , EditCompressors(zipsig)   , DontClose
    "Z&OO"          , EditCompressors(zoosig)   , DontClose
    "&7ZIP"         , EditCompressors(s7zsig)   , DontClose
end

proc Configuration()
    integer cid

    cid = GetBufferId()
    ConfigurationMenu()
    GotoBufferId(configurationvars)
    SaveAs(datfile, _OVERWRITE_)
    GotoBufferId(cid)
end

#ifdef TREE_HERE
proc Tree()
    Disable(FileManagerKeys)
    if Lower(lasttree) <> Lower(flist_path[1..3])
        lasttree = flist_path[1..3]
        if NOT ExecMacro("tree " + lasttree + " -h-")
            Set(MacroCmdLine, "")
        endif
    else
        if NOT ExecMacro("ViewTree")
            Set(MacroCmdLine, "")
        endif
    endif
    if Length(Query(MacroCmdLine))
        SetUpPickList(Query(MacroCmdLine))
    endif
    Enable(FileManagerKeys)
end
#endif

proc copy_path_to_win_clip()
    CopyToWinClip(GetFullPath())
end

/*
Add "filelist.tse" to the edit history, and then envoke "SaveAs()".

Called by:  Assigned to <Alt F> in listkeys.

*/
string attr_pos[] = Chr(_READONLY_) + Chr(_HIDDEN_) + Chr(_SYSTEM_) + Chr(_VOLUME_) + Chr(_DIRECTORY_) + Chr(_ARCHIVE_)
string attr_ltr[] = "rhsvda"
string proc mPBAttribute()
    integer a, i
    string astr[6] = "      "

    a = PBAttribute()
    for i = 1 to Length(attr_pos)
        if a & Asc(attr_pos[i])
            astr[i] = attr_ltr[i]
        endif
    endfor

    return (astr)
end

string proc mPBSize()
    return (iif(PBAttribute() & _DIRECTORY_,
            "   <Dir>", PBSizeStr()))
end

/**************************************************************************
  List buffer needs to be hidden to prevent conflicts with other macros.
  However, we don't want undo info originally, so create it as _SYSTEM_.
 **************************************************************************/
integer proc PBToAscii(integer pb_id)
    integer list_id
    string path[MAXPATH]

    Message("Creating list file...")
    PushPosition()

    list_id = CreateTempBuffer()

    GotoBufferId(pb_id)
    BegFile()
    repeat
        path = PBName()
        AddLine(Format(mPBSize():12; PBDateStr(); PBTimeStr(); mPBAttribute()), list_id)
        GotoBufferId(list_id)
        GotoPos(CurrLineLen() + 2)
        InsertText(path)
        GotoBufferId(pb_id)
    until not Down()

    GotoBufferId(pb_id)
    PopPosition()

    GotoBufferId(list_id)
    BufferType(_HIDDEN_)
    BegFile()
    return (list_id)
end

proc mSaveListFile()
    integer id

    PushPosition()
    id = PBToAscii(GetBufferId())
    AddHistoryStr("filelist.tse", _EDIT_HISTORY_)
    SaveAs()
    PopPosition()
    AbandonFile(id)
end mSaveListFile

proc mOpenFile()
    EndProcess(-1)
    UpdateDisplay()
    EditFile()
end

/***********************************************************************
    The Menu system for FileManager
***********************************************************************/

menu FileMenu()
    History
    keydef = FileManagerKeys

   "Create New F&ile List..."           , AddFiles(FORCE) // only what you enter
   "Add &Files to List..."              , AddFiles(FALSE) //add additional
   "&Narrow View to Tagged Files"       , Narrow()
   "&Hide Tagged Files"                 , HideTaggedFiles()
   "Hide Direct&ories"                  , HideDirs()
   ""                                   ,                       ,Divide
   "Change &Attribute of File(s) "     , ChangeAttr(0)
   "Change &Time/Date of File(s) "     , SetDateTime()
   ""                                   ,                       ,Divide
   "&Copy File(s)..."                   , CopyIt()
   "&Delete File(s)"                    , DelIt()
   "&Edit File(s)"                      , Loadit()
   "&Move File(s)..."                   , MoveIt()
   "&Rename File(s)..."                 , RenameIt()
   "&View Current File"                 , ViewIt("")
   "Edit File Here"                     , edit_file_here()
   ""                                   ,                       ,Divide
   "Manage Com&press Files "           , ZipManager()
   ""                                   ,                       ,Divide
   "&Update Configuration "            , Configuration()
   ""                                   ,                       ,Divide
   "E&xit"                              , EndProcess(-1)
end

menu DirMenu()
    History
    keydef = FileManagerKeys

   "&Log New Drive..."                  , ChangeDrv()
   "&Change Directory..."               , ChangeDir()
   "Change to Current Working Directory", RebuildPickListThisPath(CurrDir())
   "Set As Current &Working Directory"  , SetCurrDriveDir(flist_path)
   "&Select Attributes to Display "    , SelectAttribs(FALSE)
//   "&Tree "                            , Tree()
   ""                                   ,                       , Divide
   "&Make Directory..."                 , mMakeDirectory()
   "&Remove Directory"                  , mRemoveDirectory()
   ""                                   ,                       , Divide
   "&Disk Space..."                     , DiskSpace()
   "&Freshen List"                      , RebuildPickList()
end

menu TagMenu()
    History
    title = "Tag:"
    keydef = FileManagerKeys

    "&All Files"                        , TagAll(Chr(MARK_CHAR))
    "&Current File"                     , MarkIt(_MARK)
    "By &Name..."                       , TagName()
    "By &Extension..."                  , TagExtension()
    "By &Date..."                       , TagDate()
    "By &Size..."                       , TagSize()
    "By A&ttributes "                  , TagAttributes()
    ""                                  , , Divide
    "T&oggle Current Tag  <SpaceBar>"   , MarkIt(_TOGGLE)
    "To&ggle All Tags"                  , InvertTags()
    ""                                  , , Divide
    "&Untag All"                        , TagAll(" ")
end

menu PrintMenu()
    History
    title = "Print:"
    keydef = FileManagerKeys

    "&Current File"                     , PrintIt(FALSE)
    "&Tagged Files"                     , PrintIt(TRUE)
end

menu ExecuteMenu()
    History
    title = "Execute:"
    keydef = FileManagerKeys

   "&Start Current File"                , mStart()
   "Current File With &Pause"           , Run(_DEFAULT_)
   "Current File With&out Pause"        , Run(_DONT_PROMPT_)
   "Dos &Command..."                    , mDos()
   "&Dos Shell"                         , mShell()
   "&Where Is"                          , mWhereIs()
   "&Explorer here"                     , StartPgm("Explorer", flist_path)
   "Cop&y Path to Windows Clipboard"    , copy_path_to_win_clip()
end

menu UtilityMenu()
    "Save List to File"                 , mSaveListFile()
    "&Open File Here"                   , mOpenFile()
end

menu HelpMenu()
    History
    keydef = FileManagerKeys

    "&Key Assignments..."               , QuickHelp(FileManagerHelp)
end

menubar FileManagerMenu()
    History

    "&File"         , FileMenu()
    "&Dir"          , DirMenu()
    "&Tag"          , TagMenu()
    "&Print"        , PrintMenu()
    "&Sort"         , SortMenu()
    "&Execute"      , ExecuteMenu()
    "&Util"         , UtilityMenu()
    "&Help"         , HelpMenu()
end

proc mFMLeftBtn()
    case MouseHotSpot()
        when _MOUSE_MARKING_
            GotoMouseCursor()
            // dblclk at midnight will fail
            if GetClockTicks() - lastLeft <= DBLCLK_THRESHHOLD
                // toggle back to orig state
                MarkIt(_TOGGLE)
                // view it
                ViewIt("")
                return()
            endif
            MarkIt(_TOGGLE)
            lastLeft = GetClockTicks()
        when _NONE_, _MOUSE_CLOSE_, _MOUSE_ZOOM_, _MOUSE_HRESIZE_,
                _MOUSE_VRESIZE_
            FileManagerMenu()
        otherwise
            ProcessHotSpot()
    endcase
end

proc mFMRightBtn()
    // do nothing for now
end

/***********************************************************************
    The help screen.
***********************************************************************/
helpdef FileManagerHelp
    x = 2
    y = 2
    width = 76
    height = 22
    title = "F-List Help"

    "F10 or /            Menu"
    "<Alt d>             DirMenu"
    "<Alt e>             ExecuteMenu"
    "<Alt f>             FileMenu"
    "<Alt h>             HelpMenu"
    "<Alt p>             PrintMenu"
    "<Alt S>             SortMenu"
    "<Alt t>             TagMenu"
    ""
    "<.>                 Change to parent dir"
    "<Grey*>             Tag"
    "<Grey+>             Tag"
    "<SpaceBar>          Tag"
    "<Ctrl a>            Change Attribute"
    "<Ctrl c>            Copy File(s)"
    "<Del>               Delete File(s)"
    "<Ctrl Del>          Remove File(s) from list"
    "<Ctrl e>            Edit File(s)"
    "<Ins>               Add Files to list"
    "<Ctrl i>            Change Directory"
    "<Ctrl m>            Move file(s)"
    "<Ctrl p><c>         Print current file"
    "<Ctrl p><m>         Print tagged files"
    "<Ctrl r>            Rename file(s)"
    "<Ctrl Enter>        Run without pause"
    "<CtrlShift Enter>   Run with pause"
    "<Ctrl t>            Change time/date"
    "<Ctrl u>            Untag All files"
    "<Ctrl v>            View"
    "<Ctrl z>            Manage compress files"
    ""
    "<CursorDown>        Down"
    "<CursorUp>          Up"
    "<PgUp>              PageUp"
    "<PgDn>              PageDown"
    "<Home>              BegFile"
    "<End>               EndFile"
    ""
    "<Escape>            Exit"
end

string helplinestr[] = "{Ctrl: A}-ChAttr {C}-Copy {E}-Edit {I}-Cd {M}-Move {R}-Rename {T}-Date/Time {V}-View"

keydef FileManagerKeys
    <alt CursorLeft>    RebuildPickListThisPath(flist_path + "..")
    <Grey*>             MarkIt(_TOGGLE)    Down()
    <Grey+>             MarkIt(_TOGGLE)    Down()
    <SpaceBar>          MarkIt(_TOGGLE)    Down()
    <Ctrl SpaceBar>     MarkIt(_TOGGLE)    Down()
    <Ctrl a>            ChangeAttr(0)
    <Ctrl c>            CopyIt()
    <Del>               DelIt()
    <Ctrl Del>          HideTaggedFiles()
    <Ctrl e>            Loadit()
    <Enter>             Loadit()
    <GreyEnter>         Loadit()
    <Ins>               AddFiles(FALSE)
    <Ctrl i>            ChangeDir()
    <Ctrl m>            MoveIt()
    <Ctrl r>            RenameIt()
    <Ctrl Enter>        Run(_DONT_PROMPT_)
    <CtrlShift Enter>   Run(_DEFAULT_)
    <Ctrl t>            SetDateTime()
    <Ctrl u>            TagAll(" ")
    <Ctrl v>            ViewIt("")
    <Ctrl z>            ZipManager()

    <F1>                QuickHelp(FileManagerHelp)

    <CursorDown>        Down()
    <CursorUp>          Up()
    <PgUp>              PageUp()
    <PgDn>              PageDown()
    <Home>              BegFile()
    <End>               EndFile()
    <F5>                ScrollToTop()
    <Shift F5>          ScrollToCenter()
    <f9>                mShell()
    <Alt f9>            mDos()

    <f10>               FileManagerMenu()

    <Alt d>             FileManagerMenu("d")
    <Alt e>             FileManagerMenu("e")
    <Alt f>             FileManagerMenu("f")
    <Alt h>             FileManagerMenu("h")
    <Alt p>             FileManagerMenu("p")
    <Alt S>             FileManagerMenu("s")
    <Alt t>             FileManagerMenu("t")
    <Alt u>             FileManagerMenu("u")

    <Escape>            EndProcess(-1)

    <LeftBtn>           mFMLeftBtn()
    <RightBtn>          mFMRightBtn()
end

proc ListStartUp()
    Unhook(ListStartUp)                 // do not want keydef for other lists
    if Enable(FileManagerKeys)
        ListFooter(helplinestr)
    endif
    PutTopLine()
    BreakHookChain()
end

integer proc initialSetupPickList(string path)
    if SetUpPickList(path)
        return (TRUE)
    endif

    return (SetUpPickList("*.*"))
end

proc main()
    integer id, showmainmenu, save_readflags
    string pickfilesortorder[8], filespec[MAXPATH], cmdline[255]

    if lpFreeBytesAvailableToCallerHi endif // just to keep compiler from noting it is unused

    Set(Break, ON)
    cmdline = Query(MacroCmdLine)
    id = GetBufferId()
    PushBlock()
    UnmarkBlock()
    firstfileid = 0
    taggedfiles = 0
    taggedsize_st = "0"

    if cmdline[1:2] == "-z"
        cmdline = Trim(cmdline[3:Length(cmdline)])
        Viewit(ExpandTilde(cmdline))
    else
        HideMouse()
        showmainmenu = Set(ShowMainMenu, Off)
//        pickfileflags = Set(PickFileFlags, _ADD_DIRS_|_ADD_SLASH_|_DIRS_AT_TOP_)
        pickfileflags = Set(PickFileFlags, Query(PickFileFlags) | _ADD_DIRS_)
        pickfilesortorder = Set(PickFileSortOrder, lpickfilesortorder)
        save_readflags = Set(ReadFlags, Query(ReadFlags) | FN_COMP)

        realtextattr = Query(TextAttr)

        filespec = flist_path
        if cmdline[1:2] == '-c'
            filespec = SplitPath(CurrFilename(), _DRIVE_|_PATH_) + "*.*"
        elseif Length(cmdline) and cmdline <> "-r"  // restore
            filespec = cmdline
        endif

        GotoBufferId(viewid)

        if Length(cmdline) or GetNewDir(filespec)
            if Length(filespec) == 0
                filespec = fileslisted
            endif
            if initialSetupPickList(mQuotePath(filespec))
                Set(X1, 1)
                Hook(_LIST_STARTUP_, ListStartUp)
                lList("", Min(255, Query(ScreenCols)), Query(ScreenRows) - 2, _ENABLE_SEARCH_|_FIXED_HEIGHT_|_BLOCK_SEARCH_|_ANCHOR_SEARCH_)
                UnHook(ListStartUp)
            endif
        endif

        ShowMouse()
        Set(ShowMainMenu, showmainmenu)
        Set(PickFileFlags, pickfileflags)
        Set(PickFileSortOrder, pickfilesortorder)
        Set(ReadFlags, save_readflags)
    endif

    GotoBufferId(id)
    PopBlock()

    if firstfileid
        GotoBufferId(firstfileid)
        EditFile(QuotePath(CurrFilename()))        // !!! Force calling of hooks
    endif
end

/**************************************************************************
  See if the specified pgm exists in the actual (LoadDir(1)) or the
  (possibly redirected) LoadDir().
 **************************************************************************/
proc find_editor_program(string fn, var string path)
    path = SplitPath(LoadDir(1), _DRIVE_|_PATH_) + fn
    if not FileExists(path)
        path = LoadDir() + fn
        if not FileExists(path)
            path = ""
        endif
    endif
    path = QuotePath(path)
end

proc WhenLoaded()
    integer id

    id = GetBufferId()

    find_editor_program("tee32.exe", tee32)
    validdrives = CreateTempBuffer()
    storeid = CreateTempBuffer()
    configurationvars = CreateTempBuffer()
    copyid = CreateTempBuffer()
    viewinzipid = CreateTempBuffer()
    listid = CreateTempBuffer()
    scrapbuffer = CreateTempBuffer()
    viewid = CreateTempBuffer()
    DisplayMode(_DISPLAY_PICKFILE_)             // set display to pickfile
    if NOT (validdrives AND storeid AND viewid AND configurationvars
            AND copyid AND listid AND viewinzipid AND scrapbuffer)
        Warn("Error initializing")
        PurgeMacro(CurrMacroFileName())
    endif

    lpickfilesortorder = Query(PickFileSortOrder)
    GotoBufferId(configurationvars)
    PushBlock()
    datfile = LoadDir() + "f.dat"
    BinaryMode(-1)
    InsertFile(datfile, _DONT_PROMPT_)
    SetupCompressVars(zipsig)
    if lFind("config", "g^")
        Down()
        ReadLineStr(binaryload)
        ReadLineStr(hexload)
        ReadLineStr(exe)
        ReadLineStr(lpickfilesortorder)
        ReadLineInt(searchattribs)
        ReadLineInt(skipdrive)
    else
        Warn("Bogus config file (f.dat) - Compression options will not work")
    endif
    PopBlock()
    GotoBufferId(id)
    cd_his = GetFreeHistory("F:ChDir")
    mvcpy_his = GetFreeHistory("F:MvCpy")
end

proc WhenPurged()
    AbandonFile(validdrives)
    AbandonFile(storeid)
    AbandonFile(viewid)
    AbandonFile(configurationvars)
    AbandonFile(copyid)
    AbandonFile(viewinzipid)
    AbandonFile(listid)
    AbandonFile(scrapbuffer)
end

