/*************************************************************************
  DelBlank    Replaces multiple blank lines with one, or removes them all

  Author:     SemWare

  Date:       Feb 7, 1997

  Overview:

  This macro will remove blank lines from a file or block, starting at the
  current position or at the beginning of the file or block.
  Optionally, consecutively occurring blank lines can be reduced into
  one.

  To run this macro, select it from the Potpourri, or run it by name
  via the ExecMacro command (default: <ctrl x>)

  Usage notes:

  If the cursor is within a marked block, the macro prompts to confirm using
  the block vs. the entire file.

  If the cursor is not at the beginning of the file/block, the macro prompts
  whether to start at the current position or at the beginning of the
  file/block.

  The macro prompts the user whether to delete all blank lines or to
  compress multiple consecutive blank lines into one.

  Copyright 1992-1997 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged by
  SemWare provided that this statement, including the above copyright
  notice, is not removed; and provided that no fee or other remuneration
  is received for distribution.  You may add your own copyright notice
  to cover new matter you add to the macro, but SemWare Corporation will
  neither support nor assume legal responsibility for any material added
  or any changes made to the macro.

*************************************************************************/

integer proc HowMany(integer lines_to_check, integer use_block)
    integer lines_left

    lines_left = iif(use_block, Query(BlockEndLine), NumLines()) - CurrLine()
    return (100 - (lines_left * 100 / lines_to_check))
end

menu StartWhereMenu()
    "&Start at Beginning of File/Block"
    "&Work from the Current Line Onwards"
    "&Cancel"
end

integer proc StartHere(integer use_block)
    if (use_block and CurrLine() > Query(BlockBegLine)) or
        (CurrLine() > 1)
        case StartWhereMenu("Delete Blank Lines")
            when 0, 3
                return (-1)
            when 1
                return (FALSE)
            when 2
        endcase
    endif
    return (TRUE)
end

menu DelWhatMenu()
    "Squeeze Multiple Blank Lines into &One"
    "Delete &All Blank lines"
    "&Cancel"
end

integer proc DelWhat()
    case DelWhatMenu("Delete Blank Lines")
        when 0, 3
            return (-1)
        when 1
            return (TRUE)
    endcase
    return (FALSE)
end

integer proc UseBlock()
    integer result

    if isCurrLineInBlock()
        result = MsgBox("Delete Blank Lines", "Limit operation to currently marked block?", _YES_NO_CANCEL_)
        case result
            when 0
                return (-1)
            when 1
                return (TRUE)
        endcase
    endif
    return (FALSE)
end

integer proc TryNextLine(integer use_block)
    return (Down() and (use_block == FALSE or isCurrLineInBlock()))
end

integer proc TryKillLine(integer use_block)
    return (KillLine() and (use_block == FALSE or isCurrLineInBlock()))
end

proc mRemoveTrailingWhite(integer use_block)
    integer last_line

    if Query(RemoveTrailingWhite)
        PushPosition()
        last_line = iif(use_block, Query(BlockEndLine), NumLines())
        repeat
        until not Down() or CurrLine() >= last_line
        PopPosition()
    endif
end

string BLANK_LINE[] = "^[ \t]*$"

proc main()
    integer use_block, lines_to_check, leave_one, start_here, count
    string fopts[12]

    Set(Break, ON)

    use_block = UseBlock()
    if use_block == -1
        return ()
    endif

    start_here = StartHere(use_block)
    if start_here == -1
        return ()
    endif

    leave_one = DelWhat()
    if leave_one == -1
        return ()
    endif

    PushPosition()

    if not start_here
        if use_block
            GotoBlockBegin()
        else
            BegFile()
        endif
    endif

    fopts = "x"
    if use_block
        fopts = fopts + "l"
    endif

    lines_to_check = iif(use_block, Query(BlockEndLine), NumLines()) - CurrLine()
    count = 0

    mRemoveTrailingWhite(use_block)

    while lFind(BLANK_LINE, fopts)
        if leave_one
            if not TryNextLine(use_block)
                break
            else
                Message(HowMany(lines_to_check, use_block), '% complete')

                // while a blank line is found
                while PosFirstNonWhite() == 0
                    // if we can, delete it - if we can't assume eof
                    if TryKillLine(use_block)
                        count = count + 1
                    else
                        goto loop_exit
                    endif
                endwhile
            endif
        elseif not TryKillLine(use_block)
            break
        else
            count = count + 1
            if CurrLine() mod 100 == 0
                // Display percentage left
                Message(HowMany(lines_to_check, use_block), '% complete')
            endif
        endif
    endwhile
    loop_exit:

    Message(count, " blank line(s) deleted.")
    PopPosition()
end
