/*************************************************************************
Column-mode Performs editing on each row of a column block

            Overview:

            This macro takes effect when one has a column block marked, the
            cursor is in said block, and certain editing keys are pressed.

            When <del> is pressed, the current character is deleted on all lines
            of the column block.

            Similar actions take place for other deletion keys.

            In the same manner, when a character is typed, the same character is
            repeated on all lines of the block.

            Keys:

                <del>           delete current char
                <backspace>     delete char left

                <ctrl del>      delete word right
                <ctrl t>        delete word right

                <alt del>       delete to end-of-line
                <f6>            delete to end-of-line

                <grey *>        paste from editor clipboard
                <ctrl v>        paste from Windows clipboard

                <ctrl ->        type character from above line

            Usage notes:

            To activate this macro, simply select it from the Potpourri menu.
            Then mark a column block, and use the appropriate keys.

            To have it always available, autoload it - Macro, Autoload List.

  Author: SemWare, 2006
  Change log:
    26 Feb 2015 - (SEM) added deltoeol and paste
    29 Dec 2015 - (SEM) set wordwrap off when pasting
    29 May 2020 - (SEM) added delrightword (ctrl-t, ctrl-del)
 *************************************************************************/

proc resize_block(integer delta)
    integer blk_bgln = Query(BlockBegLine)
    integer blk_edln = Query(BlockEndLine)
    integer blk_bgcl = Query(BlockBegCol)
    integer blk_edcl = Query(BlockEndCol)

    blk_edcl = blk_edcl + delta
    if delta < 0 or Query(Insert)
        UnMarkBlock()
        if blk_edcl >= blk_bgcl
            MarkColumn(blk_bgln, blk_bgcl, blk_edln, blk_edcl)
        endif
    endif
end

proc on_selfinsert()
    integer marking, line
    string key[1] = Chr(Query(Key))

    if isCurrLineInBlock() == _COLUMN_
        Left()
        if isCursorInBlock() == _COLUMN_
            line = CurrLine()
            PushPosition()
            marking = Set(marking, off)
            GotoLine(Query(BlockBegLine))
            repeat
                if CurrLine() <> line
                    InsertText(key)
                    Left()
                endif
            until not Down() or not isCursorInBlock()
            PopPosition()
            Right()
            resize_block(1)
            Set(Marking, marking)
        else
            Right()
        endif
    endif
end

proc my_copy_char_above()
    integer marking, c, sv

    if isCursorInBlock() == _COLUMN_
        PushPosition()                  // save position
        sv = Set(RemoveTrailingWhite, Off)
        c = iif(up(), CurrChar(), -1)   // Try for char on line above
        Set(RemoveTrailingWhite, sv)
        PopPosition()

        if c >= 0
            PushPosition()
            marking = Set(Marking, Off)
            Set(Key, c)
            GotoLine(Query(BlockBegLine))
            repeat
                InsertText(Chr(c))
                Left()
            until not Down() or not isCursorInBlock()
            PopPosition()                   // back to where we were
            Right()
            resize_block(1)
            Set(Marking, marking)
        endif
    else
        PressKey(Query(Key))
    endif
end

proc my_delch()
    integer marking
    if isCursorInBlock() == _COLUMN_
        PushPosition()
        marking = Set(marking, off)
        GotoLine(Query(BlockBegLine))
        repeat
            DelChar()
        until not Down() or not isCursorInBlock()
        PopPosition()
        resize_block(-1)
        Set(Marking, marking)
    else
        DelChar()
    endif
end

proc my_deltoeol()
    integer marking
    if isCursorInBlock() == _COLUMN_
        PushPosition()
        marking = Set(marking, off)
        GotoLine(Query(BlockBegLine))
        repeat
            DelToEOL()
        until not Down() or not isCursorInBlock()
        PopPosition()
        Set(Marking, marking)
    else
        DelChar()
    endif
end

proc my_delrightword()
    integer marking
    if isCursorInBlock() == _COLUMN_
        PushPosition()
        marking = Set(marking, off)
        GotoLine(Query(BlockBegLine))
        repeat
            DelRightWord()
        until not Down() or not isCursorInBlock()
        PopPosition()
        Set(Marking, marking)
    else
        DelRightWord()
    endif
end

proc my_backspace()
    integer marking, col
    if isCursorInBlock() == _COLUMN_
        col = CurrCol()
        PushPosition()
        marking = Set(marking, off)
        GotoLine(Query(BlockBegLine))
        repeat
            BackSpace()
            Right()
        until not Down() or not isCursorInBlock()
        PopPosition()
        if col == CurrCol()
            Left()
        endif
        resize_block(-1)
        Set(Marking, marking)
    else
        BackSpace()
    endif
end

proc my_paste()
    string text[255] = ""
    integer i, result = 2, ww

    ww = Set(WordWrap, off)
    if isCursorInBlock() == _COLUMN_
        PushPosition()
        if GotoBufferId(Query(ClipboardId))
            If NumLines() == 1
                text = GetText(1, 255)
            endif
        endif
        PopPosition()
    endif

    if text <> ""
        result = MsgBoxEx("", "Paste Text", "[&Respect Column mode];[&Paste normally];[&Cancel]")
    endif

    case result
        when 0, 3
        when 1
            for i = 1 to length(text)
                PressKey(Asc(text[i]))
            endfor
        when 2      Paste()
    endcase
    Set(WordWrap, ww)
end

proc my_pastereplacefromwinclip()
    PasteReplaceFromWinclip()
end

forward keydef typeable_keys
integer enabled

proc normal_command()
    if isCursorInBlock() == _COLUMN_
        if not enabled
            Enable(typeable_keys)
            enabled = True
        endif
    elseif enabled
        Disable(typeable_keys)
        enabled = False
    endif
end

proc nonedit_command()
    if enabled
        Disable(typeable_keys)
        enabled = False
    endif
end

proc WhenLoaded()
    Hook(_BEFORE_COMMAND_, normal_command)
    Hook(_AFTER_COMMAND_, normal_command)
    Hook(_BEFORE_NONEDIT_COMMAND_, nonedit_command)
    Hook(_AFTER_NONEDIT_COMMAND_, nonedit_command)
    Hook(_ON_SELFINSERT_, on_selfinsert)
end

keydef typeable_keys
<del>           my_delch()
<backspace>     my_backspace()

<ctrl t>        my_delrightword()
<ctrl del>      my_delrightword()

<f6>            my_deltoeol()
<alt del>       my_deltoeol()

<grey*>         my_paste()
<ctrl v>        my_pastereplacefromwinclip()

<ctrl ->        my_copy_char_above()

end
