/*************************************************************************
  CmpFiles    Compares the current file to a backup file

  Author:     Ian Campbell (Contributing User)

  Date:       Jun 16, 1993 (Original: Ian Campbell)
              Jul 15, 1994 (Revised: Ian Campbell)
              Nov 21, 1996 (Revised: Chris Antos)
                    Fix w32 problems including long filenames.

                    Add a shortcut so that if you just type in a path
                    instead of a filename, it will look in that path for the
                    same filename you already have loaded.

              Oct  2, 1997
                    Fix broken Save block to file
                    Add <alt x> as a alternative exit key

              Mar 19, 2001
                    If only two files are loaded, Cmpfiles will prompt to allow
                    those files to be compared.

              Feb  6, 2002 (Revised: Ross Boyd)
                    Cosmetic fix to mDisplayHelpScreen() for non OEM fonts.

              Feb 13, 2002 Ross Boyd
                    General code cleanup. (Functionality remains the same.)
                    FlashMessage() changed to DisplayMessage() because Win32 defaults
                    to non-flashing text attributes.
                    Replaced mMessageBox() and mYesNoCancel() with MsgBox().
                    Replaced ScrollToRow(Query(WindowRows)/2) with ScrollToCenter()

              Apr  6, 2002 Chris Antos, Ross Boyd, SEM
                    Merged Chris Antos' <Alt E> edit-mode code.
                    Made 'replace indicators' OEM characters in AfterUpdate()

              Jan 2004 Ross Boyd: Yet another 'spaces in the filename' problem
                    fixed.  Change EditFile() to EditThisFile().

              Nov 2008 SEM: Added <F7> as a synonym for <Enter> for compatibility with
                    Visual Source Safe.

              Dec 2010 SEM: - Only call DLL's under Windows.
                            - Add ignore case option
                            - Save window setup by using state macro.

              May 2017 SEM: - if started with 2 windows, keep them.

              Aug 2020 SEM: - if comparing already loaded files,
                              don't ask to unload one.

              Aug 2022 SEM: - Use internal CompareLines() so we can compare lines
                    > 255 characters.

  Version:    1.20

  Overview:

  The user will be prompted for a backup filename.  This backup file
  may be inside an "ARC", "ARJ", "LZH", or "ZIP" file, or it may be a
  stand alone file.  If the file is inside an archive, then the
  filename must be the same as the current filename.  Note that there
  may be many different files inside the archive, but only the one
  that matches the current filename will be extracted.

  The two files will be split (either vertically or horizontally).
  Cursor movement keystrokes will be fed to both windows so that they
  move simultaneously, in sync with each other.

  Text may be scanned, and differences will be highlighted.

  The video mode may be changed from 25 to 36 to 44 to 50 lines to
  show more detail.

  F1 is available for help.

  Whitespace characters (spaces and tabs) are automatically converted
  to a single space character by default, in an effort to reduce false
  comparisons.  This option may be toggled by pressing the "W" key.

  Once a re-sync has been accomplished, both windows will update to
  the end of the difference text, with the re-synced lines adjacent to
  each other on line 12 of the screen.  Difference text will be marked
  in both windows, for easy comparison.  Press the grey- key to
  retreat to the BEGINNING of the difference text, then press the
  grey+ key to move back to the END of the difference text.  Hit the
  enter again to continue the search process, -- to find and highlight
  the next difference.

  Keys:       (none)

  Usage notes:

  This macro is designed to be run from PROJECTS or may be executed
  directly from Potpourri.  It is automatically purged when done.

  LIMITATIONS: -Line length must be under 256 characters.

               -although whitespace errors will NOT cause files that
                are already in sync to de-sync (unless the whitespace
                filter is turned off), TSE's basic search mechanism is
                used to re-sync, and whitespace cannot be filtered out
                here.  This is usually not too big of an issue unless
                you completely en-tab, or de-tab one of the two files.
                Then re-syncing will NOT take place.

               -Your file decompressor must be in the path, or a "File
                Not Found"  error message will result.

  The usual disclaimer -- Use these macros at your own risk! I will in
  NO WAY be responsible for ANY problems resulting in their use.

*************************************************************************/

constant MAXPATH = 255      // _MAXPATH_
constant YES_NO = 2         // _YES_NO_

constant SkipLines = 20         // walk through 20 lines at a time
constant SearchRange = 250      // scan 250 lines at a time
constant ShortRangeScan = 0     // signifies a tight, line by line scan
constant MediumRangeScan = 1    // signifies a somewhat relaxed scan
constant LongRangeScan = 2      // scan entire file, in blocks, for a match
constant NO_VIDEO_FLG = 0       // the video mode is not changing
constant VIDEO_FLG = 1          // the video mode is changing
constant BEGINNING = 0
constant ENDING = 1

// change to FALSE for HORIZONTAL default
constant WinDefaultVert = TRUE  // TRUE = VERTICAL windows default

integer  cmp2bkup_file_history
integer  Depth = 20             // number of consecutive lines that must
                                // match before text is found
integer CID1                    // original file ID
integer CID2                    // backup file ID

integer End_Line1 = -1           // marking pointer for original file
integer End_Line2 = -1           // marking pointer for backup file
integer Begin_Line1 = -1         // marking pointer for original file
integer Begin_Line2 = -1         // marking pointer for backup file
integer WinID                   // original Window ID
integer FilterWhiteSpaceFlg = 1 // permission to filter whitespace chars
integer ignore_case = 0         // permission to ignore case
integer MouseState              // 0 = no mouse activity
                                // 1 = first mouse clicked
                                // 2 = autorepeat mouse clicking
String  GS1[255] = ""           // general purpose string number-1
String  GS2[255] = ""           // general purpose string number-2
String  GS3[255] = ""           // general purpose string number-3

integer g_cwba = -1
integer g_cwbt = -1

integer hookI
integer hookBUD
integer hookAUD
integer g_fBusy = FALSE
integer compare_existing        // Are we comparing already loaded files?

integer proc BorderAttr()
    return (iif(g_cwba >= 0, g_cwba, Query(CurrWinBorderAttr)))
end

proc SaveHookState()
    hookI = SetHookState(OFF, _IDLE_)
    hookBUD = SetHookState(OFF, _BEFORE_UPDATE_DISPLAY_)
    hookAUD = SetHookState(OFF, _AFTER_UPDATE_DISPLAY_)
end

proc RestoreHookState()
    SetHookState(hookI, _IDLE_)
    SetHookState(hookBUD, _BEFORE_UPDATE_DISPLAY_)
    SetHookState(hookAUD, _AFTER_UPDATE_DISPLAY_)
end

proc AfterUpdate()
    string s[1]
    integer x, y, i, wid

    // draw indicator showing which direction copying will occur in

    if WindowId() in 1..2
        i = 0

        Set(Attr, Query(CurrWinBorderAttr))

        wid = WindowId()
        GotoWindow(2)
        x = Query(WindowX1)
        GotoWindow(wid)

        if x > 2
            s = iif(wid == 1, "", "")
            x = iif(wid == 1, Query(WindowX1)+Query(WindowCols), Query(WindowX1)-1)
            y = Query(ScreenRows)/3
            i = max(1, Query(ScreenRows)/2 - y)
        else
            i = 0
        endif

        if i
            VGotoXYAbs(x, y)
            PutOemChar(s)  // JHB - (OEM fix)
            VGotoXYAbs(x, y+i)
            PutOemChar(s)
            VGotoXYAbs(x, y+i*2)
            PutOemChar(s)
        endif
    endif
end

proc SaveWinState()
    g_cwba = Set(CurrWinBorderAttr, Color(Bright Red on Black))
    g_cwbt = Set(CurrWinBorderType, 4)
    Hook(_AFTER_UPDATE_STATUSLINE_, AfterUpdate)
end

proc RestoreWinState()
    UnHook(AfterUpdate)
    Set(CurrWinBorderType, g_cwbt)
    Set(CurrWinBorderAttr, g_cwba)
    g_cwbt = -1
    g_cwba = -1
end

/*** mDirExists *********************************************************
 *                                                                      *
 * Check for the existence of a directory and return non zero if it     *
 * does exist.                                                          *
 *                                                                      *
 * Called by:   mGetValidFileName(), mLoadBackupFile()                  *
 *                                                                      *
 * Enter With:  the name of the directory                               *
 *                                                                      *
 * Returns:     non zero if the directory exists                        *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mDirExists(string s)
    return(FileExists(s) & _DIRECTORY_)
end mDirExists

/*** DisplayMessage *****************************************************
 *                                                                      *
 * Display a message on the top line.  The message will display         *
 * for the DisplayDelay time period.                                    *
 *                                                                      *
 * Called by:   mReplaceBlock(), mCompareToBkup()                       *
 *                                                                      *
 * Enter With:  Message to display, whether or not to update display    *
 *              on exit, the time to display in seconds.                *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc DisplayMessage(string Msg, integer Update, integer DisplayDelay)
    integer OldClockTicks

    Message(Msg)
    UpdateDisplay()
    OldClockTicks = GetClockTicks()
    repeat
        ExecHook(_NONEDIT_IDLE_)
    until ((GetClockTicks() > OldClockTicks + (DisplayDelay * 18))
            or (GetClockTicks() < OldClockTicks)
            or (KeyPressed()))
    Message(Msg)
    if Update
        UpdateDisplay()
    endif
end DisplayMessage

/*** mGS1EqualsGS2 ******************************************************
 *                                                                      *
 * Compares two strings.  If they are equal,                            *
 * return TRUE, if not, strip off superfluous whitespace, and repeat    *
 * the comparison.                                                      *
 *                                                                      *
 * Called by: mCheckMultiLine(), mFindTextLine(),                       *
 *            mFindFirstLineMatch(), mSearchTwoFiles()                  *
 *                                                                      *
 * Returns:   TRUE if the two strings compare, FALSE otherwise.         *
 * Notes:     None                                                      *
 *                                                                      *
 ************************************************************************/

integer proc mGS1EqualsGS2(integer line1, integer line2, integer b1, integer b2)
    if CompareLines(line1, line2, 0, b1, b2) == 0
        return (True)
    endif

    if ignore_case and FilterWhiteSpaceFlg
        if CompareLines(line1, line2, _IGNORE_CASE_|_FILTER_SPACES_, b1, b2) == 0
            return (True)
        endif
        return (False)
    endif

    if ignore_case
        if CompareLines(line1, line2, _IGNORE_CASE_, b1, b2) == 0
            return (True)
        endif
    endif

    if FilterWhiteSpaceFlg
        if CompareLines(line1, line2, _FILTER_SPACES_, b1, b2) == 0
            return (True)
        endif
    endif

    return (False)
end mGS1EqualsGS2

/*** mCheckMultiLine ****************************************************
 *                                                                      *
 * If a single line comparison between the two files is TRUE, then      *
 * mCheckMultiLine() will be called to see if more than one line        *
 * matches.  mCheckMultiLine() checks up to "Depth" (a global) lines    *
 * and all must match before TRUE will be returned.  The first mismatch *
 * causes FALSE to be returned.                                         *
 *                                                                      *
 * Called by: mFindTextLine(), mFindFirstLineMatch()                    *
 *                                                                      *
 * Returns:   TRUE if all lines match, FALSE otherwise.                 *
 * Notes:     None                                                      *
 *                                                                      *
 ************************************************************************/

integer proc mCheckMultiLine()  //@@@ here 02 Dec 2016  3:47 pm
    integer i, ReturnCode = TRUE, IDComingIn = GetBufferID()
    integer line1, line2

    GotoBufferID(CID1)
    PushPosition()
    GotoBufferID(CID2)
    PushPosition()
    i = 1
    while i <= Depth            // match all lines
        GotoBufferID(CID1)      // go to first window
        Down()
        line1 = CurrLine()
        GotoBufferID(CID2)      // go back to second window
        Down()
        line2 = CurrLine()
        if not mGS1EqualsGS2(line1, line2, CID1, CID2)
            ReturnCode = FALSE  // cancel the first line match
            break
        endif
        i = i + 1               // next line to match
    endwhile
    PopPosition()
    GotoBufferID(CID1)
    PopPosition()
    GotoBufferID(IDComingIn)
    Return(ReturnCode)
end mCheckMultiLine

/**************************************************************************
  02/11/2002 SEM: Because of the way this macro works, (it apparently
  assumes character blocks) don't change the following proc to use
  MarkLine(line1, line2), as it will break the macro when you use the 'text
  transfer' feature when comparing.
 **************************************************************************/
proc mMarkBlock(integer line1, integer line2)
    UnMarkBlock()

    GotoLine(line1)
    BegLine()
    MarkChar()
    GotoLine(line2)
    MarkChar()
end

/*** mFindTextLine ******************************************************
 *                                                                      *
 * Get a line of text from window ID "CIDRef1".  Search through         *
 * "MaxLines" of text in Window ID "CIDRef2" for the line.  If the      *
 * line is found, do a CheckMultiLine check to make sure it's the right *
 * one.  Return the line number if it is found and confirmed, -1 if not *
 * found, and -2 if the end of the file is reached.                     *
 *                                                                      *
 * Called by: mSearchForText()                                          *
 *                                                                      *
 * Enter With:  Window ID to get the text from.                         *
 *              Window ID block to scan for the text.                   *
 *              The number of lines to scan for text in the second      *
 *              window.                                                 *
 * Returns:     The line number if line found.                          *
 *              -1 if line not found.                                   *
 *              -2 if end of file reached.                              *
 * Notes:       None                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mFindTextLine(integer CIDRef1, integer CIDRef2,
    integer MaxLines)   //@@@ here - hard one
    integer Line = -1, MyCID = GetBufferID()
    integer line1, line2, b1, b2

    GotoBufferID(CIDRef1)
    BegLine()
    GS1 = GetText(1, 255)               // read text line from first file
    line1 = CurrLine()
    b1 = GetBufferId()
    GotoBufferID(CIDRef2)
    BegLine()
    PushPosition()
    UnMarkBlock()                       // unmark any existing mark
    MarkChar()
    GotoLine(CurrLine() + MaxLines)     // mark a block of MaxLines
    MarkChar()
    PopPosition()
    PushPosition()
    If IsBlockInCurrFile()
        Line = iif (lFind(GS1,"WL"), CurrLine(), -1) // search the block
    else
        //End of file reached, CANNOT synchronize files
        Line = -2
    endif
    if Line > 0
        GS2 = GetText(1, 255)           // do a full line check
        line2 = CurrLine()
        b2 = GetBufferId()
        if mGS1EqualsGS2(line1, line2, b1, b2)              // do a full line check
            if not mCheckMultiLine()
                Line = -1
            endif
        else
            Line = -1
        endif
    endif
    PopPosition()
    GotoBufferID(MyCID)
    return(Line)
end mFindTextLine

/*** mPositionText ******************************************************
 *                                                                      *
 * Goto the line just past the difference text for both files.          *
 *                                                                      *
 * Called by: mSearchForText()                                          *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       None                                                    *
 *                                                                      *
 ************************************************************************/

proc mPositionText()
    GotoBufferID(CID1)
    if End_Line1 > 0
        GotoLine(End_Line1)  // position just past the marked text
    endif
    GotoBufferID(CID2)
    if End_Line2 > 0
        GotoLine(End_Line2)  // position just past the marked text
    endif
end mPositionText

/*** mIsEscKeyPressed ***************************************************
 *                                                                      *
 * Provides a mechanism for escaping from a looping macro by pressing   *
 * the escape key.  All keystrokes are emptied from the editors buffer. *
 *                                                                      *
 * Called by: mSearchForText(), mSearchTwoFiles                         *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     TRUE if escape pressed, otherwise FALSE                 *
 *                                                                      *
 * Notes:       As well as detecting whether or not escape has been     *
 *              pressed, this macro also "eats" ALL outstanding         *
 *              keystrokes, whether they're escapes or not!             *
 *                                                                      *
 ************************************************************************/

integer proc mIsEscKeyPressed()
    integer PressedKey = 0

    while KeyPressed()
         PressedKey = GetKey()
    endwhile
    return(PressedKey == <Escape>)
end mIsEscKeyPressed

/*** mSearchForText *****************************************************
 *                                                                      *
 * Does either a short range, medium range, or a long range search for  *
 * text.  Basically, take a line from the first file, search MaxLines   *
 * through the second file for an exact match, then repeat the process  *
 * by taking a line from the second file, and searching the first file  *
 * for an exact match.  Continue searching until either the text is     *
 * found, the end of a file is reached, or until the while loop         *
 * expires (does not expire for long range scan).  Continue to advance  *
 * through each file as failures occur, either by one line or           *
 * "SkipLines" until a match is found (on either side).  Once a match   *
 * is found, move the cursor back through each file by "SearchRange".   *
 * This allows for a smaller depth value to be used, and also helps     *
 * to ensure that only the first match will be found (by subroutine     *
 * "mFindFirstLineMatch()").  Do NOT go back further than the beginning *
 * point.                                                               *
 *                                                                      *
 * Called by:   mSyncFirstLines()                                       *
 *                                                                      *
 * Enter With:  two integers                                            *
 *                                                                      *
 * Returns:     1 if a match occurs.                                    *
 *              0 if no match found.                                    *
 *             -1 if end of file reached or escape key pressed.         *
 *                                                                      *
 * Notes:       If the files CANNOT be resynced, then the cursor will   *
 *              be placed at the beginning of the differences, and      *
 *              NO difference text will be marked.                      *
 *                                                                      *
 ************************************************************************/

string SyncMessage1[24]
string SyncMessage2[24]
integer proc mSearchForText(integer RangeFlg)
    integer MyLine1, MyLine2, MaxLines1, MaxLines2
    integer UpLines, i = SkipLines, DownLines = SkipLines, DecLoopValue = 1

    End_Line1 = -1                   // invalidate pointer
    End_Line2 = -1                   // invalidate pointer

    GotoBufferID(CID1)              // go to the original file
    GotoLine(Begin_Line1)            // start point for file-1
    MaxLines1 = NumLines()          // assume all lines scanned (long range)
    MyLine1 = Begin_Line1            // displays line numbers on the message line

    GotoBufferID(CID2)              // go to the backup file
    GotoLine(Begin_Line2)            // start point for file-2
    MaxLines2 = NumLines()          // assume all lines scanned (long range)
    MyLine2 = Begin_Line2            // displays line numbers on the message line

    case RangeFlg
        when ShortRangeScan
            SyncMessage1 = " <Synchronizing Files>  "
            SyncMessage2 = " >Synchronizing Files<  "
            MaxLines1 = SkipLines
            MaxLines2 = MaxLines1
            DownLines = 1                   // scan every line per miss

        when MediumRangeScan    // if a medium scan, restrict the lines
            SyncMessage1 = "<<Synchronizing Files>> "
            SyncMessage2 = ">>Synchronizing Files<< "
            MaxLines1 = SearchRange
            MaxLines2 = MaxLines1

        when LongRangeScan
            SyncMessage1 = "**Synchronizing Files** "
            SyncMessage2 = "++Synchronizing Files++ "
            DecLoopValue = 0                // never fall out of while loop
    endcase

    while i > 0                             // begin the search
        Message(SyncMessage1, MyLine1)      // show user what and where

        // search through "MaxLines2" lines in the backup file for
        // the current line in the original file
        End_Line2 = mFindTextLine(CID1, CID2, MaxLines2)
        // if a match is not found yet, then compare files the other way...
        if End_Line2 <= 0
            if End_Line2 == -2               // end of file?
                return(-1)
            endif
            Message(SyncMessage2, MyLine2)  // show user what and where

            // search through "MaxLines1" lines in the original file for
            // the current line in the backup file
            End_Line1 = mFindTextLine(CID2, CID1, MaxLines1)
            if End_Line1 == -2               // end of file?
                return(-1)
            endif
        endif
        if End_Line1 > 0 or End_Line2 > 0 // a match on either side?
            mPositionText()             // place cursor where text the same
            GotoBufferID(CID1)          // go to original file

            // set UpLines to either SearchRange + 50, or the distance to
            // where the compare started, whichever is smaller.
            UpLines = Min(CurrLine() - Begin_Line1, SearchRange + 50)
            GotoBufferID(CID2)          // go to backup file

            // set UpLines to the distance to where the compare
            // started, if it is smaller than the current UpLines
            UpLines = Min(CurrLine() - Begin_Line2, UpLines)
            GotoBufferID(CID1)          // go back to original file
            if UpLines > 0
                Up(UpLines)             // back up for mFindFirstLineMatch()
            endif
            GotoBufferID(CID2)          // go back to backup file
            if UpLines > 0
                Up(UpLines)             // back up for mFindFirstLineMatch()
            endif
            return(1)                   // indicate success
        endif
        GotoBufferID(CID1)              // failure, go to original file
        if not Down(DownLines)          // move down "DownLines" in original
            return(-1)
        endif
        MyLine1 = CurrLine()            // note current line for message line
        GotoBufferID(CID2)              // failure, go to backup file
        if not Down(DownLines)          // move down "DownLines" in backup
            return(-1)
        endif
        MyLine2 = CurrLine()            // note current line for message line
        if mIsEscKeyPressed()           // press escape to abort this search
            return(-1)
        endif
        // short and medium range scans have a limited loop life
        i = i - DecLoopValue
    endwhile
    return(0)                           // fallen out of while loop -- failure
end mSearchForText

/*** mFindFirstLineMatch ************************************************
 *                                                                      *
 * Compare the two files, line by line, until a match is found.  Use    *
 * mCheckMultiLine() to confirm the match to a preset depth.  If a      *
 * match is found, then set End_Line1 and End_Line2 to their respective   *
 * current lines.  Global strings GS1 and GS2 are used to hold the      *
 * text.                                                                *
 *                                                                      *
 * Called by: mSyncFirstLines().                                        *
 *                                                                      *
 * Enter With:  nothing.                                                *
 *                                                                      *
 * Returns:     TRUE if successful, FALSE if not (should ALWAYS be      *
 *              TRUE).                                                  *
 *                                                                      *
 * Notes:       None.                                                   *
 *                                                                      *
 ************************************************************************/

integer proc mFindFirstLineMatch()  //@@@ here
    integer i = SearchRange + 51
    integer line1, line2

    End_Line1 = -1
    End_Line2 = -1
    GotoBufferID(CID1)
    Message("Synchronizing Files ")
    while i
        line1 = CurrLine()
        GotoBufferID(CID2)
        line2 = CurrLine()
        if mGS1EqualsGS2(line1, line2, CID1, CID2)
            if mCheckMultiLine()
                GotoBufferID(CID1)
                End_Line1 = CurrLine()
                GotoBufferID(CID2)
                End_Line2 = CurrLine()
                return(TRUE)               // success
            endif
        endif
        Down()
        GotoBufferID(CID1)
        Down()
        i = i - 1
    endwhile
    Warn ("*** ALGORITHM FAILURE ***  Sorry, but I can't track this part...")
    return(FALSE)                           // failure
end mFindFirstLineMatch

/*** mUpdateWindowOne ***************************************************
 *                                                                      *
 * Updates the new position of the original file into Window-1          *
 *                                                                      *
 * Called by: mSyncFirstLines(), mSearchTwoFiles()                      *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       None                                                    *
 *                                                                      *
 ************************************************************************/

proc mUpdateWindowOne()
    GotoWindow(2)               // make sure in "working window"
    GotoBufferID(CID1)          // go to the original file
    PushPosition()              // keep track of where we are
    GotoWindow(1)               // switch to window where original file goes
    PopPosition()               // Update Window-1 with new file position
    GotoWindow(2)               // back to working window
    GotoBufferID(CID2)          // Put the backup file back in window-2
end mUpdateWindowOne

/*** mSyncFirstLines ****************************************************
 *                                                                      *
 * Window-2 is assumed on entry.  Search both files, mark any           *
 * difference text, and advance to the line just past the difference    *
 * text (right where the two files are resynced).                       *
 *                                                                      *
 * Called by:   mSearchTwoFiles()                                       *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       None                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mSyncFirstLines()
    integer ReturnCode, WarnFlg = 0
    constant MinDepth = 3, MediumDepth = 8, DeepDepth = 25

    GotoBufferID(CID1)              // access the original file
    Begin_Line1 = CurrLine()         // mark the beginning line of the compare
    GotoBufferID(CID2)              // access the backup file
    Begin_Line2 = CurrLine()         // mark the beginning line of the compare
    Depth = MinDepth                // MinDepth lines must match
    ReturnCode = mSearchForText(ShortRangeScan)
    if ReturnCode <> 1              // not found in short range scan?
        Depth = MediumDepth         // MediumDepth lines must match
        ReturnCode = mSearchForText(MediumRangeScan)
        if ReturnCode <> 1          // not found in medium range scan?
            Depth = DeepDepth       // DeepDepth lines must match
            ReturnCode = mSearchForText(LongRangeScan)
            if ReturnCode <> 1
                WarnFlg = 1
            endif
        endif
    endif
    if ReturnCode == 1              // text found in one of the scans?
        Depth = MinDepth            // MinDepth lines must match

        // home in exactly on a line by line match (note the low depth)
        ReturnCode = mFindFirstLineMatch()
    endif
    if ReturnCode <> -1
        GotoWindow(WinID)
    else
        GotoBufferID(CID1)
        GotoLine(Begin_Line1)        // position to first mismatched line
        GotoBufferID(CID2)
        GotoLine(Begin_Line2)        // position to first mismatched line
        End_Line1 = -1               // end of file, cancel any marking
        End_Line2 = -1               // end of file, cancel any marking
        UnMarkBlock()               // unmark any block on a failure
    endif
    return(not WarnFlg)
end mSyncFirstLines

/*** mCenterBothWindows *************************************************
 *                                                                      *
 * Center both windows vertically, restore any horizontal Xoffset,      *
 * and update the display.                                              *
 *                                                                      *
 * Called by:   mSearchTwoFiles(), mToggleWindowTypes()                 *
 *                                                                      *
 * Enter With:  the Xoffset requirements for both windows               *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       None                                                    *
 *                                                                      *
 ************************************************************************/

proc mCenterBothWindows(integer MyXoffset1, integer MyXoffset2)
    GotoWindow(1)
    if CurrCol() < MyXoffset1 + 1
        GotoColumn(MyXoffset1 + 1)
    endif
    GotoXoffset(MyXoffset1)             // move whole screen horizontally
    ScrollToCenter()  // center the cursor line vertically

    GotoWindow(2)
    if CurrCol() < MyXoffset2 + 1
        GotoColumn(MyXoffset2 + 1)
    endif
    GotoXoffset(MyXoffset2)             // move whole screen horizontally
    ScrollToCenter()  // center the cursor line vertically
end mCenterBothWindows

/*** PopWinCentered *****************************************************
 *                                                                      *
 * This macro opens up a Popup Window, and centers it on the screen.    *
 * The co-ordinates of the window are stored globally in PopX1, PopX2,  *
 * PopY1, and PopY2.                                                    *
 *                                                                      *
 * Called by:   mShowClipboardHelp()                                    *
 *                                                                      *
 * Enter With:  Width, height, box type, and window message.            *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       PopX1, PopX2, PopY1, and PopY2 are globally made        *
 *              available for other routines who want to know the       *
 *              popup window's dimensions.                              *
 *                                                                      *
 ************************************************************************/

integer PopX1, PopY1, PopX2, PopY2
proc mPopWinCentered(integer WinWidth, integer WinHeight, integer BoxType,
    string BoxMessage, integer WindowColor)

    PopX1 = ((Query(ScreenCols) - WinWidth) / 2) + 1
    PopX2 = PopX1 + WinWidth - 1
    PopY1 = ((Query(ScreenRows) - WinHeight) / 2) + 1
    if PopY1 == 1
        PopY1 = 2
    endif
    PopY2 = PopY1 + WinHeight - 1
    PopWinOpen(PopX1, PopY1, PopX2, PopY2, BoxType, BoxMessage, WindowColor)
end mPopWinCentered

/*** mScanningFilesMsg **************************************************
 *                                                                      *
 * This macro opens a box on the screen while actively comparing the    *
 * two files.                                                           *
 *                                                                      *
 * Called by:   mSearchTwoFiles()                                       *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mScanningFilesMsg()
    Set(Attr, Query(MsgAttr))
    mPopWinCentered(43, 6, 4, "MESSAGE",        // open up a centered box
        BorderAttr())
    ClrScr()
    WriteLine(" Comparing original file to backup file.")
        Write(" Line Differences will be ")
        Set(Attr, Query(BlockAttr))
        Write("HILITED")
        Set(Attr, Query(MsgAttr))
    WriteLine(".")
    WriteLine("")
    Write(' Please wait...')
end mScanningFilesMsg

/*** mSearchTwoFiles ****************************************************
 *                                                                      *
 * Do a line by line compare of the two files, hilite blocks of         *
 * difference text in each file, and resync the two files.              *
 *                                                                      *
 * Called by:   mProcessKeystrokes                                      *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       None                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mSearchTwoFiles()  //@@@ here
    integer MyXoffset1, MyXoffset2, i = 0, EscapeKeyIsPressed = FALSE
    integer OldCursor, n, m, ReturnCode = 1
    integer line1, line2, b1, b2

    OldCursor = Set(Cursor, OFF)
    WinID = WindowID()              // save the current window ID
    GotoWindow(1)
    n = NumLines()
    MyXoffset1 = CurrXoffset()      // save the Xoffset
    GotoWindow(2)                   // switch to the first window
    m = NumLines()
    n = Min(m,n)
    MyXoffset2 = CurrXoffset()      // save the Xoffset
    GotoBufferID(CID2)              // make sure backup file in window-2
    mScanningFilesMsg()             // tell the user what we're up to
    loop                            // permanent loop
        i = i + 1
        line1 = CurrLine()
        b1 = GetBufferId()
        GotoBufferID(CID1)          // switch to original file
        line2 = CurrLine()
        b2 = GetBufferId()
        if not mGS1EqualsGS2(line1, line2, b1, b2)
            mUpdateWindowOne()  // update window one's position
            GotoWindow(WinID)   // restore the original window
            ReturnCode = mSyncFirstLines()  // mark difference and resync
            mUpdateWindowOne()  // update window one's position
            GotoWindow(WinID)   // restore the original window
            break               // exit
        endif
        if i >= 153             // avoid updates on every line (skip 75)
            if WinID == 1
                i = 0
                Message(CurrLine(), " of ", n, "...")
                if mIsEscKeyPressed()
                    EscapeKeyIsPressed = TRUE
                endif
            endif
        endif
        if not down()
            mUpdateWindowOne()  // exiting, update Window-1's Position
            break               // at end of file, exit loop
        endif
        GotoBufferID(CID2)      // switch back to the backup file
        if i >= 153             // avoid updates on every line (skip 75)
            if WinID == 2
                i = 0
                Message(CurrLine(), " of ", n, "...")
                if mIsEscKeyPressed()
                    EscapeKeyIsPressed = TRUE
                endif
            endif
        endif
        if not down()
            GotoBufferID(CID1)
            Up()                // files different legnths -- correct down!
            GotoBufferID(CID2)
            mUpdateWindowOne()  // exiting, update Window-1's Position
            break               // at end of file, exit compare
        endif
        if EscapeKeyIsPressed       // the escape key exits the loop
            mUpdateWindowOne()      // update window one's position
            break                   // exit
        endif
    endloop
    PopWinClose()
    mCenterBothWindows(MyXoffset1, MyXoffset2)
    Set(Cursor, OldCursor)
    UpdateDisplay()
    GotoWindow(WinID)               // restore the current window ID
    return(ReturnCode)
end mSearchTwoFiles

/*** mMarkDifference ****************************************************
 *                                                                      *
 * Mark difference text in MyWindow, file CID, from MyBeginLine         *
 * to MyEndLine.                                                        *
 *                                                                      *
 * Called by:   mMarkDifferenceText()                                   *
 *                                                                      *
 * Enter With:  the Window to mark, the file ID to mark, line to begin  *
 *              marking on, and the line to end marking on.             *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       None                                                    *
 *                                                                      *
 ************************************************************************/

proc mMarkDifference(integer MyWindow, integer CID, integer MyBeginLine,
    integer MyEndLine)

    if MyBeginLine <> -1 and MyEndLine <> -1
        GotoWindow(MyWindow)
        GotoBufferID(CID)
        PushPosition()

        mMarkBlock(MyBeginLine, MyEndLine)

        PopPosition()
    endif
end mMarkDifference

/*** mHiliteDiffWin *****************************************************
 *                                                                      *
 * Mark difference text in MyWindow, file CID, from MyBeginLine         *
 * to MyEndLine.                                                        *
 *                                                                      *
 * Called by:   mMarkDifferenceText()                                   *
 *                                                                      *
 * Enter With:  the Window to hilite, the file ID to hilite, line to    *
 *              begin hiliting on, and the line to end hiliting on.     *
 *                                                                      *
 * Returns:     Nothing                                                 *
 *                                                                      *
 * Notes:       TSE only allows one window to be marked at one time.    *
 *              Since I wanted two separate marks, I had to simulate    *
 *              one of them.  This is the routine that does the         *
 *              simulation.                                             *
 *                                                                      *
 ************************************************************************/

proc mHiliteDiffWin(integer MyWindow, integer CID, integer MyBeginLine,
    integer MyEndLine)
    integer i = 1, j = Query(WindowRows)
    integer BlockColor = Query(BlockAttr)
    integer CursorInBlockAttr = Query(CursorInBlockAttr)
    integer WindowColumns
    integer CurrentCursorRow = CurrRow()

    GotoWindow(MyWindow)
    GotoBufferID(CID)
    WindowColumns = Query(WindowCols)
    UpdateDisplay()                     // must update BEFORE coloring text!
    if MyBeginLine <> -1 and MyEndLine <> -1
        PushPosition()
        // make video window dimensions parallel the editing window dimensions
        Window(Query(WindowX1),Query(WindowY1),
            Query(WindowX1) + Query(WindowCols),
            Query(WindowY1) + Query(WindowRows))
        GotoRow(1)                      // start at upper left corner of screen
        BegLine()
        // scan all window rows, and hilite any lines that need it.
        while i <= j
            if CurrLine() >= MyBeginLine and CurrLine() < MyEndLine
                // set the virtual screen cursor
                VGotoXY(CurrCol(), CurrRow())
                if CurrentCursorRow == i
                    // color the text now with the BlockAttr.
                    PutAttr(CursorInBlockAttr, WindowColumns)
                else
                    PutAttr(BlockColor, WindowColumns)
                endif
            endif
            down()
            i = i + 1
        endwhile
        // disconnect video window dimensions from editing window dimensions
        FullWindow()
        PopPosition()
    endif
end mHiliteDiffWin

/*** mMarkDifferenceText ************************************************
 *                                                                      *
 * Mark difference text in the "other Window", and hilite the           *
 * difference text in this window.                                      *
 *                                                                      *
 * Called by:   mSyncTwoScreenKeys(), mProcessKeystrokes()              *
 *              mCompareToBkup()                                        *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       Only one of the windows is marked, the other is         *
 *              hilited to simulate a mark.                             *
 *                                                                      *
 ************************************************************************/

proc mMarkDifferenceText(integer ForceMark)
    integer CurrentWinID = WindowID()

    BufferVideo()
    if CurrentWinID == 1
        if ForceMark
            mMarkDifference(2, CID2, Begin_Line2, End_Line2)
        endif
        mHiliteDiffWin(1, CID1, Begin_Line1, End_Line1)
    else
        if ForceMark
            mMarkDifference(1, CID1, Begin_Line1, End_Line1)
        endif
        mHiliteDiffWin(2, CID2, Begin_Line2, End_Line2)
    endif
    UnBufferVideo()
end mMarkDifferenceText

/*** mRollLeft **********************************************************
 *                                                                      *
 * TSE seems to have problems handling hard tabs.  Basically, RollLeft  *
 * seems to get stuck when the cursor hits column 1 while on a tab.     *
 * This macro prevents this.                                            *
 *                                                                      *
 * Called by:   mSyncTwoScreenKeys()                                    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mRollLeft()
    if CurrCol() == 1 and CurrXoffset() == 0
        return()
    endif
    if CurrCol() == 1
        Right()
    endif
    RollLeft()
end mRollLeft

/*** mRollRight *********************************************************
 *                                                                      *
 * TSE seems to have problems handling hard tabs.  Basically, RollRight *
 * seems to get stuck when the cursor hits column 1 while on a tab.     *
 * This macro prevents this.  It also tries to keep the cursor as far   *
 * left as possible.                                                    *
 *                                                                      *
 * Called by:   mSyncTwoScreenKeys()                                    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mRollRight()
    // get the next xoffset position
    integer NewXoffset = CurrXoffset() + 1

    GotoXoffset(NewXoffset)             // try to scroll over one
    if CurrXoffset() == NewXoffset      // did this work?
        Left()                          // might be able to move cursor left
        if CurrXoffset() <> NewXoffset  // did this cause xoffset to fail?
            Right()                     // put cursor back
            GotoXoffset(NewXoffset)     // put new xoffset back
        endif
    else                                // failure to scroll over by one
        Right()                         // move cursor right to allow xoffset
        GotoXoffset(NewXoffset)         // try again to scroll over one
        Left()                          // put cursor back
        if CurrXoffset() <> NewXoffset  // did this not work?
            Right()                     // move cursor right a 2nd time
            GotoXoffset(NewXoffset)     // this time scroll over must be ok
        endif
    endif
end mRollRight

/*** mPositionAtMark ****************************************************
 *                                                                      *
 * This macro resyncs text to either the beginning of the marked        *
 * difference text, or to the next line following the marked            *
 * difference text for both files.  Use this macro to visually keep     *
 * things in sync.                                                      *
 *                                                                      *
 * Called by:   mSyncTwoScreenKeys()                                    *
 *                                                                      *
 * Enter With:  the window to position, and either the beginning or     *
 *              end of the marked area.                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mPositionAtMark(integer MyWindow, integer MyPosition)
    if MyPosition == BEGINNING
        if MyWindow == 1
            if Begin_Line1 <> -1 and End_Line1 <> -1
                GotoLine(Begin_Line1)
            endif
        else
            if Begin_Line2 <> -1 and End_Line2 <> -1
                GotoLine(Begin_Line2)
            endif
        endif
    else
        if MyWindow == 1
            if Begin_Line1 <> -1 and End_Line1 <> -1
                GotoLine(End_Line1)
            endif
        else
            if Begin_Line2 <> -1 and End_Line2 <> -1
                GotoLine(End_Line2)
            endif
        endif
    endif
end mPositionAtMark

/*** mHandleWhiteSpace **************************************************
 *                                                                      *
 * This macro switches between filtering out all whitespace sequences   *
 * to a single space (for line by line comparisons), or not doing this. *
 * The default is to ignore multiple whitespace differences.            *
 *                                                                      *
 * Called by:   mCompareToBkup()                                        *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mHandleWhiteSpace()

    FilterWhiteSpaceFlg = FilterWhiteSpaceFlg ^ 1

    MsgBox("","WhiteSpace " + iif(FilterWhiteSpaceFlg == 1,
            "is now FILTERED (default).",
            "is now UNFILTERED."))

end mHandleWhiteSpace

proc mHandleIgnoreCase()

    ignore_case = ignore_case ^ 1

    MsgBox("","Case " + iif(ignore_case,
            "is now ignored.",
            "is now respected (default)."))

end mHandleIgnoreCase

/*** mDisplayHelpScreen *************************************************
 *                                                                      *
 * This macro displays Help information for keystrokes / mouseclicks.   *
 *                                                                      *
 * Called by:   mProcessKeystrokes                                      *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mDisplayHelpScreen()
    integer OldCursor, use3d = iif((Query(SpecialEffects) & _USE_3D_CHARS_), _USE3D_, 0)

    OldCursor = Set(Cursor, OFF)
    mPopWinCentered(78, 24, 2, "CMPFILES HELP SCREEN, Press F10 For Menu System",BorderAttr())
    Set(Attr, Query(HelpBoldAttr))
    ClrScr()
    Set(Attr, Query(HelpTextAttr))
    PutOemStrXY(1, 1, " ", 0, use3d)
    Set(Attr, Query(HelpBoldAttr))
    Write("SPECIAL FUNCTIONS")
    Set(Attr, Query(HelpTextAttr))
    PutOemLine(" Ŀ", 30)

    PutOemStrXY(1, 2,"   FUNCTION Ŀ   KEYBOARD Ŀ  MOUSE Ŀ ", 0, use3d)
    PutOemStrXY(1, 3,"  Activate CMPFILES Menus      F10               Click On Status Line     ", 0, use3d)
    PutOemStrXY(1, 4,"                                                 Mouse Menu Item          ", 0, use3d)
    PutOemStrXY(1, 5,"  Roll current window up/down  Shift CursorUp/Dn None                     ", 0, use3d)
    PutOemStrXY(1, 6,"  Page current window up/down  Shift PgUp/Dn     None                     ", 0, use3d)
    PutOemStrXY(1, 7,"  Begin/Continue Comparison    Enter             Window Zoom Button  []  ", 0, use3d)
    PutOemStrXY(1, 8,"  Exit Macro                   Escape            Window Close Button []  ", 0, use3d)
    PutOemStrXY(1, 9,"  Horizontal/Vertical Windows  Spacebar          Mouse Menu Item          ", 0, use3d)
    PutOemStrXY(1,10,"  Replace Block<-Other Window  R                 Mouse Menu Item          ", 0, use3d)
    PutOemStrXY(1,11,"  Switch Windows               Tab               Point At Window & Click  ", 0, use3d)
    PutOemStrXY(1,12,"  Toggle Whitespace Filter     W                 None                     ", 0, use3d)
    PutOemStrXY(1,13,"", 0, use3d)

    PutOemStrXY(1,14, " ", 0, use3d)
    Set(Attr, Query(HelpBoldAttr))
    Write("DUAL WINDOW FUNCTIONS")
    Set(Attr, Query(HelpTextAttr))
    PutOemLine(" Ŀ", 28)
    PutOemStrXY(1,15,"   FUNCTION Ŀ   KEYBOARD Ŀ    MOUSE Ŀ    ", 0, use3d)
    PutOemStrXY(1,16,"  End Of Both Windows           Ctrl End, Ctrl PgDn    Mouse Menu Item    ", 0, use3d)
    PutOemStrXY(1,17,"  Beginning Of Both Windows     Ctrl Home, Ctrl PgUp   Mouse Menu Item    ", 0, use3d)
    PutOemStrXY(1,18,"  Jump To Line                  J                      Mouse Menu Item    ", 0, use3d)
    PutOemStrXY(1,19,"  Find                          Ctrl F                 Mouse Menu Item    ", 0, use3d)
    PutOemStrXY(1,20,"  Repeat Find                   Ctrl L                 Mouse Menu Item    ", 0, use3d)
    PutOemStrXY(1,21,"  Begin/End Of Difference Text  F11/F12                Mouse Menu Item    ", 0, use3d)
    PutOemStrXY(1,22,"", 0, use3d)
    if GetKey() == <F10>                // menu requested?
        PushKey(<F10>)
    endif
    PopWinClose()
    Set(Cursor, OldCursor)
end mDisplayHelpScreen

/*** mToggleWindowTypes *************************************************
 *                                                                      *
 * This macro adjusts the windows when either a video change occurs, or *
 * a a Vertical / Horizontal change occurs.                             *
 *                                                                      *
 * Called by:   mProcessKeystrokes                                      *
 *                                                                      *
 * Enter With:  The current window ID, and VideoFlg set to a 1 if       *
 *              a video mode change is to occur.                        *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mToggleWindowTypes(integer CurrentWinID, integer VideoFlg, integer Size)
integer MyXoffset1, MyXoffset2
    string BkupFileName[MAXPATH]

    GotoWindow(2)
    MyXoffset2 = CurrXoffset()
    BkupFileName = CurrFilename()
    GotoWindow(1)
    MyXoffset1 = CurrXoffset()
    OneWindow()
    if GetGlobalInt("VWindowFlg")
        if VideoFlg             // if video mode change, keep same window type
            Set(CurrVideoMode, size)
            VWindow()
            SetGlobalInt("VWindowFlg", TRUE)       // vertical windows
        else                    // toggle window types
            HWindow()
            SetGlobalInt("VWindowFlg", FALSE)       // vertical windows
        endif
    else
        if VideoFlg             // if video mode change, keep same window type
            Set(CurrVideoMode, size)
            HWindow()
            SetGlobalInt("VWindowFlg", FALSE)       // vertical windows
        else                    // toggle window types
            VWindow()
            SetGlobalInt("VWindowFlg", TRUE)       // vertical windows
        endif
    endif
    EditThisFile(BkupFileName)
    mCenterBothWindows(MyXoffset1, MyXoffset2)
    GotoWindow(CurrentWinID)
end mToggleWindowTypes

/*** mEscapeProcess *****************************************************
 *                                                                      *
 * This macro prepares for a macro exit.  It marks the difference text, *
 * and stores the information in a global area which stays after the    *
 * macro shuts down.  It switches to one window, either the original,   *
 * or the backup window (whichever one the cursor is in), and leaves    *
 * any text marked in this window.                                      *
 *                                                                      *
 * Called by:   mProcessKeystrokes                                      *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mEscapeProcess()
    integer MyWindow = WindowID(), MainFileID, ReturnCode

    GotoWindow(iif(WindowID() == 1, 2, 1))  // toggle to other window
    mMarkDifferenceText(TRUE)               // hilite difference info
    GotoWindow(iif(WindowID() == 1, 2, 1))  // toggle to first window

    GotoWindow(1)
    MainFileID = GetBufferID()
    GotoWindow(2)
    // get the backup filename
    GS2 = SplitPath(CurrFilename(), _NAME_ | _EXT_)
    Set(Attr, Query(MsgAttr))

    ReturnCode = TRUE
    if not compare_existing
        case MsgBox("","Unload 'Compare to' File?", _YES_NO_CANCEL_)
            when 1                              // yes
                QuitFile()
                GoToBufferID(MainFileID)
                OneWindow()
                ReturnCode = TRUE               // continue with macro exit
            when 2                              // no
                GotoWindow(MyWindow)            // back to original window
                ReturnCode = TRUE               // continue with macro exit
            when 0,3                            // escape
                GotoWindow(MyWindow)            // back to original window
                ReturnCode = FALSE
        endcase
    endif
    if ReturnCode                           // really exiting the macro?
        // preserve these integers in case Cmp2BkUp restarts
        SetGlobalInt("BeginLine1Save", Begin_Line1)
        SetGlobalInt("BeginLine2Save", Begin_Line2)
        SetGlobalInt("EndLine1Save", End_Line1)
        SetGlobalInt("EndLine2Save", End_Line2)
    endif
    return (ReturnCode)
end mEscapeProcess

/*** mGotoLines *********************************************************
 *                                                                      *
 * This macro changes the current window to the entered line number,    *
 * while the other window is changed by a common differential.          *
 *                                                                      *
 * Called by:   mProcessKeystrokes                                      *
 *                                                                      *
 * Enter With:  The current window ID.                                  *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mGotoLines(integer CurrentWinID)
integer LineDiff
String  LastGotoLine[11] = ""

    if Ask("Go to line:", LastGotoLine, _GOTOLINE_HISTORY_)
        LineDiff = val(LastGotoLine) - CurrLine()
        GotoLine(val(LastGotoLine))
        GotoWindow(iif(CurrentWinID == 1, 2, 1))
        GotoLine(CurrLine() + LineDiff)
        GotoWindow(CurrentWinID)
    endif
end mGotoLines

/*** mProcessMouse ******************************************************
 *                                                                      *
 * This macro handles all of the supported mouse functions.             *
 *                                                                      *
 * Called by:   mProcessKeystrokes                                      *
 *                                                                      *
 * Enter With:  The mouse button pressed                                *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       All mouse functions, except _MOUSE_VRESIZE_ and         *
 *              _MOUSE_HRESIZE_, are simulated via pushed keystrokes.   *
 *                                                                      *
 ************************************************************************/

proc mProcessMouse(integer Button)
    GotoWindow(MouseWindowID())
    case MouseHotSpot()
        when _NONE_
            if Button == <RightBtn>
                PushKey(<Grey+>)
            else
                PushKey(<Grey->)
            endif
        when _MOUSE_CLOSE_
            PushKey(<Escape>)               // terminate the macro
        when _MOUSE_ZOOM_
            PushKey(<Enter>)                // begin the scan
        when _MOUSE_UP_
            PushKey(<CursorUp>)
        when _MOUSE_DOWN_
            PushKey(<CursorDown>)
        when _MOUSE_PAGEUP_
            PushKey(<PgUp>)
        when _MOUSE_PAGEDOWN_
            PushKey(<PgDn>)
        when _MOUSE_LEFT_
            PushKey(<CursorLeft>)
        when _MOUSE_RIGHT_
            PushKey(<CursorRight>)
        when _MOUSE_TABLEFT_
            PushKey(<CursorLeft>)
        when _MOUSE_TABRIGHT_
            PushKey(<CursorRight>)
        when _MOUSE_VRESIZE_
            ProcessHotSpot()
        when _MOUSE_HRESIZE_
            ProcessHotSpot()
        when _MOUSE_VWINDOW_
            PushKey(<Spacebar>)
        when _MOUSE_HWINDOW_
            PushKey(<Spacebar>)
    endcase
    return()
end mProcessMouse

integer proc mMarkHilitedText()
    integer MyBeginLine, MyEndline

    if WindowID() == 1
        MyBeginLine = Begin_Line1
        MyEndline = End_Line1
    else
        MyBeginLine = Begin_Line2
        MyEndline = End_Line2
    endif
    if MyBeginLine == -1 or MyEndLine == -1 or MyEndLine < MyBeginLine
        return (Warn("Block not marked!"))
    else
        mMarkBlock(MyBeginLine, MyEndLine)
        if not IsBlockMarked()
            return (Warn("Block not marked!"))
        endif
    endif
    return (TRUE)
end

/**************************************************************************
  Save the currently hilited text
 **************************************************************************/
proc mSaveBlock()
    PushBlock()
    if mMarkHilitedText()
        SaveBlock()
    endif
    PopBlock()
end

/*** mCopyBlock *********************************************************
 *                                                                      *
 * Copy the current highlighted block to the clipboard.                 *
 *                                                                      *
 * Called by:   the EditMenu()                                          *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mCopyBlock()
    PushBlock()
    if mMarkHilitedText()
        Copy()
    endif
    PopBlock()
end mCopyBlock

/*** mReplaceBlock ******************************************************
 *                                                                      *
 * When text is compared and differences are encountered, both files    *
 * will have the difference text highlighted.  This macro will replace  *
 * the highlighted block in the current file with the highlighted block *
 * in the other file.                                                   *
 *                                                                      *
 * Called by:   mProcessKeystrokes()                                    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mReplaceBlock()
    integer MyBeginLine, MyEndline
    string fn[MAXPATH]

    if WindowID() == 1
        MyBeginLine = Begin_Line1
        MyEndline = End_Line1
    else
        MyBeginLine = Begin_Line2
        MyEndline = End_Line2
    endif
    GotoWindow(iif(WindowID() == 1, 2, 1))
    fn = CurrFilename()
    GotoWindow(iif(WindowID() == 1, 2, 1))

    if MyBeginLine <> -1 and MyEndLine <> -1
        if CurrLine() < MyBeginLine or CurrLine() > MyEndLine
            DisplayMessage("Cursor NOT positioned correctly -- nothing done!",
                FALSE, 1)
            return()
        endif
        PushKey(<Tab>)
        if MsgBox("CMPFILES Query",
                'Transfer marked text from:  ' +
                '"' + Upper(fn) + '"' +
                ' to replace marked text in:  ' +
                '"' + Upper(CurrFilename()) + '"' +
                '?',
                YES_NO) <> 1
            return()
        endif
        PushPosition()
        PushBlock()

        mMarkBlock(MyBeginLine, MyEndLine)

        if IsBlockMarked()
            KillBlock()
        endif
        PopBlock()
        PushBlock()
        if IsBlockMarked()
            CopyBlock()
            GotoBlockBegin()
            MyBeginLine = CurrLine()
            GotoBlockEnd()
            MyEndLine = CurrLine()
        else
            MyBeginLine = CurrLine()
            MyEndLine = CurrLine()
        endif
        if WindowID() == 1
            Begin_Line1 = MyBeginLine
            End_Line1 = MyEndline
        else
            Begin_Line2 = MyBeginLine
            End_Line2 = MyEndline
        endif
        PushKey(<F11>)              // resync both differences
        PopBlock()
        PopPosition()
    else
        DisplayMessage("NO difference text found yet!", FALSE, 1)
    endif
end mReplaceBlock

/*** mPaste *************************************************************
 *                                                                      *
 * Paste the main clipboard contents into the current file.             *
 *                                                                      *
 * Called by:   The EditMenu()                                          *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mPaste()
    PushKey(<Tab>)      // default to [No]
    if MsgBox("CMPFILES Query", 'Paste clipboard text into "' +
            Upper(CurrFilename()) + '"?', YES_NO) == 1
        Paste()
    endif
end mPaste

/*** mDoubleFind ********************************************************
 *                                                                      *
 * Find the same text in both windows.                                  *
 *                                                                      *
 * Called by:   mProcessKeystrokes()                                    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mDoubleFind()
    GotoWindow(iif(WindowID() == 1, 2, 1))
    PushPosition()
    Right()
    if Find()
        KillPosition()
    else
        PopPosition()
    endif
    GotoWindow(iif(WindowID() == 1, 2, 1))
    if Query(key) <> <escape>
        RepeatFind()
    endif
end mDoubleFind

/*** mDoubleRepeatFind **************************************************
 *                                                                      *
 * Do a repeat find for the same word in both windows.                  *
 *                                                                      *
 * Called by:   mProcessKeystrokes()                                    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mDoubleRepeatFind()
    GotoWindow(iif(WindowID() == 1, 2, 1))
    RepeatFind()
    GotoWindow(iif(WindowID() == 1, 2, 1))
    if Query(key) <> <escape>           // escape character?
        RepeatFind()
    endif
end mDoubleRepeatFind

/*** mDoubleFindWordAtCursor ********************************************
 *                                                                      *
 * Find the word pointed to by the cursor in both windows.              *
 *                                                                      *
 * Called by:   mProcessKeystrokes()                                    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mDoubleFindWordAtCursor()
    string s[80]

    s = GetWord(TRUE)
    GotoWindow(iif(WindowID() == 1, 2, 1))
    if Length(s)
        Find(s, Query(FindOptions) + '+')
    endif
    GotoWindow(iif(WindowID() == 1, 2, 1))
    if Length(s)
        if Query(key) <> <escape>           // escape character?
            RepeatFind()
        endif
    endif
end mDoubleFindWordAtCursor

keydef EditInPlaceKeys
<Alt E>             EndProcess()
<Shift Escape>      EndProcess()
end
integer proc EditInPlace()
    integer idWin = WindowID()
    integer id1
    integer id2
    string st1[MAXPATH]
    string st2[MAXPATH]
    integer cwba

    GotoWindow(1)
    id1 = GetBufferId()
    st1 = CurrFilename()
    GotoWindow(2)
    id2 = GetBufferId()
    st2 = CurrFilename()
    GotoWindow(idWin)

    Begin_Line1 = -1
    Begin_Line2 = -1
    End_Line1 = -1
    End_Line2 = -1
    UnmarkBlock()

    RestoreWinState()
    RestoreHookState()
    cwba = Set(CurrWinBorderAttr, Color(bright red on black))
    Enable(EditInPlaceKeys)
    UpdateDisplay()
    DisplayMessage("Editing file...      <Alt E> or <Shift Escape> to resume comparison.", FALSE, 1)
    Process()
    Disable(EditInPlaceKeys)
    Set(CurrWinBorderAttr, cwba)
    SaveHookState()
    SaveWinState()

    if GetBufferId(st1) <> id1 or GetBufferId(st2) <> id2
        return (FALSE)
    endif

    OneWindow()
    GotoBufferId(id1)
    if GetGlobalInt("VWindowFlg")
        VWindow()
    else
        HWindow()
    endif
    GotoWindow(2)
    GotoBufferId(id2)
    GotoWindow(idWin)

    DisplayMessage("Resuming comparison...", FALSE, 1)

    return (TRUE)
end

/*** CPBMENUS --- PULLDOWN MENUS FOR CMP2BKUP ***************************
 *                                                                      *
 * SOFTWARE:    CMP2BKUP                                                *
 * VERSION:     1.00                                                    *
 * DATE:        May 8, 1994                                             *
 * REV. DATE:   July 15th, 1994                                         *
 * AUTHOR:      Ian Campbell                                            *
 * TYPE:        Include file for CMP2BKUP                               *
 *                                                                      *
 ************************************************************************/

/*** mAbout *************************************************************
 *                                                                      *
 * Provides an "About Box" for CMP2BKUP.                                *
 *                                                                      *
 * Called by:   menu item Help/About                                    *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

string stTitleAbout[] = "About CmpFiles"
datadef dAbout
"      CMPFILES - Version 1.20"
""
"   Split the screen and compare   "
"  two text files.  Find and mark"
"   the next difference block in"
"  each file.  Resync both files"
"  to the end of these difference"
"             blocks."
""
"     Written by Ian Campbell"
"         July 15th, 1994"
""
"   Modifications by Chris Antos"
"        1996 through 2002"
""
end

proc mAbout()
    if CreateTempBuffer()
        PushBlock()
        InsertData(dAbout)
        MsgBoxBuff("", Addr(stTitleAbout), 0)
        AbandonFile()
        PopBlock()
    endif
end mAbout

Menu FileMenu()
    history

    "&Edit              <Alt E>"    ,   PushKey(<Alt E>)
    "&Save"                         ,   SaveFile()
    "Save &As..."                   ,   SaveAs()
    ""                              ,                       , Divide
    "E&xit CMPFILES   <Escape>"    ,   PushKey(<escape>)
end

Menu EditMenu()
    history

    "&Copy             <Ctrl C>",   mCopyBlock()
    "&Paste...         <Ctrl V>",   mPaste()
    ""                          ,                       , Divide
    "&Replace...       <R>"     ,   PushKey(<r>)
    "&Block to File... <Alt W>" ,   mSaveBlock()
end

Menu ViewMenu()
    history

    "&Vert/Horiz Window <SpaceBar>" ,   PushKey(<spacebar>)
    "&Resize..."                    ,   ResizeWindow()
    "Other &Window           <Tab>" ,   GotoWindow(iif(WindowID() == 1, 2, 1))
    ""                              ,                       , Divide
    "&25-Line                     " ,   mToggleWindowTypes(WindowID(),
                                            VIDEO_FLG, _25_LINES_)
    "3&6-Line                     "  ,   mToggleWindowTypes(WindowID(),
                                            VIDEO_FLG, _36_LINES_)
    "&44-Line                     "  ,   mToggleWindowTypes(WindowID(),
                                            VIDEO_FLG, _44_LINES_)
    "&50-Line                     "  ,   mToggleWindowTypes(WindowID(),
                                            VIDEO_FLG, _50_LINES_)
end

Menu SearchMenu()
    history

    "Begin/Continue &Comparison <Enter>"   ,   PushKey(<enter>)
    "Difference Blocks &Begin     <F11>"   ,   PushKey(<F11>)
    "Difference Blocks &End       <F12>"   ,   PushKey(<F12>)
    ""                                     ,                       , Divide
    "&Find...                  <Ctrl F>"   ,   PushKey(<Ctrl F>)
    "Find &Again               <Ctrl L>"   ,   PushKey(<Ctrl L>)
    "Find &Word at Cursor       <Alt =>"   ,   PushKey(<Alt =>)
    ""                                  ,                          , Divide
    "Be&ginning of Files    <Ctrl PgUp>"   ,   PushKey(<Ctrl PgUp>)
    "En&d of Files          <Ctrl PgDn>"   ,   PushKey(<Ctrl PgDn>)
    "Go to &Lines...           <Ctrl J>"   ,   PushKey(<Ctrl J>)
    ""                                     ,                       , Divide
    "&Toggle WhiteSpace            <W>"   ,   PushKey(<w>)
    "&Ignore Case                   <I>"   ,   PushKey(<i>)
end

Menu HelpMenu()
    "&Help Screen  "                   ,   mDisplayHelpScreen()
    "&About  "                         ,   mAbout()
end

MenuBar MainMenu()
    history

    "&File"    ,    FileMenu()
    "&Edit"    ,    EditMenu()
    "&View"    ,    ViewMenu()
    "&Search"  ,    SearchMenu()
    "&Help"    ,    HelpMenu()
end

/*** CMP2BKUP.KEY -- KEY FILE FOR CMP2BKUP ******************************
 *                                                                      *
 * SOFTWARE:    CMP2BKUP                                                *
 * VERSION:     1.00                                                    *
 * DATE:        May 8, 1993                                             *
 * REV. DATE:   July 15th, 1994                                         *
 * AUTHOR:      Ian Campbell                                            *
 * TYPE:        Include file for CMP2BKUP                               *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 * This file does not handle keystrokes in the conventional sense,      *
 * rather, it sets up a GetKey() loop and handles the keyboard          *
 * directly, rather than implicitely, or via a keydef file.             *
 *                                                                      *
 ************************************************************************/

/*** mSyncTwoScreenKeys *************************************************
 *                                                                      *
 * This macro is called twice, once for one window, and again           *
 * for the other window.  Keystrokes which are to be reflected          *
 * simultaneously into both windows are handled here.                   *
 * If you don't like the dual screen keystroke choices, then here is    *
 * the place to change them!                                            *
 *                                                                      *
 * Called by:   mProcessKeystrokes()                                    *
 *                                                                      *
 * Enter With:  the keystroke to handle, and the window to do it in.    *
 *                                                                      *
 * Returns:     TRUE is the keystroke was recognized, otherwise FALSE.  *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mSyncTwoScreenKeys(integer keystroke, integer win)
    integer CurrentWinID = WindowID(), ReturnCode = TRUE

    GotoWindow(win)                     // switch to the requested window
    case keystroke
        when <CursorUp>, <GreyCursorUp>
            if CurrLine() > (Query(WindowRows) / 2)
                RollUp()
            else
                Up()
            endif
        when <CursorDown>, <GreyCursorDown>
            if CurrLine() > (Query(WindowRows) / 2)
                RollDown()
            else
                Down()
            endif
        when <Grey->, <F11>
            mPositionAtMark(win, BEGINNING)
        when <Grey+>, <F12>
            mPositionAtMark(win, ENDING)
        when <Ctrl Home>, <Ctrl GreyHome>, <Ctrl PgUp>, <Ctrl GreyPgUp>
            BegFile()
        when <Ctrl End>, <Ctrl GreyEnd>, <Ctrl PgDn>, <Ctrl GreyPgDn>
            EndFile()
            BegLine()
        when <CursorRight>, <GreyCursorRight>
            mRollRight()
        when <CursorLeft>, <GreyCursorLeft>
            mRollLeft()
        when <PgUp>, <GreyPgUp>
            PageUp()
        when <PgDn>, <GreyPgDn>
            PageDown()
        when <Home>, <GreyHome>
            BegLine()
        when <End>, <GreyEnd>
            EndLine()
        when <WheelUp>  WheelUp()
        when <WheelDown>WheelDown()
        otherwise
            ReturnCode = FALSE
    endcase
    ScrollToCenter()  // center the cursor line vertically
    GotoWindow(CurrentWinID)
    return(ReturnCode)
end mSyncTwoScreenKeys

/*** mProcessKeystrokes *************************************************
 *                                                                      *
 * This macro processes and handles all keystrokes, including ones that *
 * must be passed simultaneously through to both windows.               *
 * If you don't like the single screen keystroke choices, here is       *
 * the place to change them!                                            *
 *                                                                      *
 * Called by:   mCompareToBkup()                                        *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       The main macro loop exists here.  All macro processing  *
 *              is suspended via the call to GetKey() (until a key is   *
 *              actually pressed).                                      *
 *                                                                      *
 ************************************************************************/

proc mProcessKeystrokes()
    integer keystroke, CurrentWinID, MustMark = 0, CurrentTime

    SaveWinState()
    UpdateDisplay(_ALL_WINDOWS_REFRESH_)

    loop                            // loop until break
        if Query(ShowHelpLine)
            VGotoXYAbs(1, iif(Query(StatusLineAtTop), Query(ScreenRows), 1))
            PutHelpLine("{COMPARE:}  {R}-Replace {Enter}-Next {MENU}-Menu {F1}-Help")
            Set(Attr, Query(HelpTextAttr))
            ClrEol()
        endif

        CurrentWinID = WindowID()
        if isMacroLoaded("whtspc")
            GotoWindow(1)
            ExecMacro("WHTSPC_DrawWhitespace")
            GotoWindow(2)
            ExecMacro("WHTSPC_DrawWhitespace")
            GotoWindow(CurrentWinID)
        endif
        keystroke = GetKey()        // wait for a key to be pressed
        case keystroke
            when <F10>, <Alt M>, <CtrlAlt M>, <Ctrl K>
                PushKey(<Enter>)
                MainMenu()
            when <r>, <Alt R>, <Shift R>
                mReplaceBlock()
            when <Ctrl C>
                mCopyBlock()
            when <Ctrl V>
                mPaste()
            when <Alt W>
                mSaveBlock()
            when <Ctrl L>, <l>, <Shift L>
                mDoubleRepeatFind()
            when <Alt =>
                mDoubleFindWordAtCursor()
            when <Ctrl F>, <Alt S>
                mDoubleFind()
            when <Alt CursorDown>, <Alt GreyCursorDown>
                PushKey(<CursorDown>)
                ResizeWindow()
            when <Alt CursorLeft>, <Alt GreyCursorLeft>
                PushKey(<CursorLeft>)
                ResizeWindow()
            when <Alt CursorRight>, <Alt GreyCursorRight>
                PushKey(<CursorRight>)
                ResizeWindow()
            when <Alt CursorUp>, <Alt GreyCursorUp>
                PushKey(<CursorUp>)
                ResizeWindow()
            when <Shift PgUp>
                PageUp()
            when <Ctrl CursorUp>, <Ctrl GreyCursorUp>, <Shift CursorUp>
                RollUp()
            when <Shift PgDn>
                PageDown()
            when <Ctrl CursorDown>, <Ctrl GreyCursorDown>, <Shift CursorDown>
                RollDown()
            when <c>, <Shift C>, <v>, <Shift V>
                mToggleWindowTypes(CurrentWinID, VIDEO_FLG, 0)
            when <Enter>, <GreyEnter>, <F7>
                if not mSearchTwoFiles()
                    DisplayMessage("Cannot resync files past this point...",
                        FALSE, 1)
                endif
                MustMark = 1
            when <Escape>, <Alt Q>, <Alt X>
                if mEscapeProcess()
                    goto Out
                endif
                MustMark = 1
            when <F1>, <h>, <Shift H>, <Alt H>
                mDisplayHelpScreen()
            when <g>, <Shift G>, <Alt G>, <j>, <Ctrl J>, <Shift J>, <Alt J>
                mGotoLines(CurrentWinID)
            when <Spacebar>
                // vertical / horizontal toggle
                mToggleWindowTypes(CurrentWinID, NO_VIDEO_FLG, 0)
            when <LeftBtn>, <RightBtn>
                if MouseHotSpot() == _NONE_
                    while MouseKeyHeld()    // wait for the mouse to be released
                    endwhile
                    PushKey(<enter>)
                    MainMenu()
                endif
                if MouseState == 1
                    CurrentTime = GetClockTicks()
                    while MouseKeyHeld()
                        if GetClockTicks() > CurrentTime + Query(MouseHoldTime)
                            break
                        endif
                    endwhile
                    MouseState = 2          // full speed ahead state
                endif
                if MouseKeyHeld()
                    PushKey(keystroke)
                    mProcessMouse(keystroke)
                    if MouseState == 0      // entry state
                        MouseState = 1      // pause state
                    endif
                else
                    MouseState = 0          // reset to the entry state
                endif
                MustMark = 1
            when <Tab>, <Shift Tab>
                GotoWindow(iif(WindowID() == 1, 2, 1))
                ScrollToCenter()  // center the cursor line
                MustMark = 1
            when <w>, <Shift W>
                mHandleWhiteSpace()
            when <i>, <Shift I>
                mHandleIgnoreCase()
            when <Alt E>
                if not EditInPlace()
                    Warn("Unable to find original buffers, terminating CMPFILES.")
                    goto Out
                endif
            otherwise
                if mSyncTwoScreenKeys(keystroke, 1)     // window one
                    mSyncTwoScreenKeys(keystroke, 2)    // window two
                elseif keystroke <> <Alt>
                    DisplayMessage('Invalid Key! <' + KeyName(keystroke) + '>  <Enter> = Compare, <F1> = Help, <F10> = Menu', FALSE, 2)
                endif
        endcase
        mMarkDifferenceText(MustMark)       // mark differences on screen now
        MustMark = 0
    endloop

Out:
    RestoreWinState()
end mProcessKeystrokes

/*** mListIt ************************************************************
 *                                                                      *
 * Display files in the editor's ring.  Allow cursor movement,          *
 * and when mListIt exits, place the cursor on the line that was        *
 * selected, allowing a primative picklist selection.                   *
 *                                                                      *
 * Called by:   mListOpenFiles()                                        *
 *                                                                      *
 * Enter With:  The title of the window, and the window width           *
 *                                                                      *
 * Returns:     Non zero if exited with ENTER, 0 if exited with ESCAPE. *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mListIt(string title, integer width)
    width = width + 4
    if width > Query(ScreenCols)
        width = Query(ScreenCols)
    endif
    return (List(title, width))
end mListIt

/*** mGetNameFromRing ***************************************************
 *                                                                      *
 * See if a name (drive, path, and extention stripped off) is in the    *
 * ring.  If it is, then edit that file, and return a 1 to say that     *
 * we've done it.  If not, then just return a 0.                        *
 *                                                                      *
 * Called by:   mCompareToBkup()                                        *
 *                                                                      *
 * Enter With:  the filename string to check for.                       *
 *                                                                      *
 * Returns:     TRUE if filename found, FALSE if not found.             *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mGetNameFromRing(string filename)
    integer start_file, filelist, id, maxl, total, n
    string fn[MAXPATH] = "", fn2[MAXPATH] = ""

    n = NumFiles() + (BufferType() <> _NORMAL_)
    if n == 0
        return ()
    endif
    maxl = 0
    total = 0
    start_file = GetBufferId()                 // Save current buffer
    filelist = CreateTempBuffer()
    if filelist == 0
        warn("Can't create filelist")
        return ()
    endif
    GotoBufferID(start_file)
    id = GetBufferid()
    while n
        fn = CurrFilename()
        if Lower(filename) == Lower(SplitPath(fn, _NAME_)) and CurrExt() <> ".cbk"
            fn2 = fn
            total = total + 1
            if length(fn) > maxl
                maxl = length(fn)
            endif
            GotoBufferID(filelist)
            AddLine(fn)
            GotoBufferID(id)
        endif
        NextFile(_DONT_LOAD_)
        id = GetBufferid()
        n = n - 1
    endwhile
    GotoBufferID(filelist)
    BegFile()
    PushBlock()
    UnMarkBlock()
    MarkStream()
    EndFile()
    MarkStream()
    Sort()
    PopBlock()
    BegFile()
    if total > 1
        while not mListIt("Select Original File", maxl)
        endwhile
        fn2 = GetText(1, sizeof(fn2))
    endif
    AbandonFile(filelist)
    if Length(fn2)
        EditThisFile(fn2)
    else
        GotoBufferID(start_file)
    endif
end mGetNameFromRing

string stCurrFilename[MAXPATH] = ""
proc GuessFilename()
    if FileExists(GetText(1, CurrLineLen())) & _DIRECTORY_
        PushPosition()
        EndLine()
        if Left()
            if not isDirSeparator(Chr(CurrChar()))
                EndLine()
                InsertText(GetDirSeparator(), _INSERT_)
            endif
            if FileExists(GetText(1,
                    CurrLineLen())+SplitPath(stCurrFilename, _NAME_|_EXT_))
                EndLine()
                InsertText(SplitPath(stCurrFilename, _NAME_|_EXT_), _INSERT_)
            endif
        endif
        PopPosition()
    endif
end

keydef FilenamePromptKeys
<Enter>         GuessFilename() EndProcess(TRUE)
<GreyEnter>     GuessFilename() EndProcess(TRUE)
end

proc FilenamePromptStartup()
    Enable(FilenamePromptKeys)
end

proc FilenamePromptCleanup()
    Disable(FilenamePromptKeys)
end

/*** mGetValidFileName **************************************************
 *                                                                      *
 * This routine will only allow a legit filename.   Picklists are       *
 * provided to resolve it.  Both the current directory, and then the    *
 * current filename's directory are sequentially checked.               *
 *                                                                      *
 * Called by:   mLoadBackupFile()                                       *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     FALSE if user exits "Ask" with escape, TRUE if filename *
 *              resolved.                                               *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mGetValidFileName()
    integer FileThere, PathQualified, i, NameLength
    string name[MAXPATH] = ""
    string s1[MAXPATH] = ""

    loop
        PathQualified = FALSE
        UpdateDisplay()
        stCurrFilename = CurrFilename()
        Hook(_PROMPT_STARTUP_, FilenamePromptStartup)
        Hook(_PROMPT_CLEANUP_, FilenamePromptCleanup)
        FileThere = AskFilename('Compare "'
            + Upper(SplitPath(CurrFilename(), _NAME_ | _EXT_)) +
                '" to:  ["ARC", "ARJ", "LZH", and "ZIP" files are also OK]',
            GS3, _MUST_EXIST_, cmp2bkup_file_history)
        UnHook(FilenamePromptStartup)
        UnHook(FilenamePromptCleanup)
        if not FileThere
            return(FALSE)
        endif
        if Length(GS3)
            if GS3 == ".*"
                GS3 = "*.*"
            endif
            if Length(SplitPath(GS3, _DRIVE_ | _PATH_))
                if SplitPath(GS3, _DRIVE_ | _PATH_) <> "."
                    PathQualified = TRUE
                endif
            endif
            if not isDirSeparator(GS3[Length(GS3)])
                if mDirExists(GS3)
                    GS3 = GS3 + GetDirSeparator()
                endif
            endif
        endif
        GS3 = ExpandPath(GS3)
        if not Length(SplitPath(GS3, _EXT_))    // no extention?
            if not FileExists(GS3)
                GS3 = GS3 + ".*"
            endif
        endif
        i = 1
        name = Splitpath(GS3, _NAME_ | _EXT_)
        NameLength = Length(name)
        while i <= NameLength
            // any wildcard in name or extention?
            if name[i] == '*' or name[i] == '?'
                // wildcard in name but no extention?
                if not Length(SplitPath(GS3, _EXT_))
                    // make sure that all extentions are checked
                    GS3 = GS3 + ".*"
                endif
                s1 = ""
                if FileExists(GS3)
                    FileThere = TRUE
                    s1 = PickFile(GS3)
                endif
                if not Length(s1)
                    if not PathQualified        // path not specified?
                        //paths different?
                        if Lower(SplitPath(GS3, _DRIVE_ | _PATH_)) <>
                            Lower(SplitPath(CurrFilename(), _DRIVE_ | _PATH_))
                            // try the current directory
                            if FileExists(SplitPath(CurrFilename(),
                                _DRIVE_ | _PATH_) + SplitPath(GS3,
                                _NAME_ | _EXT_))
                                FileThere = TRUE
                                s1 = PickFile(SplitPath(CurrFilename(),
                                    _DRIVE_ | _PATH_) + SplitPath(GS3,
                                    _NAME_ | _EXT_))
                            endif
                        endif
                    endif
                endif
                GS3 = s1
                break           // exit while loop
            endif
            i = i + 1
        endwhile
        if Length(GS3)          // got something in GS3?
            if not FileExists(GS3) and not PathQualified
               // try the current directory
                GS3 = SplitPath(CurrFilename(), _DRIVE_ | _PATH_)
                    + SplitPath(GS3, _NAME_ | _EXT_)
            endif
        endif
        if Length(GS3)
            if FileExists(GS3)
                break               // then exit loop
            endif
        endif
        if not FileThere
            // force an error message
            PickFile(">")
        endif
        GS3 = ""
    endloop
    return(TRUE)
end mGetValidFileName

/*** mLoadBackupFile ****************************************************
 *                                                                      *
 * Prompt the user for the file to compare.  This may be any filename   *
 * in any subdirectory.  Alternatively, this file may exist inside an   *
 * archived backup ("ARC", "ARJ", "LZH", AND "ZIP" archives are         *
 * currently supported).  Wildcards are also supported, and are         *
 * resolved through picklists.                                          *
 *                                                                      *
 * Once a backup file has been chosen, the screen is split vertically,  *
 * and the backup file is placed into the second window.                *
 *                                                                      *
 * For archived files, the file must first be extracted to a            *
 * temporary subdirectory before it can be used.  The environment is    *
 * first checked to see if a "TEMP=subdirectory" entry exists.  If it   *
 * does exist, then the backup file will be extracted to this           *
 * subdirectory.  If this entry does not exist, then a "TSETEMP"        *
 * subdirectory is created.  The backup file is then extracted into     *
 * this subdirectory, and loaded into Window-2 of TSE.  It is then      *
 * renamed to the same pathname as the original, but with an extention  *
 * of "CBK" (Compare BacKup).  The backup file, and the "TSETEMP"       *
 * subdirectory (if created) are then erased from the disk.             *
 *                                                                      *
 * Called by:   mCompareToBkup()                                        *
 *                                                                      *
 * Enter With:  GS1 is preset to the current filename.                  *
 *              GS2 is preset to the drive:path\ for the current        *
 *              filename.                                               *
 *                                                                      *
 * Returns:     TRUE if successful, otherwise FALSE                     *
 *                                                                      *
 * Notes:       If the "CBK" file already exists in TSE's ring of       *
 *              files, then use that file instead subtracting it again  *
 *              from the disk.  (The CBK file must be deleted before a  *
 *              another one can be extracted from the disk).            *
 *                                                                      *
 ************************************************************************/

integer proc mLoadBackupFile(string arg)
    integer TempDirIsMine = 0
    string Extention[5] = ""
    string TempDir[MAXPATH] = ""
    string TempFileName[MAXPATH] = CurrFilename()
    integer OldCursor

    OldCursor = Query(Cursor)
    GS3 = arg
    if GS3 == "" and not GetGlobalInt("BeginLine1Save")
        GS3 = "*.ZIP"
    endif
    if not mGetValidFileName()             // get a valid filename
        return(FALSE)
    endif

    SetGlobalInt("VWindowFlg", WinDefaultVert)
    OneWindow()
    if GetGlobalInt("VWindowFlg")
        VWindow()
    else
        HWindow()
    endif
    Extention = SplitPath(GS3, _EXT_)
    case Lower(Extention)
        when ".zip", ".lzh", ".arc", ".arj"
            if not FileExists(GS3)
                Warn("File does not exist!")
                return(FALSE)
            endif
            TempDir = GetEnvStr("TEMP")
            if not Length(TempDir)
                TempDir = GS2 + "TSETEMP"
                TempDirIsMine = TRUE
            else
                if isDirSeparator(TempDir[Length(TempDir)])
                    TempDir = SubStr(Tempdir, 1, Length(TempDir) - 1)
                endif
                // does the stated subdirectory exist?
                if not mDirExists (TempDir)
                    // no, then create my own!
                    TempDir = GS2 + "TSETEMP"
                    TempDirIsMine = TRUE
                endif
            endif
            if TempDirIsMine
                if not mDirExists(TempDir)
                    Dos ("mkdir " + TempDir, _DONT_CLEAR_|_DONT_PROMPT_)
                endif
            endif
            Set(Cursor, OFF)
            mPopWinCentered(29, 4, 4, "MESSAGE", Query(CurrWinBorderAttr))
            Set(Attr, Query(MsgAttr))
            ClrScr()
            Write(" Decompressing backup file")
            GotoXY(1, 2)
            Write(" Please wait...")
            case Lower(Extention)
                when ".zip"
                    // Pkunzip must be in the path (or the same directory) for this to work
                    // or a rather BOGUS "FILE NOT FOUND" error message will result.
                    Dos ("PKUNZIP -o " + GetShortPath(GS3) + " " + GetShortPath(GS1) + " " + GetShortPath(TempDir+"\") + ">NUL",
                        _DONT_CLEAR_ | _DONT_PROMPT_)
                when ".lzh"
                    Dos ("LHA e " + GetShortPath(GS3) + " " + GetShortPath(TempDir+"\") + " " + GetShortPath(GS1) + ">NUL",
                        _DONT_CLEAR_ | _DONT_PROMPT_)
                when ".arj"
                    Dos ("ARJ e " + GetShortPath(GS3) + " " + GetShortPath(TempDir+"\") + " " + GetShortPath(GS1) + ">NUL",
                        _DONT_CLEAR_ | _DONT_PROMPT_)
                when ".arc"
                    if FileExists(TempDir + "\" + GS1)
                        EraseDiskFile(TempDir + "\" + GS1)
                    endif
                    Dos ("PKXARC " + GetShortPath(GS3) + " " + GetShortPath(GS1) + " " + GetShortPath(TempDir+"\") + ">NUL",
                        _DONT_CLEAR_ | _DONT_PROMPT_)
            endcase
            PopWinClose()
            Set(Cursor, OldCursor)
            if not FileExists(TempDir + GetDirSeparator() + GS1)
                Warn(GS1, " not found in ", GS3)
                return(FALSE)
            endif
            if EditThisFile(TempDir + GetDirSeparator() + GS1)
                // assume this CBK already exists in current dir, and say "YES"
                // to the overwrite query when changing the name to it.
                PushKey(<Y>)
                ChangeCurrFilename(GS2 + SplitPath(CurrFilename(), _NAME_)
                    + ".cbk")
                mIsEscKeyPressed()          // flush the <Y> key (if not needed)
                FileChanged(FALSE)          // say no change occurred in file
            endif
            EraseDiskFile(TempDir + GetDirSeparator() + GS1)
            if TempDirIsMine
                Dos ("rmdir " + QuotePath(TempDir), _DONT_CLEAR_ | _DONT_PROMPT_)
            endif
        otherwise
            if EquiStr(GS3,TempFileName)
                Message("CANNOT COMPARE FILE TO ITSELF!")
                return(FALSE)
            endif
            if FileExists(GS3)
                EditThisFile(GS3)
            else
                return(FALSE)
            endif
    endcase
    return(TRUE)
end mLoadBackupFile

/*** mCompareToBkup *****************************************************
 *                                                                      *
 * Check and see if a "CBK" file is the current filename. If it is,     *
 * try to open the first file found in the ring that matches that       *
 * filename.  If the current filename is not a "CBK" file, then try     *
 * and find a "CBK" file in the editor's ring that matches the current  *
 * filename (minus the extention).  If found, then immediately load it  *
 * it in window-2, in preparation for synchronized comparisons.  If     *
 * not, then call mLoadBackupFile(), ask the user for a filename, and   *
 * load the requested backup.  This backup may exist inside of an       *
 * archive file ("ARC", "ARJ", "LZH", and "ZIP" formats are supported). *
 *                                                                      *
 * Called by:   main()                                                  *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mCompareToBkup(string arg)
    integer WasBackup = 0

    // current file is the CBK file?
    if CurrExt() == ".cbk"
        WasBackup = 1                       // flag backup file active

        // switch main window to original file (best guess!)
        mGetNameFromRing(Splitpath(CurrFilename(), _NAME_))

        // still editing the CBK file?
        if CurrExt() == ".cbk"
            Warn("Sorry, the original file is no longer loaded,")
            return()
        endif
    endif
    GS1 = Splitpath(CurrFilename(), _NAME_ | _EXT_)
    GS2 = Splitpath(CurrFilename(), _DRIVE_ | _PATH_)
    // respect passed arg even if a .cbk file is already loaded
    if Length(arg)
        GS3 = arg
    else
        GS3 = GS2 + SplitPath(CurrFilename(), _NAME_) + ".cbk"
    endif
    BegLine()
    if GetBufferID(GS3)                 // is a CBK file already extracted?
        OneWindow()
        if GetGlobalInt("VWindowFlg")
            VWindow()
        else
            HWindow()
        endif
        GotoWindow(2)
        EditThisFile(GS3)                       // edit the CBK file now
        BegLine()
        GotoWindow(1)
        ScrollToCenter()
        GotoWindow(2)
        ScrollToCenter()

        // restore macro's previous values
        Begin_Line1 = GetGlobalInt("BeginLine1Save")
        Begin_Line2 = GetGlobalInt("BeginLine2Save")
        End_Line1 = GetGlobalInt("EndLine1Save")
        End_Line2 = GetGlobalInt("EndLine2Save")
    else
        if not mLoadBackupFile(arg)
            GotoWindow(1)
            OneWindow()
            return()
        endif
    endif
    GotoWindow(2)
    CID2 = GetBufferID()            // get the backup file ID
    GotoWindow(1)
    CID1 = GetBufferID()            // get the original file ID
    if WasBackup
        GotoWindow(2)               // if started as backup, stay in backup
    endif
    UpdateDisplay()
    mMarkDifferenceText(TRUE)
    ScrollToCenter()  // center the cursor line vertically
    DisplayMessage('CMPFILES Loaded -- press <F10> for the Menu System',
        TRUE, 1)
    mProcessKeystrokes()
end mCompareToBkup

integer proc same_file_in_2_windows()
    integer w_id, b_id, b_id2

    if NumWindows() <> 2
        return (true)
    endif

    if isZoomed()
        return (true)
    endif

    w_id = WindowId()
    b_id = GetBufferId()

    NextWindow()

    b_id2 = GetBufferId()

    GotoWindow(w_id)

    return (b_id == b_id2)
end

string proc get_arg()
    string arg[_MAX_PATH_]
    integer x1, x2

    compare_existing = false
    if NumWindows() == 2 and not same_file_in_2_windows()
        case MsgBox("Cmpfiles", "Compare Windows?", _YES_NO_CANCEL_)
            when 1
                x1 = Query(WindowX1)
                NextWindow()
                x2 = Query(WindowX1)
                arg = CurrFilename()
                NextWindow()
                if x1 < x2      // horizontal
                    SetGlobalInt("VWindowFlg", TRUE)       // vertical windows
                endif
                compare_existing = true
                return (arg)
        endcase
    endif

    if NumFiles() == 2
        case MsgBox("Cmpfiles", "Compare currently loaded files?", _YES_NO_CANCEL_)
            when 1
                NextFile()
                arg = CurrFilename()
                NextFile()
                compare_existing = true
                return (arg)
        endcase
    endif

    return ("")
end

/*** Main ***************************************************************
 *                                                                      *
 * This is the main entry point for Cmp2BkUp.  When the macro           *
 * completes, it will then be purged from memory.                       *
 *                                                                      *
 * Called by:   TSE as an external macro.                               *
 *                                                                      *
 * Enter With:  command line can be a filename to compare against.      *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       This is the entry point, and the purge point for this   *
 *              entire macro collection.                                *
 *                                                                      *
 ************************************************************************/

proc WhenLoaded()
    cmp2bkup_file_history = GetFreeHistory("Cmp2bkup:SecondaryFile")
end

string temp_fn[_MAX_PATH_]

proc Main()
    string temp_dir[_MAX_PATH_] = GetEnvStr("TEMP")
    string arg[255] = Query(MacroCmdLine)

    if g_fBusy
        Warn("CMPFILES is already active, use <Shift-Escape> to resume CMPFILES.")
        return()
    endif

    if arg == ""
        arg = get_arg()
    endif

    temp_fn = QuotePath(MakeTempName(temp_dir))
    ExecMacro("state -l -s -f" + temp_fn)

    SaveHookState()
    g_fBusy = TRUE
    mCompareToBkup(arg)                            // start things rolling now
    g_fBusy = FALSE
    PurgeMacro(CurrMacroFilename())      // purge the macro
    RestoreHookState()

    ExecMacro("state -l -r -f" + temp_fn)
    EraseDiskFile(temp_fn)
end Main
