/*************************************************************************
  ClipView    Simple ClipBoard Viewer utility for PROJECTS

  Author:     Ian Campbell (Contributing User)

  Date:       Jul 23, 1993 (Original: Ian Campbell)
              Jul 15, 1994 (Revised: Ian Campbell)
              Feb  9, 2002 Made menu non-OEM font compliant.

  Version:    1.00

  Overview:

  This file provides a simple ClipBoard viewer.  It allows ClipBoards
  "A" through "Z" to be viewed.  It was written as a support file for
  PROJECTS.

  Keys:       (none)

  Usage notes:

  This macro may be executed from the PROJECTS macro, or may be
  executed directly from Potpourri or the "Execute Macro:" prompt.

*************************************************************************/

integer WID = 0                     // global clipboard window ID

/*** mAdvanceClipBoard **************************************************
 *                                                                      *
 * If the "f12" key is pressed, then paste the next clipboard to        *
 * the buffer named "clipboard".  If the "f11" key is pressed, then     *
 * paste the previous clipboard to buffer "clipboard".                  *
 *                                                                      *
 * Called by:   mViewClipBoard()                                        *
 *                                                                      *
 * Enter With:  the key pressed, clipboard named type, the clipboard    *
 *              letter, and the clipboard name.                         *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mAdvanceClipBoard(integer MyKey, var integer ClipBoardIsNamed,
    var string ClipBoardLetter, var string ClipBoardName)
    integer i

    if not ClipBoardIsNamed
        case MyKey
            when <Grey+>, <F12>
                ClipboardLetter = "A"
            when <Grey->, <F11>
                ClipboardLetter = "Z"
        endcase
    endif
    i = 1
    while i <= 26
        case MyKey
            when <Grey->, <F11>
                if ClipBoardIsNamed or i > 1
                    if ClipBoardLetter == "A"
                        ClipBoardIsNamed = FALSE
                        break
                    else
                        ClipBoardLetter =
                            Chr(Asc(ClipBoardLetter) - 1)
                    endif
                endif
            when <Grey+>, <F12>
                if ClipBoardIsNamed or i > 1
                    if ClipBoardLetter == "Z"
                        ClipBoardIsNamed = FALSE
                        break
                    else
                        ClipBoardLetter =
                            Chr(Asc(ClipBoardLetter) + 1)
                    endif
                endif
        endcase
        ClipBoardName = "+++" + ClipBoardLetter
        if GetBufferID(ClipBoardName)
            ClipboardIsNamed = TRUE
            break
        endif
    i = i + 1
    endwhile
end mAdvanceClipBoard

/*** PopWinCentered *****************************************************
 *                                                                      *
 * This macro opens up a Popup Window, and centers it horizontally on   *
 * the screen.  The co-ordinates of the window are stored globally in   *
 * PopX1, PopX2, PopY1, and PopY2.                                      *
 *                                                                      *
 * Called by:   mShowClipboardHelp()                                    *
 *                                                                      *
 * Enter With:  Width, height, box type, and window message.            *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       PopX1, PopX2, PopY1, and PopY2 are globally made        *
 *              available for other routines who want to know the       *
 *              popup window's dimensions.                              *
 *                                                                      *
 ************************************************************************/

integer PopX1, PopY1, PopX2, PopY2
proc mPopWinCentered(integer WinWidth, integer WinHeight, integer BoxType,
    string BoxMessage, integer WindowColor)

    PopX1 = ((Query(ScreenCols) - WinWidth) / 2) + 1
    PopX2 = PopX1 + WinWidth -1
    PopY1 = ((25 - WinHeight) / 2) + 2
    PopY2 = PopY1 + WinHeight -1
    PopWinOpen(PopX1, PopY1, PopX2, PopY2, BoxType, BoxMessage, WindowColor)
end mPopWinCentered

/*** mWriteLineWithAttr *************************************************
 *                                                                      *
 * This macro writes a line of text to a popup window on the screen.    *
 * If the LineToColor matches the Current Line number, the line is      *
 * colored to MenuSelectLtrAttr.                                        *
 *                                                                      *
 * Called by:   mWriteTextInClipPopup()                                 *
 *                                                                      *
 * Enter With:  the text string to write, the current line number,      *
 *              and the line number to hilite.                          *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mWriteLineWithAttr(string CharLine, integer CurrentLine,
    integer LineToColor)
    Set(Attr, (iif(LineToColor == CurrentLine, Query(MenuSelectAttr),
        Query(MenuTextAttr))))

    // JHB - Replaced this line for non-OEM font compliance
    // WriteLine(CharLine)
    PutOemLine(CharLine,Length(CharLine))
    VGotoXY(1,VWhereY()+1)

end mWriteLineWithAttr

/*** mWriteTextInClipPopup **********************************************
 *                                                                      *
 * This macro writes text to a popup window on the screen               *
 *                                                                      *
 * Called by:   mViewClipBoard()                                        *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mWriteTextInClipPopup(integer LineToColor)

    BufferVideo() // JHB - Added video buffering to improve redraw
    ClrScr()
    mWriteLineWithAttr(' Main Clipboard       <Home>    ', 1, LineToColor)
    mWriteLineWithAttr(' Named Clipboard      <A>...<Z> ', 2, LineToColor)
    mWriteLineWithAttr(' Previous Clipboard   <F11>     ', 3, LineToColor)
    mWriteLineWithAttr(' Next Clipboard       <F12>     ', 4, LineToColor)
    mWriteLineWithAttr('', -1, LineToColor)
    mWriteLineWithAttr(' Clear Clipboard      <Alt C>   ', 6, LineToColor)
    mWriteLineWithAttr(' Paste                <Alt P>   ', 7, LineToColor)
    mWriteLineWithAttr(' Paste Over           <Ctrl P>  ', 8, LineToColor)
    mWriteLineWithAttr('', -1, LineToColor)
    Set(Attr, (iif(LineToColor == 10, Query(MenuSelectAttr),
        Query(MenuTextAttr))))
                 Write(' Exit = <RightBtn> or <Escape>  ')
    UnBufferVideo() // JHB - Added video buffering to improve redraw
end mWriteTextInClipPopup

/*** mHandleMenuItem ****************************************************
 *                                                                      *
 * This macro determines what menu function is being called base on     *
 * the value in HiliteItemCount.  The appropriate keystroke that will   *
 * later be required to do the function is then pushed.  Finally, a     *
 * decision is made whether or not to later pop the menu back up via    *
 * the ForceUpMenu flag.                                                *
 *                                                                      *
 * Called by:   mProcessPopupMenu()                                     *
 *                                                                      *
 * Enter With:  a flag that will control whether or not the menu is     *
 *              "stickey", and the line number to color hilite.         *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mHandleMenuItem(var integer ForceUpMenu, integer HiliteItemCount)
    case HiliteItemCount
        when 1          // main clipboard
            PushKey (<Home>)
            ForceUpMenu = TRUE
        when 2          // named clipboard (first)
            PushKey (<Grey+>)
            PushKey (<Home>)
            ForceUpMenu = TRUE
        when 3          // previous clipboard
            PushKey (<Grey->)
            ForceUpMenu = TRUE
        when 4          // next clipboard
            PushKey (<Grey+>)
            ForceUpMenu = TRUE
        when 6      // clear clipboard
            PushKey (<Alt C>)
            ForceUpMenu = TRUE
        when 7      // normal paste
            PushKey (<Alt P>)
            ForceUpMenu = TRUE
        when 8      // overwrite paste
            PushKey (<Ctrl P>)
            ForceUpMenu = TRUE
    endcase
end mHandleMenuItem

/*** mHandleClipMouse ***************************************************
 *                                                                      *
 * Determine whether or not a mouse hot spot was clicked.  If yes, then *
 * determine whether or not to obey the request.  Return TRUE if a      *
 * hotspot, FALSE if not a hotspot.                                     *
 *                                                                      *
 * Called by:   mProcessPopupMenum()                                    *
 *                                                                      *
 * Enter With:  a flag telling whether or not the clipboard menu is     *
 *              up or not up.                                           *
 *                                                                      *
 * Returns:     1 if a clipboard window hotspot clicked.                *
 *              0 if no hot spot clicked.                               *
 *              -1 if another window hotspot clicked (calling routine   *
 *              usually ignores this).                                  *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mHandleClipMouse(integer MenuIsUp)
    case MouseHotSpot()
        when _MOUSE_ZOOM_, _MOUSE_UP_, _MOUSE_DOWN_,
            _MOUSE_PAGEUP_, _MOUSE_PAGEDOWN_, _MOUSE_LEFT_,
            _MOUSE_RIGHT_, _MOUSE_VELEVATOR_, _MOUSE_HELEVATOR_,
            _MOUSE_TABLEFT_, _MOUSE_TABRIGHT_, _MOUSE_VRESIZE_,
            _MOUSE_HRESIZE_
            if MouseWindowID() == WID               // clipboard window?
                return(1)
            else
                if MenuIsUp
                    PushKey(<Spacebar>)             // keep the menu up
                endif
                return(-1)
            endif
        when _MOUSE_HWINDOW_, _MOUSE_VWINDOW_, _NONE_
            return(-1)
        when _MOUSE_CLOSE_
            if MouseWindowID() == WID               // clipboard window?
                PushKey(<Escape>)                   // shut it down
            else
                if MenuIsUp
                    PushKey(<Spacebar>)             // keep the menu up
                endif
            endif
            return(-1)
    endcase
    return(0)
end mHandleClipMouse

/*** mProcessPopupMenu **************************************************
 *                                                                      *
 * This macro is activated when either <spacebar>, <f10>, or <LeftBtn>  *
 * is pressed.  Mouse clicks, menu keystrokes, and cursor keys are all  *
 * handled here.                                                        *
 *                                                                      *
 * Called by:   mViewClipBoard                                          *
 *                                                                      *
 * Enter With:  a flag that will control whether or not the menu is     *
 *              "sticky", and the line number to color hilite.          *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mProcessPopupMenu(var integer ForceUpMenu, var integer HiliteItemCount)
    integer Keystroke
    string MyAsciiKey[2] = ""

    loop
        Keystroke = GetKey()
        case Keystroke
            when <LeftBtn>
                case mHandleClipMouse(TRUE)
                    when 1
                        PushKey(Keystroke)
                        ForceUpMenu = TRUE
                    when 0
                        // mouse click within the box?
                        if Query(MouseX) > PopX1 and Query(MouseX) < PopX2 and
                            Query(MouseY) > PopY1 and Query(MouseY) < PopY2
                            HiliteItemCount = Query(MouseY) - PopY1
                            mHandleMenuItem(ForceUpMenu, HiliteItemCount)
                        endif
                endcase
                return()
            when <RightBtn>             // drop out of the viewer
                return()
            when <CursorDown>           // move hilites down
                repeat
                    HiliteItemCount = iif(HiliteItemCount == 10, 1,
                        HiliteItemCount + 1)
                until
                    HiliteItemCount <> 5 and HiliteItemCount <> 9
            when <CursorUp>
                repeat
                    HiliteItemCount = iif(HiliteItemCount == 1, 10,
                        HiliteItemCount - 1)
                until
                    HiliteItemCount <> 5 and HiliteItemCount <> 9
            when <Escape>
                return()
            when <Enter>, <GreyEnter>
                mHandleMenuItem(ForceUpMenu, HiliteItemCount)
                return()
            when <PgUp>, <PgDn>, <CursorLeft>, <CursorRight>,
                <Ctrl PgDn>, <Ctrl End>, <Ctrl PgUp>, <Ctrl Home>
                PushKey(Keystroke)
                ForceUpMenu = TRUE
                return()
            when <Home>, <Grey+>, <F12>, <Grey->, <F11>, <Alt C>, <Alt P>,
                <Ctrl P>
                Pushkey(Keystroke)
                case Keystroke
                    when <Home>
                        HiliteItemCount = 1
                        ForceUpMenu = TRUE
                    when <Grey->, <F11>
                        HiliteItemCount = 3
                        ForceUpMenu = TRUE
                    when <Grey+>, <F12>
                        HiliteItemCount = 4
                        ForceUpMenu = TRUE
                    when <Alt C>
                        HiliteItemCount = 6
                        ForceUpMenu = TRUE
                    when <Alt P>
                        HiliteItemCount = 7
                        ForceUpMenu = TRUE
                    when <Ctrl P>
                        HiliteItemCount = 8
                        ForceUpMenu = TRUE
                endcase
                return()
            otherwise
                MyAsciiKey = Upper(Chr(Keystroke & 0xff))
                if (Asc(MyAsciiKey) >= Asc("A") and
                    Asc(MyAsciiKey) <= Asc("Z")) or
                    (Asc(MyAsciiKey) >= Asc("a") and
                    Asc(MyAsciiKey) <= Asc("z"))
                    PushKey(Keystroke)
                    HiliteItemCount = 2
                    ForceUpMenu = TRUE
                    return()
                endif
        endcase
        mWriteTextInClipPopup(HiliteItemCount)
    endloop
end mProcessPopupMenu

/*** UpdateClipboard ****************************************************
 *                                                                      *
 * Determine the clipboard type (main or named clipboard).  Paste the   *
 * clipboard data into an empty clipboard viewer buffer.  Check for     *
 * two action situations -- PasteIntoApplication and ClearClipBoard.    *
 * If PasteIntoApplication is set to an <Alt P>, then paste that        *
 * clipboard's data into the main file.  If PasteIntoApplication is     *
 * set to an <Ctrl P>, then do a paste-over into the main file.  If     *
 * ClearClipBoard is set, then clear that clipboard's data.             *
 *                                                                      *
 * Called by:   mViewClipBoard()                                        *
 *                                                                      *
 * Enter With:  - the clipboard name.                                   *
 *              - the clipboard type.                                   *
 *              - either a 0 (do nothing), an <Alt P> (paste), or a     *
 *                <Ctrl P> (paste over).                                *
 *              - the original file ID (before the clipboard came up).  *
 *              - true to clear the clipboard, false to do nothing.     *
 *                                                                      *
 *                                                                      *
 * Returns:     true tells the parent to continue, false tells the      *
 *              parent to execute a break.                              *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc UpdateClipboard(string ClipBoardName, integer ClipBoardIsNamed,
    integer PasteIntoApplication, integer MyOldID, var integer ClearClipBoard)
    integer OldClipBoardID

    EmptyBuffer()
    if ClipBoardIsNamed
        if GetBufferID(ClipBoardName)
            OldClipBoardID = Set(ClipBoardID, GetBufferID(ClipBoardName))
            // show named clipboard data
            Paste()
            if PasteIntoApplication
                GotoBufferID(MyOldID)   // switch to application
                if PasteIntoApplication == <Alt P>
                    Paste()             // paste into application
                else
                    Paste(_OVERWRITE_)
                endif
                // back to original clipboard
                Set(ClipBoardID, OldClipBoardID)
                return(FALSE)
            elseif ClearClipBoard
                AbandonFile(GetBufferID(ClipBoardName))
                // drop to the previous clipboard
                PushKey(<Grey->)
                ClearClipBoard = FALSE
            endif
            Set(ClipBoardID, OldClipBoardID)
        endif
    else                            // Main clipboard
        Paste()                     // show main clipboard data
        if PasteIntoApplication
            GotoBufferID(MyOldID)
            if PasteIntoApplication == <Alt P>
                    Paste()             // paste into application
                else
                    Paste(_OVERWRITE_)
                endif
            return(FALSE)
        elseif ClearClipBoard
            GotoBufferID(Query(ClipBoardID))
            EmptyBuffer()           // empty the main clipboard
            PushKey(<Home>)         //lets see if we got it
            ClearClipBoard = FALSE
        endif
    endif
    return(TRUE)
end UpdateClipboard

/*** mYesNoMenu *********************************************************
 *                                                                      *
 * Display two menu items -- Yes, and No.  Center the menu for an 80 by *
 * 25 screen.                                                           *
 *                                                                      *
 * Called by:   mYesNo                                                  *
 *                                                                      *
 * Enter With:  the text to display at the top of the menu box.         *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

Menu mYesNoMenu()
    x = (80 - 23) / 2
    y = ((24 - 4) / 2) + 1
    Width = 23 + 4
    " &Yes"
    " &No"
end mYesNoMenu

/*** mYesNo *************************************************************
 *                                                                      *
 * Open up a centered menu, place a "yes" on one line and a "no" on the *
 * other line, and hilite the "yes" item.  Determine whether a "Y" or   *
 * "N" were selected, and return a "1" for "yes" and a "0" for "no".    *
 *                                                                      *
 * Called by:   mViewClipBoard()                                        *
 *                                                                      *
 * Enter With:  the string to display.                                  *
 *                                                                      *
 * Returns:     Return a 1 if yes clicked, or "Y" pressed, or a 0 if no *
 *              clicked or "N" pressed, or "Esc" pressed.               *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

integer proc mYesNo(string s)
integer OldCursor

mYesNoSpin:
    OldCursor = Set(Cursor, OFF)                    // cursor off
    mYesNoMenu(s)                       // display the menu
    Set(Cursor, OldCursor)              // restore the cursor
    case MenuKey()                      // interpret the key pressed in menu
        when <y>, <Shift Y>
            return(1)
        when <n>, <Shift N>, <Escape>   // just fall out of case
        when <Enter>, <GreyEnter>, <LeftBtn>
            if MenuOption() == 1
                return(1)
            endif
        otherwise
            Goto mYesNoSpin             // invalid key, try again
    endcase
    return(0)                           // return no
end mYesNo

/*** mPopUpClipMenu *****************************************************
 *                                                                      *
 * Open up the clipboard menu, update the message line, and process     *
 * keystrokes/mouseclicks.                                              *
 *                                                                      *
 * Called by:   mViewClipBoard()                                        *
 *                                                                      *
 * Enter With:  an integer to set if the menu must popup again after    *
 *              exit, the item in the menu to hilite, the clipboard     *
 *              letter for the message, and whether or not a named      *
 *              clipboard is active.                                    *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mPopUpClipMenu(var integer ForceUpMenu, var integer HiliteItemCount,
    String ClipBoardLetter, integer ClipBoardIsNamed)
    integer OldCursor
    integer OldMsgAttr

    OldCursor = Set(Cursor, OFF)

    OldMsgAttr = Set(MsgAttr, Query(MenuTextLtrAttr))
    if ClipBoardIsNamed
        Message('NAMED CLIPBOARD-', ClipBoardLetter)
    else
        Message('MAIN CLIPBOARD   ')
    endif
    Set(MsgAttr, OldMsgAttr)
    mPopWinCentered(34, 12, 4, "Clipboard Menu",
        Query(MenuBorderAttr))
    mWriteTextInClipPopup(HiliteItemCount)
    mProcessPopupMenu(ForceUpMenu, HiliteItemCount)
    PopWinClose()
    Set(Cursor, OldCursor)
end mPopUpClipMenu

/*** mViewClipBoard *****************************************************
 *                                                                      *
 * Create a temporary buffer, and call it "clipboard".  Paste the       *
 * contents of the main clipboard into this buffer.  Start up a         *
 * keyboard loop, and output a help message on the status line,         *
 * initially identifying the file as the MAIN CLIPBOARD.  Then wait     *
 * for a keystroke.                                                     *
 *                                                                      *
 * If the "grey+" key or the "F12" key is pressed, then copy the next   *
 * clipboard to the buffer named "clipboard".  If the "grey-" key or    *
 * the "F11" key is pressed, then copy the previous clipboard to        *
 * buffer "clipboard".                                                  *
 *                                                                      *
 * If "Alt-P" is pressed, then paste the currently viewed clipboard     *
 * into the original file in the original window.                       *
 *                                                                      *
 * If "Escape" is pressed, then close the "clipboard" buffer, and       *
 * exit the clipboard viewer macro.                                     *
 *                                                                      *
 * Treat "CursorUp" and "CursorDown" requests as "RollUp" and           *
 * "RollDown" requests.                                                 *
 *                                                                      *
 * Respect the "PageUp" and "PageDown" keys.                            *
 *                                                                      *
 * If either "Enter", or the "SpaceBar" is pressed, then return to the  *
 * main clipboard.                                                      *
 *                                                                      *
 * Respect the vertical mouse buttons.                                  *
 *                                                                      *
 * If an alphabetic is pressed ("A" - "Z") try to go to that Named      *
 * ClipBoard, if it exists.  If it does not exist, just ignore the      *
 * keystroke.                                                           *
 *                                                                      *
 *                                                                      *
 * Called by:   Main()                                                  *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc mViewClipBoard()
    integer CID, MyKey = <SpaceBar>, ClipBoardIsNamed = FALSE
    integer MyOldID, PasteIntoApplication = 0, UpdateNeeded = FALSE
    integer ClearClipBoard = FALSE, HiliteItemCount = 10
    integer ForceUpMenu = FALSE
    string ClipBoardName[5] = "", ClipBoardLetter[2] = "A", MyAsciiKey[2] = ""
    string HMsg[63] = '  -  press SPACEBAR or LEFT MOUSE BUTTON for a Clipboard Menu'
    integer OldCursor, OldMsgAttr

    OldCursor = Set(Cursor, OFF)
    PushBlock()             // preserve any existing marking
    MyOldID = GetBufferID()
    CID = CreateBuffer("ClipBoard")
    WID = WindowID()
    if CID
        Paste()
        loop                // start up a clipboard loop
            FileChanged(FALSE)
            UpdateDisplay()
            OldMsgAttr = Query(MsgAttr)
            Set(MsgAttr, Query(MenuTextLtrAttr))
            if ClipBoardIsNamed
                Message('NAMED CLIPBOARD-', ClipBoardLetter,
                    iif(ForceUpMenu == FALSE, HMsg, ""))
            else
                Message('MAIN CLIPBOARD   ',
                    iif(ForceUpMenu == FALSE, HMsg, ""))
            endif
            Set(MsgAttr, OldMsgAttr)
            if not KeyPressed()
                if ForceUpMenu
                    ForceUpMenu = FALSE
                    PushKey(<Spacebar>)
                endif
            endif
            MyKey = GetKey()
            case MyKey
                when <Spacebar>, <F10>, <Enter>
                    mPopUpClipMenu(ForceUpMenu, HiliteItemCount,
                        ClipBoardLetter, ClipBoardIsNamed)
                when <Grey+>, <F12>, <Grey->, <F11>
                    UpdateNeeded = TRUE
                    mAdvanceClipBoard(MyKey, ClipBoardIsNamed,
                        ClipBoardLetter, ClipBoardName)
                when <Alt P>
                    UpdateNeeded = TRUE
                    if mYesNo("Paste Into Application?") == 1
                        PasteIntoApplication = MyKey
                        ForceUpMenu = FALSE
                    endif
                when <Ctrl P>
                    UpdateNeeded = TRUE
                    if mYesNo("Paste Over Application?") == 1
                        PasteIntoApplication = MyKey
                        ForceUpMenu = FALSE
                    endif
                when <Alt C>
                    UpdateNeeded = TRUE
                    if mYesNo("Clear Clipboard?") == 1
                        ClearClipBoard = TRUE
                    endif
                when <Escape>, <RightBtn>
                    UpdateDisplay()
                    break
                when <CursorUp>
                    RollUp()
                when <CursorDown>
                    RollDown()
                when <CursorRight>
                    RollRight()
                when <CursorLeft>
                    RollLeft()
                when <PgUp>
                    PageUp()
                when <PgDn>
                    PageDown()
                when <Ctrl PgDn>, <Ctrl End>
                    EndFile()
                when <Ctrl PgUp>, <Ctrl Home>
                    BegFile()
                when <Home>
                    UpdateNeeded = TRUE
                    ClipBoardIsNamed = FALSE
                when <LeftBtn>          // handle the mouse
                    case mHandleClipMouse(FALSE)
                        when 1
                            ProcessHotSpot()
                        when 0
                            mPopUpClipMenu(ForceUpMenu, HiliteItemCount,
                                ClipBoardLetter, ClipBoardIsNamed)
                    endcase
                otherwise
                    MyAsciiKey = Upper(Chr(MyKey & 0xff))
                    if (Asc(MyAsciiKey) >= Asc("A") and
                        Asc(MyAsciiKey) <= Asc("Z")) or
                        (Asc(MyAsciiKey) >= Asc("a") and
                        Asc(MyAsciiKey) <= Asc("z"))
                        UpdateNeeded = TRUE
                        if GetBufferID("+++" + MyAsciiKey)
                            ClipBoardName = "+++" + MyAsciiKey
                            ClipBoardIsNamed = TRUE
                            ClipBoardLetter = MyAsciiKey
                        endif
                    endif
            endcase
            if UpdateNeeded
                UpdateNeeded = FALSE
                if not UpdateClipboard(ClipBoardName, ClipBoardIsNamed,
                    PasteIntoApplication, MyOldID, ClearClipBoard)
                    break
                endif
            endif
            GotoBufferID(CID)               // back to viewer buffer
        endloop
        AbandonFile(CID)
    endif
    PopBlock()              // restore any existing marking
    GotoBufferID(MyOldID)
    Set(Cursor, OldCursor)
    UpdateDisplay()
end mViewClipBoard

/*** Main ***************************************************************
 *                                                                      *
 * This macro is executed when clipview is executed.  It is             *
 * automatically purged from memory when it completes.                  *
 *                                                                      *
 * Called by:   assigned to the "Ctrl S" key in "KEYSPRJ.KEY" and       *
 *              in "KEYSALT.KRY".                                       *
 *                                                                      *
 * Enter With:  nothing                                                 *
 *                                                                      *
 * Returns:     nothing                                                 *
 *                                                                      *
 * Notes:       none                                                    *
 *                                                                      *
 ************************************************************************/

proc Main()
    mViewClipBoard()
    PurgeMacro("clipview")
end Main

