/**************************************************************************
  A simple macro that saves changed files to:   temp_dir + fn + save_ext

  If you ever crash during a session, gather latest versions from your
  tempdir, by looking for the ".save" files.

  Files are only auto-saved if they are changed, and then only during
  idle time.  The files current modified status when the file was
  last auto-saved is stored in a buffer variable, and is used to keep
  from auto-saving a changed (but not changed since the last autosave)
  file.

  tempdir is cleaned up upon normal program termination.

  The TEMP or TMP environment variable must be set in order for this macro
  to locate the temporary directory.

  27 Apr 2020 SEM - use GetTempPath()
 **************************************************************************/

constant WAIT_TIME = 1000               // about 10 seconds
string temp_dir[_MAXPATH_],             // path from TEMP or TMP
    auto_save_ext[] = ".save",          // default extension
    save_dir[] = ".TSE",                // "/tmp/.TSE"
    changed_state[255]

integer idle_time, all_saved

string proc AutoSaveFn()
    return (temp_dir + SplitPath(CurrFilename(), _NAME_|_EXT_) + auto_save_ext)
end

integer proc ValidFilename(string fn)
    string bogus[5] = "<>?*|"
    integer i

    for i = 1 to Length(bogus)
        if Pos(bogus[i], fn) > 0
            return (FALSE)
        endif
    endfor
    return (TRUE)
end

#if 0
proc Show(string msg)
    vGotoXYAbs(80, 1)
    PutStr(msg, Query(StatusLineAttr))
end
#endif

proc Idle()
    integer id, i, n

    if Query(IdleTime) < WAIT_TIME
        return ()
    endif

    // See if we need to do anything
    if Query(IdleTime) >= idle_time and all_saved
        idle_time = Query(IdleTime)
        return ()
    endif
#if 0
    Show("AutoSave")
#endif
    all_saved =  FALSE
    id = GetBufferId()

    n = NumFiles() + (BufferType() <> _NORMAL_)
    for i = 1 to n
        if BufferType() == _NORMAL_ and Length(CurrFilename()) and ValidFilename(CurrFilename()) and FileChanged()
            if GetBufferInt(changed_state) <> FileChanged()
                if SaveAs(AutoSaveFn(), _OVERWRITE_|_QUIET_)
                    SetBufferInt(changed_state, FileChanged())
                endif
#if 0
                warn("saved")
            else
                warn("skipped")
#endif
            endif
            if KeyPressed()
                break
            endif
        elseif (i mod 5 == 0) and KeyPressed()
            break
        endif
        NextFile(_DONT_LOAD_)
    endfor

    GotoBufferId(id)
    if i >= NumFiles()
        all_saved = TRUE
    endif
    idle_time = Query(IdleTime)
#if 0
    Show("        ")
#endif
end

proc KillAutosave()
    if GetBufferInt(changed_state)
        EraseDiskFile(AutoSaveFn())
    endif
end

proc KillLotsOfAutosave()
    integer id

    id = GetBufferId()

    do NumFiles() + (BufferType() <> _NORMAL_) times
        KillAutosave()
        NextFile(_DONT_LOAD_)
    enddo

    GotoBufferId(id)
end

proc autosave_list()
    EditFile(temp_dir + "*.*")
end

proc autosave_all_changed()
    Warn("not implemented")
end

proc autosave_current()
    SaveAs(AutoSaveFn(), _OVERWRITE_|_QUIET_)
end

menu autosave_menu()
title = "AutoSave"
    "AutoSave &Current File",            autosave_current()
    "AutoSave &All Changed Files",       autosave_all_changed()
    "Show AutoSave &Directory",          autosave_list()
    "&Cancel"
end

public proc functions()
    autosave_menu()
end

proc WhenPurged()
    KillLotsOfAutosave()
end

proc WhenLoaded()
    string dir[_MAX_PATH_] = CurrDir()

    temp_dir = GetTempPath()

    if not FileExists(temp_dir + save_dir)
        ChDir(temp_dir)
        MkDir(save_dir)
        ChDir(dir)
    endif

    temp_dir = temp_dir + save_dir
    temp_dir = AddTrailingSlash(temp_dir)

    changed_state = SplitPath(CurrMacroFilename(), _NAME_) + ":last autosaved state"

    Hook(_IDLE_, Idle)
    Hook(_NONEDIT_IDLE_, Idle)
    Hook(_AFTER_FILE_SAVE_, KillAutosave)
    Hook(_ON_FILE_QUIT_, KillAutosave)
    Hook(_ON_ABANDON_EDITOR_, KillLotsOfAutosave)
end

proc main()
    case Query(MacroCmdLine)
        when "-save_current"        autosave_current()
        when "-save_all_changed"    autosave_all_changed()
        when "-list"                autosave_list()
        otherwise                   autosave_menu()
    endcase
end
