/*************************************************************************
  Adjust      Adjusts text in a marked block to the right, left or center

  Author:     SemWare

  Date:       May  5, 1993 (Steve Watkins)

   Sep 15, 2000 Two bug fixes submitted by Ross Boyd.  Thanks Ross!

   Feb 27, 2001 Supports 3 command line arguments: "left", "right", or
                "center". (Bill Stewart)
  Overview:

  This macro allows you to adjust text within a block by aligning it
  either on the left or right, or by centering it.  If a column block
  is marked, the macro operates on the text in the block using the
  leftmost and/or rightmost column boundary (as appropriate).  If a
  line or character block is marked, it operates on the text on the
  lines spanned by the block, using the left and/or right margin (as
  appropriate).  A selection menu issued by the macro allows you to
  choose whether you want to left adjust, right adjust, or center the
  text.

  Keys:       (none)

  Usage notes:

  This macro does not have any key assignments.  To use, simply select
  it from the Potpourri menu.

  Copyright 1992-2001 SemWare Corporation.  All Rights Reserved Worldwide.

  Use, modification, and distribution of this SAL macro is encouraged by
  SemWare provided that this statement, including the above copyright
  notice, is not removed; and provided that no fee or other remuneration
  is received for distribution.  You may add your own copyright notice
  to cover new matter you add to the macro, but SemWare Corporation will
  neither support nor assume legal responsibility for any material added
  or any changes made to the macro.

*************************************************************************/

constant    LEFT_JUSTIFY   = 0x01,
            RIGHT_JUSTIFY  = 0x02,
            CENTER_JUSTIFY = 0x04

constant    MAX_LINE_LEN = MAXLINELEN

integer beg_line, beg_col, end_line, end_col

proc CheckBreak()
    if KeyPressed() and GetKey() == <Escape>
        Alarm()
        Message("Justification aborted at line ", CurrLine())
        GotoLine(end_line + 1)
    endif
end

proc JustifyMargins(integer flags)
    integer delta

    repeat
        if flags & (LEFT_JUSTIFY | CENTER_JUSTIFY)
            delta = beg_col - PosFirstNonWhite()
            if delta > MAX_LINE_LEN - PosLastNonWhite()
                delta = MAX_LINE_LEN - PosLastNonWhite()
            endif
            ShiftText(delta)
        endif

        if flags & (RIGHT_JUSTIFY | CENTER_JUSTIFY)
            delta = end_col - PosLastNonWhite()
            if -delta > PosFirstNonWhite()
                delta = -PosFirstNonWhite() + 1
            endif
            ShiftText(iif(flags & CENTER_JUSTIFY, delta/2, delta))
        endif

        CheckBreak()

    until not Down() or CurrLine() > end_line
end

proc JustifyBlock(integer flags)
    integer count

    count = 0
    repeat
        if flags & (LEFT_JUSTIFY | CENTER_JUSTIFY)
            GotoColumn(beg_col)
            if isWhite() and CurrChar() >= 0
                WordRight()
                ShiftText(beg_col - CurrCol())
            endif
        endif

        if flags & (RIGHT_JUSTIFY | CENTER_JUSTIFY)
            GotoColumn(end_col)
            if (CurrChar() < 0 or isWhite()) and PosLastNonWhite() > 0
                WordLeft()
                EndWord()
                if CurrCol() > beg_col
                    if flags & CENTER_JUSTIFY
                        GotoColumn((end_col + CurrCol() + 1)/2)
                    endif
                    ShiftText(end_col - CurrCol() + 1)
                endif
            endif
        endif

        count = count + 1
        if count mod 400 == 0
            CheckBreak()
            Message("Justifying text... Press <Escape> to abort"; CurrLine())
        endif

    until not Down() or CurrLine() > end_line
end

proc JustifyText(integer type)
    string ws[32]
    integer use_margins, col, curr_line, curr_col, curr_xoffset

    curr_line = CurrLine()
    curr_col = CurrCol()
    curr_xoffset = CurrXOffset()
    if isBlockinCurrFile()
        if lFind(chr(9),"gl")
            Warn("Real tabs not supported")
        else
            col = CurrCol()
            ws = Set(WordSet, ChrSet("~ \t"))

            PushBlock()
            PushPosition()

            use_margins = FALSE
            beg_line = Query(BlockBegLine)
            beg_col  = Query(BlockBegCol)
            end_line = Query(BlockEndLine)
            end_col  = Query(BlockEndCol)
            GotoBlockBegin()

            if isBlockMarked() <> _COLUMN_
                UnMarkBlock()
                use_margins = TRUE
                beg_col = Query(LeftMargin)
                end_col = Query(RightMargin)
                if beg_col == 0
                    beg_col = 1
                endif
            endif

            if (end_col > beg_col)
                Message("Justifying text... Press <Escape> to abort")
                GotoLine(beg_line)
                if use_margins
                    JustifyMargins(type)
                else
                    JustifyBlock(type)
                endif
                UpdateDisplay()
            else
                Warn("Left margin must be < Right margin")
            endif

            PopPosition()
            PopBlock()

            Set(Wordset,ws)

            GotoColumn(col)
        endif
    else
        Warn("Block not in current file")
    endif
    GotoLine(curr_line)
    GotoColumn(curr_col)
    GotoXOffset(curr_xoffset)
end

menu JustifyMenu()
    "&Left Justify"     ,   JustifyText(LEFT_JUSTIFY)
    "&Right Justify"    ,   JustifyText(RIGHT_JUSTIFY)
    "&Center Justify"   ,   JustifyText(CENTER_JUSTIFY)
end

proc main()
    case Lower(Query(MacroCmdLine))
        when      "left"   JustifyText(LEFT_JUSTIFY)
        when      "right"  JustifyText(RIGHT_JUSTIFY)
        when      "center" JustifyText(CENTER_JUSTIFY)
        otherwise          JustifyMenu()
    endcase
end

